"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.directorySnapshot = exports.synthSnapshotWithPost = exports.synthSnapshot = exports.mkdtemp = exports.execProjenCLI = exports.TestProject = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const glob_1 = require("glob");
const __1 = require("..");
const logging = require("../logging");
const util_1 = require("../util");
const PROJEN_CLI = require.resolve('../../bin/projen');
logging.disable(); // no logging during tests
class TestProject extends __1.Project {
    constructor(options = {}) {
        const tmpdir = mkdtemp();
        super({
            name: 'my-project',
            outdir: tmpdir,
            clobber: false,
            logging: {
                level: __1.LogLevel.OFF,
            },
            ...options,
        });
    }
    postSynthesize() {
        fs.writeFileSync(path.join(this.outdir, '.postsynth'), '# postsynth');
    }
}
exports.TestProject = TestProject;
function execProjenCLI(workdir, args = []) {
    const command = [
        process.execPath,
        PROJEN_CLI,
        ...args,
    ];
    return util_1.exec(command.map(x => `"${x}"`).join(' '), { cwd: workdir, stdio: 'inherit' });
}
exports.execProjenCLI = execProjenCLI;
const autoRemove = new Set();
// Hook to automatically remove temporary directories without needing each
// place to actually handle this specifically.
afterAll((done) => {
    // Array.from used to get a copy, so we can safely remove from the set
    for (const dir of Array.from(autoRemove)) {
        try {
            // Note - fs-extra.removeSync is idempotent, so we're safe if the
            // directory has already been cleaned up before we get there!
            fs.removeSync(dir);
        }
        catch (e) {
            done.fail(e);
        }
        autoRemove.delete(dir);
    }
    done();
});
function mkdtemp() {
    const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-test-'));
    autoRemove.add(tmpdir);
    return tmpdir;
}
exports.mkdtemp = mkdtemp;
function synthSnapshot(project) {
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = 'true';
        project.synth();
        return directorySnapshot(project.outdir);
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function synthSnapshotWithPost(project) {
    try {
        project.synth();
        return directorySnapshot(project.outdir);
    }
    finally {
        fs.removeSync(project.outdir);
    }
}
exports.synthSnapshotWithPost = synthSnapshotWithPost;
function directorySnapshot(root, options = {}) {
    var _a;
    const output = {};
    const files = glob_1.glob.sync('**', {
        ignore: (_a = options.excludeGlobs) !== null && _a !== void 0 ? _a : [],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (path.extname(filePath) === '.json') {
            content = fs.readJsonSync(filePath);
        }
        else {
            content = fs.readFileSync(filePath, 'utf-8');
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
//# sourceMappingURL=data:application/json;base64,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