# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['proc_wrapper']

package_data = \
{'': ['*']}

install_requires = \
['Jinja2==3.0.3']

extras_require = \
{'docs': ['Sphinx>=3.5.4,<4.0.0',
          'sphinx-rtd-theme>=1.0.0,<2.0.0',
          'myst-parser>=0.17.2,<0.18.0']}

setup_kwargs = {
    'name': 'cloudreactor-procwrapper',
    'version': '4.0.0',
    'description': 'Wraps the execution of processes so that a service API endpoint (CloudReactor) can monitor and manage them. Also implements retries, timeouts, and secret injection from AWS into the environment.',
    'long_description': '# cloudreactor-procwrapper\n\n<p align="center">\n  <a href="https://github.com/CloudReactor/cloudreactor-procwrapper/actions?query=workflow%3ACI">\n    <img src="https://img.shields.io/github/workflow/status/CloudReactor/cloudreactor-procwrapper/CI/main?label=CI&logo=github&style=flat-square" alt="CI Status" >\n  </a>\n  <a href="https://cloudreactor-procwrapper.readthedocs.io">\n    <img src="https://img.shields.io/readthedocs/cloudreactor-procwrapper.svg?logo=read-the-docs&logoColor=fff&style=flat-square" alt="Documentation Status">\n  </a>\n  <a href="https://codecov.io/gh/CloudReactor/cloudreactor-procwrapper">\n    <img src="https://img.shields.io/codecov/c/github/CloudReactor/cloudreactor-procwrapper.svg?logo=codecov&logoColor=fff&style=flat-square" alt="Test coverage percentage">\n  </a>\n</p>\n<p align="center">\n  <a href="https://python-poetry.org/">\n    <img src="https://img.shields.io/badge/packaging-poetry-299bd7?style=flat-square&logo=data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAA4AAAASCAYAAABrXO8xAAAACXBIWXMAAAsTAAALEwEAmpwYAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAJJSURBVHgBfZLPa1NBEMe/s7tNXoxW1KJQKaUHkXhQvHgW6UHQQ09CBS/6V3hKc/AP8CqCrUcpmop3Cx48eDB4yEECjVQrlZb80CRN8t6OM/teagVxYZi38+Yz853dJbzoMV3MM8cJUcLMSUKIE8AzQ2PieZzFxEJOHMOgMQQ+dUgSAckNXhapU/NMhDSWLs1B24A8sO1xrN4NECkcAC9ASkiIJc6k5TRiUDPhnyMMdhKc+Zx19l6SgyeW76BEONY9exVQMzKExGKwwPsCzza7KGSSWRWEQhyEaDXp6ZHEr416ygbiKYOd7TEWvvcQIeusHYMJGhTwF9y7sGnSwaWyFAiyoxzqW0PM/RjghPxF2pWReAowTEXnDh0xgcLs8l2YQmOrj3N7ByiqEoH0cARs4u78WgAVkoEDIDoOi3AkcLOHU60RIg5wC4ZuTC7FaHKQm8Hq1fQuSOBvX/sodmNJSB5geaF5CPIkUeecdMxieoRO5jz9bheL6/tXjrwCyX/UYBUcjCaWHljx1xiX6z9xEjkYAzbGVnB8pvLmyXm9ep+W8CmsSHQQY77Zx1zboxAV0w7ybMhQmfqdmmw3nEp1I0Z+FGO6M8LZdoyZnuzzBdjISicKRnpxzI9fPb+0oYXsNdyi+d3h9bm9MWYHFtPeIZfLwzmFDKy1ai3p+PDls1Llz4yyFpferxjnyjJDSEy9CaCx5m2cJPerq6Xm34eTrZt3PqxYO1XOwDYZrFlH1fWnpU38Y9HRze3lj0vOujZcXKuuXm3jP+s3KbZVra7y2EAAAAAASUVORK5CYII=" alt="Poetry">\n  </a>\n  <a href="https://github.com/pre-commit/pre-commit">\n    <img src="https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white&style=flat-square" alt="pre-commit">\n  </a>\n</p>\n<p align="center">\n  <a href="https://pypi.org/project/cloudreactor-procwrapper/">\n    <img src="https://img.shields.io/pypi/v/cloudreactor-procwrapper.svg?logo=python&logoColor=fff&style=flat-square" alt="PyPI Version">\n  </a>\n  <img src="https://img.shields.io/pypi/pyversions/cloudreactor-procwrapper.svg?style=flat-square&logo=python&amp;logoColor=fff" alt="Supported Python versions">\n  <img src="https://img.shields.io/pypi/l/cloudreactor-procwrapper.svg?style=flat-square" alt="License">\n</p>\n\nWraps the execution of processes so that an API server\n([CloudReactor](https://cloudreactor.io/))\ncan monitor and manage them.\nAvailable as a standalone executable or as a python module.\n\n## Features\n\n* Runs either processes started with a command line or a python function you\nsupply\n* Implements retries and time limits on wrapped processes\n* Injects secrets from AWS Secrets Manager, AWS S3, or local files and extracts\nthem into the process environment (for command-lines) or configuration (for\nfunctions)\n* When used with the CloudReactor service:\n  * Reports when a process/function starts and when it exits, along with the\n  exit code\n  * Sends heartbeats, optionally with status information like the number of\n  items processed\n  * Prevents too many concurrent executions\n  * Stops execution when manually stopped in the CloudReactor dashboard\n\n## How it works\n\nFirst, secrets and other configuration are fetched and resolved from\nproviders like AWS Secrets Manager, AWS S3, or the local filesystem.\n\nJust before your code runs, the module requests the API server to create a\nTask Execution associated with the Task name or UUID which you pass to the\nmodule.\nThe API server may reject the request if too many instances of the Task are\ncurrently running, but otherwise records that a Task Execution has started.\nThe module then passes control to your code.\n\nWhile your code is running, it may report progress to the API server,\nand the API server may signal that your Task stop execution (due to\nuser manually stopping the Task Execution), in which\ncase the module terminates your code and exits.\n\nAfter your code finishes, the module informs the API server of\nthe exit code or result. CloudReactor monitors Tasks to ensure they\nare still responsive, and keeps a history of the Executions of Tasks,\nallowing you to view failures and run durations in the past.\n\n### Auto-created Tasks\n\nBefore your Task is run (including this module),\nthe [AWS ECS CloudReactor Deployer](https://github.com/CloudReactor/aws-ecs-cloudreactor-deployer)\ncan be used to set it up in AWS ECS,\nand inform CloudReactor of details of your Task.\nThat way CloudReactor can start and schedule your Task, and setup your\nTask as a service.\nSee [CloudReactor python ECS QuickStart](https://github.com/CloudReactor/cloudreactor-python-ecs-quickstart)\nfor an example.\n\nHowever, it may not be possible or desired to change your deployment process.\nInstead, you may configure the Task to be *auto-created*.\n\nAuto-created Tasks are created the first time your Task runs.\nThis means there is no need to inform the API server of the Task details\n(during deployment) before it runs.\nInstead, each time the module runs, it informs the API server of the\nTask details at the same time as it requests the creation of a Task Execution.\nOne disadvantage of auto-created Tasks is that they are not available\nin the CloudReactor dashboard until the first time they run.\n\nWhen configuring a Task to be auto-created, you must specify the name\nor UUID of the Run Environment in CloudReactor that the Task is\nassociated with. The Run Environment must be created ahead of time,\neither by the Cloudreactor AWS Setup Wizard,\nor manually in the CloudReactor dashboard.\n\nYou can also specify more Task properties, such as Alert Methods and\nexternal links in the dashboard, by setting the environment variable\n`PROC_WRAPPER_AUTO_CREATE_TASK_PROPS` set to a JSON-encoded object that has the\n[CloudReactor Task schema](https://apidocs.cloudreactor.io/#operation/tasks_create).\n\n### Execution Methods\n\nCloudReactor currently supports two Execution Methods:\n\n1) [AWS ECS (in Fargate)](https://aws.amazon.com/fargate/)\n2) Unknown\n\nIf a Task is running in AWS ECS, CloudReactor is able to run additional\nTask Executions, provided the details of running the Task is provided\nduring deployment with the AWS ECS CloudReactor Deployer, or if the\nTask is configured to be auto-created, and this module is run. In the\nsecond case, this module uses the ECS Metadata endpoint to detect\nthe ECS Task settings, and sends them to the API server. CloudReactor\ncan also schedule Tasks or setup long-running services using Tasks,\nprovided they are run in AWS ECS.\n\nHowever, a Task may use the Unknown execution method if it is not running\nin AWS ECS. If that is the case, CloudReactor won\'t be able to\nstart the Task in the dashboard or as part of a Workflow,\nschedule the Task, or setup a service with the Task. But the advantage is\nthat the Task code can be executed by any method available to you,\nsuch as bare metal servers, VM\'s, Docker, AWS Lambda, or Kubernetes.\nAll Tasks in CloudReactor, regardless of execution method, have their\nhistory kept and are monitored.\n\nThis module detects which of the two Execution Methods your Task is\nrunning with and sends that information to the API server, provided\nyou configure your Task to be auto-created.\n\n### Passive Tasks\n\nPassive Tasks are Tasks that CloudReactor does not manage. This means\nscheduling and service setup must be handled by other means\n(cron jobs, [supervisord](http://supervisord.org/), etc).\nHowever, Tasks marked as services or that have a schedule will still be\nmonitored by CloudReactor, which will send notifications if\na service Task goes down or a Task does not run on schedule.\n\nThe module reports to the API server that auto-created Tasks are passive,\nunless you specify the `--force-task-passive` commmand-line option or\nset the environment variable `PROC_WRAPPER_TASK_IS_PASSIVE` to `FALSE`.\nIf a Task uses the Unknown Execution Method, it must be marked as passive,\nbecause CloudReactor does not know how to manage it.\n\n## Pre-requisites\n\nIf you just want to use this module to retry processes, limit execution time,\nor fetch secrets, you can use offline mode, in which case no CloudReactor API\nkey is required. But CloudReactor offers a free tier so we hope you\n[sign up](https://dash.cloudreactor.io/signup)\nor a free account to enable monitoring and/or management.\n\nIf you want CloudReactor to be able to start your Tasks, you should use the\n[Cloudreactor AWS Setup Wizard](https://github.com/CloudReactor/cloudreactor-aws-setup-wizard)\nto configure your AWS environment to run Tasks in ECS Fargate.\nYou can skip this step if running in passive mode is OK for you.\n\nIf you want to use CloudReactor to manage or just monitor your Tasks,\nyou need to create a Run Environment and an API key in the CloudReactor\ndashboard. The API key can be scoped to the Run Environment if you\nwish. The key must have at least the Task access level, but for\nan auto-created Task, it must have at least the Developer access level.\n\n## Installation\n\n### In Linux/AMD64 only\n\nStandalone executables built by\n[nuitka](https://nuitka.net/index.html)\nfor 64-bit Linux are available, located in `bin/nuitka`. These executables bundle\npython so you don\'t need to have python installed on your machine. They also\nbundle all optional library dependencies so you can fetch secrets from AWS\nSecrets Manager and extract them with jsonpath-ng, for example.\n\nCompared to executables built by PyInstaller (see below), they start up faster,\nand most likely are more efficient at runtime.\nHowever, they cannot be executed directly inside a Docker container. Instead, a\nseparate extraction step must be performed at Docker build time:\n\n    RUN wget -nv https://github.com/CloudReactor/cloudreactor-procwrapper/raw/4.0/bin/nuitka/linux-amd64/4.0.0/proc_wrapper.bin -O proc_wrapper_app_image \\\n      && chmod +x proc_wrapper_app_image \\\n      && ./proc_wrapper_app_image --appimage-extract \\\n      && rm ./proc_wrapper_app_image\n\nThen to run the wrapper:\n\n    ENTRYPOINT ["squashfs-root/proc_wrapper.bin"]\n\nSee the example\n[Dockerfile](tests/integration/nuitka_executable/docker_context_linux_amd64/Dockerfile) for a known working environment.\n\n### In a Linux/AMD64 or Windows 64 environment\n\nStandalone executables built by [PyInstaller](https://www.pyinstaller.org/) for 64-bit Linux and Windows are available, located in `bin/pyinstaller`.\nThese executables bundle\npython so you don\'t need to have python installed on your machine. They also\nbundle all optional library dependencies so you can fetch secrets from AWS\nSecrets Manager and extract them with jsonpath-ng, for example. Compared to\nexecutables built by nuitka, they start up slower but are able to run directly\nin a Docker container without extraction.\n\nOn a debian buster machine, the following packages (with known supported versions)\nmust be installed to run :\n\n      openssl=1.1.1d-0+deb10u5\n      libexpat1=2.2.6-2+deb10u1\n      ca-certificates=20200601~deb10u2\n\nSee the example\n[Dockerfile](tests/integration/pyinstaller_executable/docker_context_linux_amd64/Dockerfile) for a known working environment.\n\nSpecial thanks to\n[wine](https://www.winehq.org/) and\n[PyInstaller Docker Images](https://github.com/cdrx/docker-pyinstaller)\nfor making it possible to cross-compile!\n\n### When python is available\n\nInstall this module via pip (or your favourite package manager):\n\n`pip install cloudreactor-procwrapper`\n\nFetching secrets from AWS Secrets Manager requires that\n[boto3](https://boto3.amazonaws.com/v1/documentation/api/latest/index.html) is available to import in your python environment.\n\nJSON Path transformation requires that [jsonpath-ng](https://github.com/h2non/jsonpath-ng)\nbe available to import in your python environment.\n\nYou can get the tested versions of both dependencies in\n[proc_wrapper-requirements.in](https://github.com/CloudReactor/cloudreactor-procwrapper/blob/main/proc_wrapper-requirements.in)\n(suitable for use by [pip-tools](https://github.com/jazzband/pip-tools/)) or the resolved requirements in\n[proc_wrapper-requirements.txt](https://github.com/CloudReactor/cloudreactor-procwrapper/blob/main/proc_wrapper-requirements.txt).\n\n## Usage\n\nThere are two ways of using the module: wrapped mode and embedded mode.\n\n### Wrapped mode\n\nIn wrapped mode, you pass a command line to the module which it\nexecutes in a child process. The command can be implemented in whatever\nprogramming language the running machine supports.\n\nInstead of running\n\n    somecommand --somearg x\n\nyou would run\n\n    ./proc_wrapper somecommand --somearg x\n\nassuming that are using the PyInstaller standalone executable, and that\nyou configure the program using environment variables.\n\nOr, if you have python installed:\n\n    python -m proc_wrapper somecommand --somearg x\n\nHere are all the options:\n\n    usage: proc_wrapper [-h] [-v] [-n TASK_NAME] [--task-uuid TASK_UUID] [-a]\n                        [--auto-create-task-run-environment-name AUTO_CREATE_TASK_RUN_ENVIRONMENT_NAME]\n                        [--auto-create-task-run-environment-uuid AUTO_CREATE_TASK_RUN_ENVIRONMENT_UUID]\n                        [--auto-create-task-props AUTO_CREATE_TASK_PROPS]\n                        [--force-task-active]\n                        [--task-execution-uuid TASK_EXECUTION_UUID]\n                        [--task-version-number TASK_VERSION_NUMBER]\n                        [--task-version-text TASK_VERSION_TEXT]\n                        [--task-version-signature TASK_VERSION_SIGNATURE]\n                        [--execution-method-props EXECUTION_METHOD_PROPS]\n                        [--task-instance-metadata TASK_INSTANCE_METADATA] [-s]\n                        [--schedule SCHEDULE] [--max-concurrency MAX_CONCURRENCY]\n                        [--max-conflicting-age MAX_CONFLICTING_AGE]\n                        [--api-base-url API_BASE_URL] [-k API_KEY]\n                        [--api-heartbeat-interval API_HEARTBEAT_INTERVAL]\n                        [--api-error-timeout API_ERROR_TIMEOUT]\n                        [--api-final-update-timeout API_FINAL_UPDATE_TIMEOUT]\n                        [--api-retry-delay API_RETRY_DELAY]\n                        [--api-resume-delay API_RESUME_DELAY]\n                        [--api-task-execution-creation-error-timeout API_TASK_EXECUTION_CREATION_ERROR_TIMEOUT]\n                        [--api-task-execution-creation-conflict-timeout API_TASK_EXECUTION_CREATION_CONFLICT_TIMEOUT]\n                        [--api-task-execution-creation-conflict-retry-delay API_TASK_EXECUTION_CREATION_CONFLICT_RETRY_DELAY]\n                        [--api-request-timeout API_REQUEST_TIMEOUT] [-o] [-p]\n                        [-d DEPLOYMENT] [--send-pid] [--send-hostname]\n                        [--no-send-runtime-metadata]\n                        [-l {DEBUG,INFO,WARNING,ERROR,CRITICAL}] [--log-secrets]\n                        [--exclude-timestamps-in-log] [-w WORK_DIR]\n                        [-c COMMAND_LINE] [--shell-mode {auto,enable,disable}]\n                        [--no-strip-shell-wrapping]\n                        [--no-process-group-termination] [-t PROCESS_TIMEOUT]\n                        [-r PROCESS_MAX_RETRIES]\n                        [--process-retry-delay PROCESS_RETRY_DELAY]\n                        [--process-check-interval PROCESS_CHECK_INTERVAL]\n                        [--process-termination-grace-period PROCESS_TERMINATION_GRACE_PERIOD]\n                        [--enable-status-update-listener]\n                        [--status-update-socket-port STATUS_UPDATE_SOCKET_PORT]\n                        [--status-update-message-max-bytes STATUS_UPDATE_MESSAGE_MAX_BYTES]\n                        [--status-update-interval STATUS_UPDATE_INTERVAL]\n                        [-e ENV_LOCATIONS] [--config CONFIG_LOCATIONS]\n                        [--config-merge-strategy {SHALLOW,REPLACE,ADDITIVE,TYPESAFE_REPLACE,TYPESAFE_ADDITIVE}]\n                        [--overwrite_env_during_resolution]\n                        [--config-ttl CONFIG_TTL]\n                        [--no-fail-fast-config-resolution]\n                        [--resolved-env-var-name-prefix RESOLVED_ENV_VAR_NAME_PREFIX]\n                        [--resolved-env-var-name-suffix RESOLVED_ENV_VAR_NAME_SUFFIX]\n                        [--resolved-config-property-name-prefix RESOLVED_CONFIG_PROPERTY_NAME_PREFIX]\n                        [--resolved-config-property-name-suffix RESOLVED_CONFIG_PROPERTY_NAME_SUFFIX]\n                        [--env-var-name-for-config ENV_VAR_NAME_FOR_CONFIG]\n                        [--config-property-name-for-env CONFIG_PROPERTY_NAME_FOR_ENV]\n                        [--rollbar-access-token ROLLBAR_ACCESS_TOKEN]\n                        [--rollbar-retries ROLLBAR_RETRIES]\n                        [--rollbar-retry-delay ROLLBAR_RETRY_DELAY]\n                        [--rollbar-timeout ROLLBAR_TIMEOUT]\n                        ...\n\n    Wraps the execution of processes so that a service API endpoint (CloudReactor)\n    is optionally informed of the progress. Also implements retries, timeouts, and\n    secret injection into the environment.\n\n    positional arguments:\n      command\n\n    optional arguments:\n      -h, --help            show this help message and exit\n      -v, --version         Print the version and exit\n\n    task:\n      Task settings\n\n      -n TASK_NAME, --task-name TASK_NAME\n                            Name of Task (either the Task Name or the Task UUID\n                            must be specified\n      --task-uuid TASK_UUID\n                            UUID of Task (either the Task Name or the Task UUID\n                            must be specified)\n      -a, --auto-create-task\n                            Create the Task even if not known by the API server\n      --auto-create-task-run-environment-name AUTO_CREATE_TASK_RUN_ENVIRONMENT_NAME\n                            Name of the Run Environment to use if auto-creating\n                            the Task (either the name or UUID of the Run\n                            Environment must be specified if auto-creating the\n                            Task). Defaults to the deployment name if the Run\n                            Environment UUID is not specified.\n      --auto-create-task-run-environment-uuid AUTO_CREATE_TASK_RUN_ENVIRONMENT_UUID\n                            UUID of the Run Environment to use if auto-creating\n                            the Task (either the name or UUID of the Run\n                            Environment must be specified if auto-creating the\n                            Task)\n      --auto-create-task-props AUTO_CREATE_TASK_PROPS\n                            Additional properties of the auto-created Task, in\n                            JSON format. See https://apidocs.cloudreactor.io/#oper\n                            ation/api_v1_tasks_create for the schema.\n      --force-task-active   Indicates that the auto-created Task should be\n                            scheduled and made a service by the API server, if\n                            applicable. Otherwise, auto-created Tasks are marked\n                            passive.\n      --task-execution-uuid TASK_EXECUTION_UUID\n                            UUID of Task Execution to attach to\n      --task-version-number TASK_VERSION_NUMBER\n                            Numeric version of the Task\'s source code\n      --task-version-text TASK_VERSION_TEXT\n                            Human readable version of the Task\'s source code\n      --task-version-signature TASK_VERSION_SIGNATURE\n                            Version signature of the Task\'s source code (such as a\n                            git commit hash)\n      --execution-method-props EXECUTION_METHOD_PROPS\n                            Additional properties of the execution method, in JSON\n                            format. See https://apidocs.cloudreactor.io/#operation\n                            /api_v1_task_executions_create for the schema.\n      --task-instance-metadata TASK_INSTANCE_METADATA\n                            Additional metadata about the Task instance, in JSON\n                            format\n      -s, --service         Indicate that this is a Task that should run\n                            indefinitely\n      --schedule SCHEDULE   Run schedule reported to the API server\n      --max-concurrency MAX_CONCURRENCY\n                            Maximum number of concurrent Task Executions of the\n                            same Task. Defaults to 1.\n      --max-conflicting-age MAX_CONFLICTING_AGE\n                            Maximum age of conflicting Tasks to consider, in\n                            seconds. -1 means no limit. Defaults to the heartbeat\n                            interval, plus 60 seconds for services that send\n                            heartbeats. Otherwise, defaults to no limit.\n\n    api:\n      API client settings\n\n      --api-base-url API_BASE_URL\n                            Base URL of API server. Defaults to\n                            https://api.cloudreactor.io\n      -k API_KEY, --api-key API_KEY\n                            API key. Must have at least the Task access level, or\n                            Developer access level for auto-created Tasks.\n      --api-heartbeat-interval API_HEARTBEAT_INTERVAL\n                            Number of seconds to wait between sending heartbeats\n                            to the API server. -1 means to not send heartbeats.\n                            Defaults to 30 for concurrency limited services, 300\n                            otherwise.\n      --api-error-timeout API_ERROR_TIMEOUT\n                            Number of seconds to wait while receiving recoverable\n                            errors from the API server. Defaults to 300.\n      --api-final-update-timeout API_FINAL_UPDATE_TIMEOUT\n                            Number of seconds to wait while receiving recoverable\n                            errors from the API server when sending the final\n                            update before exiting. Defaults to 1800.\n      --api-retry-delay API_RETRY_DELAY\n                            Number of seconds to wait before retrying an API\n                            request. Defaults to 120.\n      --api-resume-delay API_RESUME_DELAY\n                            Number of seconds to wait before resuming API\n                            requests, after retries are exhausted. Defaults to\n                            600. -1 means to never resume.\n      --api-task-execution-creation-error-timeout API_TASK_EXECUTION_CREATION_ERROR_TIMEOUT\n                            Number of seconds to keep retrying Task Execution\n                            creation while receiving error responses from the API\n                            server. -1 means to keep trying indefinitely. Defaults\n                            to 300.\n      --api-task-execution-creation-conflict-timeout API_TASK_EXECUTION_CREATION_CONFLICT_TIMEOUT\n                            Number of seconds to keep retrying Task Execution\n                            creation while conflict is detected by the API server.\n                            -1 means to keep trying indefinitely. Defaults to 1800\n                            for concurrency limited services, 0 otherwise.\n      --api-task-execution-creation-conflict-retry-delay API_TASK_EXECUTION_CREATION_CONFLICT_RETRY_DELAY\n                            Number of seconds between attempts to retry Task\n                            Execution creation after conflict is detected.\n                            Defaults to 60 for concurrency-limited services, 120\n                            otherwise.\n      --api-request-timeout API_REQUEST_TIMEOUT\n                            Timeout for contacting API server, in seconds.\n                            Defaults to 30.\n      -o, --offline-mode    Do not communicate with or rely on an API server\n      -p, --prevent-offline-execution\n                            Do not start processes if the API server is\n                            unavailable or the wrapper is misconfigured.\n      -d DEPLOYMENT, --deployment DEPLOYMENT\n                            Deployment name (production, staging, etc.)\n      --send-pid            Send the process ID to the API server\n      --send-hostname       Send the hostname to the API server\n      --no-send-runtime-metadata\n                            Do not send metadata about the runtime environment\n      --exclude-timestamps-in-log\n                            Exclude timestamps in log (possibly because the log\n                            stream will be enriched by timestamps automatically by\n                            a logging service like AWS CloudWatch Logs)\n\n    log:\n      Logging settings\n\n      -l {DEBUG,INFO,WARNING,ERROR,CRITICAL}, --log-level {DEBUG,INFO,WARNING,ERROR,CRITICAL}\n                            Log level\n      --log-secrets         Log sensitive information\n\n    process:\n      Process settings\n\n      -w WORK_DIR, --work-dir WORK_DIR\n                            Working directory. Defaults to the current directory.\n      -c COMMAND_LINE, --command-line COMMAND_LINE\n                            Command line to execute\n      --shell-mode {auto,enable,disable}\n                            Indicates if the process command should be executed in\n                            a shell. Executing in a shell allows shell scripts,\n                            commands, and expressions to be used, with the\n                            disadvantage that termination signals may not be\n                            propagated to child processes. Options are: enable --\n                            Force the command to be executed in a shell; disable\n                            -- Force the command to be executed without a shell;\n                            auto -- Auto-detect the shell mode by analyzing the\n                            command.\n      --no-strip-shell-wrapping\n                            Do not strip the command-line of shell wrapping like\n                            "/bin/sh -c" that can be introduced by Docker when\n                            using shell form of ENTRYPOINT and CMD.\n      --no-process-group-termination\n                            Send termination and kill signals to the wrapped\n                            process only, instead of its process group (which is\n                            the default). Sending to the process group allows all\n                            child processes to receive the signals, even if the\n                            wrapped process does not forward signals. However, if\n                            your wrapped process manually handles and forwards\n                            signals to its child processes, you probably want to\n                            send signals to only your wrapped process.\n      -t PROCESS_TIMEOUT, --process-timeout PROCESS_TIMEOUT\n                            Timeout for process completion, in seconds. -1 means\n                            no timeout, which is the default.\n      -r PROCESS_MAX_RETRIES, --process-max-retries PROCESS_MAX_RETRIES\n                            Maximum number of times to retry failed processes. -1\n                            means to retry forever. Defaults to 0.\n      --process-retry-delay PROCESS_RETRY_DELAY\n                            Number of seconds to wait before retrying a process.\n                            Defaults to 60.\n      --process-check-interval PROCESS_CHECK_INTERVAL\n                            Number of seconds to wait between checking the status\n                            of processes. Defaults to 10.\n      --process-termination-grace-period PROCESS_TERMINATION_GRACE_PERIOD\n                            Number of seconds to wait after sending SIGTERM to a\n                            process, but before killing it with SIGKILL. Defaults\n                            to 30.\n\n    updates:\n      Status update settings\n\n      --enable-status-update-listener\n                            Listen for status updates from the process, sent on\n                            the status socket port via UDP. If not specified,\n                            status update messages will not be read.\n      --status-update-socket-port STATUS_UPDATE_SOCKET_PORT\n                            The port used to receive status updates from the\n                            process. Defaults to 2373.\n      --status-update-message-max-bytes STATUS_UPDATE_MESSAGE_MAX_BYTES\n                            The maximum number of bytes status update messages can\n                            be. Defaults to 65536.\n      --status-update-interval STATUS_UPDATE_INTERVAL\n                            Minimum of number of seconds to wait between sending\n                            status updates to the API server. -1 means to not send\n                            status updates except with heartbeats. Defaults to -1.\n\n    configuration:\n      Environment/configuration resolution settings\n\n      -e ENV_LOCATIONS, --env ENV_LOCATIONS\n                            Location of either local file, AWS S3 ARN, or AWS\n                            Secrets Manager ARN containing properties used to\n                            populate the environment for embedded mode, or the\n                            process environment for wrapped mode. By default, the\n                            file format is assumed to be dotenv. Specify multiple\n                            times to include multiple locations.\n      --config CONFIG_LOCATIONS\n                            Location of either local file, AWS S3 ARN, or AWS\n                            Secrets Manager ARN containing properties used to\n                            populate the configuration for embedded mode. By\n                            default, the file format is assumed to be in JSON.\n                            Specify multiple times to include multiple locations.\n      --config-merge-strategy {SHALLOW,REPLACE,ADDITIVE,TYPESAFE_REPLACE,TYPESAFE_ADDITIVE}\n                            Merge strategy for merging config files with\n                            mergedeep. Defaults to SHALLOW, which does not require\n                            mergedeep. All other strategies require the mergedeep\n                            python package to be installed.\n      --overwrite_env_during_resolution\n                            Do not overwrite existing environment variables when\n                            resolving them\n      --config-ttl CONFIG_TTL\n                            Number of seconds to cache resolved environment\n                            variables and configuration properties instead of\n                            refreshing them when a process restarts. -1 means to\n                            never refresh. Defaults to -1.\n      --no-fail-fast-config-resolution\n                            Exit immediately if an error occurs resolving the\n                            configuration\n      --resolved-env-var-name-prefix RESOLVED_ENV_VAR_NAME_PREFIX\n                            Required prefix for names of environment variables\n                            that should resolved. The prefix will be removed in\n                            the resolved variable name. Defaults to \'\'.\n      --resolved-env-var-name-suffix RESOLVED_ENV_VAR_NAME_SUFFIX\n                            Required suffix for names of environment variables\n                            that should resolved. The suffix will be removed in\n                            the resolved variable name. Defaults to\n                            \'_FOR_PROC_WRAPPER_TO_RESOLVE\'.\n      --resolved-config-property-name-prefix RESOLVED_CONFIG_PROPERTY_NAME_PREFIX\n                            Required prefix for names of configuration properties\n                            that should resolved. The prefix will be removed in\n                            the resolved property name. Defaults to \'\'.\n      --resolved-config-property-name-suffix RESOLVED_CONFIG_PROPERTY_NAME_SUFFIX\n                            Required suffix for names of configuration properties\n                            that should resolved. The suffix will be removed in\n                            the resolved property name. Defaults to\n                            \'__to_resolve\'.\n      --env-var-name-for-config ENV_VAR_NAME_FOR_CONFIG\n                            The name of the environment variable used to set to\n                            the value of the JSON encoded configuration. Defaults\n                            to not setting any environment variable.\n      --config-property-name-for-env CONFIG_PROPERTY_NAME_FOR_ENV\n                            The name of the configuration property used to set to\n                            the value of the JSON encoded environment. Defaults to\n                            not setting any property.\n\n    rollbar:\n      Rollbar settings\n\n      --rollbar-access-token ROLLBAR_ACCESS_TOKEN\n                            Access token for Rollbar (used to report error when\n                            communicating with API server)\n      --rollbar-retries ROLLBAR_RETRIES\n                            Number of retries per Rollbar request. Defaults to 2.\n      --rollbar-retry-delay ROLLBAR_RETRY_DELAY\n                            Number of seconds to wait before retrying a Rollbar\n                            request. Defaults to 120.\n      --rollbar-timeout ROLLBAR_TIMEOUT\n                            Timeout for contacting Rollbar server, in seconds.\n                            Defaults to 30.\n\nThese environment variables take precedence over command-line arguments:\n\n* PROC_WRAPPER_TASK_NAME\n* PROC_WRAPPER_TASK_UUID\n* PROC_WRAPPER_TASK_EXECUTION_UUID\n* PROC_WRAPPER_AUTO_CREATE_TASK (TRUE or FALSE)\n* PROC_WRAPPER_AUTO_CREATE_TASK_RUN_ENVIRONMENT_NAME\n* PROC_WRAPPER_AUTO_CREATE_TASK_RUN_ENVIRONMENT_UUID\n* PROC_WRAPPER_AUTO_CREATE_TASK_PROPS (JSON encoded property map)\n* PROC_WRAPPER_TASK_IS_PASSIVE (TRUE OR FALSE)\n* PROC_WRAPPER_TASK_IS_SERVICE (TRUE or FALSE)\n* PROC_WRAPPER_EXECUTION_METHOD_PROPS (JSON encoded property map)\n* PROC_WRAPPER_TASK_MAX_CONCURRENCY (set to -1 to indicate no limit)\n* PROC_WRAPPER_PREVENT_OFFLINE_EXECUTION (TRUE or FALSE)\n* PROC_WRAPPER_TASK_VERSION_NUMBER\n* PROC_WRAPPER_TASK_VERSION_TEXT\n* PROC_WRAPPER_TASK_VERSION_SIGNATURE\n* PROC_WRAPPER_TASK_INSTANCE_METADATA (JSON encoded property map)\n* PROC_WRAPPER_LOG_LEVEL (TRACE, DEBUG, INFO, WARNING, ERROR, or CRITICAL)\n* PROC_WRAPPER_LOG_SECRETS (TRUE or FALSE)\n* PROC_WRAPPER_INCLUDE_TIMESTAMPS_IN_LOG (TRUE or FALSE)\n* PROC_WRAPPER_DEPLOYMENT\n* PROC_WRAPPER_API_BASE_URL\n* PROC_WRAPPER_API_KEY\n* PROC_WRAPPER_API_HEARTBEAT_INTERVAL_SECONDS\n* PROC_WRAPPER_API_ERROR_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_API_RESUME_DELAY_SECONDS\n* PROC_WRAPPER_API_TASK_EXECUTION_CREATION_ERROR_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_TASK_EXECUTION_CREATION_CONFLICT_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_TASK_EXECUTION_CREATION_CONFLICT_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_API_FINAL_UPDATE_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_REQUEST_TIMEOUT_SECONDS\n* PROC_WRAPPER_ENV_LOCATIONS (comma-separated list of locations)\n* PROC_WRAPPER_CONFIG_LOCATIONS (comma-separated list of locations)\n* PROC_WRAPPER_OVERWRITE_ENV_WITH_SECRETS (TRUE or FALSE)\n* PROC_WRAPPER_RESOLVE_SECRETS (TRUE or FALSE)\n* PROC_WRAPPER_MAX_CONFIG_RESOLUTION_DEPTH\n* PROC_WRAPPER_MAX_CONFIG_RESOLUTION_ITERATIONS\n* PROC_WRAPPER_CONFIG_TTL_SECONDS\n* PROC_WRAPPER_FAIL_FAST_CONFIG_RESOLUTION (TRUE or FALSE)\n* PROC_WRAPPER_RESOLVABLE_ENV_VAR_NAME_PREFIX\n* PROC_WRAPPER_RESOLVABLE_ENV_VAR_NAME_SUFFIX\n* PROC_WRAPPER_RESOLVABLE_CONFIG_PROPERTY_NAME_PREFIX\n* PROC_WRAPPER_RESOLVABLE_CONFIG_PROPERTY_NAME_SUFFIX\n* PROC_WRAPPER_ENV_VAR_NAME_FOR_CONFIG\n* PROC_WRAPPER_CONFIG_PROPERTY_NAME_FOR_ENV\n* PROC_WRAPPER_SEND_PID (TRUE or FALSE)\n* PROC_WRAPPER_SEND_HOSTNAME (TRUE or FALSE)\n* PROC_WRAPPER_SEND_RUNTIME_METADATA (TRUE or FALSE)\n* PROC_WRAPPER_ROLLBAR_ACCESS_TOKEN\n* PROC_WRAPPER_ROLLBAR_TIMEOUT_SECONDS\n* PROC_WRAPPER_ROLLBAR_RETRIES\n* PROC_WRAPPER_ROLLBAR_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_MAX_CONFLICTING_AGE_SECONDS\n* PROC_WRAPPER_TASK_COMMAND\n* PROC_WRAPPER_SHELL_MODE (TRUE or FALSE)\n* PROC_WRAPPER_STRIP_SHELL_WRAPPING (TRUE or FALSE)\n* PROC_WRAPPER_WORK_DIR\n* PROC_WRAPPER_PROCESS_MAX_RETRIES\n* PROC_WRAPPER_PROCESS_TIMEOUT_SECONDS\n* PROC_WRAPPER_PROCESS_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_PROCESS_CHECK_INTERVAL_SECONDS\n* PROC_WRAPPER_PROCESS_TERMINATION_GRACE_PERIOD_SECONDS\n* PROC_WRAPPER_PROCESS_GROUP_TERMINATION (TRUE or FALSE)\n* PROC_WRAPPER_STATUS_UPDATE_SOCKET_PORT\n* PROC_WRAPPER_STATUS_UPDATE_MESSAGE_MAX_BYTES\n\nWith the exception of the settings for Secret Fetching and Resolution,\nthese environment variables are read after Secret Fetching so that they can\ncome from secret values.\n\nThe command is executed with the same environment that the wrapper script gets,\nexcept that these properties are copied/overridden:\n\n* PROC_WRAPPER_DEPLOYMENT\n* PROC_WRAPPER_API_BASE_URL\n* PROC_WRAPPER_API_KEY\n* PROC_WRAPPER_API_ERROR_TIMEOUT_SECONDS\n* PROC_WRAPPER_API_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_API_RESUME_DELAY_SECONDS\n* PROC_WRAPPER_API_REQUEST_TIMEOUT_SECONDS\n* PROC_WRAPPER_ROLLBAR_ACCESS_TOKEN\n* PROC_WRAPPER_ROLLBAR_TIMEOUT_SECONDS\n* PROC_WRAPPER_ROLLBAR_RETRIES\n* PROC_WRAPPER_ROLLBAR_RETRY_DELAY_SECONDS\n* PROC_WRAPPER_ROLLBAR_RESUME_DELAY_SECONDS\n* PROC_WRAPPER_TASK_EXECUTION_UUID\n* PROC_WRAPPER_TASK_UUID\n* PROC_WRAPPER_TASK_NAME\n* PROC_WRAPPER_TASK_VERSION_NUMBER\n* PROC_WRAPPER_TASK_VERSION_TEXT\n* PROC_WRAPPER_TASK_VERSION_SIGNATURE\n* PROC_WRAPPER_TASK_INSTANCE_METADATA\n* PROC_WRAPPER_SCHEDULE\n* PROC_WRAPPER_PROCESS_TIMEOUT_SECONDS\n* PROC_WRAPPER_TASK_MAX_CONCURRENCY\n* PROC_WRAPPER_PREVENT_OFFLINE_EXECUTION\n* PROC_WRAPPER_PROCESS_TERMINATION_GRACE_PERIOD_SECONDS\n* PROC_WRAPPER_ENABLE_STATUS_UPDATE_LISTENER\n* PROC_WRAPPER_STATUS_UPDATE_SOCKET_PORT\n* PROC_WRAPPER_STATUS_UPDATE_INTERVAL_SECONDS\n* PROC_WRAPPER_STATUS_UPDATE_MESSAGE_MAX_BYTES\n\nWrapped mode is suitable for running in a shell on your own (virtual) machine\nor in a Docker container. It requires multi-process support, as the module\nruns at the same time as the command it wraps.\n\n### Embedded mode\n\nEmbedded mode works for executing python code in the same process.\nYou include the `proc_wrapper` package in your python project\'s\ndependencies. To run a task you want to be monitored:\n\n    from typing import Any, Mapping\n\n    from proc_wrapper import ProcWrapper, ProcWrapperParams\n\n\n    def fun(wrapper: ProcWrapper, cbdata: dict[str, int],\n            config: Mapping[str, Any]) -> int:\n        print(cbdata)\n        return cbdata[\'a\']\n\n    params = ProcWrapperParams()\n    params.auto_create_task = True\n    params.auto_create_task_run_environment_name = \'production\'\n    params.task_name = \'embedded_test\'\n\n    # For example only, in the real world you would use Secret Fetching;\n    # see below.\n    params.api_key = \'YOUR_CLOUDREACTOR_API_KEY\'\n\n    # In an AWS Lambda environment, you can also pass a runtime_context\n    # parameter set the context of your Lambda entrypoint. This will send\n    # details of this execution to CloudReactor so it can be monitored and\n    # managed.\n    proc_wrapper = ProcWrapper(params=params)\n\n    x = proc_wrapper.managed_call(fun, {\'a\': 1, \'b\': 2})\n    # Should print 1\n    print(x)\n\n\nThis is suitable for running in single-threaded environments like\nAWS Lambda, or as part of a larger process that executes\nsub-routines that should be monitored.\n\nCurrently, Tasks running as Lambdas must be marked as\npassive Tasks, as the execution method is Unknown. In the near future,\nCloudReactor will support running and managing Tasks that run as\nLambdas.\n\n## Secret Fetching and Resolution\n\nA common requirement is that deployed code / images do not contain secrets\ninternally which could be decompiled. Instead, programs should fetch secrets\nfrom an external source in a secure manner. If your program runs in AWS, it\ncan make use of AWS\'s roles that have permission to access data in\nSecrets Manager or S3. However, in many scenarios, having your program access\nAWS directly has the following disadvantages:\n\n1) Your program becomes coupled to AWS, so it is difficult to run locally or\nswitch to another infrastructure provider\n2) You need to write code or use a library for each programming language you\nuse, so secret fetching is done in a non-uniform way\n3) Writing code to merge and parse secrets from different sources is tedious\n\nTherefore, proc_wrapper implements Secret Fetching and Resolution to solve\nthese problems so your programs don\'t have to. Both usage modes can fetch secrets from\n[AWS Secrets Manager](https://aws.amazon.com/secrets-manager/),\nAWS S3, or the local filesystem, and optionally extract embedded data\ninto the environment or a configuration dictionary. The environment is used to\npass values to processes run in wrapped mode,\nwhile the configuration dictionary is passed to the callback function in\nembedded mode.\n\nproc_wrapper parses secret location strings that specify the how to resolve\na secret value. Each secret location string has the format:\n\n`[PROVIDER_CODE:]<Provider specific address>[!FORMAT][|JP:<JSON Path expression>]`\n\n### Secret Providers\n\nProviders indicate the raw source of the secret data. The table below lists the\nsupported providers:\n\n| Provider Code \t| Value Prefix              \t| Provider                     \t| Example Address                                             \t| Required libs                                                               \t| Notes                                                         \t|\n|---------------\t|---------------------------\t|------------------------------\t|-------------------------------------------------------------\t|-----------------------------------------------------------------------------\t|---------------------------------------------------------------\t|\n| `AWS_SM`      \t| `arn:aws:secretsmanager:` \t| AWS Secrets Manager          \t| `arn:aws:secretsmanager:us-east-2:1234567890:secret:config` \t| [boto3](https://boto3.amazonaws.com/v1/documentation/api/latest/index.html) \t| Can also include version suffix like `-PPrpY`                 \t|\n| `AWS_S3`      \t| `arn:aws:s3:::`           \t| AWS S3 Object                \t| `arn:aws:s3:::examplebucket/staging/app1/config.json`       \t| [boto3](https://boto3.amazonaws.com/v1/documentation/api/latest/index.html) \t|                                                               \t|\n| `FILE`        \t| `file://`                 \t| Local file                   \t| `file:///home/appuser/app/.env`                                    \t|                                                                             \t| The default provider if no provider is auto-detected          \t|\n| `ENV`         \t|                           \t| The process environment      \t| `SOME_TOKEN`                                                \t|                                                                             \t| The name of another environment variable                      \t|\n| `CONFIG`      \t|                           \t| The configuration dictionary \t| `$.db`                                                      \t| [jsonpath-ng](https://github.com/h2non/jsonpath-ng)                         \t| JSON path expression to extract the data in the configuration \t|\n| `PLAIN`       \t|                           \t| Plaintext                    \t| `{"user": "postgres", "password": "badpassword"}`           \t|                                                                             \t|                                                               \t|\n\nIf you don\'t specify an explicit provider prefix in a secret location\n(e.g. `AWS_SM:`), the provider can be auto-detected from the address portion\nusing the Value Prefix. For example the secret location\n`arn:aws:s3:::examplebucket/staging/app1/config.json` will be auto-detected\nto with the AWS_S3 provider because it starts with `arn:aws:s3:::`.\n\n### Secret Formats\n\nFormats indicate how the raw string data is parsed into a secret value (which may be\na string, number, boolean, dictionary, or array). The table below lists the\nsupported formats:\n\n| Format Code \t| Extensions      \t| MIME types                                                                            \t| Required libs                                        \t| Notes                                            \t|\n|-------------\t|-----------------\t|---------------------------------------------------------------------------------------\t|------------------------------------------------------\t|--------------------------------------------------\t|\n| `dotenv`    \t| `.env`          \t| None                                                                                  \t| [dotenv](https://github.com/theskumar/python-dotenv) \t| Also auto-detected if location includes `.env.`  \t|\n| `json`      \t| `.json`         \t| `application/json`, `text/x-json`                                                     \t|                                               \t|  \t|\n| `yaml`      \t| `.yaml`, `.yml` \t| `application/x-yaml`, `application/yaml`, `text/vnd.yaml`, `text/yaml`, `text/x-yaml` \t| [pyyaml](https://pyyaml.org/)                        \t| `safe_load()` is used for security               \t|\n\nThe format of a secret value can be auto-detected from the extension or by the\nMIME type if available. Otherwise, you may need to an explicit format code\n(e.g. `!yaml`).\n\n#### AWS Secrets Manager / S3 Notes\n\n[boto3](https://boto3.amazonaws.com/v1/documentation/api/latest/index.html)\nis used to fetch secrets. It will try to access to AWS Secrets Manager\nor S3 using environment variables `AWS_ACCESS_KEY_ID` and `AWS_SECRET_ACCESS_KEY`\nif they are set, or use the EC2 instance role, ECS task role, or Lambda execution role\nif available.\n\nFor Secrets Manager, you can also use "partial ARNs"\n(without the hyphened suffix) as keys.\nIn the example above\n\n    arn:aws:secretsmanager:us-east-2:1234567890:secret:config\n\ncould be used to fetch the same secret, provided there are no conflicting secret ARNs.\nThis allows you to get the latest version of the secret.\n\nIf the secret was stored in Secrets Manager as binary, the\ncorresponding value will be set to the Base-64 encoded value.\n\n### Secret Tranformation\n\nFetching secrets can be relatively expensive and it makes sense to group related\nsecrets together. Therefore it is common to store dictionaries (formatted\nas JSON or YAML) as secrets. However, each desired environment variable\nor configuration property may only consist of a fragment of the dictionary.\nFor example, given the JSON-formatted dictionary\n\n    {\n      "username": "postgres",\n      "password": "badpassword"\n    }\n\nyou may want to populate the environment variable `DB_USERNAME` with\n`postgres`.\n\nTo facilitate this, dictionary fragments can be extracted to individual\nenvironment variables using [jsonpath-ng](https://github.com/h2non/jsonpath-ng).\nTo specify that a variable be extracted from a dictionary using\na JSON Path expression, append `|JP:` followed by the JSON Path expression\nto the secret location string. For example, if the AWS Secrets Manager\nARN\n\n    arn:aws:secretsmanager:us-east-2:1234567890:secret:db-PPrpY\n\ncontains the dictionary above, then the secret location string\n\n    arn:aws:secretsmanager:us-east-2:1234567890:secret:db-PPrpY|JP:$.username\n\nwill resolve to `postgres` as desired.\n\nIf you do something similar to get the password from the same JSON value,\nproc_wrapper is smart enough to cache the fetched dictionary, so that the\nraw data is only fetched once.\n\nSince JSON path expressions yield a list of results, the secrets fetcher\nimplements the following rules to transform the list to the final value:\n\n1. If the list of results has a single value, that value is used as the\nfinal value, unless `[*]` is appended to the JSON path expression.\n2. Otherwise, the final value is the list of results\n\n#### Fetching from another environment variable\n\nIn some deployment scenarios, multiple secrets can be injected into a\nsingle environment variable as a JSON encoded object. In that case,\nthe module can extract secrets using the *ENV* secret source. For example,\nyou may have arranged to have the environment variable DB_CONFIG injected\nwith the JSON encoded value:\n\n    { "username": "postgres", "password": "nohackme" }\n\nThen to extract the username to the environment variable DB_USERNAME you\nyou would add the environment variable DB_USER_FOR_PROC_WRAPPER_TO_RESOLVE\nset to\n\n    ENV:DB_CONFIG|JP:$.username\n\n### Secret injection into environment and configuration\n\nNow let\'s use secret location strings to\ninject the values into the environment (for wrapped mode)\nand/or the the configuration dictionary (for embedded mode). proc_wrapper\nsupports two methods of secret injection which can be combined together:\n\n* Top-level fetching\n* Secrets Resolution\n\n### Top-level fetching\n\nTop-level fetching refers to fetching a dictionary that contains multiple secrets\nand populating the environment / configuration dictionary with it.\nTo use top-level fetching, you specify the secret locations\nfrom which you want to fetch the secrets and the corresponding values are\nmerged together into the environment / configuration.\n\nTo use top-level fetching in wrapped mode, populate the\nenvironment variables `PROC_WRAPPER_ENV_LOCATIONS` with a comma-separated\nlist of secret locations, or use the command-line option\n`--env-locations <secret_location>` one or more times. Secret location\nstrings passed in via `PROC_WRAPPER_ENV_LOCATIONS` or `--env-locations`\nwill be parsed as `dotenv` files unless format is auto-detected or\nexplicitly specified.\n\nTo use top-level fetching in embedded mode, set the `ProcWrapperParams` property\n`config_locations` to a list of secret locations. Alternatively, you can set\nthe environment variable `PROC_WRAPPER_CONFIG_LOCATIONS` to a comma-separated\nlist, and this will be picked up automatically. Secret location values\nwill be parsed as JSON unless the format is auto-detected or explicitly\nspecified. The `config` argument\npassed to the your callback function will contain a merged dictionary of all\nfetched and parsed dictionary values. For example:\n\n    def callback(wrapper: ProcWrapper, cbdata: str,\n            config: Dict[str, str]) -> str:\n        return \'super\' + cbdata + config[\'username\']\n\n\n    def main():\n        params = ProcWrapperParams()\n\n        # Optional: you can set an initial configuration dictionary which will\n        # have its values included in the final configuration unless overridden.\n        params.initial_config = {\n          \'log_level\': \'DEBUG\'\n        }\n\n        # You can omit this if you set PROC_WRAPPER_CONFIG_LOCATIONS environment\n        # variable to the same ARN\n        params.config_locations = [\n            \'arn:aws:secretsmanager:us-east-2:1234567890:secret:db-PPrpY\',\n            # More secret locations can be added here, and their values will\n            # be merged\n        ]\n\n        wrapper = ProcWrapper(params=params)\n\n        # Returns \'superduperpostgres\'\n        return wrapper.managed_call(callback, \'duper\')\n\n#### Merging Secrets\n\nTop-level fetching can potentially fetch multiple dictionaries which are\nmerged together in the final environment / configuration dictionary.\nThe default merge strategy (`SHALLOW`) is just to overwrite top-level keys, with later\nsecret locations taking precedence. However, if you include the\n[mergedeep](https://github.com/clarketm/mergedeep) library, you can also\nset the merge strategy to one of:\n\n* `REPLACE`\n* `ADDITIVE`\n* `TYPESAFE_REPLACE`\n* `TYPESAFE_ADDITIVE`\n\nso that nested dictionaries / lists will be merged as well. In wrapped mode,\nthe merge strategy can be set with the `--config-merge-strategy` command-line\nargument or `PROC_WRAPPER_CONFIG_MERGE_STRATEGY` environment variable. In\nembedded mode, the merge strategy can be set in the\n`config_merge_strategy` string property of `ProcWrapperParams`.\n\n### Secret Resolution\n\nSecret Resolution substitutes configuration or environment values that are\nsecret location strings with the computed values of those strings. Compared\nto Secret Fetching, Secret Resolution is more useful when you want more\ncontrol over the names of variables or when you have secret values deep\ninside your configuration.\n\nIn wrapped mode, if you want to set the environment variable `MY_SECRET` with\na value fetched\nfrom AWS Secrets Manager, you would set the environment variable\n`MY_SECRET_FOR_PROC_WRAPPER_TO_RESOLVE` to a secret location string\nwhich is ARN of the secret, for example:\n\n    arn:aws:secretsmanager:us-east-2:1234567890:secret:db-PPrpY\n\n(The `_FOR_PROC_WRAPPER_TO_RESOLVE` suffix of environment variable names is\nremoved during resolution. It can also be configured with the `PROC_WRAPPER_RESOLVABLE_ENV_VAR_NAME_SUFFIX` environment variable.)\n\nIn embedded mode, if you want the final configuration dictionary to look like:\n\n    {\n      "db_username": "postgres",\n      "db_password": "badpassword",\n      ...\n    }\n\n\nThe initial configuration dictionary would look like:\n\n    {\n      "db_username__to_resolve": "arn:aws:secretsmanager:us-east-2:1234567890:secret:db-PPrpY|JP:$.username",\n      "db_password__to_resolve": "arn:aws:secretsmanager:us-east-2:1234567890:secret:db-PPrpY|JP:$.password",\n\n      ...\n    }\n\n(The `__to_resolve` suffix (with 2 underscores!) of keys is removed during\nresolution. It can also be configured with the `resolved_config_property_name_suffix`\nproperty of `ProcWrapperParams`.)\n\nproc_wrapper can also resolve keys in embedded dictionaries, like:\n\n    {\n      "db": {\n        "username__to_resolve": "arn:aws:secretsmanager:us-east-2:1234567890:secret:config-PPrpY|JP:$.username",\n        "password__to_resolve":\n        "arn:aws:secretsmanager:us-east-2:1234567890:secret:config-PPrpY|JP:$.password",\n        ...\n      },\n      ...\n    }\n\nup to a maximum depth that you can control with `ProcWrapperParams.max_config_resolution_depth` (which defaults to 5). That would resolve to\n\n    {\n      "db": {\n        "username": "postgres",\n        "password": "badpassword"\n        ...\n      },\n      ...\n    }\n\nYou can also inject entire dictionaries, like:\n\n    {\n      "db__to_resolve": "arn:aws:secretsmanager:us-east-2:1234567890:secret:config-PPrpY",\n      ...\n    }\n\nwhich would resolve to\n\n    {\n      "db": {\n        "username": "postgres",\n        "password": "badpassword"\n      },\n      ...\n    }\n\nTo enable secret resolution in wrapped mode, set environment variable `PROC_WRAPPER_RESOLVE_SECRETS` to `TRUE`. In embedded mode, secret\nresolution is enabled by default; set the\n`max_config_resolution_iterations` property of `ProcWrapperParams` to `0`\nto disable resolution.\n\nSecret resolution is run multiple times so that if a resolved value contains\na secret location string, it will be resolved on the next pass. By default,\nproc_wrapper limits the maximum number of resolution passes to 3 but you\ncan control this with the environment variable\n`PROC_WRAPPER_MAX_CONFIG_RESOLUTION_ITERATIONS` in embedded mode,\nor by setting the `max_config_resolution_iterations` property of\n`ProcWrapperParams` in wrapped mode.\n\n### Environment Projection\n\nDuring secret fetching and secret resolution, proc_wrapper internally maintains\nthe computed environment as a dictionary which may have embedded lists and\ndictionaries. However, the final environment passed to the process is a flat\ndictionary containing only string values. So proc_wrapper converts\nall top-level values to strings using these rules:\n\n* Lists and dictionaries are converted to their JSON-encoded string value\n* Boolean values are converted to their upper-cased string representation\n(e.g. the string `FALSE` for the boolean value `false`)\n* The `None` value is converted to the empty string\n* All other values are converted using python\'s `str()` function\n\n### Secrets Refreshing\n\nYou can set a Time to Live (TTL) on the duration that secret values are cached.\nCaching helps reduce expensive lookups of secrets and bandwidth usage.\n\nIn wrapped mode, set the TTL of environment variables set from secret locations\nusing the `--config-ttl` command-line argument or\n`PROC_WRAPPER_CONFIG_TTL_SECONDS` environment variable.\nIf the process exits, you have configured the script to retry,\nand the TTL has expired since the last fetch,\nproc_wrapper will re-fetch the secrets\nand resolve them again, for the environment passed to the next invocation of\nyour process.\n\nIn embedded mode, set the TTL of configuration dictionary values set from\nsecret locations by setting the `config_ttl` property of\n`ProcWrapperParams`. If 1) your callback function raises an exception, 2) you have\nconfigured the script to retry; and 3) the TTL has expired since the last fetch,\nproc_wrapper will re-fetch the secrets\nand resolve them again, for the configuration passed to the next invocation of\nthe callback function.\n\n## Status Updates\n\n### Status Updates in Wrapped Mode\n\nWhile your process in running, you can send status updates to\nCloudReactor by using the StatusUpdater class. Status updates are shown in\nthe CloudReactor dashboard and allow you to track the current progress of a\nTask and also how many items are being processed in multiple executions\nover time.\n\nIn wrapped mode, your application code would send updates to the\nproc_wrapper program via UDP port 2373 (configurable with the PROC_WRAPPER_STATUS_UPDATE_PORT environment variable).\nIf your application code is in python, you can use the provided\nStatusUpdater class to do this:\n\n    from proc_wrapper import StatusUpdater\n\n    with StatusUpdater() as updater:\n        updater.send_update(last_status_message=\'Starting ...\')\n        success_count = 0\n\n        for i in range(100):\n            try:\n                do_work()\n                success_count += 1\n                updater.send_update(success_count=success_count)\n            except Exception:\n                failed_count += 1\n                updater.send_update(failed_count=failed_count)\n\n        updater.send_update(last_status_message=\'Finished!\')\n\n### Status Updates in Embedded Mode\n\nIn embedded mode, your callback in python code can use the wrapper instance to\nsend updates:\n\n    from typing import Any, Mapping\n\n    import proc_wrapper\n    from proc_wrapper import ProcWrapper\n\n    def fun(wrapper: ProcWrapper, cbdata: dict[str, int],\n            config: Mapping[str, Any]) -> int:\n        wrapper.update_status(last_status_message=\'Starting the fun ...\')\n\n        success_count = 0\n        error_count = 0\n        for i in range(100):\n            try:\n                do_work()\n                success_count += 1\n            except Exception:\n\n                error_count += 1\n            wrapper.update_status(success_count=success_count,\n                    error_count=error_count)\n\n        wrapper.update_status(last_status_message=\'The fun is over.\')\n\n        return cbdata[\'a\']\n\n    params = ProcWrapperParams()\n    params.auto_create_task = True\n    params.auto_create_task_run_environment_name = \'production\'\n    params.task_name = \'embedded_test\'\n    params.api_key = \'YOUR_CLOUDREACTOR_API_KEY\'\n\n    proc_wrapper = ProcWrapper(params=params)\n    proc_wrapper.managed_call(fun, {\'a\': 1, \'b\': 2})\n\n## Example Projects\n\nThese projects contain sample Tasks that use this library to report their\nexecution status and results to CloudReactor, when deployed AWS ECS Fargate:\n\n* [cloudreactor-python-ecs-quickstart](https://github.com/CloudReactor/cloudreactor-python-ecs-quickstart)\n* [cloudreactor-java-ecs-quickstart](https://github.com/CloudReactor/cloudreactor-java-ecs-quickstart)\n\n## License\n\nThis software is dual-licensed under open source (MPL 2.0) and commercial\nlicenses. See `LICENSE` for details.\n\n## Contributors ✨\n\nThanks goes to these wonderful people ([emoji key](https://allcontributors.org/docs/en/emoji-key)):\n\n<!-- ALL-CONTRIBUTORS-LIST:START - Do not remove or modify this section -->\n<!-- prettier-ignore-start -->\n<!-- markdownlint-disable -->\n<table>\n  <tr>\n    <td align="center"><a href="https://github.com/jtsay362"><img src="https://avatars0.githubusercontent.com/u/1079646?s=460&v=4?s=80" width="80px;" alt=""/><br /><sub><b>Jeff Tsay</b></sub></a><br /><a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=jtsay362" title="Code">💻</a> <a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=jtsay362" title="Documentation">📖</a> <a href="#infra-jtsay362" title="Infrastructure (Hosting, Build-Tools, etc)">🚇</a> <a href="#maintenance-jtsay362" title="Maintenance">🚧</a></td>\n    <td align="center"><a href="https://github.com/mwaldne"><img src="https://avatars0.githubusercontent.com/u/40419?s=460&u=3a5266861feeb27db392622371ecc57ebca09f32&v=4?s=80" width="80px;" alt=""/><br /><sub><b>Mike Waldner</b></sub></a><br /><a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=mwaldne" title="Code">💻</a></td>\n    <td align="center"><a href="https://browniebroke.com/"><img src="https://avatars.githubusercontent.com/u/861044?v=4?s=80" width="80px;" alt=""/><br /><sub><b>Bruno Alla</b></sub></a><br /><a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=browniebroke" title="Code">💻</a> <a href="#ideas-browniebroke" title="Ideas, Planning, & Feedback">🤔</a> <a href="https://github.com/CloudReactor/cloudreactor-procwrapper/commits?author=browniebroke" title="Documentation">📖</a></td>\n  </tr>\n</table>\n\n<!-- markdownlint-restore -->\n<!-- prettier-ignore-end -->\n\n<!-- ALL-CONTRIBUTORS-LIST:END -->\n\nThis project follows the [all-contributors](https://github.com/all-contributors/all-contributors) specification. Contributions of any kind welcome!\n\n## Credits\n\nThis package was created with\n[Cookiecutter](https://github.com/audreyr/cookiecutter) and the\n[browniebroke/cookiecutter-pypackage](https://github.com/browniebroke/cookiecutter-pypackage)\nproject template.\n',
    'author': 'Jeff Tsay',
    'author_email': 'jeff@cloudreactor.io',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://cloudreactor.io',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
