# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['faapi']

package_data = \
{'': ['*']}

install_requires = \
['beautifulsoup4>=4.10.0,<5.0.0',
 'cfscrape>=2.1.1,<3.0.0',
 'lxml>=4.8.0,<5.0.0',
 'python-dateutil>=2.8.1,<3.0.0',
 'requests>=2.27.1,<3.0.0']

setup_kwargs = {
    'name': 'faapi',
    'version': '3.6.0',
    'description': 'Python module to implement API-like functionality for the FurAffinity.net website.',
    'long_description': '<div align="center">\n\n<img alt="logo" width="400" src="https://raw.githubusercontent.com/FurryCoders/Logos/main/logos/faapi-transparent.png">\n\n# Fur Affinity API\n\nPython library to implement API-like functionality for the [Fur Affinity](https://furaffinity.net) website.\n\n[![](https://img.shields.io/pypi/v/faapi?logo=pypi)](https://pypi.org/project/faapi/)\n[![](https://img.shields.io/pypi/pyversions/faapi?logo=Python)](https://www.python.org)\n\n[![](https://img.shields.io/github/v/tag/FurryCoders/faapi?label=github&sort=date&logo=github&color=blue)](https://github.com/FurryCoders/faapi)\n[![](https://img.shields.io/github/issues/FurryCoders/faapi?logo=github&color=blue)](https://github.com/FurryCoders/FAAPI/issues)\n[![](https://img.shields.io/github/workflow/status/FurryCoders/FAAPI/Test?label=Test&logo=githubactions)](https://github.com/FurryCoders/FAAPI/actions/workflows/test.yml)\n\n</div>\n\n## Requirements\n\nPython 3.9+ is necessary to run this\nlibrary. [<img src="https://python-poetry.org/images/logo-origami.svg" height="12"> Poetry](https://python-poetry.org)\nis used for packaging and dependency management.\n\n## Usage\n\nThe API comprises a main class `FAAPI`, two submission classes `Submission` and `SubmissionPartial`, a journal\nclass `Journal`, and a user class `User`.\n\nOnce `FAAPI` is initialized, its methods can be used to crawl FA and return parsed objects.\n\n```python\nfrom requests.cookies import RequestsCookieJar\nimport faapi\nimport orjson\n\ncookies = RequestsCookieJar()\ncookies.set("a", "38565475-3421-3f21-7f63-3d341339737")\ncookies.set("b", "356f5962-5a60-0922-1c11-65003b70308")\n\napi = faapi.FAAPI(cookies)\nsub, sub_file = api.submission(12345678, get_file=True)\n\nprint(sub.id, sub.title, sub.author, f"{len(sub_file) / 1024:02f}KiB")\n\nwith open(f"{sub.id}.json", "wb") as f:\n    f.write(orjson.dumps(dict(sub)))\n\nwith open(sub.file_url.split("/")[-1], "wb") as f:\n    f.write(sub_file)\n\ngallery, _ = api.gallery("user_name", 1)\nwith open("user_name-gallery.json", "wb") as f:\n    f.write(orjson.dumps(list(map(dict, gallery))))\n```\n\n### robots.txt\n\nAt init, the `FAAPI` object downloads the [robots.txt](https://www.furaffinity.net/robots.txt) file from FA to determine\nthe `Crawl-delay` and `disallow` values set therein. If not set in the robots.txt file, a crawl delay value of 1 second\nis used.\n\nTo respect this value, the default behaviour of the `FAAPI` object is to wait when a get request is made if the last\nrequest was performed more recently then the crawl delay value.\n\nSee under [FAAPI](#faapi) for more details on this behaviour.\n\nFurthermore, any get operation that points to a disallowed path from robots.txt will raise an exception. This check\nshould not be circumvented, and the developer of this library does not take responsibility for violations of the TOS of\nFur Affinity.\n\n### Cookies\n\nTo access protected pages, cookies from an active session are needed. These cookies can be given to the FAAPI object as\na list of dictionaries - each containing a `name` and a `value` field -, or as a `http.cookiejar.CookieJar`\nobject (`requests.cookies.RequestsCookieJar` and other objects inheriting from `CookieJar` are also supported). The\ncookies list should look like the following example:\n\n```python\ncookies = [\n    {"name": "a", "value": "38565475-3421-3f21-7f63-3d3413397537"},\n    {"name": "b", "value": "356f5962-5a60-0922-1c11-65003b703038"},\n]\n```\n\n```python\nfrom requests.cookies import RequestsCookieJar\n\ncookies = RequestsCookieJar()\ncookies.set("a", "38565475-3421-3f21-7f63-3d3413397537")\ncookies.set("b", "356f5962-5a60-0922-1c11-65003b703038")\n```\n\nTo access session cookies, consult the manual of the browser used to log in.\n\n*Note:* it is important to not logout of the session the cookies belong to, otherwise they will no longer work.\n*Note:* as of April 2022 only cookies `a` and `b` are needed.\n\n### User Agent\n\n`FAAPI` attaches a `User-Agent` header to every request. The user agent string is generated at startup in the following\nformat: `faapi/{library version} Python/{python version} {system name}/{system release}`.\n\n## Objects\n\n### FAAPI\n\nThis is the main object that handles all the calls to scrape pages and get submissions.\n\nIt holds 6 different fields:\n\n* `session: CloudflareScraper` `cfscrape` session used for get requests\n* `robots: urllib.robotparser.RobotFileParser` robots.txt handler\n* `user_agent: str` user agent used by the session (property, cannot be set)\n* `crawl_delay: float` crawl delay from robots.txt (property, cannot be set)\n* `last_get: float` time of last get (UNIX time)\n* `raise_for_unauthorized: bool = True` if set to `True`, raises an exception if a request is made and the resulting\n  page is not from a login session\n\n#### Init\n\n`__init__(cookies: list[dict[str, str]] | CookieJar)`\n\nThe class init has a single argument for the cookies. Cookies must be in the format mentioned above\nin [#Cookies](#cookies).\n\n#### Methods & Properties\n\n* `load_cookies(cookies: list[dict[str, str]] | CookieJar)`<br>\n  Load new cookies and create a new session.<br>\n  *Note:* This method removes any cookies currently in use, to update/add single cookies access them from the session\n  object.\n* `handle_delay()`<br>\n  Handles the crawl delay as set in the robots.txt\n* `check_path(path: str, *, raise_for_disallowed: bool = False) -> bool`<br>\n  Checks whether a given path is allowed by the robots.txt. If `raise_for_disallowed` is set to `True`\n  a `DisallowedPath` exception is raised on non-allowed paths.\n* `connection_status -> bool`<br>\n  Returns the status of the connection.\n* `login_status -> bool`<br>\n  Returns the login status.\n* `get(path: str, **params) -> requests.Response`<br>\n  This returns a response object containing the result of the get operation on the given URL with the\n  optional `**params` added to it (url provided is considered as path from \'https://www.furaffinity.net/\').\n* `get_parsed(path: str, *, skip_page_check: bool = False, skip_auth_check: bool = False, **params) -> bs4.BeautifulSoup`<br>\n  Similar to `get()` but returns the parsed HTML from the normal get operation. If the GET request encountered an error,\n  an `HTTPError` exception is raised. If `skip_page_check` is set to `True`, the parsed page is not checked for errors (\n  e.g. non-existing submission). If `skip_auth_check` is set to `True`, the page is not checked for login status.\n* `me() -> User | None`<br>\n  Returns the logged-in user as a `User` object if the cookies are from a login session.\n* `submission(submission_id: int, get_file: bool = False, *, chunk_size: int = None) -> tuple[Submission, bytes | None]`<br>\n  Given a submission ID, it returns a `Submission` object containing the various metadata of the submission itself and\n  a `bytes` object with the submission file if `get_file` is passed as `True`. The optional `chunk_size` argument is\n  used for the request; if left to `None` or set to 0 the download is performed directly without streaming.<br>\n  *Note:* the author `UserPartial` object of the submission does not contain the `join_date` field as it does not appear\n  on submission pages.\n* `submission_file(submission: Submission, *, chunk_size: int = None) -> bytes`<br>\n  Given a submission object, it downloads its file and returns it as a `bytes` object. The optional `chunk_size`\n  argument is used for the request; if left to `None` or set to 0 the download is performed directly without streaming.\n* `journal(journal_id: int) -> Journal`<br>\n  Given a journal ID, it returns a `Journal` object containing the various metadata of the journal.\n* `user(user: str) -> User`<br>\n  Given a username, it returns a `User` object containing information regarding the user.\n* `gallery(user: str, page: int = 1) -> tuple[list[SubmissionPartial], int]`<br>\n  Returns the list of submissions found on a specific gallery page, and the number of the next page. The returned page\n  number is set to 0 if it is the last page.\n* `scraps(user: str, page: int = 1) -> -> tuple[list[SubmissionPartial], int]`<br>\n  Returns the list of submissions found on a specific scraps page, and the number of the next page. The returned page\n  number is set to 0 if it is the last page.\n* `favorites(user: str, page: str = "") -> tuple[list[SubmissionPartial], str]`<br>\n  Downloads a user\'s favorites page. Because of how favorites pages work on FA, the `page` argument (and the one\n  returned) are strings. If the favorites page is the last then an empty string is returned as next page. An empty page\n  value as argument is equivalent to page 1.<br>\n  *Note:* favorites page "numbers" do not follow any scheme and are only generated server-side.\n* `journals(user: str, page: int = 1) -> -> tuple[list[JournalPartial], int]`<br>\n  Returns the list of submissions found on a specific journals page, and the number of the next page. The returned page\n  number is set to 0 if it is the last page.\n* `search(q: str = "", page: int = 0, **params) -> tuple[list[SubmissionPartial], int, int, int, int]`<br>\n  Parses FA search given the query (and optional other params) and returns the submissions found, and the next page\n  together with basic search statistics: the number of the first submission in the page (0-indexed), the number of the\n  last submission in the page (0-indexed), and the total number of submissions found in the search. For example if the\n  last three returned integers are 0, 47 and 437, then the page contains submissions 1 through 48 of a search that has\n  found a total of 437 submissions.<br>\n  *Note:* as of April 2022 the "/search" path is disallowed by Fur Affinity\'s robots.txt.\n* `watchlist_to(self, user: str) -> list[User]`<br>\n  Given a username, returns a list of `User` objects for each user that is watching the given user.\n* `watchlist_by(self, user: str) -> list[User]`<br>\n  Given a username, returns a list of `User` objects for each user that is watched by the given user.\n\n### JournalPartial\n\nThis object contains partial information gathered when parsing a journals folder. It contains the following fields:\n\n* `id: int` journal ID\n* `title: str` journal title\n* `date: datetime` upload date as a [`datetime` object](https://docs.python.org/3/library/datetime.html) (defaults to\n  timestamp 0)\n* `author: UserPartial` journal author (filled only if the journal is parsed from a `bs4.BeautifulSoup` page)\n* `stats: JournalStats` journal statistics stored in a named tuple (`comments` (count))\n* `content: str` journal content in HTML format\n* `mentions: list[str]` the users mentioned in the content (if they were mentioned as links, e.g. `:iconusername:`,\n  `@username`, etc.)\n* `journal_tag: bs4.element.Tag` the journal tag used to parse the object fields\n\n`JournalPartial` objects can be directly cast to a dict object or iterated through.\n\nComparison with `JournalPartial` can be made with either another `JournalPartial` or `Journal` object (the IDs are\ncompared), or an integer (the `JournalPartial.id` value is compared to the given integer).\n\n#### Init\n\n`__init__(journal_tag: bs4.element.Tag = None)`\n\n`Journal` takes one optional parameters: a journal section tag from a journals page.\n\nIf no `journal_tag` is passed then the object fields will remain at their default - empty - value.\n\n#### Methods\n\n* `url -> str`<br>\n  Property method that returns the Fur Affinity URL to the journal (`https://www.furaffinity.net/journal/{id}`).\n* `parse(journal_item: bs4.element.Tag = None)`<br>\n  Parses the stored journal tag for information. If `journal_tag` is passed, it overwrites the existing `journal_tag`\n  value.\n\n### Journal\n\nThis object contains full information gathered when parsing a journal page. It contains the same fields\nas `JournalPartial` with the addition of comments:\n\n* `id: int` journal ID\n* `title: str` journal title\n* `date: datetime` upload date as a [`datetime` object](https://docs.python.org/3/library/datetime.html) (defaults to\n  timestamp 0)\n* `author: UserPartial` journal author (filled only if the journal is parsed from a `bs4.BeautifulSoup` page)\n* `stats: JournalStats` journal statistics stored in a named tuple (`comments` (count))\n* `content: str` journal content in HTML format\n* `mentions: list[str]` the users mentioned in the content (if they were mentioned as links, e.g. `:iconusername:`,\n  `@username`, etc.)\n* `comments: list[Comments]` the comments to the journal, organised in a tree structure\n* `journal_page: bs4.BeautifulSoup` the journal page used to parse the object fields\n\n`Journal` objects can be directly cast to a dict object or iterated through.\n\nComparison with `Journal` can be made with either another `Journal` or `JournalPartial` object (the IDs are compared),\nor an integer (the `Journal.id` value is compared to the given integer).\n\n#### Init\n\n`__init__(journal_page: bs4.BeautifulSoup = None)`\n\n`Journal` takes one optional journal page argument.\n\nIf no `journal_page` is passed then the object fields will remain at their default - empty - value.\n\n#### Methods\n\n* `url -> str`<br>\n  Property method that returns the Fur Affinity URL to the journal (`https://www.furaffinity.net/journal/{id}`).\n* `parse(journal_page: bs4.BeautifulSoup = None)`<br>\n  Parses the stored journal tag for information. If `journal_tag` is passed, it overwrites the existing `journal_tag`\n  value.\n\n### SubmissionPartial\n\nThis lightweight submission object is used to contain the information gathered when parsing gallery, scraps, favorites\nand search pages. It contains only the following fields:\n\n* `id: int` submission ID\n* `title: str` submission title\n* `author: UserPartial` submission author (only the `name` field is filled)\n* `rating: str` submission rating [general, mature, adult]\n* `type: str` submission type [text, image, etc...]\n* `thumbnail_url: str` the URL to the submission thumbnail\n* `submission_figure: bs4.element.Tag` the figure tag used to parse the object fields\n\n`SubmissionPartial` objects can be directly cast to a dict object or iterated through.\n\nComparison with `Submission` can be made with either another `SubmissionPartial` or `Submission` object (the IDs are\ncompared), or an integer (the `Submission.id` value is compared to the given integer).\n\n#### Init\n\n`__init__(submission_figure: bs4.element.Tag = None)`\n\nTo initialise the object, an optional `bs4.element.Tag` object is needed containing the parsed HTML of a submission\nfigure tag.\n\nIf no `submission_figure` is passed then the object fields will remain at their default - empty - value.\n\n#### Methods\n\n* `url -> str`<br>\n  Property method that returns the Fur Affinity URL to the submission (`https://www.furaffinity.net/view/{id}`).\n* `parse(submission_figure: bs4.element.Tag = None)`<br>\n  Parses the stored submission figure tag for information. If `submission_figure` is passed, it overwrites the\n  existing `submission_figure` value.\n\n### Submission\n\nThe main class that parses and holds submission metadata.\n\n* `id: int` submission ID\n* `title: str` submission title\n* `author: UserPartial` submission author (only the `name`, `title`, and `user_icon_url` fields are filled)\n* `date: datetime` upload date as a [`datetime` object](https://docs.python.org/3/library/datetime.html) (defaults to\n  timestamp 0)\n* `tags: list[str]` tags list\n* `category: str` category\n* `species: str` species\n* `gender: str` gender\n* `rating: str` rating\n* `stats: SubmissionStats` submission statistics stored in a named tuple (`views`, `comments` (count), `favorites`)\n* `type: str` submission type (text, image, etc...)\n* `description: str` description in HTML format\n* `mentions: list[str]` the users mentioned in the description (if they were mentioned as links, e.g. `:iconusername:`,\n  `@username`, etc.)\n* `folder: str` the submission folder (gallery or scraps)\n* `file_url: str` the URL to the submission file\n* `thumbnail_url: str` the URL to the submission thumbnail\n* `prev: int` the ID of the previous submission (if any)\n* `next: int` the ID of the next submission (if any)\n* `comments: list[Comments]` the comments to the submission, organised in a tree structure\n* `submission_page: bs4.BeautifulSoup` the submission page used to parse the object fields\n\n`Submission` objects can be directly cast to a dict object and iterated through.\n\nComparison with `Submission` can be made with either another `Submission` or `SubmissionPartial` object (the IDs are\ncompared), or an integer (the `Submission.id` value is compared to the given integer).\n\n#### Init\n\n`__init__(submission_page: bs4.BeautifulSoup = None)`\n\nTo initialise the object, an optional `bs4.BeautifulSoup` object is needed containing the parsed HTML of a submission\npage.\n\nIf no `submission_page` is passed then the object fields will remain at their default - empty - value.\n\n#### Methods\n\n* `url -> str`<br>\n  Property method that returns the Fur Affinity URL to the submission (`https://www.furaffinity.net/view/{id}`).\n* `parse(submission_page: bs4.BeautifulSoup = None)`<br>\n  Parses the stored submission page for metadata. If `submission_page` is passed, it overwrites the\n  existing `submission_page` value.\n\n### Comment\n\nThis object class contains comment metadata and is used to build a tree structure with the comments and their replies.\n\n* `id: int` the comment ID\n* `author: UserPartial` the user who posted the comment\n* `date: datetime` the date the comment was posted\n* `text: str` the comment text in HTML format\n* `replies: list[Comment]` list of replies to the comment\n* `reply_to: Comment | int | None` the parent comment, if the comment is a reply. The variable type is `int` only if the\n  comment is parsed outside the parse method of a `Submission` or `Journal` (e.g. by creating a new comment with a\n  comment tag), and when iterating over the parent object (to avoid infinite recursion errors), be it `Submission`\n  , `Journal` or another `Comment`.\n* `edited: bool` `True` if the comment was edited, `False` otherwise\n* `hidden: bool` `True` if the comment was hidden, `False` otherwise (if the comment was hidden, the author and date\n  fields will default to their empty values)\n* `parent: Submission | Journal | None` the `Submission` or `Journal` object the comments are connected to\n* `comment_tag: bs4.element.Tag` the comment tag used to parse the object fields\n\n`Comment` objects can be directly cast to a dict object and iterated through.\n\nComparison with `Comment` can be made with either another comment (the IDs are compared), or an integer (\nthe `Comment.id` value is compared to the given integer).\n\n*Note:* Because each comment contains the parent `Submission` or `Journal` object (which contains the comment itself)\nand the replied comment object, some iterations may cause infinite recursion errors, for example when using\nthe `copy.deepcopy` function. If such iterations are needed, simply set the `parent` variable to `None` and\nthe `reply_to` variable to `None` or the comment\'s ID (this can be done easily after flattening the comments list\nwith `faapi.comment.flatten_comments`, the comments can then be sorted again with `faapi.comment.sort_comments` which\nwill also restore the `reply_to` values to `Comment` objects).\n*Note:* The `__iter__` method of `Comment` objects automatically removes recursion. The `parent` variable is set\nto `None` and `reply_to` is set to the comment\'s ID.\n\n#### Init\n\n`__init__(self, tag: bs4.element.Tag = None, parent: Submission | Journal = None)`\n\nTo initialise the object, an optional `bs4.element.Tag` object is needed containing the comment tag as taken from a\nsubmission/journal page.\n\nThe optional `parent` argument sets the `parent` variable described above.\n\nIf no `tag` is passed then the object fields will remain at their default - empty - value.\n\n#### Methods\n\n* `url -> str`<br>\n  Property method that returns the Fur Affinity URL to the comment (\n  e.g. `https://www.furaffinity.net/view/12345678#cid:1234567890`). If the `parent` variable is `None`, the property\n  returns an empty string.\n* `parse(tag: bs4.element.Tag = None)`<br>\n  Parses the stored tag for metadata. If `tag` is passed, it overwrites the existing `tag` value.\n\n#### Extra Functions\n\nThese extra functions can be used to operate on a list of comments. They only alter the order and structure, but they do\nnot touch any of the metadata.\n\n* `faapi.comment.sort_comments(comments: list[Comment]) -> list[Comment]`<br/>\n  Sorts a list of comments into a tree structure. Replies are overwritten.\n* `faapi.comment.flatten_comments(comments: list[Comment]) -> list[Comment]`<br/>\n  Flattens a list of comments. Replies are not modified.\n\n#### Comment Tree Graphs\n\nUsing the tree structure generated by the library, it is trivial to build a graph visualisation of the comment tree\nusing the [DOT](https://www.graphviz.org/doc/info/lang.html) language.\n\n```python\nsubmission, _ = api.submission(12345678)\ncomments = faapi.comment.flatten_comments(submission.comments)\nwith open("comments.dot", "w") as f:\n    f.write("digraph {\\n")\n    for comment in [c for c in comments if c.reply_to is None]:\n        f.write(f"    parent -> {comment.id}\\n")\n    for comment in comments:\n        for reply in comment.replies:\n            f.write(f"    {comment.id} -> {reply.id}\\n")\n    f.write("}")\n```\n\n```dot\ndigraph {\n    parent -> 157990848\n    parent -> 157993838\n    parent -> 157997294\n    157990848 -> 158014077\n    158014077 -> 158014816\n    158014816 -> 158093180\n    158093180 -> 158097024\n    157993838 -> 157998464\n    157993838 -> 158014126\n    157997294 -> 158014135\n    158014135 -> 158014470\n    158014135 -> 158030074\n    158014470 -> 158093185\n    158030074 -> 158093199\n}\n```\n\n<img alt="comments tree graph" width="400" src="https://quickchart.io/graphviz?graph=digraph%7B%0Aparent-%3E157990848%0Aparent-%3E157993838%0Aparent-%3E157997294%0A157990848-%3E158014077%0A158014077-%3E158014816%0A158014816-%3E158093180%0A158093180-%3E158097024%0A157993838-%3E157998464%0A157993838-%3E158014126%0A157997294-%3E158014135%0A158014135-%3E158014470%0A158014135-%3E158030074%0A158014470-%3E158093185%0A158030074-%3E158093199%0A%7D">\n\n_The graph above was generated with [quickchart.io](https://quickchart.io/documentation/graphviz-api/)_\n\n### UserPartial\n\nA stripped-down class that holds basic user information. It is used to hold metadata gathered when parsing a submission,\njournal, gallery, scraps, etc.\n\n* `name: str` display name with capital letters and extra characters such as "_"\n* `status: str` user status (~, !, etc.)\n* `title: str` the user title as it appears on their userpage\n* `join_date: datetime` the date the user joined (defaults to timestamp 0)\n* `user_tag: bs4.element.Tag` the user element used to parse information (placeholder, `UserPartial` is filled\n  externally)\n\n`UserPartial` objects can be directly cast to a dict object and iterated through.\n\nComparison with `UserPartial` can be made with either another `UserPartial` or `User` object (the URL names are\ncompared), or a string (the URL name is compared to the given string).\n\n#### Init\n\n`__init__(user_tag: bs4.element.Tag = None)`\n\nTo initialise the object, an optional `bs4.element.Tag` object is needed containing the user element from a user page or\nuser folder.\n\nIf no `user_tag` is passed then the object fields will remain at their default - empty - value.\n\n#### Methods\n\n* `name_url -> str`<br>\n  Property method that returns the URL-safe username\n* `url -> str`<br>\n  Property method that returns the Fur Affinity URL to the user (`https://www.furaffinity.net/user/{name_url}`).\n* `parse(user_page: bs4.BeautifulSoup = None)`<br>\n  Parses the stored user page for metadata. If `user_page` is passed, it overwrites the existing `user_page` value.\n\n### User\n\nThe main class storing all of a user\'s metadata.\n\n* `name: str` display name with capital letters and extra characters such as "_"\n* `status: str` user status (~, !, etc.)\n* `title: str` the user title as it appears on their userpage\n* `join_date: datetime` the date the user joined (defaults to timestamp 0)\n* `profile: str` profile text in HTML format\n* `stats: UserStats` user statistics sorted in a `namedtuple` (`views`, `submissions`, `favorites`, `comments_earned`\n  , `comments_made`, `journals`, `watched_by`, `watching`)\n* `info: dict[str, str]` profile information (e.g. "Accepting Trades", "Accepting Commissions", "Character Species",\n  etc.)\n* `contacts: dict[str, str]` contact links (e.g. Twitter, Steam, etc.)\n* `user_icon_url: str` the URL to the user icon\n* `user_page: bs4.BeautifulSoup` the user page used to parse the object fields\n\n`User` objects can be directly cast to a dict object and iterated through.\n\nComparison with `User` can be made with either another `User` or `UserPartial` object (the URL names are compared), or a\nstring (the URL name is compared to the given string).\n\n#### Init\n\n`__init__(user_page: bs4.BeautifulSoup = None)`\n\nTo initialise the object, an optional `bs4.BeautifulSoup` object is needed containing the parsed HTML of a submission\npage.\n\nIf no `user_page` is passed then the object fields will remain at their default - empty - value.\n\n#### Methods\n\n* `name_url -> str`<br>\n  Property method that returns the URL-safe username\n* `url -> str`<br>\n  Property method that returns the Fur Affinity URL to the user (`https://www.furaffinity.net/user/{name_url}`).\n* `parse(user_page: bs4.BeautifulSoup = None)`<br>\n  Parses the stored user page for metadata. If `user_page` is passed, it overwrites the existing `user_page` value.\n\n## Exceptions\n\nThe following are the exceptions explicitly raised by the FAAPI functions. The exceptions deriving from `ParsingError`\nare chosen depending on the content of the page. Because Fur Affinity doesn\'t use HTTP status codes besides 404, the\npage is checked against a static list of known error messages/page titles in order to determine the specific error to be\nused. If no match is found, then the `ServerError` (if the page has the "Server Error" title) or the more\ngeneral `NoticeMessage` exceptions are used instead. The actual error message parsed from the page is used as argument\nfor the exceptions, so that it can be analysed when caught.\n\n* `DisallowedPath(Exception)` The path is not allowed by the robots.txt.\n* `Unauthorized(Exception)` The user is not logged-in.\n* `ParsingError(Exception)` An error occurred while parsing the page.\n    * `NonePage(ParsingError)` The parsed page is `None`.\n    * `NotFound(ParsingError)` The resource could not be found (general 404 page or non-existing submission, user, or\n      journal).\n    * `NoTitle(ParsingError)` The parsed paged is missing a title.\n    * `DisabledAccount(ParsingError)` The resource belongs to a disabled account.\n    * `ServerError(ParsingError)` The page contains a server error notice.\n    * `NoticeMessage(ParsingError)` A notice of unknown type was found in the page.\n\n## Contributing\n\nAll contributions and suggestions are welcome!\n\nIf you have suggestions for fixes or improvements, you can open an issue with your idea, see [#Issues](#issues) for\ndetails.\n\n## Issues\n\nIf any problem is encountered during usage of the program, an issue can be opened\non [GitHub](https://github.com/FurryCoders/FAAPI/issues).\n\nIssues can also be used to suggest improvements and features.\n\nWhen opening an issue for a problem, please copy the error message and describe the operation in progress when the error\noccurred.\n',
    'author': 'Matteo Campinoti',
    'author_email': 'matteo.campinoti94@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/FurryCoders/FAAPI',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
