"use strict";
const AWSSDK = require("aws-sdk");
const AWS = require("aws-sdk-mock");
const nock = require("nock");
const sinon = require("sinon");
const provider = require("../lib/log-retention-provider");
AWS.setSDK(require.resolve('aws-sdk'));
const eventCommon = {
    ServiceToken: 'token',
    ResponseURL: 'https://localhost',
    StackId: 'stackId',
    RequestId: 'requestId',
    LogicalResourceId: 'logicalResourceId',
    PhysicalResourceId: 'group',
    ResourceType: 'Custom::LogRetention',
};
const context = {
    functionName: 'provider',
};
function createRequest(type) {
    return nock('https://localhost')
        .put('/', (body) => body.Status === type && body.PhysicalResourceId === 'group')
        .reply(200);
}
module.exports = {
    'tearDown'(callback) {
        AWS.restore();
        nock.cleanAll();
        callback();
    },
    async 'create event'(test) {
        const createLogGroupFake = sinon.fake.resolves({});
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group',
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: 'group',
            retentionInDays: 30,
        });
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: '/aws/lambda/provider',
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: '/aws/lambda/provider',
            retentionInDays: 1,
        });
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'update event with new log retention'(test) {
        const error = new Error();
        error.code = 'ResourceAlreadyExistsException';
        const createLogGroupFake = sinon.fake.rejects(error);
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Update',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '365',
                LogGroupName: 'group',
            },
            OldResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
                RetentionInDays: '30',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group',
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: 'group',
            retentionInDays: 365,
        });
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'update event with log retention undefined'(test) {
        const error = new Error();
        error.code = 'ResourceAlreadyExistsException';
        const createLogGroupFake = sinon.fake.rejects(error);
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Update',
            PhysicalResourceId: 'group',
            ResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
            },
            OldResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
                RetentionInDays: '365',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group',
        });
        sinon.assert.calledWith(deleteRetentionPolicyFake, {
            logGroupName: 'group',
        });
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'delete event'(test) {
        const createLogGroupFake = sinon.fake.resolves({});
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Delete',
            PhysicalResourceId: 'group',
            ResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.notCalled(createLogGroupFake);
        sinon.assert.notCalled(putRetentionPolicyFake);
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'responds with FAILED on error'(test) {
        const createLogGroupFake = sinon.fake.rejects(new Error('UnkownError'));
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group',
            },
        };
        const request = createRequest('FAILED');
        await provider.handler(event, context);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'does not fail when operations on provider log group fail'(test) {
        const createLogGroupFake = (params) => {
            if (params.logGroupName === '/aws/lambda/provider') {
                return Promise.reject(new Error('OperationAbortedException'));
            }
            return Promise.resolve({});
        };
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group',
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'response data contains the log group name'(test) {
        AWS.mock('CloudWatchLogs', 'createLogGroup', sinon.fake.resolves({}));
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', sinon.fake.resolves({}));
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', sinon.fake.resolves({}));
        const event = {
            ...eventCommon,
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group',
            },
        };
        async function withOperation(operation) {
            const request = nock('https://localhost')
                .put('/', (body) => { var _a; return ((_a = body.Data) === null || _a === void 0 ? void 0 : _a.LogGroupName) === 'group'; })
                .reply(200);
            const opEvent = { ...event, RequestType: operation };
            await provider.handler(opEvent, context);
            test.equal(request.isDone(), true);
        }
        await withOperation('Create');
        await withOperation('Update');
        await withOperation('Delete');
        test.done();
    },
    async 'custom log retention retry options'(test) {
        AWS.mock('CloudWatchLogs', 'createLogGroup', sinon.fake.resolves({}));
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', sinon.fake.resolves({}));
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', sinon.fake.resolves({}));
        const event = {
            ...eventCommon,
            RequestType: 'Create',
            ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group',
                SdkRetry: {
                    maxRetries: '5',
                    base: '300',
                },
            },
        };
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(AWSSDK.CloudWatchLogs, {
            apiVersion: '2014-03-28',
            maxRetries: 5,
            retryOptions: {
                base: 300,
            },
        });
        test.equal(request.isDone(), true);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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