"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Runtime = exports.RuntimeFamily = void 0;
const core_1 = require("@aws-cdk/core");
var RuntimeFamily;
(function (RuntimeFamily) {
    RuntimeFamily[RuntimeFamily["NODEJS"] = 0] = "NODEJS";
    RuntimeFamily[RuntimeFamily["JAVA"] = 1] = "JAVA";
    RuntimeFamily[RuntimeFamily["PYTHON"] = 2] = "PYTHON";
    RuntimeFamily[RuntimeFamily["DOTNET_CORE"] = 3] = "DOTNET_CORE";
    RuntimeFamily[RuntimeFamily["GO"] = 4] = "GO";
    RuntimeFamily[RuntimeFamily["RUBY"] = 5] = "RUBY";
    RuntimeFamily[RuntimeFamily["OTHER"] = 6] = "OTHER";
})(RuntimeFamily = exports.RuntimeFamily || (exports.RuntimeFamily = {}));
/**
 * Lambda function runtime environment.
 *
 * If you need to use a runtime name that doesn't exist as a static member, you
 * can instantiate a `Runtime` object, e.g: `new Runtime('nodejs99.99')`.
 */
class Runtime {
    constructor(name, family, props = {}) {
        this.name = name;
        this.supportsInlineCode = !!props.supportsInlineCode;
        this.family = family;
        this.bundlingDockerImage = core_1.BundlingDockerImage.fromRegistry(`amazon/aws-sam-cli-build-image-${name}`);
        Runtime.ALL.push(this);
    }
    toString() {
        return this.name;
    }
    runtimeEquals(other) {
        return other.name === this.name &&
            other.family === this.family &&
            other.supportsInlineCode === this.supportsInlineCode;
    }
}
exports.Runtime = Runtime;
/** A list of all known `Runtime`'s. */
Runtime.ALL = new Array();
/**
 * The NodeJS runtime (nodejs)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS = new Runtime('nodejs', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 4.3 runtime (nodejs4.3)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_4_3 = new Runtime('nodejs4.3', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 6.10 runtime (nodejs6.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_6_10 = new Runtime('nodejs6.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 8.10 runtime (nodejs8.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_8_10 = new Runtime('nodejs8.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 10.x runtime (nodejs10.x)
 */
Runtime.NODEJS_10_X = new Runtime('nodejs10.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 12.x runtime (nodejs12.x)
 */
Runtime.NODEJS_12_X = new Runtime('nodejs12.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The Python 2.7 runtime (python2.7)
 */
Runtime.PYTHON_2_7 = new Runtime('python2.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.6 runtime (python3.6)
 */
Runtime.PYTHON_3_6 = new Runtime('python3.6', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.7 runtime (python3.7)
 */
Runtime.PYTHON_3_7 = new Runtime('python3.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.8 runtime (python3.8)
 */
Runtime.PYTHON_3_8 = new Runtime('python3.8', RuntimeFamily.PYTHON);
/**
 * The Java 8 runtime (java8)
 */
Runtime.JAVA_8 = new Runtime('java8', RuntimeFamily.JAVA);
/**
 * The Java 11 runtime (java11)
 */
Runtime.JAVA_11 = new Runtime('java11', RuntimeFamily.JAVA);
/**
 * The .NET Core 1.0 runtime (dotnetcore1.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_1 = new Runtime('dotnetcore1.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.0 runtime (dotnetcore2.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_2 = new Runtime('dotnetcore2.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.1 runtime (dotnetcore2.1)
 */
Runtime.DOTNET_CORE_2_1 = new Runtime('dotnetcore2.1', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 3.1 runtime (dotnetcore3.1)
 */
Runtime.DOTNET_CORE_3_1 = new Runtime('dotnetcore3.1', RuntimeFamily.DOTNET_CORE);
/**
 * The Go 1.x runtime (go1.x)
 */
Runtime.GO_1_X = new Runtime('go1.x', RuntimeFamily.GO);
/**
 * The Ruby 2.5 runtime (ruby2.5)
 */
Runtime.RUBY_2_5 = new Runtime('ruby2.5', RuntimeFamily.RUBY);
/**
 * The Ruby 2.7 runtime (ruby2.7)
 */
Runtime.RUBY_2_7 = new Runtime('ruby2.7', RuntimeFamily.RUBY);
/**
 * The custom provided runtime (provided)
 */
Runtime.PROVIDED = new Runtime('provided', RuntimeFamily.OTHER);
//# sourceMappingURL=data:application/json;base64,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