"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaPowertoolsLayer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
/**
 * Defines a new Lambda Layer with Powertools for python library.
 */
class LambdaPowertoolsLayer extends aws_cdk_lib_1.aws_lambda.LayerVersion {
    constructor(scope, id, props) {
        const runtimeFamily = props?.runtimeFamily ?? aws_cdk_lib_1.aws_lambda.RuntimeFamily.PYTHON;
        const languageName = getLanguageNameFromRuntimeFamily(runtimeFamily);
        const dockerFilePath = path.join(__dirname, `../layer/${languageName}`);
        const compatibleArchitectures = props?.compatibleArchitectures ?? [aws_cdk_lib_1.aws_lambda.Architecture.X86_64];
        const compatibleArchitecturesDescription = compatibleArchitectures.map((arch) => arch.name).join(', ');
        console.log(`path ${dockerFilePath}`);
        super(scope, id, {
            code: aws_cdk_lib_1.aws_lambda.Code.fromDockerBuild(dockerFilePath, {
                buildArgs: {
                    PACKAGE_SUFFIX: LambdaPowertoolsLayer.constructBuildArgs(runtimeFamily, props?.includeExtras, props?.version),
                },
                // supports cross-platform docker build
                platform: getDockerPlatformNameFromArchitectures(compatibleArchitectures),
            }),
            layerVersionName: props?.layerVersionName ? props?.layerVersionName : undefined,
            license: 'MIT-0',
            compatibleRuntimes: getRuntimesFromRuntimeFamily(runtimeFamily),
            compatibleArchitectures,
            description: `Lambda Powertools for ${languageName} [${compatibleArchitecturesDescription}]${props?.includeExtras ? ' with extra dependencies' : ''} ${props?.version ? `version ${props?.version}` : 'latest version'}`.trim(),
        });
    }
    /**
     * creates build argument for the Dockerfile.
     * There are multiple combinations between version and extras package that results in different suffix for the installation.
     * With and without version, with and without extras flag.
     * We construct one suffix here because it is easier to do in code than inside the Dockerfile with bash commands.
     * For example, if we set `includeExtras=true` and `version=1.22.0` we get '[all]==1.22.0'.
     *
     */
    static constructBuildArgs(runtimeFamily, includeExtras, version) {
        let suffix = '';
        switch (runtimeFamily) {
            case aws_cdk_lib_1.aws_lambda.RuntimeFamily.PYTHON:
                if (includeExtras) {
                    suffix = '[all]';
                }
                if (version) {
                    if (version.startsWith('git')) {
                        suffix = `${suffix} @ ${version}`;
                    }
                    else {
                        suffix = `${suffix}==${version}`;
                    }
                }
                break;
            case aws_cdk_lib_1.aws_lambda.RuntimeFamily.NODEJS:
                if (version) {
                    suffix = `@${version}`;
                }
                break;
            default:
                break;
        }
        return suffix;
    }
}
exports.LambdaPowertoolsLayer = LambdaPowertoolsLayer;
_a = JSII_RTTI_SYMBOL_1;
LambdaPowertoolsLayer[_a] = { fqn: "cdk-aws-lambda-powertools-layer.LambdaPowertoolsLayer", version: "3.3.0" };
function getRuntimesFromRuntimeFamily(runtimeFamily) {
    switch (runtimeFamily) {
        case aws_cdk_lib_1.aws_lambda.RuntimeFamily.PYTHON:
            return [
                aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_7,
                aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_8,
                aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            ];
        case aws_cdk_lib_1.aws_lambda.RuntimeFamily.NODEJS:
            return [
                aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_12_X,
                aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_14_X,
                aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_16_X,
                aws_cdk_lib_1.aws_lambda.Runtime.NODEJS_18_X,
            ];
        default:
            return [];
    }
}
function getLanguageNameFromRuntimeFamily(runtimeFamily) {
    switch (runtimeFamily) {
        case aws_cdk_lib_1.aws_lambda.RuntimeFamily.PYTHON:
            return 'Python';
        case aws_cdk_lib_1.aws_lambda.RuntimeFamily.NODEJS:
            return 'TypeScript';
        default:
            return 'Unknown';
    }
}
// Docker expects a single string for the --platform option.
// getDockerPlatformNameFromArchitectures converts the Architecture enum to a string.
function getDockerPlatformNameFromArchitectures(architectures) {
    if (architectures.length == 1) {
        return architectures[0].dockerPlatform;
    }
    else {
        // if we have multiple architectures, we default to x86_64, hoping for the
        // layer not to have any architecture specific code or at least contain
        // binary code for all architectures
        return aws_lambda_1.Architecture.X86_64.dockerPlatform;
    }
}
//# sourceMappingURL=data:application/json;base64,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