"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Calendar = exports.CalendarSourceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const calendar_setup_function_1 = require("./calendar-setup-function");
/**
 * The source types for the calendar file.
 */
var CalendarSourceType;
(function (CalendarSourceType) {
    /**
     * The calendar source is an S3 Bucket.
     */
    CalendarSourceType["S3_OBJECT"] = "s3Object";
    /**
     * The calendar source is a local path.
     */
    CalendarSourceType["PATH"] = "path";
})(CalendarSourceType = exports.CalendarSourceType || (exports.CalendarSourceType = {}));
/**
 * The calendar for determining if pipeline stage should be open or closed.
 */
class Calendar {
    constructor() { }
    /**
     * Creates a calendar from a local file.
     */
    static path(options) {
        return new (class extends Calendar {
            _bind(scope) {
                const calendarBody = fs.readFileSync(path.join(options.calendarPath, options.calendarName), { encoding: 'utf-8' });
                const calendar = new CustomResourceCalendar(scope, {
                    sourceType: CalendarSourceType.PATH,
                    calendarBody,
                    calendarName: options.calendarName,
                });
                this.calendarArn = calendar.calendarArn;
                this.calendarName = calendar.calendarName;
                return calendar;
            }
        })();
    }
    /**
     * Creates a calendar from a S3 bucket.
     */
    static s3Location(options) {
        return new (class extends Calendar {
            _bind(scope) {
                const calendar = new CustomResourceCalendar(scope, {
                    sourceType: CalendarSourceType.S3_OBJECT,
                    bucketName: options.bucket.bucketName,
                    calendarName: options.calendarName,
                    roleArn: options.role?.roleArn,
                });
                this.calendarArn = calendar.calendarArn;
                this.calendarName = calendar.calendarName;
                return calendar;
            }
        })();
    }
}
exports.Calendar = Calendar;
_a = JSII_RTTI_SYMBOL_1;
Calendar[_a] = { fqn: "@cdklabs/cdk-codepipeline-extensions.Calendar", version: "0.0.14" };
/**
 * The custom resource for getting the calendar and uploading it to SSM.
 */
class CustomResourceCalendar extends Calendar {
    constructor(scope, options) {
        super();
        this.calendarName = options.calendarName;
        this.calendarArn = aws_cdk_lib_1.Arn.format({
            service: 'ssm',
            resource: 'document',
            resourceName: options.calendarName,
        }, aws_cdk_lib_1.Stack.of(scope));
        const onEvent = new calendar_setup_function_1.CalendarSetupFunction(scope, 'OnEventHandler');
        onEvent.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ssm:CreateDocument',
                'ssm:UpdateDocument',
                'ssm:DeleteDocument',
            ],
            resources: [this.calendarArn],
        }));
        const provider = new custom_resources_1.Provider(scope, 'Provider', {
            onEventHandler: onEvent,
        });
        new aws_cdk_lib_1.CustomResource(scope, 'SSMCalendarCustomResource', {
            serviceToken: provider.serviceToken,
            properties: {
                sourceType: options.sourceType,
                calendarBody: options.calendarBody,
                bucketName: options.bucketName,
                calendarName: options.calendarName,
                roleArn: options.roleArn,
            },
        });
    }
    _bind() { }
}
//# sourceMappingURL=data:application/json;base64,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