# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['spherical_functions',
 'spherical_functions.SWSH',
 'spherical_functions.SWSH_grids',
 'spherical_functions.SWSH_modes',
 'spherical_functions.WignerD',
 'spherical_functions.recursions']

package_data = \
{'': ['*']}

install_requires = \
['numba>=0.55',
 'numpy-quaternion>=2022',
 'numpy>=1.20',
 'scipy>=1.0,<2.0',
 'spinsfast>=2022,<2023']

setup_kwargs = {
    'name': 'spherical-functions',
    'version': '2022.4.1',
    'description': 'Python/numba implementation of Wigner D Matrices, spin-weighted spherical harmonics, and associated functions',
    'long_description': '# Spherical Functions <a href="https://travis-ci.org/moble/spherical_functions"><img align="right" hspace="3" alt="Status of automatic build and test suite" src="https://travis-ci.org/moble/spherical_functions.svg?branch=master"></a> <a href="https://github.com/moble/spherical_functions/blob/master/LICENSE"><img align="right" hspace="3" alt="MIT license" src="https://img.shields.io/github/license/moble/out_of_core_fft.svg"></a>\n\n| NOTE: This package will still be maintained, but *active* development has moved to the [`spherical`](https://github.com/moble/spherical) package.  While this package works well for ℓ (aka ell, L, j, or J) values up to around 25, errors start to build rapidly and turn into NaNs around 30.  The `spherical` package can readily handle values up to at least 1000, with accuracy close to ℓ times machine precision.  —Mike |\n| --- |\n\n\nPython/numba package for evaluating and transforming Wigner\'s 𝔇 matrices,\nWigner\'s 3-j symbols, and spin-weighted (and scalar) spherical harmonics.\nThese functions are evaluated directly in terms of quaternions, as well as in\nthe more standard forms of spherical coordinates and Euler\nangles.<sup>[1](#1-euler-angles-are-awful)</sup>\n\nThe conventions for this package are described in detail on\n[this page](http://moble.github.io/spherical_functions/).\n\n## Dependencies\n\nThe only true requirements for this code are `python` and the python package\n`numpy`, as well as my accompanying\n[`quaternion`](https://github.com/moble/quaternion) package (installation of\nwhich is shown below).\n\nHowever, this package can automatically use\n[`numba`](http://numba.pydata.org/), which uses [LLVM](http://llvm.org/) to\ncompile python code to machine code, accelerating most numerical functions by\nfactors of anywhere from 2 to 2000.  It is *possible* to run the code without\n`numba`, but the most important functions are roughly 10 times slower without\nit.\n\nThe only drawback of `numba` is that it is nontrivial to install on its own.\nFortunately, the best python installer,\n[`anaconda`](http://continuum.io/downloads), makes it trivial.  Just install\nthe main `anaconda` package.\n\nIf you prefer the smaller download size of\n[`miniconda`](http://conda.pydata.org/miniconda.html) (which comes with no\nextras beyond python), you\'ll also have to run this command:\n\n```sh\nconda install pip numpy numba\n```\n\n\n## Installation\n\nAssuming you use `conda` to manage your python installation (like any sane\npython user), you can install this package simply as\n\n```sh\nconda install -c conda-forge spherical_functions\n```\n\nThis should automatically download and install the package\n[`quaternion`](https://github.com/moble/quaternion), on which this package\ndepends.\n\nAlternatively, if you prefer to use `pip` (whether or not you use `conda`), you\ncan also do\n\n```sh\npip install git+git://github.com/moble/quaternion\npip install git+git://github.com/moble/spherical_functions\n```\n\nOr, if you refuse to use `conda`, you might want to install inside your home\ndirectory without root privileges.  (Anaconda does this by default anyway.)\nThis is done by adding `--user` to the above commands:\n\n```sh\npip install --user git+git://github.com/moble/quaternion\npip install --user git+git://github.com/moble/spherical_functions\n```\n\nFinally, there\'s also the fully manual option of just downloading both code\nrepositories, changing to the code directory, and issuing\n\n```sh\npython setup.py install\n```\n\nThis should work regardless of the installation method, as long as you have a\ncompiler hanging around.\n\n\n## Usage\n\nFirst, we show a very simple example of usage with Euler angles, though it\nbreaks my heart to do so:<sup>[1](#euler-angles-are-awful)</sup>\n\n```python\n>>> import spherical_functions as sf\n>>> alpha, beta, gamma = 0.1, 0.2, 0.3\n>>> ell,mp,m = 3,2,1\n>>> sf.Wigner_D_element(alpha, beta, gamma, ell, mp, m)\n\n```\n\nOf course, it\'s always better to use unit quaternions to describe rotations:\n\n```python\n>>> import numpy as np\n>>> import quaternion\n>>> R = np.quaternion(1,2,3,4).normalized()\n>>> ell,mp,m = 3,2,1\n>>> sf.Wigner_D_element(R, ell, mp, m)\n\n```\n\nIf you need to calculate values of the 𝔇<sup>(ℓ)</sup> matrix elements for many\nvalues of (ℓ, m\', m), it is more efficient to do so all at once.  The following\ncalculates all modes for ℓ from 2 to 8 (inclusive):\n\n```python\n>>> indices = np.array([[ell,mp,m] for ell in range(2,9)\n... for mp in range(-ell, ell+1) for m in range(-ell, ell+1)])\n>>> sf.Wigner_D_element(R, indices)\n\n```\n\nFinally, if you really need to put the pedal to the metal, and are willing to\nguarantee that the input arguments are correct, you can use a special hidden\nform of the function:\n\n```python\n>>> sf._Wigner_D_element(R.a, R.b, indices, elements)\n\n```\n\nHere, `R.a` and `R.b` are the two complex parts of the quaternion defined on\n[this page](http://moble.github.io/spherical_functions/) (though the user need\nnot care about that).  The `indices` variable is assumed to be a\ntwo-dimensional array of integers, where the second dimension has size three,\nrepresenting the (ℓ, m\', m) indices.  This avoids certain somewhat slower\npure-python operations involving argument checking, reshaping, etc.  The\n`elements` variable must be a one-dimensional array of complex numbers (can be\nuninitialized), which will be replaced with the corresponding values on return.\nAgain, however, there is no input dimension checking here, so if you give bad\ninputs, behavior could range from silently wrong to exceptions to segmentation\nfaults.  Caveat emptor.\n\n\n## Acknowledgments\n\nI very much appreciate Barry Wardell\'s help in sorting out the relationships\nbetween my conventions and those of other people and software packages\n(especially Mathematica\'s crazy conventions).\n\nThis code is, of course, hosted on github.  Because it is an open-source\nproject, the hosting is free, and all the wonderful features of github are\navailable, including free wiki space and web page hosting, pull requests, a\nnice interface to the git logs, etc.\n\nEvery change in this code is\n[auomatically tested](https://travis-ci.org/moble/spherical_functions) on\n[Travis-CI](https://travis-ci.org/).  This is a free service (for open-source\nprojects like this one), which integrates beautifully with github, detecting\neach commit and automatically re-running the tests.  The code is downloaded and\ninstalled fresh each time, and then tested, on python 2.7, 3.4, and 3.5.  This\nensures that no change I make to the code breaks either installation or any of\nthe features that I have written tests for.\n\nFinally, the code is automatically compiled, and the binaries hosted for\ndownload by `conda` on [anaconda.org](https://anaconda.org/moble/quaternion).\nThis is also a free service for open-source projects like this one.\n\nThe work of creating this code was supported in part by the Sherman Fairchild\nFoundation and by NSF Grants No. PHY-1306125 and AST-1333129.\n\n\n<br/>\n---\n###### <sup>1</sup> Euler angles are awful\n\nEuler angles are pretty much\n[the worst things ever](http://moble.github.io/spherical_functions/#euler-angles)\nand it makes me feel bad even supporting them.  Quaternions are\nfaster, more accurate, basically free of singularities, more\nintuitive, and generally easier to understand.  You can work entirely\nwithout Euler angles (I certainly do).  You absolutely never need\nthem.  But if you\'re so old fashioned that you really can\'t give them\nup, they are fully supported.\n',
    'author': 'Michael Boyle',
    'author_email': 'mob22@cornell.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/moble/spherical_functions',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
