"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'can add a vpn connection to a vpc with a vpn gateway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    asn: 65001,
                    ip: '192.0.2.1',
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::CustomerGateway', {
            BgpAsn: 65001,
            IpAddress: '192.0.2.1',
            Type: 'ipsec.1',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnection', {
            CustomerGatewayId: {
                Ref: 'VpcNetworkVpnConnectionCustomerGateway8B56D9AF',
            },
            Type: 'ipsec.1',
            VpnGatewayId: {
                Ref: 'VpcNetworkVpnGateway501295FA',
            },
            StaticRoutesOnly: false,
        }));
        test.done();
    },
    'with static routing'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                static: {
                    ip: '192.0.2.1',
                    staticRoutes: [
                        '192.168.10.0/24',
                        '192.168.20.0/24',
                    ],
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnection', {
            CustomerGatewayId: {
                Ref: 'VpcNetworkstaticCustomerGatewayAF2651CC',
            },
            Type: 'ipsec.1',
            VpnGatewayId: {
                Ref: 'VpcNetworkVpnGateway501295FA',
            },
            StaticRoutesOnly: true,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnectionRoute', {
            DestinationCidrBlock: '192.168.10.0/24',
            VpnConnectionId: {
                Ref: 'VpcNetworkstaticE33EA98C',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnectionRoute', {
            DestinationCidrBlock: '192.168.20.0/24',
            VpnConnectionId: {
                Ref: 'VpcNetworkstaticE33EA98C',
            },
        }));
        test.done();
    },
    'with tunnel options'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            preSharedKey: 'secretkey1234',
                            tunnelInsideCidr: '169.254.10.0/30',
                        },
                    ],
                },
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnection', {
            CustomerGatewayId: {
                Ref: 'VpcNetworkVpnConnectionCustomerGateway8B56D9AF',
            },
            Type: 'ipsec.1',
            VpnGatewayId: {
                Ref: 'VpcNetworkVpnGateway501295FA',
            },
            StaticRoutesOnly: false,
            VpnTunnelOptionsSpecifications: [
                {
                    PreSharedKey: 'secretkey1234',
                    TunnelInsideCidr: '169.254.10.0/30',
                },
            ],
        }));
        test.done();
    },
    'fails when vpc has no vpn gateway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
        test.throws(() => vpc.addVpnConnection('VpnConnection', {
            asn: 65000,
            ip: '192.0.2.1',
        }), /VPN gateway/);
        test.done();
    },
    'fails when ip is invalid'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.256',
                },
            },
        }), /`ip`.+IPv4/);
        test.done();
    },
    'fails when specifying more than two tunnel options'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            preSharedKey: 'secretkey1234',
                        },
                        {
                            preSharedKey: 'secretkey1234',
                        },
                        {
                            preSharedKey: 'secretkey1234',
                        },
                    ],
                },
            },
        }), /two.+`tunnelOptions`/);
        test.done();
    },
    'fails with duplicate tunnel inside cidr'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            tunnelInsideCidr: '169.254.10.0/30',
                        },
                        {
                            tunnelInsideCidr: '169.254.10.0/30',
                        },
                    ],
                },
            },
        }), /`tunnelInsideCidr`.+both tunnels/);
        test.done();
    },
    'fails when specifying an invalid pre-shared key'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            preSharedKey: '0invalid',
                        },
                    ],
                },
            },
        }), /`preSharedKey`/);
        test.done();
    },
    'fails when specifying a reserved tunnel inside cidr'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            tunnelInsideCidr: '169.254.1.0/30',
                        },
                    ],
                },
            },
        }), /`tunnelInsideCidr`.+reserved/);
        test.done();
    },
    'fails when specifying an invalid tunnel inside cidr'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            tunnelInsideCidr: '169.200.10.0/30',
                        },
                    ],
                },
            },
        }), /`tunnelInsideCidr`.+size/);
        test.done();
    },
    'can use metricTunnelState on a vpn connection'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnGateway: true,
        });
        const vpn = vpc.addVpnConnection('Vpn', {
            ip: '192.0.2.1',
        });
        // THEN
        test.deepEqual(stack.resolve(vpn.metricTunnelState()), {
            dimensions: { VpnId: { Ref: 'VpcNetworkVpnA476C58D' } },
            namespace: 'AWS/VPN',
            metricName: 'TunnelState',
            period: core_1.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
    'can use metricAllTunnelDataOut'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.deepEqual(stack.resolve(lib_1.VpnConnection.metricAllTunnelDataOut()), {
            namespace: 'AWS/VPN',
            metricName: 'TunnelDataOut',
            period: core_1.Duration.minutes(5),
            statistic: 'Sum',
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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