"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const net = require("net");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The VPN connection type.
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    /**
     * The IPsec 1 VPN connection type.
     */
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    /**
     * Dummy member
     * TODO: remove once https://github.com/aws/jsii/issues/231 is fixed
     */
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * Define a VPN Connection
 *
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.vpc.vpnGatewayId) {
            throw new Error('Cannot create a VPN connection when VPC has no VPN gateway.');
        }
        if (!net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    // tslint:disable:max-line-length
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    // tslint:enable:max-line-length
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        // tslint:disable:max-line-length
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                        // tslint:enable:max-line-length
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: props.tunnelOptions,
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * Return the given named metric for all VPN connections in the account/region.
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,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