"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PredefinedMetric = exports.TargetTrackingScalingPolicy = void 0;
const cdk = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
/**
 * @stability stable
 */
class TargetTrackingScalingPolicy extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error('Exactly one of \'customMetric\' or \'predefinedMetric\' must be specified.');
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error('Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.');
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || cdk.Names.uniqueId(this),
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldown && props.scaleInCooldown.toSeconds(),
                scaleOutCooldown: props.scaleOutCooldown && props.scaleOutCooldown.toSeconds(),
                targetValue: props.targetValue,
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    if (c.statistic.startsWith('p')) {
        throw new Error(`Cannot use statistic '${c.statistic}' for Target Tracking: only 'Average', 'Minimum', 'Maximum', 'SampleCount', and 'Sum' are supported.`);
    }
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics.
 *
 * @stability stable
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    PredefinedMetric["DYNAMODB_READ_CAPACITY_UTILIZATION"] = "DynamoDBReadCapacityUtilization";
    PredefinedMetric["DYANMODB_WRITE_CAPACITY_UTILIZATION"] = "DynamoDBWriteCapacityUtilization";
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
    PredefinedMetric["RDS_READER_AVERAGE_CPU_UTILIZATION"] = "RDSReaderAverageCPUUtilization";
    PredefinedMetric["RDS_READER_AVERAGE_DATABASE_CONNECTIONS"] = "RDSReaderAverageDatabaseConnections";
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION"] = "EC2SpotFleetRequestAverageCPUUtilization";
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN"] = "EC2SpotFleetRequestAverageNetworkIn";
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT"] = "EC2SpotFleetRequestAverageNetworkOut";
    PredefinedMetric["SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE"] = "SageMakerVariantInvocationsPerInstance";
    PredefinedMetric["ECS_SERVICE_AVERAGE_CPU_UTILIZATION"] = "ECSServiceAverageCPUUtilization";
    PredefinedMetric["ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION"] = "ECSServiceAverageMemoryUtilization";
    PredefinedMetric["LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION"] = "LambdaProvisionedConcurrencyUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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