# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyseto', 'pyseto.versions']

package_data = \
{'': ['*']}

install_requires = \
['cryptography>=36,<38',
 'iso8601>=1.0.2,<2.0.0',
 'passlib[argon2]>=1.7.4,<2.0.0',
 'pycryptodomex>=3.12.0,<4.0.0']

extras_require = \
{'docs': ['Sphinx[docs]>=4.3.2,<6.0.0',
          'sphinx-autodoc-typehints[docs]==1.12.0',
          'sphinx-rtd-theme[docs]>=1.0.0,<2.0.0']}

setup_kwargs = {
    'name': 'pyseto',
    'version': '1.6.9',
    'description': 'A Python implementation of PASETO/PASERK.',
    'long_description': '# PySETO - A Python implementation of PASETO/PASERK\n\n[![PyPI version](https://badge.fury.io/py/pyseto.svg)](https://badge.fury.io/py/pyseto)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/pyseto)\n[![Documentation Status](https://readthedocs.org/projects/pyseto/badge/?version=latest)](https://pyseto.readthedocs.io/en/latest/?badge=latest)\n![Github CI](https://github.com/dajiaji/pyseto/actions/workflows/python-package.yml/badge.svg)\n[![codecov](https://codecov.io/gh/dajiaji/pyseto/branch/main/graph/badge.svg?token=QN8GXEYEP3)](https://codecov.io/gh/dajiaji/pyseto)\n\n\nPySETO is a [PASETO (Platform-Agnostic SEcurity TOkens)](https://paseto.io/)/[PASERK (Platform-Agnostic Serialized Keys)](https://github.com/paseto-standard/paserk) implementation written in Python\nwhich supports all of the versions ([v1](https://github.com/paseto-standard/paseto-spec/blob/master/docs/01-Protocol-Versions/Version1.md),\n[v2](https://github.com/paseto-standard/paseto-spec/blob/master/docs/01-Protocol-Versions/Version2.md),\n[v3](https://github.com/paseto-standard/paseto-spec/blob/master/docs/01-Protocol-Versions/Version3.md) and\n[v4](https://github.com/paseto-standard/paseto-spec/blob/master/docs/01-Protocol-Versions/Version4.md)) and purposes (`public` and `local`)\nand has passed all of [the official tests](https://github.com/paseto-standard/test-vectors).\n\nYou can install PySETO with pip:\n\n```sh\n$ pip install pyseto\n```\n\nPySETO can be used in ease as follows (in case of `v4.public`):\n\n```py\nimport pyseto\nfrom pyseto import Key\n\nprivate_key_pem = b"-----BEGIN PRIVATE KEY-----\\nMC4CAQAwBQYDK2VwBCIEILTL+0PfTOIQcn2VPkpxMwf6Gbt9n4UEFDjZ4RuUKjd0\\n-----END PRIVATE KEY-----"\npublic_key_pem = b"-----BEGIN PUBLIC KEY-----\\nMCowBQYDK2VwAyEAHrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI=\\n-----END PUBLIC KEY-----"\n\n\n# Create a PASETO token.\nprivate_key = Key.new(version=4, purpose="public", key=private_key_pem)\ntoken = pyseto.encode(private_key, b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\')\n\n# Decode and verify a PASETO token.\npublic_key = Key.new(version=4, purpose="public", key=public_key_pem)\ndecoded = pyseto.decode(public_key, token)\n\nassert token == b\'v4.public.eyJkYXRhIjogInRoaXMgaXMgYSBzaWduZWQgbWVzc2FnZSIsICJleHAiOiAiMjAyMi0wMS0wMVQwMDowMDowMCswMDowMCJ9l1YiKei2FESvHBSGPkn70eFO1hv3tXH0jph1IfZyEfgm3t1DjkYqD5r4aHWZm1eZs_3_bZ9pBQlZGp0DPSdzDg\'\nassert decoded.payload == b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\'\n```\n\nSee following contents or [Documentation](https://pyseto.readthedocs.io/en/stable/) for details.\n\n## Index\n\n- [Installation](#installation)\n- [Supported PASETO Versions](#supported-paseto-versions)\n- [Supported PASERK Types](#supported-paserk-types)\n- [PASETO Usage](#paseto-usage)\n    - [Basic usage: v4.public](#basic-usage-v4public)\n    - [Basic usage: v4.local](#basic-usage-v4local)\n    - [Using serializer/deserializer for payload and footer](#using-serializerdeserializer-for-payload-and-footer)\n    - [Using Paseto class for handling registered claims](#using-paseto-class-for-handling-registered-claims)\n- [PASERK Usage](#paserk-usage)\n    - [Serializing/Deserializing PASERK](#serializingdeserializing-paserk)\n    - [Serializing PASERK ID](#serializing-paserk-id)\n    - [Key Wrapping](#key-wrapping)\n    - [Password-based Key Encryption](#password-based-key-encryption)\n    - [Asymmetric Encryption](#asymmetric-encryption)\n- [API Reference](#api-reference)\n- [Tests](#tests)\n- [Contributing](#contributing)\n\n## Installation\n\nYou can install PySETO with pip:\n\n```sh\n$ pip install pyseto\n```\n\n## Supported PASETO Versions\n\nPySETO supports all of PASETO versions and purposes below:\n\n\n|          |  v4  |  v3  |  v2  |  v1  |\n| ---------| ---- | ---- | ---- | ---- |\n| `local`  |  ✅  |  ✅  |  ✅  |  ✅  |\n| `public` |  ✅  |  ✅  |  ✅  |  ✅  |\n\n\n## Supported PASERK Types\n\nPySETO also supports [PASERK (Platform-Agnostic Serialized Keys)](https://github.com/paseto-standard/paserk).\nCurrently, following PASERK types are supported:\n\n\n|               |  v4  |  v3  |  v2  |  v1  |\n| ------------- | ---- | ---- | ---- | ---- |\n| `lid`         |  ✅  |  ✅  |  ✅  |  ✅  |\n| `sid`         |  ✅  |  ✅  |  ✅  |  ✅  |\n| `pid`         |  ✅  |  ✅  |  ✅  |  ✅  |\n| `local`       |  ✅  |  ✅  |  ✅  |  ✅  |\n| `secret`      |  ✅  |  ✅  |  ✅  |  ✅  |\n| `public`      |  ✅  |  ✅  |  ✅  |  ✅  |\n| `seal`        |  ✅  |      |  ✅  |      |\n| `local-wrap`  |  ✅  |  ✅  |  ✅  |  ✅  |\n| `secret-wrap` |  ✅  |  ✅  |  ✅  |  ✅  |\n| `local-pw`    |  ✅  |  ✅  |  ✅  |  ✅  |\n| `secret-pw`   |  ✅  |  ✅  |  ✅  |  ✅  |\n\n\n## PASETO Usage\n\nBy using this PySETO, you can easily create, decode and verify PASETO tokens. Here are sample codes that handle version 4 PySETO tokens.\n\nPlease refer to [the Documentation](https://pyseto.readthedocs.io/en/stable/) for all usage examples including other versions.\n\n### Basic usage: v4.public\n\n`v4.public` is one of current PASETO versions to be used for asymmetric authentication (public key signatures).\n\n```py\nimport pyseto\nfrom pyseto import Key\n\nprivate_key_pem = b"-----BEGIN PRIVATE KEY-----\\nMC4CAQAwBQYDK2VwBCIEILTL+0PfTOIQcn2VPkpxMwf6Gbt9n4UEFDjZ4RuUKjd0\\n-----END PRIVATE KEY-----"\npublic_key_pem = b"-----BEGIN PUBLIC KEY-----\\nMCowBQYDK2VwAyEAHrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI=\\n-----END PUBLIC KEY-----"\n\nprivate_key = Key.new(version=4, purpose="public", key=private_key_pem)\ntoken = pyseto.encode(private_key, b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\')\n\npublic_key = Key.new(version=4, purpose="public", key=public_key_pem)\ndecoded = pyseto.decode(public_key, token)\n\nassert token == b\'v4.public.eyJkYXRhIjogInRoaXMgaXMgYSBzaWduZWQgbWVzc2FnZSIsICJleHAiOiAiMjAyMi0wMS0wMVQwMDowMDowMCswMDowMCJ9l1YiKei2FESvHBSGPkn70eFO1hv3tXH0jph1IfZyEfgm3t1DjkYqD5r4aHWZm1eZs_3_bZ9pBQlZGp0DPSdzDg\'\nassert decoded.payload == b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\'\n```\n\n### Basic usage: v4.local\n\n`v4.local` is one of current PASETO versions to be used for symmetric authenticated encryption.\n\n```py\nimport pyseto\nfrom pyseto import Key\n\nkey = Key.new(version=4, purpose="local", key=b"our-secret")\ntoken = pyseto.encode(key, b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\')\n\ndecoded = pyseto.decode(key, token)\n\nassert decoded.payload == b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\'\n```\n\n### Using serializer/deserializer for payload and footer\n\nBy using `serializer` and `deserializer`, you can encode/decode a dict-typed payload and footer included in PASETO tokens into an arbitrary format.\nThe following example shows that the payload and the footer in a PASETO token are encoded/decoded as JSON formatted data.\nWhen specifing dict-typed payload, exp parameter can be used to set the expiration time (seconds) of the token.\n\n```py\nimport json\nimport pyseto\nfrom pyseto import Key\n\nprivate_key_pem = b"-----BEGIN PRIVATE KEY-----\\nMC4CAQAwBQYDK2VwBCIEILTL+0PfTOIQcn2VPkpxMwf6Gbt9n4UEFDjZ4RuUKjd0\\n-----END PRIVATE KEY-----"\npublic_key_pem = b"-----BEGIN PUBLIC KEY-----\\nMCowBQYDK2VwAyEAHrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI=\\n-----END PUBLIC KEY-----"\n\nprivate_key = Key.new(version=4, purpose="public", key=private_key_pem)\npublic_key = Key.new(version=4, purpose="public", key=public_key_pem)\ntoken = pyseto.encode(\n    private_key,\n    {"data": "this is a signed message"},\n    footer={"kid": public_key.to_paserk_id()},\n    serializer=json,\n    exp=3600,\n)\n\ndecoded = pyseto.decode(public_key, token, deserializer=json)\n\nassert decoded.payload["data"] == "this is a signed message"\nassert decoded.payload["exp"] == "2021-11-11T00:00:00+00:00"\nassert decoded.footer["kid"] == "k4.pid.yh4-bJYjOYAG6CWy0zsfPmpKylxS7uAWrxqVmBN2KAiJ"\n```\n\n### Using `Paseto` class for handling registered claims\n\nBy using `Paseto` class, you can change the default value of `exp` (the expiration date ot tokens), whether to include an `iat` claim, and other settings.\n\nNote that `pyseto.encode()` and `pyseto.decode()` are aliases to the `encode()` and `decode()` of the global "Paseto" class instance created with the default settings.\n\n```py\nimport json\nimport pyseto\nfrom pyseto import Key, Paseto\n\nprivate_key_pem = b"-----BEGIN PRIVATE KEY-----\\nMC4CAQAwBQYDK2VwBCIEILTL+0PfTOIQcn2VPkpxMwf6Gbt9n4UEFDjZ4RuUKjd0\\n-----END PRIVATE KEY-----"\npublic_key_pem = b"-----BEGIN PUBLIC KEY-----\\nMCowBQYDK2VwAyEAHrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI=\\n-----END PUBLIC KEY-----"\n\nprivate_key = Key.new(version=4, purpose="public", key=private_key_pem)\npaseto = Paseto.new(exp=3600, include_iat=True) # Default values are exp=0(not specified) and including_iat=False\ntoken = paseto.encode(\n    private_key,\n    {"data": "this is a signed message"},\n    serializer=json,\n)\n\npublic_key = Key.new(version=4, purpose="public", key=public_key_pem)\ndecoded = pyseto.decode(public_key, token, deserializer=json)\n\nassert decoded.payload["data"] == "this is a signed message"\nassert decoded.payload["iat"] == "2021-11-11T00:00:00+00:00"\nassert decoded.payload["exp"] == "2021-11-11T01:00:00+00:00"\n```\n\n## PASERK Usage\n\n[PASERK (Platform-Agnostic Serialized Keys)](https://github.com/paseto-standard/paserk) is an extension to PASETO that provides key-wrapping and serialization.\n\n### Serializing/Deserializing PASERK\n\nAs shown in the examples above, the `pyseto.Key` used for encryption and signature can be generated from PASERK or converted to PASERK as follows:\n\n```py\nimport pyseto\nfrom pyseto import Key\n\n# pyseto.Key can be generated from PASERK.\nsymmetric_key = Key.new(version=4, purpose="local", key=b"our-secret")\nprivate_key = Key.from_paserk("k4.secret.tMv7Q99M4hByfZU-SnEzB_oZu32fhQQUONnhG5QqN3Qeudu7vAR8A_1wYE4AcfCYfhayi3VyJcEfAEFdDiCxog")\npublic_key = Key.from_paserk("k4.public.Hrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI")\n\ntoken = pyseto.encode(private_key, b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\')\ndecoded = pyseto.decode(public_key, token)\n\nassert decoded.payload == b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\'\n\n# PASERK can be derived from pyseto.Key.\nassert symmetric_key.to_paserk() == "k4.local.b3VyLXNlY3JldA"\nassert private_key.to_paserk() == "k4.secret.tMv7Q99M4hByfZU-SnEzB_oZu32fhQQUONnhG5QqN3Qeudu7vAR8A_1wYE4AcfCYfhayi3VyJcEfAEFdDiCxog"\nassert public_key.to_paserk() == "k4.public.Hrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI"\n```\n\n### Serializing PASERK ID\n\n`pyseto.Key` can also be converted to PASERK ID as follows:\n\n```py\nimport pyseto\nfrom pyseto import Key\n\n# pyseto.Key can be generated from PASERK.\nsymmetric_key = Key.new(version=4, purpose="local", key=b"our-secret")\nprivate_key = Key.from_paserk("k4.secret.tMv7Q99M4hByfZU-SnEzB_oZu32fhQQUONnhG5QqN3Qeudu7vAR8A_1wYE4AcfCYfhayi3VyJcEfAEFdDiCxog")\npublic_key = Key.from_paserk("k4.public.Hrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI")\n\n# PASERK ID can be derived from pyseto.Key.\nassert symmetric_key.to_paserk_id() == "k4.lid._D6kgTzxgiPGk35gMj9bukgj4En2H94u22wVX9zaoh05"\nassert private_key.to_paserk() == "k4.secret.tMv7Q99M4hByfZU-SnEzB_oZu32fhQQUONnhG5QqN3Qeudu7vAR8A_1wYE4AcfCYfhayi3VyJcEfAEFdDiCxog"\nassert public_key.to_paserk_id() == "k4.pid.yh4-bJYjOYAG6CWy0zsfPmpKylxS7uAWrxqVmBN2KAiJ"\n```\n\n### Key Wrapping\n\nIf you call `to_paserk` with `wrapping_key`, you can get a wrapped (encrypted) PASERK with the wrapping key.\nThe wrapped PASERK can be decrypted by calling `from_paserk` with `wrapping key`.\n\nIn case of `local-wrap.pie`:\n\n```py\nimport pyseto\nfrom pyseto import Key\n\nraw_key = Key.new(version=4, purpose="local", key=b"our-secret")\nwrapping_key = token_bytes(32)\nwpk = raw_key.to_paserk(wrapping_key=wrapping_key)\ntoken = pyseto.encode(raw_key, b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\')\n\nunwrapped_key = Key.from_paserk(wpk, wrapping_key=wrapping_key)\ndecoded = pyseto.decode(unwrapped_key, token)\n\n# assert wpk == "k4.local-wrap.pie.TNKEwC4K1xBcgJ_GiwWAoRlQFE33HJO3oN9DHEZ05pieSCd-W7bgAL64VG9TZ_pBkuNBFHNrfOGHtnfnhYGdbz5-x3CxShhPJxg"\nassert decoded.payload == b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\'\n```\n\nIn case of `secret-wrap.pie`:\n\n```py\nimport pyseto\nfrom pyseto import Key\n\nraw_private_key = Key.from_paserk(\n    "k4.secret.tMv7Q99M4hByfZU-SnEzB_oZu32fhQQUONnhG5QqN3Qeudu7vAR8A_1wYE4AcfCYfhayi3VyJcEfAEFdDiCxog"\n)\nwrapping_key = token_bytes(32)\nwpk = raw_private_key.to_paserk(wrapping_key=wrapping_key)\nunwrapped_private_key = Key.from_paserk(wpk, wrapping_key=wrapping_key)\ntoken = pyseto.encode(unwrapped_private_key, b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\')\n\npublic_key = Key.from_paserk("k4.public.Hrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI")\ndecoded = pyseto.decode(public_key, token)\n\n# assert wpk == "k4.secret-wrap.pie.excv7V4-NaECy5hpji-tkSkMvyjsAgNxA-mGALgdjyvGNyDlTb89bJ35R1e3tILgbMpEW5WXMXzySe2T-sBz-ZAcs1j7rbD3ZWvsBTM6K5N9wWfAxbR4ppCXH_H5__9yY-kBaF2NimyAJyduhOhSmqLm6TTSucpAOakEJOXePW8"\nassert decoded.payload == b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\'\n```\n\n### Password-based Key Encryption\n\nIf you call `to_paserk` with `password`, you can get a wrapped (encrypted) PASERK with the password.\nThe wrapped PASERK can be decrypted by calling `from_paserk` with `passwrod`.\n\nIn case of `local-pw`:\n\n```py\nimport pyseto\nfrom pyseto import Key\n\nraw_key = Key.new(version=4, purpose="local", key=b"our-secret")\ntoken = pyseto.encode(raw_key, b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\')\n\nwpk = raw_key.to_paserk(password="our-secret")\nunwrapped_key = Key.from_paserk(wpk, password="our-secret")\ndecoded = pyseto.decode(unwrapped_key, token)\n\n# assert wpk == "k4.local-pw.HrCs9Pu-2LB0l7jkHB-x2gAAAAAA8AAAAAAAAgAAAAGttW0IHZjQCHJdg-Vc3tqO_GSLR4vzLl-yrKk2I-l8YHj6jWpC0lQB2Z7uzTtVyV1rd_EZQPzHdw5VOtyucP0FkCU"\nassert decoded.payload == b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\'\n```\n\nIn case of `secret-pw`:\n\n```py\nimport pyseto\nfrom pyseto import Key\n\nraw_private_key = Key.from_paserk(\n    "k4.secret.tMv7Q99M4hByfZU-SnEzB_oZu32fhQQUONnhG5QqN3Qeudu7vAR8A_1wYE4AcfCYfhayi3VyJcEfAEFdDiCxog"\n)\nwpk = raw_private_key.to_paserk(password="our-secret")\nunwrapped_private_key = Key.from_paserk(wpk, password="our-secret")\ntoken = pyseto.encode(unwrapped_private_key, b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\')\n\npublic_key = Key.from_paserk(\n    "k4.public.Hrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI"\n)\ndecoded = pyseto.decode(public_key, token)\n\n# assert wpk == "k4.secret-pw.MEMW4K1MaD5nWigCLyEyFAAAAAAA8AAAAAAAAgAAAAFU-tArtryNVjS2n2hCYiM11V6tOyuIog69Bjb0yNZanrLJ3afGclb3kPzQ6IhK8ob9E4QgRdEALGWCizZ0RCPFF_M95IQDfmdYKC0Er656UgKUK4UKG9JlxP4o81UwoJoZYz_D1zTlltipEa5RiNvUtNU8vLKoGSY"\nassert decoded.payload == b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\'\n```\n\n### Asymmetric Encryption\n\nAt this time, PySETO supports asymmetric encryption (key sealing) for `v2` and `v4`.\n\n```py\nimport pyseto\nfrom pyseto import Key\n\nprivate_key_pem = b"-----BEGIN PRIVATE KEY-----\\nMC4CAQAwBQYDK2VuBCIEIFAF7jSCZHFgWvC8hUkXr55Az6Pot2g4zOAUxck0/6x8\\n-----END PRIVATE KEY-----"\npublic_key_pem = b"-----BEGIN PUBLIC KEY-----\\nMCowBQYDK2VuAyEAFv8IXsICYj0paznDK/99GyCsFOIGnfY87ayyNSIvSB4=\\n-----END PUBLIC KEY-----"\n\nraw_key = Key.new(version=4, purpose="local", key=b"our-secret")\ntoken = pyseto.encode(\n    raw_key,\n    b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\',\n)\n\nsealed_key = raw_key.to_paserk(sealing_key=public_key_pem)\nunsealed_key = Key.from_paserk(sealed_key, unsealing_key=private_key_pem)\ndecoded = pyseto.decode(unsealed_key, token)\n\nassert (\n    decoded.payload\n    == b\'{"data": "this is a signed message", "exp": "2022-01-01T00:00:00+00:00"}\'\n)\n```\n\nKey searing for `v1` and `v3` have not been supported yet.\n\n\n## API Reference\n\nSee [Documentation](https://pyseto.readthedocs.io/en/stable/api.html).\n\n## Tests\n\nYou can run tests from the project root after cloning with:\n\n```sh\n$ tox\n```\n\n## Contributing\n\nWe welcome all kind of contributions, filing issues, suggesting new features or sending PRs.\n',
    'author': 'Ajitomi Daisuke',
    'author_email': 'dajiaji@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/dajiaji/pyseto',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
