#!/usr/bin/env python3
# -*- coding:utf-8; mode:python -*-
#
# Copyright 2020 Pradyumna Paranjape
# This file is part of pspman.
#
# pspman is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# pspman is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with pspman.  If not, see <https://www.gnu.org/licenses/>.
#
'''
Configuration

'''

import sys
from pathlib import Path
from typing import Any, Dict, List, Optional, Union

import xdgpspconf
import yaml

from pspman.errors import PathNameError

CONF_DISC = xdgpspconf.ConfDisc('pspman', Path(__file__))
DATA_DISC = xdgpspconf.FsDisc('pspman', 'data', Path(__file__))


class GroupDB():
    '''
    Group database information

    Args:
        **kwargs:

            * data: create object using data as __dict__
            * arg in Attributes: Hard-set attributes
            * rest are ignored

    Attributes:
        grp_path: path to Group
        name: custom-name for group [default: leaf of path]
        exists: whether path exists
        locked: locked by another process?
        clone_type: type of clones (0[default], -1, 1)

            * 1: all projects are installable
            * -1: no project is installable (all are pull-only)
            * 0: heterogenous

    '''
    def __init__(self, **kwargs):
        grp_path = kwargs.get('grp_path')
        if grp_path is None:
            self._grp_path: Optional[str] = None
        else:
            self._grp_path = str(grp_path)
        self.clone_type = int(kwargs.get('clone_type', 0))

        # Infer from parent.__dict__
        if 'data' in kwargs:
            self.merge(kwargs['data'])
            del kwargs['data']

        self.name = str(kwargs.get('name', self.get_name()))

    @property
    def locked(self) -> bool:
        if (self.grp_path / '.proc.lock').exists():
            return True
        return False

    @locked.setter
    def locked(self, *_):
        # Do nothing, shouldn't get hard-set
        return

    @locked.deleter
    def locked(self):
        # Do nothing, shouldn't get hard-set
        return

    @property
    def grp_path(self) -> Path:
        if self._grp_path:
            return Path(self._grp_path).resolve()
        raise ValueError('Value for path not provided')

    @grp_path.setter
    def grp_path(self, value: Path):
        self._grp_path = str(Path(value).resolve())

    @property
    def exists(self) -> bool:
        if self.grp_path is None:
            return False
        return self.grp_path.is_dir()

    @exists.setter
    def exists(self, *_):
        # Do nothing, shouldn't get hard-set
        return

    @exists.deleter
    def exists(self):
        # Do nothing, shouldn't get hard-set
        return

    def __repr__(self) -> str:
        '''
        Representation
        '''
        clone_type = ('heterogenous', 'install', 'pull-only')[self.clone_type]
        locked = ('No', 'Yes')[int(self.locked)]
        return f'''
        name: {self.name}
        path: {self.grp_path}
        exists: {self.exists}
        locked: {locked}
        clone_type: {clone_type}
        '''

    def get_name(self) -> str:
        '''
        Infer name from filepath
        '''
        if hasattr(self, 'name'):
            return self.name
        if self.grp_path is None:
            raise PathNameError()
        return self.grp_path.stem

    def mk_structure(self):
        '''
        Generate a directory structure for the GitGroup
        '''
        self.grp_path.mkdir(parents=True, exist_ok=True)
        dir_struct = ("bin", "etc", "include", "lib", "lib64", "libexec",
                      "programs", "share", "src", "tmp", "usr")
        for workdir in dir_struct:
            (self.grp_path / workdir).mkdir(parents=True, exist_ok=True)

    def merge(self, data: Dict[str, object]) -> None:
        '''
        Update values that are ``None`` type using ``data``.
        Doesn't change set values

        Args:
            data: source for update

        '''
        for key, val in data.items():
            setattr(self, key, self.__dict__.get(key) or val)


class MetaConfig():
    '''
    Meta-Config bearing information about all GroupDBs

    Args:
        kwargs:

            * arg in Attributes: Hard-set attributes
            * rest are ignored

    Attributes:
        meta_db_dirs: registry of all GroupDBs
        config_file: pspman configuration file
        data_dir: pspman default data directory
        opt_in: opted installation methods

    '''
    def __init__(self, **kwargs):
        self.meta_db_dirs: Dict[str, GroupDB] =\
            kwargs.get('meta_db_dirs') or {}
        custom_dir = kwargs.get('config_dir')
        if custom_dir is not None:
            self.config_file = Path(custom_dir) / 'config.yml'
            if not self.config_file.parent.is_dir():
                err_str = ' '.join((f"Configuration directory '{custom_dir}'",
                                    'does not exist. Do you mean',
                                    f'{self.config_file.parent.parent}?'))
                raise FileNotFoundError(err_str)
        else:
            self.config_file = CONF_DISC.safe_config(ext='.yml')[-1]

        self.data_dir = DATA_DISC.get_loc(improper=True,
                                          custom=kwargs.get('data_dir'),
                                          mode='rw')[-1]
        self.opt_in: List[str] = kwargs.get('opt_in') or []
        for group in (kwargs.get('meta_db_dirs') or {}).values():
            self.add(group)

    def __repr__(self) -> str:
        """
        Debug representation of object.
        """
        representation: List[str] = []
        for attr in 'meta_db_dirs', 'config_file', 'data_dir', 'opt_in':
            representation.append(f'{attr}: {getattr(self, attr)}')
        return '\n'.join(representation)

    def add(self, group: Union[GroupDB, dict]):
        '''
        Add (don't overwrite) Group

        Args:
            group: Group (or its __dict__) to be added
        '''
        if isinstance(group, dict):
            group = GroupDB(data=group)
        if group.name in self.meta_db_dirs:
            return
        if group.grp_path == self.data_dir and group.name != 'default':
            # Default prefix
            return
        self.meta_db_dirs[group.name] = group

    def remove(self, name: str):
        '''
        Remove group

        Args:
            name: name of group to remove

        '''
        if name in self.meta_db_dirs:
            del self.meta_db_dirs[name]

    def prune(self):
        '''
        If any group doesn't exist anymore, remove it
        '''
        zombies: List[str] = []
        for name, group in self.meta_db_dirs.items():
            if not group.exists:
                zombies.append(name)
        for name in zombies:
            self.remove(name)

    def load(self, configs: Dict[Path, Dict[str, Any]]) -> bool:
        '''
        Load meta configuration

        Args:
            Heirarchy of loadable configurations

        Returns:
            ``True`` if file was successfully loaded
        '''
        if not configs:
            return False
        group_vars = {}
        for group in configs.values():
            group_vars.update(group)
        for group in group_vars.values():
            self.add(group)
        return True

    def store(self):
        '''
        Refresh a configuration file

        '''
        with open(self.config_file, 'w') as conf_fh:
            for group_name, group_db in self.meta_db_dirs.items():
                yaml.dump({group_name: group_db.__dict__}, conf_fh)


def read_config(config_file: str = None) -> MetaConfig:
    '''
    Read and maintain default configurations

    Args:
        config_file: path to file containing configurations

    Returns:
        Configuration
    '''
    psp_config = CONF_DISC.read_config(
        custom=Path(config_file) if config_file else None)
    config = MetaConfig()
    if config.load(psp_config):
        config.prune()
        return config
    init_msg = [
        'PSPMan is not initialized',
        '',
        "To initialize: run without '# ':",
        "# \033[0;97;40mpspman init\033[m:",
        '',
        "Check help for omitting some installation methods: run without '# ':",
        "# \033[0;97;40mpspman init -h\033[m:",
        '',
    ]
    print("\n    ".join(init_msg), file=sys.stderr)
    return config
