"use strict";
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
function render(sm) {
    return new cdk.Stack().resolve(new stepfunctions.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
function createAppWithMap(mapFactory) {
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'my-stack');
    const map = mapFactory(stack);
    new stepfunctions.StateGraph(map, 'Test Graph');
    return app;
}
module.exports = {
    'State Machine With Map State'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const map = new stepfunctions.Map(stack, 'Map State', {
            maxConcurrency: 1,
            itemsPath: stepfunctions.Data.stringAt('$.inputForMap'),
            parameters: {
                foo: 'foo',
                bar: stepfunctions.Data.stringAt('$.bar')
            }
        });
        map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
        // THEN
        test.deepEqual(render(map), {
            StartAt: 'Map State',
            States: {
                'Map State': {
                    Type: 'Map',
                    End: true,
                    Parameters: { foo: 'foo', bar: '$.bar' },
                    Iterator: {
                        StartAt: 'Pass State',
                        States: {
                            'Pass State': {
                                Type: 'Pass',
                                End: true
                            }
                        }
                    },
                    ItemsPath: '$.inputForMap',
                    MaxConcurrency: 1
                }
            }
        });
        test.done();
    },
    'synth is successful'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: 1,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
            return map;
        });
        app.synth();
        test.done();
    },
    'fails in synthesis if iterator is missing'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: 1,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            return map;
        });
        test.throws(() => {
            app.synth();
        }, /Map state must have a non-empty iterator/, 'A validation was expected');
        test.done();
    },
    'fails in synthesis when maxConcurrency is a float'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: 1.2,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
            return map;
        });
        test.throws(() => {
            app.synth();
        }, /maxConcurrency has to be a positive integer/, 'A validation was expected');
        test.done();
    },
    'fails in synthesis when maxConcurrency is a negative integer'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: -1,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
            return map;
        });
        test.throws(() => {
            app.synth();
        }, /maxConcurrency has to be a positive integer/, 'A validation was expected');
        test.done();
    },
    'fails in synthesis when maxConcurrency is too big to be an integer'(test) {
        const app = createAppWithMap((stack) => {
            const map = new stepfunctions.Map(stack, 'Map State', {
                maxConcurrency: Number.MAX_VALUE,
                itemsPath: stepfunctions.Data.stringAt('$.inputForMap')
            });
            map.iterator(new stepfunctions.Pass(stack, 'Pass State'));
            return map;
        });
        test.throws(() => {
            app.synth();
        }, /maxConcurrency has to be a positive integer/, 'A validation was expected');
        test.done();
    },
    'isPositiveInteger is false with negative number'(test) {
        test.equals(stepfunctions.isPositiveInteger(-1), false, '-1 is not a valid positive integer');
        test.done();
    },
    'isPositiveInteger is false with decimal number'(test) {
        test.equals(stepfunctions.isPositiveInteger(1.2), false, '1.2 is not a valid positive integer');
        test.done();
    },
    'isPositiveInteger is false with a value greater than safe integer '(test) {
        const valueToTest = Number.MAX_SAFE_INTEGER + 1;
        test.equals(stepfunctions.isPositiveInteger(valueToTest), false, `${valueToTest} is not a valid positive integer`);
        test.done();
    },
    'isPositiveInteger is true with 0'(test) {
        test.equals(stepfunctions.isPositiveInteger(0), true, '0 is expected to be a positive integer');
        test.done();
    },
    'isPositiveInteger is true with 10'(test) {
        test.equals(stepfunctions.isPositiveInteger(10), true, '10 is expected to be a positive integer');
        test.done();
    },
    'isPositiveInteger is true with max integer value'(test) {
        test.equals(stepfunctions.isPositiveInteger(Number.MAX_SAFE_INTEGER), true, `${Number.MAX_SAFE_INTEGER} is expected to be a positive integer`);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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