"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chain_1 = require("../chain");
const state_graph_1 = require("../state-graph");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Returns true if the value passed is a positive integer
 * @param value the value ti validate
 */
exports.isPositiveInteger = (value) => {
    const isFloat = Math.floor(value) !== value;
    const isNotPositiveInteger = value < 0 || value > Number.MAX_SAFE_INTEGER;
    return !isFloat && !isNotPositiveInteger;
};
/**
 * Define a Map state in the state machine
 *
 * A Map state can be used to dynamically process elements of an array through sub state machines
 *
 * The Result of a Map state is the transformed array after processing through the iterator state machines.
 */
class Map extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [this];
        this.maxConcurrency = props.maxConcurrency;
        this.itemsPath = props.itemsPath;
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Define iterator state machine in Map
     */
    iterator(iterator) {
        const name = `Map ${this.stateId} Iterator`;
        super.addIterator(new state_graph_1.StateGraph(iterator.startState, name));
        return this;
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.MAP,
            Comment: this.comment,
            ResultPath: state_1.renderJsonPath(this.resultPath),
            ...this.renderNextEnd(),
            ...this.renderInputOutput(),
            ...this.renderRetryCatch(),
            ...this.renderIterator(),
            ...this.renderItemsPath(),
            MaxConcurrency: this.maxConcurrency
        };
    }
    /**
     * Validate this state
     */
    validate() {
        const errors = [];
        if (this.iteration === undefined) {
            errors.push('Map state must have a non-empty iterator');
        }
        if (this.maxConcurrency !== undefined && !exports.isPositiveInteger(this.maxConcurrency)) {
            errors.push('maxConcurrency has to be a positive integer');
        }
        return errors;
    }
    renderItemsPath() {
        return {
            ItemsPath: state_1.renderJsonPath(this.itemsPath)
        };
    }
}
exports.Map = Map;
//# sourceMappingURL=data:application/json;base64,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