"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Choice in the state machine
 *
 * A choice state can be used to make decisions based on the execution
 * state.
 */
class Choice extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [];
    }
    /**
     * If the given condition matches, continue execution with the given state
     */
    when(condition, next) {
        super.addChoice(condition, next.startState);
        return this;
    }
    /**
     * If none of the given conditions match, continue execution with the given state
     *
     * If no conditions match and no otherwise() has been given, an execution
     * error will be raised.
     */
    otherwise(def) {
        super.makeDefault(def.startState);
        return this;
    }
    /**
     * Return a Chain that contains all reachable end states from this Choice
     *
     * Use this to combine all possible choice paths back.
     */
    afterwards(options = {}) {
        const endStates = state_1.State.filterNextables(state_1.State.findReachableEndStates(this, { includeErrorHandlers: options.includeErrorHandlers }));
        if (options.includeOtherwise && this.defaultChoice) {
            throw new Error(`'includeOtherwise' set but Choice state ${this.stateId} already has an 'otherwise' transition`);
        }
        if (options.includeOtherwise) {
            endStates.push(new DefaultAsNext(this));
        }
        return chain_1.Chain.custom(this, endStates, this);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.CHOICE,
            Comment: this.comment,
            ...this.renderInputOutput(),
            ...this.renderChoices(),
        };
    }
}
exports.Choice = Choice;
/**
 * Adapter to make the .otherwise() transition settable through .next()
 */
class DefaultAsNext {
    constructor(choice) {
        this.choice = choice;
    }
    next(state) {
        this.choice.otherwise(state);
        return chain_1.Chain.sequence(this.choice, state);
    }
}
//# sourceMappingURL=data:application/json;base64,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