"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
/**
 * Metrics on the rate limiting performed on state machine execution.
 *
 * These rate limits are shared across all state machines.
 */
class StateTransitionMetric {
    /**
     * Return the given named metric for the service's state transition metrics
     *
     * @default average over 5 minutes
     */
    static metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { ServiceMetric: 'StateTransition' },
            ...props
        });
    }
    /**
     * Metric for the number of available state transitions.
     *
     * @default average over 5 minutes
     */
    static metricProvisionedBucketSize(props) {
        return StateTransitionMetric.metric("ProvisionedBucketSize", props);
    }
    /**
     * Metric for the provisioned steady-state execution rate
     *
     * @default average over 5 minutes
     */
    static metricProvisionedRefillRate(props) {
        return StateTransitionMetric.metric("ProvisionedRefillRate", props);
    }
    /**
     * Metric for the number of available state transitions per second
     *
     * @default average over 5 minutes
     */
    static metricConsumedCapacity(props) {
        return StateTransitionMetric.metric("ConsumedCapacity", props);
    }
    /**
     * Metric for the number of throttled state transitions
     *
     * @default sum over 5 minutes
     */
    static metricThrottledEvents(props) {
        return StateTransitionMetric.metric("ThrottledEvents", { statistic: 'sum', ...props });
    }
}
exports.StateTransitionMetric = StateTransitionMetric;
//# sourceMappingURL=data:application/json;base64,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