"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const chain_1 = require("./chain");
const parallel_1 = require("./states/parallel");
const state_1 = require("./states/state");
/**
 * Base class for reusable state machine fragments
 */
class StateMachineFragment extends cdk.Construct {
    get id() {
        return this.node.id;
    }
    /**
     * Prefix the IDs of all states in this state machine fragment
     *
     * Use this to avoid multiple copies of the state machine all having the
     * same state IDs.
     *
     * @param prefix The prefix to add. Will use construct ID by default.
     */
    prefixStates(prefix) {
        state_1.State.prefixStates(this, prefix || `${this.id}: `);
        return this;
    }
    /**
     * Wrap all states in this state machine fragment up into a single state.
     *
     * This can be used to add retry or error handling onto this state
     * machine fragment.
     *
     * Be aware that this changes the result of the inner state machine
     * to be an array with the result of the state machine in it. Adjust
     * your paths accordingly. For example, change 'outputPath' to
     * '$[0]'.
     */
    toSingleState(options = {}) {
        const stateId = options.stateId || this.id;
        this.prefixStates(options.prefixStates || `${stateId}: `);
        return new parallel_1.Parallel(this, stateId, options).branch(this);
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        return chain_1.Chain.start(this).next(next);
    }
}
exports.StateMachineFragment = StateMachineFragment;
//# sourceMappingURL=data:application/json;base64,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