"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A collection of connected states
 *
 * A StateGraph is used to keep track of all states that are connected (have
 * transitions between them). It does not include the substatemachines in
 * a Parallel's branches: those are their own StateGraphs, but the graphs
 * themselves have a hierarchical relationship as well.
 *
 * By assigning states to a definitive StateGraph, we verify that no state
 * machines are constructed. In particular:
 *
 * - Every state object can only ever be in 1 StateGraph, and not inadvertently
 *   be used in two graphs.
 * - Every stateId must be unique across all states in the entire state
 *   machine.
 *
 * All policy statements in all states in all substatemachines are bubbled so
 * that the top-level StateMachine instantiation can read them all and add
 * them to the IAM Role.
 *
 * You do not need to instantiate this class; it is used internally.
 */
class StateGraph {
    /**
     * @param startState state that gets executed when the state machine is launched
     * @param graphDescription description of the state machine
     */
    constructor(startState, graphDescription) {
        this.startState = startState;
        this.graphDescription = graphDescription;
        /**
         * The accumulated policy statements
         */
        this.policyStatements = new Array();
        /**
         * All states in this graph
         */
        this.allStates = new Set();
        /**
         * A mapping of stateId -> Graph for all states in this graph and subgraphs
         */
        this.allContainedStates = new Map();
        this.allStates.add(startState);
        startState.bindToGraph(this);
    }
    /**
     * Register a state as part of this graph
     *
     * Called by State.bindToGraph().
     */
    registerState(state) {
        this.registerContainedState(state.stateId, this);
        this.allStates.add(state);
    }
    /**
     * Register a Policy Statement used by states in this graph
     */
    registerPolicyStatement(statement) {
        if (this.superGraph) {
            this.superGraph.registerPolicyStatement(statement);
        }
        else {
            this.policyStatements.push(statement);
        }
    }
    /**
     * Register this graph as a child of the given graph
     *
     * Resource changes will be bubbled up to the given graph.
     */
    registerSuperGraph(graph) {
        if (this.superGraph === graph) {
            return;
        }
        if (this.superGraph) {
            throw new Error('Every StateGraph can only be registered into one other StateGraph');
        }
        this.superGraph = graph;
        this.pushContainedStatesUp(graph);
        this.pushPolicyStatementsUp(graph);
    }
    /**
     * Return the Amazon States Language JSON for this graph
     */
    toGraphJson() {
        const states = {};
        for (const state of this.allStates) {
            states[state.stateId] = state.toStateJson();
        }
        return {
            StartAt: this.startState.stateId,
            States: states,
            TimeoutSeconds: this.timeout && this.timeout.toSeconds()
        };
    }
    /**
     * Return a string description of this graph
     */
    toString() {
        const someNodes = Array.from(this.allStates).slice(0, 3).map(x => x.stateId);
        if (this.allStates.size > 3) {
            someNodes.push('...');
        }
        return `${this.graphDescription} (${someNodes.join(', ')})`;
    }
    /**
     * Register a stateId and graph where it was registered
     */
    registerContainedState(stateId, graph) {
        if (this.superGraph) {
            this.superGraph.registerContainedState(stateId, graph);
        }
        else {
            const existingGraph = this.allContainedStates.get(stateId);
            if (existingGraph) {
                throw new Error(`State with name '${stateId}' occurs in both ${graph} and ${existingGraph}. All states must have unique names.`);
            }
            this.allContainedStates.set(stateId, graph);
        }
    }
    /**
     * Push all contained state info up to the given super graph
     */
    pushContainedStatesUp(superGraph) {
        for (const [stateId, graph] of this.allContainedStates) {
            superGraph.registerContainedState(stateId, graph);
        }
    }
    /**
     * Push all policy statements to into the given super graph
     */
    pushPolicyStatementsUp(superGraph) {
        for (const policyStatement of this.policyStatements) {
            superGraph.registerPolicyStatement(policyStatement);
        }
    }
}
exports.StateGraph = StateGraph;
//# sourceMappingURL=data:application/json;base64,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