"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const parallel_1 = require("./states/parallel");
/**
 * A collection of states to chain onto
 *
 * A Chain has a start and zero or more chainable ends. If there are
 * zero ends, calling next() on the Chain will fail.
 */
class Chain {
    constructor(startState, endStates, lastAdded) {
        this.lastAdded = lastAdded;
        this.id = lastAdded.id;
        this.startState = startState;
        this.endStates = endStates;
    }
    /**
     * Begin a new Chain from one chainable
     */
    static start(state) {
        return new Chain(state.startState, state.endStates, state);
    }
    /**
     * Make a Chain with the start from one chain and the ends from another
     */
    static sequence(start, next) {
        return new Chain(start.startState, next.endStates, next);
    }
    /**
     * Make a Chain with specific start and end states, and a last-added Chainable
     */
    static custom(startState, endStates, lastAdded) {
        return new Chain(startState, endStates, lastAdded);
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        if (this.endStates.length === 0) {
            throw new Error(`Cannot add to chain: last state in chain (${this.lastAdded.id}) does not allow it`);
        }
        for (const endState of this.endStates) {
            endState.next(next);
        }
        return new Chain(this.startState, next.endStates, next);
    }
    /**
     * Return a single state that encompasses all states in the chain
     *
     * This can be used to add error handling to a sequence of states.
     *
     * Be aware that this changes the result of the inner state machine
     * to be an array with the result of the state machine in it. Adjust
     * your paths accordingly. For example, change 'outputPath' to
     * '$[0]'.
     */
    toSingleState(id, props = {}) {
        return new parallel_1.Parallel(this.startState, id, props).branch(this);
    }
}
exports.Chain = Chain;
//# sourceMappingURL=data:application/json;base64,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