#
#   Copyright EAVISE
#   Author: Maarten Vandersteegen
#   Author: Tanguy Ophoff
#
import logging
from ._base import DetectionParser, ParserType

__all__ = ['DollarParser']
log = logging.getLogger(__name__)


class DollarParser(DetectionParser):
    """ This parser is designed to parse the text based dollar detections generated by `Piotr Dollar's toolbox <dollar>`_
    and the EAVISE PeopleDetect framework.

    Keyword Args:
        class_label (string, optional): class label name of the detections (this format only supports single object class detections); Default **'?'**
        precision (int, optional): precision to use for the confidence when saving detections with this format; Default **0**

    A text file contains multiple detections over multiple images where each line in the file represents one
    detection bounding box.
    Each line is a comma separated list of values structured as follows:

        <image_id>,<x>,<y>,<w>,<h>,<score>

    =========  ===========
    Name       Description
    =========  ===========
    image_id   image identifier that this annotation belong to (integer)
    x          top left x coordinate of the bounding box in pixels (integer)
    y          top left y coordinate of the bounding box in pixels (integer)
    w          width of the bounding box in pixels (integer)
    h          height of the bounding box in pixels (integer)
    score      relative detection score not limited between boundaries (float)
    =========  ===========

    Example:
        >>> video_000.txt
            20,503,213,20,50,74.8391
            20,540,166,37,91,56.4761
            30,519,186,31,77,51.2428

    Note:
        In most cases, the detection confidence values from this format are unbounded.
        While this is not a big problem and can still be used to generate statistics like PR- or MRFPPI-curves, you might want to normalize these values.
        You could normalize the results between 0-1 by using the following code, but make sure to check the detector that was used to use the correct normalization if necessary.

        >>> df = bb.io.load('anno_dollar', 'filename')
        >>> df.confidence = (df.confidence - df.confidence.min()) / (df.confidence.max() - df.confidence.min())
    """
    parser_type = ParserType.SINGLE_FILE
    serialize_group_separator = '\n'
    extension = '.txt'

    def __init__(self, class_label='?', precision=0):
        super().__init__()
        if class_label == '?':
            log.warning('No class_label kwarg passed, defaulting to "?"')
        self.add_column('class_label', class_label)
        self.precision = precision

    def pre_serialize(self, df):
        if df.class_label.nunique() > 1:
            log.error('This parser is meant for single-class problems and as such does not have the means to store class labels. All objects will be stored as the same class.')

        try:
            df.image = df.image.astype(int)
        except ValueError:
            log.warning('Could not convert image column to integers, using categorical codes')
            df.image = df.image.cat.codes

        return df

    def serialize(self, row):
        return f'{row.image},{round(row.x_top_left)},{round(row.y_top_left)},{round(row.width)},{round(row.height)},{round(row.confidence, self.precision)}'

    def deserialize(self, rawdata, file_id=None):
        for line in rawdata.splitlines():
            elements = line.split(',')
            self.append(
                str(int(elements[0])),
                x_top_left=float(elements[1]),
                y_top_left=float(elements[2]),
                width=float(elements[3]),
                height=float(elements[4]),
                confidence=float(elements[5]),
            )
