"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTrainingJob = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
const utils_1 = require("./private/utils");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 * @stability stable
 */
class SageMakerCreateTrainingJob extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         *
         * @stability stable
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_SageMakerCreateTrainingJobProps(props);
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSize: core_1.Size.gibibytes(10),
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1),
        };
        // check that either algorithm name or image is defined
        if (!props.algorithmSpecification.algorithmName && !props.algorithmSpecification.trainingImage) {
            throw new Error('Must define either an algorithm name or training image URI in the algorithm specification');
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = props.algorithmSpecification.trainingInputMode
            ? props.algorithmSpecification
            : { ...props.algorithmSpecification, trainingInputMode: base_types_1.InputMode.FILE };
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map((config) => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return {
                    ...config,
                    dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
                };
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = props.vpcConfig.subnets ? this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     *
     * @stability stable
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error('Principal not available yet--use the object in a Task first');
        }
        return this._grantPrincipal;
    }
    /**
     * Add the security group to all instances via the launch configuration security groups array.
     *
     * @param securityGroup : The security group to add.
     * @stability stable
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTrainingJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            EnableNetworkIsolation: this.props.enableNetworkIsolation,
            RoleArn: this._role.roleArn,
            ...this.renderAlgorithmSpecification(this.algorithmSpecification),
            ...this.renderInputDataConfig(this.inputDataConfig),
            ...this.renderOutputDataConfig(this.props.outputDataConfig),
            ...this.renderResourceConfig(this.resourceConfig),
            ...this.renderStoppingCondition(this.stoppingCondition),
            ...this.renderHyperparameters(this.props.hyperparameters),
            ...utils_1.renderTags(this.props.tags),
            ...this.renderVpcConfig(this.props.vpcConfig),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {}),
                ...(spec.algorithmName ? { AlgorithmName: spec.algorithmName } : {}),
                ...(spec.metricDefinitions
                    ? { MetricDefinitions: spec.metricDefinitions.map((metric) => ({ Name: metric.name, Regex: metric.regex })) }
                    : {}),
            },
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map((channel) => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType
                            ? { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType }
                            : {}),
                        ...(channel.dataSource.s3DataSource.attributeNames ? { AttributeNames: channel.dataSource.s3DataSource.attributeNames } : {}),
                    },
                },
                ...(channel.compressionType ? { CompressionType: channel.compressionType } : {}),
                ...(channel.contentType ? { ContentType: channel.contentType } : {}),
                ...(channel.inputMode ? { InputMode: channel.inputMode } : {}),
                ...(channel.recordWrapperType ? { RecordWrapperType: channel.recordWrapperType } : {}),
            })),
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey ? { KmsKeyId: config.encryptionKey.keyArn } : {}),
            },
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: sfn.JsonPath.isEncodedJsonPath(config.instanceType.toString())
                    ? config.instanceType.toString() : `ml.${config.instanceType}`,
                VolumeSizeInGB: config.volumeSize.toGibibytes(),
                ...(config.volumeEncryptionKey ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds(),
            },
        };
    }
    renderHyperparameters(params) {
        return params ? { HyperParameters: params } : {};
    }
    renderVpcConfig(config) {
        return config
            ? {
                VpcConfig: {
                    SecurityGroupIds: core_1.Lazy.list({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
    makePolicyStatements() {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role =
            this.props.role ||
                new iam.Role(this, 'SagemakerRole', {
                    assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                    inlinePolicies: {
                        CreateTrainingJob: new iam.PolicyDocument({
                            statements: [
                                new iam.PolicyStatement({
                                    actions: [
                                        'cloudwatch:PutMetricData',
                                        'logs:CreateLogStream',
                                        'logs:PutLogEvents',
                                        'logs:CreateLogGroup',
                                        'logs:DescribeLogStreams',
                                        'ecr:GetAuthorizationToken',
                                        ...(this.props.vpcConfig
                                            ? [
                                                'ec2:CreateNetworkInterface',
                                                'ec2:CreateNetworkInterfacePermission',
                                                'ec2:DeleteNetworkInterface',
                                                'ec2:DeleteNetworkInterfacePermission',
                                                'ec2:DescribeNetworkInterfaces',
                                                'ec2:DescribeVpcs',
                                                'ec2:DescribeDhcpOptions',
                                                'ec2:DescribeSubnets',
                                                'ec2:DescribeSecurityGroups',
                                            ]
                                            : []),
                                    ],
                                    resources: ['*'],
                                }),
                            ],
                        }),
                    },
                });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'TrainJobSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        const stack = core_1.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTrainingJob = SageMakerCreateTrainingJob;
_a = JSII_RTTI_SYMBOL_1;
SageMakerCreateTrainingJob[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SageMakerCreateTrainingJob", version: "1.137.0" };
SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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