# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fuzzable', 'fuzzable.analysis']

package_data = \
{'': ['*']}

install_requires = \
['angr>=9.2.8,<10.0.0',
 'lief>=0.12.1,<0.13.0',
 'scikit-criteria>=0.7,<0.8',
 'tree-sitter>=0.20.0,<0.21.0',
 'typer[all]>=0.6.1,<0.7.0']

entry_points = \
{'console_scripts': ['fuzzable = fuzzable.__main__:app']}

setup_kwargs = {
    'name': 'fuzzable',
    'version': '2.0.6',
    'description': 'Framework for Automating Fuzzable Target Discovery with Static Analysis',
    'long_description': '# Fuzzable\n\n[![Build Status](https://github.com/ex0dus-0x/fuzzable/actions/workflows/main.yml/badge.svg)](https://github.com/ex0dus-0x/fuzzable/actions)\n[![PyPI version](https://badge.fury.io/py/fuzzable.svg)](https://badge.fury.io/py/fuzzable)\n[![Blackhat](https://raw.githubusercontent.com/toolswatch/badges/master/arsenal/usa/2022.svg)](https://raw.githubusercontent.com/toolswatch/badges/master/arsenal/usa/2022.svg)\n\nFramework for Automating _Fuzzable_ Target Discovery with Static Analysis\n\n![example](https://raw.githubusercontent.com/ex0dus-0x/fuzzable/main/extras/cli.png "CLI Example")\n\n## Introduction\n\nVulnerability researchers conducting security assessments on software will often harness the capabilities of coverage-guided fuzzing through powerful tools like AFL++ and libFuzzer. This is important as it automates the bughunting process and reveals exploitable conditions in targets quickly. However, when encountering large and complex codebases or closed-source binaries, researchers have to painstakingly dedicate time to manually audit and reverse engineer them to identify functions where fuzzing-based exploration can be useful.\n\n__Fuzzable__ is a framework that integrates both with C/C++ source code and binaries to assist vulnerability researchers in identifying function targets that are viable for fuzzing. This is done by applying several static analysis-based heuristics to pinpoint risky behaviors in the software and the functions that executes them. Researchers can then utilize the framework to generate basic harness templates, which can then be used to hunt for vulnerabilities, or to be integrated as part of a continuous fuzzing pipeline, such as Google\'s [oss-fuzz](https://github.com/google/oss-fuzz) project.\n\nIn addition to running as a standalone tool, Fuzzable is also integrated as a plugin for the [Binary Ninja ](https://binary.ninja) disassembler, with support for other disassembly backends being developed.\n\nCheck out the original blog post detailing the tool [here](https://codemuch.tech/2021/06/07/fuzzabble/), which highlights the technical specifications of the static analysis heuristics and how this tool came about. This tool is also featured at [Black Hat Arsenal USA 2022](https://www.blackhat.com/us-22/arsenal/schedule/index.html#automating-fuzzable-target-discovery-with-static-analysis-26726).\n\n## Features\n\n* Supports analyzing __binaries__ (with [Angr](https://angr.io) and [Binary Ninja](https://binary.ninja)) and\n__source code__ artifacts (with [tree-sitter](https://tree-sitter.github.io/tree-sitter/)).\n* Run static analysis both as a __standalone CLI tool__ or a __Binary Ninja plugin__.\n* __Harness generation__ to ramp up on creating fuzzing campaigns quickly.\n\n## Installation\n\nSome binary targets may require some sanitizing (ie. signature matching, or identifying functions from inlining), and therefore \n__fuzzable__ primarily uses Binary Ninja as a disassembly backend because of it\'s ability to effectively solve these problems. Therefore, it can be utilized both as a standalone tool and plugin.\n\nSince Binary Ninja isn\'t accessible to all and there may be a demand to utilize for security assessments and potentially scaling up in the cloud, an [angr](https://github.com/angr/angr)\n_fallback_ backend is also supported. I anticipate to incorporate other disassemblers down the road as well (priority: Ghidra).\n\n### Command Line (Standalone)\n\nIf you have Binary Ninja Commercial, be sure to install the API for standalone headless usage:\n\n```\n$ python3 /Applications/Binary\\ Ninja.app/Contents/Resources/scripts/install_api.py\n```\n\nInstall with `pip`:\n\n```\n$ pip install fuzzable\n```\n\n### Manual/Development Build\n\nWe use [poetry](https://python-poetry.org) for dependency management and building. To do a manual build, clone the repository with the third-party modules:\n\n```\n$ git clone --recursive https://github.com/ex0dus-0x/fuzzable\n```\n\nTo install manually:\n\n```\n$ cd fuzzable/\n\n# without poetry\n$ pip install .\n\n# with poetry\n$ poetry install\n\n# with poetry for a development virtualenv\n$ poetry shell\n```\n\nYou can now analyze binaries and/or source code with the tool!\n\n```\n# analyzing a single shared object library binary\n$ fuzzable analyze examples/binaries/libbasic.so\n\n# analyzing a single C source file\n$ fuzzable analyze examples/source/libbasic.c\n\n# analyzing a workspace with multiple C/C++ files and headers\n$ fuzzable analyze examples/source/source_bundle/\n```\n\n### Binary Ninja Plugin\n\n__fuzzable__ can be easily installed through the Binary Ninja plugin marketplace by going to `Binary Ninja > Manage Plugins` and searching for it. Here is an example of the __fuzzable__ plugin running,\naccuracy identifying targets for fuzzing and further vulnerability assessment:\n\n![binja_example](/extras/binja.png "Binary Ninja Example")\n\n## Usage\n\n__fuzzable__ comes with various options to help better tune your analysis. More will be supported in future plans and any feature requests made. \n\n### Static Analysis Heuristics\n\nTo determine fuzzability, __fuzzable__ utilize several heuristics to determine which targets are the most viable to target for dynamic analysis. These heuristics are all weighted differently using the [scikit-criteria](https://scikit-criteria.quatrope.org/en/latest/tutorial/quickstart.html) library, which utilizes _multi-criteria decision analysis_ to determine the best candidates. These metrics and are there weights can be seen here:\n\n| Heuristic             | Description                                                 | Weight |\n|-----------------------|-------------------------------------------------------------|--------|\n| Fuzz Friendly Name    | Symbol name implies behavior that ingests file/buffer input | 0.3    |\n| Risky Sinks           | Arguments that flow into risky calls (ie memcpy)            | 0.3    |\n| Natural Loops         | Number of loops detected with the dominance frontier        | 0.05   |\n| Cyclomatic Complexity | Complexity of function target based on edges + nodes        | 0.05   |\n| Coverage Depth        | Number of callees the target traverses into                 | 0.3    |\n\n> As mentioned, check out the [technical blog post](https://codemuch.tech/2021/06/07/fuzzabble/) for a more in-depth look into why and how these metrics are utilized.\n\nMany metrics were largely inspired by [Vincenzo Iozzo\'s original work in 0-knowledge fuzzing](https://resources.sei.cmu.edu/asset_files/WhitePaper/2010_019_001_53555.pdf).\n\nEvery targets you want to analyze is diverse, and __fuzzable__ will not be able to account for every edge case behavior in the program target. Thus, it may be important during analysis to _tune_ these weights appropriately to see if different results make more sense for your use case. To tune these weights in the CLI, simply specify the `--score-weights` argument:\n\n```\n$ fuzzable analyze <TARGET> --score-weights=0.2,0.2,0.2,0.2,0.2\n```\n\n### Analysis Filtering\n\nBy default, __fuzzable__ will filter out function targets based on the following criteria:\n\n* __Top-level entry calls__ - functions that aren\'t called by any other calls in the target. These are ideal entry points that have potentially very high coverage.\n* __Static calls__ - _(source only)_ functions that are `static` and aren\'t exposed through headers.\n* __Imports__ - _(binary only)_ other library dependencies being used by the target\'s implementations.\n\nTo see calls that got filtered out by __fuzzable__, set the `--list_ignored` flag:\n\n```\n$ fuzzable analyze --list-ignored <TARGET>\n```\n\nIn Binary Ninja, you can turn this setting in `Settings > Fuzzable > List Ignored Calls`.\n\nIn the case that __fuzzable__ falsely filters out important calls that should be analyzed, it is recommended to use `--include-*` arguments\nto include them during the run:\n\n```\n# include ALL non top-level calls that were filtered out\n$ fuzzable analyze --include-nontop <TARGET>\n\n# include specific symbols that were filtered out\n$ fuzzable analyze --include-sym <SYM> <TARGET>\n```\n\nIn Binary Ninja, this is supported through `Settings > Fuzzable > Include non-top level calls` and `Symbols to Exclude`.\n\n### Harness Generation\n\nNow that you have found your ideal candidates to fuzz, __fuzzable__ will also help you generate fuzzing harnesses that are (almost) ready to instrument and compile for use with either a file-based fuzzer (ie. AFL++, Honggfuzz) or in-memory fuzzer (libFuzzer). To do so in the CLI:\n\n```\n# generate harness from a candidate\n$ fuzzable create-harness target --symbol-name=some_unsafe_call\n\n# make minimal and necessary modifications to the harness\n$ vim target_some_unsafe_call_harness.cpp\n\n# example compilation for AFL-QEMU, which is specified in the comments of the generated harness\n$ clang target_some_unsafe_call_harness.cpp -no-pie -o target_some_unsafe_call_harness -ldl\n\n# create your base seeds, ideally should be more well-formed for input\n$ mkdir in/\n$ echo "seed" >> in/seed\n\n# start black box fuzzing\n$ afl-fuzz -Q -m none -i in/ -o out/ -- ./target_some_unsafe_call_harness\n```\n\nIf this target is a source codebase, the [generic source template](/templates/linux_source_harness.cpp) will be used. \n\nIf the target is a binary, the [generic black-box template](/templates/linux_closed_source_harness.cpp) will be used, which ideally can be used with a fuzzing emulation mode like [AFL-QEMU](https://github.com/mirrorer/afl/blob/master/qemu_mode/README.qemu). A copy of the binary will also be created as a shared object if the symbol isn\'t exported directly to be `dlopen`ed using [LIEF](https://lief-project.github.io).\n\nAt the moment, this feature is quite rudimentary, as it simply will create a standalone C++ harness populated with the appropriate parameters, and will not auto-generate code that is needed for any runtime behaviors (ie. instantiating and freeing structures). However, the templates created for __fuzzable__ should get still get you running quickly. Here are some ambitious features I would like to implement down the road:\n\n* Full harness synthesis - harnesses will work directly with absolutely no manual changes needed.\n* Synthesis from potential unit tests using the [DeepState](https://github.com/trailofbits/deepstate) framework _(Source only)_.\n* Immediate deployment to a managed continuous fuzzing fleet.\n\n### Exporting Reports\n\n__fuzzable__ supports generating reports in various formats. The current ones that are supported are JSON, CSV and Markdown. This can be useful if you are utilizing this as part of automation where you would like to\ningest the output in a serializable format.\n\nIn the CLI, simply pass the `--export` argument with a filename with the appropriate extension:\n\n```\n$ fuzzable analyze --export=report.json <TARGET>\n```\n\nIn Binary Ninja, go to `Plugins > Fuzzable > Export Fuzzability Report > ...` and select the format you want to\nexport to and the path you want to write it to.\n\n## Contributing\n\nThis tool will be continuously developed, and any help from external mantainers are appreciated!\n\n* Create an [issue](https://github.com/ex0dus-0x/fuzzable/issues) for feature requests or bugs that you have come across.\n* Submit a [pull request](https://github.com/ex0dus-0x/fuzzable/pulls) for fixes and enhancements that you would like to see contributed to this tool.\n\n## License\n\nFuzzable is licensed under the [MIT License](https://codemuch.tech/license.txt).\n',
    'author': 'ex0dus',
    'author_email': 'ex0dus@codemuch.tech',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://codemuch.tech/fuzzable',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
