"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sanitizeOutput = exports.directorySnapshot = exports.synthSnapshotWithPost = exports.synthSnapshot = exports.mkdtemp = exports.execProjenCLI = exports.TestProject = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const glob_1 = require("glob");
const __1 = require("..");
const logging = require("../logging");
const util_1 = require("../util");
const PROJEN_CLI = require.resolve('../../bin/projen');
logging.disable(); // no logging during tests
class TestProject extends __1.Project {
    constructor(options = {}) {
        const tmpdir = mkdtemp();
        super({
            name: 'my-project',
            outdir: tmpdir,
            clobber: false,
            logging: {
                level: __1.LogLevel.OFF,
            },
            ...options,
        });
    }
    postSynthesize() {
        fs.writeFileSync(path.join(this.outdir, '.postsynth'), '# postsynth');
    }
}
exports.TestProject = TestProject;
function execProjenCLI(workdir, args = []) {
    const command = [
        process.execPath,
        PROJEN_CLI,
        ...args,
    ];
    return util_1.exec(command.map(x => `"${x}"`).join(' '), { cwd: workdir, stdio: 'inherit' });
}
exports.execProjenCLI = execProjenCLI;
const autoRemove = new Set();
// Hook to automatically remove temporary directories without needing each
// place to actually handle this specifically.
afterAll((done) => {
    // Array.from used to get a copy, so we can safely remove from the set
    for (const dir of Array.from(autoRemove)) {
        try {
            // Note - fs-extra.removeSync is idempotent, so we're safe if the
            // directory has already been cleaned up before we get there!
            fs.removeSync(dir);
        }
        catch (e) {
            done.fail(e);
        }
        autoRemove.delete(dir);
    }
    done();
});
function mkdtemp() {
    const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-test-'));
    autoRemove.add(tmpdir);
    return tmpdir;
}
exports.mkdtemp = mkdtemp;
function synthSnapshot(project) {
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = 'true';
        project.synth();
        return directorySnapshot(project.outdir);
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function synthSnapshotWithPost(project) {
    try {
        project.synth();
        return directorySnapshot(project.outdir);
    }
    finally {
        fs.removeSync(project.outdir);
    }
}
exports.synthSnapshotWithPost = synthSnapshotWithPost;
function directorySnapshot(root, options = {}) {
    var _a;
    const output = {};
    const files = glob_1.glob.sync('**', {
        ignore: (_a = options.excludeGlobs) !== null && _a !== void 0 ? _a : [],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (path.extname(filePath) === '.json') {
            content = fs.readJsonSync(filePath);
        }
        else {
            content = fs.readFileSync(filePath, 'utf-8');
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
/**
 * Removes any non-deterministic aspects from the synthesized output.
 * @param dir The output directory.
 */
function sanitizeOutput(dir) {
    const filepath = path.join(dir, 'package.json');
    const pkg = fs.readJsonSync(filepath);
    const prev = pkg.devDependencies.projen;
    if (!prev) {
        throw new Error(`expecting "${filepath}" to include a devDependency on "projen"`);
    }
    // replace the current projen version with 999.999.999 for deterministic output.
    // this will preserve any semantic version requirements (e.g. "^", "~", etc).
    pkg.devDependencies.projen = prev.replace(/\d+\.\d+\.\d+/, '999.999.999');
    fs.writeJsonSync(filepath, pkg);
    // we will also patch deps.json so that all projen deps will be set to 999.999.999
    const depsPath = path.join(dir, '.projen', 'deps.json');
    const deps = fs.readJsonSync(depsPath);
    for (const dep of deps.dependencies) {
        if (dep.name === 'projen') {
            dep.version = dep.version.replace(/\d+\.\d+\.\d+/, '999.999.999');
        }
    }
    fs.chmodSync(depsPath, '777');
    fs.writeJsonSync(depsPath, deps);
}
exports.sanitizeOutput = sanitizeOutput;
//# sourceMappingURL=data:application/json;base64,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