"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const _resolve_1 = require("../_resolve");
test('null', () => {
    expect(_resolve_1.resolve(null)).toStrictEqual(null);
});
test('undefined', () => {
    expect(_resolve_1.resolve(undefined)).toStrictEqual(undefined);
});
test('string', () => {
    expect(_resolve_1.resolve('')).toStrictEqual('');
    expect(_resolve_1.resolve('hello')).toStrictEqual('hello');
    expect(_resolve_1.resolve('0')).toStrictEqual('0');
    expect(_resolve_1.resolve('hello world')).toStrictEqual('hello world');
    expect(_resolve_1.resolve('  ')).toStrictEqual('  ');
});
test('number', () => {
    expect(_resolve_1.resolve(0)).toStrictEqual(0);
    expect(_resolve_1.resolve(10)).toStrictEqual(10);
    expect(_resolve_1.resolve(-102)).toStrictEqual(-102);
});
test('boolean', () => {
    expect(_resolve_1.resolve(true)).toStrictEqual(true);
    expect(_resolve_1.resolve(false)).toStrictEqual(false);
});
test('array', () => {
    expect(_resolve_1.resolve([])).toStrictEqual([]);
    expect(_resolve_1.resolve([1, 2, 3])).toStrictEqual([1, 2, 3]);
});
test('object', () => {
    expect(_resolve_1.resolve(Object.create(null))).toStrictEqual({});
    expect(_resolve_1.resolve({})).toStrictEqual({});
    expect(_resolve_1.resolve({ foo: 123, hello: [1, 2, { bar: 3 }] })).toStrictEqual({
        foo: 123,
        hello: [1, 2, { bar: 3 }],
    });
});
test('functions are resolved', () => {
    expect(_resolve_1.resolve(() => 123)).toStrictEqual(123);
    expect(_resolve_1.resolve(() => 'hello')).toStrictEqual('hello');
    expect(_resolve_1.resolve(() => undefined)).toStrictEqual(undefined);
});
test('recursive resolve', () => {
    expect(_resolve_1.resolve(() => [1, 2, () => 30])).toStrictEqual([1, 2, 30]);
    expect(_resolve_1.resolve(() => ({ foo: 123, bar: () => 'bar' }))).toStrictEqual({ foo: 123, bar: 'bar' });
    expect(_resolve_1.resolve(() => ({ foo: 123, bar: () => ['bar', 'baz', { hello: () => 'world' }] }))).toStrictEqual({
        foo: 123,
        bar: ['bar', 'baz', { hello: 'world' }],
    });
});
test('context is passed to functions', () => {
    expect(_resolve_1.resolve((x) => x + 1, { args: [10] })).toStrictEqual(11);
    const calculator = {
        add: (a, b) => a + b,
        sub: (a, b) => a - b,
        mul: (a, b) => a * b,
        div: (a, b) => a / b,
        fun: (a1, b1, text) => [
            `this is ${text}`,
            (a2) => a1 * a2,
            (_, b2) => b1 + b2,
            {
                another: {
                    level: (a3, b3, text3) => `hello ${text3} ${b3} ${a3}`,
                },
            },
        ],
    };
    expect(_resolve_1.resolve(calculator, { args: [10, 2, 'fun'] })).toStrictEqual({
        add: 12,
        sub: 8,
        mul: 20,
        div: 5,
        fun: [
            'this is fun',
            100,
            4,
            { another: { level: 'hello fun 2 10' } },
        ],
    });
});
test('"undefined" values are omitted', () => {
    const r = (o) => _resolve_1.resolve(o);
    expect(r({ foo: undefined })).toStrictEqual({});
    expect(r({ foo: { bar: undefined } })).toStrictEqual({ foo: {} });
    expect(r({ foo: { bar: undefined, hello: 123 } })).toStrictEqual({ foo: { hello: 123 } });
    expect(r({ foo: [undefined] })).toStrictEqual({ foo: [] });
});
test('omitEmpty', () => {
    const r = (o) => _resolve_1.resolve(o, { omitEmpty: true });
    expect(r({ foo: {} })).toStrictEqual(undefined);
    expect(r({ foo: { bar: {} } })).toStrictEqual(undefined);
    expect(r({ foo: [] })).toStrictEqual(undefined);
    expect(r({ bar: { foo: [] } })).toStrictEqual(undefined);
    expect(r({ foo: { hello: 'world', boom: { bar: [] } } })).toStrictEqual({ foo: { hello: 'world' } });
});
//# sourceMappingURL=data:application/json;base64,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