"use strict";
const cluster_1 = require("../lib/cluster-resource-handler/cluster");
const mocks = require("./cluster-resource-handler-mocks");
module.exports = {
    setUp(callback) {
        mocks.reset();
        callback();
    },
    create: {
        async 'onCreate: minimal defaults (vpc + role)'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', mocks.MOCK_PROPS));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.configureAssumeRoleRequest, {
                RoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                RoleSessionName: 'AWSCDK.EKSCluster.Create.fake-request-id',
            });
            test.deepEqual(mocks.actualRequest.createClusterRequest, {
                roleArn: 'arn:of:role',
                resourcesVpcConfig: {
                    subnetIds: ['subnet1', 'subnet2'],
                    securityGroupIds: ['sg1', 'sg2', 'sg3'],
                },
                name: 'MyResourceId-fakerequestid',
            });
            test.done();
        },
        async 'generated cluster name does not exceed 100 characters'(test) {
            var _a, _b;
            // GIVEN
            const req = {
                StackId: 'fake-stack-id',
                RequestId: '602c078a-6181-4352-9676-4f00352445aa',
                ResourceType: 'Custom::EKSCluster',
                ServiceToken: 'boom',
                LogicalResourceId: 'hello'.repeat(30),
                ResponseURL: 'http://response-url',
                RequestType: 'Create',
                ResourceProperties: {
                    ServiceToken: 'boom',
                    Config: mocks.MOCK_PROPS,
                    AssumeRoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                },
            };
            // WHEN
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
            await handler.onEvent();
            // THEN
            test.equal((_a = mocks.actualRequest.createClusterRequest) === null || _a === void 0 ? void 0 : _a.name.length, 100);
            test.deepEqual((_b = mocks.actualRequest.createClusterRequest) === null || _b === void 0 ? void 0 : _b.name, 'hellohellohellohellohellohellohellohellohellohellohellohellohellohe-602c078a6181435296764f00352445aa');
            test.done();
        },
        async 'onCreate: explicit cluster name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                name: 'ExplicitCustomName',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.name, 'ExplicitCustomName');
            test.done();
        },
        async 'with no specific version'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                version: '12.34.56',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.version, '12.34.56');
            test.done();
        },
        async 'isCreateComplete still not complete if cluster is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'CREATING';
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { IsComplete: false });
            test.done();
        },
        async 'isCreateComplete is complete when cluster is ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp, {
                IsComplete: true,
                Data: {
                    Name: 'physical-resource-id',
                    Endpoint: 'http://endpoint',
                    Arn: 'arn:cluster-arn',
                    CertificateAuthorityData: 'certificateAuthority-data',
                },
            });
            test.done();
        },
    },
    delete: {
        async 'returns correct physical name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.onEvent();
            test.deepEqual(mocks.actualRequest.deleteClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { PhysicalResourceId: 'physical-resource-id' });
            test.done();
        },
        async 'onDelete ignores ResourceNotFoundException'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.deleteClusterErrorCode = 'ResourceNotFoundException';
            await handler.onEvent();
            test.done();
        },
        async 'isDeleteComplete returns false as long as describeCluster succeeds'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.ok(!resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete returns true when describeCluster throws a ResourceNotFound exception'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'ResourceNotFoundException';
            const resp = await handler.isComplete();
            test.ok(resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete propagates other errors'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'OtherException';
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            test.equal(error.code, 'OtherException');
            test.done();
        },
    },
    update: {
        async 'no change'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', mocks.MOCK_PROPS, mocks.MOCK_PROPS));
            const resp = await handler.onEvent();
            test.equal(resp, undefined);
            test.equal(mocks.actualRequest.createClusterRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterConfigRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
            test.done();
        },
        async 'isUpdateComplete is not complete when status is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'UPDATING';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, false);
            test.done();
        },
        async 'isUpdateComplete waits for ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, true);
            test.done();
        },
        'requires replacement': {
            'name change': {
                async 'explicit name change'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: 'new-cluster-name-1234',
                    }, {
                        name: 'old-cluster-name',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'new-cluster-name-1234',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'new-cluster-name-1234' });
                    test.done();
                },
                async 'from auto-gen name to explicit name'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: undefined,
                    }, {
                        name: 'explicit',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'MyResourceId-fakerequestid',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                    test.done();
                },
            },
            async 'subnets or security groups requires a replacement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                }, {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1'],
                        securityGroupIds: ['sg2'],
                    },
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'arn:of:role',
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                });
                test.done();
            },
            async '"roleArn" requires a replcement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                }, {
                    roleArn: 'old-arn',
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'fails if cluster has an explicit "name" that is the same as the old "name"'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // THEN
                let err;
                try {
                    await handler.onEvent();
                }
                catch (e) {
                    err = e;
                }
                test.equal(err === null || err === void 0 ? void 0 : err.message, 'Cannot replace cluster "explicit-cluster-name" since it has an explicit physical name. Either rename the cluster or remove the "name" configuration');
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it does not'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: undefined,
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it has a different name'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'new-explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'old-explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'new-explicit-cluster-name' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'new-explicit-cluster-name',
                    roleArn: 'new-arn',
                });
                test.done();
            },
        },
        'in-place': {
            'version change': {
                async 'from undefined to a specific value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '12.34',
                    }, {
                        version: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '12.34',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from a specific value to another value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '2.0',
                    }, {
                        version: '1.1',
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '2.0',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'to a new value that is already the current version'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '1.0',
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.describeClusterRequest, { name: 'physical-resource-id' });
                    test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'fails from specific value to undefined'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: undefined,
                    }, {
                        version: '1.2',
                    }));
                    let error;
                    try {
                        await handler.onEvent();
                    }
                    catch (e) {
                        error = e;
                    }
                    test.equal(error.message, 'Cannot remove cluster version configuration. Current version is 1.2');
                    test.done();
                },
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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