"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cloudformation_1 = require("@aws-cdk/aws-cloudformation");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * Fargate profiles allows an administrator to declare which pods run on
 * Fargate. This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 */
class FargateProfile extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        const role = (_a = props.podExecutionRole) !== null && _a !== void 0 ? _a : new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')],
        });
        let subnets;
        if (props.vpc) {
            const selection = (_b = props.subnetSelection) !== null && _b !== void 0 ? _b : { subnetType: ec2.SubnetType.PRIVATE };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error('Fargate profile requires at least one selector');
        }
        if (props.selectors.length > 5) {
            throw new Error('Fargate profile supports up to five selectors');
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new aws_cloudformation_1.CustomResource(this, 'Resource', {
            provider: provider.provider,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster._getKubectlCreationRoleArn(),
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: role.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.anyValue({ produce: () => this.tags.renderTags() }),
                },
            },
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
    }
}
exports.FargateProfile = FargateProfile;
//# sourceMappingURL=data:application/json;base64,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