from __future__ import annotations

import importlib
import types
import typing
from types import ModuleType
import inspect
from os import path
import os
import logging

from querky.result_shape import one_, all_, value_, status_, column_, One, All, ResultShape
from querky.conn_param_config import ConnParamConfig, First
from querky.annotation_generator import AnnotationGenerator
from querky.type_constructor import TypeConstructor
from querky.module_constructor import ModuleConstructor
from querky.base_types import TypeMetaData
from querky.query import Query
from querky.contract import Contract


logger = logging.getLogger("querky")


def to_camel_case(snake_str):
    return "".join(x.capitalize() for x in snake_str.lower().split("_"))


ShapeStringRepr = typing.Literal["one", "many", "column", "value", "status"]


QueryDef = typing.Callable[[typing.Callable[[...], str]], Query]


class Querky:
    def __init__(
            self,
            basedir: str | None = None,
            annotation_generator: AnnotationGenerator | None = None,
            contract: Contract | None = None,
            conn_param_config: ConnParamConfig | None = None,
            type_factory: typing.Callable[[Query, str], TypeConstructor] | None = None,
            subdir: str | None = "queries",
            on_before_func_code_emit: typing.Optional[typing.Callable[[typing.List[str], Query], typing.List[str]]] = None,
            on_before_type_code_emit: typing.Optional[typing.Callable[[typing.List[str], Query], typing.List[str]]] = None,
            imports: typing.Optional[typing.Set[str]] = None,
            indent: str = '    ',
            query_class: typing.Type[Query] = Query
    ):
        self.basedir = basedir

        self.on_before_func_code_emit = on_before_func_code_emit
        self.on_before_type_code_emit = on_before_type_code_emit
        self.query_class = query_class
        self.imports = imports or set()
        self.indent = indent

        self.annotation_generator = annotation_generator

        self.module_ctors: dict[types.ModuleType, ModuleConstructor] = dict()

        self.type_factory = type_factory

        if conn_param_config is None:
            conn_param_config = First(name='__conn', positional=True)

        self.conn_param_config = conn_param_config

        self.contract = contract

        self.subdir = subdir
        if self.subdir and not str.isidentifier(self.subdir):
            raise ValueError("subdir must be a valid python identifier")

        self.file_signature = "# ~ AUTOGENERATED BY QUERKY ~ #"

    def get_indent(self, i: int):
        return self.indent * i

    def create_query(
            self,
            fn: typing.Callable[[...], str],
            shape: typing.Callable[[Query], ResultShape],
            conn_param_config: ConnParamConfig | None,
            explicit_name: str | None,
            parent_query: typing.Optional[Query],
            kwargs: typing.Optional[typing.Dict[str, typing.Any]]
    ) -> Query:
        module = inspect.getmodule(fn)
        if module in self.module_ctors:
            module_ctor = self.module_ctors[module]
        else:
            filename = self.generate_filename(module)
            if not str.isidentifier(filename):
                raise ValueError(f"Generated a filename which is not a valid python identifier: {filename}")

            filedir = path.dirname(module.__file__)
            new_module_name = module.__name__.rsplit('.', maxsplit=1)[0]

            if self.subdir:
                filedir = path.join(filedir, self.subdir)
                new_module_name = f"{new_module_name}.{self.subdir}"

            fullpath = path.join(filedir, f'{filename}.py')
            new_module_name = f"{new_module_name}.{filename}"

            module_ctor = ModuleConstructor(self, module, fullpath, new_module_name, filedir)
            self.module_ctors[module] = module_ctor
        return self.query_class(
            fn,
            shape,
            module_ctor,
            self.conn_param_config or conn_param_config,
            explicit_name,
            parent_query,
            kwargs
        )

    def query(
            self,
            arg: str | TypeMetaData | Query | typing.Callable[[...], str] | None = None,
            *,
            shape: ShapeStringRepr = 'status',
            optional: bool | None = None,
            **kwargs
    ) -> QueryDef | Query:
        def wrapper(fn: typing.Callable[[...], str]) -> Query:
            nonlocal optional

            if shape in ['many', 'one']:
                if isinstance(arg, TypeMetaData):
                    raise ValueError(
                        "TypeMetaData is not supported for `many` or `one` constructors. "
                        "Use it only for `one` and `column` constructors."
                    )

                if not isinstance(arg, Query):
                    if arg is None:
                        # if we don't have a name provided for us, we're gonna create it out of the function name
                        type_name = to_camel_case(fn.__name__)
                    else:
                        type_name = arg

                    if not type_name.isidentifier():
                        raise ValueError(f"Name type should be a valid python identifier. You provided: {type_name}")
                else:
                    type_name = None

                type_name: str | None

                if shape == 'many':
                    if optional is not None:
                        raise TypeError(
                            'ALL constructor does not accept `optional` flag -- '
                            'at least an empty set will always be returned'
                        )
                    created_shape = all_(type_name)
                else:
                    if optional is None:
                        optional = True
                    created_shape = one_(type_name, optional=optional)
            elif shape in ['value', 'column']:
                if arg is None:
                    annotation = None
                else:
                    annotation = arg

                if shape == 'value':
                    if optional is None:
                        optional = True
                    created_shape = value_(annotation, optional=optional)
                else:
                    if optional is None:
                        optional = False
                    created_shape = column_(annotation, elem_optional=optional)
            elif shape == 'status':
                if optional is not None:
                    raise TypeError(
                        "STATUS constructor does not accept `optional` flag -- "
                        "query always returns a status"
                    )
                if arg is not None:
                    raise TypeError(
                        "STATUS constructor does not support annotations"
                    )

                created_shape = status_()
            else:
                raise NotImplementedError(shape)

            return self.create_query(
                fn,
                created_shape,
                self.conn_param_config,
                None,
                arg if isinstance(arg, Query) else None,
                kwargs
            )

        if callable(arg) and not isinstance(arg, Query):
            func = arg
            arg = None
            return wrapper(func)
        else:
            if isinstance(arg, Query):
                if not isinstance(arg.shape, (One, All)):
                    raise ValueError(
                        "You provided query as type argument, "
                        "yet the source query does not emit any records as per its definition. "
                        "Only queries shaped 'one' or 'many' can have their type definitions copied.\n"
                        f"Source query: {arg.unique_name}"
                    )
                if shape not in ["many", "one"]:
                    raise ValueError(
                        "Child queries can only have shape of 'one' or 'many' (as do their respective parents).\n"
                        f"Source query: {arg.unique_name}"
                    )

            return wrapper

    def generate_filename(self, module: ModuleType) -> str:
        filepath = module.__file__
        filename = path.basename(filepath).split('.', 1)[0]
        if not self.subdir:
            return f"{filename}_queries"
        else:
            return filename

    def check_file_is_mine(self, fullpath: str):
        with open(fullpath, encoding='utf-8', mode='r') as f:
            first_line = f.readline().strip()

        if first_line != self.file_signature:
            raise ValueError(
                f"{fullpath} - module's first line reads:\n"
                f"{first_line}\n"
                f"I expected it to be:\n"
                f"{self.file_signature}\n"
                f"This could be your code, so I won't overwrite this file!"
            )

    def _pre_generate(self, base_module: types.ModuleType):
        modules = []

        if base_module.__file__ is None:
            raise ValueError(
                f"Module `{base_module.__name__}` does not have __file__ defined. "
                f"If it is a directory, you should put __init__.py file inside of it and try running the script again."
            )

        directory = path.dirname(base_module.__file__)
        for filename in os.listdir(directory):
            if filename.startswith('__') or not filename.endswith('.py'):
                continue

            filepath = os.path.join(directory, filename)
            if not os.path.isfile(filepath):
                continue

            modulename = os.path.splitext(filename)[0]

            module_import_path = f'{base_module.__name__}.{modulename}'
            module = importlib.import_module(module_import_path)
            modules.append(module)

    async def generate(self, db, base_modules: typing.Collection[types.ModuleType] | None = None):
        if base_modules is not None:
            for base_module in base_modules:
                self._pre_generate(base_module)
        for module_ctor in self.module_ctors.values():
            await module_ctor.generate_module(db)

    def generate_sync(self, db, base_modules: typing.Collection[types.ModuleType] | None = None):
        if base_modules is not None:
            for base_module in base_modules:
                self._pre_generate(base_module)
        for module_ctor in self.module_ctors.values():
            module_ctor.generate_module_sync(db)

    def sign_file_contents(self, lines: list[str]) -> None:
        lines.insert(0, self.file_signature)
