# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/07_im2im_annotator.ipynb (unless otherwise specified).

__all__ = ['ImCanvas', 'Im2ImAnnotator']

# Internal Cell

import warnings
from math import ceil
from pathlib import Path
from typing import Optional, Dict, List

from ipycanvas import Canvas
from ipywidgets import (AppLayout, VBox, HBox, Button, Layout, HTML, Output)

from .base import BaseState, AppWidgetState
from .bbox_canvas import draw_img
from .capture_annotator import CaptureGrid
from .image_button import ImageButton
from .navi_widget import Navi
from .storage import JsonLabelStorage
from IPython.display import display

# Internal Cell

class Im2ImState(BaseState):
    annotations: Dict[str, Optional[List[str]]] = {}
    disp_number: int = 9
    question_value: str = ''
    n_rows: Optional[int] = 3
    n_cols: Optional[int] = 3
    image_path: Optional[str]
    im_width: int = 300
    im_height: int = 300
    label_width: int = 150
    label_height: int = 150

# Cell

class ImCanvas(HBox):
    def __init__(self, width=150, height=150, has_border=False):
        self.has_border = has_border
        self._canvas = Canvas(width=width, height=height)
        super().__init__([self._canvas])

    def _draw_image(self, image_path: str):
        self._image_scale = draw_img(
            self._canvas,
            image_path,
            clear=True,
            has_border=self.has_border
        )

    def _clear_image(self):
        self._canvas.clear()

    # needed to support voila
    # https://ipycanvas.readthedocs.io/en/latest/advanced.html#ipycanvas-in-voila
    def observe_client_ready(self, cb=None):
        self._canvas.on_client_ready(cb)

# Internal Cell

class Im2ImAnnotatorGUI(AppLayout):
    def __init__(
        self,
        app_state: AppWidgetState,
        im2im_state: Im2ImState,
        label_autosize=False,
        save_btn_clicked: callable = None,
        grid_box_clicked: callable = None,
        has_border: bool = False
    ):
        self._app_state = app_state
        self._im2im_state = im2im_state
        self.save_btn_clicked = save_btn_clicked
        self.grid_box_clicked = grid_box_clicked

        if label_autosize:
            if self._im2im_state.im_width < 100 or self._im2im_state.im_height < 100:
                self._im2im_state.set_quietly('label_width', 10)
                self._im2im_state.set_quietly('label_height', 10)
            elif self._im2im_state.im_width > 1000 or self._im2im_state.im_height > 1000:
                self._im2im_state.set_quietly('label_width', 50)
                self._im2im_state.set_quietly('label_height', 10)
            else:
                label_width = min(self._im2im_state.im_width, self._im2im_state.im_height) / 10
                self._im2im_state.set_quietly('label_width', label_width)
                self._im2im_state.set_quietly('label_height', label_width)

        self._image = ImCanvas(
            width=self._im2im_state.im_width,
            height=self._im2im_state.im_height,
            has_border=has_border
        )

        self._navi = Navi()

        self._save_btn = Button(description="Save",
                                layout=Layout(width='auto'))

        self._controls_box = HBox(
            [self._navi, self._save_btn],
            layout=Layout(
                display='flex',
                justify_content='center',
                flex_flow='wrap',
                align_items='center'
            )
        )

        self._grid_box = CaptureGrid(
            grid_item=ImageButton,
            image_width=self._im2im_state.label_width,
            image_height=self._im2im_state.label_height,
            n_rows=self._im2im_state.n_rows,
            n_cols=self._im2im_state.n_cols
        )

        self._grid_label = HTML(value="<b>LABEL</b>",)
        self._labels_box = VBox(
            children=[self._grid_label, self._grid_box],
            layout=Layout(
                display='flex',
                justify_content='center',
                flex_wrap='wrap',
                align_items='center')
        )

        if self._app_state.max_im_number:
            self._set_navi_max_im_number(self._app_state.max_im_number)

        if self._im2im_state.annotations:
            self._grid_box.load_annotations_labels(self._im2im_state.annotations)

        if self._im2im_state.question_value:
            self._set_label(self._im2im_state.question_value)

        self._im2im_state.subscribe(self._set_label, 'question_value')
        self._im2im_state.subscribe(self._image._draw_image, 'image_path')
        self._im2im_state.subscribe(self._grid_box.load_annotations_labels, 'annotations')
        self._save_btn.on_click(self._btn_clicked)
        self._grid_box.on_click(self._grid_clicked)

        super().__init__(
            header=None,
            left_sidebar=VBox([self._image, self._controls_box],
                              layout=Layout(display='flex', justify_content='center',
                                            flex_wrap='wrap', align_items='center')),
            center=self._labels_box,
            right_sidebar=None,
            footer=None,
            pane_widths=(6, 4, 0),
            pane_heights=(1, 1, 1))

    def _set_navi_max_im_number(self, max_im_number: int):
        self._navi.max_im_num = max_im_number

    def _set_label(self, question_value: str):
        self._grid_label.value = question_value

    def _btn_clicked(self, *args):
        if self.save_btn_clicked:
            self.save_btn_clicked(*args)
        else:
            warnings.warn("Save button click didn't triggered any event.")

    def _grid_clicked(self, event, name=None):
        if self.grid_box_clicked:
            self.grid_box_clicked(event, name)
        else:
            warnings.warn("Grid box click didn't triggered any event.")

    def on_client_ready(self, callback):
        self._image.observe_client_ready(callback)

# Internal Cell
def _label_state_to_storage_format(label_state):
    return [Path(k).name for k, v in label_state.items() if v['answer']]

# Internal Cell
def _storage_format_to_label_state(storage_format, label_names, label_dir):
    return {str(Path(label_dir) / label): {
        'answer': label in storage_format} for label in label_names}

# Internal Cell

class Im2ImAnnotatorController:
    debug_output = Output(layout={'border': '1px solid black'})

    def __init__(
        self,
        app_state: AppWidgetState,
        im2im_state: Im2ImState,
        storage: JsonLabelStorage,
        input_item=None,
        output_item=None,
        question=None,
    ):
        self._app_state = app_state
        self._im2im_state = im2im_state
        self._storage = storage

        self.input_item = input_item
        self.output_item = output_item

        self.images = self._storage.get_im_names(None)
        self._app_state.max_im_number = len(self.images)

        self.labels = self._storage.get_labels()
        self.labels_num = len(self.labels)

        # Tracks the app_state.index history
        self._last_index = 0

        self._im2im_state.n_rows, self._im2im_state.n_cols = self._calc_num_labels(
            self.labels_num,
            self._im2im_state.n_rows,
            self._im2im_state.n_cols
        )

        if question:
            self._im2im_state.question_value = (f'<center><p style="font-size:20px;">'
                                                f'{question}</p></center>')

    def _calc_num_labels(self, n_total: int, n_rows: int, n_cols: int) -> tuple:
        if n_cols is None:
            if n_rows is None:  # automatic arrange
                label_cols = 3
                label_rows = ceil(n_total / label_cols)
            else:  # calc cols to show all labels
                label_rows = n_rows
                label_cols = ceil(n_total / label_rows)
        else:
            if n_rows is None:  # calc rows to show all labels
                label_cols = n_cols
                label_rows = ceil(n_total / label_cols)
            else:  # user defined
                label_cols = n_cols
                label_rows = n_rows

        if label_cols * label_rows < n_total:
            warnings.warn("!! Not all labels shown. Check n_cols, n_rows args !!")

        return label_rows, label_cols

    def _update_im(self):
        # print('_update_im')
        index = self._app_state.index
        self._im2im_state.image_path = str(self.images[index])

    def _update_state(self, change=None):  # from annotations
        # print('_update_state')
        image_path = self._im2im_state.image_path

        if not image_path:
            return

        if image_path in self._storage:
            current_annotation = self._storage.get(str(image_path)) or {}
            self._im2im_state.annotations = _storage_format_to_label_state(
                storage_format=current_annotation or [],
                label_names=self.labels,
                label_dir=self._storage.label_dir
            )

    def _update_annotations(self, index: int):  # from screen
        # print('_update_annotations')
        image_path = self._im2im_state.image_path
        if image_path:
            self._storage[image_path] = _label_state_to_storage_format(
                self._im2im_state.annotations
            )

    def save_annotations(self, index: int):  # to disk
        # print('_save_annotations')
        self._update_annotations(index)
        self._storage.save()

    def idx_changed(self, index: int):
        """ On index change save old state
            and update current state for visualisation
        """
        # print('_idx_changed')
        self._app_state.set_quietly('index', index)
        self.save_annotations(self._last_index)
        # update new screen
        self._update_im()
        self._update_state()
        self._last_index = index

    @debug_output.capture(clear_output=False)
    def handle_grid_click(self, event, name):
        # print('_handle_grid_click')
        label_changed = self._storage.label_dir / name

        if label_changed.is_dir():
            # button without image - invalid
            return

        label_changed = str(label_changed)
        current_label_state = self._im2im_state.annotations.copy()

        # inverse state
        current_label_state[label_changed] = {
            'answer': not self._im2im_state.annotations[label_changed].get('answer', False)
        }

        # change traitlets.Dict entirely to have change events issued
        self._im2im_state.annotations = current_label_state

    def handle_client_ready(self):
        self._update_im()
        self._update_state()

    def to_dict(self, only_annotated: bool) -> dict:
        return self._storage.to_dict(only_annotated)

# Cell

class Im2ImAnnotator:
    """
    Represents image-to-image annotator.

    Gives an ability to itarate through image dataset,
    map images with labels for classification,
    export final annotations in json format

    """

    def __init__(
        self,
        project_path: Path,
        input_item,
        output_item,
        annotation_file_path,
        n_rows=None,
        n_cols=None,
        label_autosize=False,
        question=None,
        has_border=False
    ):
        assert input_item, "WARNING: Provide valid Input"
        assert output_item, "WARNING: Provide valid Output"

        self.app_state = AppWidgetState(uuid=id(self))

        self.im2im_state = Im2ImState(
            uuid=id(self),
            **{
                "im_height": input_item.height,
                "im_width": input_item.width,
                "label_width": output_item.width,
                "label_height": output_item.height,
                "n_rows": n_rows,
                "n_cols": n_cols,
            }
        )

        self.storage = JsonLabelStorage(
            im_dir=project_path / input_item.dir,
            label_dir=project_path / output_item.dir,
            annotation_file_path=annotation_file_path
        )

        self.controller = Im2ImAnnotatorController(
            app_state=self.app_state,
            im2im_state=self.im2im_state,
            storage=self.storage,
            input_item=input_item,
            output_item=output_item,
            question=question,
        )

        self.view = Im2ImAnnotatorGUI(
            app_state=self.app_state,
            im2im_state=self.im2im_state,
            label_autosize=label_autosize,
            has_border=has_border
        )

        self.view.save_btn_clicked = self.controller.save_annotations
        self.view.grid_box_clicked = self.controller.handle_grid_click

        # link current image index from controls to annotator model
        self.view._navi.on_navi_clicked = self.controller.idx_changed

        # draw current image and bbox only when client is ready
        self.view.on_client_ready(self.controller.handle_client_ready)

    def __repr__(self):
        display(self.view)
        return ""

    def to_dict(self, only_annotated=True):
        return self.controller.to_dict(only_annotated)