# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/04_bbox_annotator.ipynb (unless otherwise specified).

__all__ = ['BBoxAnnotator']

# Internal Cell

import warnings
from pubsub import pub
from attr import asdict
from pathlib import Path
from copy import deepcopy
from typing import Optional, List

from IPython.core.display import display
from ipywidgets import AppLayout, Button, HBox, VBox, Layout

from .mltypes import BboxCoordinate
from .base import BaseState, AppWidgetState
from .bbox_canvas import BBoxCanvas, BBoxCanvasState
from .navi_widget import Navi
from .right_menu_widget import BBoxList, BBoxVideoItem
from .storage import JsonCaptureStorage
from .custom_input.buttons import ActionButton

# Internal Cell

class BBoxState(BaseState):
    coords: Optional[List[BboxCoordinate]]
    image: Optional[Path]
    classes: List[str]
    labels: List[List[str]] = []

# Internal Cell

class BBoxCoordinates(VBox):
    """Connects the BBoxList and the states"""

    def __init__(
        self,
        app_state: AppWidgetState,
        bbox_canvas_state: BBoxCanvasState,
        bbox_state: BBoxState,
        on_btn_select_clicked: callable = None
    ):
        super().__init__()

        self._app_state = app_state
        self._bbox_state = bbox_state
        self._bbox_canvas_state = bbox_canvas_state

        """CoordinateInput maximum values that a user
        can change. 'x' and 'y' can be improved to avoid
        bbox outside of the canvas area."""
        max_coord_input_values = BboxCoordinate(*[
            self._app_state.size[0],
            self._app_state.size[1],
            self._app_state.size[0],
            self._app_state.size[1]
        ])

        self._bbox_list = BBoxList(
            max_coord_input_values=max_coord_input_values,
            on_coords_changed=self.on_coords_change,
            on_label_changed=self.on_label_change,
            on_btn_delete_clicked=self.on_btn_delete_clicked,
            on_btn_select_clicked=on_btn_select_clicked,
            classes=bbox_state.classes
        )

        if self._bbox_canvas_state.bbox_coords:
            self._bbox_list.render_btn_list(
                self._bbox_canvas_state.bbox_coords,
                self._bbox_state.labels
            )

        app_state.subscribe(self._refresh_children, 'index')
        bbox_canvas_state.subscribe(self._sync_labels, 'bbox_coords')
        self.children = self._bbox_list.children
        self.layout = Layout(
            max_height=f'{self._app_state.size[1]}px',
            display='block'
        )

    def __getitem__(self, index: int) -> BBoxVideoItem:
        return self.children[index]

    def _refresh_children(self, index: int):
        self._bbox_list.clear()
        self._render(
            self._bbox_canvas_state.bbox_coords,
            self._bbox_state.labels
        )

    def _sync_labels(self, bbox_coords: List[BboxCoordinate]):
        """Every time a new coord is added to the annotator
        it's added an empty label to the state"""
        num_classes = len(self._bbox_state.labels)

        for i in bbox_coords[num_classes:]:
            self._bbox_state.labels.append([])

        self._render(bbox_coords, self._bbox_state.labels)

    def on_coords_change(self, index: int, key: str, value: int):
        setattr(self._bbox_canvas_state.bbox_coords[index], key, value)

        pub.sendMessage(
            f'{self._bbox_canvas_state.root_topic}.coord_changed',
            bbox_coords=self._bbox_canvas_state.bbox_coords
        )

    def _render(self, bbox_coords: list, labels: list):
        self._bbox_list.render_btn_list(bbox_coords, labels)
        self.children = self._bbox_list.children

    def on_label_change(self, change: dict, index: int):
        self._bbox_state.labels[index] = [change['new']]

    def remove_label(self, index: int):
        tmp_labels = deepcopy(self._bbox_state.labels)
        del tmp_labels[index]
        self._bbox_state.set_quietly('labels', tmp_labels)

    def on_btn_delete_clicked(self, index: int):
        bbox_coords = self._bbox_canvas_state.bbox_coords.copy()
        del bbox_coords[index]
        self.remove_label(index)
        self._bbox_canvas_state.bbox_coords = bbox_coords

# Internal Cell

class BBoxAnnotatorGUI(AppLayout):
    def __init__(
        self,
        app_state: AppWidgetState,
        bbox_state: BBoxState,
        on_save_btn_clicked: callable = None
    ):
        self._app_state = app_state
        self._bbox_state = bbox_state
        self._on_save_btn_clicked = on_save_btn_clicked
        self._label_history = []

        self._navi = Navi()

        self._save_btn = Button(description="Save",
                                layout=Layout(width='auto'))

        self._undo_btn = Button(description="Undo",
                                icon="undo",
                                layout=Layout(width='auto'))

        self._redo_btn = Button(description="Redo",
                                icon="rotate-right",
                                layout=Layout(width='auto'))

        self._controls_box = HBox(
            [self._navi, self._save_btn, self._undo_btn, self._redo_btn],
            layout=Layout(
                display='flex',
                flex_flow='row wrap',
                align_items='center'
            )
        )

        self._image_box = BBoxCanvas(*self._app_state.size)

        self.right_menu = BBoxCoordinates(
            app_state=self._app_state,
            bbox_canvas_state=self._image_box.state,
            bbox_state=self._bbox_state,
            on_btn_select_clicked=self._highlight_bbox
        )

        self._annotator_box = HBox(
            [self._image_box, self.right_menu],
            layout=Layout(
                display='flex',
                flex_flow='row'
            )
        )

        # set the values already instantiated on state
        if self._app_state.max_im_number:
            self._set_max_im_number(self._app_state.max_im_number)

        if self._bbox_state.image:
            self._set_image_path(self._bbox_state.image)

        # set the GUI interactions as callables
        self._navi.on_navi_clicked = self._idx_changed
        self._save_btn.on_click(self._save_clicked)
        self._undo_btn.on_click(self._undo_clicked)
        self._redo_btn.on_click(self._redo_clicked)

        bbox_state.subscribe(self._set_image_path, 'image')
        bbox_state.subscribe(self._set_coords, 'coords')
        app_state.subscribe(self._set_max_im_number, 'max_im_number')

        super().__init__(
            header=None,
            left_sidebar=None,
            center=self._annotator_box,
            right_sidebar=None,
            footer=self._controls_box,
            pane_widths=(2, 8, 0),
            pane_heights=(1, 4, 1))

    def _highlight_bbox(self, btn: ActionButton):
        self._image_box.highlight = btn.value

    def _redo_clicked(self, event: dict):
        self._image_box.redo_bbox()
        if self._label_history:
            self._bbox_state.labels[-1] = self._label_history.pop()
        self.right_menu._refresh_children(-1)

    def _undo_clicked(self, event: dict):
        if len(self._bbox_state.labels) > 0:
            self._label_history = [self._bbox_state.labels[-1]]
        self._image_box.undo_bbox()
        self.right_menu.remove_label(-1)
        self.right_menu._refresh_children(-1)

    def _set_image_path(self, image: str):
        self._image_box._state.image_path = image

    def _set_coords(self, coords: list):
        if coords:
            tmp_coords = deepcopy(self._image_box._state.bbox_coords)
            tmp_coords.append(coords)
            self._image_box._state.bbox_coords = coords

    def _set_max_im_number(self, max_im_number: int):
        # sync the navi GUI with AppWidgetState
        self._navi.max_im_num = max_im_number

    def _idx_changed(self, index: int):
        # store the last bbox drawn before index update
        self._bbox_state.set_quietly('coords', self._image_box._state.bbox_coords)
        self._app_state.index = index

    def _save_clicked(self, *args):
        if self._on_save_btn_clicked:
            self._on_save_btn_clicked(self._image_box._state.bbox_coords)
        else:
            warnings.warn("Save button click didn't triggered any event.")

    def on_client_ready(self, callback):
        self._image_box.observe_client_ready(callback)

# Internal Cell

class BBoxAnnotatorController:
    def __init__(
        self,
        app_state: AppWidgetState,
        bbox_state: BBoxState,
        storage: JsonCaptureStorage,
        render_previous_coords: bool = True,
        **kwargs
    ):
        self._app_state = app_state
        self._bbox_state = bbox_state
        self._storage = storage
        self._last_index = 0

        app_state.subscribe(self._idx_changed, 'index')

        self._update_im(self._last_index)
        self._app_state.max_im_number = len(self._storage)
        if render_previous_coords:
            self._update_coords(self._last_index)

    def save_current_annotations(self, coords: dict):
        self._bbox_state.set_quietly('coords', coords)
        self._save_annotations(self._app_state.index)

    def _update_im(self, index: int):
        self._bbox_state.image = self._storage.images[index]

    def _update_coords(self, index: int):  # from annotations
        image_path = str(self._storage.images[index])
        coords = self._storage.get(image_path) or {}
        self._bbox_state.labels = coords.get('labels', [])
        self._bbox_state.coords = [BboxCoordinate(**c) for c in coords.get('bbox', [])]

    def _save_annotations(self, index: int, *args, **kwargs):  # to disk
        image_path = str(self._storage.images[index])
        self._storage[image_path] = {
            'bbox': [asdict(bbox) for bbox in self._bbox_state.coords],
            'labels': self._bbox_state.labels
        }
        self._storage.save()

    def _idx_changed(self, index: int):
        """
        On index change save an old state and update
        current image path and bbox coordinates for
        visualisation
        """
        self._save_annotations(self._last_index)
        self._update_im(index)
        self._update_coords(index)
        self._last_index = index

# Cell

class BBoxAnnotator:
    """
    Represents bounding box annotator.

    Gives an ability to itarate through image dataset,
    draw 2D bounding box annotations for object detection and localization,
    export final annotations in json format

    """

    def __init__(
        self,
        project_path: Path,
        input_item,
        output_item,
        annotation_file_path: Path,
        *args, **kwargs
    ):
        self.app_state = AppWidgetState(
            uuid=id(self),
            **{
                'size': (input_item.width, input_item.height),
            }
        )

        self._input_item = input_item
        self._output_item = output_item

        self.bbox_state = BBoxState(
            uuid=id(self),
            classes=output_item.classes
        )

        self.storage = JsonCaptureStorage(
            im_dir=project_path / input_item.dir,
            annotation_file_path=annotation_file_path
        )

        self.controller = BBoxAnnotatorController(
            app_state=self.app_state,
            bbox_state=self.bbox_state,
            storage=self.storage,
            **kwargs
        )

        self.view = BBoxAnnotatorGUI(
            app_state=self.app_state,
            bbox_state=self.bbox_state,
            on_save_btn_clicked=self.controller.save_current_annotations
        )

    def __repr__(self):
        display(self.view)
        return ""

    def to_dict(self, only_annotated=True):
        return self.storage.to_dict(only_annotated)