# Production Deployment Guide

## Secrets Management

### Development (Current)
The `docker-compose.yaml` file contains hardcoded credentials for local development:

```yaml
environment:
  POSTGRES_DB: market_data
  POSTGRES_USER: postgres
  POSTGRES_PASSWORD: postgres
```

**⚠️ These credentials are for development only and should never be used in production.**

### Production Recommendations

#### 1. Kubernetes Secrets
```yaml
apiVersion: v1
kind: Secret
metadata:
  name: market-data-pipeline-secrets
type: Opaque
stringData:
  DATABASE_URL: "postgres://user:password@timescale:5432/market_data"
  API_KEY: "your-api-key"
```

#### 2. HashiCorp Vault
```bash
# Store secrets in Vault
vault kv put secret/market-data-pipeline \
  database_url="postgres://user:password@timescale:5432/market_data" \
  api_key="your-api-key"
```

#### 3. AWS Secrets Manager
```bash
aws secretsmanager create-secret \
  --name "market-data-pipeline/database" \
  --secret-string '{"url":"postgres://user:password@timescale:5432/market_data"}'
```

#### 4. Environment Variables (CI/CD)
```bash
# Set in your CI/CD pipeline
export DATABASE_URL="postgres://user:password@timescale:5432/market_data"
export API_KEY="your-api-key"
```

## Configuration Management

### Environment Variables
All configuration can be overridden via environment variables with the `PIPELINE_` prefix:

```bash
# Pipeline settings
PIPELINE_BATCH_SIZE=1000
PIPELINE_FLUSH_MS=50
PIPELINE_MAX_BYTES=1048576

# Queue settings
PIPELINE_OP_QUEUE_MAX=16
PIPELINE_SINK_QUEUE_MAX=32
PIPELINE_DROP_POLICY=newest

# Sink settings
PIPELINE_SINK_WORKERS=4
PIPELINE_SINK_QUEUE_MAX=200
PIPELINE_SINK_BACKPRESSURE_POLICY=drop_oldest

# Pacing settings
PIPELINE_PACING_BUDGET_MAX_MSGS_PER_SEC=2000
PIPELINE_PACING_BUDGET_BURST=2000

# Telemetry settings
PIPELINE_ENABLE_METRICS=true
PIPELINE_ENABLE_TRACING=true
PIPELINE_METRICS_PORT=8080
```

### Configuration Files
Use `config.yaml` for default values and environment variables for overrides:

```yaml
# config.yaml
pipeline:
  batch_size: 500
  flush_ms: 100
  max_bytes: 512000

queues:
  op_queue_max: 8
  sink_queue_max: 16
  drop_policy: "oldest"
```

## Monitoring & Observability

### Prometheus Metrics
The pipeline exposes comprehensive metrics at `/metrics`:

- `mdp_source_items_total` - Ticks ingested from sources
- `mdp_source_disconnects_total` - Source disconnections
- `mdp_source_pacing_blocked_total` - Pacing violations
- `operator_bars_out_total` - Bars generated by operators
- `mdp_sink_batches_committed_total` - Successful sink commits
- `mdp_sink_batches_failed_total` - Failed sink commits
- `mdp_sink_queue_depth` - Current sink buffer size
- `mdp_sink_commit_seconds` - Commit latency histogram

### Grafana Dashboard
Import the provided dashboard JSON (`monitoring/grafana-dashboard.json`) for:
- Pipeline throughput monitoring
- Queue depth visualization
- Commit latency tracking
- Error rate monitoring
- Source status overview

### Health Checks
- **API Health**: `GET /health` - Returns service status
- **Docker Health**: Built-in health check using `curl`
- **Database Health**: Connection status via metrics

## Deployment Strategies

### Docker Compose (Development)
```bash
# Start the complete stack
docker compose up --build -d

# Run smoke tests
./scripts/smoke_test.sh
```

### Kubernetes (Production)
```yaml
apiVersion: apps/v1
kind: Deployment
metadata:
  name: market-data-pipeline
spec:
  replicas: 3
  selector:
    matchLabels:
      app: market-data-pipeline
  template:
    metadata:
      labels:
        app: market-data-pipeline
    spec:
      containers:
      - name: pipeline
        image: market-data-pipeline:latest
        ports:
        - containerPort: 8083
        env:
        - name: DATABASE_URL
          valueFrom:
            secretKeyRef:
              name: market-data-pipeline-secrets
              key: database_url
        resources:
          requests:
            memory: "256Mi"
            cpu: "250m"
          limits:
            memory: "512Mi"
            cpu: "500m"
```

### Autoscaling (Future)
```yaml
apiVersion: keda.sh/v1alpha1
kind: ScaledObject
metadata:
  name: market-data-pipeline-scaler
spec:
  scaleTargetRef:
    name: market-data-pipeline
  minReplicaCount: 1
  maxReplicaCount: 10
  triggers:
  - type: prometheus
    metadata:
      serverAddress: http://prometheus:9090
      metricName: mdp_sink_queue_depth
      threshold: '100'
```

## Security Considerations

### Network Security
- Use TLS for all external communications
- Implement network policies in Kubernetes
- Use service mesh (Istio) for advanced traffic management

### Authentication & Authorization
- Implement API key authentication for external access
- Use RBAC for internal service communication
- Rotate credentials regularly

### Data Protection
- Encrypt data at rest (database encryption)
- Encrypt data in transit (TLS)
- Implement audit logging for compliance

## Performance Tuning

### Resource Allocation
```yaml
resources:
  requests:
    memory: "256Mi"
    cpu: "250m"
  limits:
    memory: "512Mi"
    cpu: "500m"
```

### Pipeline Configuration
- **Batch Size**: Increase for higher throughput, decrease for lower latency
- **Queue Sizes**: Balance memory usage vs. buffering capacity
- **Workers**: Scale based on CPU cores and I/O patterns
- **Pacing**: Adjust based on provider limits and downstream capacity

### Database Optimization
- Use connection pooling
- Implement read replicas for analytics
- Optimize indexes for time-series data
- Consider partitioning for large datasets

## Disaster Recovery

### Backup Strategy
- Database backups (automated)
- Configuration backups (Git)
- Metrics retention (Prometheus)

### Failover Procedures
- Multi-region deployment
- Database replication
- Circuit breakers for external dependencies
- Graceful degradation under load

## Troubleshooting

### Common Issues
1. **High Queue Depth**: Increase workers or batch size
2. **High Commit Latency**: Check database performance
3. **Source Disconnections**: Verify network connectivity
4. **Memory Leaks**: Check for unbounded queues

### Debug Commands
```bash
# Check pipeline status
curl http://localhost:8083/health

# View metrics
curl http://localhost:8083/metrics

# Check logs
docker logs market-data-pipeline

# Run diagnostics
./scripts/dev.sh test-integration
```
