# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dbdeclare',
 'dbdeclare.data_structures',
 'dbdeclare.entities',
 'dbdeclare.mixins']

package_data = \
{'': ['*']}

install_requires = \
['sqlalchemy>=2.0.0,<3.0.0']

setup_kwargs = {
    'name': 'dbdeclare',
    'version': '0.0.1',
    'description': 'Declarative layer for your database.',
    'long_description': '# DbDeclare\n\n<!-- [![Release](https://img.shields.io/github/v/release/raaidarshad/dbdeclare)](https://img.shields.io/github/v/release/raaidarshad/dbdeclare) -->\n\n[![Python version](https://img.shields.io/badge/python_version-3.11-blue)](https://github.com/psf/black)\n[![License](https://img.shields.io/github/license/raaidarshad/dbdeclare)](https://img.shields.io/github/license/raaidarshad/dbdeclare)\n[![PyPI](https://img.shields.io/pypi/v/dbdeclare?color=%2334D058&label=pypi%20package)](https://pypi.org/project/dbdeclare/)\n\nA declarative layer for your database, built on [SQLAlchemy](https://github.com/sqlalchemy/sqlalchemy).\n\nThe code is [on GitHub](https://github.com/raaidarshad/postgres-declare).\n\nThe docs are [hosted on GitHub pages](https://raaidarshad.github.io/dbdeclare/).\n\nThe project is published on [PyPI](https://pypi.org/project/dbdeclare/).\n\n## Overview\n\n### What is it?\n\nDbDeclare is a Python package that helps you create and manage entities in your database cluster,\nlike databases, roles, access control, and (eventually) more. It aims to fill the gap between\nSQLAlchemy (SQLA) and infrastructure as code (IaC).\n\n### Why use it?\n\nDbDeclare does what SQLA does but for database entities beyond tables and columns. You can:\n\n- Declare desired state in Python\n- Avoid maintaining raw SQL\n- Tightly integrate your databases, roles, access control, and more with your tables\n\nFuture versions will have more features, and you will be able to:\n\n- Have version control over database changes (like [Alembic](https://github.com/sqlalchemy/alembic))\n- Define upgrades/downgrades without explicitly defining the changes (like [autogen](https://alembic.sqlalchemy.org/en/latest/autogenerate.html))\n\nAdditionally, DbDeclare is:\n\n- Typed: Type-checking done via [mypy](https://mypy.readthedocs.io/en/stable/)\n- Thoroughly commented: There are docstrings for every method and class\n- Well-tested: Though this is a new package under active development, solid test coverage is a high priority\n\nRunning SQL scripts before SQLA and after IaC can be messy and hard to maintain.\nIf you prefer to have databases, roles, and the like declared alongside your infrastructure, then there are\ngreat tools available for that, like Terraform and Pulumi\'s providers for Postgres and MySQL. So if you want\nit tied to that, great! But if, like me, you want it closer to your application code and alongside SQLA, this\ntool likely makes more sense for you.\n\n## Requirements\n\nThis requires a recent version of Python. Works with Python 3.11 or higher. We recommend\n[pyenv](https://github.com/pyenv/pyenv) to install any versions of Python you need and don\'t currently have installed.\n\nThis also requires a compatible driver/package for your database of choice, like\n[psycopg](https://www.psycopg.org/) for Postgres. You\'ll also need a functioning cluster\nto interact with (the example below shows a way to do so via Docker).\n\nSQLAlchemy is a dependency and will be installed when you install DbDeclare. DbDeclare\nworks with SQLAlchemy 2.0.0 or higher.\n\n## Installation\n\nDbDeclare is published on [PyPI](https://pypi.org/project/dbdeclare/). You can install it with `pip` or any tool\nthat uses `pip` under the hood. This is typically installed in a [virtual environment](https://docs.python.org/3/library/venv.html).\n\n```\n> pip install db-declare\n```\n\n## Example\n\nHere is a simple Postgres example. We will create a database and a user, and make sure the user\ncan connect to the database. You need a Postgres cluster/instance and a python environment.\n\nIf needed, an easy way to spin up a Postgres instance is with [Docker](https://www.docker.com/),\nspecifically the [official Postgres image](https://hub.docker.com/_/postgres):\n\n```\n> docker run --rm --name postgres -e POSTGRES_PASSWORD=postgres -p 127.0.0.1:5432:5432/tcp postgres\n```\n\nThis spins up a Postgres instance with the default database name of `postgres`, an admin user of `postgres` with the\npassword `postgres`, on port `5432`.\n\nAssuming you have a Python environment set up, DbDeclare installed, and psycopg installed (`pip install psycopg`),\nyou can create a database and a user that can connect to it like this:\n\n```Python\nfrom sqlalchemy import create_engine\n\nfrom dbdeclare.controller import Controller\nfrom dbdeclare.data_structures import GrantOn, Privilege\nfrom dbdeclare.entities import Database, Role\n\n\ndef main() -> None:\n    # declare the database\n    falafel_db = Database(name="falafel")\n    # declare the user\n    Role(\n        name="hungry_user",\n        login=True,  # (1)!\n        password="fakepassword",  # (2)!\n        grants=[GrantOn(privileges=[Privilege.CONNECT], on=[falafel_db])],  # (3)!\n    )\n\n    # create engine with admin user and default database\n    engine = create_engine(url="postgresql+psycopg://postgres:postgres@127.0.0.1:5432/postgres")  # (4)!\n    # create all entities and grant all privileges\n    Controller.run_all(engine=engine)\n\n\nif __name__ == "__main__":\n    main()\n```\n\n1. Make sure this role can log in (make it a user)\n2. Provide a password for the user to log in with\n3. Specify that this user can connect to the `falafel` database\n4. The engine to run DbDeclare must have admin privileges, so we use the `postgres` user here\n\nAfter running this script, you should be able to access the `falafel` database as `hungry_user`. You can try it out with\n`psql` (if you don\'t have it installed, find it [here](https://www.timescale.com/blog/how-to-install-psql-on-mac-ubuntu-debian-windows/)).\nIn a separate shell from where the docker run command is running, you can run:\n\n```\n> psql -h 127.0.0.1 -p 5432 -U hungry_user -d falafel\n\npassword for user hungry_user: ***\n\nfalafel=>\n```\n\nVoila! Check out the [user guide](https://raaidarshad.github.io/dbdeclare/guide) for more involved use cases.\n\n## Contributing\n\nCheck out development, testing, and contributing guidance [here](https://raaidarshad.github.io/dbdeclare/contributing).\n\n## License\n\nThis project is licensed under the terms of the [MIT license](https://github.com/raaid/dbdeclare/blob/main/LICENSE).',
    'author': 'Raaid Arshad',
    'author_email': 'raaid@protonmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://raaidarshad.github.io/dbdeclare/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.11.0,<4.0.0',
}


setup(**setup_kwargs)
