# -*- coding: utf-8 -*-

""" Hyperparameter encoder and decoder.
"""

import json


class HyperParameterEncodingError(Exception):
    """ Raised when hyperparameters cannot be successfully encoded. """


def encode(params):
    """ Converts a JSON-encodable dictionary of parameters into one that SageMaker's
        CreateTrainingJob API can accept (the API accepts only string keys and
        values, both may be at most 256 characters, and there may be at most 100
        keys).

        :param dict params:
            A possible nested JSON-encodable dictionary of parameters containing
            only string keys without dots.
        :return dict:
            A flat dictionary of parameters with only strings as key and values.
            Each key and value has at most 256 characters and there are at most
            100 parameters.

        The mapping proceeds as follows:

        * Each key must already be a string without dots (".").
        * Values that are not dictionaries are stored JSON-encoded.
        * Values that are dictionaries have all their sub-keys added with a
          prefix of the original key, followed by a dot (".").

        The inverse function is `decode(...)`.
    """
    result = {}
    for k, v in params.items():
        _check_key_format(k)
        if isinstance(v, dict):
            for subk, subv in encode(v).items():
                subk = "{}.{}".format(k, subk)
                _check_key_length(subk)
                result[subk] = subv
        else:
            v = json.dumps(v)
            _check_key_length(k)
            _check_value_length(k, v)
            result[k] = v
    _check_encoded_params(result)
    return result


def decode(params):
    """ Converts a dictionary of SageMaker hyperparameters generated by `encode(...)`
        to a JSON-encodable dictionary of parameters.
    """
    result = {}
    for k, v in params.items():
        parts = k.split(".")
        k, d = parts[0], result
        for subk in parts[1:]:
            d = d.setdefault(k, {})
            k = subk
        d[k] = json.loads(v)
    return result


def _check_key_format(k):
    if not isinstance(k, str) or "." in k:
        raise HyperParameterEncodingError(
            "Key {!r} must be a string without dots ('.')".format(k)
        )


def _check_key_length(k):
    if len(k) > 256:
        raise HyperParameterEncodingError(
            "Key {!r} must be at most 256 characters.".format(k)
        )


def _check_value_length(k, v):
    if len(v) > 256:
        raise HyperParameterEncodingError(
            "Value {!r} for key {!r} must have length at most 256.".format(v, k)
        )


def _check_encoded_params(d):
    if len(d) > 100:
        raise HyperParameterEncodingError(
            "SageMaker allows at most 100 hyperparameters but {}"
            " were specified.".format(len(d))
        )
