from abc import ABC
from abc import abstractmethod
from typing import Dict, List, OrderedDict, Tuple

from tiledb import Attr


class ObjectPartition(ABC):
    """
    Abstract class for capturing object partitions
    """

    @abstractmethod
    def init_kwargs(self) -> Dict:
        """
        Returns a dictionary containing kwargs that can be used to re-initialize the ObjectPartition.

        This is used to serialize the ObjectPartition and pass it as argument to UDF tasks.
        """
        raise NotImplementedError

    @abstractmethod
    def id(self) -> int:
        """
        Returns the id of the ObjectPartition.
        """
        raise NotImplementedError


class ObjectReader(ABC):
    """
    Abstract class that can be used to read Objects from different sources and formats.
    """

    @abstractmethod
    def init_kwargs(self) -> Dict:
        """
        Returns a dictionary containing kwargs that can be used to re-initialize the ObjectReader.

        This is used to serialize the ObjectReader and pass it as argument to UDF tasks.
        """
        raise NotImplementedError

    @abstractmethod
    def partition_class_name(self) -> str:
        """
        Returns the class name of ObjectPartition generated by this ObjectReader.

        The ObjectPartition class should be defined in the same Python file as the ObjectReader.
        """
        raise NotImplementedError

    @abstractmethod
    def metadata_array_uri(self) -> str:
        """
        Returns the URI of a TileDB array that can be used to read Object metadata.
        This array should have only one `external_id` dimension and attributes the list of TileDB attributes returned by `metadata_attributes`.

        Returns None, if a metadata array does not exist and should be materialized by object ingestion.
        """
        raise NotImplementedError

    @abstractmethod
    def metadata_attributes(self) -> List[Attr]:
        """
        Returns a list of TileDB Attributes describing the metadata of the Objects.

        Returns None, if there are no Object metadata.
        """
        raise NotImplementedError

    @abstractmethod
    def get_partitions(self, **kwargs) -> List[ObjectPartition]:
        """
        Returns a list of ObjectPartitions for the reader.
        Each partition can be read independently and used for distributed embedding and ingestion.
        """
        raise NotImplementedError

    @abstractmethod
    def read_objects(
        self, partition: ObjectPartition
    ) -> Tuple[OrderedDict, OrderedDict]:
        """
        Reads the objects corresponding to an ObjectPartition.

        Returns a tuple containing the object data and metadata respectively.
        Data and metadata are OrderedDicts having structure similar to TileDB-Py read results.
        Data and metadata should contain at least an `external_id` dimension used to identify the different objects.
        """
        raise NotImplementedError

    @abstractmethod
    def read_objects_by_external_ids(self, ids: List[int]) -> OrderedDict:
        """
        Reads the objects corresponding to a list of `external_ids`.

        Returns an OrderedDict, containing the object data, having structure similar to TileDB-Py read results.
        """
        raise NotImplementedError
