#!/usr/bin/env python3
#
# Copyright 2021 Graviti. Licensed under MIT License.
#

"""Vector, Vector2D, Vector3D.

:class:`Vector` is the base class of :class:`Vector2D` and :class:`Vector3D`. It contains the
coordinates of a 2D vector or a 3D vector.

:class:`Vector2D` contains the coordinates of a 2D vector, extending :class:`Vector`.

:class:`Vector3D` contains the coordinates of a 3D vector, extending :class:`Vector`.

"""

from itertools import zip_longest
from typing import Dict, Iterable, Optional, Sequence, Tuple, Type, TypeVar, Union

from ..utility import ReprType, UserSequence

_V = TypeVar("_V", bound="Vector")
_V2 = TypeVar("_V2", bound="Vector2D")
_V3 = TypeVar("_V3", bound="Vector3D")

_T = Union["Vector2D", "Vector3D"]


class Vector(UserSequence[float]):
    """This class defines the basic concept of Vector.

    :class:`Vector` contains the coordinates of a 2D vector or a 3D vector.

    Arguments:
        x: The x coordinate of the vector.
        y: The y coordinate of the vector.
        z: The z coordinate of the vector.

            .. code:: python

                vector2d = Vector(x=1, y=2)
                vector3d = Vector(x=1, y=2, z=3)

    """

    _data: Tuple[float, ...]

    _repr_type = ReprType.INSTANCE

    _DIMENSION: Optional[int] = None

    def __new__(  # type: ignore[misc]  # pylint: disable=unused-argument
        cls: Type[_V],
        x: float,
        y: float,
        z: Optional[float] = None,
    ) -> _T:
        """Create a new instance of :class:`Vector`.

        Arguments:
            x: The x coordinate of the vector.
            y: The y coordinate of the vector.
            z: The z coordinate of the vector.

                .. code:: python

                    vector2d = Vector(x=1, y=2)
                    vector3d = Vector(x=1, y=2, z=3)

        Returns:
            The created :class:`Vector2D` or :class:`Vector3D` object.

        """
        ReturnType = Vector2D if z is None else Vector3D

        obj: _T = object.__new__(ReturnType)
        return obj

    def __bool__(self) -> bool:
        return any(self._data)

    def __neg__(self: _V) -> _V:
        result: _V = object.__new__(self.__class__)
        result._data = tuple(-coordinate for coordinate in self._data)
        return result

    def __eq__(self, other: object) -> bool:
        if not isinstance(other, self.__class__):
            return False

        return self._data.__eq__(other._data)

    def __add__(self: _V, other: Iterable[float]) -> _V:
        """Calculate the sum of the vector and other vector.

        Arguments:
            other: The added vector.

        Returns:
            The sum vector of adding two vectors.

        """
        try:
            result: _V = object.__new__(self.__class__)
            result._data = tuple(i + j for i, j in zip_longest(self._data, other))
            return result
        except TypeError:
            return NotImplemented

    def __radd__(self: _V, other: Sequence[float]) -> _V:
        """Calculate the sum of the vector and the other vector.

        Arguments:
            other: The added vector.

        Returns:
            The sum vector of adding two vectors.

        """
        return self.__add__(other)

    def _repr_head(self) -> str:
        return f"{self.__class__.__name__}{self._data}"

    @staticmethod
    def loads(contents: Dict[str, float]) -> _T:
        """Loads a :class:`Vector` from a dict containing coordinates of the vector.

        Arguments:
            contents: A dict containing coordinates of the vector::

                {
                    "x": ...
                    "y": ...
                }
                or
                {
                    "x": ...
                    "y": ...
                    "z": ...
                }

        Returns:
            The loaded :class:`Vector2D` or :class:`Vector3D` object.

        """
        if "z" in contents:
            return Vector3D.loads(contents)
        return Vector2D.loads(contents)


class Vector2D(Vector):
    """This class defines the concept of Vector2D.

    :class:`Vector2D` contains the coordinates of a 2D vector.

    Arguments:
        x: The x coordinate of the 2D vector.
        y: The y coordinate of the 2D vector.

    """

    _DIMENSION = 2

    def __new__(  # pylint: disable=unused-argument
        cls: Type[_V2], *args: float, **kwargs: float
    ) -> _V2:
        """Create a :class:`Vector2D` instance.

        Arguments:
            args: The coordinates of the 2D vector.
            kwargs: The coordinates of the 2D vector.

        Returns:
            The created :class:`Vector2D` instance.

        """
        obj: _V2 = object.__new__(cls)
        return obj

    def __init__(self, x: float, y: float) -> None:
        self._data = (x, y)

    @classmethod
    def loads(cls: Type[_V2], contents: Dict[str, float]) -> _V2:
        """Load a :class:`Vector2D` object from a dict containing coordinates of a 2D vector.

        Arguments:
            contents: A dict containing coordinates of a 2D vector::

                {
                    "x": ...
                    "y": ...
                }

        Returns:
            The loaded :class:`Vector2D` object.

        """
        return cls(**contents)

    @property
    def x(self) -> float:
        """Return the x coordinate of the vector.

        Returns:
            X coordinate in float type.

        """
        return self._data[0]

    @property
    def y(self) -> float:
        """Return the y coordinate of the vector.

        Returns:
            Y coordinate in float type.

        """
        return self._data[1]

    def dumps(self) -> Dict[str, float]:
        """Dumps the vector into a dict.

        Returns:
            A dict containing the vector coordinate.

        """
        return {"x": self._data[0], "y": self._data[1]}


class Vector3D(Vector):
    """This class defines the concept of Vector3D.

    :class:`Vector3D` contains the coordinates of a 3D Vector.

    Arguments:
        x: The x coordinate of the 3D vector.
        y: The y coordinate of the 3D vector.
        z: The z coordinate of the 3D vector.

    """

    _DIMENSION = 3

    def __new__(  # pylint: disable=unused-argument
        cls: Type[_V3], *args: float, **kwargs: float
    ) -> _V3:
        """Create a :class:`Vector3D` instance.

        Arguments:
            args: The coordinates of the 3D vector.
            kwargs: The coordinates of the 3D vector.

        Returns:
            The created :class:`Vector3D` instance.

        """
        obj: _V3 = object.__new__(cls)
        return obj

    def __init__(self, x: float, y: float, z: float) -> None:
        self._data = (x, y, z)

    @classmethod
    def loads(cls: Type[_V3], contents: Dict[str, float]) -> _V3:
        """Load a :class:`Vector3D` object from a dict containing coordinates of a 3D vector.

        Arguments:
            contents: A dict contains coordinates of a 3D vector::

                {
                    "x": ...
                    "y": ...
                    "z": ...
                }

        Returns:
            The loaded :class:`Vector3D` object.

        """
        return cls(**contents)

    @property
    def x(self) -> float:
        """Return the x coordinate of the vector.

        Returns:
             X coordinate in float type.

        """
        return self._data[0]

    @property
    def y(self) -> float:
        """Return the y coordinate of the vector.

        Returns:
            Y coordinate in float type.

        """
        return self._data[1]

    @property
    def z(self) -> float:
        """Return the z coordinate of the vector.

        Returns:
            Z coordinate in float type.

        """
        return self._data[2]

    def dumps(self) -> Dict[str, float]:
        """Dumps the vector into a dict.

        Returns:
            A dict containing the vector coordinates.

        """
        return {"x": self._data[0], "y": self._data[1], "z": self._data[2]}
