# -*- coding: utf-8 -*-

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~IMPORTS~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~#

# Standard library imports
from collections import OrderedDict, namedtuple, Counter
import hashlib

# Third party imports
from tqdm import tqdm
import numpy as np
import pandas as pd
import jinja2

# Plotly imports
import plotly.graph_objs as go
from plotly.subplots import make_subplots
from plotly.colors import n_colors
import plotly.figure_factory as ff
import plotly.offline as py

# Local imports
from pycoMeth import __version__ as version
from pycoMeth.common import *

#~~~~~~~~~~~~~~~~~~~~~~~~Main Function~~~~~~~~~~~~~~~~~~~~~~~~#

def Comp_Report (
    methcomp_fn:str,
    gff3_fn:str,
    outdir:str="",
    n_top:int=50,
    max_tss_distance:int=100000,
    pvalue_threshold:float=0.01,
    min_diff_llr:float=1,
    verbose:bool=False,
    quiet:bool=False,
    progress:bool=False,
    **kwargs):
    """
    * methcomp_fn
        Input tsv file generated by Meth_comp (can be gzipped). At the moment only data binned by intervals with Interval_Aggregate are supported.
    * gff3_fn
        Path to an **ensembl GFF3** file containing genomic annotations. Only the transcripts details are extracted.
    * outdir
        Directory where to output HTML reports, By default current directory
    * n_top
        Number of top interval candidates for which to generate an interval report.
        If there are not enough significant candidates this is automatically scaled down.
    * max_tss_distance
        Maximal distance to transcription stat site to find transcripts close to interval candidates
    * pvalue_threshold
        pValue cutoff for top interval candidates
    * min_diff_llr
        Minimal llr boundary for negative and positive median llr. 1 is recommanded for vizualization purposes.
    """

    # Init method
    opt_summary_dict = opt_summary(local_opt=locals())
    log = get_logger (name="pycoMeth_CpG_Comp", verbose=verbose, quiet=quiet)

    log.warning("Checking options and input files")
    log_dict(opt_summary_dict, log.debug, "Options summary")

    # Create output directory
    mkdir (outdir, exist_ok=True)
    summary_fn = "pycoMeth_summary.html"

    log.warning("Loading and preparing data")

    # Parse methcomp data
    log.info("Loading Methcomp data from TSV file")
    df = pd.read_table(methcomp_fn)

    # Check that the input file was generated by methcomp from samples aggregated with Interval_Aggregate
    req_fields = [
        "chromosome","start","end","n_samples","pvalue","adj_pvalue","neg_med","pos_med",
        "ambiguous_med","unique_cpg_pos","labels","med_llr_list","raw_llr_list","raw_pos_list"]
    if not all_in(req_fields, df.columns):
        raise pycoMethError ("Invalid input file type passed. Expecting Meth_Comp TSV file generated from samples processed with Interval_Aggregate")

    # Parse GFF3 annotations
    log.info("Loading transcript info from GFF file")
    tx_df = get_ensembl_tx(gff3_fn)

    # Check that the same chromosomes ids are found in both the annotation and data file
    if not all_in(df["chromosome"], tx_df["chromosome"]):
        log.error ("Not all the chromosomes found in the data file are present in the GFF3 file. This will lead to missing transcript ids")

    # Select only significant sites
    sig_df = df[df.adj_pvalue <= pvalue_threshold]
    log.info("Number of significant intervals found (adjusted pvalue<{}): {}".format(pvalue_threshold, len(sig_df)))

    if len(sig_df)<5:
        log.error("Low number of significant sites. The summary report will likely contains errors")
    if len(sig_df)<n_top:
        log.error("Number of significant intervals lower than number of top candidates to plot")

    # List top candidates
    log.info("Generating file names for top candidates reports")
    top_dict = OrderedDict()
    top_candidates = sig_df.sort_values(by="adj_pvalue").head(n_top)
    for rank, (idx, line) in enumerate(top_candidates.iterrows()):
        coord = "{}-{}-{}".format(line.chromosome,line.start,line.end)
        fn = "pycoMeth_interval_{:04}_chr{}.html".format(rank+1, coord)
        top_dict[idx] = {"rank":rank+1, "coord":coord, "fn":fn}
    rank_fn_dict = {i["rank"]:i["fn"] for i in top_dict.values()}

    # Prepare src file for report and compute md5
    log.info("Computing source md5")
    src_file = os.path.abspath(methcomp_fn)
    md5 = md5_str(methcomp_fn)

    # Init collections
    all_cpg_dict = OrderedDict()
    top_cpg_list = []

    # Extract info from each intervals
    log.warning("Parsing methcomp data")
    log.info("Iterating over significant intervals and generating top candidates reports")
    for idx, line in tqdm(sig_df.iterrows(), total=len(sig_df), unit=" intervals", unit_scale=True, desc="\tProgress", disable=not progress):

        # collect median llr for all significant intervals
        lab_list = ["Sample {}".format(lab) for lab in str_to_list(line.labels)]
        med_list = str_to_list(line.med_llr_list)
        coord = "{}-{:,}-{:,}".format(line.chromosome,line.start,line.end)
        all_cpg_dict[coord] = {lab:llr for lab, llr in zip(lab_list, med_list)}

        # Extract more data for reports of top results
        if idx in top_dict:
            log.debug (f"Ploting top candidates: {coord}")

            # Extract data from line
            cpg_df = get_cpg_df(line)

            # Generate figures and tables
            rank = rank=top_dict[idx]["rank"]
            out_file = os.path.join(outdir, top_dict[idx]["fn"])
            heatmap_fig = cpg_heatmap(cpg_df, lim_llr=10, min_diff_llr=min_diff_llr)
            ridgeplot_fig = cpg_ridgeplot(cpg_df, box=False, scatter=True, min_diff_llr=min_diff_llr)
            interval_df = get_interval_df(line=line, rank=rank)
            transcript_df = get_close_tx_df(tx_df=tx_df, chromosome=line.chromosome, start=line.start, end=line.end, max_tss_distance=max_tss_distance)

            # Collect Interval minimal info
            top_cpg_list.append((rank, line, transcript_df, top_dict[idx]["fn"]))

            # Render interval HTML report
            write_cpg_interval_html(
                out_file = out_file,
                src_file = src_file,
                md5 = md5,
                summary_link = summary_fn,
                previous_link = prev_fn(rank_fn_dict, rank),
                next_link = next_fn(rank_fn_dict, rank),
                max_tss_distance = max_tss_distance,
                interval_df = interval_df,
                transcript_df = transcript_df,
                heatmap_fig = heatmap_fig,
                ridgeplot_fig = ridgeplot_fig)

    # Collect data at CpG interval level
    log.info("Generating summary report")

    all_cpg_df = pd.DataFrame.from_dict(all_cpg_dict)
    # # Generate figures and tables
    all_heatmap_fig = cpg_heatmap(all_cpg_df, lim_llr=4, min_diff_llr=min_diff_llr)
    all_ridgeplot_fig = cpg_ridgeplot(all_cpg_df, box=True, scatter=False, min_diff_llr=min_diff_llr)
    catplot_fig = category_barplot(all_cpg_df, min_diff_llr=min_diff_llr)
    summary_df = get_summary_df(df, sig_df)
    top_df = get_top_df(top_cpg_list)

    # Render interval HTML report
    write_summary_html(
        out_file = os.path.join(outdir, summary_fn),
        src_file = src_file,
        md5 = md5,
        summary_df = summary_df,
        top_df = top_df,
        catplot_fig = catplot_fig,
        heatmap_fig = all_heatmap_fig,
        ridgeplot_fig = all_ridgeplot_fig)

#~~~~~~~~~~~~~~~~~~~~~~~~HTML generating functions~~~~~~~~~~~~~~~~~~~~~~~~#

def write_cpg_interval_html (out_file, src_file, md5, summary_link, previous_link, next_link, max_tss_distance, interval_df, transcript_df, heatmap_fig, ridgeplot_fig):
    """Write CpG interval HTML report"""
    # Get CpG_Interval template
    template = get_jinja_template ("CpG_Interval.html.j2")

    # Render pandas dataframes and plotly figures to HTML
    interval_html = render_df(interval_df)
    transcript_html = render_df(transcript_df, empty_msg=f"No transcripts TTS found within {max_tss_distance} bp upstream or downstream")
    heatmap_html = render_fig (heatmap_fig)
    ridgeplot_html = render_fig (ridgeplot_fig)

    # Render HTML report using Jinja
    rendering = template.render(
        plotlyjs = py.get_plotlyjs(),
        version = version,
        date = datetime.datetime.now().strftime("%d/%m/%y"),
        src_file = src_file,
        md5 = md5,
        summary_link = summary_link,
        previous_link = previous_link,
        next_link = next_link,
        interval_html = interval_html,
        transcript_html = transcript_html,
        heatmap_html = heatmap_html,
        ridgeplot_html = ridgeplot_html)

    with open(out_file, "w") as fp:
        fp.write(rendering)

def write_summary_html (out_file, src_file, md5, summary_df, top_df, catplot_fig, heatmap_fig, ridgeplot_fig):
    """Write summary HTML report"""
    # Get CpG_Interval template
    template = get_jinja_template("CpG_summary.html.j2")

    # Render pandas dataframes and plotly figures to HTML
    summary_html = render_df(summary_df)
    top_html = render_df(top_df, empty_msg="No significant candidates found")
    catplot_html = render_fig (catplot_fig, empty_msg= "Not enough significant candidates to render catplot")
    heatmap_html = render_fig (heatmap_fig, empty_msg= "Not enough significant candidates to render heatmap")
    ridgeplot_html = render_fig (ridgeplot_fig, empty_msg= "Not enough significant candidates to render ridgeplot")

    # Render HTML report using Jinja
    rendering = template.render(
        plotlyjs = py.get_plotlyjs(),
        version = version,
        date = datetime.datetime.now().strftime("%d/%m/%y"),
        src_file = src_file,
        md5 = md5,
        summary_html = summary_html,
        top_html = top_html,
        catplot_html = catplot_html,
        heatmap_html = heatmap_html,
        ridgeplot_html = ridgeplot_html)

    with open(out_file, "w") as fp:
        fp.write(rendering)

#~~~~~~~~~~~~~~~~~~~~~~~~GFF parsing functions~~~~~~~~~~~~~~~~~~~~~~~~#

def get_ensembl_tx (gff_fn):
    """Simple parsing function transcript data from ensembl GFF3 files"""
    l = []
    open_fun, open_mode = (gzip.open, "rt") if gff_fn.endswith(".gz") else (open, "r")

    with open_fun (gff_fn, open_mode) as fp:

        # Check that file is a ggf3
        first_line = next(fp).strip()
        if not first_line.startswith ("##gff-version 3"):
            raise pycoMethError("Invalid annotation file format, please provide an Ensembl GFF3 file")

        for line in fp:
            if not line.startswith("#"):
                ls = line.strip().split("\t")

                # Define transcript as feature type containing RNA or transcript and with a gene as a Parent
                if ("RNA" in ls[2] or "transcript" in ls[2]) and "Parent=gene" in ls[8]:

                    d = OrderedDict()
                    d["chromosome"] = ls[0]
                    d["strand"] = ls[6]
                    d["start"] = int(ls[3])
                    d["end"] = int(ls[4])
                    d["tss"] = d["start"] if d["strand"] == "+" else d["end"]
                    d["feature type"] = ls[2]

                    # Parse attributes
                    attrs = OrderedDict()
                    try:
                        for a in ls[8].split(";"):
                            i, j = a.split("=")
                            attrs[i.lower()] = j.lower()
                    except Exception:
                        pass

                    # Extract specific attrs
                    d["transcript id"] = attrs.get("id", "-").strip("transcript:")
                    d["gene id"] = attrs.get("parent", "-").strip("gene:")
                    d["transcript biotype"] = attrs.get("biotype", "-")
                    d["transcript name"] = attrs.get("name", "-")
                    l.append(d)

        df = pd.DataFrame(l)
        df = df.fillna("")
        return df

#~~~~~~~~~~~~~~~~~~~~~~~~DataFrame functions~~~~~~~~~~~~~~~~~~~~~~~~#

def get_cpg_df (line):
    """"""
    lab_list = ["Sample {}".format(lab) for lab in str_to_list(line.labels)]
    llr_list = str_to_list(line.raw_llr_list)
    pos_list = str_to_list(line.raw_pos_list)

    cpg_df = pd.DataFrame()
    for lab, llr, pos in zip(lab_list, llr_list, pos_list):
        pos = ["{}-{:,}".format(line.chromosome,pos) for pos in pos]
        cpg_sdf = pd.DataFrame(index=pos, data=llr, columns=[lab])
        if cpg_df.empty:
            cpg_df = cpg_sdf
        else:
            cpg_df = pd.merge(cpg_df, cpg_sdf, left_index=True, right_index=True, how="outer")

    return cpg_df.T

def get_close_tx_df (tx_df, chromosome, start, end, max_tss_distance=100000):
    """Find transcripts with a TSS within a given genomic interval"""
    rdf = tx_df.query("chromosome == '{}' and tss >= {} and tss <= {}".format(chromosome, start-max_tss_distance, end+max_tss_distance))
    tss_dist = []
    for tx, line in rdf.iterrows():
        if line.tss > end:
            tss_dist.append(line.tss-end)
        elif line.tss < start:
            tss_dist.append(line.tss-start)
        else:
            tss_dist.append(0)
    rdf["distance to tss"] = tss_dist
    rdf["abs_tss"] = np.abs(tss_dist)
    rdf.sort_values("abs_tss", inplace=True)
    rdf = rdf[["distance to tss", "transcript id","gene id","transcript name","chromosome","start","end","strand","feature type","transcript biotype"]]
    return rdf

def get_interval_df (line, rank):
    """Generate a single line dataframe describing the current interval"""
    line["pvalue rank"] = f"#{rank}"
    line["length"] = line["end"]-line["start"]
    line = line[["pvalue rank", "chromosome", "start", "end", "length", "n_samples", "unique_cpg_pos", "pvalue", "adj_pvalue"]]
    df = line.to_frame().T
    df = df.rename(columns={"n_samples":"number of samples", "unique_cpg_pos":"number of CpGs", "adj_pvalue":"adjusted pvalue"})
    return df

def get_summary_df (df, sig_df):
    """Generate a single line dataframe with summary information"""
    s = pd.Series()
    s.loc["Total intervals"] = "{:,}".format(len(df))
    s.loc["Total CpGs"] = "{:,}".format(df["unique_cpg_pos"].sum())
    if sig_df.empty:
        s.loc["Significant intervals"] = 0
        s.loc["CpGs in significant intervals"] = 0
    else:
        s.loc["Significant intervals"] = "{:,}".format(len(sig_df))
        s.loc["CpGs in significant intervals"] = "{:,}".format(df["unique_cpg_pos"].sum())

    return s.to_frame().T

def get_top_df (top_cpg_list):
    """Generate a dataframe containing information for the top interval candidates"""
    # sort by pvalue rank
    top_cpg_list.sort(key=lambda r: r[0])
    # collect data
    l = []
    for rank, line, transcript_df, out_file in top_cpg_list:
        d = OrderedDict()
        d["rank"] = f"#{rank}"
        d["detailled report"] = f"<a href='{out_file}'>report link</a>"
        d["chromosome"] = line["chromosome"]
        d["start"] = line["start"]
        d["end"] = line["end"]
        d["pvalue"] = line["adj_pvalue"]

        if transcript_df.empty:
            d["closest transcript id"] = "-"
            d["closest transcript name"] = "-"
            d["distance to tss"] = "-"
        else:
            d["closest transcript id"] = transcript_df.iloc[0]["transcript id"]
            d["closest transcript name"] = transcript_df.iloc[0]["transcript name"]
            d["distance to tss"] = transcript_df.iloc[0]["distance to tss"]
        l.append(d)

    df = pd.DataFrame(l)
    return df

#~~~~~~~~~~~~~~~~~~~~~~~~Plotting functions~~~~~~~~~~~~~~~~~~~~~~~~#

def cpg_heatmap (
    df,
    methylated_color:str = 'rgb(215,48,39)',
    unmethylated_color:str = 'rgb(33,102,172)',
    ambiguous_color:str = 'rgb(240,240,240)',
    lim_llr:float = 10,
    min_diff_llr:float = 1,
    fig_width:int=None,
    fig_height:int=None):
    """
    Plot the values per CpG as a heatmap
    """
    # Cannot calculate if at least not 2 values
    if len(df.columns) <= 1:
        return None

    # Fill missing values by 0 = ambiguous methylation
    df = df.fillna(0)

    # Prepare subplot aread
    fig = make_subplots(
        rows=1,
        cols=2,
        shared_yaxes=True,
        column_widths=[0.95, 0.05],
        specs=[[{"type": "heatmap"},{"type": "scatter"}]])

    # Plot dendogramm
    dendrogram = ff.create_dendrogram(df.values, labels=df.index, orientation='left', color_threshold=0, colorscale=["grey"])
    for data in dendrogram.data:
        fig.add_trace (data, row=1, col=2)

    # Reorder rows
    labels_ordered = np.flip(dendrogram.layout['yaxis']['ticktext'])
    df = df.reindex(labels_ordered)

    # Define min_llr if not given = symetrical 2nd percentile
    if not lim_llr:
        lim_llr = max(np.absolute(np.nanpercentile(df.values, [2,98])))

    # Define colorscale
    offset = min_diff_llr/lim_llr*0.5
    colorscale = colorscale=[[0.0, unmethylated_color],[0.5-offset, ambiguous_color], [0.5+offset, ambiguous_color],[1.0, methylated_color]]

    # plot heatmap
    heatmap = go.Heatmap(name="heatmap", x=df.columns, y=df.index, z=df.values, zmin=-lim_llr, zmax=lim_llr, zmid=0, colorscale=colorscale, colorbar_title="Median LLR")
    fig.add_trace (heatmap, row=1, col=1)

    # Tweak figure layout
    fig.update_layout(
        dict1 = {'showlegend':False, 'hovermode':'closest', "plot_bgcolor":'rgba(0,0,0,0)',"width":fig_width, "height":fig_height, "margin":{"t":50,"b":50}},
        xaxis2 = {"fixedrange":True, 'showgrid':False, 'showline':False, "showticklabels":False,'zeroline':False,'ticks':""},
        yaxis2 = {"fixedrange":True, 'showgrid':False, 'showline':False, "showticklabels":False,'zeroline':False,'ticks':"", "automargin":True},
        xaxis = {"fixedrange":False, "domain":[0, 0.95], "showticklabels":False, "title":"CpG positions"},
        yaxis = {"fixedrange":True, "domain":[0, 1], "ticks":"outside", "automargin":True})

    return fig

def cpg_ridgeplot (
    df,
    methylated_color:str = 'rgb(215,48,39)',
    unmethylated_color:str = 'rgb(33,102,172)',
    ambiguous_color:str = 'rgb(100,100,100)',
    min_diff_llr:float = 1,
    lim_quantile:int = 0.0001,
    scatter:bool = True,
    box:bool = False,
    trace_width:int = 2,
    fig_width:int = None,
    fig_height:int = None):
    """
    Plot a distribution of the llr values as a ridgeplot
    """
    # Cannot calculate if at least not 2 values
    if len(df.columns) <= 1:
        return None

    # Sorted labels by median llr
    d = OrderedDict()
    for lab, row in df.iterrows():
        d[lab] = np.nanmedian(row)
    sorted_labels = [i for i,j in sorted(d.items(), key=lambda t: t[1])]

    # Define color map depending on number of samples
    cmap = n_colors(unmethylated_color, methylated_color, len(sorted_labels), colortype='rgb')

    # Find minimal and maximal llr values
    xmin, xmax = np.nanquantile(df.values, q = [lim_quantile, 1-lim_quantile])
    xmax = np.ceil(xmax)
    xmin = np.floor(xmin)

    # Create ridgeplot traces
    points="all" if scatter else False
    box_visible = True if box else False

    fig = go.Figure()
    for label, color in zip(sorted_labels, cmap):
        violin = go.Violin(
            x=df.loc[label],
            name=label,
            orientation="h",
            width=trace_width,
            hoveron="violins",
            points=points,
            box_visible=box_visible,
            box_width=trace_width/3,
            pointpos=0.2,
            jitter=0.2,
            marker_size=5,
            side="positive",
            line_color=color)
        fig.add_trace (violin)

        # Add shape for ambigous log likelihood area
    fig.add_shape(type="rect", xref="x", yref="paper", x0=-min_diff_llr, y0=0, x1=min_diff_llr, y1=1, fillcolor=ambiguous_color, opacity=0.25, layer="below", line_width=0)
    fig.add_annotation(text="Ambiguous", xref="x", yref="paper", x=0, y=1.06, showarrow=False, font_color=ambiguous_color)
    fig.add_annotation(text="Unmethylated", xref="x", yref="paper", x=(xmin+min_diff_llr)/2-min_diff_llr, y=1.06, showarrow=False, font_color=unmethylated_color)
    fig.add_annotation(text="Methylated", xref="x", yref="paper", x=(xmax-min_diff_llr)/2+min_diff_llr, y=1.06, showarrow=False, font_color=methylated_color)

    # tweak figure layout
    fig.update_layout(
        dict1 = {'showlegend':False, 'hovermode':'closest', "plot_bgcolor":'rgba(0,0,0,0)',"width":fig_width, "height":fig_height, "margin":{"t":50,"b":50}},
        xaxis={"showgrid":False, "zeroline":False, "domain":[0, 1], "title":"CpG median log likelihood ratio", "range":(xmin, xmax)},
        yaxis={"fixedrange":True, "showgrid":True, 'zeroline':False, "gridcolor":"lightgrey", "automargin":True})

    return fig

def category_barplot (
    df,
    methylated_color:str = 'rgb(215,48,39)',
    unmethylated_color:str = 'rgb(33,102,172)',
    ambiguous_color:str = 'rgb(150,150,150)',
    no_data_color:str = 'rgb(50,50,50)',
    min_diff_llr:float = 1,
    fig_width:int = None,
    fig_height:int = None):
    """
    Plot a stacked barplot of the number of intervals per category for each samples
    """
    # Cannot calculate if at least not 2 values
    if len(df.columns) <= 1:
        return None

    # Count values per categories
    d = OrderedDict()
    for sample_id, llr_list in df.iterrows():
        sample_d = OrderedDict()
        sample_d["Unmethylated"] = len(llr_list[llr_list <= -min_diff_llr])
        sample_d["Methylated"] = len(llr_list[llr_list >= min_diff_llr])
        sample_d["Ambiguous"] = len(llr_list[(llr_list > -min_diff_llr) & (llr_list < min_diff_llr) ])
        sample_d["No data"] = len(llr_list[llr_list.isna()])
        d[sample_id] = sample_d

    # Cast to dataframe and reorder per value
    count_df = pd.DataFrame.from_dict(d)
    count_df = count_df.reindex(columns=count_df.columns.sort_values())

    # Generate barplot per category
    data = []
    for status, color in [
        ("Unmethylated", unmethylated_color),
        ("Methylated", methylated_color),
        ("Ambiguous", ambiguous_color),
        ("No data", no_data_color)]:
        data.append(go.Bar(name=status, x=count_df.columns, y=count_df.loc[status], marker_color=color, marker_line_color=color, opacity=0.9))

    fig = go.Figure(data)

    # Change the bar mode
    fig.update_layout(barmode='stack', xaxis_tickangle=-45)

    # tweak figure layout
    fig.update_layout(
        barmode='stack',
        dict1 = {"plot_bgcolor":'rgba(0,0,0,0)',"width":fig_width, "height":fig_height, "margin":{"t":50,"b":50}},
        xaxis={"fixedrange":True, "showgrid":False, "tickangle":-45},
        yaxis={"fixedrange":True, "showgrid":True, 'zeroline':False, "title":"Counts per category", "gridcolor":"lightgrey"})

    return fig

#~~~~~~~~~~~~~~~~~~~~~~~~Help functions~~~~~~~~~~~~~~~~~~~~~~~~#

def get_jinja_template (template_fn):
    """Load Jinja template"""
    try:
        env = jinja2.Environment (
            loader=jinja2.PackageLoader('pycoMeth', 'templates'),
            autoescape=jinja2.select_autoescape(["html"]))

        template = env.get_template(template_fn)
        return template

    except (FileNotFoundError, IOError, jinja2.exceptions.TemplateNotFound, jinja2.exceptions.TemplateSyntaxError):
        print("\t\tFile not found, non-readable or invalid")

def render_df (df, empty_msg="No data"):
    """Render_dataframe in HTML"""
    # Return placeholder if empty
    if df.empty:
        return f"<div class='empty'><p class='empty-title h6'>{empty_msg}</p></div>"
    else:
        table = df.to_html(
            classes=["table","table-striped","table-hover", "table-scroll"],
            border=0,
            index=False,
            justify="justify-all",
            escape=False)
        return table

def render_fig (fig, empty_msg="No data"):
    """Render plotly figure in HTML"""
    # Return placeholder if empty
    if not fig:
        return f"<div class='empty'><p class='empty-title h6'>{empty_msg}</p></div>"

    fig.update_layout(margin={"t":50,"b":50})
    rendering = py.plot(
        fig,
        output_type='div',
        include_plotlyjs=False,
        image_width='',
        image_height='',
        show_link=False,
        auto_open=False)
    return rendering

def prev_fn (rank_fn_dict, rank):
    """Get the filepath for the previous interval in rank. The first interval links to the last"""
    prev_rank = len(rank_fn_dict) if rank == 1 else rank-1
    return rank_fn_dict[prev_rank]

def next_fn (rank_fn_dict, rank):
    """Get the filepath for the next interval in rank. The last interval links to the first"""
    prev_rank = 1 if rank == len(rank_fn_dict) else rank+1
    return rank_fn_dict[prev_rank]

def md5_str (fn):
    """Compute md5 has for a given file"""
    hash_md5 = hashlib.md5()
    with open(fn, "rb") as f:
        for chunk in iter(lambda: f.read(4096), b""):
            hash_md5.update(chunk)
    return hash_md5.hexdigest()
