/*
 * Decompiled with CFR 0.152.
 */
package nl.esciencecenter.xenon.filesystems;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import nl.esciencecenter.xenon.UnknownAdaptorException;
import nl.esciencecenter.xenon.XenonException;
import nl.esciencecenter.xenon.adaptors.NotConnectedException;
import nl.esciencecenter.xenon.adaptors.XenonProperties;
import nl.esciencecenter.xenon.adaptors.filesystems.FileAdaptor;
import nl.esciencecenter.xenon.adaptors.filesystems.ftp.FtpFileAdaptor;
import nl.esciencecenter.xenon.adaptors.filesystems.local.LocalFileAdaptor;
import nl.esciencecenter.xenon.adaptors.filesystems.s3.S3FileAdaptor;
import nl.esciencecenter.xenon.adaptors.filesystems.sftp.SftpFileAdaptor;
import nl.esciencecenter.xenon.adaptors.filesystems.webdav.WebdavFileAdaptor;
import nl.esciencecenter.xenon.credentials.Credential;
import nl.esciencecenter.xenon.credentials.DefaultCredential;
import nl.esciencecenter.xenon.filesystems.CopyCancelledException;
import nl.esciencecenter.xenon.filesystems.CopyMode;
import nl.esciencecenter.xenon.filesystems.CopyStatus;
import nl.esciencecenter.xenon.filesystems.DirectoryNotEmptyException;
import nl.esciencecenter.xenon.filesystems.FileSystemAdaptorDescription;
import nl.esciencecenter.xenon.filesystems.InvalidPathException;
import nl.esciencecenter.xenon.filesystems.NoSuchCopyException;
import nl.esciencecenter.xenon.filesystems.NoSuchPathException;
import nl.esciencecenter.xenon.filesystems.Path;
import nl.esciencecenter.xenon.filesystems.PathAlreadyExistsException;
import nl.esciencecenter.xenon.filesystems.PathAttributes;
import nl.esciencecenter.xenon.filesystems.PosixFilePermission;

public abstract class FileSystem
implements AutoCloseable {
    private static final String COMPONENT_NAME = "FileSystem";
    private static final HashMap<String, FileAdaptor> adaptors = new LinkedHashMap<String, FileAdaptor>();
    private final String uniqueID;
    private final String adaptor;
    private final String location;
    private final XenonProperties properties;
    private final ExecutorService pool;
    private Path workingDirectory;
    private long nextCopyID = 0L;
    private int bufferSize;
    private final HashMap<String, PendingCopy> pendingCopies = new HashMap();

    private static void addAdaptor(FileAdaptor adaptor) {
        adaptors.put(adaptor.getName(), adaptor);
    }

    private static FileAdaptor getAdaptorByName(String adaptorName) throws UnknownAdaptorException {
        if (adaptorName == null || adaptorName.trim().isEmpty()) {
            throw new IllegalArgumentException("Adaptor name may not be null or empty");
        }
        if (!adaptors.containsKey(adaptorName)) {
            throw new UnknownAdaptorException(COMPONENT_NAME, String.format("Adaptor '%s' not found", adaptorName));
        }
        return adaptors.get(adaptorName);
    }

    public static String[] getAdaptorNames() {
        return adaptors.keySet().toArray(new String[adaptors.size()]);
    }

    public static FileSystemAdaptorDescription getAdaptorDescription(String adaptorName) throws UnknownAdaptorException {
        return FileSystem.getAdaptorByName(adaptorName);
    }

    public static FileSystemAdaptorDescription[] getAdaptorDescriptions() {
        return adaptors.values().toArray(new FileSystemAdaptorDescription[adaptors.size()]);
    }

    public static FileSystem create(String adaptor, String location, Credential credential, Map<String, String> properties) throws XenonException {
        return FileSystem.getAdaptorByName(adaptor).createFileSystem(location, credential, properties);
    }

    public static FileSystem create(String adaptor, String location, Credential credential) throws XenonException {
        return FileSystem.create(adaptor, location, credential, new HashMap<String, String>(0));
    }

    public static FileSystem create(String adaptor, String location) throws XenonException {
        return FileSystem.create(adaptor, location, new DefaultCredential());
    }

    public static FileSystem create(String adaptor) throws XenonException {
        return FileSystem.create(adaptor, null);
    }

    protected FileSystem(String uniqueID, String adaptor, String location, Path workDirectory, int bufferSize, XenonProperties properties) {
        if (uniqueID == null) {
            throw new IllegalArgumentException("Identifier may not be null!");
        }
        if (adaptor == null) {
            throw new IllegalArgumentException("Adaptor may not be null!");
        }
        if (location == null) {
            throw new IllegalArgumentException("Location may not be null!");
        }
        if (workDirectory == null) {
            throw new IllegalArgumentException("EntryPath may not be null!");
        }
        if (bufferSize <= 0) {
            throw new IllegalArgumentException("Buffer size may not be 0 or smaller!");
        }
        this.uniqueID = uniqueID;
        this.adaptor = adaptor;
        this.location = location;
        this.workingDirectory = workDirectory;
        this.properties = properties;
        this.bufferSize = bufferSize;
        ThreadFactory f = r -> {
            Thread t = new Thread(r, "CopyThread-" + adaptor + "-" + uniqueID);
            t.setDaemon(true);
            return t;
        };
        this.pool = Executors.newFixedThreadPool(1, f);
    }

    private synchronized String getNextCopyID() {
        return "COPY-" + this.getAdaptorName() + "-" + this.nextCopyID++;
    }

    public String getAdaptorName() {
        return this.adaptor;
    }

    public String getLocation() {
        return this.location;
    }

    public Map<String, String> getProperties() {
        return this.properties.toMap();
    }

    public Path getWorkingDirectory() {
        return this.workingDirectory;
    }

    public String getPathSeparator() {
        return "" + this.workingDirectory.getSeparator();
    }

    public void setWorkingDirectory(Path directory) throws XenonException {
        Path wd = this.toAbsolutePath(directory);
        this.assertDirectoryExists(wd);
        this.workingDirectory = wd;
    }

    @Override
    public void close() throws XenonException {
        try {
            this.pool.shutdownNow();
        }
        catch (Exception e) {
            throw new XenonException(this.getAdaptorName(), "Failed to cleanly shutdown copy thread pool");
        }
    }

    public abstract boolean isOpen() throws XenonException;

    public abstract void rename(Path var1, Path var2) throws XenonException;

    public void createDirectories(Path dir) throws XenonException {
        Path absolute = this.toAbsolutePath(dir);
        Path parent = absolute.getParent();
        if (parent != null && !this.exists(parent)) {
            this.createDirectories(parent);
        }
        this.createDirectory(absolute);
    }

    public abstract void createDirectory(Path var1) throws XenonException;

    public abstract void createFile(Path var1) throws XenonException;

    public abstract void createSymbolicLink(Path var1, Path var2) throws XenonException;

    public void delete(Path path, boolean recursive) throws XenonException {
        Path absPath = this.toAbsolutePath(path);
        this.assertPathExists(absPath);
        if (this.getAttributes(absPath).isDirectory()) {
            Iterable<PathAttributes> itt = this.list(absPath, false);
            if (recursive) {
                for (PathAttributes p : itt) {
                    this.delete(p.getPath(), true);
                }
            } else if (itt.iterator().hasNext()) {
                throw new DirectoryNotEmptyException(this.getAdaptorName(), "Directory not empty: " + absPath.toString());
            }
            this.deleteDirectory(absPath);
        } else {
            this.deleteFile(absPath);
        }
    }

    public abstract boolean exists(Path var1) throws XenonException;

    public Iterable<PathAttributes> list(Path dir, boolean recursive) throws XenonException {
        Path absolute = this.toAbsolutePath(dir);
        this.assertDirectoryExists(dir);
        ArrayList<PathAttributes> result = new ArrayList<PathAttributes>();
        this.list(absolute, result, recursive);
        return result;
    }

    public abstract InputStream readFromFile(Path var1) throws XenonException;

    public abstract OutputStream writeToFile(Path var1, long var2) throws XenonException;

    public abstract OutputStream writeToFile(Path var1) throws XenonException;

    public abstract OutputStream appendToFile(Path var1) throws XenonException;

    public abstract PathAttributes getAttributes(Path var1) throws XenonException;

    public abstract Path readSymbolicLink(Path var1) throws XenonException;

    public abstract void setPosixFilePermissions(Path var1, Set<PosixFilePermission> var2) throws XenonException;

    protected Path toAbsolutePath(Path path) {
        this.assertNotNull(path);
        if (path.isAbsolute()) {
            return path.normalize();
        }
        return this.workingDirectory.resolve(path).normalize();
    }

    protected void streamCopy(InputStream in, OutputStream out, int buffersize, CopyCallback callback) throws IOException, CopyCancelledException {
        byte[] buffer = new byte[buffersize];
        int size = in.read(buffer);
        while (size > 0) {
            out.write(buffer, 0, size);
            callback.addBytesCopied(size);
            if (callback.isCancelled()) {
                throw new CopyCancelledException(this.getAdaptorName(), "Copy cancelled by user");
            }
            size = in.read(buffer);
        }
    }

    protected void copySymbolicLink(Path source, FileSystem destinationFS, Path destination, CopyMode mode, CopyCallback callback) throws XenonException {
        PathAttributes attributes = this.getAttributes(source);
        if (!attributes.isSymbolicLink()) {
            throw new InvalidPathException(this.getAdaptorName(), "Source is not a regular file: " + source);
        }
        destinationFS.assertParentDirectoryExists(destination);
        if (destinationFS.exists(destination)) {
            switch (mode) {
                case CREATE: {
                    throw new PathAlreadyExistsException(this.getAdaptorName(), "Destination path already exists: " + destination);
                }
                case IGNORE: {
                    return;
                }
            }
        }
        Path target = this.readSymbolicLink(source);
        destinationFS.createSymbolicLink(destination, target);
    }

    protected void copyFile(Path source, FileSystem destinationFS, Path destination, CopyMode mode, CopyCallback callback) throws XenonException {
        PathAttributes attributes = this.getAttributes(source);
        if (!attributes.isRegular()) {
            throw new InvalidPathException(this.getAdaptorName(), "Source is not a regular file: " + source);
        }
        destinationFS.assertParentDirectoryExists(destination);
        if (destinationFS.exists(destination)) {
            switch (mode) {
                case CREATE: {
                    throw new PathAlreadyExistsException(this.getAdaptorName(), "Destination path already exists: " + destination);
                }
                case IGNORE: {
                    return;
                }
                case REPLACE: {
                    destinationFS.delete(destination, true);
                }
            }
        }
        if (callback.isCancelled()) {
            throw new CopyCancelledException(this.getAdaptorName(), "Copy cancelled by user");
        }
        try (InputStream in = this.readFromFile(source);
             OutputStream out = destinationFS.writeToFile(destination, attributes.getSize());){
            this.streamCopy(in, out, this.bufferSize, callback);
        }
        catch (Exception e) {
            throw new XenonException(this.getAdaptorName(), "Stream copy failed", e);
        }
    }

    protected void performCopy(Path source, FileSystem destinationFS, Path destination, CopyMode mode, boolean recursive, CopyCallback callback) throws XenonException {
        if (!this.exists(source)) {
            throw new NoSuchPathException(this.getAdaptorName(), "No such file " + source.toString());
        }
        PathAttributes attributes = this.getAttributes(source);
        if (attributes.isRegular()) {
            this.copyFile(source, destinationFS, destination, mode, callback);
            return;
        }
        if (attributes.isSymbolicLink()) {
            this.copySymbolicLink(source, destinationFS, destination, mode, callback);
            return;
        }
        if (!attributes.isDirectory()) {
            throw new InvalidPathException(this.getAdaptorName(), "Source path is not a file, link or directory: " + source);
        }
        if (!recursive) {
            throw new InvalidPathException(this.getAdaptorName(), "Source path is a directory: " + source);
        }
        if (destinationFS.exists(destination)) {
            switch (mode) {
                case CREATE: {
                    throw new PathAlreadyExistsException(this.getAdaptorName(), "Destination path already exists: " + destination);
                }
                case IGNORE: {
                    return;
                }
            }
            attributes = destinationFS.getAttributes(destination);
            if (attributes.isRegular() || attributes.isSymbolicLink()) {
                destinationFS.delete(destination, false);
                destinationFS.createDirectory(destination);
            } else if (!attributes.isDirectory()) {
                throw new InvalidPathException(this.getAdaptorName(), "Existing destination is not a file, link or directory: " + source);
            }
        } else {
            destinationFS.createDirectory(destination);
        }
        this.copyRecursive(source, destinationFS, destination, mode, callback);
    }

    private void copyRecursive(Path source, FileSystem destinationFS, Path destination, CopyMode mode, CopyCallback callback) throws XenonException {
        Path dst;
        Path rel;
        long bytesToCopy = 0L;
        Iterable<PathAttributes> listing = this.list(source, true);
        for (PathAttributes p : listing) {
            if (callback.isCancelled()) {
                throw new CopyCancelledException(this.getAdaptorName(), "Copy cancelled by user");
            }
            if (p.isDirectory() && !this.isDotDot(p.getPath())) {
                rel = source.relativize(p.getPath());
                dst = destination.resolve(rel);
                if (destinationFS.exists(dst)) {
                    if (destinationFS.getAttributes(dst).isDirectory()) {
                        switch (mode) {
                            case CREATE: {
                                throw new PathAlreadyExistsException(this.getAdaptorName(), "Directory already exists: " + dst);
                            }
                            case REPLACE: {
                                break;
                            }
                            case IGNORE: {
                                return;
                            }
                        }
                        continue;
                    }
                    destinationFS.delete(dst, true);
                    continue;
                }
                destinationFS.createDirectories(dst);
                continue;
            }
            if (!p.isRegular()) continue;
            bytesToCopy += p.getSize();
        }
        callback.start(bytesToCopy);
        for (PathAttributes p : listing) {
            if (callback.isCancelled()) {
                throw new CopyCancelledException(this.getAdaptorName(), "Copy cancelled by user");
            }
            if (!p.isRegular()) continue;
            rel = source.relativize(p.getPath());
            dst = destination.resolve(rel);
            this.copyFile(p.getPath(), destinationFS, dst, mode, callback);
        }
    }

    protected abstract void deleteFile(Path var1) throws XenonException;

    protected abstract void deleteDirectory(Path var1) throws XenonException;

    protected abstract Iterable<PathAttributes> listDirectory(Path var1) throws XenonException;

    protected void list(Path dir, ArrayList<PathAttributes> list, boolean recursive) throws XenonException {
        Iterable<PathAttributes> tmp = this.listDirectory(dir);
        for (PathAttributes p : tmp) {
            if (this.isDotDot(p.getPath())) continue;
            list.add(p);
        }
        if (recursive) {
            for (PathAttributes current : tmp) {
                if (!current.isDirectory() || this.isDotDot(current.getPath())) continue;
                this.list(dir.resolve(current.getPath().getFileNameAsString()), list, true);
            }
        }
    }

    public synchronized String copy(Path source, FileSystem destinationFS, Path destination, CopyMode mode, boolean recursive) {
        if (source == null) {
            throw new IllegalArgumentException("Source path is null");
        }
        if (destinationFS == null) {
            throw new IllegalArgumentException("Destination filesystem is null");
        }
        if (destination == null) {
            throw new IllegalArgumentException("Destination path is null");
        }
        if (mode == null) {
            throw new IllegalArgumentException("Copy mode is null!");
        }
        String copyID = this.getNextCopyID();
        CopyCallback callback = new CopyCallback();
        Future<Void> future = this.pool.submit(() -> {
            if (Thread.currentThread().isInterrupted()) {
                throw new CopyCancelledException(this.getAdaptorName(), "Copy cancelled by user");
            }
            this.performCopy(this.toAbsolutePath(source), destinationFS, this.toAbsolutePath(destination), mode, recursive, callback);
            return null;
        });
        this.pendingCopies.put(copyID, new PendingCopy(future, callback));
        return copyID;
    }

    public synchronized CopyStatus cancel(String copyIdentifier) throws XenonException {
        if (copyIdentifier == null) {
            throw new IllegalArgumentException("Copy identifier may not be null");
        }
        PendingCopy copy = this.pendingCopies.remove(copyIdentifier);
        if (copy == null) {
            throw new NoSuchCopyException(this.getAdaptorName(), "Copy not found: " + copyIdentifier);
        }
        copy.callback.cancel();
        copy.future.cancel(true);
        XenonException ex = null;
        String state = "DONE";
        try {
            copy.future.get();
        }
        catch (ExecutionException ee) {
            ex = new XenonException(this.getAdaptorName(), ee.getMessage(), ee);
            state = "FAILED";
        }
        catch (InterruptedException | CancellationException ec) {
            ex = new CopyCancelledException(this.getAdaptorName(), "Copy cancelled by user");
            state = "FAILED";
        }
        return new CopyStatusImplementation(copyIdentifier, state, copy.callback.bytesToCopy, copy.callback.bytesCopied, ex);
    }

    public CopyStatus waitUntilDone(String copyIdentifier, long timeout) throws XenonException {
        if (copyIdentifier == null) {
            throw new IllegalArgumentException("Copy identifier may not be null");
        }
        PendingCopy copy = this.pendingCopies.get(copyIdentifier);
        if (copy == null) {
            throw new NoSuchCopyException(this.getAdaptorName(), "Copy not found: " + copyIdentifier);
        }
        XenonException ex = null;
        String state = "DONE";
        try {
            copy.future.get(timeout, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            state = "RUNNING";
        }
        catch (ExecutionException ee) {
            Throwable cause = ee.getCause();
            ex = cause instanceof XenonException ? (XenonException)cause : new XenonException(this.getAdaptorName(), cause.getMessage(), cause);
            state = "FAILED";
        }
        catch (InterruptedException | CancellationException ec) {
            ex = new CopyCancelledException(this.getAdaptorName(), "Copy cancelled by user");
            state = "FAILED";
        }
        if (copy.future.isDone()) {
            this.pendingCopies.remove(copyIdentifier);
        }
        return new CopyStatusImplementation(copyIdentifier, state, copy.callback.bytesToCopy, copy.callback.bytesCopied, ex);
    }

    public CopyStatus getStatus(String copyIdentifier) throws XenonException {
        if (copyIdentifier == null) {
            throw new IllegalArgumentException("Copy identifier may not be null");
        }
        PendingCopy copy = this.pendingCopies.get(copyIdentifier);
        if (copy == null) {
            throw new NoSuchCopyException(this.getAdaptorName(), "Copy not found: " + copyIdentifier);
        }
        XenonException ex = null;
        String state = "PENDING";
        if (copy.future.isDone()) {
            this.pendingCopies.remove(copyIdentifier);
            try {
                copy.future.get();
                state = "DONE";
            }
            catch (ExecutionException ee) {
                ex = new XenonException(this.getAdaptorName(), ee.getMessage(), ee);
                state = "FAILED";
            }
            catch (InterruptedException | CancellationException ec) {
                ex = new CopyCancelledException(this.getAdaptorName(), "Copy cancelled by user");
                state = "FAILED";
            }
        } else if (copy.callback.isStarted()) {
            state = "RUNNING";
        }
        return new CopyStatusImplementation(copyIdentifier, state, copy.callback.bytesToCopy, copy.callback.bytesCopied, ex);
    }

    protected void assertNotNull(Path path) {
        if (path == null) {
            throw new IllegalArgumentException("Path is null");
        }
    }

    protected void assertPathExists(Path path) throws XenonException {
        this.assertNotNull(path);
        if (!this.exists(path)) {
            throw new NoSuchPathException(this.getAdaptorName(), "Path does not exist: " + path);
        }
    }

    protected void assertPathNotExists(Path path) throws XenonException {
        this.assertNotNull(path);
        if (this.exists(path)) {
            throw new PathAlreadyExistsException(this.getAdaptorName(), "Path already exists: " + path);
        }
    }

    protected void assertPathIsNotDirectory(Path path) throws XenonException {
        PathAttributes a;
        this.assertNotNull(path);
        if (this.exists(path) && (a = this.getAttributes(path)).isDirectory()) {
            throw new InvalidPathException(this.getAdaptorName(), "Was expecting a regular file, but got a directory: " + path.toString());
        }
    }

    protected void assertPathIsFile(Path path) throws XenonException {
        this.assertNotNull(path);
        if (!this.getAttributes(path).isRegular()) {
            throw new InvalidPathException(this.getAdaptorName(), "Path is not a file: " + path);
        }
    }

    protected void assertPathIsDirectory(Path path) throws XenonException {
        this.assertNotNull(path);
        PathAttributes a = this.getAttributes(path);
        if (a == null) {
            throw new InvalidPathException(this.getAdaptorName(), "Path failed to produce attributes: " + path);
        }
        if (!a.isDirectory()) {
            throw new InvalidPathException(this.getAdaptorName(), "Path is not a directory: " + path);
        }
    }

    protected void assertFileExists(Path file) throws XenonException {
        this.assertPathExists(file);
        this.assertPathIsFile(file);
    }

    protected void assertDirectoryExists(Path dir) throws XenonException {
        this.assertPathExists(dir);
        this.assertPathIsDirectory(dir);
    }

    protected void assertParentDirectoryExists(Path path) throws XenonException {
        this.assertNotNull(path);
        Path parent = path.getParent();
        if (parent != null) {
            this.assertDirectoryExists(parent);
        }
    }

    protected void assertFileIsSymbolicLink(Path link) throws XenonException {
        this.assertNotNull(link);
        this.assertPathExists(link);
        if (!this.getAttributes(link).isSymbolicLink()) {
            throw new InvalidPathException(this.getAdaptorName(), "Not a symbolic link: " + link);
        }
    }

    protected void assertIsOpen() throws XenonException {
        if (!this.isOpen()) {
            throw new NotConnectedException(this.getAdaptorName(), "Connection is closed");
        }
    }

    protected boolean areSamePaths(Path source, Path target) {
        return source.equals(target);
    }

    protected boolean isDotDot(Path path) {
        this.assertNotNull(path);
        String filename = path.getFileNameAsString();
        return ".".equals(filename) || "..".equals(filename);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        FileSystem that = (FileSystem)o;
        return Objects.equals(this.uniqueID, that.uniqueID);
    }

    public int hashCode() {
        return Objects.hash(this.uniqueID);
    }

    static {
        FileSystem.addAdaptor(new LocalFileAdaptor());
        FileSystem.addAdaptor(new FtpFileAdaptor());
        FileSystem.addAdaptor(new SftpFileAdaptor());
        FileSystem.addAdaptor(new WebdavFileAdaptor());
        FileSystem.addAdaptor(new S3FileAdaptor());
    }

    private class PendingCopy {
        Future<Void> future;
        CopyCallback callback;

        public PendingCopy(Future<Void> future, CopyCallback callback) {
            this.future = future;
            this.callback = callback;
        }
    }

    class CopyCallback {
        long bytesToCopy = 0L;
        long bytesCopied = 0L;
        boolean started = false;
        boolean cancelled = false;

        CopyCallback() {
        }

        synchronized void start(long bytesToCopy) {
            if (!this.started) {
                this.started = true;
                this.bytesToCopy = bytesToCopy;
            }
        }

        synchronized boolean isStarted() {
            return this.started;
        }

        synchronized void addBytesCopied(long bytes) {
            this.bytesCopied += bytes;
        }

        synchronized void cancel() {
            this.cancelled = true;
        }

        synchronized boolean isCancelled() {
            return this.cancelled;
        }
    }

    static class CopyStatusImplementation
    implements CopyStatus {
        private final String copyIdentifier;
        private final String state;
        private final XenonException exception;
        private final long bytesToCopy;
        private final long bytesCopied;

        public CopyStatusImplementation(String copyIdentifier, String state, long bytesToCopy, long bytesCopied, XenonException exception) {
            this.copyIdentifier = copyIdentifier;
            this.state = state;
            this.bytesToCopy = bytesToCopy;
            this.bytesCopied = bytesCopied;
            this.exception = exception;
        }

        @Override
        public String getCopyIdentifier() {
            return this.copyIdentifier;
        }

        @Override
        public String getState() {
            return this.state;
        }

        @Override
        public XenonException getException() {
            return this.exception;
        }

        @Override
        public void maybeThrowException() throws XenonException {
            if (this.hasException()) {
                throw this.getException();
            }
        }

        @Override
        public boolean isRunning() {
            return "RUNNING".equals(this.state);
        }

        @Override
        public boolean isDone() {
            return "DONE".equals(this.state) || "FAILED".equals(this.state);
        }

        @Override
        public boolean hasException() {
            return this.exception != null;
        }

        @Override
        public long bytesToCopy() {
            return this.bytesToCopy;
        }

        @Override
        public long bytesCopied() {
            return this.bytesCopied;
        }

        public String toString() {
            return "CopyStatus [copyIdentifier=" + this.copyIdentifier + ", state=" + this.state + ", exception=" + this.exception + ", bytesToCopy=" + this.bytesToCopy + ", bytesCopied=" + this.bytesCopied + "]";
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CopyStatusImplementation that = (CopyStatusImplementation)o;
            return this.bytesToCopy == that.bytesToCopy && this.bytesCopied == that.bytesCopied && Objects.equals(this.copyIdentifier, that.copyIdentifier) && Objects.equals(this.state, that.state) && Objects.equals(this.exception, that.exception);
        }

        public int hashCode() {
            return Objects.hash(this.copyIdentifier, this.state, this.exception, this.bytesToCopy, this.bytesCopied);
        }
    }
}

