/*
 * Decompiled with CFR 0.152.
 */
package nl.esciencecenter.xenon.adaptors.schedulers.slurm;

import java.util.ArrayList;
import java.util.Formatter;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import nl.esciencecenter.xenon.XenonException;
import nl.esciencecenter.xenon.adaptors.schedulers.CommandLineUtils;
import nl.esciencecenter.xenon.adaptors.schedulers.JobCanceledException;
import nl.esciencecenter.xenon.adaptors.schedulers.JobStatusImplementation;
import nl.esciencecenter.xenon.adaptors.schedulers.QueueStatusImplementation;
import nl.esciencecenter.xenon.adaptors.schedulers.ScriptingUtils;
import nl.esciencecenter.xenon.filesystems.Path;
import nl.esciencecenter.xenon.schedulers.InvalidJobDescriptionException;
import nl.esciencecenter.xenon.schedulers.JobDescription;
import nl.esciencecenter.xenon.schedulers.JobStatus;
import nl.esciencecenter.xenon.schedulers.QueueStatus;
import nl.esciencecenter.xenon.schedulers.Scheduler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SlurmUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(SlurmUtils.class);
    public static final String JOB_OPTION_JOB_SCRIPT = "job.script";
    private static final String[] VALID_JOB_OPTIONS = new String[]{"job.script"};
    private static final String[] FAILED_STATES = new String[]{"FAILED", "CANCELLED", "NODE_FAIL", "TIMEOUT", "PREEMPTED", "BOOT_FAIL"};
    private static final String[] RUNNING_STATES = new String[]{"CONFIGURING", "RUNNING", "COMPLETING"};
    private static final String[] PENDING_STATES = new String[]{"PENDING", "STOPPED", "SUSPENDED", "SPECIAL_EXIT"};
    private static final String DONE_STATE = "COMPLETED";

    protected static String identifiersAsCSList(String[] jobs) {
        String result = null;
        for (String job : jobs) {
            if (job == null) continue;
            result = result == null ? job : CommandLineUtils.concat(result, ",", job);
        }
        return result;
    }

    private SlurmUtils() {
        throw new IllegalStateException("Utility class");
    }

    protected static Integer exitcodeFromString(String value) throws XenonException {
        if (value == null) {
            return null;
        }
        String exitCodeString = value.split(":")[0];
        try {
            return Integer.parseInt(exitCodeString);
        }
        catch (NumberFormatException e) {
            throw new XenonException("slurm", "job exit code \"" + exitCodeString + "\" is not a number", e);
        }
    }

    protected static JobStatus getJobStatusFromSacctInfo(Map<String, Map<String, String>> info, String jobIdentifier) throws XenonException {
        Map<String, String> jobInfo = info.get(jobIdentifier);
        if (jobInfo == null) {
            LOGGER.debug("job {} not found in sacct output", (Object)jobIdentifier);
            return null;
        }
        ScriptingUtils.verifyJobInfo(jobInfo, jobIdentifier, "slurm", "JobID", "State", "ExitCode");
        String state = jobInfo.get("State");
        Integer exitcode = SlurmUtils.exitcodeFromString(jobInfo.get("ExitCode"));
        XenonException exception = !SlurmUtils.isFailedState(state) || state.equals("FAILED") && exitcode != null && exitcode != 0 ? null : (state.startsWith("CANCELLED") ? new JobCanceledException("slurm", "Job " + state.toLowerCase(Locale.getDefault())) : new XenonException("slurm", "Job failed for unknown reason"));
        JobStatusImplementation result = new JobStatusImplementation(jobIdentifier, state, exitcode, exception, SlurmUtils.isRunningState(state), SlurmUtils.isDoneOrFailedState(state), jobInfo);
        LOGGER.debug("Got job status from sacct output {}", (Object)result);
        return result;
    }

    protected static JobStatus getJobStatusFromScontrolInfo(Map<String, String> jobInfo, String jobIdentifier) throws XenonException {
        if (jobInfo == null) {
            LOGGER.debug("job {} not found in scontrol output", (Object)jobIdentifier);
            return null;
        }
        ScriptingUtils.verifyJobInfo(jobInfo, jobIdentifier, "slurm", "JobId", "JobState", "ExitCode", "Reason");
        String state = jobInfo.get("JobState");
        Integer exitcode = SlurmUtils.exitcodeFromString(jobInfo.get("ExitCode"));
        String reason = jobInfo.get("Reason");
        XenonException exception = !SlurmUtils.isFailedState(state) || state.equals("FAILED") && reason.equals("NonZeroExitCode") ? null : (state.startsWith("CANCELLED") ? new JobCanceledException("slurm", "Job " + state.toLowerCase(Locale.getDefault())) : (!reason.equals("None") ? new XenonException("slurm", "Job failed with state \"" + state + "\" and reason: " + reason) : new XenonException("slurm", "Job failed with state \"" + state + "\" for unknown reason")));
        JobStatusImplementation result = new JobStatusImplementation(jobIdentifier, state, exitcode, exception, SlurmUtils.isRunningState(state), SlurmUtils.isDoneOrFailedState(state), jobInfo);
        LOGGER.debug("Got job status from scontrol output {}", (Object)result);
        return result;
    }

    protected static JobStatus getJobStatusFromSqueueInfo(Map<String, Map<String, String>> info, String jobIdentifier) throws XenonException {
        Map<String, String> jobInfo = info.get(jobIdentifier);
        if (jobInfo == null) {
            LOGGER.debug("job {} not found in queue", (Object)jobIdentifier);
            return null;
        }
        ScriptingUtils.verifyJobInfo(jobInfo, jobIdentifier, "slurm", "JOBID", "STATE");
        String state = jobInfo.get("STATE");
        return new JobStatusImplementation(jobIdentifier, state, null, null, SlurmUtils.isRunningState(state), false, jobInfo);
    }

    protected static QueueStatus getQueueStatusFromSInfo(Map<String, Map<String, String>> info, String queueName, Scheduler scheduler) {
        Map<String, String> queueInfo = info.get(queueName);
        if (queueInfo == null) {
            LOGGER.debug("queue {} not found", (Object)queueName);
            return null;
        }
        return new QueueStatusImplementation(scheduler, queueName, null, queueInfo);
    }

    protected static boolean isRunningState(String state) {
        for (String validState : RUNNING_STATES) {
            if (!state.startsWith(validState)) continue;
            return true;
        }
        return false;
    }

    protected static boolean isPendingState(String state) {
        for (String validState : PENDING_STATES) {
            if (!state.startsWith(validState)) continue;
            return true;
        }
        return false;
    }

    protected static boolean isDoneOrFailedState(String state) {
        return SlurmUtils.isDoneState(state) || SlurmUtils.isFailedState(state);
    }

    protected static boolean isDoneState(String state) {
        return state.equals(DONE_STATE);
    }

    protected static boolean isFailedState(String state) {
        for (String validState : FAILED_STATES) {
            if (!state.startsWith(validState)) continue;
            return true;
        }
        return false;
    }

    protected static void verifyJobDescription(JobDescription description, boolean interactive) throws XenonException {
        ScriptingUtils.verifyJobOptions(description.getJobOptions(), VALID_JOB_OPTIONS, "slurm");
        if (interactive) {
            if (description.getJobOptions().get(JOB_OPTION_JOB_SCRIPT) != null) {
                throw new InvalidJobDescriptionException("slurm", "Custom job script not supported in interactive mode");
            }
            if (description.isStartSingleProcess()) {
                throw new InvalidJobDescriptionException("slurm", "StartSingleProcess option not supported in interactive mode");
            }
            if (description.getStdin() != null) {
                throw new InvalidJobDescriptionException("slurm", "Stdin redirect not supported in interactive mode");
            }
            if (description.getStdout() != null && !description.getStdout().equals("stdout.txt")) {
                throw new InvalidJobDescriptionException("slurm", "Stdout redirect not supported in interactive mode");
            }
            if (description.getStderr() != null && !description.getStderr().equals("stderr.txt")) {
                throw new InvalidJobDescriptionException("slurm", "Stderr redirect not supported in interactive mode");
            }
            if (description.getEnvironment().size() != 0) {
                throw new InvalidJobDescriptionException("slurm", "Environment variables not supported in interactive mode");
            }
        }
        if (description.getJobOptions().get(JOB_OPTION_JOB_SCRIPT) != null) {
            return;
        }
        ScriptingUtils.verifyJobDescription(description, "slurm");
    }

    private static String getWorkingDirPath(JobDescription description, Path fsEntryPath) {
        String path;
        if (description.getWorkingDirectory().startsWith("/")) {
            path = description.getWorkingDirectory();
        } else {
            Path workingDirectory = fsEntryPath.resolve(description.getWorkingDirectory());
            path = workingDirectory.toString();
        }
        return path;
    }

    public static String[] generateInteractiveArguments(JobDescription description, Path fsEntryPath, UUID tag) {
        ArrayList<String> arguments = new ArrayList<String>();
        arguments.add("--quiet");
        arguments.add("--job-name=" + tag.toString());
        if (description.getWorkingDirectory() != null) {
            String path = SlurmUtils.getWorkingDirPath(description, fsEntryPath);
            arguments.add("--chdir=" + path);
        }
        if (description.getQueueName() != null) {
            arguments.add("--partition=" + description.getQueueName());
        }
        arguments.add("--nodes=" + description.getNodeCount());
        arguments.add("--ntasks-per-node=" + description.getProcessesPerNode());
        arguments.add("--time=" + description.getMaxRuntime());
        arguments.add(description.getExecutable());
        arguments.addAll(description.getArguments());
        return arguments.toArray(new String[arguments.size()]);
    }

    public static String generate(JobDescription description, Path fsEntryPath) {
        StringBuilder stringBuilder = new StringBuilder();
        Formatter script = new Formatter(stringBuilder, Locale.US);
        script.format("%s\n", "#!/bin/sh");
        script.format("%s\n", "#SBATCH --job-name xenon");
        if (description.getWorkingDirectory() != null) {
            String path = SlurmUtils.getWorkingDirPath(description, fsEntryPath);
            script.format("#SBATCH --workdir='%s'\n", path);
        }
        if (description.getQueueName() != null) {
            script.format("#SBATCH --partition=%s\n", description.getQueueName());
        }
        script.format("#SBATCH --nodes=%d\n", description.getNodeCount());
        script.format("#SBATCH --ntasks-per-node=%d\n", description.getProcessesPerNode());
        script.format("#SBATCH --time=%d\n", description.getMaxRuntime());
        if (description.getStdin() != null) {
            script.format("#SBATCH --input='%s'\n", description.getStdin());
        }
        if (description.getStdout() == null) {
            script.format("#SBATCH --output=/dev/null\n", new Object[0]);
        } else {
            script.format("#SBATCH --output='%s'\n", description.getStdout());
        }
        if (description.getStderr() == null) {
            script.format("%s\n", "#SBATCH --error=/dev/null");
        } else {
            script.format("#SBATCH --error='%s'\n", description.getStderr());
        }
        for (Map.Entry<String, String> entry : description.getEnvironment().entrySet()) {
            script.format("export %s=\"%s\"\n", entry.getKey(), entry.getValue());
        }
        script.format("\n", new Object[0]);
        if (!description.isStartSingleProcess()) {
            script.format("%s ", "srun");
        }
        script.format("%s", description.getExecutable());
        for (String argument : description.getArguments()) {
            script.format(" %s", CommandLineUtils.protectAgainstShellMetas(argument));
        }
        script.format("\n", new Object[0]);
        script.close();
        LOGGER.debug("Created job script:%n{} from description {}", (Object)stringBuilder, (Object)description);
        return stringBuilder.toString();
    }
}

