"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSqsToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-solutions-constructs/core");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_certificatemanager_1 = require("aws-cdk-lib/aws-certificatemanager");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_event_sources_1 = require("aws-cdk-lib/aws-lambda-event-sources");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const aws_route53_targets_1 = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
/**
 * @summary The ApiGatewayToSqsToLambda class. Class is very opinionated and does not allow for existing queues or lambdas.
 * Class assumes a pulic domain should be created and the corresponding alias in route53 shall be created
 */
class ApiGatewayToSqsToLambda extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToSqsToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToSqsToLambdaProps} props - user provided props for the construct.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { serviceName, domain, domainCertArn, route53HostedZoneId, deployDeadLetterQueue, lambdaFunction, } = props;
        // Setup the dead letter queue, if applicable
        this.deadLetterQueue = defaults.buildDeadLetterQueue(this, {
            deployDeadLetterQueue: deployDeadLetterQueue,
            deadLetterQueueProps: { queueName: `${serviceName}-dl-queue` },
        });
        // this is not very clean, by default lambda has a 3 seconds timeout
        const lambdaTimeout = lambdaFunction.timeout != aws_cdk_lib_1.Duration.seconds(3)
            ? lambdaFunction.timeout
            : aws_cdk_lib_1.Duration.seconds(30);
        // Setup the queue
        [this.sqsQueue] = defaults.buildQueue(this, `${serviceName}-queue`, {
            deadLetterQueue: this.deadLetterQueue,
            queueProps: {
                queueName: `${serviceName}-queue`,
                visibilityTimeout: lambdaTimeout ? lambdaTimeout : aws_cdk_lib_1.Duration.seconds(30),
            },
        });
        const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(this, `*.${domain}`, domainCertArn);
        const apiGatewayProps = {
            domainName: {
                domainName: `${serviceName}.${domain}`,
                certificate: certificate,
            },
            restApiName: serviceName,
        };
        // Setup API Gateway
        this.apiGateway = new aws_apigateway_1.RestApi(this, `${serviceName}-api-gateway`, apiGatewayProps);
        this.apiGatewayRole = new aws_iam_1.Role(this, `${serviceName}-api-gateway-role`, {
            assumedBy: new aws_iam_1.ServicePrincipal('apigateway.amazonaws.com'),
        });
        this.apiGatewayRole.addToPolicy(new aws_iam_1.PolicyStatement({
            resources: [this.sqsQueue.queueArn],
            actions: ['sqs:SendMessage'],
        }));
        core_1.addProxyMethodToApiResource({
            service: 'sqs',
            path: `${aws_cdk_lib_1.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: this.apiGateway.root,
            requestTemplate: 'Action=SendMessage&MessageBody=$util.urlEncode("$input.body")',
            contentType: "'application/x-www-form-urlencoded'",
            methodOptions: {
                authorizationType: aws_apigateway_1.AuthorizationType.NONE,
            },
        });
        // infering throws error, which is why we have to provide both
        const hostedZone = aws_route53_1.HostedZone.fromHostedZoneAttributes(this, `${domain}-zone`, {
            hostedZoneId: route53HostedZoneId,
            zoneName: domain,
        });
        new aws_route53_1.ARecord(this, `${serviceName}-alias-record`, {
            zone: hostedZone,
            recordName: serviceName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.ApiGateway(this.apiGateway)),
        });
        lambdaFunction.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.sqsQueue, { reportBatchItemFailures: true }));
    }
}
exports.ApiGatewayToSqsToLambda = ApiGatewayToSqsToLambda;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToSqsToLambda[_a] = { fqn: "apigateway-sqs-lambda.ApiGatewayToSqsToLambda", version: "0.0.5" };
//# sourceMappingURL=data:application/json;base64,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