"""ChartWidget组件的完整测试套件。

测试覆盖：
- 基础功能测试（创建、配置）
- 图表设置和图像处理
- 模板渲染和上下文
- 中文字体配置
- 错误处理和边界条件
"""

import base64
import io
from pathlib import Path
from unittest.mock import Mock, patch, MagicMock, mock_open
import pytest

from email_widget.widgets.chart_widget import ChartWidget


class TestChartWidgetInitialization:
    """ChartWidget初始化测试"""

    def test_init_default(self):
        """测试默认初始化"""
        widget = ChartWidget()
        assert widget._image_url is None
        assert widget._title is None
        assert widget._description is None
        assert widget._alt_text == "图表"
        assert widget._data_summary is None
        assert widget._max_width == "100%"

    def test_init_with_id(self):
        """测试带ID的初始化"""
        widget_id = "test_chart_001"
        widget = ChartWidget(widget_id=widget_id)
        assert widget.widget_id == widget_id


class TestChartWidgetBasicMethods:
    """ChartWidget基础方法测试"""

    def test_set_title(self):
        """测试设置标题"""
        widget = ChartWidget()
        title = "测试图表标题"

        result = widget.set_title(title)

        assert result is widget  # 支持链式调用
        assert widget._title == title

    def test_set_description(self):
        """测试设置描述"""
        widget = ChartWidget()
        description = "这是一个测试图表的描述"

        result = widget.set_description(description)

        assert result is widget
        assert widget._description == description

    def test_set_alt_text(self):
        """测试设置替代文本"""
        widget = ChartWidget()
        alt_text = "销售数据图表"

        result = widget.set_alt_text(alt_text)

        assert result is widget
        assert widget._alt_text == alt_text

    def test_set_data_summary(self):
        """测试设置数据摘要"""
        widget = ChartWidget()
        summary = "总销售额: 100万元，增长率: 15%"

        result = widget.set_data_summary(summary)

        assert result is widget
        assert widget._data_summary == summary

    def test_set_max_width(self):
        """测试设置最大宽度"""
        widget = ChartWidget()
        max_width = "800px"

        result = widget.set_max_width(max_width)

        assert result is widget
        assert widget._max_width == max_width


class TestChartWidgetImageHandling:
    """ChartWidget图像处理测试"""

    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    def test_set_image_url_http(self, mock_process):
        """测试设置HTTP图片URL"""
        mock_process.return_value = "data:image/png;base64,fake_data"

        widget = ChartWidget()
        url = "https://example.com/chart.png"
        result = widget.set_image_url(url)

        assert result is widget
        assert widget._image_url == "data:image/png;base64,fake_data"
        mock_process.assert_called_once_with(url, cache=True)

    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    @patch("pathlib.Path.exists")
    def test_set_image_url_local_file_exists(self, mock_exists, mock_process):
        """测试设置本地文件路径（文件存在）"""
        mock_exists.return_value = True
        mock_process.return_value = "data:image/png;base64,local_data"

        widget = ChartWidget()
        file_path = "/path/to/chart.png"
        result = widget.set_image_url(file_path, cache=False)

        assert result is widget
        assert widget._image_url == "data:image/png;base64,local_data"
        mock_process.assert_called_once_with(file_path, cache=False)

    @patch("pathlib.Path.exists")
    def test_set_image_url_local_file_not_exists(self, mock_exists):
        """测试设置本地文件路径（文件不存在）"""
        mock_exists.return_value = False

        widget = ChartWidget()
        file_path = "/nonexistent/path.png"
        result = widget.set_image_url(file_path)

        assert result is widget
        assert widget._image_url is None

    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    def test_set_image_url_pathlib_path(self, mock_process):
        """测试使用pathlib.Path对象"""
        mock_process.return_value = "data:image/png;base64,path_data"

        widget = ChartWidget()
        path_obj = Path("/path/to/chart.png")

        with patch("pathlib.Path.exists", return_value=True):
            result = widget.set_image_url(path_obj)

        assert result is widget
        assert widget._image_url == "data:image/png;base64,path_data"

    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    def test_set_image_url_data_uri(self, mock_process):
        """测试设置data URI"""
        data_uri = "data:image/png;base64,existing_data"
        mock_process.return_value = data_uri

        widget = ChartWidget()
        result = widget.set_image_url(data_uri)

        assert result is widget
        assert widget._image_url == data_uri


class TestChartWidgetChartMethods:
    """ChartWidget图表方法测试"""

    @patch("email_widget.widgets.chart_widget.ChartWidget._configure_chinese_font")
    @patch("email_widget.utils.optional_deps.check_optional_dependency")
    def test_set_chart_matplotlib(self, mock_check_dep, mock_configure_font):
        """测试设置matplotlib图表"""
        mock_check_dep.return_value = True

        # Mock matplotlib对象
        mock_plt = Mock()

        # 模拟savefig方法
        def mock_savefig(buffer, **kwargs):
            buffer.write(b"fake_chart_data")

        mock_plt.savefig = mock_savefig
        mock_plt.close = Mock()

        widget = ChartWidget()
        result = widget.set_chart(mock_plt)

        assert result is widget
        mock_configure_font.assert_called_once()
        mock_plt.close.assert_called_once()
        assert widget._image_url is not None
        assert widget._image_url.startswith("data:image/png;base64,")

    @patch("email_widget.widgets.chart_widget.ChartWidget._configure_chinese_font")
    @patch("email_widget.utils.optional_deps.check_optional_dependency")
    def test_set_chart_with_figure_object(self, mock_check_dep, mock_configure_font):
        """测试直接设置Figure对象"""
        mock_check_dep.return_value = True

        # Mock figure对象
        mock_figure = Mock()

        def mock_savefig(buffer, **kwargs):
            buffer.write(b"figure_chart_data")

        mock_figure.savefig = mock_savefig
        mock_figure.close = Mock()

        widget = ChartWidget()
        result = widget.set_chart(mock_figure)

        assert result is widget
        mock_configure_font.assert_called_once()
        mock_figure.close.assert_called_once()

    @patch("email_widget.widgets.chart_widget.ChartWidget._configure_chinese_font")
    @patch("email_widget.utils.optional_deps.check_optional_dependency")
    def test_set_chart_exception_handling(self, mock_check_dep, mock_configure_font):
        """测试图表设置异常处理"""
        mock_check_dep.return_value = True

        mock_plt = Mock()
        mock_plt.savefig.side_effect = Exception("图表保存失败")

        widget = ChartWidget()
        result = widget.set_chart(mock_plt)

        assert result is widget
        # 错误时图片URL应该被设置为None
        assert widget._image_url is None

    @patch("email_widget.widgets.chart_widget.ChartWidget._configure_chinese_font")
    @patch("email_widget.utils.optional_deps.check_optional_dependency")
    def test_set_chart_basic_functionality(self, mock_check_dep, mock_configure_font):
        """测试图表基本功能"""
        mock_check_dep.return_value = True

        mock_plt = Mock()
        mock_plt.savefig = Mock()
        mock_plt.close = Mock()

        widget = ChartWidget()
        result = widget.set_chart(mock_plt)

        assert result is widget
        mock_configure_font.assert_called_once()
        mock_plt.savefig.assert_called_once()
        mock_plt.close.assert_called_once()


class TestChartWidgetChineseFontHandling:
    """ChartWidget中文字体处理测试"""

    @patch(
        "email_widget.widgets.chart_widget.ChartWidget._import_matplotlib_font_manager"
    )
    @patch("email_widget.utils.optional_deps.check_optional_dependency")
    def test_configure_chinese_font_matplotlib_available(
        self, mock_check_dep, mock_font_manager
    ):
        """测试matplotlib可用时的中文字体配置"""
        mock_check_dep.return_value = True

        widget = ChartWidget()
        widget._configure_chinese_font()

        # 验证中文字体配置被设置
        # 实际调用的是matplotlib.pyplot，不是font_manager
        assert mock_check_dep.called

    @patch("email_widget.utils.optional_deps.check_optional_dependency")
    def test_configure_chinese_font_matplotlib_unavailable(self, mock_check_dep):
        """测试matplotlib不可用时的处理"""
        mock_check_dep.return_value = False

        widget = ChartWidget()
        widget._configure_chinese_font()

        # 应该正常返回而不抛出异常
        assert mock_check_dep.called

    @patch("matplotlib.font_manager.fontManager.ttflist", [])
    @patch("email_widget.utils.optional_deps.check_optional_dependency")
    def test_configure_chinese_font_no_fonts_found(self, mock_check_dep):
        """测试没有找到中文字体的情况"""
        mock_check_dep.return_value = True

        widget = ChartWidget()
        widget._configure_chinese_font()

        # 应该正常处理而不抛出异常


class TestChartWidgetTemplateRendering:
    """ChartWidget模板渲染测试"""

    def test_get_template_name(self):
        """测试获取模板名称"""
        widget = ChartWidget()
        assert widget._get_template_name() == "chart.html"

    def test_get_template_context_empty(self):
        """测试空上下文"""
        widget = ChartWidget()
        context = widget.get_template_context()

        assert isinstance(context, dict)
        assert context == {}  # 当没有image_url时返回空字典

    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    def test_get_template_context_with_data(self, mock_process):
        """测试包含数据的上下文"""
        mock_process.return_value = "data:image/png;base64,test_data"

        widget = ChartWidget()
        widget.set_image_url("https://example.com/chart.png")
        widget.set_title("测试标题")
        widget.set_description("测试描述")
        widget.set_alt_text("测试图表")
        widget.set_data_summary("测试摘要")
        widget.set_max_width("600px")

        context = widget.get_template_context()

        assert context["image_url"] == "data:image/png;base64,test_data"
        assert context["title"] == "测试标题"
        assert context["description"] == "测试描述"
        assert context["alt_text"] == "测试图表"
        assert context["data_summary"] == "测试摘要"
        assert "container_style" in context
        assert "img_style" in context
        assert "title_style" in context
        assert "desc_style" in context
        assert "summary_style" in context

    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    def test_template_context_styles(self, mock_process):
        """测试模板上下文样式"""
        mock_process.return_value = "data:image/png;base64,style_test"

        widget = ChartWidget()
        widget.set_image_url("test.png")  # 需要先设置图片才有context
        widget.set_max_width("800px")

        context = widget.get_template_context()

        # 如果没有图片URL，返回空字典
        if context:
            # 验证样式包含预期内容
            assert "800px" in context["container_style"]
            assert "max-width" in context["img_style"]
        else:
            # 没有图片时返回空字典
            assert context == {}


class TestChartWidgetChaining:
    """ChartWidget链式调用测试"""

    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    def test_method_chaining(self, mock_process):
        """测试方法链式调用"""
        mock_process.return_value = "data:image/png;base64,chain_data"

        widget = (
            ChartWidget()
            .set_title("链式调用标题")
            .set_description("链式调用描述")
            .set_alt_text("链式调用图表")
            .set_data_summary("链式调用摘要")
            .set_max_width("900px")
            .set_image_url("https://example.com/chain.png")
        )

        assert widget._title == "链式调用标题"
        assert widget._description == "链式调用描述"
        assert widget._alt_text == "链式调用图表"
        assert widget._data_summary == "链式调用摘要"
        assert widget._max_width == "900px"
        assert widget._image_url == "data:image/png;base64,chain_data"

    def test_partial_chaining(self):
        """测试部分链式调用"""
        widget = ChartWidget().set_title("部分标题").set_description("部分描述")

        assert widget._title == "部分标题"
        assert widget._description == "部分描述"
        assert widget._alt_text == "图表"  # 默认值
        assert widget._data_summary is None


class TestChartWidgetEdgeCases:
    """ChartWidget边界情况测试"""

    def test_empty_string_values(self):
        """测试空字符串值"""
        widget = ChartWidget()

        widget.set_title("")
        widget.set_description("")
        widget.set_alt_text("")
        widget.set_data_summary("")
        widget.set_max_width("")

        assert widget._title == ""
        assert widget._description == ""
        assert widget._alt_text == ""
        assert widget._data_summary == ""
        assert widget._max_width == ""

    def test_none_values(self):
        """测试None值处理"""
        widget = ChartWidget()

        # 这些方法应该接受字符串，传入None可能会有不同行为
        # 根据实际实现调整测试
        try:
            widget.set_title(None)
            widget.set_description(None)
        except (TypeError, AttributeError):
            # 如果不接受None，这是合理的
            pass

    def test_special_characters_in_text(self):
        """测试文本中的特殊字符"""
        widget = ChartWidget()

        special_title = "图表标题 <>&\"'"
        special_desc = "描述包含特殊字符: <script>alert('test')</script>"

        widget.set_title(special_title)
        widget.set_description(special_desc)

        assert widget._title == special_title
        assert widget._description == special_desc

    def test_unicode_text(self):
        """测试Unicode文本"""
        widget = ChartWidget()

        unicode_title = "图表标题 📊 数据可视化 🎯"
        unicode_desc = "包含Emoji和特殊Unicode字符的描述 ✅ 测试 ⭐"

        widget.set_title(unicode_title)
        widget.set_description(unicode_desc)

        assert widget._title == unicode_title
        assert widget._description == unicode_desc

    def test_very_long_text(self):
        """测试很长的文本"""
        widget = ChartWidget()

        long_title = "很长的标题" * 100
        long_desc = "很长的描述内容" * 200

        widget.set_title(long_title)
        widget.set_description(long_desc)

        assert widget._title == long_title
        assert widget._description == long_desc


class TestChartWidgetErrorHandling:
    """ChartWidget错误处理测试"""

    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    def test_image_processing_error(self, mock_process):
        """测试图像处理错误"""
        mock_process.side_effect = Exception("图像处理失败")

        widget = ChartWidget()

        # set_image_url方法会抛出异常，这是预期行为
        with pytest.raises(Exception):
            widget.set_image_url("https://example.com/error.png")

    def test_invalid_chart_object(self):
        """测试无效的图表对象"""
        widget = ChartWidget()

        # 传入无效对象
        invalid_objects = [None, "not_a_chart", 123, [], {}]

        for invalid_obj in invalid_objects:
            try:
                result = widget.set_chart(invalid_obj)
                assert result is widget
                # 应该没有设置图像URL或设置为None
                assert widget._image_url is None
            except Exception:
                # 如果抛出异常，确保是预期的类型错误
                pass

    @patch("email_widget.widgets.chart_widget.ChartWidget._configure_chinese_font")
    @patch("email_widget.utils.optional_deps.check_optional_dependency")
    def test_chart_save_error(self, mock_check_dep, mock_configure_font):
        """测试图表保存错误"""
        mock_check_dep.return_value = True

        mock_plt = Mock()
        mock_plt.savefig.side_effect = Exception("保存失败")

        widget = ChartWidget()
        result = widget.set_chart(mock_plt)

        assert result is widget
        # 错误时图片URL应该被设置为None或保持不变
        assert widget._image_url is None


class TestChartWidgetIntegration:
    """ChartWidget集成测试"""

    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    def test_complete_workflow(self, mock_process):
        """测试完整工作流程"""
        mock_process.return_value = "data:image/png;base64,complete_data"

        # 创建完整配置的图表Widget
        widget = (
            ChartWidget(widget_id="integration_test")
            .set_title("集成测试图表")
            .set_description("这是一个完整的集成测试")
            .set_alt_text("集成测试图表")
            .set_data_summary("总计: 测试数据")
            .set_max_width("1000px")
            .set_image_url("https://example.com/integration.png")
        )

        # 验证所有属性设置正确
        assert widget.widget_id == "integration_test"
        assert widget._title == "集成测试图表"
        assert widget._description == "这是一个完整的集成测试"
        assert widget._alt_text == "集成测试图表"
        assert widget._data_summary == "总计: 测试数据"
        assert widget._max_width == "1000px"
        assert widget._image_url == "data:image/png;base64,complete_data"

        # 验证模板上下文
        context = widget.get_template_context()
        assert all(
            key in context
            for key in [
                "image_url",
                "title",
                "description",
                "alt_text",
                "data_summary",
                "container_style",
                "img_style",
            ]
        )

    @patch("email_widget.widgets.chart_widget.ChartWidget._configure_chinese_font")
    @patch("email_widget.utils.image_utils.ImageUtils.process_image_source")
    def test_chart_and_image_workflow(self, mock_process, mock_configure_font):
        """测试图表和图像组合工作流程"""
        mock_process.return_value = "data:image/png;base64,image_data"

        # Mock matplotlib
        mock_plt = Mock()
        mock_figure = Mock()
        mock_canvas = Mock()

        mock_plt.gcf.return_value = mock_figure
        mock_figure.canvas = mock_canvas

        def save_to_buffer(buf, **kwargs):
            buf.write(b"chart_workflow_data")
            buf.seek(0)

        mock_canvas.print_figure.side_effect = save_to_buffer

        widget = ChartWidget()

        # 先设置图片，然后设置图表
        widget.set_image_url("https://example.com/initial.png")
        assert widget._image_url == "data:image/png;base64,image_data"

        # 设置图表会覆盖之前的图片
        widget.set_chart(mock_plt)
        assert widget._image_url.startswith("data:image/png;base64,")
        assert widget._image_url.startswith("data:image/png;base64,")

    def test_error_resilience(self):
        """测试错误恢复能力"""
        widget = ChartWidget()

        # 各种错误操作都不应该破坏Widget状态
        try:
            widget.set_image_url("/nonexistent/file.png")  # 文件不存在
            widget.set_chart("invalid_chart")  # 无效图表对象
            widget.set_title("正常标题")  # 正常操作
        except Exception:
            pytest.fail("Widget应该能够恢复错误状态")

        # 验证正常操作仍然有效
        assert widget._title == "正常标题"
