# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sqlean']

package_data = \
{'': ['*'], 'sqlean': ['grammar/*']}

install_requires = \
['black>=21.4b',
 'lark>=1.1.0',
 'pydantic>=1.8.2',
 'rich>=10.0',
 'tomli>=1.0.0',
 'typer>=0.4.0']

entry_points = \
{'console_scripts': ['sqlean = sqlean.main:app']}

setup_kwargs = {
    'name': 'sqlean',
    'version': '0.0.3',
    'description': 'Clean your SQL queries',
    'long_description': '# sqlean\n\n<p align="center">\n  <a href=\'https://coveralls.io/github/oliverxchen/sqlean\'><img src=\'https://coveralls.io/repos/github/oliverxchen/sqlean/badge.svg\' alt=\'Coverage Status\' /></a>\n  <a href="https://www.python.org/"><img alt="Python versions: 3.7, 3.8, 3.9, 3.10" src="https://img.shields.io/badge/python-3.7%20%7C%203.8%20%7C%203.9%20%7C%203.10-blue"></a>\n  <a href="https://pypi.org/project/sqlean/"><img alt="PyPI" src="https://img.shields.io/pypi/v/sqlean"></a>\n  <a href="https://github.com/psf/black"><img alt="Code style: black" src="https://img.shields.io/badge/code%20style-black-000000.svg"></a>\n  <a href="https://github.com/oliverxchen/sqlean/blob/main/LICENSE"><img alt="License: MIT" src="https://img.shields.io/badge/license-MIT-A31F34"></a>\n</p>\n\nsqlean is a Python CLI to clean your SQL queries. It aspires to be the\n[black](https://black.readthedocs.io/en/stable/) for SQL queries:\nan uncompromising query formatter that imposes a strict, strongly-opinionated\nformatting standard, with an easy workflow to achieve that standard.\n\nEarly releases of sqlean will support\n[BigQuery](https://cloud.google.com/bigquery) dialect SQL queries with or\nwithout [dbt](https://www.getdbt.com/) syntax. If there\'s sufficient interest,\nother SQL dialects can be added.\n\n**WARNING: this is very early in sqlean\'s life and it should not be used in\nproduction.**\n\n* [Usage](#usage)\n  * [Installing](#installing)\n  * [sqlean command](#sqlean-command)\n  * [Workflow](#workflow)\n  * [Configuring sqlean](#configuring-sqlean)\n* [Why](#why)\n  * [Current state of sqlean](#sqlean)\n  * [Alternatives](#alternatives)\n* [Formatting SQL](#formatting-sql)\n* [Future plans](#future-plans)\n* [Contributing](#contributing)\n  * [Reporting issues](#reporting-issues)\n  * [Writing a PR](#writing-a-pr)\n* [Important dependencies](#important-dependencies)\n\n## Usage\n\n### Installing\n\nMake sure you\'re in a virtual environment and install\nsqlean with pip:\n\n ```bash\n pip install sqlean\n ```\n\n### sqlean command\n\nThe sqlean CLI has one argument and several options. The one argument in the\nsqlean CLI is the `TARGET` which can be a file or a directory. If the `TARGET`\nis a directory, sqlean will be applied to all SQL files within the directory and\nall sub-directories.\n\nWhen sqlean is run without any options, SQL files will be changed in place so\nthat the files comply with sqlean\'s formatting rules. Here\'s an example of a\nsuccessful run (with no options):\n\n<img src="docs/img/pass_summary.png" width=640 alt="Succesful run of sqlean with summary table."/>\n\nsqlean is thoroughly tested to ensure that original queries have the same\nabstract syntax tree (AST) as the formatted query. In plain English, that means\nthat sqlean should never break a query. But sqlean is very new so if you don\'t\nfully trust sqlean yet, you can use the dry-run option (`-d`/`--dry-run`)\nto display the diff that would happen if sqlean is run for real on the file :\n\n<img src="docs/img/diff.png" width=640 alt="Dry-run of sqlean with diff."/>\n\nBecause sqlean needs to build an AST, if sqlean doesn\'t understand an SQL query\nthen it is unparsable. In this case, the example query is unparsable because\nthere\'s a problem with the query, and the verbose option (`-v`/`--verbose`)\npoints to the problem:\n\n<img src="docs/img/verbose.png" width=640 alt="Verbose run of sqlean with error message."/>\n\nBrief documentation on sqlean commands can be accessed with:\n\n```bash\nsqlean --help\n```\n\n### Workflow\n\nOnce sqlean is production ready, it is ideally used in your continuous\nintegration (CI) workflow so that a pull request that adds or changes SQL\nqueries can only be merged if sqlean runs without errors (i.e. with the final\nmessage "All files passed").\n\nWhen you first introduce sqlean into your workflow, many of your SQL files can be\nautomatically fixed. In an example production query-base of over 1,600 files,\nnearly 60% of queries could be automatically fixed.\n\nThe other SQL files are unparsable. sqlean will need further development to\ntreat these files and you can help by [reporting the issue](#reporting-issues)\nso that the problem can be addressed.\n\nIn the mean time though, the unparsable files can be marked so that sqlean knows\nto ignore them in CI runs. You can do this automatically by running sqlean with\nthe ignore option (`-i`/`--write-ignore`), or just by inserting this comment as\nthe first line of the file that you want ignored:\n\n```sql\n# sqlean ignore\n```\n\nIf some of the issues are addressed by a future version of sqlean, you can\nupgrade sqlean and automatically remove the ignore indicators with the force\n(`-f`/`--force`) option.\n\n### Configuring sqlean\n\nCLI options are for options that can change from one run of `sqlean` to another.\nProject level configuration will not change from one run to another, and must be\nset in [`pyproject.toml`](https://snarky.ca/what-the-heck-is-pyproject-toml/).\nOne exception is the `TARGET` directory or file that can be both a project level\nsetting and also change from one run to another, so it can appear in both\n`pyproject.toml` or as a CLI argument.\n\nIn `pyproject.toml`, sqlean options go in a `[tool.sqlean]` section. The options\nare:\n\n* `includes`: a single path or list of paths to include when sqlean is run. This\n  will be over-ridden by the `TARGET` argument in the CLI when it is supplied.\n  The default value is `None`, in which case `TARGET` must be used.\n* `indent_size`: an integer that indicates the number of spaces in the indent.\n  The default value is 2.\n* `whisper`: (NOT YET IMPLEMENTED) a boolean which should be set to `true` to\n  make all SQL reserved words and function names all lower case. The default\n  value is `false`, which makes all SQL reserved words and function names\n  all-caps.\n* `max_line_length`: (NOT YET IMPLEMENTED) an integer value which adds\n  additional line breaks if a line is longer than this value. The default value\n  is `-1` to indicate that no maximum will be applied.\n* `dialect`: (NOT YET IMPLEMENTED) a string to indicate the SQL dialect used in\n  the project. The default value is `"BIGQUERY"`\n\n## Why\n\nIn the last few years, the usage of SQL in big data processing has steadily\nincreased with the emergence of computational engines like [Google Cloud\nBigQuery](https://cloud.google.com/bigquery), [Amazon\nRedshift](https://aws.amazon.com/redshift/),\n[Snowflake](https://www.snowflake.com/) and others with SQL interfaces.\n\nLeveraging on the growth of SQL, [dbt](https://www.getdbt.com) has gained\nwide-spread adoption as a tool to orchestrate the running of SQL queries. While\nanalytics and feature generation pipelines were previously written\nprimarily in programming languages like Python or Scala, they are now mostly dbt\ncode: a combination of SQL and [jinja](https://jinja.palletsprojects.com/)\ntemplating.\n\nThe tooling around dbt code does not have the maturity as tooling around\nolder languages like Python. We see code formatting and styling as a critical\npart of tooling around a language: consistently formatted code makes it easier to\ncollaborate within a team by making it faster and easier to read other people\'s\ncode.\n\n[Black](https://black.readthedocs.io/) has emerged as the most popular code\nformatter for Python for a couple reasons:\n\n1. There are few options so there is little to argue about when setting it up\nfor the first time.\n1. If two versions of a piece of code have the same AST, they will have the same\nblack formatted output (modulo minor differences in line spacing).\n\nThe strong consistency in formatting means that sometimes the output looks a bit\nstrange or at least different than what you\'re used to. But over time our brains\nadjust and it becomes faster to read black formatted code.\n\nsqlean aspires to play the same rule with SQL/dbt queries. By using sqlean,\nyou\'ll give up control on exactly how your queries are formatted and styled, but\nyou\'ll gain strong consistency in the formatting. No more nitpicking in code\nreviews and no more getting used to other team members\' query styles.\n\n### Current state of sqlean\n\nThese are very early, experimental days in the development of sqlean so these\nnumbers will change. On a production code base of 1,615 queries, 58.7% could be\nparsed and formatted and the other 42.3% could not be parsed. It takes 8.8\nseconds for sqlean to attempt to parse and fix these queries. The time increases\nas more queries can be parsed, so it is expected to take roughly 15 seconds when\nmost or all of the queries can be parsed.\n\n### Alternatives\n\nThere are other SQL formatting/linting tools out there and some may be more\nsuitable for you.\n\n#### [SQLFluff](https://www.sqlfluff.com/)\n\n> SQLFluff is a dialect-flexible and configurable SQL linter. Designed with ELT\napplications in mind, SQLFluff also works with Jinja templating and dbt.\nSQLFluff will auto-fix most linting errors, allowing you to focus your time on\nwhat matters.\n\nsqlfluff is the main alternative to sqlean with wide community adoption and\nactive contributions (4000+ Github stars and 150 contributors). sqlfluff has a\nwide array of rules that can be customized. Time to auto-format queries can vary\nwidely. Running sqlfluff on the production code base of 1,615 queries took too\nlong, so a subset of 617 queries was used for comparison:\n\n|                         | sqlean | sqlfluff |\n| :---------------------- | -----: | -------: |\n| Time to auto-format (s) | 3.8    | 1325.3   |\n| # files auto-formatted  | 388    | 228      |\n| Version                 | 0.0.3  | 0.13.1   |\n\nWhile sqlfluff\'s performance depends on the starting queries, on this sample\nsqlean was able to run nearly 350 times faster than sqlfluff and was able to\nauto-format 70% more files than sqlfluff. sqlfluff was run with the default\nrules using the command: `sqlfluff fix -f --dialect bigquery`.\n\nLanguage: Python\n\n#### [ZetaSQL](https://github.com/google/zetasql)\n\n> ZetaSQL defines a language (grammar, types, data model, and semantics) as well\nas a parser and analyzer.\n\nIt is a Google project (but not official) that is used for BigQuery, Cloud\nSpanner and Dataflow. An improved formatter is in progress, and a\n[fork](https://github.com/Matts966/zetasql-formatter) provides a formatter with\npreserved comments.\n\nLanguage: C++\n\n#### [SQLLanguageServer](https://github.com/joe-re/sql-language-server)\n\nSQLLanguageServer provides auto-completion for editors, parsing and linting and\nhas a VS Code extension. It has support for MySQL, PostgreSQL and SQLite3.\n\nLanguage: TypeScript\n\n#### [dbt-sql-formatter](https://github.com/dbt-labs/dbt-sql-formatter)\n\nFrom dbt Labs, unfortunately this is in an early stage and hasn\'t had\nnew commits in two years.\n\nLanguage: Python\n\n#### [sqlparse](https://github.com/andialbrecht/sqlparse)\n\n> sqlparse is a non-validating SQL parser for Python. It provides support for\nparsing, splitting and formatting SQL statements.\n\nLanguage: Python\n\n#### [SQLint](https://github.com/purcell/sqlint)\n\n> SQLint is a simple command-line linter which reads your SQL files and reports\nany syntax errors or warnings it finds.\n\nLanguage: Ruby\n\n## Formatting SQL\n\nWhenever a formatting choice needs to be made, the guiding question should be:\nif black formatted SQL queries, what choice would black make? By following this\nprinciple, sqlean formatted code will at least look familiar to the analytics\nengineers who use Python and black.\n\nThis is a non-exhaustive list of formatting changes that sqlean makes:\n\n* SQL reserved words like SELECT, AS, FROM, JOIN, etc and function names are\n  written all-caps. A future `--whisper` option will force these to be all\n  lower-case.\n* Select lists are one item per line, with no exceptions. Even if there are only\n  two or three items in the list and writing them on a single line would make a\n  query more compact, there are two main reasons to write them on separate lines:\n  * If more items are added to the list later, more effort on editting is needed\n    and the git diff will be harder to read.\n  * Not making an exception for short lists makes select lists more\n    visually/spatially consistent and therefore faster to read.\n* An explicit "AS" will be added to all alias statements for select items or\n  tables. From _The Zen of Python_:\n  > Explicit is better than implicit.\n* SQL syntax allows both double and single quotes. sqlean processes quoted\n  strings with black so that double quotes are usually used (with exceptions for\n  strings that contain double-quotes).\n* Character spacing: a single space after commas, no space between function\n  names and left parentheses, etc.\n* BigQuery now allows trailing commas in select lists, which obviates\n  the need for leading commas (which were convenient, but aesthetically ugly).\n  sqlean will insert a trailing comma in every select list. The primary reason\n  is to reduce the size of future diffs. If no trailing comma is present:\n\n  ```sql\n  SELECT\n    field_1,\n    field_2,\n    field_3\n  FROM\n    table\n  ```\n\n  then when you add another item, two lines need to be changed and there will be\n  three lines in the diff:\n\n  ```diff\n     field_2,\n  -  field_3\n  +  field_3,\n  +  field_4\n  FROM\n  ```\n\n  If instead there is a trailing comma:\n\n  ```sql\n  SELECT\n    field_1,\n    field_2,\n    field_3,\n  FROM\n    table\n  ```\n\n  then adding an item results in only a single line in the diff:\n\n  ```diff\n     field_3,\n  +  field_4,\n  FROM\n  ```\n\n## Future plans\n\n* Reduce unparsable queries\n* VSCode extension\n* Styling improvements, e.g. dealing with long lines\n* Support for other SQL dialects.\n* SQL hygiene: e.g. warnings for select items in a join query that don\'t have a\n  table reference.\n\n## Contributing\n\n`sqlean` will need a community effort to be capable of parsing all valid SQL\nqueries.\n\n### Reporting issues\n\nIf the query is correctly written but it is unparsable by sqlean, then sqlean\'s\ngrammar needs to be extended to accommodate the valid query. You can help by\nreporting the issue and providing a minimal query that demonstrates the parse\nerror.\n\n### Writing a PR\n\nThe primary PRs for non core-developers are adding to sqlean\'s grammar and\nchanging its styling. We\'ve setup the testing to make it easier to add to the\ngrammar so that more and more elements of valid SQL can be parsed.\n\nSnapshot testing that is somewhat similar to snapshot testing in\n[Jest](https://jestjs.io/docs/snapshot-testing) is used to make sure that:\n\n* when the grammar is changed, parsing changes in expected ways\n* when formatting changes, the format in all preceding snapshot tests change in\n  expected ways\n\n#### Snapshot files\n\nSnapshot files are contained in `sqlean/tests/snapshots` directory.\nEach snapshot file is divided into 3 parts, with each part separated by a line\nwith three dashes:\n\n```text\n---\n```\n\n* The first part is minimal working example (MWE) for the SQL query. This is the\n  only part that needs to be human-written.\n* The second part is the MWE formatted by sqlean, for easy comparison with the\n  raw query written in the first part. This part will be auto-generated.\n* The third part is the AST as represented by a parse tree. This makes is easier\n  to understand what the grammar is doing to parse the query, and how the\n  formatting should work. This part will be auto-generated.\n\n#### Conventions\n\nThe snapshots are located in the `sqlean/tests/snapshots` directory, under\ndifferent sub-directories. The sub-directories are grouped by grammar elements.\nWithin each sub-directory, the files are prefixed by a three digit integer to\npreserve ordering that is natural to the grammar rather than alphabetical\nordering.\n\nThe snapshot files have extension `.snapshot`. If you set up your editor to\nrecognise the `.snapshot` file as an `R` file, you\'ll get decent syntax\nhighlighting of the parse tree.\n\n#### Adding to the grammar\n\nIf you run `sqlean` on a file that contains a valid SQL/dbt query but\n`sqlean` marks it as "unparsable", this indicates that there is an element of\nyour file that is not in `sqlean`\'s grammar. You can contribute to the grammar\nso that the element and the file can be parsed and formatted.\n\n1. If a query of yours cannot be parsed by sqlean, run sqlean on the single file\n   with the verbose option to identify what is causing the problem:\n\n   ```bash\n   sqlean -v <your_file_name>.sql\n   ```\n\n1. Once the unparsable element has been identified, write a minimal working\n   example (MWE) of an SQL query that contains the element, and which sqlean\n   cannot parse.\n1. Identify the sub-directory in the `sqlean/tests/snapshots` directory where\n   the element should go, or if there needs to be a new sub-directory.\n1. Within the sub-directories, create a new file named according to the\n   [conventions](#conventions) and put your MWE in the file.\n1. Run the snapshot test for this file:\n\n   ```bash\n   make snapshot L=tests/snapshots/<sub_dir>/<new_file>\n   ```\n\n   This will fail with an error printed in the snapshot file which can guide\n   you on modifying the grammar so that the file can be parsed.\n\n   If there are a number of different new files, you can use the `M` (match)\n   argument instead of the `L` (location) argument to match a string within the\n   file names. For example,\n\n   ```bash\n   make snapshot M=dbt\n   ```\n\n   will run the snapshot tests on all snapshot files that contain "dbt" in the\n   file name.\n1. The grammar for sqlean is defined in `.lark` files in `sqlean/sqlean/grammar`.\n   sqlean uses [Lark](https://lark-parser.readthedocs.io/en/latest/) to parse queries\n   once the grammar is defined. You can read the Lark docs or just short cut to the\n   [Lark cheat sheet](https://lark-parser.readthedocs.io/en/latest/_static/lark_cheatsheet.pdf).\n1. As you modify the grammar, you can check the output of the parser by running\n   the `make snapshot L/M=...` command and checking what is printed in the second or\n   third section of the snapshot file. Errors will appear in the second and\n   third section if there is a problem with the grammar.\n1. You\'re done modifying the grammar once the parse tree appears in the third section\n   of the snapshot file and the tree is as you want it.\n\n#### Change the formatting\n\nOnce you\'re happy with the parse tree, you can move on to formatting the output.\n\n1. If the parse tree is output in the third section, there may be a styling\n   related error in the second section. The error will indicate a node that is\n   missing, which you can add in `sqlean/sqlean/sql_styler.py`.\n1. Similar to the grammar, as you modify the formatting, you can check the output\n   of the formatter by running the `make snapshot L/M=...` command and checking what is printed\n   in the second section of the snapshot file. Errors will appear in the second\n   section if there is a problem with the formatter.\n1. Once you\'re happy with the formatting, you should run the tests on all existing\n   snapshot files with:\n\n   ```bash\n   make snapshot\n   ```\n\n   If your parsing or formatting changed other files, it will show up here. These\n   file changes will show up in git so it will be easy to see whether the change\n   was intentional or not.\n\n## Important dependencies\n\nA few dependencies are critical to sqlean. We are eternally grateful for these\namazing projects:\n\n* [Lark](https://github.com/lark-parser/lark) is used to parse queries into a\n  tree that can then be styled. sqlean\'s parsing is lightning fast because of\n  lark.\n* [Black](https://github.com/psf/black) is used to help style some bits in\n  queries, and of course is a role model for sqlean.\n* [Rich](https://github.com/Textualize/rich) makes the output from sqlean easier\n  for humans to read.\n* [Typer](https://github.com/tiangolo/typer) made the sqlean CLI fast to\n  develop.\n',
    'author': 'Oliver Chen',
    'author_email': 'oliverxchen@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/oliverxchen/sqlean',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
