"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MachineImageType = exports.DefaultCapacityType = exports.CoreDnsComputeType = exports.CpuArch = exports.NodeType = exports.EksOptimizedImage = exports.Cluster = exports.ClusterLoggingTypes = exports.KubernetesVersion = exports.EndpointAccess = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const ssm = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const YAML = require("yaml");
const alb_controller_1 = require("./alb-controller");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const instance_types_1 = require("./instance-types");
const k8s_manifest_1 = require("./k8s-manifest");
const k8s_object_value_1 = require("./k8s-object-value");
const k8s_patch_1 = require("./k8s-patch");
const kubectl_provider_1 = require("./kubectl-provider");
const managed_nodegroup_1 = require("./managed-nodegroup");
const oidc_provider_1 = require("./oidc-provider");
const bottlerocket_1 = require("./private/bottlerocket");
const service_account_1 = require("./service-account");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * Endpoint access characteristics.
 */
class EndpointAccess {
    constructor(
    /**
     * Configuration properties.
     *
     * @internal
     */
    _config) {
        this._config = _config;
        if (!_config.publicAccess && _config.publicCidrs && _config.publicCidrs.length > 0) {
            throw new Error('CIDR blocks can only be configured when public access is enabled');
        }
    }
    /**
     * Restrict public access to specific CIDR blocks.
     * If public access is disabled, this method will result in an error.
     *
     * @param cidr CIDR blocks.
     */
    onlyFrom(...cidr) {
        if (!this._config.privateAccess) {
            // when private access is disabled, we can't restric public
            // access since it will render the kubectl provider unusable.
            throw new Error('Cannot restric public access to endpoint when private access is disabled. Use PUBLIC_AND_PRIVATE.onlyFrom() instead.');
        }
        return new EndpointAccess({
            ...this._config,
            // override CIDR
            publicCidrs: cidr,
        });
    }
}
exports.EndpointAccess = EndpointAccess;
_a = JSII_RTTI_SYMBOL_1;
EndpointAccess[_a] = { fqn: "@aws-cdk/aws-eks.EndpointAccess", version: "1.154.0" };
/**
 * The cluster endpoint is accessible from outside of your VPC.
 * Worker node traffic will leave your VPC to connect to the endpoint.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @param cidr The CIDR blocks.
 */
EndpointAccess.PUBLIC = new EndpointAccess({ privateAccess: false, publicAccess: true });
/**
 * The cluster endpoint is only accessible through your VPC.
 * Worker node traffic to the endpoint will stay within your VPC.
 */
EndpointAccess.PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: false });
/**
 * The cluster endpoint is accessible from outside of your VPC.
 * Worker node traffic to the endpoint will stay within your VPC.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC_AND_PRIVATE.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @param cidr The CIDR blocks.
 */
EndpointAccess.PUBLIC_AND_PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: true });
/**
 * Kubernetes cluster version
 */
class KubernetesVersion {
    /**
     *
     * @param version cluster version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * Custom cluster version
     * @param version custom version number
     */
    static of(version) { return new KubernetesVersion(version); }
}
exports.KubernetesVersion = KubernetesVersion;
_b = JSII_RTTI_SYMBOL_1;
KubernetesVersion[_b] = { fqn: "@aws-cdk/aws-eks.KubernetesVersion", version: "1.154.0" };
/**
 * Kubernetes version 1.14
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_14 = KubernetesVersion.of('1.14');
/**
 * Kubernetes version 1.15
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_15 = KubernetesVersion.of('1.15');
/**
 * Kubernetes version 1.16
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_16 = KubernetesVersion.of('1.16');
/**
 * Kubernetes version 1.17
 * @deprecated Use newer version of EKS
 */
KubernetesVersion.V1_17 = KubernetesVersion.of('1.17');
/**
 * Kubernetes version 1.18
 */
KubernetesVersion.V1_18 = KubernetesVersion.of('1.18');
/**
 * Kubernetes version 1.19
 */
KubernetesVersion.V1_19 = KubernetesVersion.of('1.19');
/**
 * Kubernetes version 1.20
 */
KubernetesVersion.V1_20 = KubernetesVersion.of('1.20');
/**
 * Kubernetes version 1.21
 */
KubernetesVersion.V1_21 = KubernetesVersion.of('1.21');
/**
 * EKS cluster logging types
 */
var ClusterLoggingTypes;
(function (ClusterLoggingTypes) {
    /**
     * Logs pertaining to API requests to the cluster.
     */
    ClusterLoggingTypes["API"] = "api";
    /**
     * Logs pertaining to cluster access via the Kubernetes API.
     */
    ClusterLoggingTypes["AUDIT"] = "audit";
    /**
     * Logs pertaining to authentication requests into the cluster.
     */
    ClusterLoggingTypes["AUTHENTICATOR"] = "authenticator";
    /**
     * Logs pertaining to state of cluster controllers.
     */
    ClusterLoggingTypes["CONTROLLER_MANAGER"] = "controllerManager";
    /**
     * Logs pertaining to scheduling decisions.
     */
    ClusterLoggingTypes["SCHEDULER"] = "scheduler";
})(ClusterLoggingTypes = exports.ClusterLoggingTypes || (exports.ClusterLoggingTypes = {}));
class ClusterBase extends core_1.Resource {
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesResource` object.
     */
    addManifest(id, ...manifest) {
        return new k8s_manifest_1.KubernetesManifest(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` construct
     */
    addHelmChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    /**
     * Defines a CDK8s chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param chart the cdk8s chart.
     * @returns a `KubernetesManifest` construct representing the chart.
     */
    addCdk8sChart(id, chart, options = {}) {
        const cdk8sChart = chart;
        // see https://github.com/awslabs/cdk8s/blob/master/packages/cdk8s/src/chart.ts#L84
        if (typeof cdk8sChart.toJson !== 'function') {
            throw new Error(`Invalid cdk8s chart. Must contain a 'toJson' method, but found ${typeof cdk8sChart.toJson}`);
        }
        const manifest = new k8s_manifest_1.KubernetesManifest(this, id, {
            cluster: this,
            manifest: cdk8sChart.toJson(),
            ...options,
        });
        return manifest;
    }
    addServiceAccount(id, options = {}) {
        return new service_account_1.ServiceAccount(this, id, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Installs the AWS spot instance interrupt handler on the cluster if it's not
     * already added.
     */
    addSpotInterruptHandler() {
        if (!this._spotInterruptHandler) {
            this._spotInterruptHandler = this.addHelmChart('spot-interrupt-handler', {
                chart: 'aws-node-termination-handler',
                version: '0.18.0',
                repository: 'https://aws.github.io/eks-charts',
                namespace: 'kube-system',
                values: {
                    nodeSelector: {
                        lifecycle: user_data_1.LifecycleLabel.SPOT,
                    },
                },
            });
        }
        return this._spotInterruptHandler;
    }
    /**
     * Connect capacity in the form of an existing AutoScalingGroup to the EKS cluster.
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addAutoScalingGroupCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    connectAutoScalingGroupCapacity(autoScalingGroup, options) {
        var _e, _f, _g, _h;
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        // allow traffic to/from managed node groups (eks attaches this security group to the managed nodes)
        autoScalingGroup.addSecurityGroup(this.clusterSecurityGroup);
        const bootstrapEnabled = (_e = options.bootstrapEnabled) !== null && _e !== void 0 ? _e : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = options.machineImageType === MachineImageType.BOTTLEROCKET ?
                user_data_1.renderBottlerocketUserData(this) :
                user_data_1.renderAmazonLinuxUserData(this, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        // https://docs.aws.amazon.com/eks/latest/userguide/worker.html
        core_1.Tags.of(autoScalingGroup).add(`kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
            // exclude security groups to avoid multiple "owned" security groups.
            // (the cluster security group already has this tag)
            excludeResourceTypes: ['AWS::EC2::SecurityGroup'],
        });
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        let mapRole = (_f = options.mapRole) !== null && _f !== void 0 ? _f : true;
        if (mapRole && !(this instanceof Cluster)) {
            // do the mapping...
            core_1.Annotations.of(autoScalingGroup).addWarning('Auto-mapping aws-auth role for imported cluster is not supported, please map role manually');
            mapRole = false;
        }
        if (mapRole) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn,
            });
        }
        const addSpotInterruptHandler = (_g = options.spotInterruptHandler) !== null && _g !== void 0 ? _g : true;
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice && addSpotInterruptHandler) {
            this.addSpotInterruptHandler();
        }
        if (this instanceof Cluster) {
            // the controller runs on the worker nodes so they cannot
            // be deleted before the controller.
            (_h = this.albController) === null || _h === void 0 ? void 0 : _h.node.addDependency(autoScalingGroup);
        }
    }
}
;
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 */
class Cluster extends ClusterBase {
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param id the id of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props) {
        var _e, _f, _g, _h, _j, _k, _l, _m;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * If the cluster has one (or more) FargateProfiles associated, this array
         * will hold a reference to each.
         */
        this._fargateProfiles = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const stack = core_1.Stack.of(this);
        this.prune = (_e = props.prune) !== null && _e !== void 0 ? _e : true;
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.kubectlLambdaRole = props.kubectlLambdaRole ? props.kubectlLambdaRole : undefined;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.vpcSubnets = (_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        const selectedSubnetIdsPerGroup = this.vpcSubnets.map(s => this.vpc.selectSubnets(s).subnetIds);
        if (selectedSubnetIdsPerGroup.some(core_1.Token.isUnresolved) && selectedSubnetIdsPerGroup.length > 1) {
            throw new Error('eks.Cluster: cannot select multiple subnet groups from a VPC imported from list tokens with unknown length. Select only one subnet group, pass a length to Fn.split, or switch to Vpc.fromLookup.');
        }
        // Get subnetIds for all selected subnets
        const subnetIds = Array.from(new Set(flatten(selectedSubnetIdsPerGroup)));
        this.logging = props.clusterLogging ? {
            clusterLogging: [
                {
                    enabled: true,
                    types: Object.values(props.clusterLogging),
                },
            ],
        } : undefined;
        this.endpointAccess = (_g = props.endpointAccess) !== null && _g !== void 0 ? _g : EndpointAccess.PUBLIC_AND_PRIVATE;
        this.kubectlEnvironment = props.kubectlEnvironment;
        this.kubectlLayer = props.kubectlLayer;
        this.kubectlMemory = props.kubectlMemory;
        this.onEventLayer = props.onEventLayer;
        this.clusterHandlerSecurityGroup = props.clusterHandlerSecurityGroup;
        const privateSubnets = this.selectPrivateSubnets().slice(0, 16);
        const publicAccessDisabled = !this.endpointAccess._config.publicAccess;
        const publicAccessRestricted = !publicAccessDisabled
            && this.endpointAccess._config.publicCidrs
            && this.endpointAccess._config.publicCidrs.length !== 0;
        // validate endpoint access configuration
        if (privateSubnets.length === 0 && publicAccessDisabled) {
            // no private subnets and no public access at all, no good.
            throw new Error('Vpc must contain private subnets when public endpoint access is disabled');
        }
        if (privateSubnets.length === 0 && publicAccessRestricted) {
            // no private subnets and public access is restricted, no good.
            throw new Error('Vpc must contain private subnets when public endpoint access is restricted');
        }
        const placeClusterHandlerInVpc = (_h = props.placeClusterHandlerInVpc) !== null && _h !== void 0 ? _h : false;
        if (placeClusterHandlerInVpc && privateSubnets.length === 0) {
            throw new Error('Cannot place cluster handler in the VPC since no private subnets could be selected');
        }
        if (props.clusterHandlerSecurityGroup && !placeClusterHandlerInVpc) {
            throw new Error('Cannot specify clusterHandlerSecurityGroup without placeClusterHandlerInVpc set to true');
        }
        const resource = this._clusterResource = new cluster_resource_1.ClusterResource(this, 'Resource', {
            name: this.physicalName,
            environment: props.clusterHandlerEnvironment,
            roleArn: this.role.roleArn,
            version: props.version.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
            ...(props.secretsEncryptionKey ? {
                encryptionConfig: [{
                        provider: {
                            keyArn: props.secretsEncryptionKey.keyArn,
                        },
                        resources: ['secrets'],
                    }],
            } : {}),
            kubernetesNetworkConfig: props.serviceIpv4Cidr ? {
                serviceIpv4Cidr: props.serviceIpv4Cidr,
            } : undefined,
            endpointPrivateAccess: this.endpointAccess._config.privateAccess,
            endpointPublicAccess: this.endpointAccess._config.publicAccess,
            publicAccessCidrs: this.endpointAccess._config.publicCidrs,
            secretsEncryptionKey: props.secretsEncryptionKey,
            vpc: this.vpc,
            subnets: placeClusterHandlerInVpc ? privateSubnets : undefined,
            clusterHandlerSecurityGroup: this.clusterHandlerSecurityGroup,
            onEventLayer: this.onEventLayer,
            tags: props.tags,
            logging: this.logging,
        });
        if (this.endpointAccess._config.privateAccess && privateSubnets.length !== 0) {
            // when private access is enabled and the vpc has private subnets, lets connect
            // the provider to the vpc so that it will work even when restricting public access.
            // validate VPC properties according to: https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html
            if (this.vpc instanceof ec2.Vpc && !(this.vpc.dnsHostnamesEnabled && this.vpc.dnsSupportEnabled)) {
                throw new Error('Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled. Use `enableDnsHostnames: true` and `enableDnsSupport: true` when creating the VPC.');
            }
            this.kubectlPrivateSubnets = privateSubnets;
            // the vpc must exist in order to properly delete the cluster (since we run `kubectl delete`).
            // this ensures that.
            this._clusterResource.node.addDependency(this.vpc);
        }
        this.adminRole = resource.adminRole;
        // we use an SSM parameter as a barrier because it's free and fast.
        this._kubectlReadyBarrier = new core_1.CfnResource(this, 'KubectlReadyBarrier', {
            type: 'AWS::SSM::Parameter',
            properties: {
                Type: 'String',
                Value: 'aws:cdk:eks:kubectl-ready',
            },
        });
        // add the cluster resource itself as a dependency of the barrier
        this._kubectlReadyBarrier.node.addDependency(this._clusterResource);
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        this.clusterSecurityGroupId = resource.attrClusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = resource.attrEncryptionConfigKeyArn;
        this.clusterSecurityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterSecurityGroup', this.clusterSecurityGroupId);
        this.connections = new ec2.Connections({
            securityGroups: [this.clusterSecurityGroup, securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        // we can use the cluster security group since its already attached to the cluster
        // and configured to allow connections from itself.
        this.kubectlSecurityGroup = this.clusterSecurityGroup;
        // use the cluster creation role to issue kubectl commands against the cluster because when the
        // cluster is first created, that's the only role that has "system:masters" permissions
        this.kubectlRole = this.adminRole;
        this._kubectlResourceProvider = this.defineKubectlProvider();
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // if an explicit role is not configured, define a masters role that can
        // be assumed by anyone in the account (with sts:AssumeRole permissions of
        // course)
        const mastersRole = (_j = props.mastersRole) !== null && _j !== void 0 ? _j : new iam.Role(this, 'MastersRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // map the IAM role to the `system:masters` group.
        this.awsAuth.addMastersRole(mastersRole);
        if (props.outputMastersRoleArn) {
            new core_1.CfnOutput(this, 'MastersRoleArn', { value: mastersRole.roleArn });
        }
        commonCommandOptions.push(`--role-arn ${mastersRole.roleArn}`);
        if (props.albController) {
            this.albController = alb_controller_1.AlbController.create(this, { ...props.albController, cluster: this });
        }
        // allocate default capacity if non-zero (or default).
        const minCapacity = (_k = props.defaultCapacity) !== null && _k !== void 0 ? _k : DEFAULT_CAPACITY_COUNT;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === DefaultCapacityType.EC2 ?
                this.addAutoScalingGroupCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== DefaultCapacityType.EC2 ?
                this.addNodegroupCapacity('DefaultCapacity', { instanceTypes: [instanceType], minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = (_l = props.outputConfigCommand) !== null && _l !== void 0 ? _l : true;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        this.defineCoreDnsComputeType((_m = props.coreDnsComputeType) !== null && _m !== void 0 ? _m : CoreDnsComputeType.EC2);
    }
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_ClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromClusterAttributes);
            }
            throw error;
        }
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Fetch the load balancer address of a service of type 'LoadBalancer'.
     *
     * @param serviceName The name of the service.
     * @param options Additional operation options.
     */
    getServiceLoadBalancerAddress(serviceName, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_ServiceLoadBalancerAddressOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getServiceLoadBalancerAddress);
            }
            throw error;
        }
        const loadBalancerAddress = new k8s_object_value_1.KubernetesObjectValue(this, `${serviceName}LoadBalancerAddress`, {
            cluster: this,
            objectType: 'service',
            objectName: serviceName,
            objectNamespace: options.namespace,
            jsonPath: '.status.loadBalancer.ingress[0].hostname',
            timeout: options.timeout,
        });
        return loadBalancerAddress.value;
    }
    /**
     * Fetch the load balancer address of an ingress backed by a load balancer.
     *
     * @param ingressName The name of the ingress.
     * @param options Additional operation options.
     */
    getIngressLoadBalancerAddress(ingressName, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_IngressLoadBalancerAddressOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getIngressLoadBalancerAddress);
            }
            throw error;
        }
        const loadBalancerAddress = new k8s_object_value_1.KubernetesObjectValue(this, `${ingressName}LoadBalancerAddress`, {
            cluster: this,
            objectType: 'ingress',
            objectName: ingressName,
            objectNamespace: options.namespace,
            jsonPath: '.status.loadBalancer.ingress[0].hostname',
            timeout: options.timeout,
        });
        return loadBalancerAddress.value;
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Note that if you specify `updateType: RollingUpdate` or `updateType: ReplacingUpdate`, your nodes might be replaced at deploy
     * time without notice in case the recommended AMI for your machine image type has been updated by AWS.
     * The default behavior for `updateType` is `None`, which means only new instances will be launched using the new AMI.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * In addition, the [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addAutoScalingGroupCapacity(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_AutoScalingGroupCapacityOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAutoScalingGroupCapacity);
            }
            throw error;
        }
        if (options.machineImageType === MachineImageType.BOTTLEROCKET && options.bootstrapOptions !== undefined) {
            throw new Error('bootstrapOptions is not supported for Bottlerocket');
        }
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImageType === MachineImageType.BOTTLEROCKET ?
                new bottlerocket_1.BottleRocketImage({
                    kubernetesVersion: this.version.version,
                }) :
                new EksOptimizedImage({
                    nodeType: nodeTypeForInstanceType(options.instanceType),
                    cpuArch: cpuArchForInstanceType(options.instanceType),
                    kubernetesVersion: this.version.version,
                }),
        });
        this.connectAutoScalingGroupCapacity(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
            machineImageType: options.machineImageType,
            spotInterruptHandler: options.spotInterruptHandler,
        });
        if (nodeTypeForInstanceType(options.instanceType) === NodeType.INFERENTIA) {
            this.addNeuronDevicePlugin();
        }
        return asg;
    }
    /**
     * Add managed nodegroup to this Amazon EKS cluster
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @param id The ID of the nodegroup
     * @param options options for creating a new nodegroup
     */
    addNodegroupCapacity(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_NodegroupOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addNodegroupCapacity);
            }
            throw error;
        }
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth() {
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer url.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuerUrl() {
        return this._clusterResource.attrOpenIdConnectIssuerUrl;
    }
    /**
     * If this cluster is kubectl-enabled, returns the OpenID Connect issuer.
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     * @attribute
     */
    get clusterOpenIdConnectIssuer() {
        return this._clusterResource.attrOpenIdConnectIssuer;
    }
    /**
     * An `OpenIdConnectProvider` resource associated with this cluster, and which can be used
     * to link this cluster to AWS IAM.
     *
     * A provider will only be defined if this property is accessed (lazy initialization).
     */
    get openIdConnectProvider() {
        if (!this._openIdConnectProvider) {
            this._openIdConnectProvider = new oidc_provider_1.OpenIdConnectProvider(this, 'OpenIdConnectProvider', {
                url: this.clusterOpenIdConnectIssuerUrl,
            });
        }
        return this._openIdConnectProvider;
    }
    /**
     * Adds a Fargate profile to this cluster.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param id the id of this profile
     * @param options profile options
     */
    addFargateProfile(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_FargateProfileOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addFargateProfile);
            }
            throw error;
        }
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Internal API used by `FargateProfile` to keep inventory of Fargate profiles associated with
     * this cluster, for the sake of ensuring the profiles are created sequentially.
     *
     * @returns the list of FargateProfiles attached to this cluster, including the one just attached.
     * @internal
     */
    _attachFargateProfile(fargateProfile) {
        this._fargateProfiles.push(fargateProfile);
        // add all profiles as a dependency of the "kubectl-ready" barrier because all kubectl-
        // resources can only be deployed after all fargate profiles are created.
        this._kubectlReadyBarrier.node.addDependency(fargateProfile);
        return this._fargateProfiles;
    }
    /**
     * Adds a resource scope that requires `kubectl` to this cluster and returns
     * the `KubectlProvider` which is the custom resource provider that should be
     * used as the resource provider.
     *
     * Called from `HelmResource` and `KubernetesResource`
     *
     * @param resourceScope the construct scope in which kubectl resources are defined.
     *
     * @internal
     */
    _attachKubectlResourceScope(resourceScope) {
        constructs_1.Node.of(resourceScope).addDependency(this._kubectlReadyBarrier);
        return this._kubectlResourceProvider;
    }
    defineKubectlProvider() {
        const uid = '@aws-cdk/aws-eks.KubectlProvider';
        // since we can't have the provider connect to multiple networks, and we
        // wanted to avoid resource tear down, we decided for now that we will only
        // support a single EKS cluster per CFN stack.
        if (this.stack.node.tryFindChild(uid)) {
            throw new Error('Only a single EKS cluster can be defined within a CloudFormation stack');
        }
        return new kubectl_provider_1.KubectlProvider(this.stack, uid, { cluster: this });
    }
    selectPrivateSubnets() {
        const privateSubnets = [];
        const vpcPrivateSubnetIds = this.vpc.privateSubnets.map(s => s.subnetId);
        const vpcPublicSubnetIds = this.vpc.publicSubnets.map(s => s.subnetId);
        for (const placement of this.vpcSubnets) {
            for (const subnet of this.vpc.selectSubnets(placement).subnets) {
                if (vpcPrivateSubnetIds.includes(subnet.subnetId)) {
                    // definitely private, take it.
                    privateSubnets.push(subnet);
                    continue;
                }
                if (vpcPublicSubnetIds.includes(subnet.subnetId)) {
                    // definitely public, skip it.
                    continue;
                }
                // neither public and nor private - what is it then? this means its a subnet instance that was explicitly passed
                // in the subnet selection. since ISubnet doesn't contain information on type, we have to assume its private and let it
                // fail at deploy time :\ (its better than filtering it out and preventing a possibly successful deployment)
                privateSubnets.push(subnet);
            }
        }
        return privateSubnets;
    }
    /**
     * Installs the Neuron device plugin on the cluster if it's not
     * already added.
     */
    addNeuronDevicePlugin() {
        if (!this._neuronDevicePlugin) {
            const fileContents = fs.readFileSync(path.join(__dirname, 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            this._neuronDevicePlugin = this.addManifest('NeuronDevicePlugin', sanitized);
        }
        return this._neuronDevicePlugin;
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) || core_1.Token.isUnresolved([subnet.subnetId]) ? '' : ` ${subnet.subnetId}`;
                    core_1.Annotations.of(this).addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                core_1.Tags.of(subnet).add(tag, '1');
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType,
                        },
                    },
                },
            },
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2),
        });
    }
}
exports.Cluster = Cluster;
_c = JSII_RTTI_SYMBOL_1;
Cluster[_c] = { fqn: "@aws-cdk/aws-eks.Cluster", version: "1.154.0" };
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends ClusterBase {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id);
        this.props = props;
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.clusterArn = this.stack.formatArn(cluster_resource_1.clusterArnComponents(props.clusterName));
        this.kubectlRole = props.kubectlRoleArn ? iam.Role.fromRoleArn(this, 'KubectlRole', props.kubectlRoleArn) : undefined;
        this.kubectlSecurityGroup = props.kubectlSecurityGroupId ? ec2.SecurityGroup.fromSecurityGroupId(this, 'KubectlSecurityGroup', props.kubectlSecurityGroupId) : undefined;
        this.kubectlEnvironment = props.kubectlEnvironment;
        this.kubectlPrivateSubnets = props.kubectlPrivateSubnetIds ? props.kubectlPrivateSubnetIds.map((subnetid, index) => ec2.Subnet.fromSubnetId(this, `KubectlSubnet${index}`, subnetid)) : undefined;
        this.kubectlLayer = props.kubectlLayer;
        this.kubectlMemory = props.kubectlMemory;
        this.clusterHandlerSecurityGroup = props.clusterHandlerSecurityGroupId ? ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterHandlerSecurityGroup', props.clusterHandlerSecurityGroupId) : undefined;
        this.kubectlProvider = props.kubectlProvider;
        this.onEventLayer = props.onEventLayer;
        this.prune = (_e = props.prune) !== null && _e !== void 0 ? _e : true;
        let i = 1;
        for (const sgid of (_f = props.securityGroupIds) !== null && _f !== void 0 ? _f : []) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgid));
            i++;
        }
        if (props.clusterSecurityGroupId) {
            this._clusterSecurityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterSecurityGroup', this.clusterSecurityGroupId);
            this.connections.addSecurityGroup(this._clusterSecurityGroup);
        }
    }
    get vpc() {
        if (!this.props.vpc) {
            throw new Error('"vpc" is not defined for this imported cluster');
        }
        return this.props.vpc;
    }
    get clusterSecurityGroup() {
        if (!this._clusterSecurityGroup) {
            throw new Error('"clusterSecurityGroup" is not defined for this imported cluster');
        }
        return this._clusterSecurityGroup;
    }
    get clusterSecurityGroupId() {
        if (!this.props.clusterSecurityGroupId) {
            throw new Error('"clusterSecurityGroupId" is not defined for this imported cluster');
        }
        return this.props.clusterSecurityGroupId;
    }
    get clusterEndpoint() {
        if (!this.props.clusterEndpoint) {
            throw new Error('"clusterEndpoint" is not defined for this imported cluster');
        }
        return this.props.clusterEndpoint;
    }
    get clusterCertificateAuthorityData() {
        if (!this.props.clusterCertificateAuthorityData) {
            throw new Error('"clusterCertificateAuthorityData" is not defined for this imported cluster');
        }
        return this.props.clusterCertificateAuthorityData;
    }
    get clusterEncryptionConfigKeyArn() {
        if (!this.props.clusterEncryptionConfigKeyArn) {
            throw new Error('"clusterEncryptionConfigKeyArn" is not defined for this imported cluster');
        }
        return this.props.clusterEncryptionConfigKeyArn;
    }
    get openIdConnectProvider() {
        if (!this.props.openIdConnectProvider) {
            throw new Error('"openIdConnectProvider" is not defined for this imported cluster');
        }
        return this.props.openIdConnectProvider;
    }
    get awsAuth() {
        throw new Error('"awsAuth" is not supported on imported clusters');
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props = {}) {
        var _e, _f, _g;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_eks_EksOptimizedImageProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.nodeType = (_e = props.nodeType) !== null && _e !== void 0 ? _e : NodeType.STANDARD;
        this.cpuArch = (_f = props.cpuArch) !== null && _f !== void 0 ? _f : CpuArch.X86_64;
        this.kubernetesVersion = (_g = props.kubernetesVersion) !== null && _g !== void 0 ? _g : LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? this.cpuArch === CpuArch.X86_64 ?
                'amazon-linux-2/' : 'amazon-linux-2-arm64/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux-2-gpu/' : '')
            + (this.nodeType === NodeType.INFERENTIA ? 'amazon-linux-2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
_d = JSII_RTTI_SYMBOL_1;
EksOptimizedImage[_d] = { fqn: "@aws-cdk/aws-eks.EksOptimizedImage", version: "1.154.0" };
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Standard instances
     */
    NodeType["STANDARD"] = "Standard";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
    /**
     * Inferentia instances
     */
    NodeType["INFERENTIA"] = "INFERENTIA";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * CPU architecture
 */
var CpuArch;
(function (CpuArch) {
    /**
     * arm64 CPU type
     */
    CpuArch["ARM_64"] = "arm64";
    /**
     * x86_64 CPU type
     */
    CpuArch["X86_64"] = "x86_64";
})(CpuArch = exports.CpuArch || (exports.CpuArch = {}));
/**
 * The type of compute resources to use for CoreDNS.
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    /**
     * Deploy CoreDNS on EC2 instances.
     */
    CoreDnsComputeType["EC2"] = "ec2";
    /**
     * Deploy CoreDNS on Fargate-managed instances.
     */
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
/**
 * The default capacity type for the cluster
 */
var DefaultCapacityType;
(function (DefaultCapacityType) {
    /**
     * managed node group
     */
    DefaultCapacityType[DefaultCapacityType["NODEGROUP"] = 0] = "NODEGROUP";
    /**
     * EC2 autoscaling group
     */
    DefaultCapacityType[DefaultCapacityType["EC2"] = 1] = "EC2";
})(DefaultCapacityType = exports.DefaultCapacityType || (exports.DefaultCapacityType = {}));
/**
 * The machine image type
 */
var MachineImageType;
(function (MachineImageType) {
    /**
     * Amazon EKS-optimized Linux AMI
     */
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    /**
     * Bottlerocket AMI
     */
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
function nodeTypeForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.gpu.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU :
        instance_types_1.INSTANCE_TYPES.inferentia.includes(instanceType.toString().substring(0, 4)) ? NodeType.INFERENTIA :
            NodeType.STANDARD;
}
function cpuArchForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.graviton2.includes(instanceType.toString().substring(0, 3)) ? CpuArch.ARM_64 :
        instance_types_1.INSTANCE_TYPES.graviton.includes(instanceType.toString().substring(0, 2)) ? CpuArch.ARM_64 :
            CpuArch.X86_64;
}
function flatten(xss) {
    return Array.prototype.concat.call([], ...xss);
}
//# sourceMappingURL=data:application/json;base64,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