"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Origin = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const cdk = require("@aws-cdk/core");
const http_origin_1 = require("./http-origin");
/**
 * An Origin that is backed by an S3 bucket.
 *
 * If the bucket is configured for website hosting, this origin will be configured to use the bucket as an
 * HTTP server origin and will use the bucket's configured website redirects and error handling. Otherwise,
 * the origin is created as a bucket origin and will use CloudFront's redirect and error handling.
 *
 * @stability stable
 */
class S3Origin {
    /**
     * @stability stable
     */
    constructor(bucket, props = {}) {
        this.origin = bucket.isWebsite ?
            new http_origin_1.HttpOrigin(bucket.bucketWebsiteDomainName, {
                protocolPolicy: cloudfront.OriginProtocolPolicy.HTTP_ONLY,
                ...props,
            }) :
            new S3BucketOrigin(bucket, props);
    }
    /**
     * The method called when a given Origin is added (for the first time) to a Distribution.
     *
     * @stability stable
     */
    bind(scope, options) {
        return this.origin.bind(scope, options);
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 */
class S3BucketOrigin extends cloudfront.OriginBase {
    constructor(bucket, { originAccessIdentity, ...props }) {
        super(bucket.bucketRegionalDomainName, props);
        this.bucket = bucket;
        if (originAccessIdentity) {
            this.originAccessIdentity = originAccessIdentity;
        }
    }
    bind(scope, options) {
        if (!this.originAccessIdentity) {
            // Using a bucket from another stack creates a cyclic reference with
            // the bucket taking a dependency on the generated S3CanonicalUserId when `grantRead` is called,
            // and the distribution having a dependency on the bucket's domain name.
            // Fix this by parenting the OAI in the bucket's stack when cross-stack usage is detected.
            const bucketStack = cdk.Stack.of(this.bucket);
            const bucketInDifferentStack = bucketStack !== cdk.Stack.of(scope);
            const oaiScope = bucketInDifferentStack ? bucketStack : scope;
            const oaiId = bucketInDifferentStack ? `${cdk.Names.uniqueId(scope)}S3Origin` : 'S3Origin';
            this.originAccessIdentity = new cloudfront.OriginAccessIdentity(oaiScope, oaiId, {
                comment: `Identity for ${options.originId}`,
            });
            this.bucket.grantRead(this.originAccessIdentity);
        }
        return super.bind(scope, options);
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
//# sourceMappingURL=data:application/json;base64,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