"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DmarcReporter = exports.DmarcAlignment = exports.DmarcPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const route53 = require("aws-cdk-lib/aws-route53");
const constructs_1 = require("constructs");
const receiver_1 = require("../email-receiver/receiver");
/**
 * The DMARC policy to apply to messages that fail DMARC compliance.
 */
var DmarcPolicy;
(function (DmarcPolicy) {
    /**
     * Do not apply any special handling to messages that fail DMARC compliance.
     */
    DmarcPolicy["NONE"] = "none";
    /**
     * Quarantine messages that fail DMARC compliance. (usually by sending them to spam)
     */
    DmarcPolicy["QUARANTINE"] = "quarantine";
    /**
     * Reject messages that fail DMARC compliance. (usually by rejecting them outright)
     */
    DmarcPolicy["REJECT"] = "reject";
})(DmarcPolicy || (exports.DmarcPolicy = DmarcPolicy = {}));
/**
 * The DMARC alignment mode.
 */
var DmarcAlignment;
(function (DmarcAlignment) {
    /**
     * Relaxed alignment mode.
     */
    DmarcAlignment["RELAXED"] = "relaxed";
    /**
     * Strict alignment mode.
     */
    DmarcAlignment["STRICT"] = "strict";
})(DmarcAlignment || (exports.DmarcAlignment = DmarcAlignment = {}));
/**
 * Creates a DMARC record in Route 53 and invokes a Lambda function to process incoming reports.
 */
class DmarcReporter extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const emailAddress = props.emailAddress ?? `dmarc-reports@${props.hostedZone.zoneName}`;
        new receiver_1.EmailReceiver(this, 'EmailReceiver', {
            recipients: [emailAddress],
            function: props.function,
            afterRule: props.afterRule,
            receiptRuleSet: props.receiptRuleSet,
        });
        const dmarcRecordValue = [
            'v=DMARC1',
            `p=${props.dmarcPolicy}`,
            `rua=${[emailAddress, ...(props.additionalEmailAddresses ?? [])]
                .map((address) => `mailto:${address}`)
                .join(',')}`,
        ];
        if (props.dmarcSubdomainPolicy) {
            dmarcRecordValue.push(`sp=${props.dmarcSubdomainPolicy}`);
        }
        if (props.dmarcPercentage) {
            dmarcRecordValue.push(`pct=${props.dmarcPercentage}`);
        }
        if (props.dmarcDkimAlignment) {
            dmarcRecordValue.push(`adkim=${props.dmarcDkimAlignment === DmarcAlignment.RELAXED ? 'r' : 's'}`);
        }
        if (props.dmarcSpfAlignment) {
            dmarcRecordValue.push(`aspf=${props.dmarcSpfAlignment === DmarcAlignment.RELAXED ? 'r' : 's'}`);
        }
        // Create Route 53 DMARC Record
        new route53.TxtRecord(this, 'DmarcRecord', {
            zone: props.hostedZone,
            recordName: `_dmarc.${props.hostedZone.zoneName}`,
            values: [dmarcRecordValue.join('; ')],
        });
    }
}
exports.DmarcReporter = DmarcReporter;
_a = JSII_RTTI_SYMBOL_1;
DmarcReporter[_a] = { fqn: "cloudstructs.DmarcReporter", version: "0.9.14" };
//# sourceMappingURL=data:application/json;base64,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