"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Skill = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: MIT-0
 */
const path = require("path");
const cdk = require("aws-cdk-lib");
const ask = require("aws-cdk-lib/alexa-ask");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const assets = require("aws-cdk-lib/aws-s3-assets");
const customResources = require("aws-cdk-lib/custom-resources");
const ALEXA_SERVICE_PRINCIPAL = 'alexa-appkit.amazon.com';
const BACKEND_LAMBDA_PERMISSION_ACTION = 'lambda:InvokeFunction';
;
class SkillBase extends cdk.Resource {
}
;
;
/**
 * Defines an Alexa Skill.
 *
 * @resource Alexa::ASK::Skill
 */
class Skill extends SkillBase {
    constructor(scope, id, props) {
        var _b;
        // Validate that SSM SecureString was not supplied--Alexa::ASK::Skill does not support SSM SecureString references.
        const resolvedClientSecret = cdk.Tokenization.resolve(props.lwaClientSecret, {
            scope,
            resolver: new cdk.DefaultTokenResolver(new cdk.StringConcat()),
        });
        const resolvedRefreshToken = cdk.Tokenization.resolve(props.lwaRefreshToken, {
            scope,
            resolver: new cdk.DefaultTokenResolver(new cdk.StringConcat()),
        });
        if (resolvedClientSecret.includes('ssm-secure')) {
            throw new Error('Invalid prop: lwaClientSecret; SSM SecureString is not supported. Use Secrets Manager secret instead.');
        }
        if (resolvedRefreshToken.includes('ssm-secure')) {
            throw new Error('Invalid prop: lwaRefreshToken; SSM SecureString is not supported. Use Secrets Manager secret instead.');
        }
        super(scope, id);
        // Role giving CfnSkill resource read-only access to skill package asset in S3.
        const askResourceRole = new iam.Role(this, 'AskResourceRole', {
            assumedBy: new iam.ServicePrincipal(ALEXA_SERVICE_PRINCIPAL),
        });
        // Skill package S3 asset.
        const skillPackageAsset = new assets.Asset(this, 'SkillPackageAsset', {
            path: props.skillPackagePath,
            readers: [askResourceRole],
        });
        // Alexa Skill with override that injects the endpoint Lambda Function in the skill manifest.
        const resource = new ask.CfnSkill(this, 'Resource', {
            vendorId: props.alexaVendorId,
            skillPackage: {
                s3Bucket: skillPackageAsset.s3BucketName,
                s3Key: skillPackageAsset.s3ObjectKey,
                s3BucketRole: askResourceRole.roleArn,
                ...props.endpointLambdaFunction && {
                    overrides: {
                        manifest: {
                            apis: {
                                custom: {
                                    endpoint: {
                                        uri: (_b = props.endpointLambdaFunction) === null || _b === void 0 ? void 0 : _b.functionArn,
                                    },
                                },
                            },
                        },
                    },
                },
            },
            authenticationConfiguration: {
                clientId: props.lwaClientId,
                clientSecret: props.lwaClientSecret.toString(),
                refreshToken: props.lwaRefreshToken.toString(),
            },
        });
        // Set resource skillId to Alexa Skill resource Skill ID.
        this.skillId = resource.ref;
        // This section is only necessary if a Lambda Function was supplied in the props.
        if (props.endpointLambdaFunction) {
            // Create placeholder Lambda Permission to allow Alexa Skill to pass endpoint validation.
            // Permission will be replaced with another containing event source validation after Alexa Skill is created.
            const initialLambdaPermission = new lambda.CfnPermission(this, 'InitialLambdaPermission', {
                functionName: props.endpointLambdaFunction.functionArn,
                principal: ALEXA_SERVICE_PRINCIPAL,
                action: BACKEND_LAMBDA_PERMISSION_ACTION,
            });
            // Skill must be created after the initial Lambda Permission resource is in place to prevent endpoint validation errors.
            resource.addDependsOn(initialLambdaPermission);
            // Lambda Function that retrieves the StatementId of the initial Lambda Permission for use by other custom resources.
            const getPermissionStatementIdFunction = new lambda.Function(this, 'GetLambdaPermissionStatementIdFunction', {
                runtime: lambda.Runtime.PYTHON_3_8,
                handler: 'index.lambda_handler',
                code: lambda.Code.fromAsset(path.join(__dirname, '../custom-resource-runtime/get-lambda-permission-statement-id-handler')),
                initialPolicy: [
                    new iam.PolicyStatement({
                        actions: ['lambda:GetPolicy'],
                        resources: [props.endpointLambdaFunction.functionArn],
                    }),
                ],
            });
            // Custom resource for managing lifecycle of GetLambdaPermissionStatementIdFunction Lambda Function.
            const getPermissionStatementIdCustomResource = new cdk.CustomResource(this, 'GetLambdaPermissionStatementIdCustomResource', {
                serviceToken: new customResources.Provider(this, 'Provider', { onEventHandler: getPermissionStatementIdFunction }).serviceToken,
                properties: {
                    lambda_function_arn: props.endpointLambdaFunction.functionArn,
                    service_principal_to_match: ALEXA_SERVICE_PRINCIPAL,
                    action_to_match: BACKEND_LAMBDA_PERMISSION_ACTION,
                },
            });
            // Custom resource code must run after the initial Lambda Permission resource is in place.
            getPermissionStatementIdCustomResource.node.addDependency(initialLambdaPermission);
            // Get custom resource result for use by other custom resources.
            const permissionStatementId = getPermissionStatementIdCustomResource.getAttString('statement_id');
            // Policy for AwsCustomResource resources.
            const awsCustomResourcePolicy = customResources.AwsCustomResourcePolicy.fromStatements([
                new iam.PolicyStatement({
                    actions: [
                        'lambda:RemovePermission',
                        'lambda:AddPermission',
                    ],
                    resources: [props.endpointLambdaFunction.functionArn],
                }),
            ]);
            // SDK call to be used for RemovePermissionCustomResource.
            const removePermissionStatementSdkCall = {
                service: 'Lambda',
                action: 'removePermission',
                parameters: {
                    FunctionName: props.endpointLambdaFunction.functionArn,
                    StatementId: permissionStatementId,
                },
                ignoreErrorCodesMatching: 'ResourceNotFoundException',
                physicalResourceId: customResources.PhysicalResourceId.of(`RemovePermission-${this.skillId}`),
            };
            const removePermissionCustomResource = new customResources.AwsCustomResource(this, 'RemovePermissionCustomResource', {
                policy: awsCustomResourcePolicy,
                onCreate: removePermissionStatementSdkCall,
                onUpdate: removePermissionStatementSdkCall,
                onDelete: removePermissionStatementSdkCall,
            });
            // RemovePermissionCustomResource code must run after the Alexa Skill has been created to ensure the intial Lambda Permission is in place upon Alexa Skill creation.
            removePermissionCustomResource.node.addDependency(resource);
            // SDK call to be used for AddPermissionCustomResource.
            const addPermissionStatementSdkCall = {
                service: 'Lambda',
                action: 'addPermission',
                parameters: {
                    FunctionName: props.endpointLambdaFunction.functionArn,
                    StatementId: permissionStatementId,
                    Principal: ALEXA_SERVICE_PRINCIPAL,
                    Action: BACKEND_LAMBDA_PERMISSION_ACTION,
                    EventSourceToken: this.skillId,
                },
                physicalResourceId: customResources.PhysicalResourceId.of(`AddPermission-${this.skillId}`),
            };
            const addPermissionCustomResource = new customResources.AwsCustomResource(this, 'AddPermissionCustomResource', {
                policy: awsCustomResourcePolicy,
                onCreate: addPermissionStatementSdkCall,
                onUpdate: addPermissionStatementSdkCall,
            });
            // AddPermissionCustomResource code must run after RemovePermissionCustomResource code has run to prevent attempts to create Permission with redundant StatementIds.
            addPermissionCustomResource.node.addDependency(removePermissionCustomResource);
        }
    }
    /**
     * Reference an existing Skill,
     * defined outside of the CDK code, by Skill ID.
     */
    static fromSkillId(scope, id, skillId) {
        class Import extends SkillBase {
            constructor() {
                super(...arguments);
                this.skillId = skillId;
            }
        }
        return new Import(scope, id);
    }
}
exports.Skill = Skill;
_a = JSII_RTTI_SYMBOL_1;
Skill[_a] = { fqn: "cdk-alexa-skill.Skill", version: "2.0.2" };
;
//# sourceMappingURL=data:application/json;base64,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