# SomeTools

一些常用工具

[![MIT licensed](https://img.shields.io/badge/license-MIT-blue.svg?style=plastic)](./LICENSE)
[![pypi](https://img.shields.io/pypi/v/SomeTools.svg?style=plastic)](https://pypi.org/project/SomeTools/)
[![wheel](https://img.shields.io/pypi/wheel/SomeTools.svg?style=plastic)](https://pypi.org/project/SomeTools/)
[![pyversions](https://img.shields.io/pypi/pyversions/SomeTools.svg?style=plastic)](https://pypi.org/project/SomeTools/)
[![Downloads](https://pepy.tech/badge/SomeTools?style=plastic)](https://pepy.tech/badge/SomeTools)

---------------------------------------------

一些方便日常使用的工具

* 将输入的任何类型的日期字符串类型转化为datetime.datetime类型的日期对象
* 移除一个字符串中的回车换行空格制表符等内容
* 将中文字符转为其拼音的首字母
* 更加方便的日志记录工具
* 繁体简体中文汉字转换工具
* redis创建连接同步、异步工具
* 正则表达式提取字符串中的中文内容工具


## Installation

``` bash
$ pip install SomeTools -i https://pypi.python.org/simple
```


## Getting Started

### 1.同步工具使用

``` pycon
from sometools import Common_tools


class Demo(Common_tools):
    def __init__(self, *args, **kwargs):
        # kwargs['log_file_rec'] = True
        # kwargs['log_file_name'] = 'ConsumptionService'
        super(Demo, self).__init__(*args, **kwargs)


if __name__ == '__main__':
    demo_ins = Demo()

    # 将输入的任何类型的日期字符串类型转化为datetime.datetime类型的日期对象(北京时间UTC+8)(Converts any type of date string type entered to a date object of type datetime.datetime)(beijing time UTC+8)
    demo_ins.logger(uuid1='uuid1', uuid2='uuid2').info(
        f"{demo_ins.str_to_obj('2012-12-12 12:12:12')}{type(demo_ins.str_to_obj('2012-12-12 12:12:12'))}")
    demo_ins.logger().info(
        f"{demo_ins.str_to_obj('11-May-2021 07:03 EDT')}{type(demo_ins.str_to_obj('11-May-2021 07:03 EDT'))}")

    # 移除一个字符串中的回车换行空格制表符等内容(Remove carriage return, newline space tabs, etc., from a string)
    temp_str = 'abc abc \n abc \r'
    demo_ins.logger().info(f"before clean string{temp_str}")
    temp_str = Demo.clean_string('abc abc \n abc \r')
    demo_ins.logger().info(f"after clean string{temp_str}")

    # 将中文字符转为其拼音的首字母(Convert Chinese characters to the first letter of their pinyin)
    demo_ins.logger().info(f"{Demo.get_pinyin('中国外汇交易中心')}")

    # 更加方便的日志记录工具(More convenient logging tool)
    demo_ins.logger().info(f"{Demo.get_pinyin('中国外汇交易中心')}")
    demo_ins.logger().debug(f"{Demo.get_pinyin('中国外汇交易中心')}")
    demo_ins.logger().warning(f"{Demo.get_pinyin('中国外汇交易中心')}")
    demo_ins.logger().error(f"{Demo.get_pinyin('中国外汇交易中心')}")

    # 中文繁体简体互转
    demo_ins.logger().info(f"繁体转简体 眾議長與李克強會談->{Demo.traditional_chinese_to_simplified('眾議長與李克強會談')}")
    demo_ins.logger().info(f"简体转繁体 众议长与李克强会谈->{Demo.simplified_chinese_to_traditional_chinese('众议长与李克强会谈')}")

    # url编码解码
    demo_ins.logger().info(f"url编码->{Demo.url_encode('https://www.baidu.com/s?wd=中国')}")
    demo_ins.logger().info(f"url解码->{Demo.url_decode('https://www.baidu.com/s?wd=%E4%B8%AD%E5%9B%BD')}")

    # redis使用
    redis_conn = demo_ins.get_sync_redis_conn(redis_host='10.1.90.29',
                                              redis_port='6379',
                                              redis_db=15,
                                              redis_pwd='lhdna2016'
                                              )
    msg = redis_conn.set('temp_key1', 'test string1')
    demo_ins.logger().info(f"redis set {msg}")
    msg = redis_conn.get('temp_key1')
    demo_ins.logger().info(f"redis get {msg}")

    # 正则提取中文
    temp_str = demo_ins.extract_one_chinese("""downloadPdf1('http://www.sse.com.cn/disclosure/bond/announcement/company/c/2021-03-22/4135530025747110334559080.pdf','厦门建发股份有限公司2021年面向专业投资者公开发行可续期公司债券（第一期）发行公告','2021-03-22','1015','pdf');""")
    demo_ins.logger().info(f"正则提取单个中文 {temp_str}")
    temp_str = demo_ins.extract_multi_chinese("""downloadPdf1('http://www.sse.com.cn/disclosure/bond/announcement/company/c/2021-03-22/4135530025747110334559080.pdf','厦门建发股份有限公司2021年面向专业投资者公开发行可续期公司债券（第一期）发行公告','2021-03-22','1015','pdf');""")
    demo_ins.logger().info(f"正则提取多个中文 {temp_str}")

    # 编码处理
    _temp_content = b'<!DOCTYPE HTML>\n<html>\n\n<head><meta name="templateId" content="d2b039de7f564c3882858a6a1655f8c1" >\n    <meta http-equiv="Content-Type" content="text/html; charset=utf-8" >\n    <meta name="apple-mobile-web-app-capable" content="yes" >\n    <meta name="apple-mobile-web-app-status-bar-style" content="black" >\n    <meta content="telephone=no" name="format-detection" >\n    <meta http-equiv="X-UA-Compatible" content="IE=edge" >\n    <meta content="width=device-width, initial-scale=1.0, minimum-scale=1.0, maximum-scale=1.0,user-scalable=no" name="viewport" >\n    <meta name="keywords" content="\xe7\xbb\xbf\xe8\x89\xb2,\xe4\xbd\x8e\xe7\xa2\xb3,\xe9\x93\xb6\xe8\x81\x94,\xe4\xba\xa7\xe5\x93\x81,\xe4\xbc\x81\xe4\xb8\x9a,\xe5\x8f\x91\xe5\xb1\x95,\xe7\x9b\xae\xe6\xa0\x87,\xe4\xb8\xad\xe5\x9b\xbd,\xe9\x87\x91\xe8\x9e\x8d,\xe5\xae\x9e\xe7\x8e\xb0">\n    <meta name="description" content="\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe9\x93\xb6\xe8\xa1\x8c\xe5\x90\x88\xe4\xbd\x9c\xe9\x83\xa8\xe6\x80\xbb\xe7\xbb\x8f\xe7\x90\x86\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a\xe8\xb7\xb5\xe8\xa1\x8c\xe2\x80\x9c\xe6\x94\xaf\xe4\xbb\x98\xe4\xb8\xba\xe6\xb0\x91\xe2\x80\x9d\xe7\x90\x86\xe5\xbf\xb5 \xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\xb0\x86\xe6\x8e\xa2\xe7\xb4\xa2\xe6\x9b\xb4\xe5\xa4\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xba\xa7\xe5\x93\x81\xe5\x92\x8c\xe6\x9c\x8d\xe5\x8a\xa1-\xe6\x8d\xae\xe6\x82\x89\xef\xbc\x8c\xe8\xbf\x91\xe6\x97\xa5\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe8\x81\x94\xe5\x90\x88\xe4\xb8\x8a\xe6\xb5\xb7\xe7\x8e\xaf\xe5\xa2\x83\xe8\x83\xbd\xe6\xba\x90\xe4\xba\xa4\xe6\x98\x93\xe6\x89\x80\xe3\x80\x81\xe5\x95\x86\xe4\xb8\x9a\xe9\x93\xb6\xe8\xa1\x8c\xef\xbc\x8c\xe5\x85\xb1\xe5\x90\x8c\xe5\x8f\x91\xe5\xb8\x83\xe9\x93\xb6\xe8\x81\x94\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe5\x8d\xa1\xe4\xba\xa7\xe5\x93\x81\xe3\x80\x82">\n    <title data-server-rendered="true">\n        \xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe9\x93\xb6\xe8\xa1\x8c\xe5\x90\x88\xe4\xbd\x9c\xe9\x83\xa8\xe6\x80\xbb\xe7\xbb\x8f\xe7\x90\x86\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a\xe8\xb7\xb5\xe8\xa1\x8c\xe2\x80\x9c\xe6\x94\xaf\xe4\xbb\x98\xe4\xb8\xba\xe6\xb0\x91\xe2\x80\x9d\xe7\x90\x86\xe5\xbf\xb5 \xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\xb0\x86\xe6\x8e\xa2\xe7\xb4\xa2\xe6\x9b\xb4\xe5\xa4\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xba\xa7\xe5\x93\x81\xe5\x92\x8c\xe6\x9c\x8d\xe5\x8a\xa1-\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\n        </title>\n    <link rel="stylesheet" href="http://lib.news.cn/common/reset.css" >\n    <link rel="stylesheet" href="http://lib.news.cn/swiper/swiper3.4.2/swiper.min.css" >\n    <link rel="stylesheet" href="http://www.xinhuanet.com/detail/detail_2021new.css" >\n    <script src="http://res.wx.qq.com/open/js/jweixin-1.6.0.js"></script>\n    <script src="http://lib.news.cn/common/share.js"></script>\n    <script src="http://lib.news.cn/jquery/jquery1.12.4/jquery.min.js"></script>\n\t\n\t\n<meta name="pagetype" content="1"><meta name="filetype" content="0"><meta name="publishedtype" content="1"><meta name="author" content=""><meta name="catalogs" content=""><meta name="subject" content=""><meta name="contentid" content="202109080e3a78a113e34a95b4962a28e0f21844"><meta name="publishdate" content="2021-09-08"><meta name="source" content="\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91"><script src="//news.cn/webdig/xinhua_webdig.js" language="javascript" type="text/javascript" async></script></head>\n<body>\n    <div class="fix-ewm domPC" dataType="content" preview="ds_"><img src="http://www.news.cn/detail2020/images/ewm.png" width="94" alt> <div class="fxd"><span class="fxd-wx"></span> <a href="javascript:void(0)" class="fxd-wb"></a> <span class="fxd-xcx"></span> <span class="fxd-khd"></span> <div class="fxd-wx-ewm"><img src="/money/20210908/0e3a78a113e34a95b4962a28e0f21844/zxcode_202109080e3a78a113e34a95b4962a28e0f21844.jpg"></div> <div class="fxd-xcx-ewm"><img></div> <div class="fxd-khd-ewm"><img src="http://www.news.cn/politics/newpage2020/images/qrcode-app.png"></div></div></div>\n\t<div class="domPC">\n    \t<script src="http://www.news.cn/2021homepro/scripts/smallTop.js"></script>\n\t</div>\n    <div class="domMobile">\n        <script src="http://www.news.cn/2021mobile/scripts/toppx.js"></script>\n    </div>\n\t<div class="topAd">\n\t\t<div class="domPC"><ins data-ycad-slot="2166"></ins></div>\n\t\t<div class="domPC"><ins data-ycad-slot="2167"></ins></div>\n\t</div>\n    <div class="header domPC" dataType="content" preview="ds_"><div class="header-top clearfix"><div class="header-nav left"><a href="http://www.news.cn/" target="_blank">\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91</a> &gt; <a class="curColumn"></a> &gt; \xe6\xad\xa3\xe6\x96\x87</div></div> <div class="header-cont clearfix"><div class="header-time left"><span class="year"><em>2021</em></span> <span class="day"><em>09</em>/<em>08</em></span> <span class="time">17:35:14</span></div> <div class="source">\xe6\x9d\xa5\xe6\xba\x90\xef\xbc\x9a\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\n           \t</div> <div class="head-line clearfix"><h1><span class="title">\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe9\x93\xb6\xe8\xa1\x8c\xe5\x90\x88\xe4\xbd\x9c\xe9\x83\xa8\xe6\x80\xbb\xe7\xbb\x8f\xe7\x90\x86\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a\xe8\xb7\xb5\xe8\xa1\x8c\xe2\x80\x9c\xe6\x94\xaf\xe4\xbb\x98\xe4\xb8\xba\xe6\xb0\x91\xe2\x80\x9d\xe7\x90\x86\xe5\xbf\xb5 \xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\xb0\x86\xe6\x8e\xa2\xe7\xb4\xa2\xe6\x9b\xb4\xe5\xa4\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xba\xa7\xe5\x93\x81\xe5\x92\x8c\xe6\x9c\x8d\xe5\x8a\xa1</span> <span class="btn-audio"></span></h1> <audio id="audioDom" loop="loop" src class="hide"></audio> <div class="pageShare"><div class="setFont">\xe5\xad\x97\xe4\xbd\x93\xef\xbc\x9a\n                        <span id="fontSmall">\xe5\xb0\x8f</span> <span id="fontNormal" class="active">\xe4\xb8\xad</span> <span id="fontBig">\xe5\xa4\xa7</span></div> <div class="share">\xe5\x88\x86\xe4\xba\xab\xe5\x88\xb0\xef\xbc\x9a<a href="javascript:void(0)" class="wx"></a><a href="javascript:void(0)" class="wb"></a><a href="javascript:void(0)" class="xcx"></a><a href="javascript:void(0)" class="khd"></a> <div datatype="content"><div class="wx-ewm"><img src="/money/20210908/0e3a78a113e34a95b4962a28e0f21844/zxcode_202109080e3a78a113e34a95b4962a28e0f21844.jpg"></div></div> <div class="xcx-ewm"><img></div> <div class="khd-ewm"><img src="http://www.news.cn/politics/newpage2020/images/qrcode-app.png"></div></div></div></div></div></div>\n    <div class="adv domMob">\n          <div class="advCont" style="display:none"><ins data-ycad-slot="2305"></ins></div>\n          <div class="advShow"></div>\n        </div>\n    <div class="mheader domMobile" dataType="content" preview="ds_"><h1><span class="title">\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe9\x93\xb6\xe8\xa1\x8c\xe5\x90\x88\xe4\xbd\x9c\xe9\x83\xa8\xe6\x80\xbb\xe7\xbb\x8f\xe7\x90\x86\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a\xe8\xb7\xb5\xe8\xa1\x8c\xe2\x80\x9c\xe6\x94\xaf\xe4\xbb\x98\xe4\xb8\xba\xe6\xb0\x91\xe2\x80\x9d\xe7\x90\x86\xe5\xbf\xb5 \xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\xb0\x86\xe6\x8e\xa2\xe7\xb4\xa2\xe6\x9b\xb4\xe5\xa4\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xba\xa7\xe5\x93\x81\xe5\x92\x8c\xe6\x9c\x8d\xe5\x8a\xa1</span></h1> <div class="info">\n\t\t\t2021-09-08 17:35:14\n\t\t\t<span>\n            \xe6\x9d\xa5\xe6\xba\x90\xef\xbc\x9a\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\n\t\t\t</span></div></div>\n    <div class="main clearfix">\n        <div class="main-left left">\n            <div id="detail" dataType="content" preview="ds_"><span id="detailContent"><p id="XVKhXou3WrkVW2Qk" style="text-indent: 2em;">\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\xe5\x8c\x97\xe4\xba\xac9\xe6\x9c\x888\xe6\x97\xa5\xe7\x94\xb5\xef\xbc\x88\xe7\x8e\x8b\xe9\x9b\xa8\xe9\xa6\xa8\xef\xbc\x89\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe9\x93\xb6\xe8\xa1\x8c\xe5\x90\x88\xe4\xbd\x9c\xe9\x83\xa8\xe6\x80\xbb\xe7\xbb\x8f\xe7\x90\x86\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xe6\x97\xa5\xe5\x89\x8d\xe6\x8e\xa5\xe5\x8f\x97\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\xe8\xae\xb0\xe8\x80\x85\xe4\xb8\x93\xe8\xae\xbf\xe6\x97\xb6\xe8\xa1\xa8\xe7\xa4\xba\xef\xbc\x8c\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe4\xb8\x80\xe7\x9b\xb4\xe4\xbb\xa5\xe6\x9d\xa5\xe9\x87\x8d\xe8\xa7\x86\xe6\x95\xb0\xe5\xad\x97\xe5\x8c\x96\xe5\x92\x8c\xe4\xbd\x8e\xe7\xa2\xb3\xe5\x8c\x96\xe5\x8f\x91\xe5\xb1\x95\xef\xbc\x8c\xe6\x9c\xaa\xe6\x9d\xa5\xe5\xb0\x86\xe5\x85\x85\xe5\x88\x86\xe5\x8f\x91\xe6\x8c\xa5\xe6\x94\xaf\xe4\xbb\x98\xe4\xba\xa7\xe5\x93\x81\xe5\xb9\xb3\xe5\x8f\xb0\xe5\x92\x8c\xe6\x9e\xa2\xe7\xba\xbd\xe4\xbd\x9c\xe7\x94\xa8\xef\xbc\x8c\xe6\x8c\x81\xe7\xbb\xad\xe8\x81\x94\xe5\x90\x88\xe5\x95\x86\xe4\xb8\x9a\xe9\x93\xb6\xe8\xa1\x8c\xe7\xad\x89\xe4\xba\xa7\xe4\xb8\x9a\xe5\x90\x84\xe6\x96\xb9\xe9\x99\x86\xe7\xbb\xad\xe6\x8e\xa8\xe5\x87\xba\xe5\x90\x84\xe9\x93\xb6\xe8\xa1\x8c\xe7\x9a\x84\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe5\x8d\xa1\xe4\xba\xa7\xe5\x93\x81\xef\xbc\x8c\xe7\xa0\x94\xe5\x8f\x91\xe6\x9b\xb4\xe5\xa4\x9a\xe6\x99\xae\xe6\x83\xa0\xe4\xbe\xbf\xe6\xb0\x91\xe7\x9a\x84\xe7\xbb\xbf\xe8\x89\xb2\xe9\x87\x91\xe8\x9e\x8d\xe6\x96\xb0\xe4\xba\xa7\xe5\x93\x81\xe3\x80\x81\xe6\x96\xb0\xe4\xb8\x9a\xe6\x80\x81\xe3\x80\x81\xe6\x96\xb0\xe6\xa8\xa1\xe5\xbc\x8f\xef\xbc\x8c\xe5\x8a\xa9\xe5\x8a\x9b\xe2\x80\x9c3060\xe2\x80\x9d\xe5\x8f\x8c\xe7\xa2\xb3\xe7\x9b\xae\xe6\xa0\x87\xe5\xa6\x82\xe6\x9c\x9f\xe5\xae\x9e\xe7\x8e\xb0\xe3\x80\x82</p>\n<p id="BymDsgl6oKZX6z3V" style="text-indent: 2em;">\xe6\x8d\xae\xe6\x82\x89\xef\xbc\x8c\xe8\xbf\x91\xe6\x97\xa5\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe8\x81\x94\xe5\x90\x88\xe4\xb8\x8a\xe6\xb5\xb7\xe7\x8e\xaf\xe5\xa2\x83\xe8\x83\xbd\xe6\xba\x90\xe4\xba\xa4\xe6\x98\x93\xe6\x89\x80\xe3\x80\x81\xe5\x95\x86\xe4\xb8\x9a\xe9\x93\xb6\xe8\xa1\x8c\xef\xbc\x8c\xe5\x85\xb1\xe5\x90\x8c\xe5\x8f\x91\xe5\xb8\x83\xe9\x93\xb6\xe8\x81\x94\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe5\x8d\xa1\xe4\xba\xa7\xe5\x93\x81\xe3\x80\x82\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe9\x93\xb6\xe8\xa1\x8c\xe5\x8d\xa1\xe5\x8c\x85\xe5\x90\xab\xe4\xbc\x81\xe4\xb8\x9a\xe7\x89\x88\xe5\x8f\x8a\xe4\xb8\xaa\xe4\xba\xba\xe7\x89\x88\xe4\xb8\xa4\xe5\xa5\x97\xe5\x8d\xa1\xe4\xba\xa7\xe5\x93\x81\xe4\xbd\x93\xe7\xb3\xbb\xef\xbc\x8c\xe4\xbb\xa5\xe9\x93\xb6\xe8\x81\x94\xe4\xba\xa4\xe6\x98\x93\xe7\xb3\xbb\xe7\xbb\x9f\xe4\xb8\xad\xe4\xbc\x81\xe4\xb8\x9a\xe5\x92\x8c\xe4\xb8\xaa\xe4\xba\xba\xe4\xba\xa4\xe6\x98\x93\xe8\xa1\x8c\xe4\xb8\xba\xe6\x95\xb0\xe6\x8d\xae\xe4\xb8\xba\xe8\xae\xa1\xe7\xae\x97\xe5\x9f\xba\xe7\xa1\x80\xef\xbc\x8c\xe6\x9e\x84\xe5\xbb\xba\xe9\x93\xb6\xe8\x81\x94\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\xa7\xaf\xe5\x88\x86\xe4\xbd\x93\xe7\xb3\xbb\xef\xbc\x8c\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2\xe5\x8a\xa9\xe5\x8a\x9b\xe4\xbc\x81\xe4\xb8\x9a\xe7\xa2\xb3\xe4\xb8\xad\xe5\x92\x8c\xe8\xbd\xac\xe5\x9e\x8b\xef\xbc\x9b\xe5\x8f\xa6\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2\xe6\xa0\xb9\xe6\x8d\xae\xe7\x94\xa8\xe6\x88\xb7\xe7\xbb\xbf\xe8\x89\xb2\xe6\xb6\x88\xe8\xb4\xb9\xe8\xa1\x8c\xe4\xb8\xba\xe8\xb4\xa1\xe7\x8c\xae\xe6\x9e\x84\xe5\xbb\xba\xe9\x93\xb6\xe8\x81\x94\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe5\x8d\xa1\xe5\x8a\xa8\xe6\x80\x81\xe6\x9d\x83\xe7\x9b\x8a\xe4\xbd\x93\xe7\xb3\xbb\xef\xbc\x8c\xe5\x90\x91\xe4\xbc\x81\xe4\xb8\x9a\xe5\x92\x8c\xe7\xa4\xbe\xe4\xbc\x9a\xe5\xa4\xa7\xe4\xbc\x97\xe4\xbc\xa0\xe6\x92\xad\xe7\xa2\xb3\xe4\xb8\xad\xe5\x92\x8c\xe6\xa6\x82\xe5\xbf\xb5\xe3\x80\x82</p>\n<p id="9hQA7PyWzezRjHbr" style="text-indent: 2em;"><strong>\xe8\xb7\xb5\xe8\xa1\x8c\xe2\x80\x9c\xe6\x94\xaf\xe4\xbb\x98\xe4\xb8\xba\xe6\xb0\x91\xe2\x80\x9d \xe6\x8e\xa8\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe9\x93\xb6\xe8\xa1\x8c\xe5\x8d\xa1</strong></p>\n<p id="BFGNLEvCTT4bhdxQ" style="text-indent: 2em;"><strong>\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\xef\xbc\x9a\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe4\xba\x8e\xe8\xbf\x91\xe6\x97\xa5\xe5\x8f\x91\xe5\xb8\x83\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe9\x93\xb6\xe8\xa1\x8c\xe5\x8d\xa1\xe7\x9a\x84\xe5\x88\x9d\xe5\xbf\x83\xe6\x98\xaf\xe4\xbb\x80\xe4\xb9\x88\xef\xbc\x9f\xe6\x83\xb3\xe5\x80\x9f\xe6\xad\xa4\xe4\xbc\xa0\xe6\x92\xad\xe4\xbb\x80\xe4\xb9\x88\xe6\xa0\xb7\xe7\x9a\x84\xe7\x90\x86\xe5\xbf\xb5\xef\xbc\x9f</strong></p>\n<p id="HdPeGFmrUR4JHaQ1" style="text-indent: 2em;"><strong>\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a</strong>\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe4\xb8\x80\xe7\x9b\xb4\xe4\xbb\xa5\xe6\x9d\xa5\xe9\x87\x8d\xe8\xa7\x86\xe6\x95\xb0\xe5\xad\x97\xe5\x8c\x96\xe5\x92\x8c\xe4\xbd\x8e\xe7\xa2\xb3\xe5\x8c\x96\xe5\x8f\x91\xe5\xb1\x95\xef\xbc\x8c\xe8\x87\xaa\xe5\xae\x9e\xe7\x8e\xb0\xe7\xa2\xb3\xe8\xbe\xbe\xe5\xb3\xb0\xe7\xa2\xb3\xe4\xb8\xad\xe5\x92\x8c\xe7\x9b\xae\xe6\xa0\x87\xe6\x8f\x90\xe5\x87\xba\xe4\xbb\xa5\xe6\x9d\xa5\xef\xbc\x8c\xe9\x93\xb6\xe8\x81\x94\xe4\xb8\x80\xe7\x9b\xb4\xe5\x9c\xa8\xe6\x80\x9d\xe8\x80\x83\xe5\xa6\x82\xe4\xbd\x95\xe7\xbb\x93\xe5\x90\x88\xe8\x87\xaa\xe8\xba\xab\xe4\xb8\x9a\xe5\x8a\xa1\xe5\x8f\x91\xe5\xb1\x95\xef\xbc\x8c\xe7\xa7\xaf\xe6\x9e\x81\xe5\x93\x8d\xe5\xba\x94\xe7\xa2\xb3\xe8\xbe\xbe\xe5\xb3\xb0\xe7\xa2\xb3\xe4\xb8\xad\xe5\x92\x8c\xe7\x9b\xae\xe6\xa0\x87\xe4\xbb\xa5\xe5\x8f\x8a\xe7\xbb\xbf\xe8\x89\xb2\xe9\x87\x91\xe8\x9e\x8d\xe5\x8f\x91\xe5\xb1\x95\xe8\xa6\x81\xe6\xb1\x82\xe3\x80\x82\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe4\xbd\x9c\xe4\xb8\xba\xe9\x93\xb6\xe8\xa1\x8c\xe5\x8d\xa1\xe8\xbd\xac\xe6\x8e\xa5\xe6\xb8\x85\xe7\xae\x97\xe6\x9c\xba\xe6\x9e\x84\xef\xbc\x8c\xe4\xb8\x8e\xe5\x95\x86\xe4\xb8\x9a\xe9\x93\xb6\xe8\xa1\x8c\xe3\x80\x81\xe5\x95\x86\xe6\x88\xb7\xe3\x80\x81\xe7\x94\xa8\xe6\x88\xb7\xe6\x9c\x89\xe7\x9d\x80\xe5\xa4\xa9\xe7\x84\xb6\xe7\x9a\x84\xe7\xb4\xa7\xe5\xaf\x86\xe8\x81\x94\xe7\xb3\xbb\xef\xbc\x8c\xe6\x89\x80\xe4\xbb\xa5\xe5\x9c\xa8\xe5\xae\x9e\xe7\x8e\xb0\xe2\x80\x9c\xe5\x8f\x8c\xe7\xa2\xb3\xe2\x80\x9d\xe7\x9b\xae\xe6\xa0\x87\xe4\xbd\x9c\xe4\xb8\xba\xe9\x87\x8d\xe8\xa6\x81\xe6\x88\x98\xe7\x95\xa5\xe5\x86\xb3\xe7\xad\x96\xe8\xa2\xab\xe6\x8f\x90\xe5\x87\xba\xe6\x97\xb6\xef\xbc\x8c\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\xb0\xb1\xe2\x80\x9c\xe6\x94\xaf\xe4\xbb\x98\xe4\xb8\xba\xe6\xb0\x91\xe2\x80\x9d\xe7\x9a\x84\xe7\x90\x86\xe5\xbf\xb5\xef\xbc\x8c\xe5\xb8\x8c\xe6\x9c\x9b\xe6\x8e\xa8\xe5\x87\xba\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\x9b\xb8\xe5\x85\xb3\xe9\x93\xb6\xe8\xa1\x8c\xe5\x8d\xa1\xe4\xba\xa7\xe5\x93\x81\xef\xbc\x8c\xe4\xbb\xa5\xe9\x93\xb6\xe8\xa1\x8c\xe5\x8d\xa1\xe4\xb8\xba\xe7\xba\xbd\xe5\xb8\xa6\xef\xbc\x8c\xe5\x90\x91\xe4\xbc\x81\xe4\xb8\x9a\xe5\x92\x8c\xe4\xb8\xaa\xe4\xba\xba\xe6\x8e\xa8\xe5\xb9\xbf\xe6\x99\xae\xe5\x8f\x8a\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\x90\x86\xe5\xbf\xb5\xef\xbc\x8c\xe5\xbc\x95\xe5\xaf\xbc\xe4\xbc\x81\xe4\xb8\x9a\xe5\x92\x8c\xe4\xb8\xaa\xe4\xba\xba\xe8\x8a\x82\xe8\x83\xbd\xe5\x87\x8f\xe6\x8e\x92\xef\xbc\x8c\xe5\x8a\xa9\xe5\x8a\x9b\xe7\xa2\xb3\xe8\xbe\xbe\xe5\xb3\xb0\xe7\xa2\xb3\xe4\xb8\xad\xe5\x92\x8c\xe7\x9b\xae\xe6\xa0\x87\xe5\xa6\x82\xe6\x9c\x9f\xe5\xae\x9e\xe7\x8e\xb0\xe3\x80\x82</p>\n<p id="lTspzdysVGbzuYwX" style="text-indent: 2em;"><strong>\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\xef\xbc\x9a\xe9\x93\xb6\xe8\x81\x94\xe6\xad\xa4\xe6\xac\xa1\xe5\x8f\x91\xe8\xa1\x8c\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe9\x93\xb6\xe8\xa1\x8c\xe5\x8d\xa1\xe7\x9a\x84\xe5\x8f\x97\xe4\xbc\x97\xe6\x9c\x89\xe5\x93\xaa\xe4\xba\x9b\xef\xbc\x9f\xe6\x94\xaf\xe6\x8c\x81\xe5\x93\xaa\xe4\xba\x9b\xe7\xbb\xbf\xe8\x89\xb2\xe6\xb6\x88\xe8\xb4\xb9\xe8\xa1\x8c\xe4\xb8\xba\xef\xbc\x9f</strong></p>\n<p id="xJDSoYTiB5CPP2by" style="text-indent: 2em;"><strong>\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a</strong>\xe2\x80\x9c\xe5\x8f\x8c\xe7\xa2\xb3\xe2\x80\x9d\xe7\x9b\xae\xe6\xa0\x87\xe7\x9a\x84\xe5\xae\x9e\xe7\x8e\xb0\xef\xbc\x8c\xe9\x9c\x80\xe8\xa6\x81\xe4\xbc\x81\xe4\xb8\x9a\xe5\x92\x8c\xe4\xb8\xaa\xe4\xba\xba\xe7\x9a\x84\xe5\x85\xb1\xe5\x90\x8c\xe5\x8a\xaa\xe5\x8a\x9b\xe3\x80\x82\xe5\x9c\xa8\xe5\x89\x8d\xe6\x9c\x9f\xe7\x9a\x84\xe5\xb8\x82\xe5\x9c\xba\xe8\xb0\x83\xe7\xa0\x94\xe8\xbf\x87\xe7\xa8\x8b\xe4\xb8\xad\xef\xbc\x8c\xe6\x88\x91\xe4\xbb\xac\xe4\xba\x86\xe8\xa7\xa3\xe5\x88\xb0\xe4\xbc\x81\xe4\xb8\x9a\xe5\x92\x8c\xe4\xb8\xaa\xe4\xba\xba\xe9\x83\xbd\xe5\xad\x98\xe5\x9c\xa8\xe7\x9b\xb8\xe5\xba\x94\xe7\x9a\x84\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\x94\x9f\xe4\xba\xa7\xe5\x92\x8c\xe7\x94\x9f\xe6\xb4\xbb\xe9\x9c\x80\xe6\xb1\x82\xef\xbc\x8c\xe6\x89\x80\xe4\xbb\xa5\xe9\x93\xb6\xe8\x81\x94\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe5\x8d\xa1\xe5\x9c\xa8\xe4\xba\xa7\xe5\x93\x81\xe7\xa0\x94\xe5\x8f\x91\xe9\x98\xb6\xe6\xae\xb5\xe5\xb0\xb1\xe8\xae\xbe\xe8\xae\xa1\xe4\xba\x86\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe5\xaf\xb9\xe5\x85\xac\xe4\xba\xa7\xe5\x93\x81\xe5\x92\x8c\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xaa\xe4\xba\xba\xe4\xba\xa7\xe5\x93\x81\xef\xbc\x8c\xe5\xb8\x8c\xe6\x9c\x9b\xe6\x8e\xa8\xe5\x8a\xa8\xe6\x9b\xb4\xe5\xa4\x9a\xe7\x9a\x84\xe4\xbc\x81\xe4\xb8\x9a\xe5\x92\x8c\xe4\xb8\xaa\xe4\xba\xba\xe8\x83\xbd\xe5\xa4\x9f\xe5\x8f\x82\xe4\xb8\x8e\xe5\x88\xb0\xe6\x88\x91\xe4\xbb\xac\xe7\x9a\x84\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe8\xa1\x8c\xe5\x8a\xa8\xe4\xb8\xad\xe3\x80\x82\xe5\x85\xb6\xe4\xb8\xad\xe5\xaf\xb9\xe5\x85\xac\xe4\xba\xa7\xe5\x93\x81\xe9\x9d\xa2\xe5\x90\x91\xe5\xa4\xa7\xe4\xb8\xad\xe5\x9e\x8b\xe4\xbc\x81\xe4\xb8\x9a\xe3\x80\x81\xe5\xb0\x8f\xe5\xbe\xae\xe4\xbc\x81\xe4\xb8\x9a\xe3\x80\x81\xe4\xb9\xa1\xe9\x95\x87\xe4\xbc\x81\xe4\xb8\x9a\xe7\xad\x89\xe5\x90\x84\xe7\xb1\xbb\xe5\x9e\x8b\xe5\x85\xb7\xe6\x9c\x89\xe7\xbb\xbf\xe8\x89\xb2\xe5\x8f\x91\xe5\xb1\x95\xe9\x9c\x80\xe6\xb1\x82\xe7\x9a\x84\xe4\xbc\x81\xe4\xb8\x9a\xef\xbc\x8c\xe4\xb8\xaa\xe4\xba\xba\xe4\xba\xa7\xe5\x93\x81\xe9\x9d\xa2\xe5\x90\x91\xe4\xbb\xa5\xe5\xb9\xb4\xe8\xbd\xbb\xe5\xae\xa2\xe7\xbe\xa4\xe3\x80\x81\xe6\x96\xb0\xe8\x83\xbd\xe6\xba\x90\xe6\xb1\xbd\xe8\xbd\xa6\xe8\xbd\xa6\xe4\xb8\xbb\xe7\xad\x89\xe4\xb8\xba\xe4\xbb\xa3\xe8\xa1\xa8\xe7\x9a\x84\xe5\xaf\xb9\xe7\x8e\xaf\xe4\xbf\x9d\xe7\x90\x86\xe5\xbf\xb5\xe8\xae\xa4\xe7\x9f\xa5\xe5\xba\xa6\xe8\xbe\x83\xe9\xab\x98\xe7\x9a\x84\xe5\xa4\xa7\xe4\xbc\x97\xe5\xae\xa2\xe7\xbe\xa4\xe3\x80\x82</p>\n<p id="zG3J2c6GV74VObud" style="text-indent: 2em;">\xe7\x8e\xb0\xe9\x98\xb6\xe6\xae\xb5\xef\xbc\x8c\xe6\x88\x91\xe4\xbb\xac\xe8\x81\x94\xe5\x90\x88\xe5\x95\x86\xe4\xb8\x9a\xe9\x93\xb6\xe8\xa1\x8c\xef\xbc\x8c\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2\xef\xbc\x8c\xe9\x80\x9a\xe8\xbf\x87\xe5\x8f\x91\xe8\xa1\x8c\xe8\x99\x9a\xe6\x8b\x9f\xe5\x8d\xa1\xe6\x88\x96\xe7\x8e\xaf\xe4\xbf\x9d\xe6\x9d\x90\xe8\xb4\xa8\xe7\x9a\x84\xe5\x8d\xa1\xe7\x89\x87\xe3\x80\x81\xe5\x90\x91\xe6\x8c\x81\xe5\x8d\xa1\xe4\xba\xba\xe6\x8f\x90\xe4\xbe\x9b\xe7\x94\xb5\xe5\xad\x90\xe8\xb4\xa6\xe5\x8d\x95\xe7\xad\x89\xe6\x96\xb9\xe5\xbc\x8f\xef\xbc\x8c\xe4\xbb\x8e\xe4\xba\xa7\xe5\x93\x81\xe6\x9c\xac\xe8\xba\xab\xe5\xb0\xbd\xe5\x8f\xaf\xe8\x83\xbd\xe5\x87\x8f\xe5\xb0\x91\xe7\xa2\xb3\xe6\x8e\x92\xe6\x94\xbe\xe3\x80\x82\xe5\x8f\xa6\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2\xef\xbc\x8c\xe4\xbe\x9d\xe6\x89\x98\xe9\x93\xb6\xe8\x81\x94\xe4\xba\xa4\xe6\x98\x93\xe7\xb3\xbb\xe7\xbb\x9f\xe6\x95\xb0\xe6\x8d\xae\xef\xbc\x8c\xe6\x88\x91\xe4\xbb\xac\xe6\x9e\x84\xe5\xbb\xba\xe4\xba\x86\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\xa7\xaf\xe5\x88\x86\xe4\xbd\x93\xe7\xb3\xbb\xef\xbc\x8c\xe6\x94\xaf\xe6\x8c\x81\xe4\xb8\xaa\xe4\xba\xba\xe5\x9c\xa8\xe5\x85\xac\xe4\xba\xa4\xe5\x87\xba\xe8\xa1\x8c\xe3\x80\x81\xe5\x9c\xb0\xe9\x93\x81\xe5\x87\xba\xe8\xa1\x8c\xe3\x80\x81\xe5\x85\xb1\xe4\xba\xab\xe5\x8d\x95\xe8\xbd\xa6\xe9\xaa\x91\xe8\xa1\x8c\xe3\x80\x81\xe4\xba\x91\xe9\x97\xaa\xe4\xbb\x98\xe5\x85\xac\xe7\xbc\xb4\xe3\x80\x81\xe6\x96\xb0\xe8\x83\xbd\xe6\xba\x90\xe8\xbd\xa6\xe5\x85\x85\xe7\x94\xb5\xe3\x80\x81\xe9\xab\x98\xe9\x93\x81\xe5\x8a\xa8\xe8\xbd\xa6\xe5\x87\xba\xe8\xa1\x8c\xe7\xad\x89\xe5\x85\xad\xe5\xa4\xa7\xe4\xbd\x8e\xe7\xa2\xb3\xe5\x9c\xba\xe6\x99\xaf\xe9\x80\x9a\xe8\xbf\x87\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe5\x8d\xa1\xe6\xb6\x88\xe8\xb4\xb9\xe7\xb4\xaf\xe8\xae\xa1\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\xa7\xaf\xe5\x88\x86\xe3\x80\x82\xe5\x90\x8e\xe7\xbb\xad\xef\xbc\x8c\xe6\x88\x91\xe4\xbb\xac\xe5\xb0\x86\xe8\x81\x94\xe5\x90\x88\xe5\x95\x86\xe4\xb8\x9a\xe9\x93\xb6\xe8\xa1\x8c\xef\xbc\x8c\xe7\xba\xb3\xe5\x85\xa5\xe6\x9b\xb4\xe5\xa4\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe6\xb6\x88\xe8\xb4\xb9\xe5\x9c\xba\xe6\x99\xaf\xef\xbc\x8c\xe4\xbb\x8e\xe6\x8c\x81\xe5\x8d\xa1\xe4\xba\xba\xe7\x9a\x84\xe8\xa1\xa3\xe9\xa3\x9f\xe4\xbd\x8f\xe8\xa1\x8c\xe7\x94\xa8\xe5\x90\x84\xe6\x96\xb9\xe9\x9d\xa2\xe5\x80\xa1\xe5\xaf\xbc\xe5\x92\x8c\xe9\xbc\x93\xe5\x8a\xb1\xe7\xbb\xbf\xe8\x89\xb2\xe6\xb6\x88\xe8\xb4\xb9\xe8\xa1\x8c\xe4\xb8\xba\xe3\x80\x82</p>\n<p id="azrImmlxNwBxPNn4" style="text-indent: 2em;"><strong>\xe2\x80\x9c\xe8\x87\xaa\xe8\xba\xab+\xe7\x94\x9f\xe6\x80\x81\xe5\x9c\x88\xe2\x80\x9d\xe5\x8f\x8c\xe5\x8a\xa0\xe7\xa0\x81\xef\xbc\x8c\xe5\x85\xb1\xe4\xbf\x83\xe7\xa2\xb3\xe5\x87\x8f\xe6\x8e\x92 \xe5\x8a\xa9\xe5\x8a\x9b\xe2\x80\x9c\xe5\x8f\x8c\xe7\xa2\xb3\xe2\x80\x9d\xe7\x9b\xae\xe6\xa0\x87\xe5\xae\x9e\xe7\x8e\xb0</strong></p>\n<p id="T6QxrlzAKSX3pS6A" style="text-indent: 2em;"><strong>\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\xef\xbc\x9a\xe5\x9c\xa8\xe5\xbd\x93\xe5\x89\x8d\xe7\xa2\xb3\xe8\xbe\xbe\xe5\xb3\xb0\xe7\xa2\xb3\xe4\xb8\xad\xe5\x92\x8c\xe5\x8f\x91\xe5\xb1\x95\xe7\x9b\xae\xe6\xa0\x87\xe7\x9a\x84\xe5\xbc\x95\xe9\xa2\x86\xe4\xb8\x8b\xef\xbc\x8c\xe9\x93\xb6\xe8\x81\x94\xe5\xa6\x82\xe4\xbd\x95\xe5\x8a\xa9\xe5\x8a\x9b\xe2\x80\x9c3060\xe2\x80\x9d\xe5\x8f\x8c\xe7\xa2\xb3\xe7\x9b\xae\xe6\xa0\x87\xe7\x9a\x84\xe5\xae\x9e\xe7\x8e\xb0\xef\xbc\x9f</strong></p>\n<p id="gi9iykoeFzt6meWV" style="text-indent: 2em;"><strong>\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a</strong>\xe4\xbd\x9c\xe4\xb8\xba\xe4\xb8\x80\xe5\xae\xb6\xe8\xb4\x9f\xe8\xb4\xa3\xe4\xbb\xbb\xe7\x9a\x84\xe5\x9b\xbd\xe6\x9c\x89\xe4\xbc\x81\xe4\xb8\x9a\xef\xbc\x8c\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\xb0\x86\xe4\xbb\x8e\xe6\x8e\xa8\xe8\xbf\x9b\xe8\x87\xaa\xe8\xba\xab\xe7\xa2\xb3\xe5\x87\x8f\xe6\x8e\x92\xe5\x92\x8c\xe5\xb8\xa6\xe5\x8a\xa8\xe9\x93\xb6\xe8\x81\x94\xe7\x94\x9f\xe6\x80\x81\xe5\x9c\x88\xe5\x85\xb1\xe5\x90\x8c\xe7\xa2\xb3\xe5\x87\x8f\xe6\x8e\x92\xef\xbc\x8c\xe5\x8a\xa9\xe5\x8a\x9b\xe2\x80\x9c\xe5\x8f\x8c\xe7\xa2\xb3\xe2\x80\x9d\xe7\x9b\xae\xe6\xa0\x87\xe5\xae\x9e\xe7\x8e\xb0\xe3\x80\x82\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2\xef\xbc\x8c\xe6\x88\x91\xe4\xbb\xac\xe5\xb0\x86\xe4\xbb\x8e\xe8\x87\xaa\xe8\xba\xab\xe7\x9a\x84\xe7\xbb\x8f\xe8\x90\xa5\xe6\xb4\xbb\xe5\x8a\xa8\xe3\x80\x81\xe5\x9b\xad\xe5\x8c\xba\xe5\xbb\xba\xe8\xae\xbe\xe3\x80\x81\xe5\x91\x98\xe5\xb7\xa5\xe8\xa1\x8c\xe4\xb8\xba\xe5\xbc\x95\xe5\xaf\xbc\xe7\xad\x89\xe6\x96\xb9\xe9\x9d\xa2\xe5\x85\xa8\xe9\x9d\xa2\xe5\x87\x8f\xe5\xb0\x91\xe7\xa2\xb3\xe6\x8e\x92\xe6\x94\xbe\xef\xbc\x8c\xe5\x8a\x9b\xe4\xba\x89\xe6\x97\xa9\xe6\x97\xa5\xe5\xae\x9e\xe7\x8e\xb0\xe7\xa2\xb3\xe4\xb8\xad\xe5\x92\x8c\xe3\x80\x82\xe5\x8f\xa6\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2\xef\xbc\x8c\xe6\x88\x91\xe4\xbb\xac\xe5\xb0\x86\xe5\xb8\xa6\xe5\x8a\xa8\xe5\x90\x84\xe6\x88\x90\xe5\x91\x98\xe6\x9c\xba\xe6\x9e\x84\xef\xbc\x8c\xe7\xa7\xaf\xe6\x9e\x81\xe8\xb7\xb5\xe8\xa1\x8c\xe7\xa2\xb3\xe4\xb8\xad\xe5\x92\x8c\xe8\xa6\x81\xe6\xb1\x82\xef\xbc\x8c\xe6\x8e\xa8\xe5\x87\xba\xe6\x9b\xb4\xe5\xa4\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe9\x87\x91\xe8\x9e\x8d\xe4\xba\xa7\xe5\x93\x81\xe5\x92\x8c\xe6\x9c\x8d\xe5\x8a\xa1\xef\xbc\x8c\xe6\x94\xaf\xe6\x8c\x81\xe4\xbc\x81\xe4\xb8\x9a\xe5\x92\x8c\xe4\xb8\xaa\xe4\xba\xba\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe5\x8f\x91\xe5\xb1\x95\xe8\xaf\x89\xe6\xb1\x82\xef\xbc\x8c\xe5\xb9\xb6\xe8\x81\x94\xe5\x90\x88\xe5\x90\x84\xe6\x88\x90\xe5\x91\x98\xe6\x9c\xba\xe6\x9e\x84\xef\xbc\x8c\xe5\x80\x9f\xe5\x8a\xa9\xe9\x93\xb6\xe8\x81\x94\xe5\x8d\xa1\xe7\x9a\x84\xe5\x8f\x91\xe8\xa1\x8c\xef\xbc\x8c\xe4\xbc\xa0\xe6\x92\xad\xe7\xa2\xb3\xe4\xb8\xad\xe5\x92\x8c\xe7\x90\x86\xe5\xbf\xb5\xef\xbc\x8c\xe5\x8a\xa9\xe5\x8a\x9b\xe4\xbc\x81\xe4\xb8\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe5\x8f\x91\xe5\xb1\x95\xef\xbc\x8c\xe5\x80\xa1\xe5\xaf\xbc\xe4\xb8\xaa\xe4\xba\xba\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\x94\x9f\xe6\xb4\xbb\xe3\x80\x82</p>\n<p id="KodbISSvkS6iOavB" style="text-indent: 2em;"><strong>\xe6\x9c\xaa\xe6\x9d\xa5\xe5\xb0\x86\xe6\x95\xb4\xe5\x90\x88\xe4\xbc\x98\xe5\x8a\xbf\xe8\xb5\x84\xe6\xba\x90 \xe6\x8e\xa2\xe7\xb4\xa2\xe6\x9b\xb4\xe5\xa4\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xba\xa7\xe5\x93\x81\xe5\x92\x8c\xe6\x9c\x8d\xe5\x8a\xa1</strong></p>\n<p id="LZhf0dgYyPIKSF2s" style="text-indent: 2em;"><strong>\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\xef\xbc\x9a\xe6\x82\xa8\xe8\xae\xa4\xe4\xb8\xba\xef\xbc\x8c\xe9\x87\x91\xe8\x9e\x8d\xe8\xa1\x8c\xe4\xb8\x9a\xe5\x9c\xa8\xe5\x8f\x8c\xe7\xa2\xb3\xe7\x9a\x84\xe8\x83\x8c\xe6\x99\xaf\xe4\xb8\x8b\xe9\x9d\xa2\xe4\xb8\xb4\xe7\x9d\x80\xe5\x93\xaa\xe4\xba\x9b\xe5\x8f\x91\xe5\xb1\x95\xe6\x9c\xba\xe9\x81\x87\xe4\xbb\xa5\xe5\x8f\x8a\xe6\x8c\x91\xe6\x88\x98\xef\xbc\x9f</strong></p>\n<p id="uuPNEBKLaWMTzX0n" style="text-indent: 2em;"><strong>\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a</strong>\xe2\x80\x9c\xe5\x8f\x8c\xe7\xa2\xb3\xe2\x80\x9d\xe7\x9b\xae\xe6\xa0\x87\xe7\x9a\x84\xe6\x8f\x90\xe5\x87\xba\xe5\xb0\x86\xe6\x98\xaf\xe7\xa4\xbe\xe4\xbc\x9a\xe7\x94\x9f\xe4\xba\xa7\xe8\xa6\x81\xe7\xb4\xa0\xe8\xb5\x84\xe6\xba\x90\xe9\x85\x8d\xe7\xbd\xae\xe5\x92\x8c\xe7\x94\x9f\xe6\xb4\xbb\xe6\x96\xb9\xe5\xbc\x8f\xe8\xb0\x83\xe6\x95\xb4\xe7\x9a\x84\xe4\xb8\x80\xe6\xac\xa1\xe5\xb7\xa8\xe5\xa4\xa7\xe5\x8f\x98\xe9\x9d\xa9\xef\xbc\x8c\xe5\x90\x84\xe7\xb1\xbb\xe5\x9e\x8b\xe4\xbc\x81\xe4\xb8\x9a\xe9\x83\xbd\xe9\x9d\xa2\xe4\xb8\xb4\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe5\x8f\x91\xe5\xb1\x95\xe7\x9a\x84\xe8\xa6\x81\xe6\xb1\x82\xe3\x80\x82\xe4\xbc\x81\xe4\xb8\x9a\xe7\x9a\x84\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe8\xbd\xac\xe5\x9e\x8b\xe6\xb6\x89\xe5\x8f\x8a\xe8\x9e\x8d\xe8\xb5\x84\xe8\xb4\xb7\xe6\xac\xbe\xe3\x80\x81\xe6\x94\xaf\xe4\xbb\x98\xe7\xbb\x93\xe7\xae\x97\xe3\x80\x81\xe4\xbf\x9d\xe9\x99\xa9\xe7\xad\x89\xe5\x90\x84\xe7\xb1\xbb\xe9\x87\x91\xe8\x9e\x8d\xe9\x9c\x80\xe6\xb1\x82\xef\xbc\x8c\xe9\x87\x91\xe8\x9e\x8d\xe8\xa1\x8c\xe4\xb8\x9a\xe5\xb0\x86\xe5\x9c\xa8\xe5\x85\xb6\xe4\xb8\xad\xe5\x8f\x91\xe6\x8c\xa5\xe4\xb8\xbe\xe8\xb6\xb3\xe8\xbd\xbb\xe9\x87\x8d\xe7\x9a\x84\xe4\xbd\x9c\xe7\x94\xa8\xef\xbc\x8c\xe6\x8b\xa5\xe6\x9c\x89\xe5\xb9\xbf\xe9\x98\x94\xe7\x9a\x84\xe5\x8f\x91\xe5\xb1\x95\xe7\xa9\xba\xe9\x97\xb4\xe5\x92\x8c\xe6\x9c\xba\xe4\xbc\x9a\xe3\x80\x82</p>\n<p id="y925U998QpTGp6RF" style="text-indent: 2em;">\xe6\x88\x91\xe4\xbb\xac\xe7\x9b\xb8\xe4\xbf\xa1\xef\xbc\x8c\xe6\x9c\x89\xe9\x87\x91\xe8\x9e\x8d\xe8\xa1\x8c\xe4\xb8\x9a\xe7\x9a\x84\xe7\xa7\xaf\xe6\x9e\x81\xe5\x8f\x82\xe4\xb8\x8e\xef\xbc\x8c\xe7\xa2\xb3\xe9\x87\x91\xe8\x9e\x8d\xe4\xba\xa7\xe5\x93\x81\xe5\xb8\x82\xe5\x9c\xba\xe5\xb0\x86\xe5\xbf\xab\xe9\x80\x9f\xe5\x8f\x91\xe5\xb1\x95\xe6\x88\x90\xe7\x86\x9f\xef\xbc\x8c\xe9\x87\x91\xe8\x9e\x8d\xe8\xa1\x8c\xe4\xb8\x9a\xe5\xb0\x86\xe5\x8b\x87\xe7\xab\x8b\xe6\xbd\xae\xe5\xa4\xb4\xef\xbc\x8c\xe5\x88\x87\xe5\xae\x9e\xe6\x8e\xa8\xe8\xbf\x9b\xe2\x80\x9c\xe5\x8f\x8c\xe7\xa2\xb3\xe2\x80\x9d\xe7\x9b\xae\xe6\xa0\x87\xe7\x9a\x84\xe5\xa6\x82\xe6\x9c\x9f\xe5\xae\x9e\xe7\x8e\xb0\xe3\x80\x82</p>\n<p id="s5tVhxEfRrDVwjxY" style="text-indent: 2em;"><strong>\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\xef\xbc\x9a\xe6\x82\xa8\xe8\xae\xa4\xe4\xb8\xba\xef\xbc\x8c\xe9\x93\xb6\xe8\x81\x94\xe5\x9c\xa8\xe6\x9e\x84\xe5\xbb\xba\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\xa7\xaf\xe5\x88\x86\xe4\xbd\x93\xe7\xb3\xbb\xe4\xb8\xad\xe6\x89\xae\xe6\xbc\x94\xe4\xbb\x80\xe4\xb9\x88\xe6\xa0\xb7\xe7\x9a\x84\xe8\xa7\x92\xe8\x89\xb2\xef\xbc\x9f\xe6\x9c\xaa\xe6\x9d\xa5\xe5\x9c\xa8\xe7\xbb\xbf\xe8\x89\xb2\xe9\x87\x91\xe8\x9e\x8d\xe6\x96\xb9\xe9\x9d\xa2\xe8\xbf\x98\xe4\xbc\x9a\xe4\xbd\x9c\xe5\x93\xaa\xe4\xba\x9b\xe5\xb0\x9d\xe8\xaf\x95\xef\xbc\x9f</strong></p>\n<p id="Ol1pwNfNLNvaEuFb" style="text-indent: 2em;"><strong>\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a</strong>\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\x9c\xa8\xe6\x9e\x84\xe5\xbb\xba\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\xa7\xaf\xe5\x88\x86\xe4\xbd\x93\xe7\xb3\xbb\xe4\xb8\xad\xe5\x90\x8c\xe6\x97\xb6\xe6\x89\xae\xe6\xbc\x94\xe7\x9d\x80\xe8\xa7\x84\xe5\x88\x99\xe5\x88\xb6\xe5\xae\x9a\xe8\x80\x85\xe5\x92\x8c\xe6\xb4\xbb\xe5\x8a\xa8\xe5\x8f\x91\xe8\xb5\xb7\xe8\x80\x85\xe7\x9a\x84\xe8\xa7\x92\xe8\x89\xb2\xe3\x80\x82\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2\xef\xbc\x8c\xe9\x93\xb6\xe8\x81\x94\xe6\x9e\x84\xe5\xbb\xba\xe4\xba\x86\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\xa7\xaf\xe5\x88\x86\xe4\xbd\x93\xe7\xb3\xbb\xe5\x8f\x8a\xe7\x9b\xb8\xe5\xba\x94\xe7\x9a\x84\xe6\xb4\xbb\xe5\x8a\xa8\xe7\x8e\xa9\xe6\xb3\x95\xef\xbc\x8c\xe9\x92\x88\xe5\xaf\xb9\xe6\x80\xa7\xe5\x9c\xb0\xe5\xaf\xb9\xe6\x8c\x81\xe5\x8d\xa1\xe4\xba\xba\xe7\x9a\x84\xe7\xbb\xbf\xe8\x89\xb2\xe6\xb6\x88\xe8\xb4\xb9\xe8\xa1\x8c\xe4\xb8\xba\xe8\xbf\x9b\xe8\xa1\x8c\xe5\xa5\x96\xe5\x8a\xb1\xef\xbc\x8c\xe7\x94\xa8\xe6\xb4\xbb\xe5\x8a\xa8\xe8\xa7\x84\xe5\x88\x99\xe5\x8e\xbb\xe5\xbc\x95\xe5\xaf\xbc\xe5\x92\x8c\xe9\xbc\x93\xe5\x8a\xb1\xe5\xa4\xa7\xe4\xbc\x97\xe8\xb7\xb5\xe8\xa1\x8c\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\x94\x9f\xe6\xb4\xbb\xe3\x80\x82\xe5\x8f\xa6\xe4\xb8\x80\xe6\x96\xb9\xe9\x9d\xa2\xef\xbc\x8c\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\x9f\xba\xe4\xba\x8e\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe7\xa7\xaf\xe5\x88\x86\xe4\xbd\x93\xe7\xb3\xbb\xef\xbc\x8c\xe8\x81\x94\xe5\x90\x88\xe5\x90\x84\xe5\xae\xb6\xe5\x95\x86\xe4\xb8\x9a\xe9\x93\xb6\xe8\xa1\x8c\xe5\x8f\x91\xe8\xa1\x8c\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xb8\xbb\xe9\xa2\x98\xe5\x8d\xa1\xef\xbc\x8c\xe5\xb8\xa6\xe9\xa2\x86\xe5\x92\x8c\xe5\x8f\xb7\xe5\x8f\xac\xe9\x93\xb6\xe8\xa1\x8c\xe4\xb8\x9a\xe5\x85\xb1\xe5\x90\x8c\xe5\x8f\x82\xe4\xb8\x8e\xe5\x88\xb0\xe7\xbb\xbf\xe8\x89\xb2\xe9\x87\x91\xe8\x9e\x8d\xe6\x9c\x8d\xe5\x8a\xa1\xe4\xb8\xad\xef\xbc\x8c\xe4\xb9\x9f\xe8\xae\xa9\xe6\x9b\xb4\xe5\xa4\x9a\xe6\x8c\x81\xe5\x8d\xa1\xe4\xba\xba\xe5\x8f\x82\xe4\xb8\x8e\xe5\x88\xb0\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe8\xa1\x8c\xe5\x8a\xa8\xe4\xb8\xad\xe3\x80\x82</p>\n<p id="f9AJUXjxTqqCrzyn" style="text-indent: 2em;">\xe6\x9c\xaa\xe6\x9d\xa5\xef\xbc\x8c\xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\xb0\x86\xe8\x81\x94\xe5\x90\x88\xe5\x95\x86\xe4\xb8\x9a\xe9\x93\xb6\xe8\xa1\x8c\xe3\x80\x81\xe8\xa1\x8c\xe4\xb8\x9a\xe5\x90\x88\xe4\xbd\x9c\xe4\xbc\x99\xe4\xbc\xb4\xef\xbc\x8c\xe6\x95\xb4\xe5\x90\x88\xe4\xbc\x98\xe5\x8a\xbf\xe8\xb5\x84\xe6\xba\x90\xef\xbc\x8c\xe6\x8e\xa2\xe7\xb4\xa2\xe6\x9b\xb4\xe5\xa4\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xba\xa7\xe5\x93\x81\xe5\x92\x8c\xe6\x9c\x8d\xe5\x8a\xa1\xe3\x80\x82</p></span> <div id="articleEdit"><span class="tiyi1 domPC"><a href="javascript:void(0);" class="advise">\xe3\x80\x90\xe7\xba\xa0\xe9\x94\x99\xe3\x80\x91</a> <div id="advisebox01" class="tiyi01" style="display:none"><div><iframe id="jc_link1" border="0" marginwidth="0" framespacing="0" marginheight="0" frameborder="0" noresize="noresize" scrolling="no" vspale="0" style="width:600px;height:350px; float:left;"></iframe></div> <div class="tiyi03"><div id="jc_close1" style="cursor:pointer"><img src="http://www.news.cn/images/syicon/space.gif" width="24" height="24" border="0"></div></div></div></span> <span class="editor"> \xe3\x80\x90\xe8\xb4\xa3\xe4\xbb\xbb\xe7\xbc\x96\xe8\xbe\x91:\xe4\xb8\x87\xe5\x98\x89\xe3\x80\x91 </span></div></div>\n\t\t\t\n\t\t\t\n\t\t\t\n\t\t\t \n            <div class="nextpage clearfix" style="display:none;">\n                <div class="nextpage-tit">\xe9\x98\x85\xe8\xaf\xbb\xe4\xb8\x8b\xe4\xb8\x80\xe7\xaf\x87\xef\xbc\x9a</div>\n                <div class="nextpage-cont"><a href="#"></a></div>\n            </div>\n\t\t\t\n\t\t\t<div class="ad_right domPC" style="margin-bottom:20px"><ins data-ycad-slot="399"></ins> </div>\n\t\t\t\n\t\t\t<div class="domPC"><ins data-ycad-slot="2168"></ins> </div>\n\t\t\t<div class="domPC"><ins data-ycad-slot="2169"></ins> </div>\n\t\t\t\n            <div class="bookList clearfix domPC">\n                <ul>\n                </ul>\n            </div>\t\t\n\t\t\t<div class="domPC"><ins data-ycad-slot="2169"></ins></div>\n\n        </div>\n        <div class="main-right right">\n\t\t\t<div class="ad_right domPC"><ins data-ycad-slot="2174"></ins> </div>\n            <div class="columBox hotList">\n                <div class="col-tit">\n                    <span><a href="http://www.news.cn/2021homepro/rsznb/index.htm" target="_blank">\xe6\x96\xb0\xe9\x97\xbb\xe7\x83\xad\xe6\xa6\x9c</a></span>\n                </div>\n                <div class="col-cont">\n                    <ul>\n                    </ul>\n                </div>\n            </div>\n\t\t\t<div class="ad_right domPC"><ins data-ycad-slot="2175"></ins> </div>\n            <div class="columBox book">\n                <ul>\n                </ul>\n            </div>\n\t\t\t<div class="ad_right domPC"><ins data-ycad-slot="2176"></ins> </div>\n\t\t\t<div class="ad_right domPC"><ins data-ycad-slot="2177"></ins> </div>\n            <div class="columBox video">\n                <div class="col-tit">\n                    <span><a href="http://www.news.cn/videopro/" target="_blank">\xe6\x96\xb0\xe5\x8d\x8e\xe8\xa7\x86\xe9\xa2\x91</a></span>\n                </div>\n                <div class="col-cont">\n                    <ul dataType="ds" data="datasource:b07af04f61af43b7a28a15fa6aa30212" preview="ds_"><li><div class="pic"><a href="http://www.news.cn/multimediapro/20210629/9a965f685ff14c9fa074f140d976da4b/c.html" target="_blank"><img src="/money/20210908/0e3a78a113e34a95b4962a28e0f21844/20210630/324484144dc74505b6303bf29d1da5c1/883ec26132ef4dde9ca128d7e9ac9b3d_666.gif"></a> <div class="tit"><div><a href="http://www.news.cn/multimediapro/20210629/9a965f685ff14c9fa074f140d976da4b/c.html" target="_blank">\xe2\x80\x9c\xe7\x99\xbd\xe9\xb9\xa4\xe2\x80\x9d\xe8\x88\x9e\xe9\x87\x91\xe6\xb2\x99</a></div></div></div></li><li><div class="pic"><a href="http://www.xinhuanet.com/2021-05/04/c_1211140315.htm" target="_blank"><img src="/money/20210908/0e3a78a113e34a95b4962a28e0f21844/20210507/d2d5cb97dc2e4c2f8631bc88171c5d91/56a7336620c44fdb927c83b17077bd34_275x225_e7b31d77-55ab-4147-bbb4-c50b75c7e8552021571540534.gif"></a> <div class="tit"><div><a href="http://www.xinhuanet.com/2021-05/04/c_1211140315.htm" target="_blank" rel="noopener">\xe8\x88\x9e\xe5\x8a\xa8\xe7\x99\xbe\xe5\xb9\xb4\xe8\x8a\xb3\xe5\x8d\x8e</a></div></div></div></li><li><div class="pic"><a href="https://xhpfmapi.zhongguowangshi.com/vh512/share/9941981?channel=qq" target="_blank"><img src="/money/20210908/0e3a78a113e34a95b4962a28e0f21844/20210507/edc10cd86680425aa4ffb58fb6b963f3/acb9d0ed057f4933a4fc83335f96f40d_7777.jpg"></a> <div class="tit"><div><a href="https://xhpfmapi.zhongguowangshi.com/vh512/share/9941981?channel=qq" target="_blank">\xe2\x80\x9c\xe7\xa1\xac\xe6\xa0\xb8\xe2\x80\x9d\xe5\x94\x90\xe5\xb1\xb1\xe7\x9a\x84\xe7\xba\xa2\xe8\x89\xb2\xe5\x8d\xb0\xe8\xae\xb0\xe4\xb8\x8e\xe7\xbb\xbf\xe8\x89\xb2\xe5\x8f\x91\xe5\xb1\x95</a></div></div></div></li></ul>\n                </div>\n            </div>\n        </div>\n    </div>\n<div class="adv domMob">\n          <div class="advCont" style="display:none"><ins data-ycad-slot="2306"></ins></div>\n          <div class="advShow"></div>\n        </div>\n    <div class="foot">\n        <script src="http://lib.news.cn/common/foot.js"></script>\n    </div>\n\t<div class="domPC" style="margin-bottom:20px"><ins data-ycad-slot="2262"></ins> </div>\n    <div dataType="content" preview="ds_"><div id="fontsize" style="display:none">\n                \n            </div> <div id="wxpic" style="display:none"><img src></div> <div id="wxtitle" style="display:none">\n                \xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe9\x93\xb6\xe8\xa1\x8c\xe5\x90\x88\xe4\xbd\x9c\xe9\x83\xa8\xe6\x80\xbb\xe7\xbb\x8f\xe7\x90\x86\xe5\x8d\x95\xe9\x95\xbf\xe8\x83\x9c\xef\xbc\x9a\xe8\xb7\xb5\xe8\xa1\x8c\xe2\x80\x9c\xe6\x94\xaf\xe4\xbb\x98\xe4\xb8\xba\xe6\xb0\x91\xe2\x80\x9d\xe7\x90\x86\xe5\xbf\xb5 \xe4\xb8\xad\xe5\x9b\xbd\xe9\x93\xb6\xe8\x81\x94\xe5\xb0\x86\xe6\x8e\xa2\xe7\xb4\xa2\xe6\x9b\xb4\xe5\xa4\x9a\xe7\xbb\xbf\xe8\x89\xb2\xe4\xbd\x8e\xe7\xa2\xb3\xe4\xba\xa7\xe5\x93\x81\xe5\x92\x8c\xe6\x9c\x8d\xe5\x8a\xa1\n            </div></div>\n\t<script>\n\t\t//\xe5\xbe\xae\xe4\xbf\xa1\xe5\x88\x86\xe4\xba\xab\xe5\x9b\xbe\xe5\x8a\x9f\xe8\x83\xbd\n\t\tvar wxfxPic = $.trim($("#wxpic").find("img").attr("src"));\n\t\tvar wxfxTit = $.trim($("#wxtitle").html());\n\t\tvar detaiWxPic = $("#wxsharepic").attr("src");\n\t\tif(wxfxPic==""){\n\t\t\twxfxPic = \'http://lib.news.cn/common/sharelogo.jpg\';\n\t\t}else{\n            wxfxPic = "http://www.news.cn"+wxfxPic\n        }\n\t\tconsole.log("wxfxPic",wxfxPic);\n\t\twxConfig({\n\t\t\ttitle: wxfxTit,\n\t\t\tdesc: \'\xe6\x96\xb0\xe5\x8d\x8e\xe7\xbd\x91\xef\xbc\x8c\xe8\xae\xa9\xe6\x96\xb0\xe9\x97\xbb\xe7\xa6\xbb\xe4\xbd\xa0\xe6\x9b\xb4\xe8\xbf\x91\xef\xbc\x81\',\n\t\t\tlink: window.location.href,\n\t\t\timgUrl: wxfxPic\n\t\t});\n\t</script>\n    <script src="http://www.news.cn/2021homepro/scripts/fixedNav.js"></script>\t\n    <script src="http://www.news.cn/json/bangdan/top1.json"></script>\n    <script src="http://www.news.cn/2021detail/js/xh-column.js"></script>\n    <script src="pager.js"></script>\n    <script src="http://www.xinhuanet.com/detail/detail_2021new.js"></script>\n    <script src="http://www.news.cn/2021detail/js/booklist.js"></script>\n    <script src="http://www.news.cn/2021detail/js/xh-video.js"></script>\n    <script type="text/javascript" src="http://a2.news.cn/js/xadndelayed.js"></script>\n    <script src="http://www.news.cn/mobile/20210315mobile/scripts/ad_mobile.js"></script>\n</body>\n</html>'
    print(f'检测bytes内容编码结果：{demo_ins.char_detect(_temp_content)}')
    print(f'解码bytes内容转为str结果：{demo_ins.char_to_str(_temp_content)}')

    # 图片处理
    temp_aa = demo_ins.img_blurred(r'E:\Users\Administrator\Documents\SomeTools\win11_sunrise.jpg')
    print(f"图片模糊后路径{temp_aa}")
    temp_aa, img_path = demo_ins.img_generate_verification_code(r'E:\Users\Administrator\Documents\SomeTools\Arial.ttf', r'E:\Users\Administrator\Documents\SomeTools\\')
    print(f"根据字符串{temp_aa}生成验证码{img_path}")

    # 查看内存信息
    print(f'内存信息：{demo_ins.os_memory_info()} 可用内存剩余：{demo_ins.os_memory_info().free}')
    print(f'CPU信息：{demo_ins.os_cpu_info()}')
    print(f'硬盘信息：{demo_ins.os_disk_info()}')
    print(f'网络信息：{demo_ins.os_net_info()}')
```


### 2.异步工具使用

``` pycon
import asyncio
import platform

if not (platform.system() == 'Windows'):
    import uvloop

    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())  # 使用 uvloop 来替换 asyncio 内部的事件循环。

from sometools.async_tools import CommonAsyncTools


class Demo(CommonAsyncTools):
    def __init__(self, *args, **kwargs):
        super(Demo, self).__init__(*args, **kwargs)
        # 事件循环
        task_number = kwargs.get('TASK_NUMBER')
        loop = asyncio.get_event_loop()
        tasks = [asyncio.ensure_future(self.ready(**kwargs)) for _ in range(task_number)]
        if platform.system() == 'Windows':
            loop.set_debug(True)
        loop.run_until_complete(asyncio.wait(tasks))

    async def ready(self, **kwargs):
        # 获取链接
        self.aio_redis_conn = await self.get_async_redis_conn(redis_host=kwargs.get('redis_host'),
                                                              redis_port=kwargs.get('redis_port'),
                                                              redis_db=kwargs.get('redis_db'),
                                                              redis_pwd=kwargs.get('redis_pwd'))
        # 使用
        while 1:
            msg = await self.aio_redis_conn.set('temp_key', 'test string')
            print(f'set {msg}')
            msg = await self.aio_redis_conn.get('temp_key')
            print(f'get {msg}')
            await asyncio.sleep(5)
            if __name__ == "__main__":
                break


if __name__ == '__main__':
    # 异步工具使用方法示例
    demo_ins = Demo(TASK_NUMBER=1, redis_host='10.1.90.29',
                    redis_port='6379',
                    redis_db=15,
                    redis_pwd='lhdna2016')
    print('start')
```