# flake8: noqa: E501
# SHA256 hashes of kernel files
from __future__ import annotations

from dataclasses import dataclass


@dataclass
class AesKey:
    key: bytes
    iv: bytes

    def __init__(self, key: bytes | str, iv: bytes | str):
        self.key = bytes.fromhex(key) if isinstance(key, str) else key
        self.iv = bytes.fromhex(iv) if isinstance(iv, str) else iv

    def __repr__(self) -> str:
        return f"AesKey(key={self.key.hex()!r}, iv={self.iv.hex()!r})"


@dataclass
class ChaCha20Key:
    key: bytes
    iv: bytes

    def __init__(self, key: bytes | str, iv: bytes | str):
        self.key = bytes.fromhex(key) if isinstance(key, str) else key
        self.iv = bytes.fromhex(iv) if isinstance(iv, str) else iv

    def __repr__(self) -> str:
        return f"ChaCha20Key(key={self.key.hex()!r}, iv={self.iv.hex()!r})"


@dataclass
class ChaCha20Seed:
    key: bytes

    def __init__(self, key: bytes | str):
        self.key = bytes.fromhex(key) if isinstance(key, str) else key

    def __repr__(self) -> str:
        return f"ChaCha20Seed(key={self.key.hex()!r})"


KERNEL_KEY_MAP = {
    # FGR_60F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "a5030ba41680b1e855ed347b18d0e0d7d0cd7b5399098e3e1a23fbe75380cf41": ChaCha20Key(
        "dbd7d3a12fe7035464d9ce343d810b6187c247f629593ec3b4b10122e403473a", "f4b097fa46868bab133740678fa589c0"
    ),
    # FGR_60F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "9453f83f302b65501d484ad830c1cbfaad88e3d1b9148f125972aee7b5761ada": ChaCha20Key(
        "f44153720fadc950b22bc37d4684e19cf47e4d9f916d1309e795ffafe249db81", "f90687623058bbeed3f17742c00f7e41"
    ),
    # FGR_60F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "15b7a3dd321252930f50e1919e360f777b0710247824177bbe806ccefdddcd1e": ChaCha20Key(
        "25b02156a8a4641a4f56b321d5ac5c15a9ab496c105b030bd41e164c9a96aec9", "cc299ce688fd32918941c211319b9616"
    ),
    # FGR_60F_3G4G-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "22002eefe7ae49c70227151fe84aa2fbe8f0362334b49166b9a24b2c7b423cd3": ChaCha20Key(
        "22cd97ce8a901d2c209e115bc95a2615c1042a26413840dfe71dd4743845989e", "c5c96c1e3d463c24a2b364703a618102"
    ),
    # FGR_60F_3G4G-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "23fd52352ff14e873c77c303d0a51669672c9e0bdcd9c4d9ef16263110134d94": ChaCha20Key(
        "15f1260a89f9a4132dce404b1bd2983c28df2dbdfd1f0af95310ab87bc75c729", "727e32526ccf5fdf930052712cdeac2f"
    ),
    # FGR_60F_3G4G-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "967acc7a67bda76f9d88070d12da403c1906b50be2449c57acda62feb9fb41ae": ChaCha20Key(
        "7c0e5ce5381a7063ba3a2b9dbdb731a1d9daaed663c59a10bc9c3d06f51a21aa", "cd44408b1f46bbac9ca7db4193bbdeaa"
    ),
    # FGT_1000D-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "a324a6655e5350f367c8696e4598f66ad750d6d77a2cf2ab9d6f2401781e3416": ChaCha20Key(
        "2c4a4ae16fab78d21769a9816dad7b4f26d3c8d1f98ca4921ea217fa8f70c0c1", "f9fd7d9883fea9e44bbc6a0856e1cddd"
    ),
    # FGT_1000D-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "188a76caeb9f1dde70312d6a5f87760ad331eefb8dd309b6ac40bff58de05bb9": ChaCha20Key(
        "73e07e127c2362678743e0c6038347062d09e4de7268a8517a3707d2632001d4", "1662074295905672270cfc185ae44dcd"
    ),
    # FGT_1000D-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "ae0dd86f1330b3dbc38d72535f8de51e99080305b09db43a92f6c1d80fc98c1e": ChaCha20Key(
        "bb01a7b65205c2452aa23641ef927046de13cbff20d87903efc2ba7959026f5f", "e9b60fe7867b87f94bdfb72ba26c8cd5"
    ),
    # FGT_100E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "64b7c2c194c1c2cff87454d54bdaa79c4f635d3a452f9c9eed7558956957d961": ChaCha20Key(
        "bc9bbe0cf699bc2e0a40eecd91f7bf436793f4cc9d317661b5ee2122347616af", "b68f31b1efbe74d06e9de18ec7ab8791"
    ),
    # FGT_100E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "9fa81b725fcc7541d738b01f18aebb42c6ace8e859b77e1ab0c9c3d206281232": ChaCha20Key(
        "fdf30619861a1b7418e6c86f7c1ffa9be3c350be7afda65d770a1904dde5e1b1", "beb9eb00f50b5e3612b2c5322aa73b98"
    ),
    # FGT_100E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "735de8d52a492265b9dc6a38dcf0a0f3c673fcfc9bdd5ca50f5adddb3c990e9d": ChaCha20Key(
        "a9f752273789ab5aa669ef72a2647d8937df8ab9854c6c170476cac53a3423ae", "99af9f67a2e053f0d67a21b8046a34f1"
    ),
    # FGT_100EF-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "1182c027f0b62bba7694f5f6d650763e5edae099ddf3c0658427135c469a9465": ChaCha20Key(
        "edd5f0ac2690847d0cad120e8ca29e30c888e7f3a9fe67c7b45cef21d0b7d501", "dee30f0f3a19d0b4d6e0e7a2aafe0d00"
    ),
    # FGT_100EF-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "a02ca3db93ddb86d6749b2d16c66dde40d03d53031bd287eea09d4c8148c6cdd": ChaCha20Key(
        "22a7dfc3dd2658483494982c10890802b7e2b31eaaeeaa9982e42a433f45373c", "8fb477775739a1d25ee3b96257c81c16"
    ),
    # FGT_100EF-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "484aca157f2009ed509b6ffa9cca1ac76b57856a5497cb2285b465e37cf46757": ChaCha20Key(
        "8f6f3afb52bae455f2ed3972b46ea6f77538a2c231a041f5fc2a1bcc049b7737", "c1aaa11be8a7899d0527b70df2f12003"
    ),
    # FGT_100F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "b95ff715ab12199945d7133f0b9f05bb59471207d2a37e2c469655263c90119c": ChaCha20Key(
        "054bd2d7c3154b958f627b8fbd130d3dac30eb717435cd53542c940b43f2871b", "0656e3e675049ca959501d22167ac517"
    ),
    # FGT_100F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "cab595e798478d25a6a65a9202f97b395fbe9b3c7ff37b31b2833fa5455bb9a4": ChaCha20Key(
        "b70c8c42ac83cf85da3d21c2004cf2738076f3b21d850b97f95efe6a5745cbda", "30a94cc09d4bdd0d4ad22d910c87e211"
    ),
    # FGT_100F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "ce232bb25b688a358ae759baae99e2236b0f37147bcea9bdd9ddae223cda2bbc": ChaCha20Key(
        "2060839c574455695e97e32695175de6048a8db596a94c702272a49f5ad91175", "363adb80c6e79b08c0f52c4f5f5feeda"
    ),
    # FGT_101E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "b61467f4543f7347323b2dc5d0152ba08d0d7edbae1b50ec056b80fbdc36a327": ChaCha20Key(
        "ec7402474ce765e6eab29e67d02949d8ac9fd490b5a10c9b171ba531f97d77be", "c0cfec9ebf783e8da381f16cc1121f7d"
    ),
    # FGT_101E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "fc341cf2c5484411de5e8d6e55601cc60a607078ff2ab699ff7f74f75a081c74": ChaCha20Key(
        "082bbc1a3709b024013c93f15784ccd0b7bf48f7fd205fa1aeb0ad6f7930afc1", "41f8bcdf46c7f7ebb7b21c8041fe1f3c"
    ),
    # FGT_101E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "b8aa2889f348c0847c3da8a2357762f5b22905f458b8634274239fc934ed835d": ChaCha20Key(
        "bfd3af4dd6f36b93e99b282b9a98f7b39b30b6b34dc368afcab30973889f28ba", "60916ad3c606b785342369eafa85d992"
    ),
    # FGT_101F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "fd29b4aec38ced0e07a09fe984b1aa3994d667c03cc695d9674229ba67850630": ChaCha20Key(
        "d60157dba242f623389069b05858954e83ff731392f7df61016589555319f774", "d24444ed204cafb6a141f717cf3fd008"
    ),
    # FGT_101F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "884ef7f3b25e1f5861ee38ab12567df68a41bb86cd1ddd397a0a1b2bc079d412": ChaCha20Key(
        "36baa68bbc7ba052f71f32fdee3aed2670efa6700f03eb4aa117351a1f265ac8", "f3413e6c3cfdb6ea1126e72239763420"
    ),
    # FGT_101F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "89407bf8ce49dbbfec6981c2eb494aea8bc42dc903c0b03500706a5a350145a3": ChaCha20Key(
        "ad63ba3b35c69c075b61abf9f507ac081492dcdd4b45709f1f143850007b087f", "19cff81ec0768c5d2299ee697b1d8d9b"
    ),
    # FGT_1100E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "fdd82a05f3cf384cb460e683b6e99fc70bbfff9bf0229ac606bd066cf287f718": ChaCha20Key(
        "cca60764166f672e551a02f9fdc7918a17acd637c2290076246d59dcdf4c7918", "13fc079c58fdbd1d2751998dc7668b82"
    ),
    # FGT_1100E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "3f5edd461d733dad514b16d62c93cc49e8e5e9045f3ec0be6df6ec5baf571923": ChaCha20Key(
        "2890f5135078816c5ff7c7527b073a2b227d6bab8f229001d6d4b9a2ad1b79c0", "76d219c383ae042d4bb4806a8e1408bd"
    ),
    # FGT_1100E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "23f591404998b6d70092da8b6e1ebc3d0a9563c57c05538d5d808fb86f2729ce": ChaCha20Key(
        "e8b7556a5ade40a14631c6f6ad3d6b7421d62fecf7b74f0c585acd0dec79a2f6", "6f5efdd94b3437e69f902f016c1c1f54"
    ),
    # FGT_1101E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "3a8aabfc26deb915e0888baa87c337f51bb331b6d54d07382f1d9c7e1127b78e": ChaCha20Key(
        "fc7207e68a43f53a780ead9710c3bb0f4e9b68b8c9664390ec9cff4ac18d9c2e", "d6cb98202b1c6f7b9c05a240b6624d3d"
    ),
    # FGT_1101E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "216fdd47063920f356db37043b92255c622b65eda1530e609ee3539998efd749": ChaCha20Key(
        "4f1c935b44a832a542e240174887a44f9835abe68b8c2fcbbd9203a91f2837fe", "25a0ed64f760e6df8dbfc7567fa10869"
    ),
    # FGT_1101E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "a5de55d6a58e241134b65847d24bf1ec97ef924e42cc69c86b861d5ba9d81885": ChaCha20Key(
        "e0c83f25b86188087b5e3109a2adc1025bac2e6d2e0a8c43c4d5f7d95575aa7b", "35bd35414eab763d49c71f8b0eb26cc2"
    ),
    # FGT_140E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "6c900aeab6ac4dba9a24e07e1868e8c2c3259a943a2ee9aa80ec651ab9a34eaa": ChaCha20Key(
        "3518e076d1bc3c80af37595cc4de1e77b84b8127ca8a8c85d4e31a35c37e5fcc", "7b777c81ac79824bbba6e133fa2e2cf6"
    ),
    # FGT_140E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "09e5d69938b044dd00f4b9b92b56d7287e2866fac7aea76826db3ba5a6b9069b": ChaCha20Key(
        "15d64751afe95604d9aaf7f0a43add33132a9a79fd0c79a2b80a2f7c81f4c053", "10d5f93b40b0d8a3b11ac70d6aa3120b"
    ),
    # FGT_140E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "778e1f45ac1b823c02ca6d5c9ad5b6b5e12970b98fcef2210b2517fc62495844": ChaCha20Key(
        "e19f2c8f80c19d8766c99e30ab37c0797d3ee3b6589e931101ec673b75558f0a", "eb3386a157b6b66a64075206b35d5916"
    ),
    # FGT_140E_POE-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "47ef71e6d1a4fa06862fecc69b64f97af3aff942be019709edacba5722532674": ChaCha20Key(
        "8a5641e4cfb25882e65be1696b223d4a2edc7b018098f3a78e519ddcece22133", "760fb74f789cb4e9d31df45ab01c4e24"
    ),
    # FGT_140E_POE-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "b6b34b606ed8031ebaf16823efa03a090a5c70f66d567b56c71b49e54b79aeff": ChaCha20Key(
        "e4c4a0d72362fbcf565551a29730a660631d4bf5a6383a0dd2b9767d0ebff7de", "5785216b6211f6253108b4e91484a051"
    ),
    # FGT_140E_POE-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "4529b179234753e4cb80a16493a43aadafdc31e71853c676bc34eda0b6236f55": ChaCha20Key(
        "c029bc6d7e179355e556c8e4e3262fd0f14600e17a6a9844e6fc1a06b3348c1b", "214a434de4c60b373add3dbf33ddd4ca"
    ),
    # FGT_1500D-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "db7c4356fb1341c11a676d8c9f674cd6f0fbbf0c83d087cb12d04de200706cb3": ChaCha20Key(
        "860a46fa5bd8253fb0a55162de3ad3bc3dce25f378c408e65eec6b420825de12", "a43e30497cee9a0a262d112e61dc099f"
    ),
    # FGT_1500D-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "2d9c84635c742a35f1464345c4d939a1e7bf6a4b2085a2603ec3588dac1b6ecd": ChaCha20Key(
        "b3d3fdf96ca83172cf0eac82a244b9362864671254d9fdae8bd591c1d0d45b25", "2bad35a7a930ba6bd0d50013471415ed"
    ),
    # FGT_1500D-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "2581f1b8c2eb569eceac594f438ce2aceef04f1f60c8561130ec4aca917390c0": ChaCha20Key(
        "2f69b297171bdf04d93a44af13643cc7a261fb576ed67df67df286c24b29ec84", "105e3120d852d2b870b0c169daab3943"
    ),
    # FGT_1800F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "1789baff1c7ed08f0866a36e6004c64a5ab91369ece3b1f8f72e8870becea64a": ChaCha20Key(
        "fdbecd24e93ea0339d5079878271fdf251de6b8adcee7bfd3b9c1242d025749c", "febf9e5c0cce75f9c2fd6f11470712b6"
    ),
    # FGT_1800F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "cd19000663c255ef8f35efd4a2f7b179ae9174750b121f89e0b5e26ca6f041ee": ChaCha20Key(
        "cc266a68b16e857da85cefc3c028e88d8d94226b213e36ec808587c832125fe3", "7fa64d21723e1b95929b253356f2010b"
    ),
    # FGT_1800F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "d63f841174f66df749104b2722a8ad86adb0758abe4c183121f6f299faeb9108": ChaCha20Key(
        "fb9a3cb4761c10aa8343ccf5580f4176627517d921ce21824a285fb403ab8944", "50a7d6e582519ae2f9ff11fa6ce7c9ae"
    ),
    # FGT_1801F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "27f58df4371e50df3bf62337217e8bc7dc540dc93bb0efc8766c7c65931f7c87": ChaCha20Key(
        "88dd502edc9bafe2b998c11baca0fe5da77f784f96c1b8bc6ceb2722136a11fa", "a4301dabff5aa4685903ea523c038a05"
    ),
    # FGT_1801F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "cbc2c245bbc51e5f0c42007eb373c763b4c5c3c499c14ec6b5715f395e8d3b0e": ChaCha20Key(
        "145f9f989c2ac09b019e95494ec678b199d7e4e19301cfbb7276a5c5fdbb1670", "fee92cb8f9fbe53391780d6433eb00a3"
    ),
    # FGT_1801F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "c50e0a83f62b9ae7a98bed1f893176fd71309d23d3c694b3af87b02579720b7c": ChaCha20Key(
        "4595320c2a6df86eafc82ab883d5e8b40f2dce6470e98eac5aa3dff491041077", "4ad0c4d2427af429fa6fdf9a786a1271"
    ),
    # FGT_2000E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "cb42ffe1a3c82da6912006d860e4dabe78dc9207778e3fcd19a51afe1d8e9a39": ChaCha20Key(
        "ab454aaa3eea0844cfa3c46092a751203ae552b2c6aea0dee8f18006bc9ac500", "f163202c735f98e8594425a28803f0e2"
    ),
    # FGT_2000E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "dba56d5eb0224ee600cebbe0d4dc3f034fd4f7318443d93d3cad839753f3113f": ChaCha20Key(
        "54185ff9d4c71fe90ac095539646019290a874c7137b54206ca984e9ebb462dd", "c964945052c9c818ac13f503f4a91b4d"
    ),
    # FGT_2000E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "db770140735d48b4b1a481619ff5f64bd898bfb4eca0dcbeb51c87c7bf64e1cd": ChaCha20Key(
        "08c642d76fe374a3a213cb632da2048c6e41e589b5f9c4b1b72f07845de84957", "4d7401d08c5936af02c78129e8765bc8"
    ),
    # FGT_200E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "e326c20d8faa0bb43e271e33c7e8604af6d6525c4653b54c7835babd9bc8888b": ChaCha20Key(
        "84b3c4b2d931e6174b8202dc9b6fe1ca957a089df77b9f7d17da80efd2b6d994", "83c3378e753b87d139eb5d89c43440c5"
    ),
    # FGT_200E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "4f5f1af29e98b5267b4ac81f25799d4486af8ea89708139e9a19da6f32e4f1ca": ChaCha20Key(
        "b360ffcc7691abd6932fae226f68c5351ac7239f10e8ab643b9a5c37ccc82aa1", "2e2e00b88b657d13214b613aec99f6dd"
    ),
    # FGT_200E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "449ce32a67cabf7a483e9e2b644ec1e60a96e8174a259f4d6da173ba01397adb": ChaCha20Key(
        "c8f978d9f8553fa6e41775f1bd5efc3bde53c2c39be310b09daa7b4fa12036f1", "aff724b9ed1de051a743b7966fc9020e"
    ),
    # FGT_200F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "4882adff2893c5084a9d8202aa9167b60803346fe8b5c1ac908e530a6519a492": ChaCha20Key(
        "11268d2bdd36f83e2788b23d704a9115207910da8b4382ea6e7d577510fcdb9f", "51ff434976d3366f22a910770dcf6788"
    ),
    # FGT_200F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "a46e2439701d8fc98738ec0462124aaa7a99a6da2d143a3681ffed76f0ae19cb": ChaCha20Key(
        "dcd9f01a115cdbca370a5856a4ea35c9b708899ec99aff399bb26ba68618d784", "b0366d3c629923e7533521aebab0411e"
    ),
    # FGT_200F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "b9b4ff142af0653799e659d702b9bf86659cf3815fc2e9e9c4fa408c59118302": ChaCha20Key(
        "dc6bc1470c4ed9c1f2bec934da2dc1c93a76dc7ea44f48933cf782c28d194b4d", "a1b13559e7c0857f20d1e4c42d90a3da"
    ),
    # FGT_201E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "cd4fac91febb196214a8bdf901c96d668965e96ea6b72f6748f6c0c743c18d6d": ChaCha20Key(
        "917a7e4609098814fafaf584650a20882ba8f410270bca3a1109abaf66564740", "31bda82a8a2e55503176922e85f89c89"
    ),
    # FGT_201E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "1fd330914e8ba73c1744d4a43aedbbaf34197f5835fc90fb2e40a34e1370a0cd": ChaCha20Key(
        "cd7d283b5b499420418a2e8de194d5835d101e400b8cf4fa503add0c5851672a", "bdd99d43bb796f11f1eb442e7ad562b7"
    ),
    # FGT_201E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "ef5472dcf462091e0923c53aad9abbe767fc2e89726401f6c4449f0cc8ea78fa": ChaCha20Key(
        "1489a71aa2a615f9078ec3e6953cb41feb31534ebd218cc147c7713d13818a82", "32fa4ec0d9675ff7de112f6bf1a9b828"
    ),
    # FGT_201F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "0120ba74abbd6927dec1cd8a54b79e96d237e2e7b34be07ff1f84a229ad25355": ChaCha20Key(
        "5e19c5db3e8f5b51995f7868abbe1c8459f4460bb00091f0c0113fb571e34cea", "2c669a5e284115693748f6c45ee52086"
    ),
    # FGT_201F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "5f81e3f61bfaa9d22f0a07ed70ee4928db23b638d671803b664dc7dbd1251876": ChaCha20Key(
        "59b3a33b588868e7e2ca3c3f293b1c2dfb2d99123aabf6f142b675dd32f00d95", "20f802b45c6c10359a886390c5706f24"
    ),
    # FGT_201F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "b7c7d9f98263ab44ae3b19df7c7cb31d053a526cb3890ff102c979dcd20f9698": ChaCha20Key(
        "edce2d91f743c10ad529102bf8751331de50d15065e2a1231aecd3eff67ea4e3", "7fdf6df2a6020fc9d286a44bedffdd3c"
    ),
    # FGT_2200E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "e42595a5e407f88139499b8b6ae2c47d18652c8829ab75f2275d4b533986a064": ChaCha20Key(
        "8ca3e470f6c4c50f3361f48c4e451e109ee7381226cc99f38bc121d941e16a2b", "18c01e5417a2e1c3c3d935952e565c4b"
    ),
    # FGT_2200E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "e1ed0bff032734760b5025eeb8ce82474d71a34261392deb72faa9503137c4a5": ChaCha20Key(
        "fd4159d36581433ceb3a106765e376e214f2a323327cdb513fc07e5f548584a9", "75138b9a7c1e4bba5fab3c83120c2b0e"
    ),
    # FGT_2200E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "37a40c1c7edd646b9a6a4cc881bae9344fed58715a929cca595d98cc8a1b5e22": ChaCha20Key(
        "2cb113a043072da87eb2fe4b1ad37ad544c329964d80483a0f20e29bcee0ced4", "5c3328740aebefcab61ad1db36a02e67"
    ),
    # FGT_2201E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "d1a20bc8889dfd94df06c835dbbc3f816b7fc5f261b17bfafe69ae5fd4643778": ChaCha20Key(
        "4c41b6138e2589c2264c03a0431a4dfef91516013e749aa969e5080f75be9b61", "15007570158404888ea5b464d2be5214"
    ),
    # FGT_2201E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "c78a205796c5ff755b6eef0fd3d4997951c404a7e84ff43a20925ba3442957e5": ChaCha20Key(
        "e4704f9883f060611c19aabb66ce7fc60d4b3a04bec77088efb653b9575cb49e", "f83c64a388072e1aae728f1672be715c"
    ),
    # FGT_2201E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "46bfd46068f9eaccdad7cc6700aeef42b380ec76eb224cf46bd2d5ac0a10a9f9": ChaCha20Key(
        "53d46db6e27b0b174789bd496fa56e0989d0c492af8c6dee072653ec6db9e224", "8038a8d5f1ed90b9ff5cbfe6fd7a8331"
    ),
    # FGT_2500E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "3ae351994d411514495410c51f40d25c35eaeb4b505a6bf90244a1e976f046b3": ChaCha20Key(
        "2256e6fa01eb104eedab60e6a7013c4e1ef7cb2af08cc768dfaf3375a194d138", "53c2223ed2e1b94d1500f3f1154cd0a0"
    ),
    # FGT_2500E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "f696181f4a7f3da1bcb6ba36ab09845f93cc75245b4dd99c5866d06b3611b355": ChaCha20Key(
        "5060bca0f96a4f8eb5d31749d55a6b2cbd200fb4e9348f42a5aff55176a1ea9e", "351f79a7132b0097b6985110196c47e6"
    ),
    # FGT_2500E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "b54b4cc4aa9ba381007833ef2ad2d4061cfe729f3848674d53dd450053801fdd": ChaCha20Key(
        "e4bfb57cf1fd2c12f8594758986068328f2fa3b993fb7e24f5248a254ad5ff70", "f42654e3f89eca91c75d4a4bb2609a23"
    ),
    # FGT_2600F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "8481b0a456a54fc155b5e7a7e487c3d170e10ede86483e188bef008e442f9068": ChaCha20Key(
        "8138a0b8efd19994ec59bbd93fc20e9022521fd1a49f4c8b33691efbde4f6da5", "28d8ad843ea3dcd690d2a1e9f2e64e7b"
    ),
    # FGT_2600F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "8bcabb364a6933a3b02b12bb0c123aaf93069886f4f2581cd3298f88fa999c92": ChaCha20Key(
        "72613b8a156aa39e49436261b82c2b5a755f4250846e609e140cc1ba83d3d528", "42cddc61f19ce50b819dd82857473927"
    ),
    # FGT_2600F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "74ba61c49ebbef0c5f93ce0c2fae9a893edfc49fd860eeaace0a201a416b07f7": ChaCha20Key(
        "6a26e84eb0216e6383beb5c42fa491f8a07e0781868652808e9670a0ae31d654", "41c906ec3888b54a4a5223178b344978"
    ),
    # FGT_2601F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "e1042a3b9d448ddc93e34889a5d8f99521511edd725ed3c7dd663ef1c23908ee": ChaCha20Key(
        "8c948b29ea5e18ee9963e7622322f21810bba35a7fdc6369a9787573c27d367e", "7e20812b15fb88bb7b9174d07479a6dd"
    ),
    # FGT_2601F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "74d0e81b658b95d17c39da64226da4c761e7d2565000812740418c52dc66cc40": ChaCha20Key(
        "1a8fd81ce4a20b4566f3c7bbb315a277b9f9d23a831525de32731156a199434b", "a24c65a62aeae047e97cc8a9804215e9"
    ),
    # FGT_2601F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "9bcf806c2822319dd609fcdd6c724756d217643a46c5ce2474e10f918ef39091": ChaCha20Key(
        "cc3d16e706b8768910da106bc680558158c76c86468a7d05835717a91eb94b83", "2f9be80c5ff9d26ea5de8ec84e0ee087"
    ),
    # FGT_3000D-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "8439ad2d67f7b47b7ef1694777151e5a7bfac7c9fd7737ac0320f4c30c7b3b36": ChaCha20Key(
        "60f3276ff78eb7b0cd4dcba059e75d2f48f6e681ae63f8adfbc3e94295eb17fb", "a88bab9f7e7837f82455082798172649"
    ),
    # FGT_3000D-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "443d8a690c6b0ca74bdbca63f65716e0241b09a92a19101fa9e97eba12ad9b2b": ChaCha20Key(
        "10cea5c3cb617fc0df11ec0f19ddf6a4fc1d35bd2e4d570979732506dc1c7325", "1a85cc22b8d522c0e6782f66a1c75b7f"
    ),
    # FGT_3000D-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "cf04aa87530f2229a32b1756e08dcd94bf9fa689ba632ff38cf96db11470e77d": ChaCha20Key(
        "9b01446aa6169658de6284fe9dea3396f9acd7b6345c4258e22f86a028de9876", "c3c6b7e3dc1c3954369420baef179370"
    ),
    # FGT_3000F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "fe88aafe8d25a4c18bd1f139ed1e43536deace3f09561633f2f134b084ac9372": ChaCha20Key(
        "6061cfd62a0243b67df5c3cd26f58e0e23f1a6daa68a6cdf53467f8809e21df5", "8c88660d52db400aaa934c01c4053ea9"
    ),
    # FGT_3000F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "2f4d2a7952adc42b9bb70a61c4cd16b2a6621e8a05b530058bcd21b3964ee347": ChaCha20Key(
        "b2602e0dfaf1f7cc92a43a89f8550ad35533ae6f0ce3913ee80ac2d1be2462dc", "9ee50a40279c110b3c1c3eb2f8ae82bd"
    ),
    # FGT_3000F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "d9001172e8245abeaae9c275e8bf54e0a76bbcead8b3461608fd8b5bb1dfa8f6": ChaCha20Key(
        "8fe699d58c4e070538600bf99bb4f2bfa6097bfb9f74f6ba844e21ff63e236c1", "bdbacf5c75213b1069381f063c8554c1"
    ),
    # FGT_3001F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "4e70d500b62a787c52a27399dce94118afca18bc9cd4d866f92a6327ad30a009": ChaCha20Key(
        "4a26569030be303fd91cb4e0cdfba6d027335722f6b9f63f9637a3213b6c6460", "60933ee1f749d750f611d9e208cb4ab0"
    ),
    # FGT_3001F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "ada026af32dc91e7562e72db75f01e48d2e220f71296e54069e1398cb77ca44f": ChaCha20Key(
        "5096bca4205dc10e8fa2ea89f9a6838f9646d6d6bcae98c1e9ce2df8c796b918", "6636588d659d287debd44d59b0d9bceb"
    ),
    # FGT_3001F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "ef98ddad1af2fb83577e5402f5a847ed6acd784224b10cc58a7cfbc47b982067": ChaCha20Key(
        "e0b8eb3505d5c084fa2c8518ff0a520b960f6b85f92dbffc9f221217ee484f1e", "abf4ffaf5857cda588dac6cc48c99b19"
    ),
    # FGT_300E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "2de1020b092469bbdb2134924c55f9e25251241e613b044fd9fea4716e5c9f38": ChaCha20Key(
        "55161d815d62b35052807b6f3f2f602cc124769c46f31d4069b92636dbadab44", "7870a96420f26d26ea17bfd9901d52fb"
    ),
    # FGT_300E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "28c4c382da32321b61094fb1c628e629e47fb9632cca67e7336fbc9b5d216819": ChaCha20Key(
        "b5373e07ae7ac396575b02a056f57490060856f53b2186f1aacf1ff4e5e03f88", "857bf55099f4bb4794d864dd394896aa"
    ),
    # FGT_300E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "849594b71e2c38d40ac8be8192b6d00b4a9fa896aa132c8ef189ddfab6d41eff": ChaCha20Key(
        "092a69ed6763c2932f5db24fe62da00c0290e06fc8a76d3fff5f4c8833030e03", "5be98b14db18e9b7e877ef4c59f75c38"
    ),
    # FGT_301E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "933c6e9a0c299ac3123bf4e8f8fb20c28b9b1bbcd29c421055f4a1487f254e0f": ChaCha20Key(
        "ccd773099d0685398d8eaa6ecd4fbb8ffc4405d5088e01a92d2515723fcfb8e1", "d5cd0fd923f1cb29b76114597d1a7fb8"
    ),
    # FGT_301E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "00c55c5db504ded0df6ac79f8ddd0cd58e59d2c97892fb1092e97147a223e16f": ChaCha20Key(
        "c581f6f9f34540fb7af25aa84c0d22d9bd459edb2b408b761c78c1ee28716e8b", "6e4831034c6fe1306d83bbbb66dcfe40"
    ),
    # FGT_301E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "cc2c7e418626ac10f10316284d68c605e6890c0b5307baaaaca66c0a30ff599c": ChaCha20Key(
        "3b391023d6c17e859a5f12f71a7383440f5a4fd43b5e47695688015c8826bfcf", "3d74b3e346d22b80faad855b2c84f35a"
    ),
    # FGT_3100D-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "7f63b86dffef0eb21e7f021fdb784f06991d5ef50722b0896ddad1035ea8639e": ChaCha20Key(
        "e2ed1f56e387837ac14d8528ed05a7f5e23adb9f37252bbdb2b34ac015467671", "d831f029bf9695a5ddc06a346821a4a1"
    ),
    # FGT_3100D-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "dc3a17eff228d3bb1c8269e27efef4cdaf4c43c29897c8369841eaeb544c74fa": ChaCha20Key(
        "af0b574a68595b3c9e09fdee6c86b3718720ee960054c6e2f9e05ba911f0d1e8", "45b243412ba9ba77b5c9994ff8cb6151"
    ),
    # FGT_3100D-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "87386fe5c84f7f870570abb6a6004243c6516bc7afeeafbd9dfd09927d162201": ChaCha20Key(
        "3c6000a3f909999db8b8307f1703988464a3ce6b63efa8c4429544a8db009b22", "a44bd2ba35f92063cf06fd2c524561b8"
    ),
    # FGT_3200D-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "20cbfe0fa3498822dd03d4f19511368719d0edae02556ca238a09b1ef2940281": ChaCha20Key(
        "291eba56ace9cff8e70c621773d97e416095fc9598dd6ddad6d74420acc4dd39", "6c37828f9ac1c0cd75e23d87fc0fe77a"
    ),
    # FGT_3200D-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "ca1e78f09ddf7a74283c63d6ade9daf8b4d1da554d9fdddc624564fb0da0b153": ChaCha20Key(
        "8f5e7c89971cd916be546839d3ae292a7d2b3a89b8c1457c47baa4020868242c", "8eccb1409d1a84d978401eb15e09e5ab"
    ),
    # FGT_3200D-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "1416d35ae31ac20498db763fcd1a278939ed4a52e89ca390cb3b81f36d425875": ChaCha20Key(
        "5792a1ca03e6f947d5011b51abe13eb30c5ff5f530ccbb050b13123f2da9762d", "6511b9e6662fdd4e457a99e1a5e2fde0"
    ),
    # FGT_3300E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "febf89cb3799aae74582b2eef1d6561736ffe75352387c07888f445195ec7830": ChaCha20Key(
        "1d061eef4a41258b3a17e1ea330d4e18191fcdbb1e8322494da495a9dba52ef4", "6c3555450f6ed1bb9d14db9a75b9ad60"
    ),
    # FGT_3300E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "1945f79d9280a15fd52e23e5e20dd107a32efa5a1c7c7dbd7b40feccd74106c9": ChaCha20Key(
        "db98cc1ba0923ff57272eee9eafe32ca18c0f4d38c927429ae3d94dd04113f73", "52c33c7d3a945fd2be7a793fa089eb3f"
    ),
    # FGT_3300E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "5baef324ce2ede16fe8e23c00d5e7c5ce92a738068693e677f6df6afdf847189": ChaCha20Key(
        "8056f6e01aba785519b61c1e193544640b786e957377437440e8959945fb8d06", "6734050f065ebab3019c6387578cb31c"
    ),
    # FGT_3301E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "b001913a0f69014d84fcc54489572bab8be67dab376ddacad0d649d253e6296d": ChaCha20Key(
        "7f25ff8d8ac659d22244c4456d042e44a2ab8d7a3d14815aec166af0651676f7", "804392f10a61efc5b97fe1d00df86968"
    ),
    # FGT_3301E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "c088b44025774b67f41c87088955ce83c36a7135e18f8db10e6af239d6ff5575": ChaCha20Key(
        "d8abbd1198881d4dc619bd509f08a803bcdc5876aa63109ac5504e800d5702cf", "07492f7226c58559f918d8befa5618d8"
    ),
    # FGT_3301E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "1369ea8b708e00bc4f41e184ced5c48ba3c80bf62c6f878f742a084e30ee2a86": ChaCha20Key(
        "b9e2efd021efbe2d75f96b146fd8e9e2877bd42facaf83bb959ace1563ce838d", "1ea977f09517eac0ef90fd9b6dac9d50"
    ),
    # FGT_3400E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "5136ad4f1f1c9639ad6187a4eb12b36d6e9f3c6e8eedd37f5ce3d05726344e48": ChaCha20Key(
        "18c445c842485088e9f873e5dfb2393920e0e05d632142bc55bd428a06043ce8", "de0c4226cd447fa37f076e53184ddc8f"
    ),
    # FGT_3400E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "119cdf0dd98ea168900f25ae0c5c71155343204388918ea6e13bdb819e981957": ChaCha20Key(
        "23e8c92a429e9edcbd2b0c3e4dc7dcf05492f8464dc9ab8d61b528e312667fb4", "ded14e333c8edb34d5169803f7b3ad6c"
    ),
    # FGT_3400E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "3b9e05885b498acb4c115a9f5cf674338286c18de1c752a66df96ef7860c5929": ChaCha20Key(
        "41cafacbe7add8157d1bc65a9590ef0fb16ac48b68990d45f06f80bbcf499170", "60ed747d0623188fc83f756cc7ece690"
    ),
    # FGT_3401E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "27dbd7263425d1cd3decfac36d2eba02d33f8fe7ae15ab330b9fdbc4bfe01699": ChaCha20Key(
        "4787f8cf47b34a427d7e9176eea5da58910a83a8573898b87110533b39f0d62e", "203af5b809694e155ea0f44a66fbd462"
    ),
    # FGT_3401E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "e889ea0fa6218756b18991396de1226e12cffb2a62097dfc8b7a6f5f5b67fd9e": ChaCha20Key(
        "3fc3b21b630af03cc45dbd40d38e3c21ebe896178b8428493f5e86fda7b67c9d", "c08c3dc6a82962cb5542e39ef8997eed"
    ),
    # FGT_3401E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "83a4af30cc92ea7f0803e7ba9e940171188826f0f457914679fe3eccdd89ccb0": ChaCha20Key(
        "4339df1054ab036dc25e60de671ea32bf511c72ebe97d433f8a7e236e3ee1945", "01e9d4aa24d4bb8026cdf38095fb40e6"
    ),
    # FGT_3500F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "bebb21ef735285c91864ba7c2378c4cdcc2ca1bbaf8d9814d6aebe4b79e54025": ChaCha20Key(
        "061e279bb7574e2096491b05aea3fea94845e5255bff8926954c7e746c194f8e", "e90b64ea2cfda288a316120a86efae8b"
    ),
    # FGT_3500F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "0491f0e40d9d9088766106d7f3b4eea5a17bb0ae578b99954e06a22e33ac79c3": ChaCha20Key(
        "c2cf962844d8adfc9029505e9521dba0ccd990cc3401798bb07ddb09d9322ec2", "76af2ed07f28b83394eb4b5c8d7be04b"
    ),
    # FGT_3500F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "51d7490606c81b5169b3f05696ab742242b3ab8812a131a7657f1cd446918b6b": ChaCha20Key(
        "8dcf0342b1bf79b3b2c2ca7a760bfa21b73d3bffedd1ed3ec76910a55d36e352", "a317ca2ec7b2ed2ea666cdaa719d4d3f"
    ),
    # FGT_3501F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "c433fabfc4405b6933f12ba007418460f50a1734814ce9f07b5efb35a300129b": ChaCha20Key(
        "0dc55c377f0a1cf7233e0199cb6201283cca3d09d5eea705dc7c1725588755bb", "489a812114b0076c9d3d39e85cd8bf0e"
    ),
    # FGT_3501F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "ddb3c1b111453ac2772a736556584a4c1c53dbc3a9ca84684700c6673e0c0786": ChaCha20Key(
        "30faef60f0946761113bc7ca41f87f79182289ff6a486c8531a8e03eef7fc1ef", "1418276b14c4afc696a00846835739e0"
    ),
    # FGT_3501F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "9317d287d64e9856eb175620eba7cf275a671468ecbc40b63982ac3e6d7f56fa": ChaCha20Key(
        "c569e37c4e6a241582e028cb39aacb1ab0cfbcd22a72e2a75eb78c0d55331d96", "a46a54950308591012b9aa6ba2a06a23"
    ),
    # FGT_3600E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "42336502a4223dac2b50201853c7c6f9958397897f6f803752ec776cc8257448": ChaCha20Key(
        "707c2fcbe1e5040aac2896c5cef1b2d593755dab0015042a0e6ab8ae674576b3", "9d5f7185b0624067e29f5abe2ee4f7a1"
    ),
    # FGT_3600E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "7848fd0a6541ec6992f5bec281bbe1e84514b88cfb5ffcea94e63271b86c3363": ChaCha20Key(
        "108ee9f4ce935c7e9ffe17e1e2dd89806082f2616c28046a99539af2209c06b4", "7e85a604460ed7684e913f364d83c4ff"
    ),
    # FGT_3600E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "dd84fd0f605a81a27f00999757a5e7413af8ee277973b4479cfb0ce9d6080fbc": ChaCha20Key(
        "a9e1025d4e438c5caeaf969ca649530f8784c9adb1c0f3776880ab013a68c8e3", "6c9b91d46a6572abe75186e454c8b6b0"
    ),
    # FGT_3601E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "f41db78601669d109bb84921cc4962b2398d751ff240267ecfcbc719190f57d1": ChaCha20Key(
        "502c3f08c9ccf3e8eb5a5633ae02aa2507d37e562462e3cd749dfee0fcf941d6", "62f00fcf2848810b20275aa3de87ec0d"
    ),
    # FGT_3601E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "875485520769a4ffbf5a7d031ed0e8e1e763b6c79b05ca4c5148f83ee95f47cc": ChaCha20Key(
        "4fed99de98679372366708553cf929441597db5dd9903b3f2164f62d1920262d", "23649218fdf14c7d8e558a4cee21c8b7"
    ),
    # FGT_3601E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "b9dced15857143ecaa04e05b375c3d08f8014d65daf4531931d3b763353e2ff6": ChaCha20Key(
        "f3ebda451c3214a2a20e328e9cb52d5c4c1ce002b6a855ec6ee8c6466e77b20d", "782aaebc4af980fe6e37965a2308bac3"
    ),
    # FGT_3700D-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "6c76cdada2bf4e3eabbc30ab96c8b756b4e9d42feba5a3e72717bb35192508fc": ChaCha20Key(
        "de98908f7c66511f18bc317c6fb42cb35b52539252ec3b1842a98f0f3695ba85", "505a037978972c4e5a694a53bcfaf2ca"
    ),
    # FGT_3700D-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "361e4df2d40d2c19da65bb2677e580ebd086e4ecb417d43fc0df39c8aa6ee255": ChaCha20Key(
        "bda635a34f02b5cd566dac245103786b1a979019f8acf792bb498feaa3baa844", "b31c00ef197677e2bfe9f9537babdf8f"
    ),
    # FGT_3700D-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "db77685440a66c0bcc7a895a7179aac156d92221b90f32a5b7505f460ebf365a": ChaCha20Key(
        "12fac7ad487c00720979ba716db38bfcfa956c421c6309da871a533d2e797888", "94fae278d1fd2e7f1c80fb973ae40d6d"
    ),
    # FGT_3800D-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "d23d91f3314eda68b8cc07ca34e7d6809cfe0991530b616e9301372b6db57ebe": ChaCha20Key(
        "4707f370c74ec0ac7956d056037fcf48ee5552f96b2426d49c571d0c2d748e33", "898ccc3528a187a3b8e0e8c7a7dd5492"
    ),
    # FGT_3800D-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "136826365c37f9cb84332161e78cc73d839836ad12457b9775ee803a807b27eb": ChaCha20Key(
        "97b89a25df7e90e9e633983aff618c17ef57a649ef87072ae2e49c0abcbc0fa3", "c3ef20e80824ac7aeb18625bec670e3c"
    ),
    # FGT_3800D-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "9f7a2acfbcc8141ab448de8c291718de7f942c871200f4705922d6140efd07e3": ChaCha20Key(
        "486a473ec8820e849257427818d19c72b1f227fe71bed319ac6eabcf779d3e52", "e514b5a4f78b05f2bae09364ba7580ed"
    ),
    # FGT_3960E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "85386029172dcbf816ae635a3a51e2f42594cd4395625ecf8800f274c526c1d5": ChaCha20Key(
        "a0361002a6e7ca19f43252d3687f5807927577cae30c398bc270915520c95a82", "5a58749b3358f12e1243ef14c02137cd"
    ),
    # FGT_3960E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "fb1b2b6a6ad4aa8984ac312cf08f2678fcbceab9fae386a9349952d6ac1621d6": ChaCha20Key(
        "c6eed3bbbea3eef30c18d5245587519b614a97fc04e51b6876a4dba4b5b0e0ef", "cde4e160cba16732db294aec1d0f60f9"
    ),
    # FGT_3960E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "77c71b91a1cd1be01f40dd4a852df2132e457fa693ee8f399887b12e7ecc4890": ChaCha20Key(
        "9aea67e26d9c9a40b0fd347e8afcca49cbfa9fe8bc3db3542c850f2bd76b475b", "839a1d7ba705e6233049b311ba0ac762"
    ),
    # FGT_400E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "f1ae08932d90063b35077f5b8bddcf49603389252a93caa937948d2243de6ee3": ChaCha20Key(
        "0577937829e64eb9bdf87560c0351a737db09c4c46a423e266cb36ee14452579", "c9b7fef1800ff1455a73b7aae08248e6"
    ),
    # FGT_400E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "b423e2b3635ad634f265aad9b597a7cb1bdacbf05f6bd89f77274dd100d3e698": ChaCha20Key(
        "7ff5cc06e15e0449bf3bdaf99f10d4b0c959fa4c7ebe9ad73687702826933de8", "e854d3e0a094af5cd02e7df55672c5c0"
    ),
    # FGT_400E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "7d71d9703d5e15eb12074dd5b16b1740c00a133deee476cdba35e017a5e9fcfc": ChaCha20Key(
        "60641a98256cec6a3c7ca02f1c93f37454d1670a1929e66b6d4918a89ee78460", "0ff6bed528e36a52437b6dd250ce690e"
    ),
    # FGT_400F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "bd22a7ef3c02cf07ac09034d9840bdaf5cca5dd8d63818e85bc3b82c4e4ceea4": ChaCha20Key(
        "0d1528941cc4f9dfd4b9cee723e8fb991493ef9804a99acc3f1a743ea8a566c2", "882b57a41bf74a71d192d443f5f7649a"
    ),
    # FGT_400F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "d97b5678a494f27606ff9ffb1694d30148bf965079bccbe79963433cf9c6d5bc": ChaCha20Key(
        "85c2c616beb93eb1c2f49863a5ef68954329ad624ec502d38c90ea3959207823", "74da701c97454cc06728358e6e21e1a5"
    ),
    # FGT_400F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "d50a39d5ab13a8dbb4c78b733077c7d5906431ce00e32038cc2a6c267567a6d1": ChaCha20Key(
        "bf45cfbd86db613f36037bffe19317029ce1bc6d371a4f58d6dd927a57018d7d", "80c301f3bfbe132292a370a90cd8ad64"
    ),
    # FGT_401E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "cd070188bec475ec9a03e778df006324d79652ebc6418222f4fb5c3a16d761b8": ChaCha20Key(
        "7fe089d3b8aba5f18cec192a8f64ec9f7a5c0eb49d677d263e785f7bdd76a561", "17648619cff6f65a913cdf3aaafacd0c"
    ),
    # FGT_401E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "825b44c159081c0115de7364baf9dfce49f7c0ef5af7150891d3b958ea838849": ChaCha20Key(
        "cb85b4a136690a8601cc26a2f7e1a773682656a4c6fffd1266f3fe0dc452d909", "ce8ad8672225ffe42009d349bac0eb52"
    ),
    # FGT_401E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "e4d630132f3b491eab91b7cd7b8e59ccfc4f4321786ae26dc138161f5cc80424": ChaCha20Key(
        "68192a981ca7978ce3b91a05d635d034477ae3ba5d9827919597e3f22e46d12f", "c5f1b94679cd6e5e86822743eab7b6f9"
    ),
    # FGT_401F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "6ad2525572bba65b681734dca15bf17bdecc3ff7eff7e2402e4fd67da4b30cc1": ChaCha20Key(
        "c835a540c87ed1c6d231660e5fec7cc4cfed4f85f19d439232c169c74a72349d", "cfeedc758a72e2901e56d9aaee617650"
    ),
    # FGT_401F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "1e7f83f668a1aaff30a7b0955ebbfe1b6c20cee9e0a6fe396f36b634cd1a0b90": ChaCha20Key(
        "e388af9c1c2403577a76fb943431e2f3af0e967aa7d957179da50ebc180c27d8", "34fbe7930dd8c7b8e19484464a77d705"
    ),
    # FGT_401F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "00d21fb21066ad0d2ac9dc1dba421fcad1de251ed80c78fc99de65ce97e46a49": ChaCha20Key(
        "cba72164a5768b1126202a5ce91adddd5e7f9a8936fecbf8935e6bbdd2eb7efc", "9d46998fc222b4020e0b7395718aabbf"
    ),
    # FGT_40F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "2f3e4fce6f71fdbdee02d0cf331bfbd2ad813e895aca1c153bab084d9da785b2": ChaCha20Key(
        "1e8cd6e53423d3e9597329c9149f95f5b4864fdd726d9fba6a0c7f4cddd924df", "d8f1bad8cabf4e3600fc78bb121119bc"
    ),
    # FGT_40F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "595d478079f1c78bbb20aaf70996de8f02f5dd4e9b2cf722e4bc122887cb55e3": ChaCha20Key(
        "3c6c1db15d65c0ec58fdb6c51f2d835f273192774103894de2f4244815908742", "767b5cdae927fadc64b63c592d726318"
    ),
    # FGT_40F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "d10a4af00b33b0a99d51c76946f813c54ae0e2600d79d849747fa76baa31b8a4": ChaCha20Key(
        "beb60f0c2bcd5c978cc57ab509e82c9382be0be4a8c85b5bf48ca2b1b44f0f5c", "2050da7cc6091d98452f7b6ea32e1f41"
    ),
    # FGT_40F_3G4G-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "d6027a9343eefd3a4d854d0e822334a7f7861a8362328d7ffdb5fb067e438ccb": ChaCha20Key(
        "8128adae3e90d99f7b33b3df6dc32ef0dcacd5ba053b6d5384113059424dda0f", "422614c284cfc40cf85aeb4532c30e7f"
    ),
    # FGT_40F_3G4G-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "2d6149944a807f40b5c7139e5902a8b5c81a9c9ae76f3b60b8535cdc681684c4": ChaCha20Key(
        "2ed9df653861818b3247af69ba3d11906772f8989d47e22d9e9e162e51a95c6d", "4acff4798040b2f8c6a78cfb67080d47"
    ),
    # FGT_40F_3G4G-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "0d0c9d5fadacf45ecfde7a15d92a7bd6b39c8b97f7a7a76619ec5447ade8fb61": ChaCha20Key(
        "bc402ba4780659195b9dfc8bd624d779b2644bff7c64d07a123c370a7251e5f2", "4a960caf5de8e5466319d10058b93b59"
    ),
    # FGT_4200F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "6b2bef1bd55ea7578ca9e5eaa4acd4342fb258dd7ffb72b630f59b0a86621267": ChaCha20Key(
        "4068bc24500a687d415993d67300322fe4757ae2cafb9ee929b7204673bdf74e", "b5f71e62f97ce73fcbe9e563d83656dc"
    ),
    # FGT_4200F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "f8643f50bb12460422e66983907fd816f68a3aebb3de79262024248f4cf373fe": ChaCha20Key(
        "fd88b98c62d45d035a85b7e52a69c9ed3fce8e4b114b542fe79bfbe6a7ed9eb7", "a0b6aeeab372f156820ee1c1564fe040"
    ),
    # FGT_4200F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "ce49504ce4aaa330edf990d4b74d23396c0395f53b0838e138cf5de05f52afb8": ChaCha20Key(
        "e35097da0643da3a750623166c7e3efa635e109d0a6acfbe1443c173efc0956e", "69eec49d150dd28f6d0465fcd87c8d1a"
    ),
    # FGT_4201F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "3c442d6afdf336d8c022d6ed31829f81243775b36ccb3405909818dabb981b32": ChaCha20Key(
        "9928ea5177b59c3441fd6057c72178562efa40eaaeabfb8f695a52255b11f69a", "b78fa44c5b9d705110fa7914afc67b76"
    ),
    # FGT_4201F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "ab9a1a5aa43c31a4ea2ab366767a01231ad53f91809217f4539adaa8e7b3775f": ChaCha20Key(
        "89cd3a171dd0ee98a4ddc5bd82482e23bc04737c738dabbec91477d7f2604473", "74f145f53a6d90b6e489a16df4d49f78"
    ),
    # FGT_4201F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "5e4663d684ff314f22d1da3010efe60df04fc7f176312da2775498b77c90ab57": ChaCha20Key(
        "a8c21d6e21379670a9ea4c2bea0a23c88f2da184f55a237c8712d20d445630dc", "73a279d24c211b32c6bab0bbf85af275"
    ),
    # FGT_4400F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "3cd82378813fe59c636510eb993c742a574b30dcf4f88061b9ee7c7aef4e1668": ChaCha20Key(
        "111e53e9ee100ab54492c08919f9c0230b980248981e28da74021f323aa52cab", "81b3d8038731d9a535f443337f085c14"
    ),
    # FGT_4400F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "e3dbd0f75d79947fce75637469468d241886e8c1afa5dfedac6fec58b0ed23ae": ChaCha20Key(
        "32f9ec7a8dace909f2459904d52a4a66dfe0dde9a107c352be423f975064d93d", "be05fb3076d76c7d8cbbd9329c1332a7"
    ),
    # FGT_4400F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "954ab82d39a0ea8953ca3b8f2be04d4a81c0a8f44d51cc174320333e6fd735c2": ChaCha20Key(
        "1c7881ce47d2fcdebddac1f3adc569d80c6f5f8a58467f5b36b015fe268eff09", "5bd013dc513e09f8e075ff7efc43f7eb"
    ),
    # FGT_4401F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "6c45941b226e60d4315be7d9c15c5e84f9a60a4cb27d4b2dfa12419fadaf5c7e": ChaCha20Key(
        "9f5319ef6b176f7c13c5891e6d10fc71fcb7ed21f493822c24c5ce21794431c6", "2af2bdfb34a10a96b60d0ca8981189ee"
    ),
    # FGT_4401F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "ca6983b3c1174be08f299a8639cd7547955f92aa7cc971777b5d75d510d382fc": ChaCha20Key(
        "0f9282a30952fe5f2fe240909278f0e18b1c9841d8891f80216eaeaf992264ec", "abb574928bcce6bbb123d197dd6a4d47"
    ),
    # FGT_4401F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "0e22193d6b3bc49dbbf2acf2e998ac56f1041f912394f87fc52e6db938ba2543": ChaCha20Key(
        "6e392a596392c301fc573cf852f296876d51f4a3682a8d560e8f072e746aaa6e", "937c810896fe1ce1b5ea8e378735143e"
    ),
    # FGT_5001E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "3f4cab5ee02412070335c77774aa5ea340196583d008a396c54d43554f215253": ChaCha20Key(
        "f6e06de73e03ed956a2dc9f5e14be1ea8df041c95a77e68de4b3789833602a81", "f82d91d580f16cbaa1ba464c8eba28ab"
    ),
    # FGT_5001E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "0148644a27198a3b960782605f8a04efe591bc5d1ec44d653411195877dfe1f1": ChaCha20Key(
        "74e392264f239e93c0783127f49807fa1f1f4517b8b588f3c2d45cf5367d2539", "fea21c7a14457a2203b2ac9223d284aa"
    ),
    # FGT_5001E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "823e008629ff89482f65db109e8353e9a82e392ef4d0c9a3266953d96896ea98": ChaCha20Key(
        "3acd8e3fbfb36a48871a3a2c1ce66181fc6f5281067a65171276b1919eb25e35", "a928c3715de27147ba0a6dde34064801"
    ),
    # FGT_500E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "543ebc9d56dd866683067ef3c8eedc1d40c78db902c586597fcb575833a8cd04": ChaCha20Key(
        "c33351f067879f622c7c819e109d8c97acb317591639729aece7a085f597862a", "bef00dccc7b0a7bb3b24f4d963f11d49"
    ),
    # FGT_500E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "5e288ddd9b93c464f6a3024c1b43a1609204357b163f670f02a75e8f5b2a65b0": ChaCha20Key(
        "f886a6452ffb86090d5cc0affe66b8d63ec76d80ab1721c944d7a18c466efd3d", "d7bfef5424b6dd9463db6e38b6071273"
    ),
    # FGT_500E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "2fa55993352ca20567483f4a391feca44692dea93d10e55c07e3c7b5e4ca04c8": ChaCha20Key(
        "f22e31e905cb3aa0a074de96bb0f3f67597337e2812ab72c2efe86e649b75d4c", "ee9a9607810562ab088a5af5fa110f81"
    ),
    # FGT_501E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "79920820959c20b04b95e0bcc4b5cd1294ae3209dbc423933266bd66160a9984": ChaCha20Key(
        "4bc20a65df349727c86bc04201e5f36d8a078bc8be36d0eaa2cb6590e01552d1", "99d91c657019bb98d815bff0c62b4f6c"
    ),
    # FGT_501E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "e9bab1e7c319a9962dd4ecad2590cbb933ac78118ae7c7b6aaf3cb120b32bfdd": ChaCha20Key(
        "fe29caa9b2c23ed518f6ec18005af644792ec5fe50557a90dd832f7fad646f82", "9d5bb389b02bdc4e9c99054b8b144002"
    ),
    # FGT_501E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "3cb36fa0d216d6c2e576a84f1b8832b6e34f1e309dc0bbd6d189804542e689f3": ChaCha20Key(
        "6a1a38119939a2219924a56022f29ee97556a1f7a04584f4e2f7f29e05073d7f", "eec31642dfc22bcf6787520789ac49a3"
    ),
    # FGT_600E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "f82cbeb68ccb5905567ace0d3ccb8905bf6ef0a73a6d355eab5e8a7ad1e3e533": ChaCha20Key(
        "b7324343893bd2162f6aac24aebedbff2fb6496002448de4264b782d5c7db7cb", "20617d305a09c924e4dacaca2e1178cb"
    ),
    # FGT_600E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "caf09b387bd274bec3e74ebef4275a2090fe1af382d5ddc86e6f43a26a29b798": ChaCha20Key(
        "8eea2c022acd50ebf61714367da19acc2e0c25ed334bb73d281f6add3916d717", "ef16e07613cd40bba6127d411c27f11e"
    ),
    # FGT_600E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "481869716258e0131965c29d57c42ebb3c69ae2fede98b3db8d1e83cc60123e6": ChaCha20Key(
        "61d50a43bc0e3f21f828da7b3aaad682200340a5bbd0034011c0d1b89ef2a5cc", "717a0c173ddcb2b66918d73010c7fd60"
    ),
    # FGT_600F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "19c13a0d8a6640cccee57fa4b5ad341a560d57c896ab6a2b7b3e61543e079af2": ChaCha20Key(
        "b6139533412906f6409922286984a071ebb6d2eb041b5e28bdb1cd2571f86fc1", "ad1d3c277aece94ea6cf832822185337"
    ),
    # FGT_600F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "6d5af37771aaacec7e739527938cfccfe96a1a94476d5b62f471d472c2c7cdd9": ChaCha20Key(
        "de9b94a6725bf445bd2709ceae2b47ff7e494d9e7bb004a3768d1fae88e4603e", "68fb0475bb6e577092ecd7020fc1a18e"
    ),
    # FGT_600F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "d85dbfdf56b992c118585e875677fed8a466ebe3d48ae760f6e2624bc07d7f69": ChaCha20Key(
        "bde922c0272f99a83e0df5f2f5a5365d97d797814808df45232097446c27d7bb", "22c490db0a39dc98a1e41c6ab8e5e5cb"
    ),
    # FGT_601E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "fcadfaa202d1c1e1d7f376de8011563b5bf57eff49ce6bac0b664ee5ec8704f6": ChaCha20Key(
        "939a6f1b0e80122e56438def3c1b3f3ce427f0bea2e49ca0219ae632c6dbf601", "8e26a12c6ded9f4673c1fd1df038c33d"
    ),
    # FGT_601E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "bfc640f6368dcdc93c15a02b2b2e5e17a65482f8f04335759beecc8b1b085312": ChaCha20Key(
        "48648058123ecd5fb2937960c9cd663be961763fb4545a13be8278e6678562a5", "bd7f0a911ea90cdda3922dcb264fdf4d"
    ),
    # FGT_601E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "59c8c24a664891348f47e65b2da64ceea16f0c037c2ff32b6a3a1857f0c613fa": ChaCha20Key(
        "f24b67b449469c736d200326a269b5f8727254c9d491c00af953437906c90f72", "5cd3ce08b9281817cc059e8dcfba74a1"
    ),
    # FGT_601F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "91d4e5ee40a877ab28307ac43d8cb2f9cc07bedfbd2c2c7e30a0dda3b3b71993": ChaCha20Key(
        "3328149df0e2f218f32214d3b7c2b77a5e9839597f70edc489b65f22dcdce9f1", "0fdbb178d3416b8b90b4645a20ad4645"
    ),
    # FGT_601F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "ade73e10fe54ff203e0566bb0c281a7dbdd5cfd4635de1674c27b6bedbe45ada": ChaCha20Key(
        "3111aa569dc2db763cb29038ecc424000c9a955e631bbabe905206adc6a47062", "4eb3f699d4c07dc895be4b3e35f44069"
    ),
    # FGT_601F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "2b5ddbb913b61692866db620d7e6a612a00e9c5719b0709bbd712b050a642da0": ChaCha20Key(
        "072aecec18189661707f1d981a629c33c703552e90c786a26ea5185be9ad1430", "147649beb2fa0433811397d8272b102e"
    ),
    # FGT_60E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "651c52d6225a68f729ddb83faccd28656b5710bf059743545e8a741e0d7b04a3": ChaCha20Key(
        "fbe32b6d0f823291c44b0154a48d43b458da2ad704b74a643a9f973874b07df9", "03eed70d5d5dee59e726655f27e62bc8"
    ),
    # FGT_60E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "c03e654cecb02485dea30de86b7cdba1a7568e82c4e57294a9543c41dcbd0ee9": ChaCha20Key(
        "0c6a4e32b128949f976d1caa2faf057499052d0bb2e14dc63e5417bc0199ab3a", "7ab30ccb9067a102f5b801729b58e200"
    ),
    # FGT_60E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "7d2cd5728b74ece2bbd8133cee1750d53ec243fe2b21db8c020d98f30a44a5bc": ChaCha20Key(
        "86941563b876833cb80490558f3834295771c4241ea98f5288a565a1ca857e16", "6abdee0d38912533a80eabfc4cc978c6"
    ),
    # FGT_60E_DSL-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "f767e6466af474b2c0f1e1296f11cc5e8bc88faa6e2eab42aa9eb7b4ca3b87ee": ChaCha20Key(
        "ac357598760dbd5ad9ed992abdc33808cb3750b01d1f05f22fdd9dc639837a94", "6b3b94a1bbb25f1d20e7971c1971e157"
    ),
    # FGT_60E_DSL-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "80d4bf247249f4a56d141b75ee1731c533d576993771c652fd585901a5cde184": ChaCha20Key(
        "0b3a93bf03a36eaca1292e050f1916499fb423821f962eae8865f3356ded3d19", "fc1c2fe12ac81228b99ee63a5e88f967"
    ),
    # FGT_60E_DSL-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "dee0a8cb94cc30ad6a9de45b4a5058713c4161d4d41b37a893c25e3848f507c4": ChaCha20Key(
        "1aec0c4ea0b09760a7b4349661933d2f064c024302f817ced0d1c0436871824a", "efae3223129a6219f6e7cf8aa3b2a5ee"
    ),
    # FGT_60E_DSLJ-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "f16f0c89fac3f3267e4576f0afb5911b8128284deb5b6772990fd7f20be40115": ChaCha20Key(
        "b67e9a96015e8185891bae4ed07dd9fb2c16f681f7daedf2d663849cdca95f04", "77b3baa07b0c2d4b3fd21f4188d51fb8"
    ),
    # FGT_60E_DSLJ-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "ca559c4ae92d1492ad4c9adb965253545d0df34896a61a117e78f5a7ad90768f": ChaCha20Key(
        "8e007f36c7bed6214f99c91c8f5616e59112cbb39f027d66b81138d0a0c6188f", "282c162460812756e099190638167c75"
    ),
    # FGT_60E_DSLJ-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "9a432a44cb7651d1ee1bcbb86303bc2a0794602690dbb30ce798f27ae04ccdb8": ChaCha20Key(
        "ae62466dbde8e70adbd8a53cd6629878c470d39242e0743f3d0327d32c9a8232", "d1851ad7e9bdbdc7860e5485ad8798f5"
    ),
    # FGT_60E_POE-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "4004cf7f99b00877dd5eeefc498905ebec048fca47b4596f5abbd9849884cdc0": ChaCha20Key(
        "bfa531ae977914892dd7af5b56560f135d654a4e5c33d94eb06c7cade0c1d152", "35be2f9181d4f0611c273b1b2723a40c"
    ),
    # FGT_60E_POE-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "af5e5e7f9b35bb62bcae36566f3957653ec409adb496861a85595b00423f5130": ChaCha20Key(
        "e283587848716d7d8b256c4886f56d48551b9918f339174eb7bc6ca391ae3f1d", "4d38e8a0b92a46976f8f9ce03ca8b0c7"
    ),
    # FGT_60E_POE-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "0ee2e4698cdf616fd4abffffd1f7a5c3019ab15f5c9cfa854d90363f121a175b": ChaCha20Key(
        "daa67ca53fbda987016f554e955babe6f5ceeff71166feb2a21af7ccf44d0e83", "c6bf808a46fcfd4556657dc06019582b"
    ),
    # FGT_60F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "2268b572f94c70d395283c37533ad6047548fb26b6a19f9903dcef0054b47f85": ChaCha20Key(
        "61e37d69b034b39ba9f466cd9dd4ac054fb51c5c5cb7cfa1380f874bbf7d1e01", "487a10653fe93faaf319ed44c9811073"
    ),
    # FGT_60F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "71eac26e3b0888c3fb457f54e373177f6b585e8005d84cb3abea73a0e89912cd": ChaCha20Key(
        "5678f594da6fecf6b7234b3e0bec3b71e51063689c7d2fafd75fc55530861513", "6dbed7815794e97f64095fe9dcb54379"
    ),
    # FGT_60F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "9dd56208c1f3f9f01c66927db1636e564d49184c7dc0abd543751e320ef61ef9": ChaCha20Key(
        "9a716c2cfe43de919ee8e9e302de527507064f75bd2faf95b491bf05d8baf127", "ad55369cfddfa0564a0a7c52219ff58c"
    ),
    # FGT_61E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "5c125e6c5e9a6ef69be23aff6e18bfaf19fc1042db8f420742b2db4dbb769da9": ChaCha20Key(
        "dc9bc7288848fe63e044857beeada799dca9ad377b12016487e86eb93b7a2301", "ff8b8b6fd3e423e39890ee39015996bd"
    ),
    # FGT_61E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "963d7f9e5319990c351ebd346e10d9d77183a6033697bab2366135c04bb49ad0": ChaCha20Key(
        "990d27cc49c1c82211e21a7ab4b516bf7d530de16bfc9451854cdd17ec432d75", "9e1e6576c8990f9c4d2b5afb15d1b3c0"
    ),
    # FGT_61E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "bd10c97a0acf4f4669a43b549c698a9f25c0e83ca3259fdf1ada47c15221d5bb": ChaCha20Key(
        "ba992771a29b7e50980d9c904e3c1545d8765cc3aa978a94d09bdf379e08a192", "a308fec2596ddebcebfd9d9bccde16dc"
    ),
    # FGT_61F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "dce495c14b43c3db25ee6891d8de7bec7497d563b315df120ecd42092b2f38a5": ChaCha20Key(
        "fa43b3620f639ec9c6f89795ce42f34967faa678c4e7082b8f11e0e414dbaa18", "938bdd0bc001cb3db8b3748c4c9cd058"
    ),
    # FGT_61F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "5657d111711dd402b4d7f7cac40fc925d053cdb1d8b56074edb15de903e75612": ChaCha20Key(
        "2634f8780867806c4333b1cf7c3d1dc3cafa07cbc8823e668f609f518fabb9eb", "3ac717eeca92037fd809a41263dbbf25"
    ),
    # FGT_61F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "af6b7ea8c0604d99089e7ea8de3bffc7fc28d85b0812d0cce9cc5e4433228587": ChaCha20Key(
        "b2404a182abab9acb3c71daf3fd4cd3d2803376e209c552fa83d814d36f599ea", "14c9289eca6b618aac70fe9287f0fdd1"
    ),
    # FGT_70F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "ceb3eea23127dc3205b2dd3b36e1cb8800395c0bd03ce4e31efe455e829ec729": ChaCha20Key(
        "49e9ce645632a549141a7659bdb4bb5bb646e9f96633bdd56198736256b79eb5", "7cf82ae2b451341dc33363c0bc1af1a0"
    ),
    # FGT_70F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "21781d38e4303b2842759104b7e11cce7ba0fbcbca08bf68ed1144d09c8486a3": ChaCha20Key(
        "030d5b85ea015c532163fea2d5cb1f968547bde3fe7c990b73f5f23a0dde6cd1", "11a220656e355d1b618178df679c93ff"
    ),
    # FGT_70F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "fb1b5340080398f9891f2fb238840a3f927c351e2a5f03fdcd1ddfda2b1a8c08": ChaCha20Key(
        "1eadb7b172d3544daa4487095b29a4758d18e5d040fed2f5931bd95be37b9497", "80722514137532b9842e0b43cbe8389f"
    ),
    # FGT_71F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "608998e2733eb59df9634d52835876bc229807c4b8dc0ae2bbd1d1168628cbee": ChaCha20Key(
        "d0664aa8083ca6b1060d648ddd14095343ef1a30db15608b245a35934b7de502", "2712f68b439cdd260ef72e1a0e67d1cc"
    ),
    # FGT_71F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "2d83c1600103258721f1c3a03dad294bf7cf91bd5322328fccd8647c4f898b78": ChaCha20Key(
        "a471e7c25e58672b28d8f796942b9e07f9896d6907442762bf22c93dd9f33f3f", "5ea0c946d6a1cb27a40f36020059df31"
    ),
    # FGT_71F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "492f8e5dc017f245082419214264a513f996dbee187156cc6e63861f43300d2a": ChaCha20Key(
        "efe070dbc64c4ae9a3c64c6408207efc1ce90b6b76940ae25e9e8bd5a700f3e9", "188ed5470db7ad8316539fc5bf85090e"
    ),
    # FGT_800D-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "4a455b8e843427521546a7bcb53f82d1477181ff20ab8aea97d7631ad84d45e9": ChaCha20Key(
        "2a174d90539054e3765a02a6bf82370d7f8c77a5e863f3c5fa2932f49b6e477d", "131c4a892eba62f14e2b16ee492e36fd"
    ),
    # FGT_800D-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "b077ccc93c2efee7a17e79ada9ff2621d4b9018dd325ee8389182b8dad7bc648": ChaCha20Key(
        "9a3c71925c39e691aa15834fd58f04fe786ea40f617708ce6527bb3968e888eb", "6106ca86894f533879e0c0f3d9798dbe"
    ),
    # FGT_800D-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "9b12dbf3fc4a2d821197733492b50b09139278d4edf3c35c06a3e88cde61abea": ChaCha20Key(
        "023696afae6a607a0a7ea3148d7f6538a6c05f56959b939251892802b0dc5f50", "1f4f30cbeb134f1b6ab9d9f79a87b10e"
    ),
    # FGT_80E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "68c1ef3b34062ebe953c5f1bd4bf67312a9598dec21fd182ac5d3bc2127df7ab": ChaCha20Key(
        "4bbbb0a4e4e9bcddde13699b35b880bc42ba49bede58abd22c2f343e10147111", "8e94f6cbbf93014b0dca8bcb455d4f3a"
    ),
    # FGT_80E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "1784b6ddc1b6a473cc21b6937b853db2065fe2260cec17a548082774e2d34ba4": ChaCha20Key(
        "979e5220beed72adf94835ce18dbad4cdcf44d84403af069daca3fe43cbd80cb", "5b12686ff8f23d86466760bbd9c91d58"
    ),
    # FGT_80E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "28c7f48d5c51a99370ef99c4aa0ccaa49b1f3dc4fa1c59253fcd66c571dfd774": ChaCha20Key(
        "522b6f95ca2e40d4a93000979b4c7a5578889a59effe756a5dfc5d02261d25a3", "1eb7c703f697ced7f4539cf8a6a13dae"
    ),
    # FGT_80E_POE-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "82fc27aabb01a2832bc10ab77933ff909671c7bcf73c80540e0b811708deaa31": ChaCha20Key(
        "3d0879a4c346daf58d34cd06efde941c8cc00a741c29168c592b8f52e51cbceb", "47f6fe744a09fbe96c93dd3c03ebf8e5"
    ),
    # FGT_80E_POE-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "8fc19e0e09484ac5885c532abd95a123ab5aed5b5972c641a24b73e6fafe8ce1": ChaCha20Key(
        "6e467652aae777d72f4af1008d016c14567bb80df9f8dd91c26824dd584e549e", "7f88e05b816a58408bb572108b41a8e1"
    ),
    # FGT_80E_POE-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "4b01b5ba01fb19bbad239751c87afe7ed0284a3a80df0fa69e5308d3512b0be5": ChaCha20Key(
        "291a3bf924508a93cdeea334594fee71e72e7ec56b83397847901e045d292367", "3b3dec237bd00e8dcb896c988665b56c"
    ),
    # FGT_80F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "e4d99de726229ebad0f968e5273106b29d92515af15fdc1ee3f4598bc1590e94": ChaCha20Key(
        "e45d74b7929d993dc7fc5c16afdbcd01a84b11f04466e73e1d4ba4adfd2c46cf", "30462ad1f2c25a37b4ada461307c9627"
    ),
    # FGT_80F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "afed629da8c3725a5da7c94f2570484d16cc17681756b976b49128d79d26c039": ChaCha20Key(
        "231087b9a269cf54b7a1964549b25897eaa4b59cf8a335c07600f590105ece59", "1ab4d4ad9ea3a1ae17a82654db4f671c"
    ),
    # FGT_80F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "d37e1f05f2844a4b54ab564531e783c22f44848508d54d6a05b67642d3dcc94e": ChaCha20Key(
        "e03f81bc83108ab195428bdfaa1599c62e6ebfeeffdbec20e4cc03278464f684", "a94d2a25fc67efe76e2ec5291b3ba219"
    ),
    # FGT_80F_BP-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "fad7951f311ee1ea050ab4f70cb081016ee1b4afe4a0f765a4d0285790611d68": ChaCha20Key(
        "840cea7f03d346102ab9cc803e5ca81d5f0be39a3db27d2afd2233f7ece6fed1", "80f002266675bbf9003fa95d30fb76fd"
    ),
    # FGT_80F_BP-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "e081a0f20ed2e38c8b3a946aca17699ecb59c6c7413b5264ab3e61584e4fdb64": ChaCha20Key(
        "53575f9446e3ea25ffcd1ce2f8f6d368d85300537bba4b8175487acf5b62cf35", "ccdec29e5c63eb5487781059f271e4a5"
    ),
    # FGT_80F_BP-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "151461b99f83b66732e6d6e26f84f567d5a8aa16e0ab2f958fa995d8675124e4": ChaCha20Key(
        "43ebfb56639f42edab9f908e16c17d9f21a86a67c2fece3396edc819059f8c62", "d2e43704517adb373f1aedbd68834cfe"
    ),
    # FGT_80F_POE-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "9b62fe54f57a74736a43ec0d69e1473c51d0746fe014766068a4e4ef4e5a47a1": ChaCha20Key(
        "0d7002d7d6d7d34b843d84cb81b2cb8458ebe56dbc633f9128b9a851ec2f8e1b", "8af5eb1c7523b6795127e2520b705677"
    ),
    # FGT_80F_POE-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "15258cac2bd2d2a6e87ce3278f07ae2847db1bed6a1adff45515e06b0c746c3e": ChaCha20Key(
        "faa90ea641a20ff5fc12b37b9dc0a2e981ee4a4a2c519a87d66ea62ebadabf42", "4dacea055cb356cef2db7eafa02fd797"
    ),
    # FGT_80F_POE-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "09e1019090f4aa7b6d204ab4c6d8be14454e00f7f5f30dcd24f9a4a6ba4fcb73": ChaCha20Key(
        "abaa81782a77aeeb196a42249bd29a16378adc4276c05666d89e92cfd10b21bc", "1c6642f98b3eb103784ca6d8f3b0381e"
    ),
    # FGT_81E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "00484b2113c84ab06e511f196f6910662cfc9151c8b55ef96f9a04cd5bfba2d0": ChaCha20Key(
        "7e33e32968d8564959b7d71ce1955594a12c3c0b16ad00b490f6c993ab413882", "c5385488e84b88aef75609605da4efd9"
    ),
    # FGT_81E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "258541ce8cf9bc0cb370b9543332406f001604ef3fc0c75bd7cb4f470505f31c": ChaCha20Key(
        "077b5f30cbe2bf5d28bb927ec149f062ba723d90732499a1fe4571f5889b46ef", "3d486cf2492c20004bab4e896fdc4f9b"
    ),
    # FGT_81E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "c2d081e3eb085ed4aaf2fe4e7b2645f038f129ea7de40619adf9885dddb64610": ChaCha20Key(
        "05502ba37c8fddf14040a4b17801b135f298f5886cfddb09e0a126310b78c81a", "9e431ae9d628de29b5f7e91ac3f86acb"
    ),
    # FGT_81E_POE-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "841a8127992fa543afc5d19f3f66a08d8984c5ca90a9195c253d95e9e954658d": ChaCha20Key(
        "abf5cdcd73671c89a0262071b6ef94cb6fdd6722f3ab3e8c34448cc2b6ed7191", "c86fb81765077118d0602058b3e3f673"
    ),
    # FGT_81E_POE-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "e6c6f74917052c6038a767e9612fdf3ef0f838ef93d759671917ed2b39189a6e": ChaCha20Key(
        "ed57800a7ad8fbf7b9d58f6c0af10868f0206a13d7a61451ba40c0192786878c", "1f96862691f38c9e334bc999306cc68c"
    ),
    # FGT_81E_POE-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "e132c2b8f22ade46130e55a442a487837de36290cba86de6f4ad178534e9a6db": ChaCha20Key(
        "15a118064154c9dee265b40da8fe805149874d9de766c3928b6f553bd560125f", "a80c3a2f17dc9c974dda0975ee8e9595"
    ),
    # FGT_81F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "502166597dfbb17845c8f6f771c4b94ddd683d606420e5abf3a1473e63b03ab9": ChaCha20Key(
        "615baee3ba76f4c76394d6570433781ce77a5f8c0573bb70c23c6e8e30382439", "d6c183947696245a7209e6d4dedb1d30"
    ),
    # FGT_81F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "f08a1693fb945e4f8b35015c59c44763d862ec018de44e06fe3baaf900a2c6ee": ChaCha20Key(
        "28e745a46fe22c6e77879860de68ee9c6be62ea4f0feb5a9d39667bd769d5f10", "b76542182979c62251fe2c4147f78264"
    ),
    # FGT_81F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "abf9019a6d97d29139e56071abf0d17565e5297f31bd4a4c627258b8bf6e72ee": ChaCha20Key(
        "bc8ad92775e05bf3ae08a559a2ebe30ad2ff5cd18470a381fadb825514029c6e", "558dbc73568803577948eba91033834c"
    ),
    # FGT_81F_POE-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "6b4e3906fdea6954a5bafbc3de88624b0d48d152d2d2cbb97d436e5638b29fc8": ChaCha20Key(
        "8ca0a98157ed127d4315903808b735dc9c29b4481f93818aca548332c54d3b67", "edf0445b38e8d41a1e1081af0f737f80"
    ),
    # FGT_81F_POE-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "f16971d2261995a12fe567d78e5e585b8ddc0bb449395742639aeef7adda0579": ChaCha20Key(
        "5e6b8df49dc14ae1d482cdb06ee6812ef1708cce1c2cc7e2b095cdbef8f2c419", "e2f73765bda9f40e91539709a1415a46"
    ),
    # FGT_81F_POE-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "390f221b6b9159600147f7393a70bfdcfc448f62c7d722522d57626b3a70e215": ChaCha20Key(
        "1b566a21c9b49ccb8bc961cf07c5784f75d40f2a01c2d3a5bd8759efa7ae45ac", "c6d660e51ce4291b4b47be7eff548a56"
    ),
    # FGT_900D-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "8000da4fd49514c3c0e81a7b508d3e69dcb7d26b257259e9d0550a0aab94b133": ChaCha20Key(
        "eb1fc0a1df174420e28a1aa34529cc14c22d2ad35b2fb4feb4ba5aaf1f3e2e1d", "bdad559588bafef22516f086bf76bc06"
    ),
    # FGT_900D-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "1b6146fb9c2fee842969ed77ba2166367923e302945bf96207e38a24281e7ff3": ChaCha20Key(
        "77a1fa90337e3d0058ba252fddf1728c7265e62f1d44d2f484ab486189b866ee", "5353e1ae234244b4841961294d8c43a3"
    ),
    # FGT_900D-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "e8260c1a064e3b342f876d29c39c7e056f755de8aa67a759defa41391fe23e7e": ChaCha20Key(
        "82dfbe8a0b6a7edd521dd8f0407f8dbbb33ca62b8115ec661f74a28bd70c1116", "1978ca64e26f072782bcedeaef67604f"
    ),
    # FGT_90E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "59f6d96b6ee759ee2193377a4aafaa9e83612da2ce0a5a702ad87973312be601": ChaCha20Key(
        "ccd078574cce65c2337161ae2cb04f4e8ee51e852230c468ce5d92575aeb1837", "653eaa3702ad1bbe48b1c22eb98f2da7"
    ),
    # FGT_90E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "97a70bf5080a78c92ea1582f47dbc6d346d5ec60c25938653f7c41b8730b43ad": ChaCha20Key(
        "29686254ba7bc65066068a245ddb53d31e186d54873a3c9b06714d38fa27554e", "c533638bb01cbc07c1afc375461e608e"
    ),
    # FGT_90E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "0c1d03d2124828680703cfb88106e8648f206b5d3a2b65f8086ebe6b4fe0141b": ChaCha20Key(
        "103ced07379792ff433437cd85867d14448db8adb7eebaaea7488f521bb8382b", "e78749241df3e518795d32cd84183fed"
    ),
    # FGT_91E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "26c2c6472c5d6ebc164ea6d3c98a4d055e57dfd39ac8ea3cadeb35ba9c90272e": ChaCha20Key(
        "128cac2d7151cabb8035968db27ec12fac3fd5d2b3c0fd3b032165c45a552ac3", "6fdf12baef669b12fd4c9d6419601623"
    ),
    # FGT_91E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "c6bd31541b0e44febcfe977fec52c5279f126549274c360bbb6be2a8ce8d143b": ChaCha20Key(
        "0a316209e07bffcc961d2f1c13d2a7d5f06b452c9e7b13b499b09c97ec707523", "28b09c1fccaff36e61dfc24afe81e8cc"
    ),
    # FGT_91E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "b3e71ed1a6d1fbfae94a9bd6813cfb932426051686dc040c0f91180895c06695": ChaCha20Key(
        "c92772b61c0b56e0171cfb1a6193708a37dc8eafd3b3553cbd6b4f534b1b90b0", "c65fd33e8dedde1055e6ce70cf734fa9"
    ),
    # FGT_VM64-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "ae6106cde43a0a15449fcae8e516e5cb7c7f89573d5f901e2d1cdb84917375e8": ChaCha20Key(
        "70ef25c87689d46c42d09fd6d60d5d8ad46f1226c0508872343610cda3676715", "c1f50081c1dfabbbd37fd3acdc23ecd6"
    ),
    # FGT_VM64-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "d66ab19f099668585c42e860266de76b5e3cc8eb3149bce6058f57c4b66db02a": ChaCha20Key(
        "dce0d1c33d4599ff60566ee1bba719ccd93af5f0f35d47fce391f8f7677febc3", "c0d407295046d439a5a52c0cbf98b406"
    ),
    # FGT_VM64-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "8499ddc51db5c6310d2afd7fe00283316ac0e28ed10b74f82aa8035bb269ad51": ChaCha20Key(
        "a78637e1039a7c6d19f8a717e092a5c68aa414329f214a947417c843acadac6c", "27006ee1608b73d34facdc2a979994a7"
    ),
    # FGT_VM64_AWS-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "06d378dc45db596a0b91b52feaf42f11b55d988130007f448b6be70e4127d3a0": ChaCha20Key(
        "23a9ae88f0947ab433852390a4ba3411a2589b571230e08a37eee68197986d82", "3c51ae938de8e3f9c14eccbf5d5ba148"
    ),
    # FGT_VM64_AWS-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "dd1ee24daec5ad31eb864cd3ea674b81b82ced13ad12e22b239c0cd6cde3097b": ChaCha20Key(
        "2a867535f3af0f952954031fb7bb7a0380396770314ba5d415b757806b30faaa", "ea523a000cf24221800400ca6fe99e73"
    ),
    # FGT_VM64_AWS-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "cffeef54f1112c33e1e180ebbd2ca798ae98340c0699f31af98051df96af3d29": ChaCha20Key(
        "7449ad32bebc531f6bdb1443f7e05f80c4470e6847a66536c54cd101975387e2", "fc6d4f48106a62fcd46e1afa11be8727"
    ),
    # FGT_VM64_AZURE-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "07e392ef6dceea4381f151aa31de50a08f3186ca9eeeaa69d0c4a9b8ad19ec18": ChaCha20Key(
        "9a7bec9de6eef0b6ed3e3708e43587d90ee50914729ad49548c48d74d6574104", "13acdac91d2c1783d248f29938d923c5"
    ),
    # FGT_VM64_AZURE-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "0aeeba84beed38ae36c2832bcbe0f509fb9fc6cda8810e81170ddb287d2e9a43": ChaCha20Key(
        "a6e9bb9bca90087ef1f1f2d8fca1ce8cc2314d48a62f8fd17a840143f6f562d4", "43e00321e38b14cb45f2c323c707b2f2"
    ),
    # FGT_VM64_AZURE-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "55f4534938efadf438d6518cf48c38c0019b1245f613c53e9c9eedc4670c0768": ChaCha20Key(
        "2650de82d32edcd029e5d3caa7cb3ca9ff87d0334409f6e88f66bb7d87521b69", "7c738ee6806ce4679b76e814a27a7290"
    ),
    # FWF_40F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "a8cb473ba7e8a7386d56167c646551931276007e14fbf999097edebb1be8ccde": ChaCha20Key(
        "1d2b049a51b7e3a5e376df35df89483a3fcab48d33ed1cf7d75aa1b2231e4456", "774a4a24d93ac6875ecee8d109d44c14"
    ),
    # FWF_40F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "47c46077fa827b1bd22241fe85c58189244a0b8b05bdbd72bc9e535667002f78": ChaCha20Key(
        "72eb1af28e3b3e422116045ed27b0908808a546578ba9bc5da65045c80b827f3", "fbd10557924c0052100ac5dd56026ff3"
    ),
    # FWF_40F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "d5a6db0d81bae0208830c615bd4aefd31f6a328f2da47d1318f0b320ad197767": ChaCha20Key(
        "8778c9065dbafc70de24fa73610a856ecc99930ecb59ec37187962e8aaec83a7", "15005d9555177b7f6b88372236ecb90f"
    ),
    # FWF_40F_3G4G-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "2c1fb5c3a21a59b5fee4050862d582ad7aefecb4a47301c28651f82e3b281716": ChaCha20Key(
        "0f6fe3584db19773c4f2e7e592a23adae4f3c5ff9a600bb0bd53eb3a210786b6", "ffcaf69b638f574e8cf70c55074ce11b"
    ),
    # FWF_40F_3G4G-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "fdc1c9350fc6c7b028083d96a7c53b3004de62a61e2b62fab11f48a780874f4f": ChaCha20Key(
        "a25ead82a64b3adaeb1a2a21dd5fbf41bb1bb258af79f0e52377bd1cc9a80f4a", "d22a665d9ece009bcfa60dc20b9ce73e"
    ),
    # FWF_40F_3G4G-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "04a86531a31690cf710706741b6d0d67c537a592adfe8864a1ab8b1bee4d122f": ChaCha20Key(
        "9413bd098ae216cf66c60d1bb5e06fbdeca55c1828a8ce9761a1ea936b5500b1", "0f77b3ce495fe4d600e1ef77f34359fd"
    ),
    # FWF_60E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "d4a82c288c14b4f115aa44a0c42b5181c67f42b8a3cb5154a390f4e102c4e784": ChaCha20Key(
        "d8bbb9ca7fef38bc9fc18cf2284b2f0539b7e1c8221b8121ba84177b91f549df", "3d2bca75478c40b4da55c638ab0df8ab"
    ),
    # FWF_60E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "f1fc965aa0c4ba7074b4c2ee711f4a9e76f280e8fe33584014a01fcfa105fdaa": ChaCha20Key(
        "35318abe4bff8377b1f9c0bd2eea05c6cd14e9301444f3253749330ac5047b63", "677840325c4fe567dfdf549535faf2bf"
    ),
    # FWF_60E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "dedd00ce7f9a5d07109b8a4ab7c484c8a82d7b379298c8930a4c2a0f7fba59e5": ChaCha20Key(
        "0653de23b88590ef2fac029b20538251627d3377dc772ffab2139b322fdb5ebf", "c3af6b0f531eaa797bbbf52cb03ee9cd"
    ),
    # FWF_60E_DSL-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "d82855712c93c67a0b714f161fa00db148e4c76b8167245f671e810fb1474984": ChaCha20Key(
        "89f2820be88726101ff6a42a8c3e583682f09c7c48c8a0a9899fad705f842259", "76983f056d192984c786e38e64b902f2"
    ),
    # FWF_60E_DSL-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "eac43c2ed5dfdcf864797cc415b4265f4b9a9882de539fcfebc344cc61364eed": ChaCha20Key(
        "4352f716b449cbf8bc3ac26e91ecfc35865cc97408e939eaa6e6f419a7d0b85a", "f379cc3e46f2cc4e51fea00d9602fd71"
    ),
    # FWF_60E_DSL-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "d1e7d8a336eae487559b5d7ee590db8ad2449df5b1b67fc1f874190305ce9411": ChaCha20Key(
        "955beacdb8c1b106e828a8c8e1afa839d25d67e4169acd796e407491599ee59d", "e9459fe46b39e8ca3db06a1a07162acc"
    ),
    # FWF_60E_DSLJ-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "3de4f7258a9b26c9dd0fbdc515295895ef06e4d3eeb46817e1118c394c5c17bc": ChaCha20Key(
        "f2a2496b0a4e3c918bda5fbe7ddf490434294290d6520fc8daae0ddd7bb3736a", "e7a5d13c241ad5975590db26d14b5ffe"
    ),
    # FWF_60E_DSLJ-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "59906a669fba68542235acd02547160872df4098fc1018758233f91698d2c0f0": ChaCha20Key(
        "674d6878154e7dfb470607af4177d32ee129758583b8fb8233ea0bdb3cf5cd44", "d387448e49fd79b9f014e9652a12d545"
    ),
    # FWF_60E_DSLJ-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "35ab48260e50c53f0bb4a1c679b4438a285761422046e59879de76b6d2c6929b": ChaCha20Key(
        "f6b4e4f378fe136aadfc602fcd44bd89c034df3207852b2e6deef17229989618", "53fa6b30c7a7e447870ca7d96104ebff"
    ),
    # FWF_60F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "8e1a29733a6fd9d165eaabeb34bf2dfd20bc017c5b2d32624b71de51e6a27364": ChaCha20Key(
        "8068350da9fb928b4bdc8909865663ac7e50801fa67e9b40ec55b45c0c50fb6a", "758dd0c1592860e75b3a3e8ac52afb98"
    ),
    # FWF_60F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "81996be13bbb1de44f6d6405a8097b17a9baf0db624d46934b3101b5c1b0885b": ChaCha20Key(
        "e1ea1b9eeafb7d4e08680b6fe28458fb64d252cae5d8fa721acff14a7a50a5cb", "12db325a67a434d3ec027aab7f77f0f8"
    ),
    # FWF_60F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "b355ce6482c18fecb5647f8f513b2ee8c69dbf320e5377d0eb81c3bdb67fb2d1": ChaCha20Key(
        "37fc63ef34b4fd14c94a1c397eddf5b025ee22d9217b89517ec1f9284d9234a5", "3dcd5b4fab6f067a4469351d75093420"
    ),
    # FWF_61E-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "6b202f7fbfa187ba8661e437e6cce4e5460fa6687e70dc825a5b71c2d5966305": ChaCha20Key(
        "f3bf2399962b586536c859e4e9bdefe81f17cea5b0d44c12bc4527d26774b2aa", "736f1e2c19b06c11460a4d03ce7ec17b"
    ),
    # FWF_61E-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "8b553f7c655ded2d874ceb04384cf8c33b2b407d880b504b9092f0f45abb2eb8": ChaCha20Key(
        "957decaa1b3a3f2a400bd1d64dd224d362fd8813ee033b3a46ee203dc697990c", "c40ed3ab849ecb5ef7f952c6d1e224db"
    ),
    # FWF_61E-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "0a8f63556a5e31eb08137e7a16d52391faba6016efc2a81557566130754897bf": ChaCha20Key(
        "5a749ed62847bb314d5c7e49a94b2277f065f7e7155d67006a3a0e2410186ccd", "111bfdad34038807473a9af38631a588"
    ),
    # FWF_61F-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "c035e86f129679d92a6dc878a14427cf4f0640465e3af03b1ac7b89d0d9575ce": ChaCha20Key(
        "e503ad530cabefb742c2f7426223e0be23aa94b44a65db1e722b93621ddf52bd", "b8e957a8ab931f8878cdc46fde45175b"
    ),
    # FWF_61F-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "88ae4afcaf0a135968b83524ecfec6e79bbfe5157f04b57ef5dbb48434bc29eb": ChaCha20Key(
        "ad3d3680e5f82858b79ef2d32f5b8241404228c058d88a1d08c7bf8557b1d612", "3408c3cf25c671e896b3c7aa6c2c065d"
    ),
    # FWF_61F-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "8157ce0551d322f2b1060c350e6e938154e4b1f7ae7820ee9ecffe8b8ab45e53": ChaCha20Key(
        "6948f546490f7bbc41ae8af71c3bae910d35864247d8acb6a4470310bb72f545", "d6b05a69882c8c5590dfcbbc290321db"
    ),
    # FWF_80F_2R-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "e0eb5e16bcf861f9b52666bce44e7aaba24940b83f2ee2dddfb1040260449d75": ChaCha20Key(
        "b08fc4b4202deb169cff7ae09470609d82e542b0f805163819f051cb64827ef4", "c558a41c731eef345a42f38e0ec2848f"
    ),
    # FWF_80F_2R-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "3afcb66a3a1f3ac6965faeae78a48e084296f5637d0c3c2c3f527ec8ee801740": ChaCha20Key(
        "78cdaf07d3db0ffc2f1ef164e060e3cd23aefcf95b794b795f8c7e4887d36163", "567076e487413467a84221fefb51f6c4"
    ),
    # FWF_80F_2R-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "0b72b5e543c633fd2368a3414d0b391176ac177ca9c71e1900a3a0cff9e88e43": ChaCha20Key(
        "6d9af8aad03994da396205ff5c1c98cb58ab288eded4f598eedf630375c2b3a1", "2347f9c352daf8a621b538250000e4b4"
    ),
    # FWF_81F_2R-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "4f88618d9b8bb6936770a1013e749a9b054928acb8e31c71a53f7a04431d03cc": ChaCha20Key(
        "435a8fe331f2c484e730adc48e691f27853cf0c465530341dae91e6e7c509913", "ed0b8058e24c864596ce693e2fab87b1"
    ),
    # FWF_81F_2R-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "f69cf91a4f1e50c7e6d3f2e48742ce441a87caf52c38b59dc501b8a3e16230a5": ChaCha20Key(
        "94a946a79a628d2a3734e31a33696d3321916b6a06a66027c8c3f209dc5d302e", "2cf7c563c4dac1af6188048eee7dc73b"
    ),
    # FWF_81F_2R-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "a21ad751fd935502f39f9f89dc797e92aa5510495d165785a90c9e7601416226": ChaCha20Key(
        "52f184dc1a34631d4d5f191d1fe79e612584eaa7d928dc69aa9914072cf36a79", "85ab3adc739c55a6f506de7807153d3b"
    ),
    # FWF_81F_2R_3G4G_POE-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "1d14065b4e978f7a446ae61bd5da5390d2aa4ef4f2267a72fb95caa75d454e1f": ChaCha20Key(
        "f7a40dd37a942f8a43b1bf6356c1657332c4975b20ab40173a24cc1a8dbcb04c", "344d0f4402570a595cfb6eb5ebc7c128"
    ),
    # FWF_81F_2R_3G4G_POE-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "1edc329909295569cfa955f814dbfcfdd81bb806d37a83cba690bf8af35e9994": ChaCha20Key(
        "b090c159c0daf5a9520e16599df35eb8a49d94ff5a8d672b216d694923843f0d", "bec37b25c663fdf3dc9cbb2aee190baa"
    ),
    # FWF_81F_2R_3G4G_POE-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "745a0b9d672aa75d2ec2bdb6fafdba8eecd8a24d0d46f3189c39633d4d79abfa": ChaCha20Key(
        "00c32b9e74d457bc915b1f3590eb24f2d468e258bf5c9812d5810c08e79da7e7", "0543fbe11ed6201b5f7e2c90bcfc79a9"
    ),
    # FWF_81F_2R_POE-v7.0.12FIPS-CC-70-16-build9223-FORTINET
    "8c03a11426ef7b64a8688a14ac362e77a9a0aaedc36410124f70fabb1c1c781c": ChaCha20Key(
        "37bf2db060cf5d9a888ee13bb8f0800223d62e676c5cd5277b4284b4d088d166", "4e2246943ad2b08abe0259f8cb209136"
    ),
    # FWF_81F_2R_POE-v7.0.12FIPS-CC-70-18-build9406-FORTINET
    "4f4d40fb12654bb7e944d31008e446d05506420f818a2dc67cd0042929181f4a": ChaCha20Key(
        "117cc969d968cb2947598d174ba5182a9766fd70acab4a5a7a7d208e17f1fa0e", "da376f60f15243f361ee4d1e81b3c6f4"
    ),
    # FWF_81F_2R_POE-v7.0.12FIPS-CC-70-19-build9485-FORTINET
    "e4f83993d23bd2f496c29e0a86b5a8f192d7b89879685ef8e32f0fe1ae74594d": ChaCha20Key(
        "46b225a147e1fdcd5825a8214fd5bce1bd682ecd4566e63149fe904406662240", "1dc11c232fd1a7aefc10fdc9e2db62af"
    ),
    # FFW_3980E-v7.0.13.M-build0566-FORTINET
    "2be771c0b844cc61544a59a9507fc7765392c3281164f6e41bbfbba58403ac21": ChaCha20Key(
        "c28758151f4be7911463a785580737340b0732c353a3d73c3f8756ff1aefa93a", "bd0318b73124fa73e145ca481a507b70"
    ),
    # FFW_VM64-v7.0.13.M-build0566-FORTINET
    "e27d98df263c742ea0a5c7f5e2b4cc748d9689d1e762e5a40d26ce51873a3275": ChaCha20Key(
        "cfeb8acdbd125a9bd028850b67869d8c33542b8107454794730ab398ee0c4787", "ae5a3fefa9865d4f2f2b47909b0f6ac5"
    ),
    # FFW_VM64_KVM-v7.0.13.M-build0566-FORTINET
    "11ab52963d99ebee535e1742ce0822d77be18ee95e0ff5553a7d2b1432b289bc": ChaCha20Key(
        "ecea1bd0dd31030d6fa28110ef55e2e5ae1b442ab21f2876307fb1c877ec64a9", "e2036e83fd1e7be1e0d130d3b8f8f685"
    ),
    # FGR_60F-v7.0.13.M-build0566-FORTINET
    "ed4bd0e324eb73c0a2049921da751431a4e301a359d08dae72ebfdf133d5c42a": ChaCha20Key(
        "be75b7b73af2d59c11865cf44bb5a6c1625ff87cfd288b6e92a220687725c001", "47e946bf3671fec8094e1d4588e7543a"
    ),
    # FGR_60F_3G4G-v7.0.13.M-build0566-FORTINET
    "440de1accc377194d965ca40639c3b1e6ce989b7d1ae68ad4f1cf5d6a8c1fd3c": ChaCha20Key(
        "ff48556981ef1d598121eff6f1eccd90e25e277cf550242c4ba56f045dae98dc", "cee6ecb0f0e9f1077e3efb5fa055951e"
    ),
    # FGR_70F-v7.0.13.M-build5396-FORTINET
    "70d6ffe2219ec9f033ccfc505d4d19f34977a7ef37a06ba9dbb9dfa5065f0f6f": ChaCha20Key(
        "8418b778893e2df5e744a035622b0c775ac8ca6ec933196e689be20eb2193794", "2865cdd261dd16919d4bcdaf638b99c7"
    ),
    # FGR_70F_3G4G-v7.0.13.M-build5396-FORTINET
    "96bcdf4564700eac9ec6cfdf3b49b319a498f0ec0a364fbb3189b07145c53c13": ChaCha20Key(
        "4c746b5ce38fe4d6df81279899b9948116436bf3606f567d4eb854e4bf99641f", "8c7d967e963e3c03b140b98514685a7e"
    ),
    # FGT_1000D-v7.0.13.M-build0566-FORTINET
    "ce1e36d474f6e9151b2c0dfbe9c9d1f63dcf81d34bc48e8fc9d78f3527cde800": ChaCha20Key(
        "45af1602a969a240be6ef502d955eb505eca6bb6bf04281622a64c4d244462a3", "1dadc72e054e337228a99d6c4aa1ae3e"
    ),
    # FGT_1000F-v7.0.13.M-build6903-FORTINET
    "a69cefa0393ce65ed300008a0170565a7b3356438f67d20df20544d5bc300555": ChaCha20Key(
        "029b1206847d208861b8ee6c06c5c5d00e6b0d5659c84a5df0bb12d43fffa9b1", "91d29fa3cfb2720782586187cbd1be9f"
    ),
    # FGT_1001F-v7.0.13.M-build6903-FORTINET
    "8feaac706d94b095149da908bae9fcb3a7548241b896767ff783b910bcecc02c": ChaCha20Key(
        "058dcad891b17db60571b8e527bc51935a2e97bceb12857e574146220ec4fc5f", "0ba89074760d7bc36d6a57369561386d"
    ),
    # FGT_100E-v7.0.13.M-build0566-FORTINET
    "1ddbf383db010a7cd51fe856f88266f68bf978c3e36c2b1ed04b59960ee2096b": ChaCha20Key(
        "889336d66952f92d18b3ed41e6a60e268e227eb2fffad3cdd6fbd1529b826771", "8573c79d031889a94c55d67f4d5d86dd"
    ),
    # FGT_100EF-v7.0.13.M-build0566-FORTINET
    "64cab568249339eced097eac297cd992f1083b19646bc48be0800ff9565ebaa7": ChaCha20Key(
        "595fd5b873a3197e48adadfbd948bdb3e3f77db2336c38d17abf7d2567d19e81", "e0747dc84708bc11cd424182ec96184c"
    ),
    # FGT_100F-v7.0.13.M-build0566-FORTINET
    "c6ba7cfe4c12856b82bf248d52ade494194ac1e6ddb1b60bf7da5508befa1161": ChaCha20Key(
        "b61eb6802438236043d54bbcabb83d9712482efdee6521b836e9a8ea7568f6b9", "cb9bec5fc5e890ef54c1a1e3b4acdce8"
    ),
    # FGT_101E-v7.0.13.M-build0566-FORTINET
    "e09ca1cc27e9215757169be238b61110f2ba5d12d3bf093ce4687b29b96b0c49": ChaCha20Key(
        "83d7fd79e9e26ec5fabd8b0dc2c158d000238faf4331625327370dc2328ee052", "83dee167e771206380198df5b6c29151"
    ),
    # FGT_101F-v7.0.13.M-build0566-FORTINET
    "184f0da0fa5b659318fda77b2122263f84d912f711ebd0ae87467f82127129d1": ChaCha20Key(
        "5295570c0c3a757c03fc35d19febd38e87f7db122c164bb850d505e98f75a631", "dca91397fe8d8e468c91bdca20c8ccf6"
    ),
    # FGT_1100E-v7.0.13.M-build0566-FORTINET
    "2755690462848ec5d955d0f55d9d497204c46d98f5af688de0fe6a7539d41815": ChaCha20Key(
        "9c4700eeb28cd3a76c4236211e4b62dff9f607a2d73e8a7fe23d747a971c41b3", "b363b966fca3fa467f4da1267df78fa7"
    ),
    # FGT_1101E-v7.0.13.M-build0566-FORTINET
    "09a525295f782a437d5de8da61815a6b8404f9a2df4c348c29499ff96b10e475": ChaCha20Key(
        "797ad4da4df4fa3340bc730f65762787c1ad9f4bf9863a871c2267fe0afec43e", "4ed06455d8850a8fc20c4b81113d1925"
    ),
    # FGT_1200D-v7.0.13.M-build0566-FORTINET
    "8623b6bb7d52a998bbdbe34e6a6b5dae8608dd0b946d75ace36577485b24edca": ChaCha20Key(
        "2b3817a8ca1fe98c457333c0ab43722e7dbc8a6e7c010b7f6facecaf20237416", "7406e7648aa7200872d785c1cbba65bb"
    ),
    # FGT_140E-v7.0.13.M-build0566-FORTINET
    "f358346fbf5e5333fa927b8d0726c21bab2b426bf89259e8d3da2faea8d32cc2": ChaCha20Key(
        "6f222f064de178d0b798d268bd34d116e69a30d3b318da4d2778901adf63b29b", "27011e0a1db74d24481014b0617132d0"
    ),
    # FGT_140E_POE-v7.0.13.M-build0566-FORTINET
    "affced747e49a9240b9756c761c1e60f87af46d1f4285657077169aad70aae0a": ChaCha20Key(
        "e89ecfa2a221be54b0b3f0c582d1d340db3669094003d0def2740cee287edd8e", "6ea1fc01c964968987e3c401fc61b256"
    ),
    # FGT_1500D-v7.0.13.M-build0566-FORTINET
    "bb54847ece7d20b23bd6746054980f689c91b257de6f5a5104249e6ae0acd11b": ChaCha20Key(
        "1b7e16b5c5169a58b860a071e14d47ea75e19b8c654751ab382457062ef9278b", "c493887b6c1d0c17c3bf548252522664"
    ),
    # FGT_1500DT-v7.0.13.M-build0566-FORTINET
    "f64fdc2d4815ffe2602d2de1ee91776cd5835568f4ad86ed706d33986f1ad1b4": ChaCha20Key(
        "ccb6824c6e6ecd7b8314f1d97bbd557b45e5c678d427e9ffd561fb16df30d5ed", "f55dee4ff091e5b0e458ced55a4dfdef"
    ),
    # FGT_1800F-v7.0.13.M-build0566-FORTINET
    "77d202c3b193f52736b97525aebb8eadaf20fece2625639628a9f278b407cb11": ChaCha20Key(
        "3af6d6b85a498d8beb2055b789944f28d8bbabddf521d8ce2337511e35bb26f3", "fb3ab61cffb1f5f71b96c7330740c58a"
    ),
    # FGT_1801F-v7.0.13.M-build0566-FORTINET
    "332f257b1875b12497fef71322ac5b43dd7985c7c59681d4ccc43bee42f0a2b2": ChaCha20Key(
        "f3658375a2d0da68c33583e2d1fb4e7d2daecc2665d45dd00b6d29aa46cc56c2", "ede6e574ae3d24bab413f58952c63f9c"
    ),
    # FGT_2000E-v7.0.13.M-build0566-FORTINET
    "74e2fb1a96874708c5a365e23074bb833c6b2c2f77dac5440de6546353fb41ea": ChaCha20Key(
        "03a0173843a45e3ef9f9bc7354ef7a2a22038fd34321ca8c79cf5e838f93363a", "79b3b99b875aeeadee44efba470fd799"
    ),
    # FGT_200E-v7.0.13.M-build0566-FORTINET
    "25513c2b54229dcc317b717720e9723c402b925e78a2f464227e567f0c420b74": ChaCha20Key(
        "373ddd668b6671bdaf28150eaff800a145263c5ecca632e0135dd928eef58d28", "f5349bc8cebe2c07eb3dd0e253877367"
    ),
    # FGT_200F-v7.0.13.M-build0566-FORTINET
    "25aefd1633292bb633ab69bad8172edd4823dd82ba980e3f2a57498d55de9085": ChaCha20Key(
        "f398fd22436f3b33910279cbdd92a2b2fde8b182ad6ba522b63119b7fd35c489", "679d05e4a81c1ff04684ef80a1e811d8"
    ),
    # FGT_201E-v7.0.13.M-build0566-FORTINET
    "7ada8ffc51994307c84775be96f4e674dfc3fa55ebcd5e8177097106ef09b9cb": ChaCha20Key(
        "df350feaf528a8e1f0da139a5c1555c02b864df41bd20edebb99c3c952810cfa", "e580a4a621bab513d2c1ae0830fbc7e6"
    ),
    # FGT_201F-v7.0.13.M-build0566-FORTINET
    "b002729aabf8d90efed7d0a0df474963acb2f9b9126a341f9b096692712da0e9": ChaCha20Key(
        "ac3537b944d08a5afdc76b8e4d5b7708d1505ecb31d6196f7481c85db50ed41e", "774f5c6b0e08108d027152bb799d5036"
    ),
    # FGT_2200E-v7.0.13.M-build0566-FORTINET
    "e27d64b560dc092af7b06d941cd48a51bddcf654a54c589315ef7b91b41ef55d": ChaCha20Key(
        "ec5a5fb9c65ccf0f6702a8350b405e5f19bddf26c0f4ee95d3e9a4a53dc2d87b", "2558a4883a4c6f4a2ccce57f53c381c0"
    ),
    # FGT_2201E-v7.0.13.M-build0566-FORTINET
    "0028874650e0e351e336363145bdaf7fe860ee4c70112cb17ebd4eef2765d39c": ChaCha20Key(
        "dbf18e3e4e0029376b1449626b110e81a2a92fefb86850e9f1a0cba1db99222f", "4ab4a269a432e1215129b388752a4178"
    ),
    # FGT_2500E-v7.0.13.M-build0566-FORTINET
    "ec9b5a8fb947072a4543eac08773762bc11070b3a0f7e44760faad56b39310d3": ChaCha20Key(
        "160a4aa9d413093a2b184ee57593d4d46f68db3e521c90957841d2595a71cf7b", "99f3910af14e2e959e9210ac8a05e8cf"
    ),
    # FGT_2600F-v7.0.13.M-build0566-FORTINET
    "fedb35511e15713e029daf42fbe7e7cd13ef7d054e497f3d1ec3e8897f60544c": ChaCha20Key(
        "7f9632555ad352d23d848b78626b05155cef73b4d7451db76b1771e87b5c272e", "d9ca2c34549370786f8f9b4ea31c0b39"
    ),
    # FGT_2601F-v7.0.13.M-build0566-FORTINET
    "b8d946cc89294d219db99859213eb02dea083c089d4434c78c3644af5b95cc8f": ChaCha20Key(
        "234dead04192e4342fc0124c07ed6fa162c2292f0b68bf86703ad00f2bb25a85", "49b16d4c1203cb02daa0b4e0076b328c"
    ),
    # FGT_3000D-v7.0.13.M-build0566-FORTINET
    "74bf9f6e89df48c173c253a657a0a2cfb26366fa94f878e846eec55075901e57": ChaCha20Key(
        "c05ecba7784982a2288c62544b412f90bed72a236355c087d056844e55402294", "c529253bbd1548a308d3a25a0f685db6"
    ),
    # FGT_3000F-v7.0.13.M-build0566-FORTINET
    "a0ac87f260d59b6b2315c6c6d2af7c758e3245e9f5d5c1201613129583aeac46": ChaCha20Key(
        "44bbbfd667dcdd4d5e9ffaa03a2dd8377086ab74c29ed5c1345ec6128edd20ab", "1fe95a6f9d8e9058b462ca59fa37ba2f"
    ),
    # FGT_3001F-v7.0.13.M-build0566-FORTINET
    "9182381a146086cb5fcaa5f52429ff860fd20a62faf35c729da8c0c3185bd57a": ChaCha20Key(
        "4ad441db08152060e48910b933175370a0c36c59e2dbd36ccc909027b5b1ad34", "cf9835189e26c454c941b00e74ea4241"
    ),
    # FGT_300E-v7.0.13.M-build0566-FORTINET
    "e1a43472e40174cd34dcdae47fa9caf404fb81bc70245f2dfa10801922968e0a": ChaCha20Key(
        "e4559c436e8a64947b1981c6dc256301440ee90bd53ee5bba811859a26e8da8c", "cddba491c53e7ab9fb511c8c63427d97"
    ),
    # FGT_301E-v7.0.13.M-build0566-FORTINET
    "186b784fc5bc969a46417aec2aaa96e8459a8e8a2e7531300cec70d32f2c9784": ChaCha20Key(
        "0c9e4ef8593e85dbd0e07b34d8c18914c15cc18096ac9edb197de4e8a83fcccf", "89984e4cbfee8479e809eade54639609"
    ),
    # FGT_3100D-v7.0.13.M-build0566-FORTINET
    "012cef950471e877455ec8904728a5396c42ed314726a2b74e78ff77a6e8fe24": ChaCha20Key(
        "f67181fa3ab468ce48bcbb95b000aa6f2ac88b14abf14221e87779d3058c0d28", "65ec9b2ed2ded36152d7d34fac0dae39"
    ),
    # FGT_3200D-v7.0.13.M-build0566-FORTINET
    "f3a6297a78b2ef37c52683c96fcdbbcaaf33803c56c30570fbe43a6330830a83": ChaCha20Key(
        "b9f223c005910594a5b1af12a0720be85878eff088a90a57a28840cb3d558d34", "0381555f77d4039c92a59ed49615a7a2"
    ),
    # FGT_3200F-v7.0.13.M-build6912-FORTINET
    "f90483186f81558e7b945a627c2cc121e73aeab5d4aeab3b47c9ef7f75426b4d": ChaCha20Key(
        "6798507b1fd8f8cdf4e9c9ff2c71e5b02c55e1b0776db4fa04d410ade3c6bb02", "495c85ca3efdf3e60c195996a0736392"
    ),
    # FGT_3201F-v7.0.13.M-build6912-FORTINET
    "f22803b3791d8005319903e7676770dc57e916af9d7ca270f6a78e0326369fad": ChaCha20Key(
        "eebed725c54ded6385bd31b3eeb768921ce7593f337a7a29a7d609261a350be7", "8f651eaa6fd7688dfe263c2ce19a4d2a"
    ),
    # FGT_3300E-v7.0.13.M-build0566-FORTINET
    "d3cde6c0a0bc4d267aa76f08af3c26d434006fd0cf498e7d8dd0181677cf7eab": ChaCha20Key(
        "53c7120489168f49abd198400973c4e001e3a6a0eb43ac24832e6384877c0b06", "062b70cd757b387b30682dc895e58b8c"
    ),
    # FGT_3301E-v7.0.13.M-build0566-FORTINET
    "b0572a6f3ab7761393736991f48d474771e79c9da9e60c5d14fda799e4a56ec3": ChaCha20Key(
        "d396fcc7399112aca10542b5afc3dfb25d1de09adb5f08f33158db468d28a274", "54d216f2464b12bb29b685bd1c8ab88c"
    ),
    # FGT_3400E-v7.0.13.M-build0566-FORTINET
    "6e7bed0ec7c31c63a5d82d837f9d237b09a7fd3bc865109be8fe92f4df7e32b6": ChaCha20Key(
        "29bdef2b74b20813f5ed073633b70920f2515f0c3743e29a0b0e1aa097dd9b32", "b384d0098878a77e2a9b9e07b9c22bd9"
    ),
    # FGT_3401E-v7.0.13.M-build0566-FORTINET
    "5d6456a4fdac5a79d8b3f2a66eea7ba47d0adf5f750f306fc3e80caca9195401": ChaCha20Key(
        "b8005086198351403970d85d511c554a377551458cd90310da9be6613fa64609", "86e6f5e5390e70c002d8e71d549e5cac"
    ),
    # FGT_3500F-v7.0.13.M-build0566-FORTINET
    "19c288c66935e11fe7a3e2d0c132e79659f76d2b2b4be42a25f9bc874f85b76a": ChaCha20Key(
        "63ae87be0ddc686baa731b635ea1aed07a7141963a64f1ddac069071e50a7aff", "a3b29fc5efd4a8e143bd35da870febcc"
    ),
    # FGT_3501F-v7.0.13.M-build0566-FORTINET
    "11a6c1b2cabfaaee221730182fa0cf127d6977e03ea73bf62230aaf97fdb5ff0": ChaCha20Key(
        "f1cb5957d8d0f03f17aa7735b4844f4d289fde35fb9c0751232696dd52ce939d", "3dfe196adc3ccbe996b7a9cf691af94f"
    ),
    # FGT_3600E-v7.0.13.M-build0566-FORTINET
    "6e33b2c17da5bee44641ae119c71f8cb52369823f1006655511a910ddc77e899": ChaCha20Key(
        "67a07ec22a22abdbb688ad06f8e2c8323ee6ea3ce27d49e2093e3c83dfe7c75c", "b33ed956137f62d4dbff8186990625e9"
    ),
    # FGT_3601E-v7.0.13.M-build0566-FORTINET
    "cf17cd6d81285d523362293889c2082bd25d00a9e7c62d202f5b592d9efce913": ChaCha20Key(
        "bfac98bca97ad90cf52f45bcdb277a9ab721dfdd4501ccfee89ff98f3e3f0304", "cc9c50cf4f209d2b85c35ac47b260852"
    ),
    # FGT_3700D-v7.0.13.M-build0566-FORTINET
    "eac69d9c3deb6919756dd7b95df7a8cd8020129ba1f8ff7c92abac19c05db552": ChaCha20Key(
        "793f7a778ab7d0528fab3cbee4994fbedc6ac74fea6fc4084bb58681cafff066", "70e7a6182862f987dd2d955981f15664"
    ),
    # FGT_3700F-v7.0.13.M-build6912-FORTINET
    "90931b7ac8990bc62604befcc2f9ae66c6c862bc06748f441171a6026e8b06f1": ChaCha20Key(
        "4765d9258205b8dcad5301608b4f4ef83a190d44be895656c3c0d958f9b18601", "2fb58865f5ec73b233462df2370f5171"
    ),
    # FGT_3701F-v7.0.13.M-build6912-FORTINET
    "2c454e68b2d30cb75fc7a6d386b8bc1a47e563e5efd63159cdc898243b9d7876": ChaCha20Key(
        "9f5e1701ff3c306acdb3ba52aa21fe8352c871af8ec06b64d8599a8c4125bb70", "bff1e0a473e26dd974cf9d60c2df2bd5"
    ),
    # FGT_3800D-v7.0.13.M-build0566-FORTINET
    "1cf89d1bfd18dea85fa47ce507d0c51c9618670d0c29455af938ebd08743bd30": ChaCha20Key(
        "a7f521031866a47c6edd4336e39db0b00c1b718a6b52fcc72eb78315d67b3445", "215902bcdd8a30b053621e19063079ba"
    ),
    # FGT_3960E-v7.0.13.M-build0566-FORTINET
    "a19604b95dfb74edb27ed14929511b1bf19daae34a3a35f5fb0d6f305e9d063d": ChaCha20Key(
        "2b9050856322d615245121dd35fb65b3305381cbf3d68fd4d7ae0d4f252d5bd0", "3f134e2880e1dcd6896fb7629b200d66"
    ),
    # FGT_3980E-v7.0.13.M-build0566-FORTINET
    "3b912602a2f8bf334c3bd8d6e8bf1b9564834e025b9c3049381ac05812822633": ChaCha20Key(
        "a0c43c44be6bd56db6cef38f8c85ec565492d8a04c4a9cbe40ffacf4a3f7d360", "dfe91da873753e11ef1b581c5276e1c0"
    ),
    # FGT_400E-v7.0.13.M-build0566-FORTINET
    "999290009f7768d08ab05b36724bcb6cd866cab29351d02d82ab4f7c8607d78f": ChaCha20Key(
        "608bc559c9c3a35d4d9eb10d4158edbc84f43932eadd69b41fc7264733167841", "43f4b231100f9e73e2584bf367563316"
    ),
    # FGT_400E_BP-v7.0.13.M-build0566-FORTINET
    "d8be6c74372f0427f4c940fa5683767705eb5755acca22ccdeec63cc2443d8a4": ChaCha20Key(
        "64ecb25979834b79fefa634ccd25ef959ece515a20d1ca30acfaa562f77ca182", "a31b75b107a3e43e6df087a7a2aa4442"
    ),
    # FGT_400F-v7.0.13.M-build0566-FORTINET
    "f671dea3538ec44a8ce9af39f09bf2d1b01b916403792c6b192f625b61f42426": ChaCha20Key(
        "270c0b942956b131839ad665ed0db357d7b0d29504c17edcfcc33c45b4352b7e", "34aa0e89ef13a464d9e3b87797da7371"
    ),
    # FGT_401E-v7.0.13.M-build0566-FORTINET
    "bd6498552183ffe464dc0727ca80e32e17a9671b5ce49206c76182c120876101": ChaCha20Key(
        "c5972ee243c348a2c66ee93eb7f9950a71e914364aeec622a495aeb5e2fe805e", "a423e2357155489e817c7cbf0a9d4474"
    ),
    # FGT_401F-v7.0.13.M-build0566-FORTINET
    "21187790ae0f6cbc9f7de40f20c7870386abf5aa37ba5697f5d576c4abc5138c": ChaCha20Key(
        "be26a3c7c3048a86f73f12f2219a16d37bcc4a6350fd76660f83d26e57ee9975", "f1d30d94cded14a9d1cb7cf578716d67"
    ),
    # FGT_40F-v7.0.13.M-build0566-FORTINET
    "835f7e6528640898e140bc8966557a631a11d10c0cecb6f9a6ce493d48528993": ChaCha20Key(
        "66539b176e8e95c28d7ce3e91d21fdde63c05488fb246976a2e37dfa2ad8c576", "210986fc252d9bb4e7329af28c8574ee"
    ),
    # FGT_40F_3G4G-v7.0.13.M-build0566-FORTINET
    "bd794d4d213ff1f4d504e30e8ae689056f9871663f5de34d31e9ddc18610f3c6": ChaCha20Key(
        "d50ad8e028c22e5030f475e0d68f0f539c6fa00c8042ce4f3e66bdaafb26dbc3", "891ec5dc3b3a07803406f016388a5e81"
    ),
    # FGT_4200F-v7.0.13.M-build0566-FORTINET
    "0247785804ff7eaa338e1deea2d052c4eb12c60c0ff93cae1f9dc77e92395d82": ChaCha20Key(
        "aed73decc6e1657a12f46742130ce9a322bd9a57cd504e6354f62f9aa09e4b54", "25727d92fc6613f3094462e1885938ad"
    ),
    # FGT_4201F-v7.0.13.M-build0566-FORTINET
    "7e9ae5512311330a055448a5ed38aad420a98ce1acaf1c3e7817d987b8e84c1d": ChaCha20Key(
        "3b50319a18b219a715384f8002c9000ee320ee7db5af29fefdb07225142a4693", "6ac1f4a4e7817169ae7578c9b2ead926"
    ),
    # FGT_4400F-v7.0.13.M-build0566-FORTINET
    "daa6ea92b0f61f01917d078367cf13780bf70f69d747e988612f2e48b76dc368": ChaCha20Key(
        "970627729eb57dfb10e8c51399c58a36407faf6e8334b11b481a84b004cc9ea0", "6a0d0bdfdb7615a040280fa9d64b8478"
    ),
    # FGT_4401F-v7.0.13.M-build0566-FORTINET
    "aaee44a24e1e0ed27525fd2d41859409ab7ba2d9995a0c1e3c84fb1c20b79eda": ChaCha20Key(
        "8ea1e9468c9e4807aa521eaf6b281abb3810e64a599585758601807ae6afeee2", "5df213d2d2fd5d530fb69e28ebfaf5a9"
    ),
    # FGT_4800F-v7.0.13.M-build6912-FORTINET
    "9482bbe44f20136966365c42cf71f339051f831ebd58ac12997adbdb2f9a2e41": ChaCha20Key(
        "ef06c6affeecb5b66ea9ae3c6d8ea2e30fc5946b516f8f6817bbe9e269a12aa1", "271ffe26b942b2a2c27c151dab1a359e"
    ),
    # FGT_4801F-v7.0.13.M-build6912-FORTINET
    "3ce2547ce86dbc9bebe04a393a833c681356499db00c3c6804a01daab58f4f52": ChaCha20Key(
        "a4c009f15b899f5d6760491453227939b658ca50e54dbe7d927635f8dba040bf", "d601fa4f9891679e8e6ef5d93efaf2ee"
    ),
    # FGT_5001E-v7.0.13.M-build0566-FORTINET
    "1e1eab6198da7e4cf7f421dbac8d11e504bff75e8bfc0deaef520d495fb1283e": ChaCha20Key(
        "388125f1bdfe99b228ef681e636db5ac92d2ffed69956803c370d5771983ada1", "c875a6c8190222c2b9c89aaed21ae621"
    ),
    # FGT_5001E1-v7.0.13.M-build0566-FORTINET
    "755fab88c89510a38453ae05a5a0850e80c545dab85434dc102b22a74bc83ed2": ChaCha20Key(
        "c88181e0623a2c40ce02dbe94fdb909c0416bcb6ba2226b15728448f3b87caa2", "3e5326a5d76b9b88866c58b4bf21c759"
    ),
    # FGT_500E-v7.0.13.M-build0566-FORTINET
    "94495a94bb6ad81ff13c0392290a09194ddedd8b80c72d24a3b049311e8f116b": ChaCha20Key(
        "71a07633b1ca42de68465b7b3faddcd62ce5bb17ec570c8d3196e8ec701b4519", "e91223495706f0ca8d24b2bf21a56877"
    ),
    # FGT_501E-v7.0.13.M-build0566-FORTINET
    "70509903e810a12fef2404e973195de7382a9092212691865a6299ba4ba8358a": ChaCha20Key(
        "247c38ea8967897f1ee6d7f75f3eb994249a43782b0496938491912aabf54784", "d3a7f53fcb7e686b47e1401b128ed095"
    ),
    # FGT_600E-v7.0.13.M-build0566-FORTINET
    "2a5f91a655f9500dc5ee156aa2386eb2654735a720ca5da7483aca2da46ffd72": ChaCha20Key(
        "7b4804f94c82ce624e8aeeedcc5685e3a91c57f4db9fe06b2501c867f9bc530f", "d10e6babc87f1234fdab5492ca1e2681"
    ),
    # FGT_600F-v7.0.13.M-build0566-FORTINET
    "f80443b21d3f032064fc4f4da78b07e5fb4060819d66f397253914d818257875": ChaCha20Key(
        "a7465746a1ac47b88fed4b9f9fcaec9cf0fcc44b7084db01c68d639593aeee82", "40401c5c7195ce3eb3f8311b6d0d055c"
    ),
    # FGT_601E-v7.0.13.M-build0566-FORTINET
    "f5295e2b8bfd59f9ed151f37c4cda16d236a62734fc58c735d085e2a109939f1": ChaCha20Key(
        "0e550e8185b0a1bb982a429f3e22bd7f126861ee311ba056fd237287c2ef17b3", "3a9295b29c3721812ed2d6c3ce0e2bbb"
    ),
    # FGT_601F-v7.0.13.M-build0566-FORTINET
    "1c263af26819549f55a8d441d3b8906b954f615edc054e19ea0ecff647fb4864": ChaCha20Key(
        "01e868ace8395588c8ff8e6cc078a4c0541691360d79f9c714e54fb103c03b21", "2e8d6892f402c8e495ffd3d356fa5605"
    ),
    # FGT_60E-v7.0.13.M-build0566-FORTINET
    "aaff977486edeb0ae188426e715ed4cca828cd6909e4e2b8f5135fc4c1fed219": ChaCha20Key(
        "4eb9f6551021f8347382b3a777a1d49bc357061183e6e0c754ffc9faf89e5277", "629627818dd4c147454477d27f2d279c"
    ),
    # FGT_60E_DSL-v7.0.13.M-build0566-FORTINET
    "cc962ac75467eb42f79e421c8b393e30037d5531d191359eecb1b506f214e5d4": ChaCha20Key(
        "e204d9c6c5f53480548dd68037b87be1a028a0c445c3eefca29985dd9d8c4b51", "787e897e14d33a7eba639b7edda6a722"
    ),
    # FGT_60E_DSLJ-v7.0.13.M-build0566-FORTINET
    "5541a7087d4700185eecef1b61cea1e699479a743e098052a49b4f2b7679719c": ChaCha20Key(
        "9d09a11ee1b627c2aa844f97314db313c673fb5e0cd87544e684cb7a94f14b0c", "f4682350332390198d6883a9339cae3c"
    ),
    # FGT_60E_POE-v7.0.13.M-build0566-FORTINET
    "0ce1813f876eaa0e5ed7af23c4a78e03acf719cacef2de0fd741f64b48307971": ChaCha20Key(
        "f954f2d49077844b718cfa659c2bf05fef817fd6b72b91c62468223df2479962", "74fe9f90a0b0156476d3fd4ea4e3b458"
    ),
    # FGT_60F-v7.0.13.M-build0566-FORTINET
    "d1b159238795ba3d90e864797df2bedffb55ba906ed78a839a299407449f7708": ChaCha20Key(
        "a74d31b077b65a9bc47202b2f00111892dfdf74caa82ae46551081ae505f8c8e", "f0c10498648d5704a85fc04096cf93a2"
    ),
    # FGT_61E-v7.0.13.M-build0566-FORTINET
    "1b5ea9f257bab2ad12d838ed77b174aaaf406fa9f1c5e6e828d924bc3bc64620": ChaCha20Key(
        "e8660b9ea4f2d9155947702e1ea1cf6b253288837eabe3bfb807b5df6f9d9690", "de295fd85ce8f32007b05f68eb3bd30b"
    ),
    # FGT_61F-v7.0.13.M-build0566-FORTINET
    "94dbfb83eb2e48b63a303f09f54df4b36051dd09e3602b122a669b4da9d723e4": ChaCha20Key(
        "e96e24ae4db6ef369c02cadd0e1b6cd85ec4a6f9e6a3b0645f5f7d6ad1fc2737", "6287699bd1bad08a9b25618add09a8d2"
    ),
    # FGT_70F-v7.0.13.M-build0566-FORTINET
    "2871da81afe338742889ad715d97b7fc4a54ab5f8429a22a5f697d190016519c": ChaCha20Key(
        "2b5bf41a186d0d727144a137cc5a83b91f030b77b0b299fb4220136827bb773a", "56c998762967f0ae28eb173a8f9fda6b"
    ),
    # FGT_71F-v7.0.13.M-build0566-FORTINET
    "112f796d7994d2727a59a5bb1429de92b4671c61fa86c117333dacfc6882757f": ChaCha20Key(
        "41f7097b5f257f49878b84477b3b8ea45988bb312ef291473eed37ea67ea6378", "618d51366a9efafef2177becad74032c"
    ),
    # FGT_800D-v7.0.13.M-build0566-FORTINET
    "374ec1f3a08de1e06df249a700dc187ea6d1189d4c8aadcd0a8b8eb723d1cb75": ChaCha20Key(
        "9f9829e9afc7b69910ae9dc62b096af62ad3afc0c9343a95a10f5670581c78b9", "a02a0c3ffb918f105689408e38a19053"
    ),
    # FGT_80E-v7.0.13.M-build0566-FORTINET
    "a2bc20cadc423b50b3a5584a74be07def7bc32644057b1f6ac51bbf122f3aba0": ChaCha20Key(
        "f26f47c4dc7efcd8c57c8757386c58207149161e276b247357d1b28b5aaf374e", "57fab0a22ff8ce2278ccc24063b05d73"
    ),
    # FGT_80E_POE-v7.0.13.M-build0566-FORTINET
    "652bf6ea845b3e474c289e00241f577e880bdef9bd2838288b58497091666c9a": ChaCha20Key(
        "66277899bfc6afe571aef4584297986a3561a4072076ec15350ea4dddb4ec540", "ede8c9e7ad3feed7f8d4c0ed9cc62866"
    ),
    # FGT_80F-v7.0.13.M-build0566-FORTINET
    "e9c8373b6a9adaa14bbbc8cd311d8a6e7bc16c059696a5157463ba466509ba67": ChaCha20Key(
        "92ab38246d8705361ceab53a1e451b0772deb9bd1638f30d334944cefdfa22aa", "15611e1c16064ceb65ac84acfb7554e0"
    ),
    # FGT_80F_BP-v7.0.13.M-build0566-FORTINET
    "329df2163e7bdcfb540b3b70484a9ee83fe7a0c6470b0206e0ab7a8e71f16abc": ChaCha20Key(
        "f4c572bcce5ed2feaaca12c4655e76e37f4338401eea3a5223a4ff6a3ec93b92", "e178b6557a8ff61590b95b5fcf1d0ccc"
    ),
    # FGT_80F_DSL-v7.0.13.M-build6913-FORTINET
    "0ca45a1ab57d155f6a911ade44f249e81212a7670295cd6123d9a3aa468d1e01": ChaCha20Key(
        "da69cc0da200c1602869cdc46ec23b5a05006a2367b73360fbc7bb2e326abe89", "ea34cff38c0db76c50617df078fcd604"
    ),
    # FGT_80F_POE-v7.0.13.M-build0566-FORTINET
    "f939bc5b5eaaebc9c54fde6227f5a1b555f489ac39b1eab89e7d58b69b64be99": ChaCha20Key(
        "b7d8eb2c3cb87c968cca6c0b3c10729f66613dc51ba638e5ee06f8a434aeef51", "b12b54f9f3cf2c3646a600e73a8c0333"
    ),
    # FGT_81E-v7.0.13.M-build0566-FORTINET
    "0ae8998dee4179e630bfc5cf76f08b420a9ee14d223eb6fc93a3370661bd99d1": ChaCha20Key(
        "e279a23a377e996e6bb6640e1a387837322ae263175f2a23ee3409d69bf9c590", "7a76a494e3636f48da0d098ad928227d"
    ),
    # FGT_81E_POE-v7.0.13.M-build0566-FORTINET
    "a3837387a999faff93f3c3aae9ddb5fb0f9e89404ef839925cea8f89e5b5db74": ChaCha20Key(
        "89027ab0db5460eadb1c4671f673aa457e55ac847a6a86fe00d7f18e590a346a", "3aa592b1f33cd5e39caec1fd155f063b"
    ),
    # FGT_81F-v7.0.13.M-build0566-FORTINET
    "64dd486045ff5cbcf5d568bc352bc4185d5a3ab71de287a0f6613f4662a7697b": ChaCha20Key(
        "12fe5d67a2d0e12d8c490feab84efed7f31b31fdf599dbd8f7c73d6f99fc356f", "7eb8e19e71e28bd56d0bcb4575b00c1b"
    ),
    # FGT_81F_POE-v7.0.13.M-build0566-FORTINET
    "849bf32906cac7120d4f6d61db2aae884b702cf2a443ce535374bfa71953a473": ChaCha20Key(
        "e0a3ef7c0bb58dac84181c07cb19d732ab688bf40f2ca8c42eee8230aff5eb08", "9081ab12ba8037121b6031375d3b5b75"
    ),
    # FGT_900D-v7.0.13.M-build0566-FORTINET
    "f8ad6eed3f1c7cc081d8dd72fbaf33b03487bc2070c7496d9bd20785379dd77b": ChaCha20Key(
        "41af3119409c049a63a9ef0ca1c555d5f6b0b42d130bef5901c789ac9714dc91", "b3bee3bdf148e06710cb8f572f3936a9"
    ),
    # FGT_900G-v7.0.13.M-build6902-FORTINET
    "e6e0623d43256af5a7068a7c9a21ff9e7910854d4f3b6c7c20560f9eb0e60aea": ChaCha20Key(
        "6415cf0046f44d713784ca21b515ae4654985136fd244b450e884c073388553f", "99136b9186dd0dfb1551aafbd9ea65a5"
    ),
    # FGT_901G-v7.0.13.M-build6902-FORTINET
    "f54839e997804ab50cd349cf4565f266d17c734009af2ad544155e212053f6c4": ChaCha20Key(
        "8d93284eb813728c8672c8b7a812e7a3d316563840ebfa4204d3cb7cb53f20c6", "51e14ae844be401a8ad153c27649ff02"
    ),
    # FGT_90E-v7.0.13.M-build0566-FORTINET
    "7fded80e4fe0fa91f15b4af47b4785f2c0522941fbc9e8fee9ff0ddab526f60a": ChaCha20Key(
        "68bef1e1e2ec798257197650e1de29fa149aca1be15091fb6171e03c8538a667", "a97db76bef303484b6058628a6fe41ef"
    ),
    # FGT_90G-v7.0.13.M-build7121-FORTINET
    "468eb7068023035aaea727f45c0c3d5035ea2cbb0fe154bdfa152957f1d95e8d": ChaCha20Key(
        "5241bfa34b6f09a342fa47cd855540b615fdb453c1baa0aa1b4077164dc7f0fb", "bf5d34d6f81a21f10d739c0792cd29d5"
    ),
    # FGT_91E-v7.0.13.M-build0566-FORTINET
    "29d69d1119fe46c3885953a3492885bf3fb4127bd4432614d5aeb661a41ef2d1": ChaCha20Key(
        "de7a9d7454e54d96e9a3ca3b2bc8805cecd9ba74e86cff9c0f65670e8a911666", "311e58c3aa57ba40bcb90ec13d322fe6"
    ),
    # FGT_91G-v7.0.13.M-build7121-FORTINET
    "d256aeb88c09e466235f37fb297cf6ee0bcfa18430b7786046942ee75e0a93bd": ChaCha20Key(
        "bcd8a2341e091545385a683f921b19e1b8fe1a0fa5fef6597a1830653763f3fa", "19576a40d56c587aad4470620a67825f"
    ),
    # FGT_ARM64_AWS-v7.0.13.M-build0566-FORTINET
    "ad90c19c11b13898023f08b8b5963f06d2555b1187b04ee47b290e8ffcfe3034": ChaCha20Key(
        "edfa61bb14e535f4e20676421618be394c97d91ce07e8106b0198e3ff69fc477", "559208f9230a8a9c8ba768f2c4e5af89"
    ),
    # FGT_ARM64_KVM-v7.0.13.M-build0566-FORTINET
    "6262f9948fe53870b144326fe50dfdeb202b81461609bbea54b91a05164f7585": ChaCha20Key(
        "6b4244062dd92b3333591d97f810632e79c1e9a940487b59edd76bf4736a2ffb", "d6449ccf89fdc47a66e8524f5a81808a"
    ),
    # FGT_ARM64_OCI-v7.0.13.M-build0566-FORTINET
    "c7d6e53eb599083dfc991b96590137c94d7ac4b971c97cec54ae77ee11b3d29e": ChaCha20Key(
        "7d8d4aa7f2477e5f9ba6de8266700dfcdc7d0379f3ad182360cbcf53fceec4ff", "c7a6ec4c727ba8f5a35010b3b9dfcdc8"
    ),
    # FGT_VM64-v7.0.13.M-build0566-FORTINET
    "25cb2c8a419cde1f42d38fc6cbc95cf8b53db41096d0648015674d8220eba6bf": ChaCha20Key(
        "c87e13e1f7d21c1aca81dc13329c3a948d6e420d3a859f3958bd098747873d08", "87486a24637e9a66f09ec182eee25594"
    ),
    # FGT_VM64_ALI-v7.0.13.M-build0566-FORTINET
    "2aa7faceb9a0b29235ce64dab0f552e9433fcd6081818710726593584b2e77b6": ChaCha20Key(
        "60fbcc1a35d877256014517373862d1394184c968ea08adcd97fe01ce4a10119", "5bfdddc7fcbf327e3669ee8112c100fc"
    ),
    # FGT_VM64_AWS-v7.0.13.M-build0566-FORTINET
    "c907c6237a14a52c60eaed344c910a0e1a2e4cd017d60b2406bc18d23299a245": ChaCha20Key(
        "6740da9b78014fb3e4c18cd748112924bdf39b973bb876ff2e19f30f2367c627", "404168e81353dd9a10401e081d74f6e2"
    ),
    # FGT_VM64_AZURE-v7.0.13.M-build0566-FORTINET
    "8d97cb5ec598bd81ceb8791344f89206d26c5d5752a6ae2ac942bf38695e5466": ChaCha20Key(
        "9368b7c395089ab7a972172b4382cb10d01760c0138db8f758a988bf0899a727", "8db1bfe592c008749a8394a9b9e141ef"
    ),
    # FGT_VM64_GCP-v7.0.13.M-build0566-FORTINET
    "4647a0de16b20af15e407db242eabd686b86fb5d1b8436be38df69803e80cd1c": ChaCha20Key(
        "6c42fd57625f5f094c6ffd4d62691536212aea938ca5094c6f4cb27acaed78aa", "9bec66d11d95acce653abba5fb2113b0"
    ),
    # FGT_VM64_HV-v7.0.13.M-build0566-FORTINET
    "30bafb7b6dfcfa11b15315582d26a43e52351ee9a27797f45400f3a9460c8969": ChaCha20Key(
        "c83a7d7fdacfa595f9658d68f6e88194745e5482ad596c42a9dc835a65bb62bd", "d5f8e9319d975f75f5d950cfd1aa2e23"
    ),
    # FGT_VM64_IBM-v7.0.13.M-build0566-FORTINET
    "3b843a585fce7f0e34cbd1fbae92f7d4eb4ad24c5634665e4196a864a2774a89": ChaCha20Key(
        "dd52f130c8ca25a109236aab5e8f081e4e2135be0d7771c1468a3fbd51d25975", "0c90827000347dc667f9116b5b3752d1"
    ),
    # FGT_VM64_KVM-v7.0.13.M-build0566-FORTINET
    "96404e366ac402a1abd895ad2ff9bb7520fdf50bda8678a7b80b6c87bf547932": ChaCha20Key(
        "e9f117b3930abdf36eea7f0419c4df72b30570285c91009e62a8515266b9a5eb", "1c13717cef7e461f862db077618cd7c1"
    ),
    # FGT_VM64_OPC-v7.0.13.M-build0566-FORTINET
    "e91ba1c7f125066ed7a3b57855892592af9a9c2b7a3860dc04d269adb6a09088": ChaCha20Key(
        "d037361be71ce10f4af05d33caca77077d17de1c7dbff2e604ce8a1aaa5fb628", "5e5b847936c9567fc34108a995391a29"
    ),
    # FGT_VM64_RAXONDEMAND-v7.0.13.M-build0566-FORTINET
    "277cf57bcf11003a4f50df85f2cb9fcb44716144588052d4f26f8881d4ec8afc": ChaCha20Key(
        "c296c98c3f4ae9dc4853f57245dd2ba64dba97c12980474b31206facdb4201b1", "047ebf84858974abdd159346243f0116"
    ),
    # FGT_VM64_SVM-v7.0.13.M-build0566-FORTINET
    "c504304a600969ff47da1150a65a9e709c54defa099f1f3a90f13b383f43031a": ChaCha20Key(
        "c875b60a71414b0d0a3c043cafa34a65a52c3d24bb9f87ff0397fc52865e6640", "9e7ad15f4183538658e324650f07f678"
    ),
    # FGT_VM64_VMX-v7.0.13.M-build0566-FORTINET
    "5459644455c356b823371cc20a0140ff840e9b489fd71419da3467323fc9deac": ChaCha20Key(
        "6924444d45dd76329a2c0670241e9e3dbbf1d6049ae17854fa17a397196282fa", "e5718d29a2b653c9b1a4a18a462c43e9"
    ),
    # FGT_VM64_XEN-v7.0.13.M-build0566-FORTINET
    "a5261d354a6076c91fd434f929f53313b6507bd242911943efc7cb17511b01df": ChaCha20Key(
        "3bafc3503fd83f94c4ca07dd4a5fc6d93beb5c6fa96b9285529dc718801ba1bc", "5933960d17ca7ffd8709a8e46cfc66cf"
    ),
    # FOS_VM64-v7.0.13.M-build0566-FORTINET
    "d0c78b38525ddd20d098f813d26ddde80ab7ddbeba764fb85d25d79c423cc54d": ChaCha20Key(
        "f6bdb0078ff41b1579cd1a93d390206346da620ebaec5a17dba9b0f120259b8d", "438639432ee6c3fd2a2d6db6989bb4b5"
    ),
    # FOS_VM64_HV-v7.0.13.M-build0566-FORTINET
    "d63111b1c6920149b5b1b3879e7520ab4bc7af8935f30eafe30d27ba1459a2f4": ChaCha20Key(
        "74b5cdb5e9bb544e8f6fd8393428c4cae010d08c7c5dc0bd3acebcbeafae6e4e", "7e3b61ea6d78671c48d8aa242b611f83"
    ),
    # FOS_VM64_KVM-v7.0.13.M-build0566-FORTINET
    "ea073d4d7ff0e75dbc3421aadb1eca897b62fcfb1d56553fec267f4657f00f7c": ChaCha20Key(
        "1fe82df074f717e934ad85b48c33dba80266be8b99c5276d0100ec50fef2e19d", "03bbee6cc9479d5d5a8713ff87cf20d8"
    ),
    # FOS_VM64_XEN-v7.0.13.M-build0566-FORTINET
    "9311c7baeea78e59f076aa60dbcebc2e66715e631ed66ecfabf7fe3462f80d20": ChaCha20Key(
        "9742d94d5fedaf05ac767f98ee956b13426158fbff2480bbe121ea3f6740ded3", "4483a4e993b948f9c73b227c3ba8fbd3"
    ),
    # FWF_40F-v7.0.13.M-build0566-FORTINET
    "5118227e08cb201714724f33bcc10b2bc970f3dd742335f329ec16269d114196": ChaCha20Key(
        "6c4c0be41385247f7a8c480f16a466dfcba89c2c2f8e2c15f4c98d28d658aaa0", "d19f4f860cdd25163ac52a31fd5d3794"
    ),
    # FWF_40F_3G4G-v7.0.13.M-build0566-FORTINET
    "306752ab7a4ba8c94c8de9c3524c22f1c5fafaa9cbf1c65b7a65cd295ad46c87": ChaCha20Key(
        "8168872e5f5217cff85ca81c8d3379341c4a0c47554e161871ba5663226c7589", "4854d7df285ba6551712c3674a8106f0"
    ),
    # FWF_60E-v7.0.13.M-build0566-FORTINET
    "7d4b8ab9895409bb9aa2df1b3af75e45cce5d29762b01659ec5b294ff9be13ba": ChaCha20Key(
        "a7404b93e0d3fc1c211f135599a5f651e9fe2dedd5d9e37d3ef939acc37ad629", "3934a524d7f2dfe2a8b0ff02ecd4739f"
    ),
    # FWF_60E_DSL-v7.0.13.M-build0566-FORTINET
    "e20e73a1c133c55d717d6dc97b95aa88decea6e241415891c3564cccd5199cdf": ChaCha20Key(
        "0307d336e257e57b97ee8e3fb7b76a438e181314b595630dd04000cc9437fd58", "46dd291a40fd2f86d165f88268fd5912"
    ),
    # FWF_60E_DSLJ-v7.0.13.M-build0566-FORTINET
    "29433d147bc102b7dfb802569eae9686744b8d17db6fb2045de0b98eaee7ad62": ChaCha20Key(
        "89956b19012081265d3a668d9838bfd869f4c363fcb5c45a07c405631a68b732", "bc0e07adc3379711caa860921e31ad4b"
    ),
    # FWF_60F-v7.0.13.M-build0566-FORTINET
    "a7d1aa6e332393d775cb679e5363252f1ae3417636f9b4b0997b7f175f3c7fb0": ChaCha20Key(
        "97a5e989988603360a63d41a14a6a96c33705836b514e6830438d1404e279ca7", "e3f91c9af5ec9d723967d6e6a5771ba1"
    ),
    # FWF_61E-v7.0.13.M-build0566-FORTINET
    "ada2ce4d14e30292beb27ec02007ca7107adee3d555a36df80c089ff3e0f4a0b": ChaCha20Key(
        "fb14b50bdf88c209e076419ab597b865331f4e325124d3a72da8cdfbbc131dce", "ac5f21e5926c201e44ad440a06084660"
    ),
    # FWF_61F-v7.0.13.M-build0566-FORTINET
    "068a74274af6417a91bcb86bec10a25cce321e4e1e2524b0eaaf75e88d95fc7e": ChaCha20Key(
        "82d66c74612154ff1728d60cdfbf08d56ae2e7233cc13d368640fa1f996ce116", "b922fe8b5a44e4d0475a7d09baa0ab09"
    ),
    # FWF_80F_2R-v7.0.13.M-build0566-FORTINET
    "aabed28bb01c70f08dbbcd942ff7b4c387de3ae3ac4db75205c357b11c409cdd": ChaCha20Key(
        "501519616bf615238e7d776075bb39a6c50c1748c207973a56ea65234cd06c07", "2851f9359ea241bd71cc200e30fe7fd9"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.0.13.M-build6913-FORTINET
    "380a47aab4cf26e5c97aff3c6a7c9dc7537c92700273c7c83f35d35ad6549d36": ChaCha20Key(
        "44665877ceaad076806bfbe617b4b8ca640743b4b27984a488a2581e6ef4c1a0", "d361b4d1159f96e3dbef7910032470ee"
    ),
    # FWF_81F_2R-v7.0.13.M-build0566-FORTINET
    "075c91613dd20e85c55b089e6efeb0e581de28018539be63585043d082153849": ChaCha20Key(
        "4127936245f6096dcf973733e8072d51c92f99f1d0b658e9f989601f4924f1a8", "0300fef3b16b934b7ffafece635e750d"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.0.13.M-build6913-FORTINET
    "1a8e28b07188612e30c1be6fefb00fd330d09d3750cb6151200302181fa99f00": ChaCha20Key(
        "3fef0f82540050deb98b02bb7f1d0bedc343bfeb597031a30e93a5e173f727fb", "249d803e45f638911a95bc4228183c7e"
    ),
    # FWF_81F_2R_3G4G_POE-v7.0.13.M-build0566-FORTINET
    "e932f01b5e6202b98b0bd8764dc30c1291155b57bba515ff17da2b8c0a6f7eb9": ChaCha20Key(
        "5e43f99a8b0ed7572b7de8682b79ef8bde6a8039dd01cbb318cb914f6c29d144", "084f9f0e83109e6c607c9678bae260f4"
    ),
    # FWF_81F_2R_POE-v7.0.13.M-build0566-FORTINET
    "d3aa5a52ad06be3df86672784e3f76697a58725f8a87b7fc86a8674ad2fe9be0": ChaCha20Key(
        "8b1cd9aab318e9d72bfc0670ddcf5a21bc881bcc86b42a4777fcc6abd38b39a9", "36d5895a981f092a2b001b10a4a467ee"
    ),
    # FFW_3980E-v7.0.14.M-build0601-FORTINET
    "a494ec1713ab75a5ab58a847f096951e2de7ba899bef1a9a88a9c94d8efc4749": ChaCha20Key(
        "bb48ece8482e277f307479b8923796aed3b536e83f5fadc0d758b36192626762", "5394687ae6c679a74c7901267dfb9bb3"
    ),
    # FFW_VM64-v7.0.14.M-build0601-FORTINET
    "4f487c3f8fb31ec0cfbf881f38c59b87933a0fd43713963efad309990b18bca4": ChaCha20Key(
        "de54d7020140ccec9c01785f64696a35ec109129a5a8ff22f61dabd6850d4644", "bcb87a3cf58b614f83d6d34b077b5468"
    ),
    # FFW_VM64_KVM-v7.0.14.M-build0601-FORTINET
    "1c57cb3bcb646bebdbf9140e65ddfb3cb6bc579f60a900fe77a623f74944dfc1": ChaCha20Key(
        "f02f10bb43f0f0513dedcbe8b4db78bee0b2d7ace75e2ec6cbfbac6e65343e88", "5cd3b34d491f99f5d30b3f676ccf3c9d"
    ),
    # FGR_60F-v7.0.14.M-build0601-FORTINET
    "a61be94a8f3d89e854f7828dea1f5342386af191b5adeab1f6d5c90144ed88e5": ChaCha20Key(
        "c2c5cf9c929a3787a05d30035cdbb6f85ed79dfb941fd6b8fbc4ad2903a68fd0", "f8bcdddffb3e1673eb0450660afc7c41"
    ),
    # FGR_60F_3G4G-v7.0.14.M-build0601-FORTINET
    "7826b6d3878a17b8d631bfc8312b48fa241d1a0de0e4f3694847963911c9316e": ChaCha20Key(
        "922ded4f6f5ed745be172bbe803fd795ec51be398cce3738e31fdfc1bc3427b7", "01a6a463513d921edd183a98a62d1cef"
    ),
    # FGR_70F-v7.0.14.M-build7175-FORTINET
    "10fe8536c66586bc7ad51a3dca7038c5111121810e42a5f077883bb6b37bf988": ChaCha20Key(
        "1f9b8023ba39b82abd2876c63fc9937d4101795007c4e57f2a7a44e79b6618ad", "d01b87c5b1f75d595b5dc17ae45eab34"
    ),
    # FGR_70F_3G4G-v7.0.14.M-build7175-FORTINET
    "dc6fd1df4db0421e679a99b03f65620d83de55d46bf5fbc9ba942a8f764cdd2a": ChaCha20Key(
        "ceb51821af0c8265105fb4a2656a96b70dc5998b7c4d067931e884af07c886ff", "d8fc07500f7f230460fc67c51b5817f7"
    ),
    # FGT_1000D-v7.0.14.M-build0601-FORTINET
    "90f32843f15f73fc0a47020ba855a386b4893e2c0f511ae40f7c8778ebeed447": ChaCha20Key(
        "eef595024b146555c1f29e17a6ad9f129840e3c4e465987122de37355e1df88c", "bb2bee68707ae99663a5740cf8b20c24"
    ),
    # FGT_1000F-v7.0.14.M-build7164-FORTINET
    "b8dec3b4ed4e0cf1719af453d7fef6d7e23ad24e9663501f8c766d9dd3167aa9": ChaCha20Key(
        "b6bd362e23408f5eda689d43471121dce279db02b290883bda1539bf43728ca6", "66919a4fda61b58275e6c397b04a2f81"
    ),
    # FGT_1001F-v7.0.14.M-build7164-FORTINET
    "00b0c8070fa41f1a78f0282c9f6db86318d91c245db15282ed1c5563eb960df4": ChaCha20Key(
        "0c01854b6b4a46f143a6a7546fc9d91c6bde09bb0cb0bff0489dd0e36636bd32", "65506b02b4b168f36646729899b38543"
    ),
    # FGT_100E-v7.0.14.M-build0601-FORTINET
    "735236bbc9954ad998c396ba772f8b027a4144025b9a843b90fa550504c2ea94": ChaCha20Key(
        "f0f1f4c444d96d4b54430fad0d1f2d54ae0f6adbb816e26b8f6773a3311eae90", "64097ef98482d85b875b5ecc35f65fe5"
    ),
    # FGT_100EF-v7.0.14.M-build0601-FORTINET
    "dcc0b24fecaca4488f6580cac4c43b1bd0b7dd7324456bcc73e25afaaee93885": ChaCha20Key(
        "6e45bd1af3f2328e07ef245e1614622b1644811dcace014c3ad9068a2476ac2c", "58d592dac3ae404c9207592147eee490"
    ),
    # FGT_100F-v7.0.14.M-build0601-FORTINET
    "7d077857e8d4f8c53727f3767de9ef873e73231dab91a42001006b7e21862a6c": ChaCha20Key(
        "c0a4af4e0d6e7ef2ec37bf01357233b383d2d071cf4d5fd749d36b76667c628e", "36bfe96a0eb980c96521500917b23777"
    ),
    # FGT_101E-v7.0.14.M-build0601-FORTINET
    "9d724eb129e2d4313bfb69132460e04956937726a3424dd87d744e15386c70ef": ChaCha20Key(
        "34d770d12ffebcad143891e605df1868bf31129dd5dacb341b73dac8aa465918", "3e850b2baa616cdc1396d916243ba8df"
    ),
    # FGT_101F-v7.0.14.M-build0601-FORTINET
    "ca0f7d7209638a905f1773d7e90a0409747d56ff745ac3aba91f4674462b8f5e": ChaCha20Key(
        "b4820e3aa8c7a62f65c0c26f5a7bd2337e2743db5ad716b877271facbb0eb181", "de2030b3c257f8102289b0a78fc5e074"
    ),
    # FGT_1100E-v7.0.14.M-build0601-FORTINET
    "5d3d611afc5eddb3bab82a702a790ab3ed2e84e1a406c1c76283cfce451888ae": ChaCha20Key(
        "271e6a5af1a7ba95955af8a43e7f53dbc51f2f0aabbf46eb67cf076141c0d79b", "d65a8fa834621b5fc657cea46354e5d3"
    ),
    # FGT_1101E-v7.0.14.M-build0601-FORTINET
    "3baca8d39c545b19095a5fe411a4dad59c194a16c6b16335c39e8b24db0f138a": ChaCha20Key(
        "9f7fba2d0da094f2f1dca713406a03c94a0e6e09423392a355e7aafbeb72006f", "4441c2ff05c141263effab494cc38bd5"
    ),
    # FGT_1200D-v7.0.14.M-build0601-FORTINET
    "97d48b909b13bc60938605d1ea353ee575f831c87cf5839856d7775e71761c74": ChaCha20Key(
        "3e3ea73b9c49c0984cf35d000b0dfab6a583cea62609507967971c8c79f204fd", "1fb6cda3184a6290d819457c0354eaea"
    ),
    # FGT_120G-v7.0.14.M-build7212-FORTINET
    "641e57836b9872984b25b73797fd012af30d3297a77791ae1abd5b24f5094596": ChaCha20Key(
        "a9bf96fcf279cdbec61dd6a66ac2a4547a7c598d6953ca4242132250490f0396", "14af2d4af8fc6f1130af3807fd517ca4"
    ),
    # FGT_121G-v7.0.14.M-build7212-FORTINET
    "c99cbec6d296a36fa5b8a4c88bfcde81a3e05946ea285dd2fb62778669a945cf": ChaCha20Key(
        "1fadfdf8cca2aa74f82a80b1674dc3dbc542195ff048abb049bdf33993f9d424", "31900d7d1900b381d746b7d8e5e27e46"
    ),
    # FGT_140E-v7.0.14.M-build0601-FORTINET
    "31dd7a552850a316df085d1978df62e55db7fa7bd0710467aa1f02bd9605a9c6": ChaCha20Key(
        "21319908cc781e96ea99f2e48f26e486da96a00afc770af390e7c98141a1d5ec", "edf75cc42bd0efbacbf52e1fb99dcc69"
    ),
    # FGT_140E_POE-v7.0.14.M-build0601-FORTINET
    "2b512ff51b4d867d768daba763dcb17f9508f24bb97eb1cf523cc55083af5a0e": ChaCha20Key(
        "db9a1fedd4e677e1ddd982795ac6c56fafffb4a74ff19a79aa227adc96294a2b", "0f5d34f983b9d965910dfefcfddc0459"
    ),
    # FGT_1500D-v7.0.14.M-build0601-FORTINET
    "47784c7a35366df7f37285ad71fa388b738ed32ebc5f3b8106b4bade58c5761a": ChaCha20Key(
        "cf81d17db1d2aa12d4068741adbd3b64c579e71997eb2851c92969301c837b5f", "351c2e60f57d05a7f6e1de15a1f81de6"
    ),
    # FGT_1500DT-v7.0.14.M-build0601-FORTINET
    "2993b578f1ded6bd5ca51db8c0066feb2656497f85ee10f54fc77236686cf760": ChaCha20Key(
        "0e41d56148d0602f226c4090570fcecc6e8b91a97d551c2a2e2e3c66ad189148", "6c71792d7651144448bc8f60034482ff"
    ),
    # FGT_1800F-v7.0.14.M-build0601-FORTINET
    "56e07f7590bf925fe957671a1a6584633b7a3e9992cd0c8d2c02b1f696e6af9d": ChaCha20Key(
        "8f797901c90c7b029e72817a846516dc44f5269850e5dd76a3fc56d4894447e4", "fd78e7b5cc93984d473523847cae442c"
    ),
    # FGT_1801F-v7.0.14.M-build0601-FORTINET
    "834b193bbfe0053c52807bea8f572c802e5644c2f1d5e969168677d3d45ecb43": ChaCha20Key(
        "cb3a2ea4e1990f03058d8979cfe804eee984176230e101bd4e50203325319057", "edcf71c6b1998024a9ba569a1280e762"
    ),
    # FGT_2000E-v7.0.14.M-build0601-FORTINET
    "ca45f4ac8491b51c6819b5dc003f3bd0a4fa137c9bfbf82213ebb3181aef4e55": ChaCha20Key(
        "3694a331624f5bc9a39c3ba32eda046bf1351f3b0874d0d8c0c6e210fe39b437", "ab860b8e50468e76c09e60e2fafb6743"
    ),
    # FGT_200E-v7.0.14.M-build0601-FORTINET
    "79134729e422a25c5aca1f9c846c59925d17f0089d457303732a4e103104ff3e": ChaCha20Key(
        "3fa042301038e798fb24035b8de4d0628b4a6cc491d75ea80ddb95ffa0d3316d", "54e1bdbd993bd601076e2df8ca5bb915"
    ),
    # FGT_200F-v7.0.14.M-build0601-FORTINET
    "dcad268fc1c7d4b66f2f5957df41025a57d8fbae575eb9ea0f793d22dfd6a64c": ChaCha20Key(
        "610cc73ecf0556e0cfa649e31e46433e2b482120078886981c29a274a081e958", "01b74b46e0ba6dd7af5e3893a8e296d5"
    ),
    # FGT_201E-v7.0.14.M-build0601-FORTINET
    "790a0ba68ce3cfa20c926228c733d79e64a3ebbe8d42c400b5aed99f444375a1": ChaCha20Key(
        "d24378af8413cafdeb7db1475ebead81f3264a16c7c5411ab42a48279924e0a5", "82aafb890fbfd2594f0dfd65af5a3e8d"
    ),
    # FGT_201F-v7.0.14.M-build0601-FORTINET
    "4241d305621ea9753fe9f1a4ef731b6e31b1fabe32e3a1316e3e70759f6b05cc": ChaCha20Key(
        "038f66c326d44f5fdaf02b41aa0123fefd9cef9c13a7e101b1c823d759f8d591", "cf493277240ddaacab994fb2abe1c6e2"
    ),
    # FGT_2200E-v7.0.14.M-build0601-FORTINET
    "23623030ad1a66adc595ee730523a59f4557ce14501a79e9abae168ddf453569": ChaCha20Key(
        "bff8dc305e7c348967ec29c856a8ecd93a3d481e33ad776cfeee2d118e1825e9", "78ec6909df614aa0aa6f365e30a0704e"
    ),
    # FGT_2201E-v7.0.14.M-build0601-FORTINET
    "64dbd3a15bb53e46efed797da3d3bc26ef1735768b312bcbefa4a4699d5d6a96": ChaCha20Key(
        "0eb788f18fff571a78d0ebf661112eeebcf42985d24a1d2d4fcb83833fc79749", "ce482ab43f58590de5f78a3f924fa218"
    ),
    # FGT_2500E-v7.0.14.M-build0601-FORTINET
    "cdbee601e76482ab6f68767e62a9f5626e9e7a954870009fa7ea2255869ec9fc": ChaCha20Key(
        "a3c25168ddb00f08be86308b06259965c3a49dc745759cd1c689c5a80caf2677", "54a5afdf778890c91c03c5e9a46bcb7e"
    ),
    # FGT_2600F-v7.0.14.M-build0601-FORTINET
    "785ecba04eb830cc0f2aac7ce52270f3475f2f9fab7c6471161e8d1fef242645": ChaCha20Key(
        "0fbb70ac73c0688810ea604bf02f57532cdf93ddd5dd6d3a602f0af87d162cac", "46c5ac75b23e7158b50d17c6fa77b1c5"
    ),
    # FGT_2601F-v7.0.14.M-build0601-FORTINET
    "e692ef51d58f1f39aabe0875dadf40e71cf0491e633a37fce3dc494f67242c3b": ChaCha20Key(
        "9b0743646f95814dd5d7d0264b1618adb86ed08389048f6838f693e6e39ac411", "299bb180c40d747c6a63653658eedffa"
    ),
    # FGT_3000D-v7.0.14.M-build0601-FORTINET
    "5bca1b904c7063ab02e345da6ae6515e8f96f3152c6c6de84060a1a574a5b6ea": ChaCha20Key(
        "b18de35babeee5b9dce83859ede93ef19b36f2ab36814819adc872ac332175eb", "b4f40b1912e151ee973ccb14e3b585ca"
    ),
    # FGT_3000F-v7.0.14.M-build0601-FORTINET
    "55c51752a3948e64cc977b0c02d23e32a10a6c8b6a588a6499da7abd0f0474eb": ChaCha20Key(
        "7bb480ce594699e0ab143ab1d8ddf6e7d2cce15e9c6c0c3f0edde6677c08e04d", "514959fee7da51e2ce4737c32be4fe0e"
    ),
    # FGT_3001F-v7.0.14.M-build0601-FORTINET
    "8a7d70507e71f189d419d4f34c8ea5ba42ee42865579a24de592efb3b6d5a3f4": ChaCha20Key(
        "2a0ec397b4521f9b95ca660e44987465b66853d0857b3e213c16983f827715c4", "e92cd0fc03bfbdf70ea4e9d05e792a3d"
    ),
    # FGT_300E-v7.0.14.M-build0601-FORTINET
    "b5aafa087d85f33d90a459cfc5cfff2f73d5c22757c3b4367e7fdf447f76d87d": ChaCha20Key(
        "81be8fa438dd012b0d57fda5341d5cdfb00dc8c4ae7eda4b5e2969f1dea44512", "c9ebe2c0973cc1b5dc1c46303294fbf3"
    ),
    # FGT_301E-v7.0.14.M-build0601-FORTINET
    "da8101b29668a3166843c372c265735c5a6ebaa58cd25ca4a597e48bf93a6f9a": ChaCha20Key(
        "320cb1d84c8fb3fe31feda8fe915da39c132fb3b4246981b278807f17323dc08", "62870616fce11317393df1b3c1bf18f2"
    ),
    # FGT_3100D-v7.0.14.M-build0601-FORTINET
    "09fe9f336058d25c9b1c0bc8b4f38b9fa65e8e6f78df64eaceccfd0fb50dc131": ChaCha20Key(
        "103c9e37c30eee556d11598f48d690561c58469ab8f573cc0632463cff76e716", "fede082c5e15f2bd1b4cd5860ba8f050"
    ),
    # FGT_3200D-v7.0.14.M-build0601-FORTINET
    "538a99a8e194cc304d9a3d93d2e3d9fcea31188d4492c1746c049d26fbb91261": ChaCha20Key(
        "97fe630a0691da45b4acc33ff7496020357f5f62e4a9773f05a04d3db23480c5", "21d50ea7ace7caad370f0663e3135c8f"
    ),
    # FGT_3200F-v7.0.14.M-build7176-FORTINET
    "7983cf93dbbc6e8ff2bcd0e67a78a04b5891e1d0aad5a49c13f07910a68e3bf8": ChaCha20Key(
        "bced9f1f5b8f392a05bf755743d08daaf35669463ff58379710c032bf8fbd9f9", "587c9f562fc3f00c63b09345723f2cea"
    ),
    # FGT_3201F-v7.0.14.M-build7176-FORTINET
    "2b9d31aa8aafea08cc7b170f7fda8ce72b803d52c32b5f91f556a3785e01d063": ChaCha20Key(
        "f6f128bcad7713fea20c477c4809281c4374435531806101fea3030716212764", "00638743e67a160faf854def19adc3ef"
    ),
    # FGT_3300E-v7.0.14.M-build0601-FORTINET
    "86267ae529a9fe2b02947470f5281443696eca82d93f915a96d9c535e2ecd189": ChaCha20Key(
        "f76825473070a56a02537c73cfbe68671c596975e0c0053c0e87add13fdf395b", "5d01db8a3015a8c6409c7075379ec83a"
    ),
    # FGT_3301E-v7.0.14.M-build0601-FORTINET
    "67da66915d162960a1ca3b7a3cfe7d7c5cff507d0df66f739058b557389474f7": ChaCha20Key(
        "b08c3501c31b08b6f8fa5a3cb9a258e0e2a4921d8aff2a9393ecc241014ac52e", "1496645f163a5c65af25bc906bf8e0b4"
    ),
    # FGT_3400E-v7.0.14.M-build0601-FORTINET
    "8f652cca1c08e28f6256cef7f4c09d09e74f78597e4d8a4b115563e6dce554a4": ChaCha20Key(
        "7c744f554a32d822b3c8e9fe2207f3a074ed2e4f8c2151f1475bcdb2b17f567c", "90272ff7bd9e43e29e2ddd3fa38ee753"
    ),
    # FGT_3401E-v7.0.14.M-build0601-FORTINET
    "6f582bc1df1c454ec967f2035925f99185e4094ec358dbdb79a85cfbd930bea1": ChaCha20Key(
        "e509ff5c042fa1e6af0f26f235ce049fc160a237ebebc77bc1333d424c248af3", "bef0e582050daed2e0cec331a09fdb5f"
    ),
    # FGT_3500F-v7.0.14.M-build0601-FORTINET
    "1de873a2f877390da8315fc3ca1af261b58273aed150c82e9dc9d53373823534": ChaCha20Key(
        "d8edb3fe53fdb309f923781e999dcf484d1d7061cf12e092918251e9cce1bed1", "5deba686b4677cf6876d61dad87440e8"
    ),
    # FGT_3501F-v7.0.14.M-build0601-FORTINET
    "e3d7aa20c3d2f68ff0c6c7e7115958b6367b554b090ee960e48e31ad2438d395": ChaCha20Key(
        "233c0a4771e5676adc0b0af2c452d6c18ef87c1f6fba6989f25527745d199552", "79a1da1f5fc108e25cd8bee288038d02"
    ),
    # FGT_3600E-v7.0.14.M-build0601-FORTINET
    "08e521a98eb3d2949bf59ab8c0ec2f7e6aeaf24b7f7e5ed15985d9f5afb4706f": ChaCha20Key(
        "c0b2d809d6e5341545e29995b59863b0b56f177e6b46cf30e24cf66f2bb49e7b", "a4848ba55b689ba0b4ccecc261f2d608"
    ),
    # FGT_3601E-v7.0.14.M-build0601-FORTINET
    "bceb04119002b9f0cf4ef048cc9689ea9e77780f2719a3353e84de1203a3a075": ChaCha20Key(
        "00a41720002ded4539e666fee34c4c682cc5247c821bfd85cde741d57d9017bb", "f81715ca743bb3a504c45729df6466c3"
    ),
    # FGT_3700D-v7.0.14.M-build0601-FORTINET
    "868a7deb81f5b7b454435111a6f8a7a2023faab0da36959f45a23711ce6f1ae0": ChaCha20Key(
        "2193fae5de6a0c6730f377eeeae06f4f9052a5cced61cecd6921c2379c4c4441", "2fc9803fadfc85be0737201b5ddeabfb"
    ),
    # FGT_3700F-v7.0.14.M-build7176-FORTINET
    "65118e7148f0c96b609e806f2f716e3f6ffb53b3afe2f2e6b4243194e3afa5f4": ChaCha20Key(
        "4e61b67cd4a16bcd3ae7629825c345925041dec6d99e86d59a088eb3c000ceac", "e911964af325d46d4c53cc5244e20953"
    ),
    # FGT_3701F-v7.0.14.M-build7176-FORTINET
    "2b50e27bbbe248495d1d5ff0a19b1fc667983b44677379505147979a0e2dffcc": ChaCha20Key(
        "50b65669b264143688c0bddfbef5efb06070214cf24056f9bc75d9720483e3a7", "038e1452560345d2e2f8917dac4d08dc"
    ),
    # FGT_3800D-v7.0.14.M-build0601-FORTINET
    "b036c04c7d8cbdefb0b33e6102b826060bc8677c49330c5b6705f45a21e5e546": ChaCha20Key(
        "2cd57cc927aa7eee9a00d6a5084e711aeb612d62251fc232226da896113a9b83", "f97780c29ecb4413693bdb0909d4ce08"
    ),
    # FGT_3960E-v7.0.14.M-build0601-FORTINET
    "254583831bbb6c8cdc649a2d4c17f4498c2085da72863d9564be53bc5c544189": ChaCha20Key(
        "51d4c8ace0634b12e1b4df8eb1b248d789c2815245a3010d2f1691c795b83edd", "792079cf5e1f9c9b963be535131c5cb3"
    ),
    # FGT_3980E-v7.0.14.M-build0601-FORTINET
    "1cb3fac5aca2c7cac2991375c3ec0f34189b419a7e07ac67f3c83c1190aba93c": ChaCha20Key(
        "28f4e96035f2eaa269ac856c9a848f8cf35f6d562d32cc7fcdd1d8a51e3f421a", "4154d2d85720e0f75932e44eff190230"
    ),
    # FGT_400E-v7.0.14.M-build0601-FORTINET
    "51e663bfccfb0958b5428096eee50ee29851747ced44d342c7d5b22648aeba5e": ChaCha20Key(
        "28e2f4cd72fbd0caa6429e3d6832227174db8292db54771a95171bf00424112d", "645ebfef030a46ba326995aa908cd5a1"
    ),
    # FGT_400E_BP-v7.0.14.M-build0601-FORTINET
    "7566473b31d8688a7fdaaaf2eb6fc9ae189a97a0501c40334a401b76bbf08420": ChaCha20Key(
        "e84da6a08e3893a5b099ca1515f142e723e289650c7a82db8b420424552fd175", "0e4141cae9e27becd266d05a03c7ff24"
    ),
    # FGT_400F-v7.0.14.M-build0601-FORTINET
    "697276b4b9fba88a18231b2a5245f4bad1204fbc617b8b514d8f560af795eb9e": ChaCha20Key(
        "e227bf3195c0decac74a5bca4ccde526ebb873b19b11c1cfa145a57029f30266", "3dfc7e95249eebd95538cd6386902c20"
    ),
    # FGT_401E-v7.0.14.M-build0601-FORTINET
    "cd3c1c94dcebf799e4edf020fa4e5c1f3713b9e9b6c5c090ffdb245dbdee6452": ChaCha20Key(
        "77a27913b8d4a39cb5ae58785e3386f4b7bfd59d292820fe32492af341c14594", "4ea94276dfe239a9c0e503bc729a457e"
    ),
    # FGT_401F-v7.0.14.M-build0601-FORTINET
    "c390a83cb58328c717937ad0473e2eae5697b844cdfcd9b2a6828aebedc48bc0": ChaCha20Key(
        "3b030cf26773914e9ee9e78b53ef282c6786ee82777fbe2ac0ce158818943e78", "c7e1c67b6e45f30217c4de535ca9172e"
    ),
    # FGT_40F-v7.0.14.M-build0601-FORTINET
    "737173863495aef9f2018339a7062b7c82ce1bb2923e224a5df9dea372ef5500": ChaCha20Key(
        "228f9936477d2b24095e77dde352ea72aa11931819dd90b92ccd0efb492d30e2", "5c1effe6fd9e53a74b600fbf8ab93f07"
    ),
    # FGT_40F_3G4G-v7.0.14.M-build0601-FORTINET
    "ee3d4f0bc048a2747c1244ac7db85274f815aa4f9cdf91b63fa921987b214376": ChaCha20Key(
        "7037b6fe4c992592545dff0d045d8baab2a090f17f6baa7972cc598fe51b8669", "3c318d1def681577430f9d87dd3edbd9"
    ),
    # FGT_4200F-v7.0.14.M-build0601-FORTINET
    "d7bde4cd51e6fb22578dbb042d181024d334affcba3de9c92b680fa964196055": ChaCha20Key(
        "b785df1e426e60c16ca457716141d63ebf67f04fa030fe2a6eafba53b774f7fe", "c91f7c769afaf76fe6cb99b1254d35fc"
    ),
    # FGT_4201F-v7.0.14.M-build0601-FORTINET
    "dbf74fca4e58a2b29a85355ed792809c8019acbf31eff04fb3380cc511aae6b8": ChaCha20Key(
        "cd5d36eb204d4499ed8ec32f3b51a8055ea521222fbe73e055df020db3b0fef0", "741c200c0d82969c4a46461e7ddd2f59"
    ),
    # FGT_4400F-v7.0.14.M-build0601-FORTINET
    "b1ee8b3223939d961c6144058f761f5bfdb35850d200c2a7bfe7f14d3da72224": ChaCha20Key(
        "3eb592eff56b858dbfb682aaa45005f43a32175e2f8a89f4027b73e120659fba", "87c16a50aef9c02cbe9395a419fa829b"
    ),
    # FGT_4401F-v7.0.14.M-build0601-FORTINET
    "5aff11fe1daafeb2565de475e70c19193d25b37e25f06fb26a1f1b174b20909a": ChaCha20Key(
        "d6aec32c48635b76b95ea23bf4a5de26f82a5e7ca81ddf300e889dbde9ab976b", "f5e208c5f8c41febe538445aac0fad00"
    ),
    # FGT_4800F-v7.0.14.M-build7176-FORTINET
    "22bdb028c3a3a58c9004fcfcab5f14cd28c288eed516f65357ac56dbc3c731f9": ChaCha20Key(
        "e32049c321660b1d34192523e72a41eef0fe265b39d98ce3c3445a058845d53c", "17ea0458950dfb008b4c56345d5759c0"
    ),
    # FGT_4801F-v7.0.14.M-build7176-FORTINET
    "e32c976bbf2c3669b4836c6db060c410c60ad9747d73e59db1ba61880d83ec68": ChaCha20Key(
        "4ddeff03e0197d21716e4a998fd62eecde8ebb498baa1e512a896dec16c00037", "a9266bc3df2911e1b132dd3786b36efd"
    ),
    # FGT_5001E-v7.0.14.M-build0601-FORTINET
    "2741a006a8cdca674f00adb9fbb2b9d59e3bffade8ee30899049db8522804f48": ChaCha20Key(
        "a0243a2f1d942d4724e2655ca89080fb94c507dd0f0816409e3b646c0c677cba", "a4ed8930b27890371e7c99e349987eec"
    ),
    # FGT_5001E1-v7.0.14.M-build0601-FORTINET
    "291ef9ebfd02250e07498116d883edfc0db6f053c7861a6689fd945a7b783bd2": ChaCha20Key(
        "495b3e68303007a1298b6cb547ae0d9a2c4733c090b66b7d3d4b30601a0e9069", "056d69675f6f9259fa2f9677998e0bf9"
    ),
    # FGT_500E-v7.0.14.M-build0601-FORTINET
    "bb73048eb3431f4e937291d2d6ccdc17bc2d26a390c9f2d65fce55efdc458ef3": ChaCha20Key(
        "6207d93ac96fa7b963d9d308e2cfbbfac61c31f0fd90789a80dd96246a1c2f35", "f9e9782c771f0ff83ea24eff771c7543"
    ),
    # FGT_501E-v7.0.14.M-build0601-FORTINET
    "56729a3c795161921ac37d9eee884b06f17a2d65db344ca9e1385b192ab98be4": ChaCha20Key(
        "7b96c27eae1598a65adeb046f73927ddd19d0c79f64d396a389fb4e9fac04451", "5217410dd66648c8cf94923fed2b9dae"
    ),
    # FGT_600E-v7.0.14.M-build0601-FORTINET
    "2a63535a99e75e05894ee64dd0d6c2c35fffc5ec80fc9f4d45a541cc4d4f623f": ChaCha20Key(
        "9ee5b2463f4bf7c5de9cb3c1164251594ce1139ca3f686d45d09635627f92606", "ad049c34d1d12b663af378d830358ebc"
    ),
    # FGT_600F-v7.0.14.M-build0601-FORTINET
    "87528949d512d611c0e9502dc2f4afdf4d178301cbd176430b429bb87f0be3fa": ChaCha20Key(
        "357f97156d29e59558fa9f88f0cbd9ea34d9a5185004377e7388c499caee57c7", "ac01dc986055e23f61193201ef13c64b"
    ),
    # FGT_601E-v7.0.14.M-build0601-FORTINET
    "8a82d3d815d49351581ca8b98d0a656db431784ebeebb88aaa888e8317a6030e": ChaCha20Key(
        "f958c26cab5a3be9f8e0220467038dded17d1d8610558060fca38385f6a13d79", "d3d8ddbdf18cba812aaa42b75dba69c5"
    ),
    # FGT_601F-v7.0.14.M-build0601-FORTINET
    "878ef1774eb1bd9e2393209a04416516273beaa611ffb160ed6f879f9bb5cbd2": ChaCha20Key(
        "d20b5e066f6260795cd33802a40ab50803536c16a5e194fb50dcdc3fe2fb92d1", "a2fe7dca68e6135bd054b2721caf470d"
    ),
    # FGT_60E-v7.0.14.M-build0601-FORTINET
    "ce9f24ad4f0948c6a810e1ac467eb2aa8f09ef2a0c98a1256528a982dbbc1a34": ChaCha20Key(
        "9e3ac7a2bbc7f551a08c63421f4eea8bf6e5fb13ad9af6531f25a01e048a1a70", "185af0948d77c7e4ba84b0e0eb45a655"
    ),
    # FGT_60E_DSL-v7.0.14.M-build0601-FORTINET
    "5e90d3eb3fd594ddec726d01dcab3da2c808dcaa7d4715d2fdd1fe5c3e06cffe": ChaCha20Key(
        "4cde6e26e23c9aa701afc20378f1cfa1b5828252ae70554b4142441430776c69", "b2fc7751c8785e9e2904252d53ce0d94"
    ),
    # FGT_60E_DSLJ-v7.0.14.M-build0601-FORTINET
    "b5818c680710be0353ce77ea773e295085c087c47d6c96c941428bfd44dd4968": ChaCha20Key(
        "bcfea9c01d523feeefb031754d8255337297ab60ebf4d19d2c9a24d160d1df5e", "9cca9ec53e8a60cbf9b14c24d5ca1dcc"
    ),
    # FGT_60E_POE-v7.0.14.M-build0601-FORTINET
    "ff273bbceb70f08e95c8b151e5129588bf07daedcd0fa81370118e93fa23e180": ChaCha20Key(
        "de24cf16a47d5fea16136aa06a99657999fcd62f988938fbb546c8627e3efa89", "d0d54be2d247732b53c8b5cec021d0b6"
    ),
    # FGT_60F-v7.0.14.M-build0601-FORTINET
    "109317daab3d380954b4df3dcf5f679a51678c624937c29bacb47707175629ab": ChaCha20Key(
        "6c82cfcbb58dc01f807d28e0d02165c14f28e7bfcbbd7d16c5809639080bd235", "03b7fdad3eb33e605bb8e7d7a194902e"
    ),
    # FGT_61E-v7.0.14.M-build0601-FORTINET
    "8fe4cae3d8bbad83089bbf255df7f579cf691cb62c73e88c2cf32346ca589263": ChaCha20Key(
        "36d4876a4ca59365f292bdcd9c1a401ccd82d73ef0ecaea0cc9fca602f495b5e", "29a1abc608e79479b9aa8c04b4171ad9"
    ),
    # FGT_61F-v7.0.14.M-build0601-FORTINET
    "c8612592958665e97d9523282d82164eb53092e5d1f8d9fa0bc8dd57951afec2": ChaCha20Key(
        "ba91a6e2e3f39053576de9f61a4d9078a08348a2b705f2745e9883c848a7443d", "a8ab8ef593e84708e03d48fbb3962c73"
    ),
    # FGT_70F-v7.0.14.M-build0601-FORTINET
    "ab30f6235f217fab7f99e294efb234376984c663a326137bd68c4a30b756cc5e": ChaCha20Key(
        "81dcdd0312fa386ac3aa4bc620c2aaa7ed1b6336fddb7a13868b3dc6e579e4c5", "33f2f7ec32a91a505b4ff1f295f6d01d"
    ),
    # FGT_71F-v7.0.14.M-build0601-FORTINET
    "669354b66202f3b8614042d4765557f199ec550f8777450535a8890e14e90dbb": ChaCha20Key(
        "888dc87ba00b6ad3670a0dd73eb71371c782ac16fc879d7aa3feb3d6d1b519c7", "858e44927eb5a9a853e76016c29e54f3"
    ),
    # FGT_800D-v7.0.14.M-build0601-FORTINET
    "cfa57b9af2ee1c4594c6a6bddeb50af8a0332224c0915c905d0ed6758b6efbfa": ChaCha20Key(
        "b3fca21556853cfe8b0a6230db97f4e20c75fcc77bf7b6090b521e00f063e281", "d474c1e3a534984752472ed47244b18b"
    ),
    # FGT_80E-v7.0.14.M-build0601-FORTINET
    "fec952d2472bc256ea8f17192234103d21365aa874859020d9e1e43ecfc17ef6": ChaCha20Key(
        "c00bd4b3059a6da6e35a75e69537c691648437f101a44709406e7a93a532a59c", "057063ebe0616adf12e5b59a16ed552f"
    ),
    # FGT_80E_POE-v7.0.14.M-build0601-FORTINET
    "e1467d29d956b36b903f97a7bf42757da910dcae7387423bd7a2589120ee8df4": ChaCha20Key(
        "ff4676e59d3ba75338b678680ac552370f637e842c0e3e39203db611475e6364", "b310313058c4bbbe166c063f1451d3bc"
    ),
    # FGT_80F-v7.0.14.M-build0601-FORTINET
    "6a0229798258a433b13b7aaf123d30226fe9bcd682530ceee42f2a5315285d87": ChaCha20Key(
        "c2f4703621f6bca8a5270167af946d2a842251db6adbb5ba2c62db0bdaaf1f87", "2a4d6c5f8ed747f1f73ac7df9321d051"
    ),
    # FGT_80F_BP-v7.0.14.M-build0601-FORTINET
    "5de891fa1a5699d9dd4ad1a4814c4f01d54821ae27d11686d3065bc9c4a6e17e": ChaCha20Key(
        "edf09a7925a99b554047fc00bab37b4b936b88422df1a465104a8b18179d3237", "068d0fd52eb461d1d17f0eedb6bd1c37"
    ),
    # FGT_80F_DSL-v7.0.14.M-build7173-FORTINET
    "28843013b40ceaeae3b71402e55e780af7f0a5267fbda651f17b7fdc78d2a7af": ChaCha20Key(
        "abb8a493ccf0ab355b88671b2a94d3cae48a60073a371d960974710d944854be", "372714adbe2414077a08e60e73b34cb9"
    ),
    # FGT_80F_POE-v7.0.14.M-build0601-FORTINET
    "7d8b4786023527531cdd7d2cdb84ba67a79e0b9a1907a1dcc636108310ef9758": ChaCha20Key(
        "5af7b951f1aadd5b5f2357b09edaf0b677e71a4dee1f3274b00b033b83be4622", "773b36879968345d042cedeaa2a9ea83"
    ),
    # FGT_81E-v7.0.14.M-build0601-FORTINET
    "4699b2bd340736f45e38c4de69e70ea50818b2b05570d68d6caaaeb714a02b6b": ChaCha20Key(
        "dcf5b193fbb962a620868fe6228af80f3e022231adf8d1c7b22455511daa61db", "1105da249586f1a8e98916347fa2e8a0"
    ),
    # FGT_81E_POE-v7.0.14.M-build0601-FORTINET
    "995ae8fdd1eebd234ed3f5220fa600d74a75d215d6f30894139c01e20813d739": ChaCha20Key(
        "75c7f97b0fa344846445a2430bea2aa3d3d765033d763c8f0ba8179758029d62", "09511a9dfdb37b3a6c8a9c0afee3d63a"
    ),
    # FGT_81F-v7.0.14.M-build0601-FORTINET
    "bd6f4931c3f3c4127e4cc4eb0f5ffcd63cada2dfbd406169f4c9c95cab79cdc0": ChaCha20Key(
        "42e241fffe91f3ce8d2ce31944e59d45882b36e014fd1f0971f4fdb06a259fc0", "ec55a539997deddd5c8f1f4e78ba2d3c"
    ),
    # FGT_81F_POE-v7.0.14.M-build0601-FORTINET
    "8c5e096d0c9e72a94ba4d4432919ceb4536340a809b26ae27df2b0bba8435611": ChaCha20Key(
        "6d6a127addf6cee7be972d94d4da1d92a55de42f78cb981a2a4ef67e76b6d43f", "9d9afde33085f09a6da4bc1eedb05735"
    ),
    # FGT_900D-v7.0.14.M-build0601-FORTINET
    "d2970b368998d137e179944eed4e6a7889eacd7b4a227629a2ee88cca0c4515f": ChaCha20Key(
        "8bf1d2381db8ec94bdfd600ea6fff5e5c97b1e895730c4b2e29cf559953498c6", "ec9b1e11ba3176c038d4821dcbd61ab1"
    ),
    # FGT_900G-v7.0.14.M-build7163-FORTINET
    "62df60ead71ae271b481c391e4fe99a8422b141aefb60614b8c1f11265f0680e": ChaCha20Key(
        "a4b0e17dd7939ed3478a1fe30951d8cb7326ce31283437749eaa040dadbcb738", "204fb15b740201bfa0bf685e0e50f3b6"
    ),
    # FGT_901G-v7.0.14.M-build7163-FORTINET
    "520d2983cb42440b1f8bcc2d3ece9581808007bd0d34a643a37d9acf9444a58a": ChaCha20Key(
        "3ec494ae2732d54dca9b36170940fbda96f05cb4eaeba4a1a6335f8badfaf6e6", "938960dc382e94d73d2fdec7c8be9c34"
    ),
    # FGT_90E-v7.0.14.M-build0601-FORTINET
    "84d793d766cbd228f8c2cd36fd4f2f8a47a908a17528daf727e5849f134edb36": ChaCha20Key(
        "3980d5cd047e137a31d94e3a4e22f9204c5caca591e069af248e6be3a160a0a0", "682edf174d93ecf8a083e97f24735e51"
    ),
    # FGT_90G-v7.0.14.M-build7201-FORTINET
    "030ac85bb72cadc89ffdfd764d74688f5a3582398b23b96a9cefe275e9c63ca5": ChaCha20Key(
        "ffebb3d36bf7450a7cf49454b883aadd85fc9a3eff4c164fcb0109ed5b43e8c7", "be8d65120b6280007e15cbd87674b7b3"
    ),
    # FGT_91E-v7.0.14.M-build0601-FORTINET
    "79e1ca2bb792407eb5fdf179e2609aa3e1fb2179249690ad50ec1a755cb02467": ChaCha20Key(
        "638196d8f69b0af511fa5ecfc8dc7cff7a7ab2dc76f31b72b622f046c4f74d7b", "190ffa36574c65c3dffb179d299dc67a"
    ),
    # FGT_91G-v7.0.14.M-build7201-FORTINET
    "18da4376d0e4bc3df228d5b8bbb163bc5a282b2bcfb9ee17858758b9da975d64": ChaCha20Key(
        "aab32cbab1ef5a1963298fe9c7fdd3bc4b037d118327669d17cf4b3d4e221db9", "581add7589ddd7e8d0c088731ac2bb53"
    ),
    # FGT_ARM64_AWS-v7.0.14.M-build0601-FORTINET
    "b98d22f446ede13f2e733b8b1fbdf5500331dc6ad7a7bdf62c3a5a58907cde65": ChaCha20Key(
        "3d1047c559b546e7c79f4243143ac6fde899846a9495bc65177dfd3e2f738da0", "1bdc6a0ab03de62c96d2c4ea983160d4"
    ),
    # FGT_ARM64_KVM-v7.0.14.M-build0601-FORTINET
    "16502c8be6e982cee9c76ba895ccd6dbc26c7b2f95b4900e94dc12344a9d0b4e": ChaCha20Key(
        "a95e7342705d04fa7d23425bb8e59b680a6ec2ec005824f850c07cf172285eb4", "f995e21acaa546a74cc05f445686a3c4"
    ),
    # FGT_ARM64_OCI-v7.0.14.M-build0601-FORTINET
    "317acf4397dd2bfd339a874e28eac0a817f8ef08af3a82d102407f8026926318": ChaCha20Key(
        "77adeae20591b54afe6d6654bde260c105a89f2df119e1941c558774b9fc0ded", "640216c293b4260d3aea6243b79a4e30"
    ),
    # FGT_VM64-v7.0.14.M-build0601-FORTINET
    "67d4c913b1ceb7a62e2076ca835ebfdc67e65c7716fc604caa7552512f171197": ChaCha20Key(
        "9ba00c035bcaa97717d936f8268a973eb1dd64d19388153fad5f7849b8fdf0d8", "9df4ba40dbddcf5ec9d2983681eb1940"
    ),
    # FGT_VM64_ALI-v7.0.14.M-build0601-FORTINET
    "e26deda186c600ab07ec65d44085c27a474b57a74ba481486f4c209a82d1b8c6": ChaCha20Key(
        "bbf45b5a0c459d2fe3ed8d6fe578ec20455e764312c18e580a7dab91d9c935d5", "d2d14e120056e30b9f3a371e088505b7"
    ),
    # FGT_VM64_AWS-v7.0.14.M-build0601-FORTINET
    "4f54f970eaf5fd83e581f7c8e2613bcd196056eb9cd2f3327668ac7491df4587": ChaCha20Key(
        "cbdc05444e6881ac3be2a7d055f1ac5cad95af49baaa48129b90ac13425c8399", "0c83f80a274c0588bdabfb1970277cbe"
    ),
    # FGT_VM64_AZURE-v7.0.14.M-build0601-FORTINET
    "6f940ab0f4d25ab8621fdca9b6707ff22e18dff71d3aafdb95b7e829c8481da7": ChaCha20Key(
        "6ac5e219032251127239715fa06fa76abb92e4e1f61f01a57594b305a62fc82d", "68c121e91323ea4599152b267ea2f109"
    ),
    # FGT_VM64_GCP-v7.0.14.M-build0601-FORTINET
    "dcf2d26ec7d1715abc91c99f0dcae3e3e76650137783262f9666632974fcc814": ChaCha20Key(
        "844866268f3a6b22de691bc4fe9deddd5c760fd2c7ca993d7e86ecd029f601a1", "32dc1e5ababf20150c09c65560bd805d"
    ),
    # FGT_VM64_HV-v7.0.14.M-build0601-FORTINET
    "ad651fb1af2bb65c3810fd2d57d6a54257beb18cd51868b0a536a9f6be4fe8dc": ChaCha20Key(
        "698ba1a961f3e989f475b9877f7d616cfb34df42c4cd6a4ddff5c6b64a05e66f", "4ac33b9da53ddd8611421ca2e4c1bed0"
    ),
    # FGT_VM64_IBM-v7.0.14.M-build0601-FORTINET
    "b8ff76df1ade5a0cb344fffbcfd94e2188f6ada7b5e28d7378c30b642519f7b5": ChaCha20Key(
        "d13e628c88ee1fa4f96af03257eea6bb2cec218b855610a758a6456b2c4bd176", "c5ed5651b993b77731e866ddd2e12c2e"
    ),
    # FGT_VM64_KVM-v7.0.14.M-build0601-FORTINET
    "dbcdb3a7bcec47ab6c91ab4764c9cab010b785b8dfaa5a5c84f882fab350159b": ChaCha20Key(
        "afaca7a2bb226cb5084679faa69eb785ed9bd4f10027c9efe870acdbb1bfcbce", "c4d6a1f3dc2034979a6d958881db63a8"
    ),
    # FGT_VM64_OPC-v7.0.14.M-build0601-FORTINET
    "27871cdc6581e0e9d5665fc968874c0b365bd68e00b2ff7a8584d258a1de77d8": ChaCha20Key(
        "7e13f006c6991fb89289f68d6d67cf1a9471fb7589389c224208855ab7dce2d3", "9d1f54792b28cf77265843fcd7a91935"
    ),
    # FGT_VM64_RAXONDEMAND-v7.0.14.M-build0601-FORTINET
    "a2217497d60ab1a570eb0e3ee070da43b336e2fe02ab3c674c85eff24349fdd6": ChaCha20Key(
        "c9a26db8e492f583ba5a0e7720baf5ae46cd5374361d240ca452719857368a12", "850584a9ebbdcdf573eb40a4f2f0e69c"
    ),
    # FGT_VM64_SVM-v7.0.14.M-build0601-FORTINET
    "a4eef267ee44040fae15b83dbecccc341248cbd4ce96600a775983bd4d3b86f3": ChaCha20Key(
        "99bc1047b51c629aec881ec9825f1918a688658c00d74cd322b5ffda79d61012", "7cb403c0decd77c8165239d0f90864ca"
    ),
    # FGT_VM64_VMX-v7.0.14.M-build0601-FORTINET
    "20fd5b9fb7387c056d18a6aa51d6cd2121c45c827077dcc22f546591619c6e44": ChaCha20Key(
        "56e807db49fbcadb19cc719e9a8cd1eafb2f04aa800992089920ed75a6a7e2c7", "ad912f6f408fd649dec096d6dd5ca79f"
    ),
    # FGT_VM64_XEN-v7.0.14.M-build0601-FORTINET
    "93fc705cd98ae6317a2e57f097b8cb654b49b8168345a7041a65a045b7aed70b": ChaCha20Key(
        "5cde745df08b28e12fdbc1ff35d0f194c92936f406c1251b635b13238cf216c0", "476e7fc9407e47dce2397aede506e3ed"
    ),
    # FOS_VM64-v7.0.14.M-build0601-FORTINET
    "94e189c43d130b45eb86043ffbd1b9afd7d14c37ce1481be3149a2f8ed198476": ChaCha20Key(
        "438917b1623268c8a634f530892d1aab9b005d9efbdd85448aa78650a326bcd5", "fb8a865efc8d12357a98d09c7f4eda81"
    ),
    # FOS_VM64_HV-v7.0.14.M-build0601-FORTINET
    "33cb5cd4012736351afd26790e17ed71049bc8161abbab251fe2decbddbb86a4": ChaCha20Key(
        "8babb640582ccc0f74908da64a4ab61d29ed3e5cf1c80edc461a36bfd82329b5", "fb744f0048e4523aaa8b885616be4796"
    ),
    # FOS_VM64_KVM-v7.0.14.M-build0601-FORTINET
    "4db1709d293ef46e29b4c13042ffb22012c9ff616667548e807aba874dfed7e6": ChaCha20Key(
        "4e5a7c8c5bf1f945050c51e9e3782ee97fd595a29d56208f2aa8e748071e38d3", "a499258f33160c205eef50812e4ab7d0"
    ),
    # FOS_VM64_XEN-v7.0.14.M-build0601-FORTINET
    "64bdd9463e1be4d01e307a6e3f8578be2714e064a829224a9e3c9bc6b0a38e45": ChaCha20Key(
        "a825debf4bad44b9b57a3788fa5c16d2ac0c0a8e44bee2a93b2b1b5ae2252085", "9aef31acb73af4b8ce74727a3edf2ec5"
    ),
    # FWF_40F-v7.0.14.M-build0601-FORTINET
    "a84dfeeb19619ad45bc261b37d78490e0294cfa69ae9f22010babb423e310d1c": ChaCha20Key(
        "700ff38f82c5963f126d8c57035783a7c3cacdc64824d32b32c5691f8e360010", "dc4a7e2a52cc8e0fb0a15fc7a79f94dc"
    ),
    # FWF_40F_3G4G-v7.0.14.M-build0601-FORTINET
    "d14adc8cdb24f586c343ef3c843b97d563bd44eede19ca5292eb0b1cacda2cf2": ChaCha20Key(
        "49357b7979e89ecbce904533753ff9547cb2fb719404cea1e2b42e2703e445e9", "a5f6fe8ced97be77598e44113e5bd577"
    ),
    # FWF_60E-v7.0.14.M-build0601-FORTINET
    "5a84a1f473e99dcc9e8930c0ac44d1664490590a3b9f863a5c4681ee07dceed6": ChaCha20Key(
        "bbd7a52bc09f90eb651acb19a4e3337a82bcee1573261d0f70da515ad8aba9d2", "9888961208a3ac5bc36b08dd7c6d6911"
    ),
    # FWF_60E_DSL-v7.0.14.M-build0601-FORTINET
    "3f3e6bd69d083e5e33eb04a67f1eca0ce34233e63f80e151c74ce5719b43fd33": ChaCha20Key(
        "28f0826a28fd0b8eb15b8f158471f63419e3a89961842af8a78d638149889d06", "614524c6d28f888881a4d1f16bcdde42"
    ),
    # FWF_60E_DSLJ-v7.0.14.M-build0601-FORTINET
    "fc67d3ae2d106ead5d07f4e89ce81e520f5808815a389076b4cea71885eaaa82": ChaCha20Key(
        "d83963b5f7f0d8674d6237c65e5d27a53248189bf61e6e1d7591cd6e8c22a02b", "454ee8a69ddffa93fd0fe00afaf1013c"
    ),
    # FWF_60F-v7.0.14.M-build0601-FORTINET
    "cec44ac8f650456d8ccba1de1df767125909f540fd0edeec8d92971fadda8a8f": ChaCha20Key(
        "eeeb0e66b3c40e17e81df07d27f9a69d7e8520cf9904caf396d430b5f55827d9", "0c0355473256a34f7e220d232b6752db"
    ),
    # FWF_61E-v7.0.14.M-build0601-FORTINET
    "19faf39de5c16bd5e4869ad2de19c2b064d2736c404f79b914d58e04dac1a499": ChaCha20Key(
        "02557e65ae9e4ce0c09917dd223f6aab83544f13c0d15ec3e74a900305734058", "930958bfd759bd27ba24b2f9395cd084"
    ),
    # FWF_61F-v7.0.14.M-build0601-FORTINET
    "f87598c20675a3ad0db72e4ff1d092a81fda20e8ee62e90a044bcfe0bcf9c6e1": ChaCha20Key(
        "bc46b89a59008ff182e75ad7c1a6831fcd7ce0696e4d2b7e48dd24c8acb879f3", "32b5ca9cd3a2e393ab4fb7c73b963090"
    ),
    # FWF_80F_2R-v7.0.14.M-build0601-FORTINET
    "6759483f0ae25ceec250ee2585ab28955850532ac6f5a968470f08dcb860c0da": ChaCha20Key(
        "e4d11ea233adf84f32254ef5eb0ba1d4d8021738a4304264e5d984d4c11b6ffa", "c276e972ef942978c2c19ba0b4b4e645"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.0.14.M-build7173-FORTINET
    "d647a9dbc4d7320453f0f84684267b24fc90d549f3babde46f18bdd31cbdb454": ChaCha20Key(
        "cd9819492b4e534fb61d5f08b77ccdde3b7a64737bf458c0b5cc6d5187394da3", "bd11935d843af46ca5aaeae7f022b278"
    ),
    # FWF_81F_2R-v7.0.14.M-build0601-FORTINET
    "c38b535eac88d607b2a10ca37562af924317f03abcce0a973fb886ad1feacc49": ChaCha20Key(
        "9ac58aa6784287989ab1d7ab4729130333d5503829ab567bfbb4d0b5992d1fc7", "f76148a2cb062371dd8b4548c9d08f0a"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.0.14.M-build7173-FORTINET
    "03a818f6989cfb0a0f25982f93f427e1609346368b7d3d247b3d1408b34f8204": ChaCha20Key(
        "568a65173c1caef70c55dc7a9b514a059c0ffe59f53443c8616d069b9faec32d", "ef10a41b5e122adc3bd4e8072948fc14"
    ),
    # FWF_81F_2R_3G4G_POE-v7.0.14.M-build0601-FORTINET
    "a82a4679af6b0bb106e3d52f27349dcdadbb1d80e329c93a73df47a5afd29835": ChaCha20Key(
        "ba9175b885dd7538a3a7391bfa46ddac3567f73981bbf2a81ae73563505bb0fe", "38b8595d17091ff4217ed3715267fafb"
    ),
    # FWF_81F_2R_POE-v7.0.14.M-build0601-FORTINET
    "73ee31d8b793a1d6272d4f9a511faa75ba4b0fff8900fb1b4398299443835c16": ChaCha20Key(
        "5bc9b95ca71d87c0079955c7ff4465efc4c8088f45ef9dff5e6436428ce5a4ce", "3ae7b03f91d28d40fcaec822bf5f7d94"
    ),
    # FFW_3980E-v7.0.15.M-build0632-FORTINET
    "4d6e4ec9f72ee35ee5bebf548a4f887497c11640260b62452f2aca762e0cb7b9": ChaCha20Key(
        "e6b2c6cf39a52cb34f9a0e94685a516361ebb1b503b74cd66b72061b88470582", "3a43cb63752a4437baa695131d3585e0"
    ),
    # FFW_VM64-v7.0.15.M-build0632-FORTINET
    "cd48d12019bad955150f7e11ae97cbb32dce2bb3c28c322e395a33c6ca954018": ChaCha20Key(
        "8b13c6e68725e2d99cef4cf3f92cb83fb22b8be66bb63c78207231d028097f6b", "7d1063e6516da243a0d04cb6019748a9"
    ),
    # FFW_VM64_KVM-v7.0.15.M-build0632-FORTINET
    "172a0c0c8c61d54eaa61f0c814e07d83f9a126f175515cd96c11864914ccc3b0": ChaCha20Key(
        "f68a1d470eda8092f8c1fa6263a1197e9aee450e83919d34292d38c8604e60b5", "de549dae6282a78dca6fb7a0afcfd523"
    ),
    # FGR_60F-v7.0.15.M-build0632-FORTINET
    "4f6811e4f6ef8fcba25a19894844368d641daa284888990ff018dc58eeded818": ChaCha20Key(
        "13795c68cb4e2dfc1403906d05b3c8205f5c75af0113981be84eb1269ac43133", "485ee970d630e82eb945fb1da85bf0c0"
    ),
    # FGR_60F_3G4G-v7.0.15.M-build0632-FORTINET
    "2a83a87feb5fbce5d2d0393d391b8b5aabd698622e246aa83542b58a018d6b9d": ChaCha20Key(
        "540b4b81b3cfa3a9bc82bc1a8a076a3f25ca54d7d3b46133495ba70e64f98ccd", "90768bdee3bf865966c7c581a3dc2d06"
    ),
    # FGR_70F-v7.0.15.M-build7332-FORTINET
    "59f2f0dbb72b0bd95c3fdc6c680729d5f4208665af96011c97a883e5d7a73074": ChaCha20Key(
        "261ee7f7d7eac7c19d92d45836310669224e369c9415a87b004e3cc45b47d23e", "c6574f2d4f2eb85014bf84c9de39c015"
    ),
    # FGR_70F_3G4G-v7.0.15.M-build7332-FORTINET
    "a712545d1cac6b3839c58372ac3744871c7b8f646a2e7bcb52fe0991281cca52": ChaCha20Key(
        "c9db7153a0640d00146224c34d1fcd0d05328802c8c16e62bb68f22dcc175a6a", "8ee4006a18aacbedae84f216311eb319"
    ),
    # FGT_1000D-v7.0.15.M-build0632-FORTINET
    "9820b138d868ef7d2c163a9a5479db6580fb3ee79007ee5b79e749c398c8d1d0": ChaCha20Key(
        "fccfd5f09edc2b6ab41e3b8b322adf27415bc7949a858bde977d36a9cae8a6d5", "3f42d46e32bf309fac3dfa373bebcb73"
    ),
    # FGT_1000F-v7.0.15.M-build7330-FORTINET
    "8cbac770860b29f5b8f390f321417270c1eda13da26f215e5a3f1d1a3101b116": ChaCha20Key(
        "8c1be332f51cc5f78739b0cbbc8b0d71a820b5c7de27f92c96cde5fac5a16e25", "01662cdc4b992422bf7f9c5acbb8e4f2"
    ),
    # FGT_1001F-v7.0.15.M-build7330-FORTINET
    "018c5f2e1f7fb54fb959983dca19b76e2ff5ef88c9d884cda7f1c49ef3be063f": ChaCha20Key(
        "b597195d41c3b6d32346b56a316765d5707f343c1bee11c45af4b6101d62de6d", "0f8ad68b9413d82108b02fddeaa905a9"
    ),
    # FGT_100E-v7.0.15.M-build0632-FORTINET
    "327f8231b3bfa925c7d70f783e527e62e044dafdb2bcaa79702084aff9f2e926": ChaCha20Key(
        "3dde6f6213c8abcfa3430df153a8dc84c9ee2d570adaa42c5463258ca1be0312", "4da4a574f52eabc34729df118a106339"
    ),
    # FGT_100EF-v7.0.15.M-build0632-FORTINET
    "5b027ba639f73d4dc971a05162799221d52e25ceba90513fb2c23d6ef6bec40c": ChaCha20Key(
        "a47b1e79800fea49981d23d444aea74aac5c387086c6fe0387dff9a9e8a76e28", "d1d9444d1ec14b57f77db86d1c7a6577"
    ),
    # FGT_100F-v7.0.15.M-build0632-FORTINET
    "28916d217d290c5e43181691a282d59c2af5dd79e70869dceac94ac861ab83a1": ChaCha20Key(
        "68d8b9e9841b2d7df6fefd3cf66f0fc1b6acee01d49fc9b82fb27437d92343ae", "c4e16f3fa58f5804e1cac99c3011ab85"
    ),
    # FGT_101E-v7.0.15.M-build0632-FORTINET
    "c2b6e34aea2b7a33c3be5e2430960068d746b461af0ac04c2debe10ddcb8a751": ChaCha20Key(
        "cc87c1b944b0ad1a4e6aea67dbeae03b165d2b8279bf55f8d1bf0a020d2e9dac", "2e3663a9fc3725a9b29e0555e7ca2f5a"
    ),
    # FGT_101F-v7.0.15.M-build0632-FORTINET
    "b6126613da232c87eec9ed395ea82691e30725e8a4efd73eb472db6611274641": ChaCha20Key(
        "7c2d359e831ef9a4c44166877ec1af6e3b356d3e76527a96b74c72591e66665b", "b9c25e5ff1e5d135c2ba063c61adb599"
    ),
    # FGT_1100E-v7.0.15.M-build0632-FORTINET
    "3c0ed50094350049a13a03ad9fdc60dbd4b8ee9cadcefa89103140ee86959bb4": ChaCha20Key(
        "21909f7900ef5544c970f39c704172cd23417ef0e57c001bb801053a828dd7a8", "c4424f28699e72da1402ef911548d298"
    ),
    # FGT_1101E-v7.0.15.M-build0632-FORTINET
    "d1a7b0f9224a5888641415c8ebdf19a92316b582a23975ef1ac1156fbeee953a": ChaCha20Key(
        "0828341f28aa8842f3d8f5fb2d765e1049aadbc8dbc0bfe6b4b3477d5ace7929", "23e451ee0105ced4d797752f622356f8"
    ),
    # FGT_1200D-v7.0.15.M-build0632-FORTINET
    "3e9528426bb560cfa892b123c8f54a08dddf8a2a72b71f13711ec5ae2705db4b": ChaCha20Key(
        "43b07074db0a5e6ed78ebce0a02a8249ba78dc00c72c582393e1c47828ba58d9", "5c689cc430e3555206b2092ecc6b5b0d"
    ),
    # FGT_120G-v7.0.15.M-build7334-FORTINET
    "bf6decde40e835d15cc76694f14f806b3ebbb9c96c155dedd37298158610d52e": ChaCha20Key(
        "1ce477779f8f8a2c6cb67c8261e8a5c20a3b36d750902e1265da34ca131dca59", "f43dc441bc0123233dbce99989eefd26"
    ),
    # FGT_121G-v7.0.15.M-build7334-FORTINET
    "7f0b075d12ba6aadb69b9561373954c40fe182ce6fb5bbc11c7effc6e0a5581e": ChaCha20Key(
        "022b38b09f56ebf4f424446f1f6f6bd6f11088e9057726e42579cf8dbeac04eb", "c2c44d6f829931d6711d4708de5677db"
    ),
    # FGT_140E-v7.0.15.M-build0632-FORTINET
    "127d448278620a8cc43a7e426866fe62e8b4cac97acfb5b9d3839814cca25b5f": ChaCha20Key(
        "0e80df7317708c849139a86eef419d57cfba8268510e48f26d7713360d4fda8b", "654a25566eec6369a7a4d06c14314ee8"
    ),
    # FGT_140E_POE-v7.0.15.M-build0632-FORTINET
    "4246dc082192dbebdba19bc6d78c5e20f51420113043b7a60f6dbbb22f3cb7de": ChaCha20Key(
        "00f0a51ac8fbc62b65608dcc0df79a5a4fdb211e93a285d423cad54694e3fb2a", "5c4294992244c756b250af2f43cd1cee"
    ),
    # FGT_1500D-v7.0.15.M-build0632-FORTINET
    "bce8bcfda39f8522a82e7c269dea005b4231865096798d972c5288a5707b7518": ChaCha20Key(
        "2ff939be7e9b96c992fa5d4e4a027c9ae0c670fe7e209fba0f9d896473eb6eda", "3a6418f2d3b1cf6710a39014bf20d909"
    ),
    # FGT_1500DT-v7.0.15.M-build0632-FORTINET
    "9159d64b0a57be1fdabbf977f55d736547385b8395a648dd54a610b85a85063a": ChaCha20Key(
        "b5e6f02bfd7842ec3929764d0256e0e8390ffac2bef897194bf6e8bb7a008903", "4ec5255c06d3e46efaae0155acb5ea9f"
    ),
    # FGT_1800F-v7.0.15.M-build0632-FORTINET
    "f6bac04f0a1357b8926db657a36b9c719c6bb32f530c8d749459b9682ccf09bf": ChaCha20Key(
        "e84e246da32a5771ee0a2d87a6eb9b644ce68bd2fdc735aa6c8f7d105a8c803d", "38eb10424eb5c9994c0a7bfc4094548f"
    ),
    # FGT_1801F-v7.0.15.M-build0632-FORTINET
    "f730aaddad79c78e19be5cfe5400958d1f1cf50247f623b4c4581385cf8ec2eb": ChaCha20Key(
        "b1bdd5e0a2c4a9a0725cbafa4cb8615313df3d189bbe2f13983186cc00f58efa", "4dbcd3937b188f0bbe374e53a34b753c"
    ),
    # FGT_2000E-v7.0.15.M-build0632-FORTINET
    "9e40db9a1ba9476d3a2df0bde56eece209d49053ad074de9dc2f08077a98a219": ChaCha20Key(
        "ca46cca273b1210aa75ed0d620b18b23dc2768ff0c1dde930c9d0814863adaea", "fa91386faf58774c4fd467ab1c2f3816"
    ),
    # FGT_200E-v7.0.15.M-build0632-FORTINET
    "91ec8e670c78478463dcbfd566805841c736580f00cde059dbf3d69767c6eb26": ChaCha20Key(
        "d3edca5bb7a90df6a12e89ee947bebcdfcd4016a9672b886f984512c543745e5", "e15ad847d50e1ae129a562be881ae476"
    ),
    # FGT_200F-v7.0.15.M-build0632-FORTINET
    "5a18d1bea7db91bf39148f503b5b80545333e3ce6ad17b51cb241d029f66e956": ChaCha20Key(
        "b741b84e1193e06a7e22d31bad8ece570c9642cae3eade17f6cbcd2705b3c082", "3a8149f0ee9954561d963ecf29efb3fa"
    ),
    # FGT_201E-v7.0.15.M-build0632-FORTINET
    "3d0789d36b20d3e0ef68f608b17ba9d3b8a6e6c4ac7df6ad363ed7be7aeb2394": ChaCha20Key(
        "118e401acf079ac9ee65589ba9bdb3546cce90114d1e976d88540e1ccffd1942", "8a17e0fac0981c0de84b3a9c311e22d0"
    ),
    # FGT_201F-v7.0.15.M-build0632-FORTINET
    "54358783ad31ac91949214e71f7d0edeff451eaa39b3c628b45315b7f715416f": ChaCha20Key(
        "6513307d500adcedb118c1ae5e7291869a776fcc993b45534256516efed70ca1", "b47efd774ea08fdb05210ab47d891d5c"
    ),
    # FGT_2200E-v7.0.15.M-build0632-FORTINET
    "324f066b9912c4e83f039b621c4fb747c56d4655482b6b1294ebca4bca500380": ChaCha20Key(
        "2ca9b10326085185e197c8b11b7f3dd6e2ff5cfe8db6f514b7706f00a59b76c9", "f1f421668b650253a4095a96cf132fc6"
    ),
    # FGT_2201E-v7.0.15.M-build0632-FORTINET
    "33c02be291e0be7d6685eb353e468450b2bd1bc277835b81dd903c2a1c005105": ChaCha20Key(
        "c949ee76a094ad10aee8dfe0c4313070242531b31b2362bc248dc6d8eb1b4c12", "484979c795aa2d9ea39dc7b5ceef4779"
    ),
    # FGT_2500E-v7.0.15.M-build0632-FORTINET
    "34822171f88a5d3b0a4433689ad5c1ad41ab677ce95bacd9718620adcab6182b": ChaCha20Key(
        "e0d37759912db16354de6723d655484fea0e71e33cc3afe74fba596821423b1a", "008f3003817279c81e323831a48c7776"
    ),
    # FGT_2600F-v7.0.15.M-build0632-FORTINET
    "b0c1f8f862dee7b0ef40264f4953871ae96c41f11a21d68b1b20249c44b16f39": ChaCha20Key(
        "cb5dd703dd91f5fb7de465619559e0354f824a59c53173f9910716fbbad91610", "f16f625bbbc89fe5ff71e96e027690ab"
    ),
    # FGT_2601F-v7.0.15.M-build0632-FORTINET
    "7de83e45108e1b8e4dfccfb7bdc4bb1a9294a57e5f17c4d26fab508d7a38c731": ChaCha20Key(
        "2557180ffebbdceae52b5601ea0713962c7186499e2515aa02c82653dd830007", "068f64493237eb8428e1d264e8f6bbec"
    ),
    # FGT_3000D-v7.0.15.M-build0632-FORTINET
    "887e1d2a167f4e13e9961bd8f53a954dd602c0867a005aac345ca9a389d60546": ChaCha20Key(
        "40c8fe569afe94838fddacce998c8f84294ad8eb2cb4ef4fc117511e8eeb5393", "1d025e36fdc21afe5d80e39bb30893d0"
    ),
    # FGT_3000F-v7.0.15.M-build0632-FORTINET
    "85f18c30a6f5db3c54b27cf35fa9591a8cfb521d24d8aa9640c861b80eea4d5d": ChaCha20Key(
        "625ba698f8ad6cdba732d3d564ed7a60231378fd79855470e4ec32bd963a04ae", "88e4ebf8d6f5fcba849f9890f9bb88f1"
    ),
    # FGT_3001F-v7.0.15.M-build0632-FORTINET
    "ee372efbcbf065ac0f5b1b0daf6265cd584c727a6fa5d7eeac5d78f0f68920eb": ChaCha20Key(
        "342d6e6af9bc7935a43a87ce2fe5eb9d708ca6ae31b9d0591033ffe50e6e813f", "a496c740d66c2a8ec279c31c6b0917c5"
    ),
    # FGT_300E-v7.0.15.M-build0632-FORTINET
    "a7306a77d92f6aa0be89b114c92fc03b86cd60c5a8160455d0e0da6b2b658a5f": ChaCha20Key(
        "2e1b57c966aeb82aad056cd81ab7a67dcea9e6ff0a49512e4663c09db4939887", "8c502a18e9add766cda843c2ccc0443e"
    ),
    # FGT_301E-v7.0.15.M-build0632-FORTINET
    "1edb4a93b575cf2e99326ad69f3ebae5337be91377ea007f22e8c4d657ebd293": ChaCha20Key(
        "eb1c6b6da4996595356c66377e225e40eb02d8ebeabd3a42591af2402cf6fb9f", "3af778474283468f62aba48bb30802a9"
    ),
    # FGT_3100D-v7.0.15.M-build0632-FORTINET
    "b203f937da5b9d93ca8423389e2f2feeb3466651defbd1f78922eff7a8285367": ChaCha20Key(
        "9088ec9961c2439ff27bd35db8c33824ced15ff0a66d68b02682909c130b9945", "cca9eaa0c10927cd3bd01e8dcfa1889d"
    ),
    # FGT_3200D-v7.0.15.M-build0632-FORTINET
    "419820262e6d83c351cb4c734a526c8f535b360ec43e217218ef349adc9510cf": ChaCha20Key(
        "76ae25ec281c29251ef891d6c6fbebfafcbea47107089b01b59d2c8616c0b400", "80493e076cdc8497007886bef4aed7fd"
    ),
    # FGT_3200F-v7.0.15.M-build7331-FORTINET
    "64c9364f1af89d0d48fd26fb1a919735a86bb75d64283c3833cc3634732aae1a": ChaCha20Key(
        "7b3eeacdfd0717db1f4fbf6f34a8ba7046f3015f7084ec78ecc861efb55add21", "ab9fa86fce9481e5c09e56e017a401ab"
    ),
    # FGT_3201F-v7.0.15.M-build7331-FORTINET
    "b4d58c6035972feecf6c49deba084b31fd6c3a48fd70ddda2ed9c78ebfa042f4": ChaCha20Key(
        "7ab8ed09a43728574aa6546faf69a0533d15b452f7f82e6f6a7b270173be9469", "4e22bf8f3668dbcfc656895e800f3f07"
    ),
    # FGT_3300E-v7.0.15.M-build0632-FORTINET
    "6465cf16794bd66a8baceb7520656d1d0906d9403c096adf7a11ac38440710e6": ChaCha20Key(
        "1fee37f4d780d659e0f965510bcd10a6757bd1f8bfa4d862b0886924d8bb0d73", "116f77310262e074cdddbeb827272b9d"
    ),
    # FGT_3301E-v7.0.15.M-build0632-FORTINET
    "7ef4da3a355d7c206921bc7300a0f88f3d3588a1d5baf7dad66230260c79f4e5": ChaCha20Key(
        "c0de69d59f34a2bc0957c973e7de34643efce4e055124062a3f1fef63c0147d3", "5d2565f7f4cea2e63de04e16b1843a0a"
    ),
    # FGT_3400E-v7.0.15.M-build0632-FORTINET
    "08b5e8f70ed84cd06db482e75d95ed499edb252b4e3c35e951760425dd836057": ChaCha20Key(
        "3d599ffdf416e9f2709096be3d934a35f825d6082959f4dbdcc7cb8feb9114b2", "25a6014c62fa389cbe9d8361e16c9ea0"
    ),
    # FGT_3401E-v7.0.15.M-build0632-FORTINET
    "3dd3fa830e463339d45b6677e2b5886b7e4737925e26c554b6822f8ea13b7fe1": ChaCha20Key(
        "cf745d4534ad25b25f154531d6647c6ee8272508a7c280a662bc076d41bdb17b", "9fb91b0e9e523a1895ef5406034e7731"
    ),
    # FGT_3500F-v7.0.15.M-build0632-FORTINET
    "7ca31c6a39278e747112e2ccba39d2444ba2add4bfb42c94cf32e16b109e8de8": ChaCha20Key(
        "264fc22374106441a28ba749db01c2eb7a34c7f2dd443f779d661c4ad42725e8", "0a5bf5caa675ed9a1b651b75e61f414d"
    ),
    # FGT_3501F-v7.0.15.M-build0632-FORTINET
    "2228c5adda14c545024011046e3caa7d5be2b710a56eaa02db06c582d7c9ed73": ChaCha20Key(
        "669b0b0af485f7d35351b72ce74a2a1537391e893defb32143345c77c0972f61", "9d8ee44f06c16a140d644b482c047df7"
    ),
    # FGT_3600E-v7.0.15.M-build0632-FORTINET
    "af69fe59854fd4c0212ba2e613882f725bb3d2c4605e86bbb5de8b7ae14f123a": ChaCha20Key(
        "aeb1c1699c800ce4e2bef8902f00a43627beff3cd712811e900c093f79a25739", "58045025e2b0d4f0d728c32c1487e5e5"
    ),
    # FGT_3601E-v7.0.15.M-build0632-FORTINET
    "d8263802b9c5bfe74d72f88dfb1f015ea06b3133eb47460d597c040a5f11577f": ChaCha20Key(
        "e07c78991faa37895e9d7dbc7db5dc3c4fe52d23f598553803afc54c7950bf77", "2b1a522fae9be944745e0f2b47345b9b"
    ),
    # FGT_3700D-v7.0.15.M-build0632-FORTINET
    "c70e2c8f137cfff8c494e20cf7507311ea244f0e261142ee7b4696f17b60c1b7": ChaCha20Key(
        "d13fecbbfde6749a859b64423938544821268b2cc18dfcfb2c6ce39f8c082505", "9236a5ea333379f624e0a9362ba5f2c4"
    ),
    # FGT_3700F-v7.0.15.M-build7331-FORTINET
    "c69321544419839fb0769e696cd4fcf603e07a6df93b387a455337e95399272b": ChaCha20Key(
        "f8fa2b72ad07e8b058011020205859e651b91b91ac02e2eac753866a21a6924e", "680c2625d832a4e154e6b083f6b7b09b"
    ),
    # FGT_3701F-v7.0.15.M-build7331-FORTINET
    "f0b468874f96c13cafc9b35fa862b900c167ec7849b1b2c0a24b6ffb453bb5d4": ChaCha20Key(
        "db786da46b92f34763b53ee72e2b70d3a632290a3e84ea55faa76ec55a58dcc2", "17340d5201af79f15f36327e85aad571"
    ),
    # FGT_3800D-v7.0.15.M-build0632-FORTINET
    "7a14c1394c7fd16cd04bb27abd29be618430c7c34a2cc8f0e0ba86c7bd42398a": ChaCha20Key(
        "612e6879710c90b2419872c467e53871b599f58063592abda0a452fdc3304c76", "e0cefb015d38878c273ed40402dbdf6e"
    ),
    # FGT_3960E-v7.0.15.M-build0632-FORTINET
    "2210d90dc7721fcbebc29e14bc89bfc76072c581341ae23196e3bcb3ebe63099": ChaCha20Key(
        "08106b9353f3476576542146431ff8dcc8199208cc0328769f8a5e283c8e1506", "439a98d4d19856f550ca5178c1521ee9"
    ),
    # FGT_3980E-v7.0.15.M-build0632-FORTINET
    "da7a749f96e7a0454a2a8e41ae1c0301ff2897c059358f9f85d318eeed82cabf": ChaCha20Key(
        "1a3cf5cfa27d0a2f1e856d651a0014f9e2f2c7deea903bfbe333b4792e5388ff", "72e817b002b3956a4bda30e6e50904b2"
    ),
    # FGT_400E-v7.0.15.M-build0632-FORTINET
    "6432edd8cd2baed9bef6066ddb3a52d982a9a86430475b88449ffe86d46ab2aa": ChaCha20Key(
        "d94cf66b64d470e3b2bd0fea16bd141809e1b041367478bf1a8853b58bab9e37", "022640053528317441c9e4b649265489"
    ),
    # FGT_400E_BP-v7.0.15.M-build0632-FORTINET
    "044f7553f371d4c6393659bb87fc2cd4db4a7dae6db4799b60e4e9bb06b1b26a": ChaCha20Key(
        "40afbcb35f163135d6cbc4320beacf323d4916623b2246e009afb2dad06217a4", "c916b3420cd0e33a15fa3cdfb239b3f3"
    ),
    # FGT_400F-v7.0.15.M-build0632-FORTINET
    "c5369978696ad20417f885c49e21c2b1e26b2580e4a7c9c86e8bfd37f20c0057": ChaCha20Key(
        "021317b562f8f0750834dfc7d21594abd82c35fc27138dfd51cb32b9a1c74581", "9c3d40f4536e92e42e80389c534dfaa5"
    ),
    # FGT_401E-v7.0.15.M-build0632-FORTINET
    "a7d72f1f5920acd223defbdff376aa2b4d84fce43a25400c6224e20d47d93d21": ChaCha20Key(
        "db0ae2e265280a807b149c4ae823314d0b6a962be08f1fea528eccb9b74a6c3e", "af41b94d111e4c6ea94e00042b1874f4"
    ),
    # FGT_401F-v7.0.15.M-build0632-FORTINET
    "27d44668a358b3a68c88b806a1625aa95653920f0837f25aad3323ce679af3eb": ChaCha20Key(
        "72374750502d8cf8e125ab0a668831f8d72073ae01c5cfa6eb1d92c9b98cd6fe", "c6573cd73f2312e43f18310972d845a4"
    ),
    # FGT_40F-v7.0.15.M-build0632-FORTINET
    "66531820a9312b016b0d6ca5c05129664bc0c1631689dc5369399b1080b966ca": ChaCha20Key(
        "3f728a8f1880fd11bfb876c610d38a5c866efcb9fe295d1ef7e24b688dc1e626", "ca89f34b22da8844de8c1e998dde8c9b"
    ),
    # FGT_40F_3G4G-v7.0.15.M-build0632-FORTINET
    "009069b6c4448e234578fa4072f481b97d92af7518b50bbfea5a03cbcc38ec08": ChaCha20Key(
        "d103ac2da9bb564e3400d45b4d1bee7908f385048b521dabdde5bae9806e4b48", "a833dfcc04100d9febbef506aa2dc33f"
    ),
    # FGT_4200F-v7.0.15.M-build0632-FORTINET
    "a5a94501ed74c43d8b702228d50986fb237ee104da64d5f2f36435e63fba059b": ChaCha20Key(
        "1083ccb867f9cb44ac133aed614e64138d3e22df5da04cdcc3bea680cf3f160e", "340c322c2a5f372c9dcb312c54410eb6"
    ),
    # FGT_4201F-v7.0.15.M-build0632-FORTINET
    "dc23c12eec5c0fbc92ee088a6f5d959131696a7f076291d07f9f7e5e3b62f93a": ChaCha20Key(
        "8c7188cefd99fb225598ab1e1fd375d40295a597cde0e178283e90403406f643", "0fa96bec87c848b681e63505f184ba1b"
    ),
    # FGT_4400F-v7.0.15.M-build0632-FORTINET
    "3bb60a7a12b7e85211b28914c9bc2406e63c98e09b872c3749cae7cfb039c373": ChaCha20Key(
        "d9fcfdbb0c44dd6dc2777ccaabfdf62d437bff5fe9d429b49ebac7401da11e71", "ee7e98cab0ede7d1bacdec77bdcfcb33"
    ),
    # FGT_4401F-v7.0.15.M-build0632-FORTINET
    "c3b5c06042ed99fbc85daa67b9d076592cd4717e0d9cab20326f9d20acb2b009": ChaCha20Key(
        "1b643c9d07c66edf7434e62590c0687450e4772f6cce2a4d821188d572d5c224", "87b1df89bfd31fdeb9ad85a2878ca61a"
    ),
    # FGT_4800F-v7.0.15.M-build7331-FORTINET
    "0b78465c0302991f302bad026a7b33608c1e995b831f5b0640c3b1169c7bb18e": ChaCha20Key(
        "1c136f1e13572338d1bb3351e274ebb21e7fd2d52c0cf98e9bf9ac829844aa1e", "d36667a0b342f660098da8fbe5494dff"
    ),
    # FGT_4801F-v7.0.15.M-build7331-FORTINET
    "14ee0b40c7f989aaff21ba330b6d94079dc4df785499e98349e213e1cbebe1f1": ChaCha20Key(
        "018e5f7cc646eceb3219c759e667ce8c62fc88ed66e16c9e9457faf813e1d135", "20c8c3cb7ea47ecb9f14d27a2ce62212"
    ),
    # FGT_5001E-v7.0.15.M-build0632-FORTINET
    "57d3dc2e09411e3e6ca0b1c661257d16631b5179b6c5ce1dfa56d4d63e783034": ChaCha20Key(
        "dc22f20ee722e9205b1e2a801dfbbf445cc9b2559253e832c44376766f03306e", "75522a2bd255b79d58c1752f09e55121"
    ),
    # FGT_5001E1-v7.0.15.M-build0632-FORTINET
    "92ea603edee66d21847a984322004700ba073af6554a7c74f6d3cc76868888ab": ChaCha20Key(
        "5682affa7ae66ee9df77648a020c25caa7efbfe31c8bb64fe7c4c42f9c05930f", "4b4c26ddeeee251d53ebeb5e7989e569"
    ),
    # FGT_500E-v7.0.15.M-build0632-FORTINET
    "4fc79169c9641c8b31c8c718d734c13b201ba9fbcfbe609c9266b01391d149f3": ChaCha20Key(
        "7b0a60764f8501bd93b3b0b94a71fbf0d555672f13bf1679c93a638556b12743", "42cac3141472555f6021306164e73903"
    ),
    # FGT_501E-v7.0.15.M-build0632-FORTINET
    "6dfb52dda878fb0948562d34ef4065d3faf693a5ee1338bbe84b046a0d5bd2e8": ChaCha20Key(
        "abcef20abd866a77c47797a3fa3ecada28b0d5297f0278702af5a8141b46d41d", "3ecf87f9c3ce13ac05e796a5c0e6fc89"
    ),
    # FGT_600E-v7.0.15.M-build0632-FORTINET
    "875a522c26883037c64ac15fac194cc3023314201cbf6cc2e78ae11cb8fffec9": ChaCha20Key(
        "bde785ab453c7e4bfce7bc991acf7147dcc8ec26f5ce443fbb2abe99e71be127", "dfb66a510ef15b11d59b8b875855311e"
    ),
    # FGT_600F-v7.0.15.M-build0632-FORTINET
    "79153745aabb7c91e355b754e78ac3a9cc095064df671bce31bef41e920ed72e": ChaCha20Key(
        "ee914a1b2bce86e549ce41c8914a587d7e7111fcea0644fa53f9b4ed5c620a39", "ccf0b944570cd0eec3e14f0ae9c146ed"
    ),
    # FGT_601E-v7.0.15.M-build0632-FORTINET
    "64f4f0bb39e104b48b54ced3db9ecc97a7efbb1565b94df0cd0f44b77d080719": ChaCha20Key(
        "526fd6d83f6acfd2708a110be93bd91470679da19a1d9a535811a1b252829048", "d1af8ee765ef90df85c224232240c3a3"
    ),
    # FGT_601F-v7.0.15.M-build0632-FORTINET
    "83ee277884e80716505cc7dcdbc5fdc31fd511e4f0ef3621dd95e722e416d8a9": ChaCha20Key(
        "f4f9ca604a9c0291a9acf5e8ed3661ddb1efd5d72c518227f4e2f262d761a124", "d9263bcae6ccdbaf0e00921d156a830a"
    ),
    # FGT_60E-v7.0.15.M-build0632-FORTINET
    "f015c57f7df62267d615fecb9bb31a4388a28a2650f1702e553a29bcbb1c5184": ChaCha20Key(
        "46c3eda38c2ef100ef6e72a3f33191d1342442f23c18445044880b19ada913a4", "b60bf7a645ae8927b9f961ed0c9bc33c"
    ),
    # FGT_60E_DSL-v7.0.15.M-build0632-FORTINET
    "97bb29b4733f356402cf439d7d511f0a8a90d5a251cbd9b072a961a7152b5840": ChaCha20Key(
        "49a30b5152d5344057664a0e437b5644db02bdd886e4c0d88446f2c9e241f0a6", "bc84282c5d90f57aec5c27a470d9ea99"
    ),
    # FGT_60E_DSLJ-v7.0.15.M-build0632-FORTINET
    "f063fbe0d5f3f358b840cfcf0e913b33703f39bb7588f68c7d65f86fd898550f": ChaCha20Key(
        "903cd2d77fe4eeb66cd7bdd23fd081c1f16e5d6cd75d01c2640f6b1f139959b2", "269fd1db75e1ccba707bec0438d5eed3"
    ),
    # FGT_60E_POE-v7.0.15.M-build0632-FORTINET
    "52c88489d4fcd7148699e0a65c72c69f550490cbbb5626a8815372b4d677f4cd": ChaCha20Key(
        "15e9bf8d2126dd09881dcd709a689266b0c78e35f15998635742f4def0ad4747", "b76f0684f78a8613f81e9171d0603bd5"
    ),
    # FGT_60F-v7.0.15.M-build0632-FORTINET
    "796be9208662279cfc0c4801b94f8f229a21064a9d0f835bfb0a2f255f97496d": ChaCha20Key(
        "94a46e4d1d69489923428feb5952c6989e0b54c958bbecd755ad2b2ba4f7edc9", "5e9121dad6cdda73bf181f31212cf55e"
    ),
    # FGT_61E-v7.0.15.M-build0632-FORTINET
    "0eccac197c1ee1d5ff479b4a92c06c73cb48a041bf8bcd71932eb7d4568bc6b3": ChaCha20Key(
        "13064e9705b9c237a351fb7b958634035ed41f0fe247aee0c9505403df9f226f", "a459107ab0b2ef658d71a9dd4aec61ac"
    ),
    # FGT_61F-v7.0.15.M-build0632-FORTINET
    "7434a2c382e79c58651c799e430057dab0f80a15a9b78d2fe03be66760e458b7": ChaCha20Key(
        "1c1213cb38deb7c9416a704a7b4beada87ac772043b1c66c690d6ae2dadf4bb8", "c1a30df532fa1ad023045d4f85a609fb"
    ),
    # FGT_70F-v7.0.15.M-build0632-FORTINET
    "763ae32c3d2230343694230ec1f99b2c04743733249d329ea11224f9f642b6ab": ChaCha20Key(
        "c1c157fe8cd6c09084f3cd0138ca5f253c5a7f3022b91dc95ce3b940f55ce61f", "a0cd81687db1b49d7ef99055368fc30b"
    ),
    # FGT_71F-v7.0.15.M-build0632-FORTINET
    "d2c938333af0d933277a9229bb76be1d9a4405323bf8ebe5e56fd75845051cca": ChaCha20Key(
        "da1aa183f61db2366a93a37007742cf981067b5ab2ca0c051ec03dbdb1488c63", "5bf688661d4baec2e60821d535bfe120"
    ),
    # FGT_800D-v7.0.15.M-build0632-FORTINET
    "f42efa6ca1f50cf9c156b85775d9e72185a494806bb103de46c60924e52ae137": ChaCha20Key(
        "b38aa3776efd41d5deb3da943a9782b3ee2ef30cb33fc85b728dbdd2932e5c32", "ceef22fd77deea2feaae2522abfb22a9"
    ),
    # FGT_80E-v7.0.15.M-build0632-FORTINET
    "89b1df9838d48e296417257b1d7b5fe54d07f45fe9b3acc7f95bed0fcb1176fc": ChaCha20Key(
        "0d2f4b58d0429b5dda686f3709036e9cb9029d2ab7d16d5645385536bb19b101", "b3c8fe50a8914e2a2c4ba4dee80d88b8"
    ),
    # FGT_80E_POE-v7.0.15.M-build0632-FORTINET
    "49fb066d06bb2e6c5eaa743157377d11207128fa1528a39e2e95991ae3d874c7": ChaCha20Key(
        "892a9b24053ad99545ea6f55b3d9fefe9c8bcd88d4fe5dfaad83dd944df48519", "9b85ab647d3bd0b9f483684a7e2e44f4"
    ),
    # FGT_80F-v7.0.15.M-build0632-FORTINET
    "9578c15208c6142a71ea01c4f48f9a78c7c787cf92c8c00aa46d9ec9036449ed": ChaCha20Key(
        "334eaf8eff6f658192bdb4359d7df60d1437a75285ccbdafc59da74e39c7cfee", "dfb011002fbe3ab8782e0a6b382859b3"
    ),
    # FGT_80F_BP-v7.0.15.M-build0632-FORTINET
    "acb2d3122da9f802f4c2304e016273fcb27a17138bd76bfa9bdf70827bc62308": ChaCha20Key(
        "97128761e3ba5751def376557c0196baf6b9f09d08ba2afc04268ff708ba819b", "a0ac65bb2447f73bb4271714da597028"
    ),
    # FGT_80F_DSL-v7.0.15.M-build7338-FORTINET
    "6e7c41d02d6f8fb8b95a3a5ec76800f05862dade81e3993b0650a9802dc175e1": ChaCha20Key(
        "66ecf265833f0af0b044cbc16991f942a3052d32070dd8756378d1eea275008d", "8e2c45c2ef73e59fc6c1251f0a9672c3"
    ),
    # FGT_80F_POE-v7.0.15.M-build0632-FORTINET
    "51bb42d6f19578026e15116729ff20184d0dcd5fdf8e0458e968e64d9eda3a65": ChaCha20Key(
        "e87e396bf02f891163c92237fab3495359c08fc7f634d3f57b668013ff3026d7", "ddb47b703c8824e661ac2968bfbeed8f"
    ),
    # FGT_81E-v7.0.15.M-build0632-FORTINET
    "f465f4454095c97d70f30f0a4d1f0cdedc01fe4b9310c40daba76c9a206f4b03": ChaCha20Key(
        "a40b6f3012bea08919a838e5568e472a0f03dcd916a59a5945df62542d9181a8", "8ffbbd9f131cdc15be2ee6514e389329"
    ),
    # FGT_81E_POE-v7.0.15.M-build0632-FORTINET
    "26609ac7a1f4838f8f2d99fe911f80816b5a3abb19218fcbce43449da9c42261": ChaCha20Key(
        "e349bcdfd2553e4f5cdc32710b7b496e7c51634db290d44dce9e8fdeb251e7c8", "2f207156ee13e312ce81d7a3abd7567c"
    ),
    # FGT_81F-v7.0.15.M-build0632-FORTINET
    "cd9734cfd2d1c92066cec0f591c644b3333e15883bd3c49bcc6525de66cf68ff": ChaCha20Key(
        "5b6eb119d5cf8e1aad7c15daf7427eb7f9349a7dae01cb935ea03b9ee96ed29c", "aa33848a7d3628eba24609072bb12137"
    ),
    # FGT_81F_POE-v7.0.15.M-build0632-FORTINET
    "b9ae2b84a5db2e879e9a63a3052d365e552dc2e9cf799d3f88874b227623ea47": ChaCha20Key(
        "eec86f7bf5418df6ad7a29b10be8abfda562683b13fff1a30bb1226dda64cf0b", "b6d6d0096f22f19c1200297a426dd2bd"
    ),
    # FGT_900D-v7.0.15.M-build0632-FORTINET
    "f3fba42d531ee2d90013da65b98fedc6f34e5ebc9f889f246d934fb90e3f297d": ChaCha20Key(
        "f335132b23d5155a51328b91873befb115bd9e6966774908a526d3ea73423184", "d558b6a333999b0a4b899edba2373989"
    ),
    # FGT_900G-v7.0.15.M-build7329-FORTINET
    "41b673ae60f38af0d1588727af9dc35aa9086b6438bb525f1252201528c60359": ChaCha20Key(
        "e3a24b7c1dd82b01357d950cde5c9de5bd44b7e156195b3cddd85259d5d4173e", "db2d7f7b6feb99704c4eead181a4eec2"
    ),
    # FGT_901G-v7.0.15.M-build7329-FORTINET
    "85284043a880ac2193de67d705e8f9113bc35b0e22717b82c379ab6430191dc2": ChaCha20Key(
        "a2d5e5296631d6f9fced9938a8a8ab998ccd688d90168bbf15b9ed45c1b93b74", "19e80658199b7cfc06f23fc5e70fdb7c"
    ),
    # FGT_90E-v7.0.15.M-build0632-FORTINET
    "6156d400dd446d2d60a79536252ccb844fcabe2cb9ea1b03996bdf25a4878f80": ChaCha20Key(
        "0f1cc198b80b9b855919ed2329ead83f9800a1d69273b86bcdcb138445bab7e5", "89999f471075ac8bbb5199a350f50fbb"
    ),
    # FGT_90G-v7.0.15.M-build7342-FORTINET
    "ce61b9cef3a64d7a9be25676deff66d835be9091ac23ce5a20cd95e076eee009": ChaCha20Key(
        "6826aae6a3a36d87c3414dd9d162042b515561ff1985006303df13f92b5e5c66", "44084483215f324153bbd6a990656879"
    ),
    # FGT_91E-v7.0.15.M-build0632-FORTINET
    "ccb8ecce69e311bc80e7f02ed730654f5a14020509d8e5ad12d5836675804a73": ChaCha20Key(
        "fdf731f012abb30a7666b728cf2a4ba311074f4e9a0742d548f80c8998187897", "ee1b63f43f7b5fcc1f7602784d0b7036"
    ),
    # FGT_91G-v7.0.15.M-build7342-FORTINET
    "7f5e1a5654eb33c8a446e70f9e9782d580762de0a138566d6dc6e1ca33cf3365": ChaCha20Key(
        "85160e1648d941707bc96e802756d991d3c827900521f669fa9b7740209e05e1", "78783fcdd8a930f9e7a2c18650e79b81"
    ),
    # FGT_ARM64_AWS-v7.0.15.M-build0632-FORTINET
    "469ed5a58068b51250a3f88ae56214322a7454f4327f91eeecbc6c7d0a311cc1": ChaCha20Key(
        "5e5890df035405543ed38ae47e74abdc3156469cc2e906512ae459c422743711", "ff4207f4dad5eef50ec21442bfee3d0f"
    ),
    # FGT_ARM64_KVM-v7.0.15.M-build0632-FORTINET
    "a9d27ee5e97fca1d482637ed900e2fc2af431f55f0b93eac3f34f1eb11cdc78e": ChaCha20Key(
        "74bd1ce2b36264675271a1dae3de373cd30d55e16485fdf92247bff5750260c9", "d452d1d8954233f54e7120596e934887"
    ),
    # FGT_ARM64_OCI-v7.0.15.M-build0632-FORTINET
    "0c8144f43e134a2832d5cee1d2b6a955b31db7eb373b048ce269c2aea925756a": ChaCha20Key(
        "2df67c300e7e84e548089f11bf2a6003e396dd658229f64dbe1147151348bb2d", "5ba4f0af11065365ced5d3c3dbfdd845"
    ),
    # FGT_VM64-v7.0.15.M-build0632-FORTINET
    "479607bae53f1ef3b5dcbaa0624dc7addd666ed4d7f6a3f5017454ae945fe4f5": ChaCha20Key(
        "36c39baa993e91eca8e99451376ed5d28f11cdbdd3ba2d0897b35028a39d2d54", "9c11ae9f9209c7cbcdd30b26d689ee4a"
    ),
    # FGT_VM64_ALI-v7.0.15.M-build0632-FORTINET
    "51ef3cd0783be56b5dab52266aa8fcee4efe9580d49cf1fb4fdff3a62b61b5b9": ChaCha20Key(
        "8354273c8761a85018886646a65a1608c4a83a6bea381bce98fcc59d4951e472", "75f6e04c9e82f584fa6eb43a61e18db3"
    ),
    # FGT_VM64_AWS-v7.0.15.M-build0632-FORTINET
    "a5815bca0c2ecb74b8773657c32851b292c4aea16ff3944b39911354f43ab766": ChaCha20Key(
        "f741887109443da2833beeee0293c398510810113ef9ddf4d2dadfcd0dcd61e4", "b0e80ca1ac87d3f077826476b770dbb8"
    ),
    # FGT_VM64_AZURE-v7.0.15.M-build0632-FORTINET
    "1883912f018f18e75e2b5ebd0f84e76f4549650f7ba150f09ac21fd27fb52710": ChaCha20Key(
        "4a7342fb341d92888d8a204abe8ac350181919b615b24b256a3183d129217b01", "c1ed542faa39a332389c9f7754c2dfee"
    ),
    # FGT_VM64_GCP-v7.0.15.M-build0632-FORTINET
    "a27b560e2cd1e772016ad6ad4d93ac5e96e91ef3000f68e8c695fbc4a8f3afa3": ChaCha20Key(
        "1898f8ed484d3f97f5856f4ca5d28f11cca5980c173ae8f4580dbbc057685b0f", "4961fc95a9fae772ef39609bc64b9ff2"
    ),
    # FGT_VM64_HV-v7.0.15.M-build0632-FORTINET
    "b8a94925602a4f8ec4a8bde97e8e56279b88f2aa5eb6f060fedde447d174a8d0": ChaCha20Key(
        "f89351f443055bc5b0c635b34d9253209788cea353ba0651199c500cc3f7bbdf", "2e483ee795cc81a3ab2ed36b08ae40d2"
    ),
    # FGT_VM64_IBM-v7.0.15.M-build0632-FORTINET
    "f4a35b2373204b4db4065c5444a5e267bfbfe1cfa7169166b30c529559733312": ChaCha20Key(
        "70223a7387364dbf21ada5c1f749d1044f02166040d168e1a5f267b31722a533", "640ca7f7b5614a3aadb4dd77a56d141a"
    ),
    # FGT_VM64_KVM-v7.0.15.M-build0632-FORTINET
    "4fd8275fa2d334d3d76ba36a7aa7b1f2a24f72731f302e916fe0e882078dde72": ChaCha20Key(
        "585a2b81591cf2f1cf547bf1bb5b8beba17ab179d485a5472a870c055e768f1d", "e2b0c71240f0162a775f77806cfbc5c7"
    ),
    # FGT_VM64_OPC-v7.0.15.M-build0632-FORTINET
    "c26e2edc02c3bbde5f248760c37a5a0ee64a78a1327f957c629a1c205d3283d0": ChaCha20Key(
        "56731379cf070205c4d4181a7acc363c01a506855033ef7392ef669677ebca04", "5b8217483683e03b534fd98225eb8cf5"
    ),
    # FGT_VM64_RAXONDEMAND-v7.0.15.M-build0632-FORTINET
    "eff1a87f9a426446c6f9d8a1ea4195552a5841c85e1d71fb0a625a0ac6c6692d": ChaCha20Key(
        "c62d6d7544f6289bee83eb7676f5bdbce79cc14c941694752838a61e2880a268", "f9080998c90f4c2116609d074c741bf1"
    ),
    # FGT_VM64_SVM-v7.0.15.M-build0632-FORTINET
    "17cacb3a3567a908cd082acc14e80acdbcd75813b887344fde367131d0ca8422": ChaCha20Key(
        "5140ad11aa1c6d5ebab7d0990d51269c5697603503286a2aee7d4e74d87b740b", "af68251d4d67e19871170a80d4799496"
    ),
    # FGT_VM64_VMX-v7.0.15.M-build0632-FORTINET
    "588b85c1ba3dae69455161bcc0e9dc73083c3fc388f2bd78c00ac87bc13b0690": ChaCha20Key(
        "5aa1028cccba5a26cd6b9406415473e3696f844ea34fbb33d795b3b779ac3852", "7d07afa7fa1bc7b53a4b99c1cf25247b"
    ),
    # FGT_VM64_XEN-v7.0.15.M-build0632-FORTINET
    "5fb7c2f87d6d84716702bcc9f66cb0cef38c2cc5ec80ec483e6b992c3d7995b8": ChaCha20Key(
        "26ce84b28135567a4775f8303c53fcec9707e7342d00c3cbdb8c4fc4c938ffda", "1d85df1ef5fe2e8506022a139de52e78"
    ),
    # FOS_VM64-v7.0.15.M-build0632-FORTINET
    "f5eb5ab741ac6c772db199a2c168b2f28d68f030ff4326fa1c176db3b9f8d7ed": ChaCha20Key(
        "c27e68b8aca6fc40acbd8a0fbf4d01f042e21a1762ad37a82e3e2b40f999db2b", "3e6906b6c639279c4f4a80703360247e"
    ),
    # FOS_VM64_HV-v7.0.15.M-build0632-FORTINET
    "0150b508b4a309ea9c1383968afb6b3de3d20a100040c519e4e32e1384dd41ba": ChaCha20Key(
        "04a85d57225191dc814889e9b1fd11efa2e2eac7de83e2859ba29492772334da", "851989682ccd30697e4425ae3f9f26ee"
    ),
    # FOS_VM64_KVM-v7.0.15.M-build0632-FORTINET
    "dae914f59c467f450f81ee4b9d4992f57791fb07ad2c1c20e1e6df4ec3f330e6": ChaCha20Key(
        "cdc06bf41890d60af2663ac38711de363e5302e3f518d0ff1f080e9f69fe0d3e", "17e08e75758b061c65e2b9d5a1e8d8fe"
    ),
    # FOS_VM64_XEN-v7.0.15.M-build0632-FORTINET
    "a448e7bf056a078103d27a8419f10a1e9e852e6cfda4286c6b90e064683c79cf": ChaCha20Key(
        "24a646e509d0f8d8ea68d705121462cc99fb73ce7294fd1b976ae2d7a3ac03d4", "094a541337000f289e05d8329df43c57"
    ),
    # FWF_40F-v7.0.15.M-build0632-FORTINET
    "47307f23eec8d5941db6834e17fedd19f76a1ff89d8bf82c7cf89f3978edbdb6": ChaCha20Key(
        "495f05248f6a9f457ba89f48356e6c9f95b1b3a17ae82257a15f4e673a068988", "694e01275fc868e818772097c57b9989"
    ),
    # FWF_40F_3G4G-v7.0.15.M-build0632-FORTINET
    "e95b625d8ae9820a68fe259748e6ab71b11edfc7f835de5bab2c4a8a3af83a83": ChaCha20Key(
        "a47deffd42d547ca26b4961dcba1fd29c9b73f5b3b107c32636c6643d3f676ff", "f7d3e5ceabfebbeb9af7dce84cd09249"
    ),
    # FWF_60E-v7.0.15.M-build0632-FORTINET
    "05a356ecba9e3e5a012855cba99f2a0ca61f664aebf1f88ca0e9209ea952787d": ChaCha20Key(
        "a0310b4da5429bfc9dc6cdca517430205c7714b40aca561d2a6869c3c5b71d0f", "b81e53c461dddc1e1c030f74c666df10"
    ),
    # FWF_60E_DSL-v7.0.15.M-build0632-FORTINET
    "0d4f80be73ea027f1bb5c06d63a53ae8fd30306e5350e901ff40f147c2201c0f": ChaCha20Key(
        "f55bf3960509c4c7ec292d41649ac59938db3ead2f8deaec7c493415360fd31f", "134eb9b366b163b310ffdd352aed6ac8"
    ),
    # FWF_60E_DSLJ-v7.0.15.M-build0632-FORTINET
    "d09067ca41953af6fc6ab87b542e4c4c7fb2c6b12baf48e5304779f55786086a": ChaCha20Key(
        "1010f043aba05c6a5ee0d2393d989e1bc8045dd87d8932f428a7a4ba893c8dc7", "590d2df0fc30e8354a92d315ce712d81"
    ),
    # FWF_60F-v7.0.15.M-build0632-FORTINET
    "1e73f06615384deecd96f147582e345909a7afc156730a5f00bf91dabde16b1d": ChaCha20Key(
        "6d3d85bb2466f27ebb5ff0dd26226a5ca142170cc7e2bb60a500aa85b82bf8f6", "b5f590a51b07412ffb81104d3f66d2d7"
    ),
    # FWF_61E-v7.0.15.M-build0632-FORTINET
    "8b6f56af39dcba2fa719c92b503f656eb8d9a341eda95cd479c6464d8c2f25d2": ChaCha20Key(
        "c40d6a637528e56821ae588a4b7cf6c9a91d2f51a2d1e2a4c81d1c4f569d41cc", "214f084ace70cc7664ee445314b16566"
    ),
    # FWF_61F-v7.0.15.M-build0632-FORTINET
    "ee83accdb95a437d8769f367ca920750f31ff8ac348b278d49c3e9dfb4eff110": ChaCha20Key(
        "289a91218af391ea528e2c4ff36f1a0c657150c32169b01fd3ee461571a43217", "26efbeb52ed5b3dec0ba26c9f207c313"
    ),
    # FWF_80F_2R-v7.0.15.M-build0632-FORTINET
    "1d21ded47c45e61222f352de6f3b6868c6b7215c5daf31ce39a1e5d742e76534": ChaCha20Key(
        "6f53e52b5159af271ad7da90fe87fe56279443c22081d8143a703ae5cdc69aa8", "c2892fc1b1328e8071d8d6c97a131dbe"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.0.15.M-build7338-FORTINET
    "a1e78807b96f6e035e1709d0471def239426697b62f85691de8ceca3195da23c": ChaCha20Key(
        "c10da6abf1fbd1a7bc8fcffe1808388e8d1219daa353992a1acd4e25f14a55c6", "f849a3c6c5b37ca9727f27e09485da13"
    ),
    # FWF_81F_2R-v7.0.15.M-build0632-FORTINET
    "ed169ff84729efe8d75d426e612979e7596d3c27c4aa09b938376bce2d5e7a7e": ChaCha20Key(
        "06f9456a4ef4cfc13e7338e44f6e4ce9cf9ff6acb5ebb17e5a29857e81918cd7", "c0325f421b481dabe517438a5cc342a2"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.0.15.M-build7338-FORTINET
    "d5f6a307350c059d40d822e113c58c25a27da013cd13a40da7a4b47582f2deff": ChaCha20Key(
        "03571c8e4e644d099f06094fd5e1a30917aa9bedb1e1e0c167fb134abdd24748", "42ea91872baec2bceeaa91039b1f543d"
    ),
    # FWF_81F_2R_3G4G_POE-v7.0.15.M-build0632-FORTINET
    "49bd7c822a60ac4f6a13a42ceb3660cd1f8b9709d0c517686c6b9ecd2a20fd3f": ChaCha20Key(
        "7cffce4439bcd3caa89670acd0f11d68f05fccdd070a4f5c18cac5b605e0bc9c", "1e916833e8674d7394983e1080fe2242"
    ),
    # FWF_81F_2R_POE-v7.0.15.M-build0632-FORTINET
    "3999571f0e6e65f67ff7a35105571244b71d6af732567cd1e9e6425fb6a1e08d": ChaCha20Key(
        "061c64cb90cd146ec3cea537a15d62c0dc711c1728df91de6cf6b9f8a0260894", "c3fdeda8acfcecc84b474363eaa65ed2"
    ),
    # FFW_3980E-v7.0.16.M-build0667-FORTINET
    "388545112733d00b412d07f7c82a6a0e4a6510232fe87f511bf10290d80d55bf": AesKey(
        "6940fac6fe22ced0593bf0c63c36c62c35537e0bef8aea188c340efc28d43d8b", "888b3c5ce81c2f850e7bc6cb671035c2"
    ),
    # FFW_VM64-v7.0.16.M-build0667-FORTINET
    "3f614148faadb9a7f47e10f109ca5b5d442c8e2ecb575f0efa5ac118578d8f78": AesKey(
        "8e6f38bb122756de8f8db5f1fb52089c540381747e6e89594b305e06c5a1cfb8", "7164d0023409d21d97ba30a1019fccbd"
    ),
    # FFW_VM64_KVM-v7.0.16.M-build0667-FORTINET
    "10f9052c1f1c6cc3e816d033353b200fdc9d4f547288c726f1e7d81ece7fed0b": AesKey(
        "c8210ca6f5813f71cdcdd2e1bfb2776976b224c6803c8f2b442d3abcf4c32548", "77fa8ae45926a25d742a23505d6c9c1b"
    ),
    # FGR_60F-v7.0.16.M-build0667-FORTINET
    "c6813bae42eab54c68991c5900758059495814a65c4c28fe47ce4561a042f88b": AesKey(
        "ac39be98bef6238ac6848b39840e67c413a448c9ae11e698f3fbbf2cbb18e273", "3e40ca96d7071001ea1ded452e676a1c"
    ),
    # FGR_60F_3G4G-v7.0.16.M-build0667-FORTINET
    "0f53caafc13ab8297251dee8c09fb661917b088e0369ee123fb1c80b9a1dabda": AesKey(
        "659f48aaf5238e3eb744d7b1329e3f747031abc0eafb420d6814b35637a42103", "669185aebcdf117c839e3963614f96ef"
    ),
    # FGR_70F-v7.0.16.M-build7515-FORTINET
    "9f85ade4bfe489982a34151d3a5fdb9bd2842cc1297b3598c47f23ad63ed428f": AesKey(
        "cc3b0d0d7a7a4d43cdbb1043f5b928ee8b00bec58a30678f7db42424329392ec", "f43de24e03d9ff7c48ba4f4e6e34af65"
    ),
    # FGR_70F_3G4G-v7.0.16.M-build7515-FORTINET
    "0c3cbec493589e4005528dee26bf5cb35aa0213a828579c8f549fbb11d16f8c5": AesKey(
        "10e448953c5c3f096cca453daa3269df5189a2ef8ba83f83b3b4cd32aeb171bc", "73600673ef31f00e998304803d5a407f"
    ),
    # FGT_1000D-v7.0.16.M-build0667-FORTINET
    "2fac9de68cdd10b947f355c9a17023686986df18f203b44e449a2ccfaa7d5ad3": AesKey(
        "68a6641ba0a9f45fb34dfcf099a78149ddc5491045ddbd5879dbae306422e0e1", "e2bb6decb50dbbcde0ea4c4fb56113d6"
    ),
    # FGT_1000F-v7.0.16.M-build7523-FORTINET
    "29e4a32ebcf6f8ae7016a605075c43a49d443f0bfdc9480b7ba9a8fc49f0725d": AesKey(
        "5da8d68b8ea55250bdbb67b3f9ec511999d527ea0ab726b40075b039f8cf0b9b", "f1d5d5e53efbac5eaa6dd3d78eb2e3c9"
    ),
    # FGT_1001F-v7.0.16.M-build7523-FORTINET
    "d72147bd7ef23404b4c27a40ddbf6a0d2f5b44d5adcc1e1eb8387777a82e76aa": AesKey(
        "876499f4729370b2472ab13c1fb49c99c641e9e153fb15685aa5008029ef38b3", "df24d99f896298e49c7bb21bf5ef75ff"
    ),
    # FGT_100E-v7.0.16.M-build0667-FORTINET
    "63f1157b9b9c2eab210171f3ab0188c4acbd482cb85bda44f0c0aab2c1fb4da7": AesKey(
        "606d8a590c5a0f7fad5450c7a8cc2e2cc75fc70f0b055473f47cd6f7d795156f", "3269c9dafe5996273571c7e4da0b72d0"
    ),
    # FGT_100EF-v7.0.16.M-build0667-FORTINET
    "50b31f06c7ac424fb57aaf912e176e38659dec1d377f0f0c4f061fecf58ef60b": AesKey(
        "de1a460f29f33b0adfe54f95d8d58c44071cd043bc4d0ca1d049cdb247f34976", "d87fb8a0050ef2c00a1ee5b29152567c"
    ),
    # FGT_100F-v7.0.16.M-build0667-FORTINET
    "a5f95eb5fa844013a369095a119a3f7efb835118de11e06652c1737f2bd37b60": AesKey(
        "d4b2201a4487c6bafddf2c7d7ef5eafa0f3cec7eac9e50f99da3dc3ed5e14a9b", "53d14c9bea834ec22343f6cd35f3291c"
    ),
    # FGT_101E-v7.0.16.M-build0667-FORTINET
    "9e7a764317fa2afd7ad71c8fbc1a33722b4b810a5d65b11dcbe4774f877bd230": AesKey(
        "1f93a4c7193afe64dd0a587c0ec28268b38fac7de007d010e4b02d079cf84029", "9fd985100568c7bb845128785729c918"
    ),
    # FGT_101F-v7.0.16.M-build0667-FORTINET
    "16ab55837aee23647bc323dd26f8f3bca51b9cfda0569af44ba445d4143cea44": AesKey(
        "220bf0f6e3395fede94d6044c9ace4b5ef39e838ca0fd22194fae3d00311b419", "e46edab73562110e03435844bbc44b36"
    ),
    # FGT_1100E-v7.0.16.M-build0667-FORTINET
    "6d6bb3895eaf10f67fcd592b66abf934311c9bb29b8a995f3b6f9cc7d2ac960e": AesKey(
        "b6ad5c7a0c64775df44e291bb887d0f8475ba1571aeed11e9a3273e1be548515", "70d4528d4c75a92c21cef88654b07f55"
    ),
    # FGT_1101E-v7.0.16.M-build0667-FORTINET
    "a7aaaa855e97854f19821534b725f4a8bf878ca957895b1575592ad12fc4b4b0": AesKey(
        "f9173b7e0d50a1d06b590d8c8fc6c494d4991415b5602ad82bb83b9635f30d89", "009eb553b245d2318b8f08d3cc8f97df"
    ),
    # FGT_1200D-v7.0.16.M-build0667-FORTINET
    "f84c74993694a9fb5c687b8939429efdcf2e3b06bc97ca4200b044b4d9650631": AesKey(
        "bfa00f9443a446af8f11e652636ef2437610db3542f78435bf7bda648e884805", "fdbfd2031f5672564d2a01c02a8f6b5b"
    ),
    # FGT_120G-v7.0.16.M-build7536-FORTINET
    "f06c4fdd040ecc1007c397d9f518dedb4a5c4f0016b6ff07e58b16a2325c78a9": AesKey(
        "6b5270693380312452763b385951d39ad402811db3d780aba203fb7c9cf38fae", "96b4903a1e657b28a077943baaf265a2"
    ),
    # FGT_121G-v7.0.16.M-build7536-FORTINET
    "b51d74890b4842ce91db89b155d49ee74b6f95d439bc8219c4821cb1bf8fbd20": AesKey(
        "5a716bc90facb408ef531f6c86da0b22677aae4b5c28c43db5d2913838833391", "5aea98437dcba50f1975092bae6bb723"
    ),
    # FGT_140E-v7.0.16.M-build0667-FORTINET
    "3dc603b19852577021983df328e065d3326a1e666d855615a43ef83f89a0f306": AesKey(
        "6fa561f9197c02ea4ac4f6c6a369f8add509d1ffbede00634e565699f76ef048", "9380150c6a1bb3800ef2e94d8d881595"
    ),
    # FGT_140E_POE-v7.0.16.M-build0667-FORTINET
    "bed1941ffd05d0aeb85746ea6e79f2067c02f6f2a090916c3a70e81a22872e7d": AesKey(
        "b19f8b8e9379640df71b487d6312248426ce075398155b4e29246606e08ef60d", "64572051c8a163313f94b181d775d2ba"
    ),
    # FGT_1500D-v7.0.16.M-build0667-FORTINET
    "a037859bd99fb77f06152b260e710332ddf33e05cf770b1b84c766e0365b5987": AesKey(
        "131163a0ef47686272021745f8dda9aea834e46c808e9267a8c86e5204a928ee", "23ad25524b505a308d4339671414eafb"
    ),
    # FGT_1500DT-v7.0.16.M-build0667-FORTINET
    "d1b9184ed7c205a50c07f9e802690f8fe7fd8cb477c5153a652e467daebfc173": AesKey(
        "36cd087f9b05e1f4187ac6f7956c13815ae09ef4645a047d8bcadfc6e572590c", "b9cc2f50e34351bba25cb0fab97ebaf3"
    ),
    # FGT_1800F-v7.0.16.M-build0667-FORTINET
    "62be4583a43f7fd400cf9a7767503b418969e27f777d33f2fe203cf6f24df966": AesKey(
        "72d5fa5eb144d55990302fc55a8ddbdcff71860ad2a31bfd9abd68de624ca3bb", "213dd01e5f0d724a7ee954236b4c61ae"
    ),
    # FGT_1801F-v7.0.16.M-build0667-FORTINET
    "5083ca544ecd9d984e0309855dc789e5c433dbe1abfd164725b86c7866329084": AesKey(
        "f14aece14fd1433cc76b8e553b69a70ec78cc5c30944111c866b5a199e581158", "d9569a7271bdb2497c5566ab266b3e05"
    ),
    # FGT_2000E-v7.0.16.M-build0667-FORTINET
    "b6728e422d825c31f2be4fa59c8f1a496a27fe7f19fa08ec772c14816e3c5339": AesKey(
        "b949b5680935eebddfa92858f57ec74f2a4d9ab6dff95eff15a7c4ca54a6f43d", "e2cf906a869894a1920dee1f121c34b5"
    ),
    # FGT_200E-v7.0.16.M-build0667-FORTINET
    "87c725746c5b68eda4a7bfe46a7620cdb806cac09fe677306833a80a732e2256": AesKey(
        "b497d6e1a399dfbda7ed634834fdf81ff562e214d49d1bc6590029d5d6e3855c", "bd4cccfe255b7482df3e677cc88caa43"
    ),
    # FGT_200F-v7.0.16.M-build0667-FORTINET
    "18743a04ed3331f92493a64bd68bae73398b595f012e804f4de00d6892a13f44": AesKey(
        "8792016ae9d77bf2daaee6f0c00542726ad806fb95eb71e018f9c893b0e37e13", "d3c530a6601d9e3ce1b0887aecc0f92f"
    ),
    # FGT_201E-v7.0.16.M-build0667-FORTINET
    "5af8579e3f6428310609087a88d7895a3e5d2650b17361248a83ebc3f724c4ec": AesKey(
        "fb34e17cdd4ef256080be404f8917f9244d6ca39db2cf76e71bb15736302f03b", "ee64b0f486c8fa5a7da3f064aed277c8"
    ),
    # FGT_201F-v7.0.16.M-build0667-FORTINET
    "0c075745f04086b25b37f9b1e5aad2f24bd28b9fd54b227c769264c2a6ce46a6": AesKey(
        "0afad6532ecdea260f6de23399208016a0ff1aff7650d24bcea5318d4d5c92b3", "7068dbe047b4d237fa7f677122271504"
    ),
    # FGT_2200E-v7.0.16.M-build0667-FORTINET
    "5037b704e5238c504e6464c3e8ce06e26e97ebf4e806a55bd7da8b079183f901": AesKey(
        "3906f3337b6f9a6f133cb2550e2d7385fbc59332ce9a0e966b5cfb19994694e9", "befaa424fd37b8d0ab2b5dc5a47c0d90"
    ),
    # FGT_2201E-v7.0.16.M-build0667-FORTINET
    "ecc5ef1f606bc28f4a8e94b2fe45aff00a2f5f1a1639d1651e7ba7d2dfc92427": AesKey(
        "a7c2a19f1633dcf92fdce2ae74b0aff027500fd201d7df8d3b3cec77f53c52fc", "2059a67848a1a52d05b8dd4df40c83f8"
    ),
    # FGT_2500E-v7.0.16.M-build0667-FORTINET
    "7f872697b8bc0fd91b9004e0c16c7042c9731edc7a8b8885fc0dd719a0a7cc40": AesKey(
        "b2023db71a40bc584fb639fbd44b1f8a639e1ad86c714f981e60e8b8ad822a1e", "378321e0a8c35d8f226425dc095fe904"
    ),
    # FGT_2600F-v7.0.16.M-build0667-FORTINET
    "4721bc0d741193d924fb82d182befc5569b88ba090c3a17deec45a75ffa5d31a": AesKey(
        "f1f58fa59af0a8ac6aeb654ef08bc6c169b3e40787739399feaca30633bab37c", "b11a5e634b73d23fce1c5d56c045ed9b"
    ),
    # FGT_2601F-v7.0.16.M-build0667-FORTINET
    "bc0d7041be07cb590ea702310695a9d6dc95fbb1198d2c89f8b7f728202d85ba": AesKey(
        "0818083d69007eec080f73dab630154df68f373afe1bb06a7c7f506f79aaea0e", "fa3882962d6582cc1de1cb27455db8b3"
    ),
    # FGT_3000D-v7.0.16.M-build0667-FORTINET
    "6b56a994fdddf83b2a495a85d930aca2f483ed3f8cc41d9c722390815a2dc40e": AesKey(
        "0b09d3907fb50ba7e7ad77e91c33232c461da9d579a279d3248854397026f1bd", "bf0af4a225e2cedd74088d71c5975555"
    ),
    # FGT_3000F-v7.0.16.M-build0667-FORTINET
    "752b6506bc09f73d2244d57b7d213f14c1095a2e42fd549acf8ea6363fc6a88b": AesKey(
        "8deeead4800822a95fb0d0840b4b675da270397283568b495fbfe27ba7ccaa67", "50f689942ebfaf68849368627f49e990"
    ),
    # FGT_3001F-v7.0.16.M-build0667-FORTINET
    "47d291ebeaf84589aab233bd2f8f657532493402a63c025d6a734c51a9f79f02": AesKey(
        "9f67b2db412346bfd79cc4b4f51fcdb58afb01c18d7e3dc898836deaf676cd36", "69541a03e61879c0ede4c05976b23271"
    ),
    # FGT_300E-v7.0.16.M-build0667-FORTINET
    "3645cde67ec98679a46ef70ac264158740b1d075c0c916f248d9410a25e0f110": AesKey(
        "26378ca005028025a223ef50ef24129fe4431723d88a7107b82c92c86ec3c7a9", "71e4de843c987cf97713a584f0fc3afc"
    ),
    # FGT_301E-v7.0.16.M-build0667-FORTINET
    "359265f0548c2fc7298e6fdc9d9dec8059d409509c7f7f71d6708d3c84437aa0": AesKey(
        "573f859656c9b505012d1b786b2d9a614d2bb0c961b07917950fc3db1fbba76d", "f81e89585f22523506df3ed31ca935d1"
    ),
    # FGT_3100D-v7.0.16.M-build0667-FORTINET
    "b4475f88baa12efb3561c8be79da33e56899b3ab57d3775d5683b03dd67d6c20": AesKey(
        "e0fc99f10ba2a0e3094e84927fdd74630ff8a0cbfb02edb0c68e825e365e1026", "48fdf0d7f0fd39c04be3138feb29ee11"
    ),
    # FGT_3200D-v7.0.16.M-build0667-FORTINET
    "1cc1a688fad71137f9096b1c0e6beea6575fd98e1f5e01a0d5fc6e80b4a7ba37": AesKey(
        "546cdabaa6594327adba923673699c616b3b7db31ba2409287a02caed7e5a473", "984efc9bdfb24accc3048578e73f29a9"
    ),
    # FGT_3200F-v7.0.16.M-build7528-FORTINET
    "dc0ff8a3fcd76f821bf68c452757c3db60eaa30c2099d49cba46b7ae87b4e1bd": AesKey(
        "720e45993db2e0433559c95763c3e904e933f36372be246e8169e01da2a71578", "5adbcdd8ec36406c9253ba5d784c1956"
    ),
    # FGT_3201F-v7.0.16.M-build7528-FORTINET
    "855103a0c694571284c2b040259e31fd2a651b47143fb067deb7572e417eaafe": AesKey(
        "d276930ac5874f04c38cbf2e3ed03edd00eef3b3e29f4d9b88c6be31b2b88fc4", "ff6c94bf0e9f8128e044ac594e89b5d8"
    ),
    # FGT_3300E-v7.0.16.M-build0667-FORTINET
    "c3f03499a4ff16e5d8a813e50987920bcdf1fde82d65f7c16b3b995f48d232f2": AesKey(
        "10a64e76ebd43f1904c1cc50e86b25a45ea6413820c39103357a414f67e5e4e3", "6a5ab5a8b0a49c6bb985e54da51efea9"
    ),
    # FGT_3301E-v7.0.16.M-build0667-FORTINET
    "15d96a6201046dff4bb1362040a582220f9e1322ca60038959c7d23a27fbfb6f": AesKey(
        "63fd130619d42816b99508f2b8cacbd0863d3eb4402ebf8784b0e892cf0b9965", "53475d140f0a5870679c83021c604ee4"
    ),
    # FGT_3400E-v7.0.16.M-build0667-FORTINET
    "cd60269a382628ac21215b0f01532f8592f6357a41070cae1804771bc2720cb9": AesKey(
        "4d79e6d8e9747a6c470d2e59967a9f000af307b57a3188bf9ae9874ee2d1661e", "e848031513d98f14a11a9b856367f68a"
    ),
    # FGT_3401E-v7.0.16.M-build0667-FORTINET
    "cf06d30e1e4a73063eca51008268224bc4fb8db99a3fc6f8ceff4399d6d3388f": AesKey(
        "cf39ae2076908b5d1dfe6a9d4e9d2a6cc499f09e8ae6a9af68ac94010604df8b", "81137df996d8327c7437ebf542851f01"
    ),
    # FGT_3500F-v7.0.16.M-build0667-FORTINET
    "a15db53a8425869f5a753e94882a2160029195c8288d9a8eec697c037fabbd56": AesKey(
        "e2494a0eba3d8561e7bc486bc74847db950037d02abeee18ce56182117027fbe", "09e3dd7f5f4cf4ec85669ecec784c0ad"
    ),
    # FGT_3501F-v7.0.16.M-build0667-FORTINET
    "f74442b954f5d724af596f496111791d6ab7f1bddc6115da80dd3ad15ebf9c92": AesKey(
        "50597e17ced69a6a0a4acafda3aadb0ecefa6d2cdb5bbc377d9fbe031403f4c3", "b256f6ff681a44826dcea6d7d874e0be"
    ),
    # FGT_3600E-v7.0.16.M-build0667-FORTINET
    "cf6db36c261e01eb004043a7f5f93296423b6a0b1b880e1dc550f8bb42283ca8": AesKey(
        "af273b6a0b14e8153022367334483a9a1208d3ad2f530b8249ba3bc4d31c9b8e", "e6d4273940932b9b30078d1034824de6"
    ),
    # FGT_3601E-v7.0.16.M-build0667-FORTINET
    "b0d6c0bbd1415c9fbebee9cffaae11e64480f033e05e020b77cebe17a4c14d2c": AesKey(
        "2a6940d833093382de82578ccb4eeaff7a0dddc410b5b4e62570429d821366b3", "a19103b1f463c4e7c753cd5335c87ce3"
    ),
    # FGT_3700D-v7.0.16.M-build0667-FORTINET
    "1bb522dd49e846f4a5bc04539020815aac63451e17d8d37a936a05251e17361b": AesKey(
        "0cbbda695c2e70febbed1da5c8e4e97c909cf1dff51e8304c5563eef65dec66a", "6d56f4a33840787c84e0853b3fad9af2"
    ),
    # FGT_3700F-v7.0.16.M-build7528-FORTINET
    "4c3588be4b7f12eae760e6cd0a3709494fce2fa76a371bf98c31359fb2875b9d": AesKey(
        "d9163b5145a9b544ff21b16360847258dd22204aba0f58601dd866a944f381b9", "501a32d31e5cbecfe6a44662deeecd25"
    ),
    # FGT_3701F-v7.0.16.M-build7528-FORTINET
    "ee5a0bc708d7261224488f43ff390eabbb14afa9bbcf6d310231b5da0c918867": AesKey(
        "db4facce6c0b3ce5286ff74de9beb29f264149c7b4443e8c9c770f324a10cf2d", "9afc9b3a4832cd0f1b06d61fda8b90d1"
    ),
    # FGT_3800D-v7.0.16.M-build0667-FORTINET
    "c4ab5e8692b6b62a6aa3d89c96938b76f7a1b61908e86b53e6b9d3df1a4920c2": AesKey(
        "71e396a77308db2247fb4510e634f9b9e7e67cdd4fab3dc2f95895522b7666e0", "6cdeaa2d3962fc1836b4701a8190f893"
    ),
    # FGT_3960E-v7.0.16.M-build0667-FORTINET
    "6c5624c519d89a863ea2caec2555c937e7196b3c1c6a6b48d5a2272e453d76a2": AesKey(
        "3d563d4b07e56926cf3e779434dc2ef681560e8f7f22408edfce865666380b1c", "f1e5c95bd2f107d76551a1a575387f47"
    ),
    # FGT_3980E-v7.0.16.M-build0667-FORTINET
    "b862f887dac0496b4b7ce74baa2e8a19788ef9ea84221fb4c7182a277b1007b3": AesKey(
        "4cd92b0124dfe801d370f9dba842907a32ad1178f4e270c20cc7f94b28a6e844", "1a1d0800430dde16f8b54522518e0a8b"
    ),
    # FGT_400E-v7.0.16.M-build0667-FORTINET
    "023dccb1810368dd8eefb02e2cbea81cf7bce76a036c528fa52a16bb135ebf8c": AesKey(
        "3022857d3a851e992714b0e68f625c7b229fa4f4f7114ae13c83f3d79c7e2b46", "6a10046c15ac9c4bdd885ecfba5ed07f"
    ),
    # FGT_400E_BP-v7.0.16.M-build0667-FORTINET
    "6895ab2f51052fba41e7dce701c79e930e0408cc21cb0befb69769ed8ea5d382": AesKey(
        "6e75757ffdb0fa95e518154733d37870706d9d4d0aba9302461f8d7941581272", "befc46bfebf29e2db8832ca695f69270"
    ),
    # FGT_400F-v7.0.16.M-build0667-FORTINET
    "af7aa37635f0426b2b3c3bb6e901d30b1760e3d8bcd4fded2d14b698cb50ed48": AesKey(
        "24854ccce2b376e6083c07f1c197a83e8197d78bb3d860711559c185997a6e70", "ec4dacab32a1827366f3838b0b1309c7"
    ),
    # FGT_401E-v7.0.16.M-build0667-FORTINET
    "886ba211247908315238b0457641290c32a0712bd933c42709a46e3b9b5e6db8": AesKey(
        "d3c42facfef01683cc91ef1fbe6ac802c4a75bca72e280ba7c3053d15513ad0e", "103cbbc0ce43a17260d470740e70b62b"
    ),
    # FGT_401F-v7.0.16.M-build0667-FORTINET
    "c9418b24f400c155424eb58920551fd522e9ef6b528f58334aa4646167d6c514": AesKey(
        "c993802ab375ea14ad994c15069069cac1b468c4d33fbb9b58ce542818273f0e", "9a02aae4b95365d8a3e08cb1e14f4744"
    ),
    # FGT_40F-v7.0.16.M-build0667-FORTINET
    "c05d91d900913ceb49ad748c233f96b649692e2c88cf80c6a55c8a432c414dca": AesKey(
        "f0ec14deb72b4370583446a26ae92997ad6ec6fd8a6428054da5f3a078cf3541", "b60d3e343a51847a54b0004e315620b5"
    ),
    # FGT_40F_3G4G-v7.0.16.M-build0667-FORTINET
    "88420b348eed1aff583dd9309a2195d651acb8262857d9caea03296bc4a77993": AesKey(
        "138f03db7a06071e2c57edea007f7db0933c5df633712c01e9199803f5e0a702", "4dbfed8a650ced3b34f52a3f8f10770f"
    ),
    # FGT_4200F-v7.0.16.M-build0667-FORTINET
    "b0f717994041d8581ab429732117fbe82018474fdc04b6231d7e82d239cafc1c": AesKey(
        "b897f234545bfa0e0414a904d74ecb69d5fee6d62bf9ba2b608122234199ca97", "3c06dfe892ddab9ce92d3d2275a0ff59"
    ),
    # FGT_4201F-v7.0.16.M-build0667-FORTINET
    "da69d852471b12f1a9f625c10300722eacc6c12b7c038825e0c16e5c7b1329e1": AesKey(
        "d6154a59ad0f277fb0805c9e1bf54ba498831cd7b6787f9ced21ec03f0232574", "b9a817a0d89cb73a80f013731c08d3d0"
    ),
    # FGT_4400F-v7.0.16.M-build0667-FORTINET
    "1b2ac54ddf12a95c4d6b7f45b3b88540a4670c019510362da80aca440ab5022f": AesKey(
        "07dd9aeb046a32385a93998a237ec412e472bd8d4bd836f1ad2a3ee647b370ce", "3f3cec398e97e094ceb4ec6fa6328b33"
    ),
    # FGT_4401F-v7.0.16.M-build0667-FORTINET
    "40cce77fd74746dfa01081a024bda53cd419a566b44f449b1e17d34ced5b76ea": AesKey(
        "7e29a25e1861fd5929fcd3bf86d6f900dc41337d454a405a0769588232aad21e", "57bc18925f7d0c3054d9128a8eb87ea7"
    ),
    # FGT_4800F-v7.0.16.M-build7528-FORTINET
    "b88355bb3021e905e783705198ae1f701694631077b35ef6a57cdf52414fad2b": AesKey(
        "d5901bc4b035480778a33d2f6af31c8623f0beb8c1ff9df6bd93db8d773f4b86", "d7bef1cd87ccaa959c71e1abe3621c1c"
    ),
    # FGT_4801F-v7.0.16.M-build7528-FORTINET
    "7c823cccc795fe3b7388a548958fbac1ab5955157bfc840058c86545e1e4262a": AesKey(
        "b7c9ed763777add49dfe8980d1ea9f1875cfa7c5cd873d3ce0e2de31103ca663", "40447c6afa71846cfcb17cba5496ae65"
    ),
    # FGT_5001E-v7.0.16.M-build0667-FORTINET
    "be9a18663e2da9eb259ae860d11f862979ea687c3282dcbd8e0bc9b9bba76226": AesKey(
        "a38987edf7812437922e5a248ceb0095b3bf1d137fa40feef8e88f32388b932b", "697b221ceeea040eaf321a4f3ba4d80c"
    ),
    # FGT_5001E1-v7.0.16.M-build0667-FORTINET
    "afb8ea29842546e8896774c63becee0081242b1d1b9832580fabd83f7809598f": AesKey(
        "862a3f4f3d20e3a7221d72eae3cfa8a80c2c8dd58ba2019e043580112f647abc", "04aaa105bd2a0aedd182c0583f93af05"
    ),
    # FGT_500E-v7.0.16.M-build0667-FORTINET
    "d93fae27ae42e88a9df7549a179c7a289cfb011135bf3d96d175d405e026be5e": AesKey(
        "0b2595b3a2f51011acb488876a52836d197d855d0f22d0febea42342d661504d", "7b8d719f0939dfe863a2d7fbcf32cdf4"
    ),
    # FGT_501E-v7.0.16.M-build0667-FORTINET
    "8e726284c2c4067174f60e0e7695199f0e71f938a87ebb6042d0c8bb72d21ab2": AesKey(
        "b9cff37ad38d97f02713a209932ef2422751ec3b8ef6c07758034e910a19bba0", "7769b7145ea39b6d406cf2be9654af1d"
    ),
    # FGT_600E-v7.0.16.M-build0667-FORTINET
    "24d38f8054738193e9db4ba47ac0e06eba6ad2940e844568feefb2cfcc620475": AesKey(
        "77203d65ac1636060bf9ad458eeb4e4e1d6ecf63c8c1ef180da039f6d826df8d", "975992cc6aacc502c7782dd7effa102b"
    ),
    # FGT_600F-v7.0.16.M-build0667-FORTINET
    "76777e296a7952e3bc209112c334f02cd6e652ff640c90a3a4cf046ff5dfbb5d": AesKey(
        "0ff711ebd52a46f2ca61ec7164cbdf7c37ec988a5874744181d47013e54e3b2c", "b44e5c468db53cc436c009f18c496651"
    ),
    # FGT_601E-v7.0.16.M-build0667-FORTINET
    "95686de128cee2ec3d70b43c407813a6d4d57183eed94d38f0dd70dd01f26abf": AesKey(
        "84b7746be61b65a78913811f2f5df5b76e938941989e32b0a43b7c1b93e5615f", "d2f6cbb4724cc1838510d86489448a70"
    ),
    # FGT_601F-v7.0.16.M-build0667-FORTINET
    "68e00797189fb895145bcf1d059700bae0799299b22c9531882e676ed6814c32": AesKey(
        "29a481109ead96cffdb56bf2df8b33d03732a2b761dd0b702f094b3a3e3c9ea4", "57cc8ce2e03036bea6a4c926059f5a1f"
    ),
    # FGT_60E-v7.0.16.M-build0667-FORTINET
    "9c1168342809035a5b60077064fbcc3767d6f19362992c1b9fc2e52fcd035707": AesKey(
        "9a41c63067ed0acb0277c9da093e5b5a214affc54ea9eb537df9eee60f1496b9", "0fdf976f357a2188315c06c48ddeae04"
    ),
    # FGT_60E_DSL-v7.0.16.M-build0667-FORTINET
    "53c9f4c2b75830f0256819c9349fa4f2af8016decf3e08f6f2a50707d7fed0e7": AesKey(
        "1c24bb06bb9b47fa97d28e86a88b76ff782dfa877a704937c0e37af87eace3b1", "527a1510a34a72f89eb5946f509410b4"
    ),
    # FGT_60E_DSLJ-v7.0.16.M-build0667-FORTINET
    "986c9c7d9c8360c460db398f1bb3c8d90ad8ffcca6c58d9a1a35690edd5d18a7": AesKey(
        "e75bf092845cc7e686870decac66f33da2b41be309cab46f3e5d4dcc55f95f09", "a72577c1a60a41b842fdd8ab197c3f1f"
    ),
    # FGT_60E_POE-v7.0.16.M-build0667-FORTINET
    "0dbe69faaa2c73329b27dc3b8f6ec5346da004420b3c7ef2347f3ea7baeaedfa": AesKey(
        "2ce5e9fd0dcd985416a3640603f6535e33fca0b3b96d8df1fc4fb922cd39c724", "4016c40c5ba114ba0dbd56dca818f641"
    ),
    # FGT_60F-v7.0.16.M-build0667-FORTINET
    "a69840675eed549f3e9e4e575245eee3f61b4cc74ea091114eeaa0e294e6b93c": AesKey(
        "fb0ac719483fd43adb7059a6d6b4b622d77555ae3847917db9b430add6425419", "e1c774f7123272cc84bb7f27e9bd5f07"
    ),
    # FGT_61E-v7.0.16.M-build0667-FORTINET
    "949e3f6aef2ea6f7c8d131df5929366062635a4d614532e54a3a5a1ad7178277": AesKey(
        "f78afd1952eb5598f5e3f52bc35f035b45d157e9e241e06e906ea66e0040c292", "fc66faa13bb3f756a082393cdf40369d"
    ),
    # FGT_61F-v7.0.16.M-build0667-FORTINET
    "09770bb70eccada22dd4bf58260209d394e1038c970c023e00ccefc411a828aa": AesKey(
        "0ef37d319100a0dec19cc8d795a1f4d6b5ed67441dd1b31bac4b61ceb66d8dee", "a7ec5f0db55c2380103a7657d50ff029"
    ),
    # FGT_70F-v7.0.16.M-build0667-FORTINET
    "92f905cd2351179d2af4a4b2bce13e3fe7c0bb3c5d5d151f6420cc9d1b71aef4": AesKey(
        "6a4a30a8255f714ca7b685853b6578a92989bb3374f6a92495e6383cdc1fb8f0", "0dae644197dd14489a5298be57ef61d5"
    ),
    # FGT_71F-v7.0.16.M-build0667-FORTINET
    "92577136330fe3e38f2631786de4b2579e46d424cba1f2c7ffc62c82fab1102f": AesKey(
        "1bced655fd5659de8d6e93726a96b28c86a91652785b7d0338ad8f65d6cba58a", "71f462cff1f48e1348a54b0781bf9d6b"
    ),
    # FGT_800D-v7.0.16.M-build0667-FORTINET
    "386b4467e6eb2ce5ed065b575b57610297a87ce4f0c40b9073d2a26ad51ce73a": AesKey(
        "6501d08b0768935c4df794dbdf09c861a237f30c9fb21b46fea370aeeab6985b", "5650ac95a502e656d85c19e020335fb0"
    ),
    # FGT_80E-v7.0.16.M-build0667-FORTINET
    "5d3df035367f9494ba8e9fdae37c02035d60644fc19fd95ece4db41789ddab2f": AesKey(
        "597f164078c3996b58e633d40e9077eeb3f26bca5bab2a495efd0c505d321385", "04797ce9520b7ce7fc8ab2feea75732f"
    ),
    # FGT_80E_POE-v7.0.16.M-build0667-FORTINET
    "bc35d20edd19bb02b533694cf4c4c7fcf0ea64739deeebf4f56d65fab37df29c": AesKey(
        "d11acae28aeffaf7f7bf80ef8a1bf07c08e320e00dc2960748a3341582373d79", "b73479ee502fa158a5804bd51421fae9"
    ),
    # FGT_80F-v7.0.16.M-build0667-FORTINET
    "bd8eba963805bfaa835ae5390b4d4628885364c452441a472babfcf358235442": AesKey(
        "32048b5a67ce7b4d75d5b5f11baccb4cf04f786d877e5dde5d161e6bf3211545", "95d3c4a52d2f214b623e45ece951bf6c"
    ),
    # FGT_80F_BP-v7.0.16.M-build0667-FORTINET
    "f7063793faae2d54ef35b9bd84ecc7bc2c46c39ccdc3a2b48c0a2f26a8f3a9a6": AesKey(
        "719aad62585e77cdf689d0b7d2d4254663495d9d04dafaa9e31798a478e4600c", "9600c564d2a8e2c2b8ea4853b6f3b176"
    ),
    # FGT_80F_DSL-v7.0.16.M-build7518-FORTINET
    "b5094d764b7138a9a3499c8ab07b4ffa33159f27bf22569c8144510ae256ce8b": AesKey(
        "c5720083075098c0187f14e31ec846b69a5fa91c93a50e6b8c2ba2b40ab07417", "2e8fd5b70e6d06f48aae63aa9460e282"
    ),
    # FGT_80F_POE-v7.0.16.M-build0667-FORTINET
    "1a2ef943e16458c66cc80897f3c17332b4bb0c92a4770e22f81a76645cf55931": AesKey(
        "9006f947bcfdd8f802c2a76445cb62c6e72ec6e312103be4b19b1fb5d4b44675", "0bfcffee7274ecf21fb9a11e44fd0b26"
    ),
    # FGT_81E-v7.0.16.M-build0667-FORTINET
    "82a13da5d5e85b048ba54c871ce8c49c5c38cfbd0a3eaf173c11d4f947107c82": AesKey(
        "494286a76013bc70a177d8bfd04c114605edbd016c466ffc7692f03c63dcd699", "838d8dbc7947dd026727c5742e32f40a"
    ),
    # FGT_81E_POE-v7.0.16.M-build0667-FORTINET
    "74e5edd69390bc381b806b14ab3be12ae887fdec3351fcd2cafe937b1ccc9519": AesKey(
        "05b5292fbd8fecc7fb87e45bc71830afc2bcc7b7a855cd34a63345a1b5f90750", "3a15178229622fb54a2f9cb82651b632"
    ),
    # FGT_81F-v7.0.16.M-build0667-FORTINET
    "efa9b2580fedada8c3423665de6fcfe88d7eb8e26de25b5cb9a760b302da0923": AesKey(
        "3776e682b3e779b31bbc61ec786afeae00d9e3730061725491c3e461c72ff2c9", "ab1581842f9bdd676139ca6296a97f11"
    ),
    # FGT_81F_POE-v7.0.16.M-build0667-FORTINET
    "0918cd445b10bc42a67f99ad12ff3e3d5074c593989dcffaa2a68d1285c5309b": AesKey(
        "66a48889d0cf95858185653b3542a04bbda9216e97dc1c13e44efd134be74aa9", "a2a3a2fa8d291e664f0aa5771e4c966a"
    ),
    # FGT_900D-v7.0.16.M-build0667-FORTINET
    "081b91bdc7209e68d0c7394a98d59bfbf80525cfef6a3a85950eaba9d9d2ef29": AesKey(
        "e5a39417aa11ad7c29a146e1882c5efd56446b9af7ac5e20e326f03ee3b2048e", "1e234713663a0e6abcd9dad0c279cadf"
    ),
    # FGT_900G-v7.0.16.M-build7517-FORTINET
    "1e2fc3d378886f72925a7958eeace04f5a49f0a83dee51866be5b82f0eabbf17": AesKey(
        "1a970e809b148592221051a4ee5fc55879d26ffb5adb26f801ba54fa39d4cd81", "8ab2ecb084f384ea0974cd5c578e9c33"
    ),
    # FGT_901G-v7.0.16.M-build7517-FORTINET
    "577e7c5736f13db92e85a112cb53ff933d265423234e990ea717193f3b4cad33": AesKey(
        "a96c73954908cd558c4c6bc370efaaa5491f5a10c711ab34c9bedae19e407ffd", "98220813404d1aea4f0d7648056d6a47"
    ),
    # FGT_90E-v7.0.16.M-build0667-FORTINET
    "99be70e4fef6288a95823217b67dd4c01928e8f3664a90cf454ed7d9caf510e9": AesKey(
        "a81b979d1559645eb2320bdc7d1ab02b1c5b064964347aec6ccc9ac99c30213a", "94b9d0e876544fe52bb2429bd781371d"
    ),
    # FGT_90G-v7.0.16.M-build7535-FORTINET
    "65e0a624e887808c601452046ec0dcb5558b88633866db272236660b94660985": AesKey(
        "9d128a4828e0702b907ee5e8852e68356b074a5db25035c6f5f6f43f871ad097", "b486e6b4f4ec8a4209c7ce0c3cb44b2f"
    ),
    # FGT_91E-v7.0.16.M-build0667-FORTINET
    "34fe5d86094fd80305abef5903970880589b560ef79821a0308b337d4708fc7f": AesKey(
        "0273359159b7224ab0b08e242bbf7f0c40402fa028ebcbbc7dbb1343408bb289", "cd5d9c9400c8ffc0c4d043be42715ad3"
    ),
    # FGT_91G-v7.0.16.M-build7535-FORTINET
    "5a7067222705a486baf92feafb08e7d1cc88945e1aa6b17d84bb18c2adb7779c": AesKey(
        "5e5f3d84d6a719839147fd8169deec2e95987dbe2aa41232bda6dae7f4c1ce52", "1fcdf694ff131a1b596dda16b9d6f5e1"
    ),
    # FGT_ARM64_AWS-v7.0.16.M-build0667-FORTINET
    "d0ecee3746ecf88a6d4a5681a3fa0c88345ca083320cda33bd5b486c4013b9fa": AesKey(
        "6dae8fc5619d0a9ad9f41f0e7616680e5077f3bc19ca6ba2d6ea1f306683819a", "826f5fca8bef910655ef8bbfbab34d65"
    ),
    # FGT_ARM64_KVM-v7.0.16.M-build0667-FORTINET
    "352007ffd10416244ae5768a0abe476c8bb7729894eb61178a835baf2e434c3f": AesKey(
        "2433f4aef1e279ea8acda50644f7f410bd837be354b0feff5f528a39470dbbb5", "24fe3a46b2ae1e8ce82464e5746790d3"
    ),
    # FGT_ARM64_OCI-v7.0.16.M-build0667-FORTINET
    "1af90f6c696062b9a4969ea10340c7ac429c72405376b66adf8d96989a417ae6": AesKey(
        "1d9b09d08a41d06659b6b7af86d4c118357a5455061135fb1d92c8588fc63459", "f2f113e12e9cc5e85aebce3904f2eac5"
    ),
    # FGT_VM64-v7.0.16.M-build0667-FORTINET
    "f4da2b512ef5689bb1cadc877889b90830cc7f07b5c77673b827d63457bbab84": AesKey(
        "2f88de6a70ca4547ef0da7bfebaf7fbbde499b9627e0ece903fb5bf4ce4324b2", "83484c5461230fa5c3fead665f2259cb"
    ),
    # FGT_VM64_ALI-v7.0.16.M-build0667-FORTINET
    "27aa48e111e855f4fc73444a6df64396081e4a8d0a5c7b3d4e7b3218560958c4": AesKey(
        "515987efec260ba0e77c621f9da9a5a4f286c9cb3a632865e3b7f4f1a13fa490", "4b4af49fc274e7e568167d608b2d406b"
    ),
    # FGT_VM64_AWS-v7.0.16.M-build0667-FORTINET
    "98eadb682de4112469717a5bbbc0d90ae7c9cb5849cc19801881a76d797223ce": AesKey(
        "5e6d5ffc8a10d09ea493d11f20531d13ddf52b1112cdb999a2c9f5a08113b397", "c976447a3593856480555a0d532479f1"
    ),
    # FGT_VM64_AZURE-v7.0.16.M-build0667-FORTINET
    "c19a207694f7a231b06e645ca12dfcc5e3cbe962808f40bb5ccf758ceb5639d5": AesKey(
        "6c28250669e1f3cf24d3ed2304772d4ae7a1022bebc9ee54d2747e0dbab30fd7", "00e5af2612cd630742db27796d0e7918"
    ),
    # FGT_VM64_GCP-v7.0.16.M-build0667-FORTINET
    "ce1deeb593c618fc26460e9d7f8d81f1dcfb7f623a3c450cf3b018e1d375fadb": AesKey(
        "c1fbcdb7a7fb861348ebf495d00b7125b17fa56f0dbc2277c9e2a6b591509f27", "1bd3aea453408dcb96af3f236effcb09"
    ),
    # FGT_VM64_HV-v7.0.16.M-build0667-FORTINET
    "b455cc8579f656e5b7ac3d83f01c833df0a26b1e6812c450c10677a350e69525": AesKey(
        "dd36626e5e4b51583c02642e6254947a78cf36520129ef9408b699bf6f2d83bb", "e8a79f0a70e443b4b6635dfe72c97c31"
    ),
    # FGT_VM64_IBM-v7.0.16.M-build0667-FORTINET
    "21a721ed47a58dc4beeb70ec2f1cb734fa36026ed051a7a1e0b1b47cf84e01cf": AesKey(
        "48e07eca5db6aec1a09c7558dd55e208ed373b4e42908151723681e6d12485bc", "a1b5efe5033351d6035a1b74d29fdd0e"
    ),
    # FGT_VM64_KVM-v7.0.16.M-build0667-FORTINET
    "3a30d1cc6049035c1c89d7e7a1e372e7fa7a520545c4751b1a88770cf6c036ae": AesKey(
        "f7a23fd8e4a85ddc4394aeffe16179f752c7236a171fdea5a393ce70db5e315e", "c45cb8672840de4cc4293ebdcd6579c7"
    ),
    # FGT_VM64_OPC-v7.0.16.M-build0667-FORTINET
    "04281c48db2aacf42945e8b2b04cc97c34c9d99289a734648d4c98fec7484b5a": AesKey(
        "7748a4a57243c731181b4d6f66ef17a6a990fcea3c674488d68359821a70afe6", "c0618008992ee094b8c508539c626c64"
    ),
    # FGT_VM64_RAXONDEMAND-v7.0.16.M-build0667-FORTINET
    "7d788c4522b33f42e7598a5f9ecd7c78daf23c5eeee10aa93d6fbb34cc18d8f3": AesKey(
        "a94920b9cd95f083375ab3617f51ebe5ccd6faf4cb3a996b5fe07086872076b1", "8f4072f93c03eabb59b02f5e59a5b2b6"
    ),
    # FGT_VM64_SVM-v7.0.16.M-build0667-FORTINET
    "8f26ef78dfa6abd5145d1aa55d03b8b8b642e34890ee124c5de452c527d0c2ba": AesKey(
        "849872e6edd466a81528d4e94d6e2730c7d054a46d73c696141aaef1f6628a83", "26302f5a83678b011d43ae55dee4ac3b"
    ),
    # FGT_VM64_VMX-v7.0.16.M-build0667-FORTINET
    "b4151977288f19841c979e7c817c47c3af58ba522751464ca7f20935c834fde0": AesKey(
        "965f05067c6240d1342ee1fc14b226573a10033bbb7aaceb3ec57a75dc7c6718", "972fca4d988797f389f786f89148307b"
    ),
    # FGT_VM64_XEN-v7.0.16.M-build0667-FORTINET
    "dccee04b68461ade0e78692bfcd36965513efcbc06b32d591b3dfd8f7b4220d3": AesKey(
        "1a61730d4845fc564c2c2ccb46e19fe7a2589689f40b419577ed2a12e81e90bd", "01b3b4fb3003b2d3530af27f1e8ddede"
    ),
    # FOS_VM64-v7.0.16.M-build0667-FORTINET
    "39af60f1efae4f951abd38f184408b63e821e9517453028ee2101c70cd42b99d": AesKey(
        "902bd3885945d1436a270dc47cfe7a4b6b922ef3e15e5a895bb95dddfb99e12d", "50714b6bcaac6a58c408bf0e31e370c0"
    ),
    # FOS_VM64_HV-v7.0.16.M-build0667-FORTINET
    "6b793ea43309ed9567e1b36a787ba18f6b09a5092d32b73ba2d1dd0d2bea1f2f": AesKey(
        "e9a27263f287e30829f549f6296261499c5f7801c174f787cfb93a5320c719f2", "7e4b2055560c77a18f705ed6ccd3c7a4"
    ),
    # FOS_VM64_KVM-v7.0.16.M-build0667-FORTINET
    "5c0ab127c10a1d514d6566b7d188540302c9d363d9caf47709aef1f3715959f1": AesKey(
        "4f6bbc5519efd3545c877e82b80455eca70bc93524cd24426a08fb7ed5fabb43", "b8385005ada8d0f673f70178cd849a6f"
    ),
    # FOS_VM64_XEN-v7.0.16.M-build0667-FORTINET
    "07d4e5f3673b9e0500c306078443bbff6c30eb53dc401a29eba519896cd80157": AesKey(
        "5a41ae41d3a5f455d0064ffab1705c0e1dbb6b25421630a25b6ba349ce786cbc", "aaffacd719aec1632274e8485e629a1c"
    ),
    # FWF_40F-v7.0.16.M-build0667-FORTINET
    "f5f4a5b6420ac8061e45a197d1f98b760f398488689015b51d5bad8da8c92334": AesKey(
        "017d1b81d0709d6c38bcba42fff32ad181db387552bf91ec8aa67d6597c991cd", "a0957bc5d672d42c36ca8deb72437d7c"
    ),
    # FWF_40F_3G4G-v7.0.16.M-build0667-FORTINET
    "6d2dc4c72e4f1843fefbce63df8df61df8f9c9e698ed573bf691410af3ba60d4": AesKey(
        "49aab158ec44a611bc6d8497687ff0ec21766722764508864458db83fae97532", "6bb2bd2e8df0c0f4387e8b1284ee900e"
    ),
    # FWF_60E-v7.0.16.M-build0667-FORTINET
    "610d309a15b81e8f4bfe59886e976a482278c6e88cbe3a703cf524be31549294": AesKey(
        "6edb6af11bb9342e7cc57ae48a58a602aa1920f8457545e223ac96554fd106f5", "020b48a305a001932409504f7bd2bc43"
    ),
    # FWF_60E_DSL-v7.0.16.M-build0667-FORTINET
    "2f9fe0275bf477c6a80addb440fca006e09583b746c91165a77b9ceb760465cd": AesKey(
        "e0dec6b1fdfe50a192d59b8b65241e437a1fbc795266ce242a659bdcebb7c7c0", "8eca231eb34fa901cf5fea6a5c2775b6"
    ),
    # FWF_60E_DSLJ-v7.0.16.M-build0667-FORTINET
    "cfccfd3418738e7dc5ea2f870eba0874eef812b136c99134d0456fc12307b567": AesKey(
        "07888e800965329057f4e8bc25a897524fe6f6b16654ce238bebc0d2091ee18b", "1ec86b5376fc244e4d9eb4da7ff43964"
    ),
    # FWF_60F-v7.0.16.M-build0667-FORTINET
    "90f8e8bc0cafd7c1401001c226e67749fc971ab5d916913e39a18f14005b04c0": AesKey(
        "8cde1f1968071599fd6cfd0a8135f612d54f02fd9f17d731b4f400a289aa348f", "2c7c3f0cde7ed7ca90aae06621d8cc4e"
    ),
    # FWF_61E-v7.0.16.M-build0667-FORTINET
    "b4eff1551bd57f56d506d48a61dbf2fd1fceff485810be48b6a54e1305528e95": AesKey(
        "7dae553f4f0817a84891a8a419b5521c4f5f0edcf22bb857f6e77d84ede7d4c6", "d7f23e0f8dab6862e0ca87184292b7f6"
    ),
    # FWF_61F-v7.0.16.M-build0667-FORTINET
    "0e8bf25dc842333bcb3139006936502ac07cc6664c60e19bf20dc15daa5e4682": AesKey(
        "f051a16c824ab9f12227174be8b9283ca60a45966113566c4abd682a1f924e1f", "fd94bc8d764a61002855a53e20add462"
    ),
    # FWF_80F_2R-v7.0.16.M-build0667-FORTINET
    "4b83b4b9175a781fad1d1fe077b84a456832953141265848e67be895c438dab2": AesKey(
        "f859ec7553628acc7b92312c4d959e687efa7d948cba3dfb8181b699dc510997", "fb933d38731838d6e72d89351a1b5140"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.0.16.M-build7518-FORTINET
    "c966a8645d4a16b9e2cc2822064eb06d7ab7637aff87d826bcea32e48f64d977": AesKey(
        "f673e38c274616b13e5edeab036b1692fd2ae98915ee5fdc35659cf98a9ac78e", "0fbf29225bf07a4a73430f80ac2d12fe"
    ),
    # FWF_81F_2R-v7.0.16.M-build0667-FORTINET
    "ae30761f9666b6c033bc82b2fd1b64984481d2bc40a9653eb7827d562e02772a": AesKey(
        "d545dae495315f3957341e567e316956c337066b494a53dc3c020e2a5e4b9ff8", "0aac3101b536dad6622cc8f5709bdc8f"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.0.16.M-build7518-FORTINET
    "59f80f0bdd2419652d09dda236507caa6257fc0751ba479387df3de22b8b6072": AesKey(
        "12c8788a7695740000e3952dce0b8f91fd043d853698f104fafa79e211ff5a82", "c1b4885b6da1f7f1b60cef3b763bd88a"
    ),
    # FWF_81F_2R_3G4G_POE-v7.0.16.M-build0667-FORTINET
    "20a26e1ce0c2419b0fab5ce2b59fc9cce280df987cdb742f768d318a6a6edad3": AesKey(
        "2ab1b1c3426a5f623ae856c50c92a8d1df01f165f38ad55fdac00c835da21751", "238a80b16196ac618b803071af94bc1d"
    ),
    # FWF_81F_2R_POE-v7.0.16.M-build0667-FORTINET
    "905722e64a42f5e581d83197a649a5e13e2d28a6cf90aa72244dcdfabcde305a": AesKey(
        "aee30b9ee8e94d283d840167a7934c51e801d9e8d2de52a137b932bdbc216514", "0538b226a794495c40859d14a8fb6534"
    ),
    # FFW_3980E-v7.0.17.M-build0682-FORTINET
    "b122bfc00460eddb20b045eeb5669cb32e37009f01773957a58d9ba78f483b17": AesKey(
        "5d10fd9b760b0a631f06af5341c4683f9fe4ad5678f1d0451d5cf3f67720c8a1", "9b99e41bd28f71a8922b83bd3645c90c"
    ),
    # FFW_VM64-v7.0.17.M-build0682-FORTINET
    "907624be092efd1113ec157a188010c1316ea4db7f1fcad550c73fd5196f69ef": AesKey(
        "ebc8b847eaed622bafac3cdc298d9a18e87ba69c405336ae1cda63eea36d738d", "1b7d30241e8f4f7b5bb5b56299d3e4b4"
    ),
    # FFW_VM64_KVM-v7.0.17.M-build0682-FORTINET
    "4fef8acff40921b60e317e88b032edf80bca96863aeab017f2369d67c9129371": AesKey(
        "f4c0af1aabb34d6db069e827b9caee54b5711747369c4c6fe700cdeaba022e12", "c7144cf0def6c4a9ae2fa240b1a421a0"
    ),
    # FGR_60F-v7.0.17.M-build0682-FORTINET
    "ab4687f3d8c1279267caf1f366a940ab52d314bed1acf83c444f0d12236c9e23": AesKey(
        "630e12e2c7e4b863fb771abcf310504b7e5497f6fdfdfad36cac58221b363211", "90bb1b073f4138d8114d63fa25b562bb"
    ),
    # FGR_60F_3G4G-v7.0.17.M-build0682-FORTINET
    "118e3165ec98ef9388cb4604991714b70998e4eb165aa037b1c21265878c031e": AesKey(
        "d9fe43e6be6818c08fd955b33a6fb1d6e65e20bdc5e37e37a5f935575fe05104", "3f2ac563225135fb0ed925da44b6f3f1"
    ),
    # FGR_70F-v7.0.17.M-build7570-FORTINET
    "f97527c502f127f5db27d6b8890615b7e172ec359e0d96c3c7ebc54d32bea91f": AesKey(
        "e513d7f18e41c708316ef8f81f79f3c9051317628b14b23c7b75ba7e4df63458", "8a023df742eb5c272f2d47609d2d5ddb"
    ),
    # FGR_70F_3G4G-v7.0.17.M-build7570-FORTINET
    "8f9c7f3ebf2f02217e583372dc3d5a412ce987413c607c63a43cdef79bd1c97d": AesKey(
        "81bacf1bebf2e6d010d708a0b66dd120b056a198f839c9ca53ace434cc4d510f", "3936206da96bd84d9611095bc2e631ff"
    ),
    # FGT_1000D-v7.0.17.M-build0682-FORTINET
    "67158828ce8d8b2557b1ca26b8654dc16ce567bba639925ec2410daa74a4de7a": AesKey(
        "90edb71c2e2fc68962ffa806b628980291ff5bd0ee6090c797b75d52c8b15373", "495c7aeb330849bd0ef90cab26314827"
    ),
    # FGT_1000F-v7.0.17.M-build7574-FORTINET
    "8968b08a6bee1c521729210a8efd6937feccf6b40d85a4450e0089dc79dd0b47": AesKey(
        "adf8c8a4404fbb74b920c1005abab6da70aaae987e72536e701d50a3e6a82557", "2788c45fdad20751730bf1f548ab4f1e"
    ),
    # FGT_1001F-v7.0.17.M-build7574-FORTINET
    "cd535f53cf905dfb415fbd4b845c137a5b4577cacaeed85e66eb715fcbf71c01": AesKey(
        "048bda17332bfacbb04b600b194ef707db6b81bb7ed09f39cc38cdd1b9f2ff94", "970e3b171b1212f88fda46ce4d70a5d4"
    ),
    # FGT_100E-v7.0.17.M-build0682-FORTINET
    "404df52f9e034cd906d2d616c133380dc6ff494bb62318d2d4b2fb8993b7b28f": AesKey(
        "68bdb59aa55b9228836cecd4bc6193d373620a4de50930cd2009e46aad3b966c", "44fd6e486b70adfcc9b48bb37e09baf1"
    ),
    # FGT_100EF-v7.0.17.M-build0682-FORTINET
    "22d62b91bc3a8876a21a47f145f462457e0917831d594a6c35f5941fccd9757e": AesKey(
        "005fff1a13f8cac6871424778cd43c8d421fff51c54fb2659ae3dc2f503a28ed", "015b754b2452d3180d39a54e339a99f2"
    ),
    # FGT_100F-v7.0.17.M-build0682-FORTINET
    "e0ff555a6ebdfe8db0c942d15563dc464ebd3a67df736e5dba9ab4dec8a5a0ce": AesKey(
        "6d2ec1e57e0d65cc710081c1b17bec6cc96afdbd52b745c0c36d9cd4df6ea1c3", "bb29397ef075603527d22f384ea9ae8d"
    ),
    # FGT_101E-v7.0.17.M-build0682-FORTINET
    "a56ed3dd7ce08d451f46bfc6013c41ff01e87762ec55271b96b783309b46e5ac": AesKey(
        "edc5200e5e0d2c34c13323de2e4482f97b8cb8ca42b83d2329f930048b68a05c", "b66f7c87b7201e73c581c788c064dafd"
    ),
    # FGT_101F-v7.0.17.M-build0682-FORTINET
    "580a72d7c4c57495d640b3b206e49c5d86cfb788d3a6bede53ed7375a5d359a9": AesKey(
        "35fd802a04dac7a57639219e4590b85203498ed46dcc7fd29466910385deddb5", "b01cdfd778a067959c3917959dda2c1f"
    ),
    # FGT_1100E-v7.0.17.M-build0682-FORTINET
    "c41b01b367485c3e55af71292b97e5d195c6850e9bd8ce1f046092db9130562d": AesKey(
        "77d3583fae90aecff1557acff5737546c590b0e4bab75a239a1607171f5b9abb", "cfc484a4611fbf6fa9c678dde14e55ea"
    ),
    # FGT_1101E-v7.0.17.M-build0682-FORTINET
    "dd1027c6adf870e4d3310dcb3bdeee6b0b651eef2956b52cf0f6176ac43a8086": AesKey(
        "295c4534084142e422856b79115c7ed317561f54cb3923e77dca5f0a7e0b34c5", "1e8ab0fb3b62d57fd8e7432c73dafedc"
    ),
    # FGT_1200D-v7.0.17.M-build0682-FORTINET
    "7d0024feb0d70ed42bb607096dbd00cc05192eb831d0e1616448e6787121b5cd": AesKey(
        "05f581f7fd29fa6402e6ed4f051c5aa0b34179f329d474f01b18dfe501ea25da", "2914db6c8ffe5f8200cc9d45ca0f149f"
    ),
    # FGT_120G-v7.0.17.M-build7572-FORTINET
    "3f7fdc2e47879c42bc857416668b512da89f6a79955e9c5c6bd598fc021acbac": AesKey(
        "b11132262ef55f4931d80c7e34dc3da2c911b331b37392c2ed3a6d5865f52a0a", "0c16349d9b60c44490b2d065d1412bfa"
    ),
    # FGT_121G-v7.0.17.M-build7572-FORTINET
    "2fb0ddcef1f5d3d1995a2017fe3cbd772d471fefc4cd1aa5720dd2b432e7316f": AesKey(
        "3409052813c571d82fb92441ef33b9e21cd5fbce9713c717bd7cf87eb5db90e6", "5f7e900a1f895420f24fe57e7b836ba6"
    ),
    # FGT_140E-v7.0.17.M-build0682-FORTINET
    "08d5548c3cbea30c112dea91c1ed66f91b4131e409c5e8702d17f3c07f58e627": AesKey(
        "75fa5fd7cbdc3a5f4f33f5e1aeea61bbc10669be37d072eaae3ebec79783f0a0", "f785ef5a4a2dfb7889a6526f6c6130f8"
    ),
    # FGT_140E_POE-v7.0.17.M-build0682-FORTINET
    "fb3126e4877ff9bc326aa1cb0643c805510f05ca4684b73f7cf8a7e4493f881f": AesKey(
        "0fd3d6dc0ba66d3323c932930531485f79dc3484b989ce0c534c123c8d0fae91", "9be04bcd6b8dcc0c7b661de98e1f030c"
    ),
    # FGT_1500D-v7.0.17.M-build0682-FORTINET
    "8e5ea26fd60cc9b501333607eeae89013a4afe2d77d4ef192a2d148b9ae0d95a": AesKey(
        "923ab585249e26c0e850297976bf953e9727a64a8b88ca5122ee50be8d3b83f3", "7417aed77d4a55318533b146296a338a"
    ),
    # FGT_1500DT-v7.0.17.M-build0682-FORTINET
    "526bbe06fad3495ff73557f259abef66fcf04222de48df288ddd0b35207af4b5": AesKey(
        "39b81e0c251b30d1e0b1cd4c6bce0f0fb0cdd78f3b89176c78a46d050fd98c05", "d86e2d6f1e84e4513784651e09e71fbd"
    ),
    # FGT_1800F-v7.0.17.M-build0682-FORTINET
    "aa82584701c7bb3f992cf9f5a674c8732848f4ffac662779b30ccabc180c09e6": AesKey(
        "a1c541b6fa879af3ad7f8ae80003648425ffa3c0b7c8fc89d8b9369aeffb3812", "6f45e858be3f5ee9caf201fff1b74dfa"
    ),
    # FGT_1801F-v7.0.17.M-build0682-FORTINET
    "9f6aa71889f6224bf501f313c2385868fe9e44865b6343f9067cb54e9dc41b57": AesKey(
        "be6386069a5a525563a975d4b879da727d81f89b3c3e63b160867ebe8e996041", "f29ba1de30138ca7d781e69a0e2b9eb5"
    ),
    # FGT_2000E-v7.0.17.M-build0682-FORTINET
    "e13cb0ecc3f00cf87aa3250b5d2fec118465695e13f05bbe6b64022b6f81fef6": AesKey(
        "aae7482471cbcb52255f886b117e19957993107257ea41311e13b27cd59343e7", "2afcb0097116c59d093da0690ea617db"
    ),
    # FGT_200E-v7.0.17.M-build0682-FORTINET
    "e046dc1141f3fd7b6784ec0103238ad40336e05b0f7e89dfeda19d0365997785": AesKey(
        "f4ba2eea2ef9906f91dc0dd441a7fae7cfde5668d50fc4e68beb05372e864f9b", "063055b471430fb20043087f7ad789fd"
    ),
    # FGT_200F-v7.0.17.M-build0682-FORTINET
    "ff324056520c0cc0cb6fefd4663a54cb2f666f53a7ca998ba865ca9eb0ca0a48": AesKey(
        "1185efb1965445d4f45a5be41dfa7292a4ba9f7dbd8740614443ea77e2b4f0d6", "1fdf9c69d0bd440997f3827c6e170e61"
    ),
    # FGT_201E-v7.0.17.M-build0682-FORTINET
    "4fec69886ad8fb6d73b48d1f8600b5fecfc60d2269a7551cc2c434d8928cffb4": AesKey(
        "9eb9b3d8f671e567344bae4d78c7ea4ef0f5fe7df6a0b4317165374d30341aec", "09138916e7a7d629a2463ebc4a3cb218"
    ),
    # FGT_201F-v7.0.17.M-build0682-FORTINET
    "f2626f747dd8204ff8a0eb21575013519939ed27362f6b0cf33c6db84e7ae72f": AesKey(
        "bae5dcc4b929dc60b3b44307750044b420e4917e02030f4219db7c746a74baf7", "deb0d2f998abfd35e78e6c132fa08468"
    ),
    # FGT_2200E-v7.0.17.M-build0682-FORTINET
    "44337610b36d304bd3339b16f08a65a254df39f57f884775bcc7591d2ab9e613": AesKey(
        "8eb1ce866af23fa4fdccfdd1e23f18a0430064a3b1d653534fd502907d39217d", "5fb643c929ded3efd3aeb3e6cbda36fc"
    ),
    # FGT_2201E-v7.0.17.M-build0682-FORTINET
    "0b51679878a2b40679995499dcdfd96495604a6acfe8e9d5a5907c0ef9970b47": AesKey(
        "b175678a38b11d1a31743133b836a0f2ad47bbb7b9ffde31ff25200abcc2c31a", "d9b6ec06bf5692309a1096b94dbdd1ab"
    ),
    # FGT_2500E-v7.0.17.M-build0682-FORTINET
    "075c7ad7c31f256f19a67afc10393e4e78e08b42dab853f357c2cb0e8e75af54": AesKey(
        "63ebe8ad8c67016906fef8cdabb60f12bd520ed371f9faafbf3c9cdf22bc7a12", "08064ca23147513e83410509ab3facf3"
    ),
    # FGT_2600F-v7.0.17.M-build0682-FORTINET
    "106f4b1b571d0170bcbf73d6020b498be3ef6bcbeaf0b5fa1e2efb013b8350c5": AesKey(
        "e782ad9a8966ea225ce410b8872bd740e4ba944b69ac2927b3fd2e4550bf6ee0", "debc9fb9199ba941963b150e15a98c5c"
    ),
    # FGT_2601F-v7.0.17.M-build0682-FORTINET
    "c2ae757896b4a4ab88ab66f82bf3470a9912faab5093637a5973d3bcc666846b": AesKey(
        "79ffd25a0308c983b0d64a4bdc292db86c6e732097cf329a09b4b5fabc3bf06f", "651888ab04d90741a0e9734e99e67838"
    ),
    # FGT_3000D-v7.0.17.M-build0682-FORTINET
    "72e055aed34d9c0b57a94e1aa769e7eae0c2f0f9a635851f33029bdac9ae32d0": AesKey(
        "9401eeb66fc143d5500aa551f5742dd6552325ee463c98d6e6d79c9710cfef3f", "ad5d4cc8f9d5a45cff5c152b87094f3e"
    ),
    # FGT_3000F-v7.0.17.M-build0682-FORTINET
    "89d716512b66b4f84c53fadb757b5a9f6baaa8b40eb09b8be81c2fef54078142": AesKey(
        "472dd58ae2949f476750b2d9a0e9a5f02ae75aaefe7d73b19a4adea295e9dfc8", "07e167e8afd24a941af13ee21a572c48"
    ),
    # FGT_3001F-v7.0.17.M-build0682-FORTINET
    "f97731b3779afa1bdde21dae4345a65be066088dbeec1e670ad928622e35b86d": AesKey(
        "03a083db49b0d07a0ff0939fa0215f5c8d5f56312774571cd2ca743560c066c8", "88c0d176c3e3de2280d307b6ec842af8"
    ),
    # FGT_300E-v7.0.17.M-build0682-FORTINET
    "4d3c397c1c237241c02b48835b7170451bed8d47b814861f0a92db9d4438f788": AesKey(
        "197aaf94f7c82e2fe5f3f2e62d9efa4fa92a8552be9e41ebf93d1970e9e5af45", "e90469a41c0769201c7dbfec4c86234f"
    ),
    # FGT_301E-v7.0.17.M-build0682-FORTINET
    "e6a0785a2a7326e410e601a54340de9880c3946c1a0895058e4966a948383984": AesKey(
        "51a299f3adc06f08f952e2f66a7b8b46ed19022e10573c658c30e23417f22940", "55d6a297062d603467cdb68ed88cf9ef"
    ),
    # FGT_3100D-v7.0.17.M-build0682-FORTINET
    "639421826c8ee64028b443c503cca282718a8f7bfdfdc59d5c767fdaec693c5f": AesKey(
        "c408a7c561a7386702aab7182ae005ec926aca1934cba51069eb3cca262a7c64", "6f49863ee0ec18616ef79ff9ba88894f"
    ),
    # FGT_3200D-v7.0.17.M-build0682-FORTINET
    "c04e67e95cedcfb195108502e17f43d964559dd53fc655517d81fa66331ad5f3": AesKey(
        "098ecdd957c18eed9f114c3db5b9377fe0e94a7f53f8a061abf2dcab8c0c4172", "347123c477e69f4fc9483625e291e884"
    ),
    # FGT_3200F-v7.0.17.M-build7575-FORTINET
    "6e18d779ad7aa64e31ef89c0e32893718cbb101eb5373a1d77832eb6b3bd5b41": AesKey(
        "e7e37891370c24725bcfeba4fdc1b835492af8f9f62cfc8893588c4a9030e70c", "0cbeb9ac9de47086db312dccd24a4d7e"
    ),
    # FGT_3201F-v7.0.17.M-build7575-FORTINET
    "30fd88db84955b31b016f1980912800f105d2ad0d9ec63220179cb2cb54b3b05": AesKey(
        "cb05babf8c564b52e75fbb597aa5ed2c17a025aa7663b3a8df242b5f8be1d024", "199647a4d64ce4d15cd3d250dc4a1601"
    ),
    # FGT_3300E-v7.0.17.M-build0682-FORTINET
    "09f6fe9a1272aa120172cf5821c67604e7610997dce14a7f12d3aff96316b7d4": AesKey(
        "fa34cbb673ec75b44cd7d56a57f26d8690edc12261e8f00fe5266c6389e38a68", "37f96a43370d930bad5025daa27a5078"
    ),
    # FGT_3301E-v7.0.17.M-build0682-FORTINET
    "28c753d5295380d2ba29223abd0b9325ff72442df61fbcfba09cfe7e33cfe8e4": AesKey(
        "b139d436b38f8a0dbeea72d32e4f82f7c31eb52ea5887a0bfacb10892a017558", "65036c9b91dab416b2cdd74cbd921168"
    ),
    # FGT_3400E-v7.0.17.M-build0682-FORTINET
    "c26747799abd49d2ce837393ff3b0bf49689aeefa2955df2b93c76980ca5037d": AesKey(
        "d3b0766f0478436f3bf182fa4047071ce4ef0a3f5b69466d741d1f44d18f82ee", "f1d203d7e8028fcb27da3ec019042180"
    ),
    # FGT_3401E-v7.0.17.M-build0682-FORTINET
    "b3ec8bdf22c90c195063cdd57b225afc660a742208fedc23cd9cc2f4ade10165": AesKey(
        "2692560bf66af30e2601116113fbc3695ee65d5ef06a0bbf0d09060990d5b341", "94288fcfbf64f56d4c886dc660d24548"
    ),
    # FGT_3500F-v7.0.17.M-build0682-FORTINET
    "d4740c8435295b3e2918385afa50f212db5271fdb513efbf6aa28fbf4594c872": AesKey(
        "ddbe1be77af99b19091d8776d9ff960ac502bc33e9607a89d42be0f23d3228c7", "39506824a8a6a233d59aae6170740b77"
    ),
    # FGT_3501F-v7.0.17.M-build0682-FORTINET
    "153536ff1ff45b7059f879f0997f3e32aad18e57d0316f139a39499c182eb403": AesKey(
        "e74d375f034d3cfb190859e25dccc3e4346ec7c579032f3746d7f1754b8f5af8", "9fd1dd2eda7e2fa197b493d3b33b61d8"
    ),
    # FGT_3600E-v7.0.17.M-build0682-FORTINET
    "a2d12dcef3d630d7e7b11d3982f2e4d0983685a8d2c450e907ad6c60a771e5e8": AesKey(
        "10d0c751e9042a6726ee3b45a5cfe2455d85e392a448d9f33f630d5117205687", "95cbabdbbb27c0b47f644b18f6a9bf80"
    ),
    # FGT_3601E-v7.0.17.M-build0682-FORTINET
    "d2a8d5944ce87161cfdb84e736a278556816aeaa3f6e2095edf9fc8ddc83aff2": AesKey(
        "4964bda233b45898e85022470dc73b579ed46d3e63d98306cc52c69b7ad14a07", "d6a96cc4e37a88f46b658dbf942fa10b"
    ),
    # FGT_3700D-v7.0.17.M-build0682-FORTINET
    "06b5e85faa3986307fdfa3efb87774991fba2e274c6a077b256bf1d044354a4a": AesKey(
        "adfcc2e72960d90eaabc5b468d9b0e53553da67da8693501243eee8f1e87f1f5", "3c39e65565fc9667e49a72ffc1bdd709"
    ),
    # FGT_3700F-v7.0.17.M-build7575-FORTINET
    "70731eb769405cf0011c06ded7f4ad6cb2fcd65896488ae5af72f22b41ed32ae": AesKey(
        "d6be4630c9759f7d5cb68926bbe3cd35aab1657009cef13dccf000196670b103", "b9fbf0d0f3265f744893808710d056ef"
    ),
    # FGT_3701F-v7.0.17.M-build7575-FORTINET
    "b0a9c59be83f3e738d3cd02471fd2bef9f18fee3fca9dd15d1b272cf5b25f80e": AesKey(
        "1beff8f080546a7a25d31529374c302b7993c487763d313c7e5dafa10ee2370c", "1db4cbc6f8d90c9bd0790fecd7f9f977"
    ),
    # FGT_3800D-v7.0.17.M-build0682-FORTINET
    "a065d1f4b30d2c4781926b5f017516c70132481b258ab21c4bd788b40a50a7ce": AesKey(
        "7c2835f744b0c017197cfe616e6291bbb9874c92267583f92cf62ae5dcae3f86", "706a8e84da66f509993fab9ed7b88f71"
    ),
    # FGT_3960E-v7.0.17.M-build0682-FORTINET
    "3efc549fd7469f410dc631e80da52b5c4da1cb7eca05ed1e74beb2cffb9de092": AesKey(
        "d8b32a6f7857a18470c3ef7eac69e02554e2b643b619e6fe1ce78d58579f9437", "2cc19b373f774a51dd535d05848ab91b"
    ),
    # FGT_3980E-v7.0.17.M-build0682-FORTINET
    "607b0808e3c13a4a8072af3e356d6c2bcecb12af0a1998d5192e092402f0ae88": AesKey(
        "32736ea1a0c4129c3adc9f7d330a3ca1c8827d53269b4b503f1da7ab2fc8b542", "198df9e555fdb80a2be852fee7a40fe3"
    ),
    # FGT_400E-v7.0.17.M-build0682-FORTINET
    "feb84c517c00ea6cc38dc5e99e130ddbb413314d3577757dd0f5a29d00bfbb1d": AesKey(
        "b9d0f0131b2bed2a1afe4cac5c29dfd23348ad5ac779bc0c9d6bf8bcaca78815", "d6c785d8b8242904fc930b1fd7570acb"
    ),
    # FGT_400E_BP-v7.0.17.M-build0682-FORTINET
    "1e58bd7a29a02820e826b507fbea341476312ce296e84414390ca8df4015ad1b": AesKey(
        "632efc1469efc8f5a0523438bc078c7e965837536729cf33e20ad0006da7cb04", "2d5879222d7b8f5a15288d99a2c29980"
    ),
    # FGT_400F-v7.0.17.M-build0682-FORTINET
    "658122c5ad174a699a0040f5a95dbcfdd1ba077a2b4f81ff4c98ef5fe2153661": AesKey(
        "5b7a3259c2e1f9010bbd1efc0353b97fdec28b1379c7e99e0d4e1ecce55b1402", "75b20e12f6fbf8d8d2d9cc4d705d022c"
    ),
    # FGT_401E-v7.0.17.M-build0682-FORTINET
    "978db672be660f425730b04692ca869aecad86547220173ac3c6bac119acfbcf": AesKey(
        "d20964b4b9d3952dffaae052586da392d003b14d8fa513d222a94ec2420c5805", "641859300bc37a87e77dbcc782653420"
    ),
    # FGT_401F-v7.0.17.M-build0682-FORTINET
    "0aa55e3606a6ff083db3c6128d6478f6fca3b99ad79166301f5691ec1f85c912": AesKey(
        "76da0e86ca17653bfbd92fa0ac464b7fafb034734c8dbfed4bf938606fe9fe6e", "5a70d51ffab51ec1e56bff63ad00c06e"
    ),
    # FGT_40F-v7.0.17.M-build0682-FORTINET
    "e7362b76ea3eec6b5b12a1562181a44470f1f3fad49aa64953b726fec2511ed6": AesKey(
        "c3377954c730015f4d1a66491d72a71f1a18f4d2617f7c01a9eff29a49635011", "766f32d29afcfce35a2d5b3165a52986"
    ),
    # FGT_40F_3G4G-v7.0.17.M-build0682-FORTINET
    "47d5f21d161575c905c64f3d52059a071072ab684da958a9d4e4ff18272fa450": AesKey(
        "7ecdc0e65c36a007353cbce24228e56a52afa90df12337630ee67c278de11293", "bafbe1766cd40a9b638cb3254077d39e"
    ),
    # FGT_4200F-v7.0.17.M-build0682-FORTINET
    "778193457f9188f6af3e94f4c5d909b059b9679bf908994eb4da51987aff320a": AesKey(
        "707fac1ba552fd485d7eb502e69572f4af4192cc994da23da52bd6d19cfcf1c4", "a8b01863a474a296e115e517cb77c7f4"
    ),
    # FGT_4201F-v7.0.17.M-build0682-FORTINET
    "036fc1135c2aae3e83036ce519485045a2bc8edac07198ae36a11eae33c2d7b8": AesKey(
        "f6bb6b13be43e4138ceb7094810faab374fdc7fd4898e17d648d63b21fd3e927", "89f53f01bf9bdad2bd429772123219e6"
    ),
    # FGT_4400F-v7.0.17.M-build0682-FORTINET
    "a0c20ec3b1a744614a18bd3336603a1a50551d1e23ba554e5875a3cf5a95657e": AesKey(
        "c7882d13389e4fa673e01eb8768d614f667c166491161efaf089eb8da4f8bd2d", "fe1c15cf60be1fdce06131e059743520"
    ),
    # FGT_4401F-v7.0.17.M-build0682-FORTINET
    "f8813a6dbfc171bad60ee02746eebd45cc9e1aff0bd21894b2d3afc4a6d929b5": AesKey(
        "92f9a41d0f225a8af76a5f36f10670b4e08ea85703b293f2c815aa162f484b8b", "2be75efd6ad24907012dbd6f7e5e1ac8"
    ),
    # FGT_4800F-v7.0.17.M-build7575-FORTINET
    "dc43275c6ca781283507fc9a64fbdb115da1bd070c94fb3b220ecbbe1e2c63b3": AesKey(
        "33f402105c0ab065b3fe9bdd0340ecfad60b1f0497db69d619a8ae6224585b99", "d99b5155e1e4ba1e229223dd31355b3b"
    ),
    # FGT_4801F-v7.0.17.M-build7575-FORTINET
    "8682cdc3a3be44cd1b5f6bb94f22de75cebb7026c1677e906ce031206a063c2c": AesKey(
        "d20ecc6ace1571203519a7337e5b3fc74829ec6ed1cb3e28318dd3d751228042", "e88db1d2505758423ea4f718951c241a"
    ),
    # FGT_5001E-v7.0.17.M-build0682-FORTINET
    "e19f3021cff0be284c43d4dcf72a4772e667cc44ddda6cb6f62f7fe9b2fb3fab": AesKey(
        "b181a12444a627fe7e1db2930e94c1eb7ab4032b46661070f688b6035e65c4ac", "2aba0ce821d2127d2e9dbd8cceb5472e"
    ),
    # FGT_5001E1-v7.0.17.M-build0682-FORTINET
    "6022772910cc81d22ef31f2a4966f7f94e941b64ea5f2cd1e6c6eefbbb91d7ae": AesKey(
        "668769667fcbc13f3cba5ef3beaf08baac79d4e57e78aaff39aa0cd9d2c8eb10", "99d53cb7e970850d228267554c77176e"
    ),
    # FGT_500E-v7.0.17.M-build0682-FORTINET
    "4c4f8ff80c5e4ece7878fe7a1044f8defa134844f53dd240ceb9f7b59e5c92c5": AesKey(
        "479ee48d7d54a2f68a8d3ce7d8f4298dab5ac94bac75fd807e839cafb507f50f", "bd16c4d09dc543c4a4270e05210ac19c"
    ),
    # FGT_501E-v7.0.17.M-build0682-FORTINET
    "5a1702e264eb2225f73f2bbef33c8788ef706fe979ded8760c69a3d424c9ebfa": AesKey(
        "8bdcf2cd375a2c83a377d8ec53d2bcdace52d60ee9e9cc9906a305aaaab8419c", "8c6a6d0b7356a7920a55e0f050fcad1c"
    ),
    # FGT_600E-v7.0.17.M-build0682-FORTINET
    "7ee6ece08de4f4aaaa91f9ec09a8e3d010f2966cc2b9572a2658152d939406bf": AesKey(
        "beb389bba55bf176d8ebf4e980318d84102d2c422173145fb27f11b0d1b6ed89", "0e228e29f24ebd9e45e2dd80c77dcc56"
    ),
    # FGT_600F-v7.0.17.M-build0682-FORTINET
    "e283ea027adb9f60680c3b717267546848ec39a1c0e79949c62d77e468db90d8": AesKey(
        "11a41990ce03e88baedbb2226d4f246581ff084e6cacb76656b5f0e6cd2f0b7a", "570d5cf5845d84607d1012420f98905f"
    ),
    # FGT_601E-v7.0.17.M-build0682-FORTINET
    "f08570d0f87343b4dbb2d349724528e2c6a18b18886b1de5f191498a76a2a617": AesKey(
        "aab5d51664f26b4906c1c4e2be82ee842df2f30c5cbe1e1f4b4dd5e773182e17", "67f0b75e333eec55389149af30384d7e"
    ),
    # FGT_601F-v7.0.17.M-build0682-FORTINET
    "4f86035787de7d2feb5c5e210875b781369f3bbb4cc0500ed99cf09f19d0a3a6": AesKey(
        "887ae5e04c16a889cf89672696db022149b470df3ab977f5fe8e9a19dde212f4", "9a93039c7b5c3971b177ef84bc26f3b0"
    ),
    # FGT_60E-v7.0.17.M-build0682-FORTINET
    "db791d28613bbeadc28c80e72313842a5da3143235ea6cb274956839b924aeed": AesKey(
        "d9474eeccc4bdd81a211bdc37e9e27b1f4a5a604d6abb3e487fda91675e10c4d", "4c42142acb9574f5dc017c4eaf0baec5"
    ),
    # FGT_60E_DSL-v7.0.17.M-build0682-FORTINET
    "24ad51312aa93ec0a0fddaa1150d5b95437407c52d97686c5d255b1deb41efad": AesKey(
        "2292721fe5966e123e3c1ff3e38dd634c1bb5ebd7ad4fa2ebc99cb9daf4ab611", "73d97174d310dd5adfb43bbee24535fa"
    ),
    # FGT_60E_DSLJ-v7.0.17.M-build0682-FORTINET
    "6fd54eae61a6065afe36af6565de06e7e16a699bb98af1bf119375df8109c95d": AesKey(
        "3dfeac28084896f95ca20d349348198ea83e151e103b9be8dd5ef12272901b87", "00dbb3bfedea89734fa5519fe01d77c4"
    ),
    # FGT_60E_POE-v7.0.17.M-build0682-FORTINET
    "fbabdd90da278876f4bd8f47e8e065b80f5a02edabaef77979bbd8160b64c974": AesKey(
        "c52fcd2f531a567c30f9b04c474bbc1f4539d54e26a91b5a91bdb37d94335093", "95308ed69354c929a8e34f7f8f1ffdac"
    ),
    # FGT_60F-v7.0.17.M-build0682-FORTINET
    "fb89bacb3777bbbff9befa27303f21ef8e50da949f4226d5fab06a68e991656f": AesKey(
        "dd740a4282981006129ecf69fe7a9ea4a1246ecf692fba1581ef81a56b430d49", "008dc722034e010f1ac7d724c00d21eb"
    ),
    # FGT_61E-v7.0.17.M-build0682-FORTINET
    "a20bdc39026dd20115cd24fd742fb250bd386657443742041716bf8a95d7a736": AesKey(
        "7503c77881101d553faa5d1f34bb6a6063ba14f2e59b2df46a0f83aa6e4b0854", "76a73470309dae7bf557243c8f04af20"
    ),
    # FGT_61F-v7.0.17.M-build0682-FORTINET
    "10381c87281bee97d3c43aac64a0250413c58f80f2e4522236b94039c6b1eb65": AesKey(
        "0d0ff365a259c641972833960da64ab99243d0140d4bd018fa33ee33cef6169f", "35afc2ee395fb2d689e287bbcccf7551"
    ),
    # FGT_70F-v7.0.17.M-build0682-FORTINET
    "c433041f2dad91042ca291f8396c1e87b45feb0c555e717b660f9cb6bc12f41b": AesKey(
        "90ce7705f201e3e778256417948db98caf52bbb20818d17aa2a1c97ad9ad53ba", "047c657d315cf590c68f1cd0ffe21b81"
    ),
    # FGT_71F-v7.0.17.M-build0682-FORTINET
    "8264e343c02e960cb336c4b69f01322650c929364e26876d749186b148442f11": AesKey(
        "3cbfec38f4ee8637d6ba35a8cfdb25537d008231f37f17c76a3b16c34f4cac68", "dfe0e6ab7937c31ae84d5505d67f1728"
    ),
    # FGT_800D-v7.0.17.M-build0682-FORTINET
    "e50ecd9764857d63c080acb5007a9e80c0d1f8d1214844e012225f46e8c7d64b": AesKey(
        "2dc3712afa73a23609422eae1585bb9d13d4d07d7be38ab7eb159e32bf7c6736", "3f643f8fd53ee38360661a971136322e"
    ),
    # FGT_80E-v7.0.17.M-build0682-FORTINET
    "5e65a72b9c22a0b01538ab939406c3943aaca5e5e030570a642c2bd81a285f86": AesKey(
        "159fb4f0378b1fb161b82fc0811f3b45706f499fb4a05880644ef68cb05d6b90", "2cd5bbf77ddc973d4af263590e2bcd41"
    ),
    # FGT_80E_POE-v7.0.17.M-build0682-FORTINET
    "e946a0b3b9a21b1e01dfddc12ced230897592332e5e467d24a0134ec9d438289": AesKey(
        "5c139db1efa13d3a72a77640a5e3fbaf0cd634a7c879d1e2a293440e335045b3", "f32c2144bdda8d5e751358f8ce92af20"
    ),
    # FGT_80F-v7.0.17.M-build0682-FORTINET
    "6cc6dde5f60bfe834b7f3b0817f8f116498a65648a7758564ab3519c828e5b77": AesKey(
        "86af13197a4f5de76e4bd601f467f6549fac9a57e171122017944dcdeeb11761", "eb3cb0951f1b64207e47d931209bdff7"
    ),
    # FGT_80F_BP-v7.0.17.M-build0682-FORTINET
    "da65ccaf3bc83a99b6b1c00a94e49c489d0522cccc537917d731ca33b62d029a": AesKey(
        "6dd43c4c1564e9ad3ed23e0797c7a15cd1c3076efaeb7cd2eba3c513168089ea", "7c217e6dfbeb9ea834de3bb7057bcef0"
    ),
    # FGT_80F_DSL-v7.0.17.M-build7569-FORTINET
    "ca709c66c135aaa649e668375520953f0bba849441dd199fcc22cc1edf687865": AesKey(
        "dcd4307601f253c54cfe53997dfd93836ff229df4361c48015a4cf9887c994c2", "a4a3d64f313c718e4ec96ab044c6631f"
    ),
    # FGT_80F_POE-v7.0.17.M-build0682-FORTINET
    "c259049016ef9c61b4fd6bb962358ef1b0b97e9b0916369411361d07d504ddc6": AesKey(
        "81ea87ae2bb20a3c638c12518aec78db3018afd13527ef23c20c6b8cde5b25dd", "2fa6cb3bdad059bc0e4ddc94e868aec9"
    ),
    # FGT_81E-v7.0.17.M-build0682-FORTINET
    "db02925806d7dcdb4451bcf50c39b522113777db81d4b3c46c6420213cc5b9aa": AesKey(
        "77fde55a10bf64e730fc3ee663750f6c57002c4b88f3d9d5a2d2bf34dfce1239", "8a1b6ffc25be4e8a81ece8e6ce4af7af"
    ),
    # FGT_81E_POE-v7.0.17.M-build0682-FORTINET
    "1ce60cfa3f107f8e3cb4c90242083e2a694873c665bf63d854e3e1c6080d3eec": AesKey(
        "43052ac1821ec2b581a4e362c096f42e3c9fd08202abe09c90b5b7fad12ec92f", "38ca98002fcd140cb2bfe324b22a570e"
    ),
    # FGT_81F-v7.0.17.M-build0682-FORTINET
    "67e845a85a06b28dee265a51c03733755c8aa0333fb55160ced026bc591b129f": AesKey(
        "c3b6f852bb6e7a1a1b5d2e9f7b05269421cd1457549c3d1239211137c2bd797f", "c7822f5a9f654f128548a03a0c5c75b7"
    ),
    # FGT_81F_POE-v7.0.17.M-build0682-FORTINET
    "edf2a868b08a4eadc873e9c6294945b21ad69ae54bc7801b1c1819e51cf64a2e": AesKey(
        "51a68cfbd607fd005e6d2671cf13f0d80c108d962210b8d27d67e05bee9dcc09", "b11118e478c08a177cba3960f9aabcf4"
    ),
    # FGT_900D-v7.0.17.M-build0682-FORTINET
    "448b905733792ec0e470d4dc3a18266ec49c20604a17a5cacf59367d873de56f": AesKey(
        "8cf7c4120e4c06a1ca16bcbf223c60f18d85b5859c6d3f163a13b7a3720d44ab", "2f05e807e06a82aec8838ee4ff39475b"
    ),
    # FGT_900G-v7.0.17.M-build7573-FORTINET
    "8cfb2855c9f9647286c850167c29099fd6bab1fca11782d52cabb0305227e6c5": AesKey(
        "8bf8f9831a035887cf2986d94d75d5e1336d3855669247409df26795072c45be", "ae1dc5ed3d771a5b631e7b8e3ebaa52c"
    ),
    # FGT_901G-v7.0.17.M-build7573-FORTINET
    "176dd93ec2efa1294fde5e3226d0ecf590b664047fc0d664d4fa2df63f746d67": AesKey(
        "f1aab450c891ef15794c5600ccf82ffae8a4a3ee5316d48b013a8b2b588fee35", "df433a22b14f2a65435d839248499f40"
    ),
    # FGT_90E-v7.0.17.M-build0682-FORTINET
    "5f9c1ce51a03f37cd5b8c854aabd49a6adaa5a2ec65ed44a3ec1c4e6ad73b6d0": AesKey(
        "b839ff9f1037576076a9a758d4b1e7806ba010f2bdac1d12f57b68abdcd108ca", "d4410a780cdfdc01bfeaef12e6da23e5"
    ),
    # FGT_90G-v7.0.17.M-build7571-FORTINET
    "93a237f6ed0da31c34887f3a453df56ba242848d9586fa98dd5bafd1283643ad": AesKey(
        "b2928efc36c0d955884e0f8992923e1026626a6b33645ccb3de44f550cb0a438", "621c2fc98c415b00bdc6f3b7029f9f02"
    ),
    # FGT_91E-v7.0.17.M-build0682-FORTINET
    "cb7b57d0c6447d06fe9aeffb9175d36102eab8951bd4d474b830f272cf372199": AesKey(
        "96ec1ca2acd141758c6b6953817e7b719426a5eb1fe39487b123baa60694c509", "d635245ad62bb891ea69c2576caa6159"
    ),
    # FGT_91G-v7.0.17.M-build7571-FORTINET
    "c9f83054e8f66a6c2168f6ac406c911586fd88e5bc0a6e6d438bff7bf7262620": AesKey(
        "24b365c5358717592b8627ddbac59bde3eb3cf39fcc6fe248dbbd5069e73aabf", "56862b12219312b73a90262d324b77f6"
    ),
    # FGT_ARM64_AWS-v7.0.17.M-build0682-FORTINET
    "b750f31dffd29de19b02e5468235ec37e924b51c6b6bb9b4a138f9b9aa2af552": AesKey(
        "6e12e549781ca3fc7a52354b9b51bc66658c519a68aa81d9b74ab42853cd0cd2", "59862d282b8e97c847549790042c870c"
    ),
    # FGT_ARM64_KVM-v7.0.17.M-build0682-FORTINET
    "5e9e668d8f0de35f54a24c89623ed2293945a25eadfd8d3d4f05d4fefe7d778d": AesKey(
        "046eac5228b64c2b80c8b2aee77869ea79328e402926db23139beae672f68879", "b646cbbe6a4acb85ff653ebf6132086a"
    ),
    # FGT_ARM64_OCI-v7.0.17.M-build0682-FORTINET
    "a644c5fd02b2974cfb1e228ab5d06e99a515c9eb4d7a726786af7474279354eb": AesKey(
        "747a98fd641c508c52816b2c6a2a7340f3a9202f06e7933ab58c27f138eb0235", "20402780738a61d6e0499ceb8f22ca01"
    ),
    # FGT_VM64-v7.0.17.M-build0682-FORTINET
    "a8b62e5e17eb72b6a9bbdb8607531a28b28b60c2baba8f57e95c149862d1f718": AesKey(
        "5154f28c46f94b7c9f12474dc4d427f02fb6dc181d9b070b2a0b7f42b80811fa", "407e6bf76503e7b201912847df1791ee"
    ),
    # FGT_VM64_ALI-v7.0.17.M-build0682-FORTINET
    "6265475de10fd22ab387ac0a9ca6981987040f449c229255d37176d8011a6828": AesKey(
        "b8d99463f4d1317fce2fafd46a75e5701b6c79144b7e55cbc47e88415bf6d688", "bc82bb3d7b10417c4af8d599e8bb06cb"
    ),
    # FGT_VM64_AWS-v7.0.17.M-build0682-FORTINET
    "9bd80ccc844c5af6555bc0e476ed57f36d8a31ff1ac32e075b97bdb7369a2b3d": AesKey(
        "74707882f00b4f0706539337b3e326a9ba8f6e9f5e5b5354ff0b8a1ae43c7030", "de914db56790cbc0f7aed46bb8f2dea9"
    ),
    # FGT_VM64_AZURE-v7.0.17.M-build0682-FORTINET
    "80bf15facb5a69510bf373fc429779f7d91fa25ed3cc7d571c5012825d3e6fce": AesKey(
        "27db8e775d9c25398746402201a2fded199041bade42be753b14b80995ca0b91", "34c518fd5553ebba6245a8d8641d0754"
    ),
    # FGT_VM64_GCP-v7.0.17.M-build0682-FORTINET
    "7a6d70cbc9e4b970d529dbeb7e2b589639ee692baec3b073fb9a683bb84cb9b1": AesKey(
        "5eb3b0363406ea0df55de7feae1e18b0570da7e76d15c21911657f16325cb289", "ccb5d2198b5934c1c0bdaeb4632035af"
    ),
    # FGT_VM64_HV-v7.0.17.M-build0682-FORTINET
    "79e29d46046e2650823e32e4f399ca87f174c95930334a2aa33742e80efb74eb": AesKey(
        "1f5c7e2170a0acbf591b429d8280ef65b452635842d538695f607f1df0909655", "94431d4e0e6e33c843e01b14e16b4eab"
    ),
    # FGT_VM64_IBM-v7.0.17.M-build0682-FORTINET
    "6c3b7bf543abfdf51f2893efc54869c13959e45571bcad08f5347073539dfeac": AesKey(
        "3ac3d8db6d672d2f684dbdc9a7b7fd2443721d2bd95a741f6d1ee0e405d28598", "e403048c66ec258d0ddbae5cf4cf5644"
    ),
    # FGT_VM64_KVM-v7.0.17.M-build0682-FORTINET
    "73dea307e07ebd957714b9c1549dcc0863be0518b4f457ffe13d703d37c5dfa8": AesKey(
        "e66caf8e8469d5e32e5012cf5c7c39081598cf1a7cadcb7bb96055d9ea1b08eb", "2fed2c9ab918bf221fd1073b00cc5995"
    ),
    # FGT_VM64_OPC-v7.0.17.M-build0682-FORTINET
    "57731c4fe6b6162ea5745cd9bbb8b16c2b62c62a3632a26585b639da572240c4": AesKey(
        "adf7e1bdfe12362c7bdc63b62660c50f7638fbb2ddf7f2d11724bf16ccd09142", "84f2e20a4d25b6e23fdd5ac9cc996335"
    ),
    # FGT_VM64_RAXONDEMAND-v7.0.17.M-build0682-FORTINET
    "587be90d1d024be96043f8954f0870f8e28e351226920eaf1dfb650906ca66aa": AesKey(
        "5f925f3e15e01e39b4c8750fb3c160844b3ba3e5b4f2c77fec190f54ef025394", "3746b1bf95adb4820748abfc521e8635"
    ),
    # FGT_VM64_SVM-v7.0.17.M-build0682-FORTINET
    "f1699dbf38475f9a69cc9aca6b6b2afffeffad19071bec224f3883c3be2707b3": AesKey(
        "66f805d8cdeb0e5bc9533326c684f6cfe454309c6f1622350bdd2d77079e4df8", "6ca842e8188e6f8b32d4521a84efe8f0"
    ),
    # FGT_VM64_VMX-v7.0.17.M-build0682-FORTINET
    "9091cef3e149e9baa333918d36c0752fc8db17ca01dd1c1d19c7bfc39a86770b": AesKey(
        "774792d16512714d639634226f52d53b6e65b7cfa4b3b2580850c842d11777fb", "2a8529e4b1b5f3fed19a5d013772dd5f"
    ),
    # FGT_VM64_XEN-v7.0.17.M-build0682-FORTINET
    "1153761ddb244322c85fbf5400aff826e82a01dc24234dc4b9515632a9088dda": AesKey(
        "63b06662d19ab260f0e5408ce9945193143ff84fbb705d600d4ccc8ce81f4301", "a5b437d295d99d0d36523fdcf2ae61dd"
    ),
    # FOS_VM64-v7.0.17.M-build0682-FORTINET
    "b717b7438436567a26c4d68718601c7480e3ee2722700701c2e31f62ea77a3c9": AesKey(
        "85b05d9847fae915a71d7d481ccc583cfa6cd3ebeb28c76530846aaed2a76f7e", "bfe1132021fe3c84a7285c740a3d1f20"
    ),
    # FOS_VM64_HV-v7.0.17.M-build0682-FORTINET
    "93146c91c4add83d55febb5060bfbd5c77fb345c29621545c7ac352c23528b05": AesKey(
        "353bfec539fe9d7f6dc491d9cadd3c429f3e028e9d74251061d5df88ca929ef5", "4b48b15a86d2e1b00f7de77f3c9fd8cc"
    ),
    # FOS_VM64_KVM-v7.0.17.M-build0682-FORTINET
    "1df69bb2e73516771f27d7d905f4bf19f18db735b688c839f0610da48669b17f": AesKey(
        "1e5173abbe8f39f960610eedb3db5511b37aa8ff85df5f68c5c9449a662541e5", "47738b81d2074b243c95e0c43c3b1349"
    ),
    # FOS_VM64_XEN-v7.0.17.M-build0682-FORTINET
    "70448f5fcb78701301dc1cf57c9a7ed7cb6f491823603e41d66ed8b119946ec6": AesKey(
        "404d302c34c06d2afc3c93f37f34c7dd76fd2caba8293ed19ef07299621871c3", "0829ae0171cc46a0b21198cac4092c90"
    ),
    # FWF_40F-v7.0.17.M-build0682-FORTINET
    "b7e8dbb6ecfeabbf1f099c68808be71937c292061cf913734aef0fe5de1a146c": AesKey(
        "2f6b21a2b6575fabe87997a680f7b47a4e1497a969e98841f1b4b852f5046bf8", "ba4dc664f69894c385f59ac5d282aaea"
    ),
    # FWF_40F_3G4G-v7.0.17.M-build0682-FORTINET
    "71cb60bb01b0364f4e7af0188f5a5a47c0bfc73af9951abde82078a09daeed6e": AesKey(
        "79e68a85e13f427079075942fcf57458c58e18ba5084f162c13889a01e20108f", "f43705e2b7b7ee5eb16db0f6551b2076"
    ),
    # FWF_60E-v7.0.17.M-build0682-FORTINET
    "272e47bb315deb42e579994480f4edda3237daee3d0c55e605c47da1e5e353bc": AesKey(
        "5816cc7c2520906362890b80a9e550aadaf4cc57a58ced3664ff9303ea5d3da0", "1c010184f619dcc53f4a22250040d779"
    ),
    # FWF_60E_DSL-v7.0.17.M-build0682-FORTINET
    "e78afeec3ed9f5fdfdc654740a5ec29c63471b215fe29caebc5e889b96d0c189": AesKey(
        "ed14d7b6fc6ff4143efa95c0765def8cc99eb91774371b03bb080a15aa8d4ae5", "8887a95f51e0235c2b45733db89ce0b7"
    ),
    # FWF_60E_DSLJ-v7.0.17.M-build0682-FORTINET
    "27971ba0d1c4bef5ed38e82a95913e50af1ac3f74c11b86b3c8864fe93063f33": AesKey(
        "02df42a1a084348d0dcd2bd590018ab3aa5b2beacd26192d63d64312f544c940", "9cb268507369fe5f4cbfdb45380439a3"
    ),
    # FWF_60F-v7.0.17.M-build0682-FORTINET
    "2e1185902b0b089b464e17b4f2e030b7ca996325f9bda66c493647bd15f1244d": AesKey(
        "f0e574ccc1442bf16300e81a9e3eaa9d15df44f25c7ebf7d9de448431cace468", "970f6b9f2c1059066f633b1f8a9e4111"
    ),
    # FWF_61E-v7.0.17.M-build0682-FORTINET
    "1ce268837d0fe534c2ff87749dbf9bcaef8a1ba3d0359756a116c21b30c506e7": AesKey(
        "11e1daa11a5613f5e54d8f4f043b2966fa5695bae63bb5e12c8af86aa9aa8bda", "d38fc8753c610cf5ade354bc6b2096eb"
    ),
    # FWF_61F-v7.0.17.M-build0682-FORTINET
    "691c430ad010c573a5a594398eb4a222e92f6030e569e305b37c8e060385abb5": AesKey(
        "8fa310ed97d3425da9e514744e627161b9b653fbcc70454102b884b3895d39aa", "94b2b34071785aa204acf9e840730a99"
    ),
    # FWF_80F_2R-v7.0.17.M-build0682-FORTINET
    "0375d255f7040cdcb557305224453f6839321336ee5a62c73021cf39ca5e3780": AesKey(
        "176b8a84c3a78105006f7e6fad470f2bd022493c97bca1438ada14688631f477", "261d0ba39f7b7e24983be21a919cf5c9"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.0.17.M-build7569-FORTINET
    "1bc5b70c24f7cc102d27ce69be08f313354f569dc327bbcf210ae86360620f1d": AesKey(
        "b03fc976d9d7acade069ead3241144412eefcfbc5c1f57b40f9464c6e78551d8", "7f8d855b3af12cb574d7521bbce67dfa"
    ),
    # FWF_81F_2R-v7.0.17.M-build0682-FORTINET
    "2ccb973ee98a81c43f582e2fc97360fa3f693ab86a59b385785fd29f8b978118": AesKey(
        "456f620c9847d0a6822252151a24867c573f34074e6be53a3e7c1b93608f0f68", "14c847009b572f533146934025c890dd"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.0.17.M-build7569-FORTINET
    "2e519bebd96b2160b787cec775630e88235729d8ba91d2f6d8142e93925a1a2b": AesKey(
        "385fbde5280c3a1238b2a389036b803c3fe613caff27641b47cb7d1fdc6d607c", "68830be94e32afff957848729d3321bc"
    ),
    # FWF_81F_2R_3G4G_POE-v7.0.17.M-build0682-FORTINET
    "e43358a8ed9be50e3366c5df891b5cc88e690bd0100afc3efcf4bb30862fc664": AesKey(
        "f174e50e1b3177953a3691de1983b770bc7b83d4589b2273fa7e5c7ccfadd4d6", "8c229e39e644a8ce62e4c1f2dddf2112"
    ),
    # FWF_81F_2R_POE-v7.0.17.M-build0682-FORTINET
    "1ea98b113bb33cb32063883860a6af25e511e68f4d770098945b4d6b1fb2b4a7": AesKey(
        "649939c9716d072f0b7977323ca54b23cb06599196eafe5bb26e2e3694330dbb", "05c6cad3086faf3534606bee8d75a768"
    ),
    # FFW_1801F-v7.2.8.M-build1639-FORTINET
    "2d31c6ec46a99b9f7daa904b5f26f08f46b5738497df9add812635dbe5477419": ChaCha20Key(
        "de4cabf18e97e1b9fc5be25a02c378c3bfd8c3de06503d576538ff60476607ed", "6d8acda780536b88c2996b7bc757183a"
    ),
    # FFW_2600F-v7.2.8.M-build1639-FORTINET
    "5ff74d1613c5a6806666831ead9142ca73aede05da5c0748dd43c8bef437b825": ChaCha20Key(
        "8e16c96dc0530a8c038f1ea29b7ea6bf79959ee4bef72196f45fe49fed0704fe", "183e177fb31f585d0a8a214842a87b69"
    ),
    # FFW_3980E-v7.2.8.M-build1639-FORTINET
    "485b2659b3e907e222ab9c70114d35fd59643e7b627d15d8d22596442c0edd1f": ChaCha20Key(
        "91b20dd0e238e7969070cc759e605769ac22dd03051fef4607d6629ce987157d", "2418f00c96eaa00cbb98a88afc979324"
    ),
    # FFW_4200F-v7.2.8.M-build1639-FORTINET
    "1e4b0ad8ae5afc93ece2ad261886b2cd18b1d0a3fed0a095cd185cf037df6c63": ChaCha20Key(
        "9b93c5849274fa07488ec49d780e170555815f57536b55885e5b136808f82ee0", "2931eb787a755b63617edc7acf3de8e9"
    ),
    # FFW_4400F-v7.2.8.M-build1639-FORTINET
    "011bb8fa3bd8eeef182c4751e9368c430036919cc61c040e53c6451d25986010": ChaCha20Key(
        "d5bc6af874caf08e9164bbff2a561d8ec2c2a73fde4715740f2f1e25ea2ef753", "0293ae6a7bbe4f1f99fe589b89c1cf36"
    ),
    # FFW_4401F-v7.2.8.M-build1639-FORTINET
    "e0b84d5482b505add7963201ab6defea1f2ebfd07df1901d9349f8b89962cb73": ChaCha20Key(
        "cfa1231d629b9adf73e98550ef79bd0687006468b494dd9cd8a0ad282d2fb873", "f112428439a0704dd15493d7d2b8c941"
    ),
    # FFW_4801F-v7.2.8.M-build1639-FORTINET
    "2f6560b12cbf28321cee06d0664c0ebe53efe8ebec233b82be97c1a9730882fc": ChaCha20Key(
        "0097d398207a8c5896c58975b45cc0d6a0af86a34560e047db5c7243235d9c7f", "e66a57e6f37fedc43473f4e807e0ace4"
    ),
    # FFW_VM64-v7.2.8.M-build1639-FORTINET
    "682e5e97ab062cd5ff94bb34e403973a63135ad9c0fab0cb24dd5d74aa6dc8ef": ChaCha20Key(
        "8a1875c9efad4a9733451126f8773bf4e957110a54a3d04273f1c0ad24b2bb49", "848f1d412b430b203b30f38bce435f16"
    ),
    # FFW_VM64_KVM-v7.2.8.M-build1639-FORTINET
    "1aa1c7dc2b22ea10ca7d6b711ed09096568b21c78128d70e2740216485c26ad1": ChaCha20Key(
        "51f5ce3be488cb9b508c5b75aa277ef4378fa2189146afb1a9a6a0966c5d83dd", "853c51b97c650d39af409f58d907811a"
    ),
    # FGR_60F-v7.2.8.M-build1639-FORTINET
    "7fcc110698f2a6f716ff9d3cf314a9e6d505e51af14ebf8db0c5a1700be3c638": ChaCha20Key(
        "19f794b77f4d3588a1d83444640104ddf82f685d313d9544a13053ce17dd115e", "ec4b860e35caeb45c0368259b3475986"
    ),
    # FGR_60F_3G4G-v7.2.8.M-build1639-FORTINET
    "a52c06dc90bf5ac697fd460ffb8a71ef0c0feb330dd3f5c760dff6a7b4a59b76": ChaCha20Key(
        "286e50b0dfaebb270ba35e973e473fef43cab24a7be5bd0478a6891814d8b2f8", "6ca86e120b55347450c70eb0aedb1ea1"
    ),
    # FGR_70F-v7.2.8.M-build1639-FORTINET
    "42e9a1c1ddfe3a1964fba47de4f2acafa349e531363f6ca7d655be0488f58f8c": ChaCha20Key(
        "35f0a2036f934c9f266c42c9d48c59891468a0def3cb756bf40f1de1934cb3f4", "75769d5be5f4ffdbe4324b2befa0952b"
    ),
    # FGR_70F_3G4G-v7.2.8.M-build1639-FORTINET
    "bec3b74d38779091bc610f7d2680e4509e5abefaab76bcfbf88554f372c6a221": ChaCha20Key(
        "157d0f5e5a740a2c7ee9bf608cea7a657d9d4de963671e79261ca6bf23fc3cd0", "48a9df31144d7d1cd3d39b8c564c156b"
    ),
    # FGT_1000D-v7.2.8.M-build1639-FORTINET
    "da3519b065ca128734f40c4fe01ad568aa6c0c91732d9ef7bb314ed5b99a3bcb": ChaCha20Key(
        "d97db45260e5746dee388651b7227cfb32c6730ec413e2333856e78c849ba254", "1ea1a0ca17c1195b0feb22292ad8fa05"
    ),
    # FGT_1000F-v7.2.8.M-build1639-FORTINET
    "8758513593ada4b6ec6547121d40e4e8945fa19bb5b499759200ea3d347aad93": ChaCha20Key(
        "81b1cfa803f3d67c1cfc3a632268580193746f237397521a2a28444acaea7fc6", "40678592f0e76c1830868afdf18b871f"
    ),
    # FGT_1001F-v7.2.8.M-build1639-FORTINET
    "c44ea09b1e36c340eaa781cbcb45cd5f5c99307e979832cb61555a539ad3f941": ChaCha20Key(
        "79afe6df753b0c4e6203e2a5f7e2ab0b2ff677309f0ebfd7fe57899b4e58f679", "f1a1c2ad19fe1c7d57d037e5789b7c01"
    ),
    # FGT_100E-v7.2.8.M-build1639-FORTINET
    "d4f7941fd8e6d38c8af329d47c1a0ac7585d08ae630ea198e1eb491b5c16d741": ChaCha20Key(
        "33f263187a94f5ab2f99321efbb26be817a7bc970ff199782b9101a3253a55a0", "7f2407da679cf736a7652247df336845"
    ),
    # FGT_100EF-v7.2.8.M-build1639-FORTINET
    "1ef1f235cab072d2d044827322b4678b391728119529e3ac13c09981b1b1f251": ChaCha20Key(
        "60a02d21d738d3d07868405337268ba9ec7ef2d05fde15e08363febe77766fe3", "6dd401f0ff5fdb4c30b1e48fb77572b9"
    ),
    # FGT_100F-v7.2.8.M-build1639-FORTINET
    "2f15db4102d89bc271f75685319a7c33f5f139f1fe94904875c4d1dbf0120e2b": ChaCha20Key(
        "31666bb627e340d6fe2fcb5219b4b52b6704008a4284816dd3507483693a29cd", "dea2afe36e5dcbcc18046dd3ee1227c1"
    ),
    # FGT_101E-v7.2.8.M-build1639-FORTINET
    "b76511734f05ab5c199245ab2b3f4ae27a57a404afca49600ad4fb887bfc8af3": ChaCha20Key(
        "293ccc6a963f3178ed4631d4fdbe27ccaf35f4d4f13f1894ae4876329481356a", "a5fa18ddb65f0058f83696d82067391e"
    ),
    # FGT_101F-v7.2.8.M-build1639-FORTINET
    "56f4874b15bbd41b9a8ef8af1452d2a0f90ff13c92dd7598204113f5a2e51a41": ChaCha20Key(
        "b8711cbf074842bbaf58c052a001210b2db19630c21fd3db56e2ff0da4f9d0df", "521dcef1f0d2bef9ed26f70bec88bfb1"
    ),
    # FGT_1100E-v7.2.8.M-build1639-FORTINET
    "400c1626426adc909778b9b0a1e4296b3e49a2393cc6a55b9440ff277a47b01c": ChaCha20Key(
        "7f558c06989c343401aa32efeb9060005b228a41b535de21b74074ce70bc8de9", "94c3ec7d87418c736c1076f5238c2ac4"
    ),
    # FGT_1101E-v7.2.8.M-build1639-FORTINET
    "fde9d562688cd6f74009696a6a2b5ea8354bacf4fa6d08cb1e4f8fc5dbd27be9": ChaCha20Key(
        "cc7d58ad402fd48714b37179141c0c347d242c5c1ee03220317bd94ad39e37f2", "7175e31870a1dd2ec17694a2b06d4405"
    ),
    # FGT_140E-v7.2.8.M-build1639-FORTINET
    "b77be71f7a059d2195fa24580b21db134cdeabd74803f2db903e0c9f0f106dd4": ChaCha20Key(
        "93c28c14b9853524d402e9aaaa3a2ae49d9078b0aee74ed53c4d5b9c5750590a", "0c457f2fdfedc844199c612aca9d7aa7"
    ),
    # FGT_140E_POE-v7.2.8.M-build1639-FORTINET
    "e8a841ec19439b130cd2358ff7e58a33718e25440412931e0191a28e269b195a": ChaCha20Key(
        "fa767e2cb77a380262e9af417886228ebce927a97c00cf1160bdc329a4163ceb", "20e6132468c7abdf97badc099ba3fb3c"
    ),
    # FGT_1500D-v7.2.8.M-build1639-FORTINET
    "2dcc99be7771d760feb66fe30be75aa8ef79173aba3ca8c228ac0e1ce40415e4": ChaCha20Key(
        "da248179be42436e60557e08a3ba204e11b461c7c434f04f9fa99cdbb891bbfc", "cc71dc4ce50157b8467b57db8a0a5d5d"
    ),
    # FGT_1500DT-v7.2.8.M-build1639-FORTINET
    "67f801af69e2fb083f124366d8ef07c97df7f90c07221e43b3b741668d5da38f": ChaCha20Key(
        "01b992391d041da4b3a42dc7265db304889f0dc1000131df6c42266e578d5331", "ca16e6abb5a0aebb6f9264e2b0c75617"
    ),
    # FGT_1800F-v7.2.8.M-build1639-FORTINET
    "c381e496e84081742e42155f7a3e472e16eea3f4655a397271cc127e9cba2ca3": ChaCha20Key(
        "1dcaf90e756077a90e04af9e2f0bd120329533db167275223225b29339cb824d", "71bbab37ac57295e4fd090a5214b4321"
    ),
    # FGT_1801F-v7.2.8.M-build1639-FORTINET
    "59bb08655da603e2a59ccaca0661a2a852622f48027dfdff4d3dc8fc49cfd476": ChaCha20Key(
        "e5dd62201e11c03854ddacc7ec8d4337a94cd84b7884e758109bef366e2cdd13", "97eb9fd0a579e3585cbe47c1ed1c728d"
    ),
    # FGT_2000E-v7.2.8.M-build1639-FORTINET
    "9fc15c00d007f71e2ac94b5561fa6fe6943ea6f2b7e941ea530f40128648b14a": ChaCha20Key(
        "1bbd875c963f719e89ddba2946c98fc6a047b7f388316921858157e9fb034f1a", "8daf9ef321d1557bf4b1f5e26115f279"
    ),
    # FGT_200E-v7.2.8.M-build1639-FORTINET
    "2eed1d4d833e40a3189121b1efc07fd10e7773220a5691819ba7a5f02bb2743d": ChaCha20Key(
        "e7dd625a33e55711f79cfef72d87a6fdee68f4d9435764884fdf5fb86e5ea5ae", "1cebe148fb82e3ed3715214b57b4bef2"
    ),
    # FGT_200F-v7.2.8.M-build1639-FORTINET
    "74376516ce89edf99263d23ebf2408bf52b7290e558a0b692ce681cfe764d5c3": ChaCha20Key(
        "dc337d3f10abb009751fbfab94a3a22b14b215fea471b991114740dc0d53bbbe", "51466b32d45736f8cc254c179f55cfe6"
    ),
    # FGT_201E-v7.2.8.M-build1639-FORTINET
    "a46941c17494f49638d8f62f35898e40287f3d1437317432678463d7b096453f": ChaCha20Key(
        "e58fc1d654b89d7493cd43aee1ad4c357f85e9e693dd973cd5359c822d094311", "74f68723697dd827ce84834f8eb509da"
    ),
    # FGT_201F-v7.2.8.M-build1639-FORTINET
    "ab9cf84d0b3cdfa1eea1387d794c6f826d9e3ee49a4079028fef1c21dfa530d9": ChaCha20Key(
        "9ccfbdef2abef1a52cdf9b31e66d33b963190b5da7abe1e8caa04d0c962d7be9", "70d10025763cbc25df0ad2e4e62a36a7"
    ),
    # FGT_2200E-v7.2.8.M-build1639-FORTINET
    "15fe719e99b40d52a6cadb7bd40d77fce68c8e85190d8e721a3337ca1cbc46f8": ChaCha20Key(
        "62ef22655d1cd5ea228802842a20af24002435860dc8dcb1a1e82c5425b810bb", "6f129fd03661c3025b17c5ee29b5f898"
    ),
    # FGT_2201E-v7.2.8.M-build1639-FORTINET
    "071551e67d6fc3034f5cfd63bfe1005d76cb58ca84a9818b1fe932dd273938ed": ChaCha20Key(
        "fdace106dd4d1b97878d6708087429a072ec29dd94d3be8958de636451e0c242", "bad3d379bd26005f346b1e0578dd205a"
    ),
    # FGT_2500E-v7.2.8.M-build1639-FORTINET
    "123a47e4f9bc224a166ad41b95e86ab4a331f4d72e83d5fa31302726e52cd0bb": ChaCha20Key(
        "d7916248b1f317be242354860af8e23e82d99433c71ca0ed3782d701eb098ba4", "86898a19616305f1af11c56d74a9a6c0"
    ),
    # FGT_2600F-v7.2.8.M-build1639-FORTINET
    "94ec35e4a7a9d3a8530a212faf6e195e4b50167ef891516ed86d7b9e87c13f25": ChaCha20Key(
        "b8911458e1e83e8f9b8e710f362b90c378a7b100ef95d803c37669dd46faf1b4", "44b240ba0cd84dd305c52877c0d3ab99"
    ),
    # FGT_2601F-v7.2.8.M-build1639-FORTINET
    "975c954e03af2f8e0e62b16e693c3bfe82da5a20768b2099c6af032b758f9f12": ChaCha20Key(
        "65e865fb4974937e837d197bc056ba60f1672a4c69f41dd7eedec6bbf96ad3f6", "d82a400c1677a4928364fbff787033eb"
    ),
    # FGT_3000D-v7.2.8.M-build1639-FORTINET
    "4e1cf2c304ddd59f0286e77eed96d567489bc951e51d015c6e41a263e416cfb8": ChaCha20Key(
        "4cfb0fa1530a68c8185a61badd414c86a11abecf2c5334b2e938edd176eb1f9c", "c15d4edb087471c77e085cab74c6171f"
    ),
    # FGT_3000F-v7.2.8.M-build1639-FORTINET
    "33acbbef9d456a456bf89d7ea10cc52ed6ac150a73a622ebc1040f95f2b347d2": ChaCha20Key(
        "cb4ff0ea7cff3c6850fc64af5f3ea3268c150df2ddba3b5af8a940595d6623db", "3760d12383a2752d4d566810666aff61"
    ),
    # FGT_3001F-v7.2.8.M-build1639-FORTINET
    "8971d1adc0ac0f1262ec41fdf581fb5a3ed4e9257bb8f37e4dd19153549a5ba6": ChaCha20Key(
        "dcd809d12202f81b1acd0090328b6f48f04ed05d28664dc4093d09562985fc76", "08f38baaa32d7d0e97c2b79720e6567b"
    ),
    # FGT_300E-v7.2.8.M-build1639-FORTINET
    "09358d05cf30d20d411fdd00df3440f6573008110625d15682ad54055b726377": ChaCha20Key(
        "4d031e983b077460a60914b39aa3e8f0470db64dc2cbf14e7a702de5abf66b19", "e2450bc6226236b5e122b5b3cefccb4d"
    ),
    # FGT_301E-v7.2.8.M-build1639-FORTINET
    "0b255fbb6c123462dd1c6962182d9f12b1556824f192a9d3d820ecbd261fa4aa": ChaCha20Key(
        "154057d04dd09e20cdb3de83be7ecc8979b56bb22693456bb2aa03e3209dd1a7", "aa33ee644d1a62eefe60f0df38001114"
    ),
    # FGT_3100D-v7.2.8.M-build1639-FORTINET
    "c50a45eafa37fc542cf42fcce189c5c5b591c170e4d1d0b86363fd910f17912f": ChaCha20Key(
        "2ab7889a860f986a6d3b44c26bbee11e4baa1ec565db5ae5e02ecacb4a57f9fa", "c17f1197dca4ad20cbf061a2f7203353"
    ),
    # FGT_3200D-v7.2.8.M-build1639-FORTINET
    "d6ebf519be5506aebe75d0de98c44c6de443d5314464bd25cb59594b72e329b9": ChaCha20Key(
        "72ece48833fff9a06a9145a124b8386f1c7a18435405619205feeabd3896cb5b", "b69c176f7a9961b2759d94ad7f706499"
    ),
    # FGT_3200F-v7.2.8.M-build1639-FORTINET
    "ff59328c6ce4d50a6dce2915e80337cd23ce7647adc243d4cfba4edf4abc1241": ChaCha20Key(
        "1b5a02a3155915be1572be97b282b5377889e189f4a8faab055801093a86b626", "96ea337f0fda892b3e2bc2c203b4fd81"
    ),
    # FGT_3201F-v7.2.8.M-build1639-FORTINET
    "8dccf7a3ff03df9efea025afef475938f42a47175fd632241c26812b07de6c9d": ChaCha20Key(
        "3bf2283c92dae6049cdeae3290c1efc170cc49bcd8d41095d0e7bfd67c7d8fc0", "5931a3f261ab01d3ff705bc5b9117302"
    ),
    # FGT_3300E-v7.2.8.M-build1639-FORTINET
    "9bb80edaa3457e4cc3d3559c4d99b2ba3cd060dc3f89cc1e55d633910cb411be": ChaCha20Key(
        "8652a3cf0a8f16dc62f0438c9bda313b28cf622847a71fcc4e9a8f4b2e006dba", "6085eee578508ca8624dd5ff77d10a24"
    ),
    # FGT_3301E-v7.2.8.M-build1639-FORTINET
    "92ab1d363cfd35ff34c36742d7401ba70aefa8c635e241ea14073ca45610802c": ChaCha20Key(
        "960b9c11127392c4fe150ec71dad81c85fc73036a675e3d51ad932ff1973ca98", "ad2ca42cc4e8d32c208d987db7b97e05"
    ),
    # FGT_3400E-v7.2.8.M-build1639-FORTINET
    "c67c945e4562fdc15aef922f3422ab06e1d442602d9beb5356bcea293d66de0b": ChaCha20Key(
        "f1807577482ff3b4b90682cb203c60b59d0ef8aac420452deee2b766bade187f", "44fcf1efba453ca29a426d6d78738674"
    ),
    # FGT_3401E-v7.2.8.M-build1639-FORTINET
    "16f9efe8980d40b7c16f4eb55d3b6715c510880bfe921f17c745359becfe74bb": ChaCha20Key(
        "3dd0c902ac1f1a055b0977e18091aa04df7715b8fca2e7d2918611da1a1b31b3", "8b7c662c3f1ae5933c23d97abe9f4135"
    ),
    # FGT_3500F-v7.2.8.M-build1639-FORTINET
    "0d89ba90a8214e3af4bc076f9147472006e8a9104d86e8d1b69d9f0a79c0033b": ChaCha20Key(
        "dd5222eb3e48a1e01b2cba3f54a9f4b64a664609c7c27eb12279e53eb7159a9e", "7d0b999e20ec02b7f0acfc8fa1dab772"
    ),
    # FGT_3501F-v7.2.8.M-build1639-FORTINET
    "3144839ffeae16ad2a068caf65996ff8a3314dd016939f3088d4f8925c073159": ChaCha20Key(
        "f32f72d1ac4f0020594091b4fb6636977cf40107cfa4308175638187301534e4", "f177814c20919768db5907f20596fe47"
    ),
    # FGT_3600E-v7.2.8.M-build1639-FORTINET
    "ded7a1b6524a85f7e5f4a9921fbbe51eb45f8e2b2d9276e5781fd63ce38dd563": ChaCha20Key(
        "e6393f2f95d66c330875185b474d4b012bd987da5dd7cb3095526ab5982c40ef", "4cebcff9dd0a73de1975ac2364dd697c"
    ),
    # FGT_3601E-v7.2.8.M-build1639-FORTINET
    "48fae60fee3923cb5df7920e4cd019288cbf117dc50ccc68bab09812b02e7c94": ChaCha20Key(
        "8893b26f0ed6eee1b0944997e1e5ccf1b757ab58d74de90d94bdc457183abf89", "fa15eed3399046d8cd0e212913507ba9"
    ),
    # FGT_3700D-v7.2.8.M-build1639-FORTINET
    "1f6932917d42959b0d604a9d9c80abed99c318189305568c32a82d4de5f92f92": ChaCha20Key(
        "1583e3a4c87da586c9d1ab64a9459c7d8fd0018fdd908013340f98a20977e9d9", "faf3bf3731c3bf365aa266d38edc2779"
    ),
    # FGT_3700F-v7.2.8.M-build1639-FORTINET
    "bd8661baa00d2914fcdb7ee15b39862d15fa16244ca8c6daa033b343f42e8fb0": ChaCha20Key(
        "b2700afe783848612d2898662e623dad9a3b81e71bba3b7bc278f656ee40209a", "7521c47415ada4d4d5f3fdbc1e46dbc3"
    ),
    # FGT_3701F-v7.2.8.M-build1639-FORTINET
    "58c9d3022d58096dbee137472c6cbbabd1da3141f2409395dbeccc5a98b8aba9": ChaCha20Key(
        "52333114b268b3d872d8f34e5084c3a75f625a10b73def7856d6b0e88f4bc449", "efe35c5256fa5de6dc01b08a3ee80638"
    ),
    # FGT_3960E-v7.2.8.M-build1639-FORTINET
    "85c46ecab2a3954b0fafc9efc9ebd8c819d861ef92d82b08dcaace2e606e09a8": ChaCha20Key(
        "73d9adbc610eee234f30ab66e909a0f63b3853cd929286ce8fad68c8b5309881", "043b38bdf3d6e4f28f1c7afc3a7d9a9a"
    ),
    # FGT_3980E-v7.2.8.M-build1639-FORTINET
    "89e520ede1fff8f6a5e5494fabc38e6bb00d3115e8b132fd3157a8070a54a0ab": ChaCha20Key(
        "1f584d48d0b730ef6fafc82965022ae5c87434bf96d4273afbf9d17ed74e88e4", "0abb03ac5c83895da03e2848c0cb9629"
    ),
    # FGT_400E-v7.2.8.M-build1639-FORTINET
    "5e5c9d2d34f6aaa0d6838aa49c82fbf1e094eb654e26e424c0cb1fbed0442199": ChaCha20Key(
        "fbbc84a63faf34cd9f7692252d9e4c543c9e0eba810cd8388348fe5c96c4e283", "e5a760ef1cca3f9531468b626871b867"
    ),
    # FGT_400E_BP-v7.2.8.M-build1639-FORTINET
    "6830872999b50a5b49d63ed84eab098df94979d256e4cbfee1dee294d15eb6ea": ChaCha20Key(
        "5f1ccf622cef3ae42e9233259b7d8d795ff4e8243f3b7cdb5eadf615d20f6b20", "55bafbc99a86ad5a90ce43f1c7407409"
    ),
    # FGT_400F-v7.2.8.M-build1639-FORTINET
    "1e9065c034811b24bb71c15cb05a9160e51d28a1a73ee1ec90b1501c1ecb8129": ChaCha20Key(
        "661244beea5fe40a559ecd43c8c0779f207f20534b91c6b2520c2687c1b6f151", "e41972a55b47dc36eacf3c35ac26678a"
    ),
    # FGT_401E-v7.2.8.M-build1639-FORTINET
    "99ebefcf2f9ff42f2386df98c5c9c5049da9e96b246fd6e950f9eda68a858e39": ChaCha20Key(
        "dfff9c3e76340fff0a8750fef7967cc000fe4974c15c9f834f43723c46312198", "f6ad0adb62e1f812b45c7c66d6ae47f9"
    ),
    # FGT_401F-v7.2.8.M-build1639-FORTINET
    "5cc5e3ea2a5b82f2d42c17ccd127006d55bc43986ffb04da7df860d4a0689a24": ChaCha20Key(
        "ce3849a9c5f9eaf6e567584e802d0bc78747772c9c127f564a81377239a7b454", "01c2359d89d255d9b4dffa79cffa0ba0"
    ),
    # FGT_40F-v7.2.8.M-build1639-FORTINET
    "67d550035b825d441fec70275639a3f3d56139fd328786a22c0c6c56c2ff6fe1": ChaCha20Key(
        "b741e71610e6b62b8cdf72dc4327cce21f7724a942bb28809b214061c4d2e571", "991d40a75fb11f3a36fc680551a7b4ac"
    ),
    # FGT_40F_3G4G-v7.2.8.M-build1639-FORTINET
    "9ff6135cd94b94c745276d33c75fb3486d23c86b794f3ba947a52b245ac5a55d": ChaCha20Key(
        "9573e304344e04a780ad04bf88655c8b9b8362b9380e071ae94e231fb3fe8761", "0cd04d109cfa2a50a6865bc15440d63b"
    ),
    # FGT_4200F-v7.2.8.M-build1639-FORTINET
    "73d60af49aaec52ca295aa56b25954e7c75668023cc904702b00b96812f14f75": ChaCha20Key(
        "e6217bca062453ce7dc65a32850ba0a7a3b4636183a0635a865df1b9c9405e41", "4ef6a7bd57d6fcddfd9b5d1cb7f37f5f"
    ),
    # FGT_4201F-v7.2.8.M-build1639-FORTINET
    "1468d5903c1004df07be5669ed6cc363668e422cb927196961d2b8b6fcb29473": ChaCha20Key(
        "d0203c03a27c9431363661344cc248064e0fbffa03ad52314b3f0f402c9c9637", "9fb26d17474510abe9ac1ad25e151413"
    ),
    # FGT_4400F-v7.2.8.M-build1639-FORTINET
    "eefbb5de88542f8bfe815bca69ea4d530bfe7f9734d1e5013fc5201ce7f96f5e": ChaCha20Key(
        "c0feae00032e730b6af3f0ee74c41e8c5585cd42fc5c25efacb82e331a0d7bd6", "bb84b1ffeb99e93c6156d64654646a2a"
    ),
    # FGT_4401F-v7.2.8.M-build1639-FORTINET
    "c083ea59a6349f485b8d41d850c53edfffb8f063c72c87cc16a90902fcd536f1": ChaCha20Key(
        "ca96b6f50496b455e45ee01340b6258e63120d1fbfda1f0ed3df5a19753048a2", "c20a9e7f433a38abf433b8611bedef8c"
    ),
    # FGT_4800F-v7.2.8.M-build1639-FORTINET
    "dd90a31ef04e1081a9f814382843fa3ea412ade4d4ccf270c4f3d163ba9fad41": ChaCha20Key(
        "c2f5cf034951241e5eb0c17cc0ff4a0f40c931d077dd16eb8929fe9304a06ed4", "8eb94560b0ac02b7fd6ec2be8e194bd4"
    ),
    # FGT_4801F-v7.2.8.M-build1639-FORTINET
    "e8eaf210e7b9b519d2f3e5965751002bdd85c5111e70e33971d78559447b9366": ChaCha20Key(
        "6bde70858f1d0f4e93616500c309fabb54540208813601334f6ca2302b43dc92", "4bf8077fefb03af7dbf15f9b324622a8"
    ),
    # FGT_5001E-v7.2.8.M-build1639-FORTINET
    "23cb8f5c50427333e6fae68b4aa4ae726094dc5d84631392678db492659169f1": ChaCha20Key(
        "8ba2c7ff29f71554982e62598d70010c5e4f9b507d92601ea5f74277f871a2dc", "4b7bf0d0cbe5e37e291f0238b783999e"
    ),
    # FGT_5001E1-v7.2.8.M-build1639-FORTINET
    "cf600577e05bb7c77e219a117eacdecba39e422d25ebc2bff408de66e101b028": ChaCha20Key(
        "989d56006d24ec9cc6019f30cd0f32a52f14a62eb9ef8f41f282405adbdd4461", "0e5efdb74c27d1a6511bc177a4fd3930"
    ),
    # FGT_500E-v7.2.8.M-build1639-FORTINET
    "1193767173244b1a5e35ae816f2e1369d97bc40924cbc17f19c75605b332980d": ChaCha20Key(
        "e14a9214ce4637d8fd436d43b117ef83f1d3c4ed5700759f67eba0018d20a699", "73a9fdb051bd0e2d349e9396ea639fa9"
    ),
    # FGT_501E-v7.2.8.M-build1639-FORTINET
    "ab2d29508609568ffc905cd70d01024f0122f04eee3284a154c7a598c48c0d9e": ChaCha20Key(
        "b580e18c6f7d81bcc37c60ba307aa5fcdaa5c336a6bdff2488f8a16cf83c4af0", "f74ab9b343c1e297ff03d93875ce8649"
    ),
    # FGT_6000F-v7.2.8.M-build8735-FORTINET
    "cfdecba44683e3d008980532738d17293f9395c39af10215f123fc9ea9934c57": ChaCha20Key(
        "bf3adbe6359be17c35a52b08976924b5b3364572f58c1ad475945c1e8665b14a", "cc1f0aee1d27aac2e36ece6568f5394b"
    ),
    # FGT_600E-v7.2.8.M-build1639-FORTINET
    "144a2ee786d244a453f07e58f55550bb0fc7c5b9ba3e8210f53889fc9655dab3": ChaCha20Key(
        "0ae0bda73d801f609c4c43e017ff4741f4b301d89a39920abc674a4e51e6b5c5", "667b856073b0d47a80966f33f8fe7856"
    ),
    # FGT_600F-v7.2.8.M-build1639-FORTINET
    "4458a89a49f2ace5669cbe8582b7a070b9fe154ac01dc37e34940268cf600e8b": ChaCha20Key(
        "9c1366281c75d0d9672a3f110783a02adc9b87f3a6e7dcef8d19d0fb9cc08aaa", "653bc9c95f6766c1b33df0dc7d66af60"
    ),
    # FGT_601E-v7.2.8.M-build1639-FORTINET
    "b6f3552e2ebedfa7da3fef0fc6d6c2bdbc2fbbf48503e616f244a5e6d16c3c90": ChaCha20Key(
        "1724c6697310d58979c58a948c5c966d3d166cb1f6fbf457106f48c07cf39cc1", "d682d7f9d1b18a13131f98ebe7b4865a"
    ),
    # FGT_601F-v7.2.8.M-build1639-FORTINET
    "0fa8c119562a7b873053aa85e7b6e8f0011a8e5ee430441b191f32b04fbbdcb6": ChaCha20Key(
        "6ef08695e968d7bb6e1119c7c23ef6052f01fb0638529700bbfc5df3237bb0ab", "73eb017dc9b851ca7fedf9df60cc3abf"
    ),
    # FGT_60E-v7.2.8.M-build1639-FORTINET
    "725b29f232185ac41714a9a24dab67389f9ed13a5acb8eb0f2afe819a12ee2e7": ChaCha20Key(
        "bcbbef1623cb61ad465512c8712bc7f0e7c1c1a842825127c1c8899e2bf93b5c", "09479ae250cfe8c55d8ced4afeca2094"
    ),
    # FGT_60E_DSL-v7.2.8.M-build1639-FORTINET
    "0f26c02ac9adf80c8ba6cc6d581acd31a42b39b17be7e41fafcf972e5986bcb4": ChaCha20Key(
        "feaad2a775908047d93c45de1bab08d709beaac4fcc47f740964ab006e19df19", "9e1ac4e837467398b1ca5766536f4371"
    ),
    # FGT_60E_DSLJ-v7.2.8.M-build1639-FORTINET
    "01530a78f9cb6bd7b7fac32d0d060a63574d1a117dc2352db22f2f1464033ed8": ChaCha20Key(
        "8b6452623480059ada4fb8be5927e69692d6f393065840d1b0854292f3539f15", "ed08c3760daecf2e3fd112c48d070b1d"
    ),
    # FGT_60E_POE-v7.2.8.M-build1639-FORTINET
    "512623fbbd555a6c04a2225fadc95a0ec1f4212d68f3dfc68095b25e3ddd7ab1": ChaCha20Key(
        "0fd7657848727590fd1e021ae462520e284dab8cc71ad8cb3cad67ed53d7cd2b", "6cc45db8cab11372b758a4c70af34000"
    ),
    # FGT_60F-v7.2.8.M-build1639-FORTINET
    "12c2bf744769a83132b7bc6f64dab9db0345404ac2c11dcb1c288eeaf621dcc0": ChaCha20Key(
        "9e169849728c756880c1e1847928c1f830243b0b4c90dd4a5f603bb5561f8a6e", "1658ff05e2ff286549fa891e852e2dc1"
    ),
    # FGT_61E-v7.2.8.M-build1639-FORTINET
    "8b6d0fadab2240b3f55b5d5aa719b27289f45274c7e0c40e9bbe3a51fadeb349": ChaCha20Key(
        "72d74d41652f1f6a322df2091cd01d8c912cb27ee615287f251f5c88ac398df4", "9fd2da05ab295b2f89d184db9fbbe45d"
    ),
    # FGT_61F-v7.2.8.M-build1639-FORTINET
    "e7e03c9b9bbe529230e8b5e95909faec1c3f0cd1ec0e27491b2e9819f80d8f4c": ChaCha20Key(
        "b354d69ef67aff5833f5b6fe018fa5ea40ada12db87b0440eeadd22b30c87eba", "42b02d535361337258bf3874601bf657"
    ),
    # FGT_7000E-v7.2.8.M-build1639-FORTINET
    "35e3aa6075d90ff662b74bc315701bf3484a2a4db54ff3291b04573ede3e48ce": ChaCha20Key(
        "982d47958b35f14140f44e004524df125473d6613b147df8b62fbc2d2c049866", "a84ff3483d9127631419a60f97e4bfc1"
    ),
    # FGT_7000F-v7.2.8.M-build8735-FORTINET
    "7fe2f1240e2d19249835555212d52de08a1b1ab7564c69206e9ad49955b594ee": ChaCha20Key(
        "277f6096bdf6ed5407170873fb3ac8eeee19a02b4e077450977b95f4146e0488", "d419aae6f7ce520b839d6ff853baef20"
    ),
    # FGT_70F-v7.2.8.M-build1639-FORTINET
    "8672d1a597f0397a8e539ac350088d4cefbe58dd1366ad1d52ccc3339c921fe8": ChaCha20Key(
        "f08dbea9ba536f99b45e0d6a30d5166cc9fd38c140da55f6f15681541bc3303c", "b37aab25625c06415b89254f9b4eec5d"
    ),
    # FGT_71F-v7.2.8.M-build1639-FORTINET
    "1f04969f3f8b682c7cd021dcc190f63e9bc3a5f9da960a8e1956d155c988f42c": ChaCha20Key(
        "5db24d3d944e933739751fcccf6ad6fccd2fa0b6dfa5c580ded9d360237ddc0f", "9fe227289be08b9894fb91a2ce9f0864"
    ),
    # FGT_800D-v7.2.8.M-build1639-FORTINET
    "e000770e4e41f01083ace46c71066d24872f478bc0fe43b35a05bd5b9e3f225f": ChaCha20Key(
        "be9813d56790b6d586a84a76727f399bfcc34b202381c93e17a5fae3fe22d36b", "8a2ffb097a39c1aa9ab656fa98c41664"
    ),
    # FGT_80E-v7.2.8.M-build1639-FORTINET
    "2319996e6e08a37c96b7336655bc8c04bdbc4d5e5a0472fcfc3d7bd6fe0f3d2e": ChaCha20Key(
        "1f3760ade8dff8fb5169a41045b37158a373bf46797e6a97b39479290436a66c", "7f0cc46c7be324a217d082243ab3fe7b"
    ),
    # FGT_80E_POE-v7.2.8.M-build1639-FORTINET
    "5986cc9590b3b444d7ee4e6ca79e6c430cffadca39042cdb415b0dd98f3f29c5": ChaCha20Key(
        "50385c2698d610e9cb1d1c26330e22ba9f54ed3b24a50ce67aa8d85d69bc7322", "bab71ac6e52b2f63661f2efd3645deec"
    ),
    # FGT_80F-v7.2.8.M-build1639-FORTINET
    "e0266aec7fb02f034ac44b7681ae0344db122b800296d377f4d549f1360b9d38": ChaCha20Key(
        "3fc6a64e81c1845980223d7b5c6399657e71e0cdc8cf1efaba44025b6a2be5d8", "f492c5498be1e5ee8d566a68c8f08faf"
    ),
    # FGT_80F_BP-v7.2.8.M-build1639-FORTINET
    "7dfabf93e7ab3b1d2f4e7fff57c0588e115917a28cd6f8fb75aa7b452e9aa225": ChaCha20Key(
        "fe66b5b2bc7a3f34941ab01bb56a86d15d1c6630d88c60186e4908d4c297c5e0", "5fd5997c5b827772dab1e509c80746a4"
    ),
    # FGT_80F_POE-v7.2.8.M-build1639-FORTINET
    "ff9bf529e843caa1c9b9173595b94e210e8e0522578cf7f227532b2a065f65bb": ChaCha20Key(
        "04a2d6872fdfa3c08cf6ec8ef93d11df5bb1e2d9109d0dbfbdfcadca06203019", "ec45ffbad837e5c6787457833c8d9a0a"
    ),
    # FGT_81E-v7.2.8.M-build1639-FORTINET
    "d1fadba4fffb09e5e43ca10cbcf5528978a17df053ca6c32a61343a2f273dc53": ChaCha20Key(
        "2489d25fb5f456af8eb267c2953ee4df6e60043c23dafbb95b363cdbcbb7abe7", "9132a5ea97dbf600b4cf0d977ba0c9bb"
    ),
    # FGT_81E_POE-v7.2.8.M-build1639-FORTINET
    "b5ed3fc6bdd35ecb805c1bd6cbd822c6ae152b52fae464d5575f274bedcb62ef": ChaCha20Key(
        "50de681c1fd6d835e6ca333417c603261bc45b8cf4303fdb75c39d6bc94e3807", "d72042bb961555f31cc0b647b4955f89"
    ),
    # FGT_81F-v7.2.8.M-build1639-FORTINET
    "8071e1ea458566f748aa8caee8849123ad385a4670cb03ed682e2002f482092e": ChaCha20Key(
        "e1a942be5dbed517f12ff3efdcf2116d483b49c8bb4b205f88dbc997fb47de80", "a641a94c5ef90d20641e256ccad6d1f8"
    ),
    # FGT_81F_POE-v7.2.8.M-build1639-FORTINET
    "96b05ef12513a38e41d601f75d0c5838ff4461aa4c43d8c54c0cc971e75e5f8a": ChaCha20Key(
        "d6868697754857c5a4e2c327dcca96266bb8776d094078aaacf05699543d468d", "541ae8ed0d7799ab9521773b2e3983c0"
    ),
    # FGT_900D-v7.2.8.M-build1639-FORTINET
    "b9b3c849c343c68620f710e0a33ba01d8465d279b8934917613500ed8e2718b5": ChaCha20Key(
        "b995e3ab8df6618823f3838ef8b487f9dee7e8183e82234c0753fe45901a54b8", "65a0f47374dc58f3d9f23d51370ccf3e"
    ),
    # FGT_900G-v7.2.8.M-build1639-FORTINET
    "8b3aadadc54412d947f630128cfce0cc0f509d5a85d5cfd560f055f20db68f43": ChaCha20Key(
        "c4c9a9e89bc1cd0fde7f41a5a27a1b4fc88017b88db8def26440a033ae4cece3", "1473d67b81bc6ce46c8fb6cd5ac29106"
    ),
    # FGT_901G-v7.2.8.M-build1639-FORTINET
    "9ae95f3b78b9997f7533535ad28cf716cf5e9704840b5454318a5e113f5de833": ChaCha20Key(
        "738a8ddd56071a0dfe066bc5f9ecf1b76c654ece28bc20b441b11cb1f5190c06", "3aa8ed2b4c6f5b36a596fa79e473bdfd"
    ),
    # FGT_90E-v7.2.8.M-build1639-FORTINET
    "69f0ee654be0e16d31e93f6968162d322f09591aa8015bcafc81b03b3fa14cf2": ChaCha20Key(
        "c2c69eb80a7520b057dff0c1f15ac4595da9892eb0eeab3ce0d8ea46232b4b5f", "b471a4bcff81bc687236e0729f78ff12"
    ),
    # FGT_91E-v7.2.8.M-build1639-FORTINET
    "70ac8dc6ab78b83485a483bd9e55e8e09ffaed903828c32581b519bfc2aebebb": ChaCha20Key(
        "89eef48fdd0eddc7fe6a401687eb23f624be47861184add880197c96a4c1df9b", "3753dc58eeeaba0b3cb4fbfe46fa1094"
    ),
    # FGT_ARM64_AWS-v7.2.8.M-build1639-FORTINET
    "f2b5a88fe8656b1e46c3b309c8e43ca19af17e853b3fee62127feed059c219b2": ChaCha20Key(
        "22654bf237199c3a21e11edbeeb5d7a2c083ccd5115401cff07d4f1e13773f0e", "12979874279a59e0f0584972d24d6e20"
    ),
    # FGT_ARM64_AZURE-v7.2.8.M-build1639-FORTINET
    "be0aff4ab2f152ea3af89dbc71a5a499b5884e5d4e6a093b3e8f5593e0608ef4": ChaCha20Key(
        "a8c7c53c88a988c07b5bd9aa7246431e4dd5f047be158f7ec3d243bdc7b9d764", "bc64ef717a91170ea2ad2c8d683df32c"
    ),
    # FGT_ARM64_GCP-v7.2.8.M-build1639-FORTINET
    "22f7a941e6a30489696a7f1fd3569b167cd0cfb6867e461390e1e03e01dd32bd": ChaCha20Key(
        "00b4bc435d0d924744e8fa2aba6d810f808523e26d559e7f9847b1c60efa0b0f", "721002af6853ed806d7b7018fe357d06"
    ),
    # FGT_ARM64_KVM-v7.2.8.M-build1639-FORTINET
    "de9cb2968db91cae133ecb890cd5f683385e751f3768706cccd8e092e4d574a0": ChaCha20Key(
        "2ce3c19d0f5ef281b772b3a8a815cabae742e5fa99d70404b94c0633b005efc8", "54156ce63d7f2f8be9f714c73603963e"
    ),
    # FGT_ARM64_OCI-v7.2.8.M-build1639-FORTINET
    "033d14a2399ff6e33ebdff070b5abba6114441320874c61de24e89474efae708": ChaCha20Key(
        "1ab595d4576dc826743e0ac4af8e65ec9e3e9f5ee70106d5aacb649c91044a3e", "19a196c772ba23d153e7ef6b8b36473f"
    ),
    # FGT_VM64-v7.2.8.M-build1639-FORTINET
    "9d3451ca02d3827f552c9392a3409bfd4711f4c1ff5fb674294011831ba9e27f": ChaCha20Key(
        "306551582ec43b637a7bccfbf548565cd21222472c9d07f045dfec2cad07d309", "497fe3b47d26885c5101a2fd197eaa5e"
    ),
    # FGT_VM64_ALI-v7.2.8.M-build1639-FORTINET
    "3dca951b572b2c58a98eba0aa20607304e58b6b095a3d6767fb5017bb0dfea7c": ChaCha20Key(
        "4dbfc0bc5b05b1c3dbfb2a6800534cc64aebf74afc5cedbe6d3d1d76c6431cad", "8c373f714fba0c2a801bcf5f20ee45c0"
    ),
    # FGT_VM64_AWS-v7.2.8.M-build1639-FORTINET
    "bd75703f10697c652ab88ba038ed19fdd0d3c7aca2dc528af040cd5e47c12435": ChaCha20Key(
        "ca5d15e59e181d27df5f1f488d478d71d1e10c6843232538d460f27fd6de4e4a", "369d5f92831e7a81078ae009e4a84656"
    ),
    # FGT_VM64_AZURE-v7.2.8.M-build1639-FORTINET
    "522e878eaf40f1e890f0567229e237a4e57635a40ea5db7f625c6d222a0f9880": ChaCha20Key(
        "8aa05b7e5f7f4b9086c3cc3a55aaeb6cf711b07663e5513f9bae0e8d8d0b705c", "34efa60b0c87e9ac39eb205174327c21"
    ),
    # FGT_VM64_GCP-v7.2.8.M-build1639-FORTINET
    "7fa13de4f0c85920965595a1087615af09e7654252ab5d26dff086e87dd8887d": ChaCha20Key(
        "72f12167d8c92fe58b0078c9c410b20fdf516f574e877d3e3d0dcb821d3a5bfa", "8891ed820828d2087af35d9b4791d1ed"
    ),
    # FGT_VM64_HV-v7.2.8.M-build1639-FORTINET
    "af599bc90d0d7ffe1b63b3908b609e7ecf400e3cc67e23cc2fda37d5a52cab48": ChaCha20Key(
        "f4b04db757ffac09655c912b043a85df841d9b4408a70830be0c92d4102d8501", "bac7c6608315746aa235e3307482ecbf"
    ),
    # FGT_VM64_IBM-v7.2.8.M-build1639-FORTINET
    "ee4fafa83c972673d6cc69383bc27db96389fbe3f9856bcf1251c506601f89fd": ChaCha20Key(
        "17efd61f9d3e0e921f9ec246b6628bdfe6decd5ac3766f0f73c0c3f065c9bd52", "7d00d699e50d88314992dd7de4c9169d"
    ),
    # FGT_VM64_KVM-v7.2.8.M-build1639-FORTINET
    "a000468f1f5bc7320a6574c99d060f2e3cf391f6377bbddd6b86f593d9abdd24": ChaCha20Key(
        "88a38c08743bcccb2a68b4ac0f350f146ad94b00ba270add21cea47a1026e857", "9f00fef5b5fd9b5d3d31a5e6f5f38c9c"
    ),
    # FGT_VM64_OPC-v7.2.8.M-build1639-FORTINET
    "9cb0435a06ab1e8125c717845c1756812c694e1368841db8ed1a249dda0556ac": ChaCha20Key(
        "1fa00de817c3fca6ec5af4ebe84e7f1c54e57a14495a6fa7e4d7bff84b0e13a3", "f26a29893ea145c8a71ccfa70b78ee37"
    ),
    # FGT_VM64_RAXONDEMAND-v7.2.8.M-build1639-FORTINET
    "563e121e9ac01f718af057fc9b6ce5d4147d00fee5a94876f2795da4c3687b41": ChaCha20Key(
        "551c363d8c4ba29eeee194fd7f2025beba51ee962ba536297088288f134a5292", "45a17c0819b3761fcb7be0b5552ba109"
    ),
    # FGT_VM64_SVM-v7.2.8.M-build1639-FORTINET
    "0e8b48577a8debe385458dee4554f69c0163a28dd3299920fa1eed24ac59c7b5": ChaCha20Key(
        "b5d7cb57d7bd3d063e9e764ecf9c81c7f1799aec71a46e09a0fe5c582d4f75df", "63086ccd8daf2e3c23160e3af4f7f071"
    ),
    # FGT_VM64_VMX-v7.2.8.M-build1639-FORTINET
    "5276fe8b131eaab6b0d8e675c8e90d8838d3881deada31848f592eb2a76a43c3": ChaCha20Key(
        "9793c8778387df048b428d406c38472e89a64f389f3e95b4896a211dd523bd21", "60e8714760233655718367818b4a0862"
    ),
    # FGT_VM64_XEN-v7.2.8.M-build1639-FORTINET
    "5a484fe3db50085e7b5a1015fc6990887d7664da0beb2a14615f18afab68e472": ChaCha20Key(
        "6df05de09babcf98abfdf4603b3741bb5e43cab655127d385397b788152871ef", "5cf7abd1f9f555b108aebcbd0f442ab3"
    ),
    # FOS_VM64-v7.2.8.M-build1639-FORTINET
    "60db1a46fe4b6dc60cebe1e7aed0d238b0fac01e40582e1410a31cbcd4f73457": ChaCha20Key(
        "a745d1b99144c6af467d1a4559cd265c20a505b7d28d48787ab57acd48c32a57", "e42d9474010be7559dc690da8a122187"
    ),
    # FOS_VM64_HV-v7.2.8.M-build1639-FORTINET
    "e2878851cedf364f562535811cbc39ffd70b54d85c0cec2bf4d52ef97c233520": ChaCha20Key(
        "3d227f52bb2c033ebab703b8cac03d83b5beb23e3d266acc3cf2b7d91d7a55d9", "cbf894d299d066e3bfc4b1d048fdd473"
    ),
    # FOS_VM64_KVM-v7.2.8.M-build1639-FORTINET
    "636256a9b7bd1d8028f276ef77609a726953750b9aa85d345aa26e0fcefd3db7": ChaCha20Key(
        "1cdae2182c4b3211265e1bbb777495070bae25c0c1e71f0caa62dcbf85b4ec74", "62075a25412520083976a50b159a94c9"
    ),
    # FOS_VM64_XEN-v7.2.8.M-build1639-FORTINET
    "d63744a7c9e662a67a33cc3a252fba8404bc393ce66f3fdb6f2897785fa547b5": ChaCha20Key(
        "09938211dfe413bf77cd1e2884e3bd52d32ccdba33da9b8996c0569c1c649f9a", "f6f46c89048e691f1b1b671bdd530d5b"
    ),
    # FWF_40F-v7.2.8.M-build1639-FORTINET
    "695b59e4db67893ddbb33b308af3d2dc2697c73425e81aaf5061819fad6ef670": ChaCha20Key(
        "124347d126fb155e5342907e291fd1b9c4db16d4bfeccf08b109e3b1b6d8b035", "368927e9c1c06c726f479fac81f5ac46"
    ),
    # FWF_40F_3G4G-v7.2.8.M-build1639-FORTINET
    "db4533847e29b6b4fde34be0429c0dad4a9d82f93cc47d718d77b00a387962be": ChaCha20Key(
        "74ce1fdcde3c48899a9d05703c9bfa28042455c1b3a3af0e61896b293151ec33", "cd7700792dfcd0929c897d1cd8a29872"
    ),
    # FWF_60E-v7.2.8.M-build1639-FORTINET
    "1bf2d6b2368b63431070e21e6d8d4413b176b10487d1c3ef014442d07264435b": ChaCha20Key(
        "2a291f4cafb998c717cd7e99ac86ece2287ff56c143ddd650980c5b2e865d904", "ac6f1430633f68f0f258dde9eeda4044"
    ),
    # FWF_60E_DSL-v7.2.8.M-build1639-FORTINET
    "ae4fc1daf10d7255f62c7bfb948aade7701d24e7a0c90e139c45c4ffd2f5b739": ChaCha20Key(
        "d25af58ce2e1027d36c316af16e224be5deab1f3349c5280e9126647f6f3be36", "7c0fda7e55b5ff7d9dbb8b4a3d39abab"
    ),
    # FWF_60E_DSLJ-v7.2.8.M-build1639-FORTINET
    "ca47f2dc15257b6fd0ea14edf8869bf1399e4c14c88ceb891e93e9db9ae05196": ChaCha20Key(
        "4f4abfe532b65814540bca83c999ef47f41483c7ad5774d75ddb94ece13d171d", "a8ce7d35d48bab62249f8f9014b12826"
    ),
    # FWF_60F-v7.2.8.M-build1639-FORTINET
    "79ab1db62f85eb3ca5ea75a77ea52a8749be99d5860845863bc13026fe2a3962": ChaCha20Key(
        "4a1f3b40cb2ae6fca5be541872adbfb80ad64af32c598cfb7c22c71a415f1926", "67c7d122cb52baa5f47a90904c5c5888"
    ),
    # FWF_61E-v7.2.8.M-build1639-FORTINET
    "afc039bedf734c7cb6d0d3529808a97f203d1826efd4e52c6088a52dd48efc4b": ChaCha20Key(
        "b3891cc6c1a7a09feb66374a4ab2402e303d15754641393ea2fbb7ed756df0df", "1de7b498e97739a49ed01dd723d2f523"
    ),
    # FWF_61F-v7.2.8.M-build1639-FORTINET
    "6c365073db1a93357f04f1c9431c86aad55585593a701159af49dc86c0e3eb85": ChaCha20Key(
        "4904b4c6cbfd21134b9b7a113089d99eba19151a8f350be22b442d0062e2f16a", "c609688334031eee9bea4b89d7ea82b2"
    ),
    # FWF_80F_2R-v7.2.8.M-build1639-FORTINET
    "d9da17398d621abeaee5f890a83f904301c98f5db4c7f48a176948e5e0e9ea0c": ChaCha20Key(
        "9efb58a7a2e73b511d9f03b9c10c5046e1a366fe3e6944d59d56ca05967af7fc", "e5172c3540e41c859e1d4305bd169785"
    ),
    # FWF_81F_2R-v7.2.8.M-build1639-FORTINET
    "ad88f19465669176912911e6351940fac6e9c051bf1b7769152bd995860ba959": ChaCha20Key(
        "c136c75215a8dc2d0688ed4a7acbe6970e90a6874006c5389bda498a992a9d18", "7f9ec26641f34aa634dbb308d4d9c65f"
    ),
    # FWF_81F_2R_3G4G_POE-v7.2.8.M-build1639-FORTINET
    "7db83d5836ff5fb91e577e16770d65ddf914fe5d5f26c25b32609de514084f13": ChaCha20Key(
        "a598ba3513305e627bd6009e22aab209e6a8d3a844b7bc6170914ff9a59c0b9e", "0db5a51a90be5ca97fd48cf1a6b20393"
    ),
    # FWF_81F_2R_POE-v7.2.8.M-build1639-FORTINET
    "175413f16d108e3d16fa7a3fffdcb9fa578d74e3d85a3fa4829a7fb88182fc0a": ChaCha20Key(
        "58525be1db5e925cbfd661594c572a942819ad91b9f2bdd85f03a7d763037db2", "8494b1253c0c1c9c0fe96d9e3c6767bb"
    ),
    # FGT_2601F-v7.2.9.M-build1688-FORTINET
    "5de79b475fa497775e6110299dd1927c5ae16c73e23c6b64331f3ff1112cf4c8": AesKey(
        "1864eb1372f5871672bc15f956783beb2d447c33f6744c694e166f3c18b23046", "a875c2c0b0302b2861a4ea01b6021bb1"
    ),
    # FFW_1801F-v7.2.9.M-build1688-FORTINET
    "7e1652daae27f59387021c40deec6a62eeaf4a5a76c8cf689c2808540395805a": AesKey(
        "e15bcd1c2c00de6e79e8f6672a74ff50ec50a8482e87caa07cd4f2947f3fe505", "0186ad6c88474fced27e41b9365372e3"
    ),
    # FFW_2600F-v7.2.9.M-build1688-FORTINET
    "b62e5e128b661e57618240b3c9ff9e110658d17654d57031313caa89fae823b9": AesKey(
        "a30650546e14e6327fcb8110a0f305a9b9bd4d5e9405f1a0e0697976f4c071c6", "76adf565d9cdeb490ac0fc710bdfe2e5"
    ),
    # FFW_3980E-v7.2.9.M-build1688-FORTINET
    "b4439e7ccfb010e44bda6975976237ac7d84cee0bf37cd93868a04a949e6c500": AesKey(
        "eb43c7cced3590898245fc8470d68ff85c932c9f05ea66cb34b6260deae51884", "6e05dbd7a5616277daccbedb3890780f"
    ),
    # FFW_4200F-v7.2.9.M-build1688-FORTINET
    "f4bc37416a8688dbf155ffa4907538a128eab591ba72cf9d2c4b4905da82eff9": AesKey(
        "412b30d8a096caf0a04faf89ea7545d47aefade9f6ae7272d192ea14f4b2b003", "a69be69b760cc7acc439b513a48f4dd4"
    ),
    # FFW_4400F-v7.2.9.M-build1688-FORTINET
    "4bd642d85ed4f553fb72cd751846ea38f66907acb9d22bdc4088c013a4a65cc0": AesKey(
        "b051c197b9d3fc545e6a86a65830ad21d62d054177702ab0490709334b48af14", "bb381818beeb204a13a0527969a44d86"
    ),
    # FFW_4401F-v7.2.9.M-build1688-FORTINET
    "10e6298f8b02334a3c93692a14586fc6745efd58a81574fcfa71b67494dcc224": AesKey(
        "1151d2bfa89611f7b01315379731aca18dc19ad069b282ff75e112ee1feeea38", "841324d1411a18c9ee5b2eee84a0440a"
    ),
    # FFW_4801F-v7.2.9.M-build1688-FORTINET
    "b0ac03719e3619ac2a271a1426872c5643c7eb0cf1b091df62f5df6b3aadf08b": AesKey(
        "5df775e30d3c595a1f31dd0086a483be4cf77ee7b8d1d40861af635e07ab2232", "7bfbe451591afff37d09fb82a5f532f8"
    ),
    # FFW_VM64-v7.2.9.M-build1688-FORTINET
    "6863d5b226eac92f754e6242ef0c08a92f97dbae925b140c1e9a034f75dc6361": AesKey(
        "2c3229b3a1d62e9ca10d22aca59568f1e0f07bf2705770a9c45d01bd30c6e199", "0d6391fa1dad0dfea64b5971ddc92b49"
    ),
    # FFW_VM64_KVM-v7.2.9.M-build1688-FORTINET
    "2626c0d18039b8f24772a0b34b9581b796ebeaa7f055bf0a96dbb4733b7db490": AesKey(
        "9d3061ccde36bbff2d9137bed1ae353ed75492b6d4ab2ff6d7bdac54d1bc840c", "b7b89e33c7c17a4848fc4d7257b28756"
    ),
    # FGR_60F-v7.2.9.M-build1688-FORTINET
    "e72c4cbca232d0b4246a3e6464a9aac20c2105eaa179d3f8185ea7e5d98a46a8": AesKey(
        "ade907ecfbd34905f19e5b65c632b3e1412e971824a30346d579138787a92da5", "6519fecadbb60073d4f775c1083ea7da"
    ),
    # FGR_60F_3G4G-v7.2.9.M-build1688-FORTINET
    "7442369a6a51c04020641c379d37230f60ed17ec3cee938eb443937226ef3930": AesKey(
        "c2e956f5ea2ff422af3cd4cced1106029e1b4f865fd168bf51b4bb9034e425c6", "e3644ba0e88ece40ad873d43db560044"
    ),
    # FGR_70F-v7.2.9.M-build1688-FORTINET
    "eaa5042dc51a6b5eee9d39c2e4a20380959053980bb633cf290dd0bb3d0502ee": AesKey(
        "20c6cd1176298c59c179d988c184cd2dd3d9566ef07da6859bb2bc6e097ad3a8", "9bca6618634623c370cd0a1ef2364657"
    ),
    # FGR_70F_3G4G-v7.2.9.M-build1688-FORTINET
    "f07f100cbfd2185deabca1c7b0acaf1dccb13ab4e7d8aa1e0321ef93fe61c09a": AesKey(
        "a0b2809ec0c822fda1043776df8dff83b6eb7d8acf437957345ef999bf3d3d58", "cadf0f0034ab2034f2544139f5d20291"
    ),
    # FGT_1000D-v7.2.9.M-build1688-FORTINET
    "408bbf3c6a4e665a85111257ff8e3a5ef903b2c9c8aa6df5bac472647fa28f4f": AesKey(
        "64f190f7a48a724b1431a16c02418245793780a72b5eb2225e3d656bf4424021", "f6e0a0257d4f51643d75b86810564356"
    ),
    # FGT_1000F-v7.2.9.M-build1688-FORTINET
    "8407fa3bc81117aa8fabe7d5d7b5875a47681e51fb54f1669b54cd09decdb5ac": AesKey(
        "df37a9384caa9a3b998b72afd104164440d4049350e1896b1caf4fb413f6c790", "fe5e5f9c29d1ec7bbf2eda62aaa6bc87"
    ),
    # FGT_1001F-v7.2.9.M-build1688-FORTINET
    "de615d2d29e85220b1ee93d79b71c363b88fdb6fd70ca18cc71c0b267e29baee": AesKey(
        "ce5af35945e2b7193462d70d977f4868baf0f56002a85c23adf093f903bede34", "fa068605634616297062ba4487c39fca"
    ),
    # FGT_100E-v7.2.9.M-build1688-FORTINET
    "4baef879d97cd9e4c4b166da583ab0ca1095389ba85e5516619254cd6a32dc54": AesKey(
        "7dc93301979432e2b875fc30d55a248e341fe20bc0f69acaf24cee34e8f58884", "f84ceadf9e3bb749a70bd61b4626c0b2"
    ),
    # FGT_100EF-v7.2.9.M-build1688-FORTINET
    "e69df2840aa3d99a8ad06d918d48529218b1f0fe720d89fb6b179d6ed61067b3": AesKey(
        "b4651ca2a40efa482900781b89e535ffdf0ebc3d947a64a89338a9b1bc0dd227", "6111fe5a391e5f32587f9e43fa3c8583"
    ),
    # FGT_100F-v7.2.9.M-build1688-FORTINET
    "96f9663415f5629de6227a2d71ac7d95bf1a2a7b2b27505779e54c377e3be472": AesKey(
        "8a36bfc0e95fd58b3fa6fe0979a66cd69cbb8104861cf561376f6266aa8a97db", "e7596b996738857c62a32332ed4dcbea"
    ),
    # FGT_101E-v7.2.9.M-build1688-FORTINET
    "6c62b506eb5a98dfbe854b2acbc5aad877d9570b6a9c7ece193b5e9fb670e0ad": AesKey(
        "ce89ebffcf5dc81906577be9cc12ba722592a0fa4bb5660bc5821baa3c4e46f1", "f610e1e68716b144086b1b92e0a5c0f2"
    ),
    # FGT_101F-v7.2.9.M-build1688-FORTINET
    "1a1d2acace9aeecf5a5540c30c216219d1b578263908d34d84ba8fe2d0714eeb": AesKey(
        "05f9027f8df9efcc851241eb6cfedc602386b65a81cde5d973800ba4cd2f1eb9", "1e45857b4c263e3208fe56ab2f1b3b59"
    ),
    # FGT_1100E-v7.2.9.M-build1688-FORTINET
    "813063118f1a19433e30ba773d058ab93f1605ad558251e08ac83c9b30ff3ec6": AesKey(
        "5e2ea22add96461173094ffb715453009fd0dd9b8506afa691ae5b61a892d1f0", "9b6535a0f51847c6204b9d59cc14dd3d"
    ),
    # FGT_1101E-v7.2.9.M-build1688-FORTINET
    "6e5884371e9afbf5da7e9d3e9625c790d4e53fc3b14e4f4b3712a40656d0441c": AesKey(
        "ccef376484973a285a32a976bd95e306766d2be966462d73007469a8876d4527", "e21d5ade373f38332c4954833c17efc2"
    ),
    # FGT_120G-v7.2.9.M-build1688-FORTINET
    "b7c96f1ca9898be23cf0b54e48467cf976de73c5ee873cdd6620d72039f247b2": AesKey(
        "72a9ef10c77650a7e49022e2d769a61c88e522f7544db4bc2d9fb1a168682a0a", "06d31f2f75c09036110d2f1f8e8f813c"
    ),
    # FGT_121G-v7.2.9.M-build1688-FORTINET
    "ce5155797322a1c0e588034f32319dae09d877805a5a9b064199ba1738c0b3f9": AesKey(
        "91831c76f2f4d334d46bc824f690dc72ac42761dec7f13d0f8fca3a22f86b39c", "bd9ae16ad6bf66def2434b72f5fa4a65"
    ),
    # FGT_140E-v7.2.9.M-build1688-FORTINET
    "962b9e59ce04036c7a42cd5adf34590dcb6716ee50425d218901868a1516bc64": AesKey(
        "10589a095dc8f57fcf1fababdcad1a8a6db34d0c635bc4b0765aa9024d1e2471", "ac4e118eb5a8cfe7b8a5905b2a0e337b"
    ),
    # FGT_140E_POE-v7.2.9.M-build1688-FORTINET
    "77742c925a8eda8c4cd2029623cd9ee75225e50055b955d0749a67d52ef73fa3": AesKey(
        "a365390160eede123e8ae588c2275c9803a64a7d4ff9c06289c7a524047b8a64", "2c877f8915ccbfc1442189af63dc454d"
    ),
    # FGT_1500D-v7.2.9.M-build1688-FORTINET
    "cc36b26998ccc4ac4bbc813d93d4bfa56398bc8644eef6ca5b5a5618eff65a39": AesKey(
        "bdaad1bb9e4b2dcd0f01b3d5d28889a6b54020b697a9ad4b7829b2a114cf4b66", "4125c3a71f258c06a6d32eeac7e2a2c5"
    ),
    # FGT_1500DT-v7.2.9.M-build1688-FORTINET
    "5e60458cc241f890ac7c801b4f798d4677141e93818155469ca5aff959e24c21": AesKey(
        "6ded4f6b342b84f95d2644c8da9efb3c30e86d4929c0ceca18723b071202fba1", "6d0fdd74216952bbcaebd0b9af072895"
    ),
    # FGT_1800F-v7.2.9.M-build1688-FORTINET
    "c5ad2b56398f103ab6bd8e00345e4334e65bacf36880f74f625226a139f7e15a": AesKey(
        "eb72b3bcddcfa2c7350ad234401ec78cc3496390bfcbf3203212a6777331af77", "d34d830c86dc3a636a0634d0172d01de"
    ),
    # FGT_1801F-v7.2.9.M-build1688-FORTINET
    "49e822d48d8b1f9befcdbc8b8f98ef7b8524794e1cf0431cf8da5f0bc6a557b8": AesKey(
        "1a06c736d386ea969030ab6b7f86150a58ed19ff922a372ff1f6afdfcca0bbf8", "7a97ba9fb6a34d1d678fec2bb365e2df"
    ),
    # FGT_2000E-v7.2.9.M-build1688-FORTINET
    "d2529ce3112053b8b8166aaaa352f81d0ef11bbd0834549247677ccd7d218b5a": AesKey(
        "b394b45d49824fad178ebc36a4ab30758390e4d3f9784b4cef46ff06cbf20225", "00f9877744e6f1d0593af692de706e28"
    ),
    # FGT_200E-v7.2.9.M-build1688-FORTINET
    "5f76130726febe2534766501177bfd2fe4cfb4b9c76221bcb6d88f1388ce2f93": AesKey(
        "ea46026a61cf1002d84b8a6784647ea42f2900ce694a24efff157848a1a99515", "cc968847a2abe0cdc9bab056cd756aee"
    ),
    # FGT_200F-v7.2.9.M-build1688-FORTINET
    "194fe47672a9bbcb4af1a40f64ea98d2e53e679d0c0eae1f114e1da8840b674a": AesKey(
        "a392e9ced94f38dc1be4a51c9b0add60a72615d7de7eeb988ed26b3fff442077", "78ed13b6bcdf0aa40de25cd215639256"
    ),
    # FGT_201E-v7.2.9.M-build1688-FORTINET
    "6a708788aec890cbce6c649160e88953562ab321e53d3193f2a8b229fe1b176d": AesKey(
        "f9d9d78facbe8efb6f50a209f2b3ee27f11511d34de051ca822967c436d18bb3", "b7e7166ce71939f9de80b164524ecbd6"
    ),
    # FGT_201F-v7.2.9.M-build1688-FORTINET
    "af24b98cb563d0a8d863b9f88e71db43249b455ef0d65ba748978601821d0c53": AesKey(
        "d1bbb6cf78ce03b899d44a78780d29d8611e0fb451963ef25d424ce933b72f5b", "0d63d6ff0c94fe7ef45e243d6518db5b"
    ),
    # FGT_2200E-v7.2.9.M-build1688-FORTINET
    "d048502b00997b44e86f8f941de5545b14172c29ccab5b937cceacd01cbcd63c": AesKey(
        "974a4ee33b94db70dd41176d388daea5813699d3d67c31dc07f69a927623eff6", "d7176549a176084c8581ee7df6a6e8c4"
    ),
    # FGT_2201E-v7.2.9.M-build1688-FORTINET
    "4cb3b750b6368f6a2fb4edbc5496208cacad983aae5a5705b0b825a460b3af6e": AesKey(
        "55847b08cdea1aa1b7cface796c3c773ef418a2a118744bde1f2fed9b749e207", "fc8788bb5f45c4e94574dc09283ca035"
    ),
    # FGT_2500E-v7.2.9.M-build1688-FORTINET
    "5c9140e2d50f4c067df5678cbabf654e9c926586e537e0e2b50ddca945fb6019": AesKey(
        "58c5f91ebf1f5ff8f3638b0238ff3059ec6eccefbf554808d490b9821b401486", "2b0448156350bb474a9e93cccfa433ae"
    ),
    # FGT_2600F-v7.2.9.M-build1688-FORTINET
    "8f22be97fe58c09c981dc53c5858e8797590d227d66ef5da1e6db6292ad5b19b": AesKey(
        "8d7593920480405a750f54a6df7a0454f66eb66df56c887db373c21ab7a8bc92", "26d140cef5edd5d17d89f130e2c47fb1"
    ),
    # FGT_3000D-v7.2.9.M-build1688-FORTINET
    "191143f7ede94a1fa2c8273bba8ef356dc23c1640dee7d36d18f32e7024bf468": AesKey(
        "5952b46fded7b93ae21cba4a195fe07c800865f5fc2d06a0906d0728b849a93c", "2c9741d4bd574ed874103c10e9f8f0ab"
    ),
    # FGT_3000F-v7.2.9.M-build1688-FORTINET
    "0aa1a837c93d4c5ddc613c27821a0bcd412600a35f51c5111446ba007507a874": AesKey(
        "3ef2016498cb17c90d1c12f1aba808f5196449e3f550d5f4385271f18c1184b9", "529760d57d292cac2695eaff53f36f68"
    ),
    # FGT_3001F-v7.2.9.M-build1688-FORTINET
    "6085cefafe8bf735e8fa6e95d8d5628818ec4d42b2d2b13004cc22c31a7b44f9": AesKey(
        "5c27327b3d99f00b818c2862a46156f18d9cd47202b2a615ff84c9d0f117972c", "727341fb0f1f2ca2fd3f17a764195260"
    ),
    # FGT_300E-v7.2.9.M-build1688-FORTINET
    "9e87583bfa8636904ddeedabd46b619b59c9a44229b620d963f971741b3e9b7c": AesKey(
        "7dd747558904a603de4124d3b6386f613ceee299ac6384b7f7f769e12b1b7bda", "16ca223c54cf792c0ece7bb52ec27031"
    ),
    # FGT_301E-v7.2.9.M-build1688-FORTINET
    "3b0fa28455f0c4d36f98a0f19aa7ae3838dc7192a8fd5d9f4e63f0fe3fdd3116": AesKey(
        "ba3b3ce16c8e1c7323c691d128eda803ad4d6de5e9e4c50318dae27cfa9a8dec", "90f8522948dcac7e6d108a5214194735"
    ),
    # FGT_3100D-v7.2.9.M-build1688-FORTINET
    "724f40d7c2d173a60bb7b3f13a00596bb8084bcd58611c95c03958fd1201d09e": AesKey(
        "b0e8808deeaad2977ab7fa6b8bab710c0dea8d5e8c3c468b9692f7192de3c20a", "8f38ab63fc608a82903e3c932336e017"
    ),
    # FGT_3200D-v7.2.9.M-build1688-FORTINET
    "35a744e375a2198116da26408f72f7b21193ffb9e0546a7b41d9f3a0bcc77efd": AesKey(
        "15e67db81b8e801d11ae8b70666a37d800ffa62082346a176f2f4e4e8d40486e", "a6fb23cc157e899fcdecd1c1018d872a"
    ),
    # FGT_3200F-v7.2.9.M-build1688-FORTINET
    "81ed79dd4e3ad18e5fbe761ad885fef70f603aec9c4ee454f33444cf0fe9d981": AesKey(
        "fb7ca1fa0dcef7d25f682f58f1f5cd0733bdf76cfe8dcb12cc47f4c522129fb6", "6df004ea3b8ebc25b7935bc5459561a0"
    ),
    # FGT_3201F-v7.2.9.M-build1688-FORTINET
    "644417614a6edbdd525971bf0947321f71275c52807019811c6d0d3c5a3603cb": AesKey(
        "b7c9f80e9524bca274d8e33a9ec72a727dd3ff8cd1245147c6a6822ed09144c3", "0e1ae35a0ad8d4a0fb816351cc169d5a"
    ),
    # FGT_3300E-v7.2.9.M-build1688-FORTINET
    "ade3e2f8cf1df79bb47ac189adf4650d912e9b3df4ac9e2bb1f0ef2f8bc10b45": AesKey(
        "82b4d62ce2c161cefce39f5d3ae8c232de89c85524ff13a6b5012fffbcb22350", "464aba38251d47eebcd1e8bd9702f7eb"
    ),
    # FGT_3301E-v7.2.9.M-build1688-FORTINET
    "9044043c8394beac7dfcb176111fae537075fee69744b949b7c99df0fa77b619": AesKey(
        "c9ffffc717468b8d83bb932dba1ff601a9a41e9332b01761e012927bb89cb0e1", "171d9e165bef3572245cf79d1625440f"
    ),
    # FGT_3400E-v7.2.9.M-build1688-FORTINET
    "7a323494e2e1e0db20db9cf3e2b8f296fd47dd0f202e56b5c2293f476e7dbbba": AesKey(
        "1231458d91711272ebd02b85d0ac3709d7e6e7bd1cfc32610bfc46f4641165e9", "dfa98d4d2c789d11d0fcc6c07c15185d"
    ),
    # FGT_3401E-v7.2.9.M-build1688-FORTINET
    "2d71b2a5508332e985ae0cf9899aed2e91a45b0c6f4259b01b905126bd403325": AesKey(
        "20ad393500c4b15776f06f6ddfc864c10e5eec5e7dba7b22a136200b6dc3a50f", "ca6ee5ed4dda92f86201d245eefba451"
    ),
    # FGT_3500F-v7.2.9.M-build1688-FORTINET
    "39d8911683608f2e48d472108e63538e77062208ecc48bf1f182115d8ade6d8c": AesKey(
        "2c405b9fe8a04483ca341393ccb9bd38a75b328e347ccfb0c588e10381648603", "ba0859f678f65449a3e1d4f372a44717"
    ),
    # FGT_3501F-v7.2.9.M-build1688-FORTINET
    "fa334969a72280e7c8e3072bfe65716999fb34b21ab17e6d6fa89340d252184d": AesKey(
        "9d4493d43fd1afb87d4aae6dc5cb72ab333b0b26bd366dce7cdfcb42cfe6cc16", "8a3aa9edc0105743050d7db23757aaa0"
    ),
    # FGT_3600E-v7.2.9.M-build1688-FORTINET
    "005d739a1e5ec6407fa99571fe1cb874ec32c562dcf298c126a78c04a3adebd7": AesKey(
        "8d621391937f4a052703af05e4cc72669f8d9281941b2546307e93b4f4695403", "17fdd93136c720c5764ebae9e062bdc3"
    ),
    # FGT_3601E-v7.2.9.M-build1688-FORTINET
    "2690414d0ecf62fe716ea5f98b8ed9b9713bf01eb0ac0153c270b9d3cfb1965e": AesKey(
        "c2c7f57627fa0d3ee7a7933a297965e0dce5f9c9d0001d9d4ec1cb1994cf1bf2", "2731d78c6ae93437abbec53fe528b761"
    ),
    # FGT_3700D-v7.2.9.M-build1688-FORTINET
    "49a147ae2708f31aac6fc42e393e307e8249485a88aafb5125821f2cb1f5fe14": AesKey(
        "5f74d3e0cf52d67df5d0d477d4c162903d9fca634c742e32e555d8fcfa8e71e3", "306cc8e4276fec92a4d6f0499d584930"
    ),
    # FGT_3700F-v7.2.9.M-build1688-FORTINET
    "5d5329276862cdfd14c21e0fc89ad4b4894205b8ece5d3eeffe2c685337328e0": AesKey(
        "3b0089894b2ceee2856ee6740b82ab8a11a1acda51bc8686fbe87a3c1258937e", "bc13d9b401f6a2aa97b400f939aadf0c"
    ),
    # FGT_3701F-v7.2.9.M-build1688-FORTINET
    "ff60f1e38f8a21a209aabcc53c46a8b2feba7c63b80819283c04c8ea6a3d232c": AesKey(
        "2e2577e942f3ac9a5fb1c25db742ca679a32178faf22e2522efb8371bdf2f374", "2156645103aec563d9ca58c492e05661"
    ),
    # FGT_3960E-v7.2.9.M-build1688-FORTINET
    "a160def698e0e0d567751d009a9dea6abb40364dbd61ae8f776235161d14537e": AesKey(
        "362a0bd6f1a4a9e590e63120db38768223da5709aca5e590bed6fd0933e6d3c2", "a5bead1de2842a5ee29f7da0e3f1327c"
    ),
    # FGT_3980E-v7.2.9.M-build1688-FORTINET
    "cddf759e2e4778f2c15a52ec434e284a57972e948c1041d97ea3c9cb1f0ff0f2": AesKey(
        "1ef047a7b74e6b5d112561b961461eb6c8ca4a2e6e5b4dbb97167578eaa24a1c", "484dd698f8f2c9e3096b686bce73082e"
    ),
    # FGT_400E-v7.2.9.M-build1688-FORTINET
    "d3ab3e8985e796f526f1ae9f080ecce17ae429039b438aea45a260ce2d83c617": AesKey(
        "3f838ba8d211c3b34cc8162af7a6d81bf8a04bf43cc168915f297f6e372af385", "53ee05e4fa66b11d6720a8bbec42e73b"
    ),
    # FGT_400E_BP-v7.2.9.M-build1688-FORTINET
    "d44361646afa1faf482c2531f8f7af01352e7be10572b82ffaee100c61dd39a5": AesKey(
        "4e8cf8123d983549598a50aadb31124993d9e2559f4d7fe8d3d0693b5769f308", "3437e0527f55ddeccb6b26ffd4a15870"
    ),
    # FGT_400F-v7.2.9.M-build1688-FORTINET
    "fb389da67fe453f3a4a32026a4231e8a516e31c09b7844b7a26e9319f2109aee": AesKey(
        "bfc4423f19449227bdf04aa4aad2b0f8682a7b30499a75d34cf5e4924a2691bd", "ad51be5936002902f897d2b1bbde3ab1"
    ),
    # FGT_401E-v7.2.9.M-build1688-FORTINET
    "9361f8eac1b7963cc2fcc9d0f71068475f99ea02cf6698da9c19d1398a251e07": AesKey(
        "9b245b21df5e6ce30624743444e3e37fa2ee78aa3fef97094068ae929f1b4bce", "b9e844b29cf139d6a2865fc8f9be1210"
    ),
    # FGT_401F-v7.2.9.M-build1688-FORTINET
    "15db4c33ba1de42ddb86a7caec368331f70b1ff2009e7e280ff7df50ec73ec06": AesKey(
        "b91e5f20155cf0a06801a3221a8fd549f7848b7232e55b3619ba844236c2fcb4", "057158b4ae1350e34b19f6b4782aca7d"
    ),
    # FGT_40F-v7.2.9.M-build1688-FORTINET
    "c72ca0706bdefdad42a6746f824c5a223b9762e2c3951ec628d9127095e98728": AesKey(
        "c8b91760abf12578907b059824a6ca3cbeb9f61461b14d335e61917a1a656496", "4c17f57ba1d75e6fd9de8c1f36c64c97"
    ),
    # FGT_40F_3G4G-v7.2.9.M-build1688-FORTINET
    "4aa04fa3e557dcf341c4d0a36323521ec3b681a8743186ee853e42951e654253": AesKey(
        "ed2a6dbf986551d17f5ff4730a95e0858fba9dbcf5b1f702922828626f8c31a9", "2ca0a894d2fdb7f4cd77a55242a9e533"
    ),
    # FGT_4200F-v7.2.9.M-build1688-FORTINET
    "8fd9d9512e589c72529300570b5aeb3750eefc5e0849b8641db5bdb862045959": AesKey(
        "1edf5c3b05b1eeb4d7ab7757fbd23a0c3c6f285d1fa3f34288ecd59fbb8e4d2d", "67fed783184ef1975627b1210d9c5df8"
    ),
    # FGT_4201F-v7.2.9.M-build1688-FORTINET
    "91ef79eb65b55a300a2f6a4b3dea5fd92299f15311782c4ee0fbf22e2dc7a0c8": AesKey(
        "053c841bf9f689d64aac9031097cf269afaf9a12d420e66eef0f2c40c895a7f2", "60460da140f479fbf4040e16c89190fc"
    ),
    # FGT_4400F-v7.2.9.M-build1688-FORTINET
    "3bb4c912147e547d6e5a1c133203a95c917bcebbe1bbbcf0ee8f9874aa45fca8": AesKey(
        "fa564e64e42e55c4057e7d8c78295a586e61480b817281d593717ed45bff15d8", "8a06839a72fea12a93f5417ec9fdd7c7"
    ),
    # FGT_4401F-v7.2.9.M-build1688-FORTINET
    "b60ef2267c0ac6fab79b361e5f832fd4b301a26508200740309cbde1eaf34483": AesKey(
        "86efd153b0341648852876cd0910cf6dfeb64928c5ee9b6bf28c882aedba63d5", "5c6306b5c6672140e5a165e54ea4088e"
    ),
    # FGT_4800F-v7.2.9.M-build1688-FORTINET
    "267dd21565befb55acb3dccee64bc9e127442735f77a7c889f06e908eb618443": AesKey(
        "7f688ff8644e8dff53496bb3d354e014cc9c9abce47bd5c6fd8bafec3031ff5d", "2af29f845c8d71786106a7b4adb994ef"
    ),
    # FGT_4801F-v7.2.9.M-build1688-FORTINET
    "e60ad709d4d1df75ef06abb8927f1e20a98512d506b56073e3720de28511d61b": AesKey(
        "49e4f9e666c5a716260496294f8ce581db673716e43e50b17bbc064c9e05affa", "cd68466dd23409fa535e57c3fe93a445"
    ),
    # FGT_5001E-v7.2.9.M-build1688-FORTINET
    "2db612e3d2f550c2fff30ec93324dbd98b22d2993e4e6feb5519489df3ececd5": AesKey(
        "ca51d5168d871c4f4ed1073078aa26d4ae325d1612505051e4f2e760266b4f3b", "467d43d98ebf4b618e10b5e8d0648988"
    ),
    # FGT_5001E1-v7.2.9.M-build1688-FORTINET
    "7a8a59bcd95929dd4bd6b6145e00d9c1c41ed709c1714f5e1842b070c81cdbbd": AesKey(
        "d6acdf199d86d2e4ab337e26bf82aab84a8e127590465cb43571b5b900e11581", "7977bda36bf099cb40463e010e93e95c"
    ),
    # FGT_500E-v7.2.9.M-build1688-FORTINET
    "625fecc798732b2348af0181e539c1049c62e9197b358fd4768833b87cf9bfcc": AesKey(
        "bf0ca970d142aa3861f64fa09a3066974bfda585795cd264a600d5175b2ce335", "d241430d74208f885d1f2bf4f9d9c932"
    ),
    # FGT_501E-v7.2.9.M-build1688-FORTINET
    "c057c6039ac7a6d2bf90e7f0b9ddef28bcdc964d86d46f01bff4ff6ccbc9808a": AesKey(
        "4b603f47a08fb14e24df9ba3bd503fd377b402d1c9cd9b9b5568f2128d60db6a", "f51abaa74c390c6d72d346a8dfeb5c98"
    ),
    # FGT_6000F-v7.2.9.M-build1688-FORTINET
    "052b62c864cd910996635c28750728d109ef2f49ca2fdc84046a008615fd0b49": AesKey(
        "2634b9450a05e4d3aca28e4a977657d2ad8041e528b05af183beee374ba3f413", "4c67ace9d62a1759e9ffd34cfd6773e5"
    ),
    # FGT_600E-v7.2.9.M-build1688-FORTINET
    "71f6c0c8c84e67fc658545a109cc660d810e23db8759d98d6a4b3257f80ad3e8": AesKey(
        "4fd32d21c4a4f8a0fe0657c597acaa1824b112d23c66ea01335f90d97618b1af", "9d4989f0ab7fd83734dd6d923113300e"
    ),
    # FGT_600F-v7.2.9.M-build1688-FORTINET
    "1de0071d80fe72eb4fdf9462f00912bd9608b2f6348df422ede0ef422b438150": AesKey(
        "b7581de8fbb187812919b5bf18213ce6f985c93bee44039cc59b709f141edfc6", "9918e250d7b4c58a6a504af77d31c919"
    ),
    # FGT_601E-v7.2.9.M-build1688-FORTINET
    "56ace740840ded10e7ff089f83e6ec33ade7ccf9e30d9c8169334dc7ab62d8c9": AesKey(
        "c99661c08a6996969b0bbbeab1f093241098858bef17048ac088942e209674dc", "09be8ab60209eb8f4ca01f9b1689ada4"
    ),
    # FGT_601F-v7.2.9.M-build1688-FORTINET
    "9b3ace40af30dea74e69d0827f223d8f7959558369360ce53a1e4145eb9c1e2b": AesKey(
        "b31baa84fd3e1929b801bda7319315854f45fb309f9c3599e26c16abbd840eb3", "5c72dc5513b6db2b4fc533efc0240e5d"
    ),
    # FGT_60E-v7.2.9.M-build1688-FORTINET
    "8d72d5ec578433daa44e874487babcac098854f02c2908b15ea9d38e526a39e6": AesKey(
        "872b530b8705639baa69c802daecc08536c3242d385a544a98baf5101c53e762", "58a15d3b55a0c92765a6b8f8d3fd8cae"
    ),
    # FGT_60E_DSL-v7.2.9.M-build1688-FORTINET
    "e0244b5496ee508e68d80a022cdfdd0524d10903569f49d191a7ab495bcacfa6": AesKey(
        "e4537ad8c75ed5cf453ee79ba2668b948f2086982c6976917ad75bfcf24056ab", "20e74ed0f8502ac6b1e42bbb4872af6a"
    ),
    # FGT_60E_DSLJ-v7.2.9.M-build1688-FORTINET
    "2cd00328d8202a8f0edc5307a41a7bdf5e1187f4c5be17235427cf6700b12cc9": AesKey(
        "cd0f4061b805aefe6b42e801195f19c0f5cb92dd674175652372af2c70b38b64", "9fd163bd1508384f26158a4e6d43786c"
    ),
    # FGT_60E_POE-v7.2.9.M-build1688-FORTINET
    "cd04d80211b1d8bd11b77777ff70c4a3f28447ede378a4c4be52c8ae603fee60": AesKey(
        "fff70d2505d9f07f42fb1f17dd7b92d7a3b58c721d017d7b68291652c507db6e", "59550763c770199e12a6dab40ab2877d"
    ),
    # FGT_60F-v7.2.9.M-build1688-FORTINET
    "7e22ee467c815e32a6f9163c3bab298d57d42459fad8c522e1dacbd964f8419f": AesKey(
        "fba3d41a4c19939739418f2e8c8cb9d823afa7345b4653a2c1133b965fc47aad", "896b199c664fbd776830e794626fa775"
    ),
    # FGT_61E-v7.2.9.M-build1688-FORTINET
    "32163ce2e6b097011f40d7701662ab1258d4986c07321ed7e2608c7119b448d4": AesKey(
        "35c37e61082c2b117a9181e4a1323dc5e175511ea0df4a5a51a90ab2594575b9", "c61fbd10eb1dbcbd9a0e40525180b7b2"
    ),
    # FGT_61F-v7.2.9.M-build1688-FORTINET
    "b0a9e0d432909e7dbd07237ceea32bdc8523c68bcdd7e41882909fe4b502042a": AesKey(
        "8cafe8ccb15fb5e40cb4c75c96c1557ec58e3d22b7ffd96fe8b5f09d1326edba", "8300d9bfa7b5eef22acd4095c7df0f86"
    ),
    # FGT_7000E-v7.2.9.M-build1688-FORTINET
    "c6cd33ea01f9d7066f22a8581069e3826d9ab98e9da004c5686ee8fe754fd1ee": AesKey(
        "d545e85c9b9f5b165213df64ecba24b1e3395238c2610e0e1b8a967403ab4873", "45a20966fa92d77b35092aaa99829841"
    ),
    # FGT_7000F-v7.2.9.M-build1688-FORTINET
    "f67067f469bfde13bd8f1431465fe1f55beab62334a2a9bf8c6018d94937f116": AesKey(
        "94882315e039af8a638fc6a39195e24cb077310b146f67248055324e0f8e5364", "30f550033d9c8fc0abda334b7f735dec"
    ),
    # FGT_70F-v7.2.9.M-build1688-FORTINET
    "15592b9855d8f1c8c863f1de586258cec790298275a240d7c47b7f5178165965": AesKey(
        "50ba68e30880e64291e72b5c7de8fc61056ecb9ca052e7bed400809c903212f7", "d2d4df1db992bf5b8b11174f34882518"
    ),
    # FGT_71F-v7.2.9.M-build1688-FORTINET
    "71d62f7e146e5cbb3ff66d7f8f48831b8ee505468788f7252702a0fa3176039f": AesKey(
        "4919ddfb28a8ac8a4ae9f285356b55743ebf71481999e80b74d5d646108c8519", "0c75886e6f994ee8d6caf8b936126b27"
    ),
    # FGT_800D-v7.2.9.M-build1688-FORTINET
    "63e1225ea9755cc00d128014b8852ed6f7a8525f3a3be4a2196e5793d133fc4f": AesKey(
        "2426433dfa0e9da6a52a8b2ae7bdde9a8d401a682abe593280f84cac0cce2ef1", "4524cafdbfb39c1e24d87df77fc2dea4"
    ),
    # FGT_80E-v7.2.9.M-build1688-FORTINET
    "88aeb7043d480a05d9a4e99b854c04978b7881438ba0bcfc5485bf8bc6d25857": AesKey(
        "bd2381e2560a84459ff5a8524d81c346dfd6a7d83797dbf9de850f7325e2d1ea", "fde00c015c9ed614eedcd1c6cf9a1333"
    ),
    # FGT_80E_POE-v7.2.9.M-build1688-FORTINET
    "4b42303eb5ee8821c799262bba8e5da5d36da56f809568f02726b5fc1b727c92": AesKey(
        "98b6d779830531e6b9b50e3a299a9cf16fadc633695014e0aa88a5faf45d76b7", "91b6069f7495a14edb023d1b111aca5b"
    ),
    # FGT_80F-v7.2.9.M-build1688-FORTINET
    "3dc88391b4c0e889a7fa14b67143a77815982eb860138662cf1cb3a9d628ac68": AesKey(
        "e0dcc9d2fe81a4f326b916a2135b98840fb1019788e114dd04acdfca91003ed9", "5166cfb5ee6493dfc6cbe84670d7e5e0"
    ),
    # FGT_80F_BP-v7.2.9.M-build1688-FORTINET
    "e59bd3717bf362ccf733ea75f3652368c85ee5dd3e359467802c1ab3dcb108f1": AesKey(
        "6d9e2dc1f77cd69d24878659555f2c830f1eba95f1f3152e390c79a120d7f607", "a13177eaceb3b36741c126ac042f004d"
    ),
    # FGT_80F_DSL-v7.2.9.M-build1688-FORTINET
    "ca592291670001046a55e1f3f7f5464c3738f9438e34ce513bdae5dd96f80c23": AesKey(
        "ecfab5cf504c21467e5c4e117fbb2b081c40dbd1802ca87ff00ffeadc4300764", "09db9358d1516cc097635a7101de6204"
    ),
    # FGT_80F_POE-v7.2.9.M-build1688-FORTINET
    "e77198aee61bd905e940dccc7969bc3c702c501fe23e82b7bd01f849db3c1ee1": AesKey(
        "71643a334e4de473322bb18c43217b5303b064c31ab71f23582957b36e1aca3c", "da4c3e47a9dcd1e6b7c25f6dbd074b3d"
    ),
    # FGT_81E-v7.2.9.M-build1688-FORTINET
    "14aa503ae3c36f2ba8b2c5cd9a7e5f6d9e80f7491a1c70ad58b691a81e584c80": AesKey(
        "db8acb0cdce991cfcb0c0ee240beb9de87b8dcd109a950fb2376e54b653ec013", "e81f7e3cc22a1ceea313eff05ca9d659"
    ),
    # FGT_81E_POE-v7.2.9.M-build1688-FORTINET
    "9f4fa8a186d1485aecaccf914732d86f9915b57cdb75def9045acbc792d1ea30": AesKey(
        "d8514daeee72218349b993adc59488c61f6e7e1c027e43689786c5f0b7cee848", "55816c9f07df744150b696be29535f52"
    ),
    # FGT_81F-v7.2.9.M-build1688-FORTINET
    "d031cd07a2fb90361cab50464025b55453f827f7cd3a68195627dc5c0844dd1a": AesKey(
        "4c8bfc6c98b0ba1b2c42a7f032264e62266019d578ad458f7138b99b99ed7939", "4e7f5955f1eb0eca4ebec9dde462fb70"
    ),
    # FGT_81F_POE-v7.2.9.M-build1688-FORTINET
    "74c271ba58111584997fa6bb3944f41f5242eaae3deb67b9312d8832918249e8": AesKey(
        "e87c5978593028dc7fcf03c97a47307a8856d54a023df73d8d8a2a173f87f86f", "10ef7dbd3b59ed49cc6bef9344731e70"
    ),
    # FGT_900D-v7.2.9.M-build1688-FORTINET
    "f7ddbeddebfc5ac645e07aae2e3b76c681de97a0a63274f66a3fdfd78e02a76c": AesKey(
        "67c454e546b84220e3a969011144985991fb2234e6c8e4953d5636ae3fa9a378", "b2a0e2365560bfb8acd46fc6351e1f0f"
    ),
    # FGT_900G-v7.2.9.M-build1688-FORTINET
    "26c72a29e32b686fa1b161ce65ebbfefdd88bbe34788d727af344382fe253c1e": AesKey(
        "e60ddf2f1509f81c4227ed5c288168f90f29eb91beef1e7529808fd22bd0e352", "384c4e574812034f07d611025aa182b4"
    ),
    # FGT_901G-v7.2.9.M-build1688-FORTINET
    "b4f1f900bce13d9a7ea6a9c502f8ddc00d8f37955f7f02a72c99d0c21bf52f5b": AesKey(
        "90cf3cdf6d82ef8959ad77b47bd02bc8152bc96b4ca66e7827760fb8cdc6d863", "accfce07d66b0eb008057fee9ed8e426"
    ),
    # FGT_90E-v7.2.9.M-build1688-FORTINET
    "e1c85f5fed2cadd7df80d858e3cad8f2d8250d126f115f39a4d64edb410986e2": AesKey(
        "11188a388419844b925d3062dd5db5b9124c47e9382a2bc8ea0b271022c00e6b", "d5ac7c70b7b7e506b15f678542ac0454"
    ),
    # FGT_90G-v7.2.9.M-build1688-FORTINET
    "4ee89eb72a1704549b669f7d3c5a079ec4fe611271721d6e27a3be36f662a8be": AesKey(
        "bf39fd552b1ef0325c2a8880972d4a6e03ea33f7410838a65a2aa6ea710c403b", "afec858091f1f77ffd9238d5128c308a"
    ),
    # FGT_91E-v7.2.9.M-build1688-FORTINET
    "64f655f78514fb56a9820a3e962c8c8bc588a807b10ddc01f86e61940c39f93f": AesKey(
        "9e3ac51e4d0ffda1ad314a744e8897e19670c5192388075406462d4757c8b7d3", "8bb3f84efc4e9990e494998d00f7e0cb"
    ),
    # FGT_91G-v7.2.9.M-build1688-FORTINET
    "b7523ca79503399d38485b29fe80c245a74f73f598820d0addfdf48383021d27": AesKey(
        "28c06dccd2fad3af2ffc0e2a94327a48a94e0649f4352e9d234d7a2bba9a975d", "c4c6637c33182911dd7f28d91fac9bd4"
    ),
    # FGT_ARM64_AWS-v7.2.9.M-build1688-FORTINET
    "3a0e4bab56dd169b9db4819d7ef9bced8c21c47ee555130fd9c6d5dca2ab1edd": AesKey(
        "5d312b815ba3bbf64f2c9e02a36c921266e192c2af8e21e6336b73fdd1fcb757", "4552a6506f36120340755d93e29df752"
    ),
    # FGT_ARM64_AZURE-v7.2.9.M-build1688-FORTINET
    "f44791e406cea04d36de251693df971ea480e9cfe74968595dcfe08be0b8a4e2": AesKey(
        "dc7a3fae05cbe1cd4e86cf48b1122c3ccbd9117a2e3c790b409a19a052c05980", "a8e80ab560a9b6c5fbe89bf6dc621546"
    ),
    # FGT_ARM64_GCP-v7.2.9.M-build1688-FORTINET
    "db9bf211f339739346b1a01eb74214bb48c84e296277e1b452988314055efa44": AesKey(
        "b50738671ea11241cd710d96f22956ae0bb77daff34949b07686f1223381026f", "9f369adb36f914664fcc5a9d7badab20"
    ),
    # FGT_ARM64_KVM-v7.2.9.M-build1688-FORTINET
    "b266cebc99353c5ec10fda5e1c983d853a9be6ff4da1b88d28d780891c68952b": AesKey(
        "fe132309e8faac82f78613d823dc93c45f492ef1c6181409b55b5d9d09654a40", "3ad16c43566c483a50d8aa147404b9b7"
    ),
    # FGT_ARM64_OCI-v7.2.9.M-build1688-FORTINET
    "55825bfb9dc3ee6520952f7db0404b2be2cd1fe46a3fd3b7338b2576078bc8af": AesKey(
        "c8caaba66b0254a816cd0d8bbe939802ad35306dd87af8c5308087ccc9c1b11e", "5ccd5f6d1c1c1de7c2ac72ee6e583faf"
    ),
    # FGT_VM64-v7.2.9.M-build1688-FORTINET
    "2471e0e1ff07eb63fb3934874f2bd9679ceac2bd9e45fdb79dc056313f868b5e": AesKey(
        "90427ca697b7df664eedd3d5ef51b6509ab7d6720a04ef337ed648a2a6bc53f3", "1c12a5536d624211900b7dc90d78d09a"
    ),
    # FGT_VM64_ALI-v7.2.9.M-build1688-FORTINET
    "be4bf81f0c7ec738c7ba060bac2c716d393165f3c392abe8b8f64b9bd83979e3": AesKey(
        "6b89e7ee5162b98e91b1344ac5ddd82847dcab38145bfd1b05693b0dbdf8264c", "4e91ba94dd81cd08c347d9f10cb60a03"
    ),
    # FGT_VM64_AWS-v7.2.9.M-build1688-FORTINET
    "6e191f8f642e82ae1dc927b466ccb379b87375d24f6a1e4c3ce7121e73e3cfbe": AesKey(
        "6e1f1e250164862d541cc3c97c44756637ffb94d4bfa069c459eed544775e6c3", "2bb0f369c06eaf4f3cfdd38defc563c7"
    ),
    # FGT_VM64_AZURE-v7.2.9.M-build1688-FORTINET
    "e2abc3e5f101556e0235779bdddb3ac28f1a2170eff8e491d6879bccd98ba7b3": AesKey(
        "95a8dc982bf6a352bf4c0a2f2eb5dbf8e89598d387bb3c54502ba1dedbafd79b", "3c57be70262a871919661696617a84df"
    ),
    # FGT_VM64_GCP-v7.2.9.M-build1688-FORTINET
    "6911cc4663035ec12f17d77d07bda467cd539601a6f3d9262adfa1cda7c6ddd8": AesKey(
        "1f1e56f0e44d1b4a52429dbeee2c29db9e1105370998ba3fc32f963a3cd50d2c", "af27f0e07c794c8ecf94a294f136739e"
    ),
    # FGT_VM64_HV-v7.2.9.M-build1688-FORTINET
    "283e37529d95af68db1220c1dcf6e83bfbfc632fabb5f2673626695e063d9fbc": AesKey(
        "470db0e2bef24dd4f0500ab6089120cc5ffe50462515ba2e1451c6f634f1b9c2", "b1d9835ce5df25062dacbe391f5009f0"
    ),
    # FGT_VM64_IBM-v7.2.9.M-build1688-FORTINET
    "d2a52f11fc3ce9e43dbc49886fa700a567964966870b82cc4d4270aa8bb35176": AesKey(
        "033e3debc59ae0fceb957d74c001bb987690011b09621c04be50c631ff3d6e48", "87ff69cdf1f304a2b137589d5e0a01cb"
    ),
    # FGT_VM64_KVM-v7.2.9.M-build1688-FORTINET
    "5f99af8c70a8360dcc2c0b9d6cd2e007a2a29b0780f394ca1c3f3a1e004f4e1f": AesKey(
        "d4619f6baa71393077ac4448334179c1f02696e38791eb3f210692dd15984ad6", "b4ca9911df2b76bb46b0f653f88de762"
    ),
    # FGT_VM64_OPC-v7.2.9.M-build1688-FORTINET
    "5e661eb5592082c32eb31658ae30bdedf7ab8f1eaad3563b43e81e0fce3f2cbd": AesKey(
        "dbd1704bf698f7838bd80066fd097a43f20792800ebd7522f902032fb9561091", "394aa76eea163ac40199aa10a2d4f970"
    ),
    # FGT_VM64_RAXONDEMAND-v7.2.9.M-build1688-FORTINET
    "cfc8eae426f54ccdccfea564f49c907627c6c4859e1ef2c09563bd9d982b0982": AesKey(
        "7f80d4b6e2ce80f03301e4e0ced0364162235b96c192eea580ac82946144bd97", "13abc87eb97a71d60d229d04a842b715"
    ),
    # FGT_VM64_SVM-v7.2.9.M-build1688-FORTINET
    "b12d2dce8a30ae2df3722625b41f744022444e4237cc392bf5f8236e8b77b88b": AesKey(
        "024f0f4eeadd3734807eef6b874b19b8ec481666f2e62a94b7b2bb2a4c1b2657", "5670b2b3705a5ddc33b719ee92efc230"
    ),
    # FGT_VM64_VMX-v7.2.9.M-build1688-FORTINET
    "9f2ff929c8895fb8ebc077b0fafc36733a9dc81cd7190177ea45b8e9aac801a7": AesKey(
        "17396da1777b824f260f588ed1b7e356ace87cfc36e86f00eda6d6af435ff2c5", "ee9a10cfa974b16d9786ac73096b231c"
    ),
    # FGT_VM64_XEN-v7.2.9.M-build1688-FORTINET
    "83c24bd381e34461cf87cb53a40db2f05eb925c99452ee0bb42fad587a85dd77": AesKey(
        "d8cceaf00c0c954eb87c7f22b30edc0701caa138efc8ffb319f4f12d86f1e358", "e63f8efbbebe0fc4771352135c28b33c"
    ),
    # FOS_VM64-v7.2.9.M-build1688-FORTINET
    "2eaabed2a2329da72a09cf564690096c5880f283d95eccaf2d628230844c88ad": AesKey(
        "43e28570e0116fe4099a44cbb35c713dacfbcd4b1d398fcfefa6480ba673ded1", "57f2d05a1c447c23432435f499da5369"
    ),
    # FOS_VM64_HV-v7.2.9.M-build1688-FORTINET
    "2b125e950ae61a849e525c697f59ef26cdbf271491b768eea66ff1a439897000": AesKey(
        "d4d5916338822254bf17781b68ac88a0fe6852cbb630b9f4b81e7b549041c2dc", "5ca45687fef8c94c2132ff0ebdf09b31"
    ),
    # FOS_VM64_KVM-v7.2.9.M-build1688-FORTINET
    "c13a142f344985bd4a39abc2bb22b5d1060edcdc1ed5f90c4189ec197fb70717": AesKey(
        "08ce2bcf7210911a51f65814b50b548db01e6baff1d813b87bc79ce76c18aa85", "39849de1d4068183e809646bfa60da90"
    ),
    # FOS_VM64_XEN-v7.2.9.M-build1688-FORTINET
    "9adeb690b05f6a32214245ef9aca8afa8d03514a43d6d1b0f1e32d9b069dd353": AesKey(
        "131322f72cf872c6cf78ebd65122d84fbf4d26319fda250e2a857b7dca8963ae", "3181fd42fca93243bf065b902ee61fe6"
    ),
    # FWF_40F-v7.2.9.M-build1688-FORTINET
    "d5311edee9e8b6974d3bc913f45530e526f6ab9066200c5e82ed7bb62a2c036b": AesKey(
        "b3089c60cf8b8909cffe539f7aa7b28eb2d7bc856f8196d354762c03bf3483f4", "c54a8735ba548a2582f0e5ec84b21fb8"
    ),
    # FWF_40F_3G4G-v7.2.9.M-build1688-FORTINET
    "e6f8b91763fdc8e920b162afd5151d0624cab2761a3406c697ee2cb2156cc00f": AesKey(
        "d231d45fb34668aa2657dd56a51b52bdadb05cc2903c410d4412b0f094cd259f", "1ea4349d10d0d429eb6e4a85101fb979"
    ),
    # FWF_60E-v7.2.9.M-build1688-FORTINET
    "94068f3f155259aeeffa9d6695113d894fad2ae898d93ff5c1d2b113cabee3ea": AesKey(
        "1fd8f82cc26602b1779e0dd2f119cb1076a4f3178360017bc5c9712caff30802", "3515b63337382dc089c82dcdee1ec465"
    ),
    # FWF_60E_DSL-v7.2.9.M-build1688-FORTINET
    "dcc34ef68980f72d139eec72852c60ec935df9105f00098dfcb12b2c6dd897cc": AesKey(
        "977a43227bba20fdc9b48edbba39367e00a53cf719efb68d874d8dd3dda1dd03", "2bdef91a0eff0bea48000d278ae4c899"
    ),
    # FWF_60E_DSLJ-v7.2.9.M-build1688-FORTINET
    "ca8b688e80c073228bfada2aa146065c705fa315dd323b7da2f0e350d468766a": AesKey(
        "a3358b804e7cfea9a3fca7d48f3d2491393c434be31e5f3e2524ae04e8439f7a", "3b31fa2950dbafc771c47ffd6b456e68"
    ),
    # FWF_60F-v7.2.9.M-build1688-FORTINET
    "469c924b39d2289fc626a569b0015426f8e6dc4be04ed38f7280f2817faa3f78": AesKey(
        "336232fd7d65d4b2f56dafb866e2bfafc0e174811a12af1aa9dffeb6ce13afc9", "6d8f6f37be5cabba45f6a549463b758b"
    ),
    # FWF_61E-v7.2.9.M-build1688-FORTINET
    "903a3743435a35dfae4084b1374cf3860789f48f5482fc64d7a1b169f3885376": AesKey(
        "22fa5d23584d4f93627626a174e64c16f15df48061eac62978f2fc48b7138f71", "033bf3e8e2653e97c769a89ec6d95152"
    ),
    # FWF_61F-v7.2.9.M-build1688-FORTINET
    "59cfc3130889913c8ab2bc37d0fd7c765cbbb1c4bf2703c2d9fb1402aa6a13f8": AesKey(
        "c8285b136962743b80dcb0831af8f53eb0c4a7cf509a3af7a81db623c3387cba", "7cc71e42ea87378db3a9f42523928169"
    ),
    # FWF_80F_2R-v7.2.9.M-build1688-FORTINET
    "13f5cd7dc3e04e47b88303d5e7283a4af1720e512fba918b426a5f0bacfa83ae": AesKey(
        "1d47a763dab9ca9af2379b7c7488a0b373ef8707af1326734c77a6d35e1a5ddc", "f53847d30eacac4d98401b90232bce0a"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.2.9.M-build1688-FORTINET
    "e99fac2dcefc640ff357d8fdf8f00e42da704bde1f05271b1e7b4882a2dd442b": AesKey(
        "a8f2666b8089d4eb596b92c88e614611a07eb8cc07298da148fb6d72f1e4a114", "5844e083a8b91101c25588170f9dbe26"
    ),
    # FWF_81F_2R-v7.2.9.M-build1688-FORTINET
    "e7ab6f347f59f75dc62670118881b9eb07eaf3d8bd154e84946338e5117b9102": AesKey(
        "1f2fd7c83c5ffa23ec743db1602735fedda93805d18ad3f17003dc18b663f2a0", "e1a417bfeecaee08a141195e1666748b"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.2.9.M-build1688-FORTINET
    "7cab1a9f3683850da4f86cf1d59bdbfd292472b08334eec54fd922425442c1f9": AesKey(
        "278f491fc193c09261f27158aca8b2654ac8e6e373b2e0f30302eef19df0f2e2", "2b7839eccd20fa0ba0cbdee00c71a45b"
    ),
    # FWF_81F_2R_3G4G_POE-v7.2.9.M-build1688-FORTINET
    "37f87fa92671da69327f2063fa76eab8a2be1e2bd5bbb35425f9836590080b40": AesKey(
        "bf9f8145d0e30776dcf6c4d0e5fd27100e3b05b6d681fc1b13966d8554d0027f", "4061e7399b282f238b11665f6c278933"
    ),
    # FWF_81F_2R_POE-v7.2.9.M-build1688-FORTINET
    "f628d5fa734b888c8d21404a5e43057fcd22b1e36ba95649b9a205fd6054a20e": AesKey(
        "c23c4ff2698e9c132f1b3669137a9a246186f6925ba4d4f32f48fec9196b8cae", "b2a61703fad80fc4efcc7b53042b9b18"
    ),
    # FFW_1801F-v7.2.10.M-build1706-FORTINET
    "3e79471321efd080be3972d36599f674463538ce3b2988726d61315ab5278aa0": AesKey(
        "8f34650819e72e78efd2004e8054681f778c7e5b7b56e47d754392a71770a985", "c98530c4468fb5e9e62928cca1679999"
    ),
    # FFW_2600F-v7.2.10.M-build1706-FORTINET
    "962bdd19dddb59a21c01733956eecb6fbc9a941a63508533ddd1663389a85a43": AesKey(
        "c059d75ddea4bf6baa6140934e9e9faf96a4ed4b993ee55fbe1da932ceee1de8", "f206d0b834fbe7bbaeabbaa2eb0f9f38"
    ),
    # FFW_3980E-v7.2.10.M-build1706-FORTINET
    "c90af5486f41d633528e5d8ba800fb3288dc849a0eccaa7f8db6aaae46f8e64e": AesKey(
        "703589939bee1a6436545afb9792040d16c5b3368f8459cd03902a1e2001e21d", "69577e71212947ecb18cbf4f9972bef1"
    ),
    # FFW_4200F-v7.2.10.M-build1706-FORTINET
    "39dc4880cab0e236fb6528b1e43325479b7684d7f6afb6a3556f26a81f2a744e": AesKey(
        "7a77b03c709d610bfac14284ce35e08bbbc3831b8c2c86f5e294ab14f108c4d0", "6a6406ab5987ae1f891d520d3348bb6f"
    ),
    # FFW_4400F-v7.2.10.M-build1706-FORTINET
    "65a407f22d67154f3c3a4a28cf13a453a5b8f8a2f92e4a51be6ee77cbb48991b": AesKey(
        "5e2e3df4740f79c20012f6f5f3a0f7d97fdd33eead37bd05731a4ba7cfbd3f8f", "0923b2ad0ff951256b1bb42cfc9298ba"
    ),
    # FFW_4401F-v7.2.10.M-build1706-FORTINET
    "94febf4f08a0c6c41675c43bd3fc616aafaa98d7b001f30800c18b3e835ceaa5": AesKey(
        "2863bed5a8ca7b11dfcf2bdbfecd773df2e6c543db4053dddb696dc415d795a6", "73ad34abc88cccb81338d6c92e645b5a"
    ),
    # FFW_4801F-v7.2.10.M-build1706-FORTINET
    "3785815b5f670d4ac091b243d1bc21f7746c2d099eddb6d88885231390aa5c57": AesKey(
        "7e9908c1e36b8769b211c323cc147106af5bcd0921911b23de0c4a17af29eccf", "0223795b20b0ba6cd892d3770a26b8dd"
    ),
    # FFW_VM64-v7.2.10.M-build1706-FORTINET
    "32e79f8e766ceefa5ca14100bbd2c080b3f0ea047bd219441e843eb4b92d9f14": AesKey(
        "7ff2ea8f0d14fb5fdeff034962d9b5ae2d4d702a8e5351a27bc629add33986d2", "9ab2a2ea578f3b1a5ef542accb197006"
    ),
    # FFW_VM64_KVM-v7.2.10.M-build1706-FORTINET
    "a4c56a74aa60464200b1513f7353cb233f2c0803c32814f7b005914b232cdcc6": AesKey(
        "adef75ca9347875e1e930dd001186793a09b372a7b18a711a48b3770b4ebc9d8", "562896ae289ddee833288c0c81e6c414"
    ),
    # FGR_60F-v7.2.10.M-build1706-FORTINET
    "9b234f754b945192be9597d7c057d8292367eadd2b393ab03a2cf1f94694fe90": AesKey(
        "4141e408a738ba1c7e9421cd6bc8372504bc945ab6391b0bee412d0b0d0153e0", "8c5e576490b94350f3c65b069363333c"
    ),
    # FGR_60F_3G4G-v7.2.10.M-build1706-FORTINET
    "b33cd6d65b0c66dd60fc910748261a53b680028f90651da28eb80db8b5e2368b": AesKey(
        "3a712e0c4b14081daf9609d9927c1ed83215c3aa50ac6bec4235356c051874e1", "297154cf856472d78cedb585f77e8884"
    ),
    # FGR_70F-v7.2.10.M-build1706-FORTINET
    "b2ee5d3c2be317d7d3ecfef08fa79b41c62cef26d9847fa417028d77e40cf361": AesKey(
        "447afd8cb7b3814313beea77f8df4af0903140d001f38c05dea27d3d07f65fcc", "40feea2df69c664ab54b2a3a7bf360d1"
    ),
    # FGR_70F_3G4G-v7.2.10.M-build1706-FORTINET
    "2f81b90b47603f2d18f377ac428ac28b59cad90f49083c29738fcd56902cd824": AesKey(
        "95d6ec2aee7c97eb9524754890d88c514babf0d4b75beb41129893e3cb42ca4f", "ea01e4697f5c1d4c26664e6d4e0cb862"
    ),
    # FGT_1000D-v7.2.10.M-build1706-FORTINET
    "d7af0a2b5803a99fb7d66b6195bd88d1deeceb7e8b8805f273a0e49af9ff9e7a": AesKey(
        "628f90e2f2b470630621c7edc1af8a567ce7f50e53f976fc99753600a80d8fa3", "203517b0568c5341fea8c4e7f9add89e"
    ),
    # FGT_1000F-v7.2.10.M-build1706-FORTINET
    "5806cff6390018344d64e1364b89a38595dab5b99ce61e7b1a2ca6adbaa16ac2": AesKey(
        "c1bfdaafbe492d92152d43b52cd03aaf86b61e003aa1c5e6fe247c6769407b93", "99271600010d8b654c059382f94f19da"
    ),
    # FGT_1001F-v7.2.10.M-build1706-FORTINET
    "39edd8f30a2c0fb9e63ac7802e4326ba7193f7121da09d2249d7e1e68230708d": AesKey(
        "a9647d0a8c5fc19a36cd478f57a6e8fd73b39924d80ab6e68f6441b1f61fd842", "0e6f24c449b8ad0d044fd25f1e4c07d5"
    ),
    # FGT_100E-v7.2.10.M-build1706-FORTINET
    "c343edaca9f31402ba2ff1925622374af730bf859f4f716de18ff1aa52060e74": AesKey(
        "704a172f8536f6472c3f27c2f734a8ff6a42e00ad87acf88d70dc859fc7cb6c1", "e6ccbf2077d1be55e391c0a15e7ad155"
    ),
    # FGT_100EF-v7.2.10.M-build1706-FORTINET
    "cef81cc5eee7f8aaf3731e03668ce211f20ad519265aee9a161d6364062606da": AesKey(
        "e2afda2d0cbf88a5fb5e5793fc776611ade685a50fa2477f5e55e86394d83eda", "b3d95881a2fc0c599f9e7957abfd1fba"
    ),
    # FGT_100F-v7.2.10.M-build1706-FORTINET
    "7b4442ced0c8b36e1398123f721421ebc11c73420e339c6d3993ff7f3c275ad1": AesKey(
        "c20e10138d51fb361fc52181f65de2f230ef5aaf532c092a961b55b48b700039", "c7eeea65be28ff95c20ee87d3b9b1561"
    ),
    # FGT_101E-v7.2.10.M-build1706-FORTINET
    "c2e6b20bd098e35d6f0513d63116b898411741d151389cbccb78b2bf06f95d58": AesKey(
        "4de51458262b5743676e5c57870830a1dd64c84849f84df3f738bc2488442f06", "57b839d46e1ae97b45b5357bba8e8d02"
    ),
    # FGT_101F-v7.2.10.M-build1706-FORTINET
    "3be571c83b9145fb5a562791f750915174a72a149abcc4d84289fa1529cc0074": AesKey(
        "58893450fa9e2485e635a9e9f1612f4040b1b27468c86ed0c93cd02f6af91d7b", "defd8d5fb760cd8a14b3477902585ae3"
    ),
    # FGT_1100E-v7.2.10.M-build1706-FORTINET
    "a8911cdc9ddef6211e789e4332f1a060a94319e90f8c1feafab493c6e485d7c2": AesKey(
        "739333736f28b4009e879847d17f2167d59f570e6216d8068ce30e2b20f020ab", "8a603f4a6340d2f6dacbebbfcbe8e055"
    ),
    # FGT_1101E-v7.2.10.M-build1706-FORTINET
    "308a952afb2e350cb20cf7f752a9c0c0b010f48175eeab2a9a9275a331bc6538": AesKey(
        "ce76ee69c78ddb0a0e64ab58ad6f1ce0880ebb2b5fdd2dd61e3afcbbdff8cec8", "7afef9af6c38039c0255a599a3842606"
    ),
    # FGT_120G-v7.2.10.M-build1706-FORTINET
    "33c36aaf985aa481ee55c1c33ff0fef062d3a13b59a17a9f71c99268820f922e": AesKey(
        "695d1fcc4654b1d7fb7f8b9f068c693298cf5ab6fb347f1534b5bd5cdd82299b", "83ae2c006e0070d8acc8a56cbd98e919"
    ),
    # FGT_121G-v7.2.10.M-build1706-FORTINET
    "13e490364aa9c8e15055c5de226b2038e861f388191cd39c7cae568e0b93ec1d": AesKey(
        "e2a1c896eae3bc349cc0c215fe4c1b407e41c1576e95b21e74d72cbd0a455561", "98e7d4c848300688246f1a9aaf21a2da"
    ),
    # FGT_140E-v7.2.10.M-build1706-FORTINET
    "a663c1d26bb2fe168105711c1ce1b9e0b7e10f8b5701bcada30f3b9f78bc628c": AesKey(
        "e2645cee4fd44257dbdfdc8bcebbdc34a126b980c7128381b02affaee5012d46", "23e80136853e4c486aa2847606a5dbea"
    ),
    # FGT_140E_POE-v7.2.10.M-build1706-FORTINET
    "a576e831f564cb4b22d8c6a70f696e41405ac479bc7a88010837721b73007c99": AesKey(
        "eded78c39f6fa759c7eeaf7ba93259505687a5864e6f9c9a9a816f2c9b4bd63c", "08e5dc330ccada9edf93e06d9e66d4d8"
    ),
    # FGT_1500D-v7.2.10.M-build1706-FORTINET
    "12433a3f0b416cd9f14fbc3439ccc2271f7550415c93d56c1699332026db1109": AesKey(
        "805b11ac064de549bedb0151bca64ca3c1a0ea764c90f2c181ba205d283a80ee", "9a2a7ef5625c1a0ea869220e970a1be1"
    ),
    # FGT_1500DT-v7.2.10.M-build1706-FORTINET
    "e53e7c945b32c157cec3ea6e12944f67738aee0b4de8051cd4a85996a9de4c43": AesKey(
        "9e9760491da70865d0a7a7eefa5abd5a73c965adeb1e063d63a03cc896d5dca1", "246d71c954136e24cfbafc40658f6f5e"
    ),
    # FGT_1800F-v7.2.10.M-build1706-FORTINET
    "3a1897459b9ccdb069b156a887d547ef3d0c93ea166c90d9bf612d77ffa967e4": AesKey(
        "b2293adf222be5ceeb6e1c6f81096a206340120ce19bb9b9bde3d8227d838768", "06ea433bd1abadf43ace0a34f7d1af94"
    ),
    # FGT_1801F-v7.2.10.M-build1706-FORTINET
    "b9baf9b4010b80d662286266a319d219c3614da05e9e75ebaf8c64148218b368": AesKey(
        "493c8d6c794c0c29a7c73d761710730fbd291067e91e8834c7e2386455a0a5b7", "f2912d35ff48ff6b754ef95c9f10033e"
    ),
    # FGT_2000E-v7.2.10.M-build1706-FORTINET
    "3688d7d24dc7d7efd84b034e38c51cc29753bfa2a21284b68e9ea6b7f33d9e70": AesKey(
        "c95fd7df4f401b0dd7e293d9945813986703dbe4b10bcf34b988f7fd80a177ab", "c4de28271044ae118c431231d88ce8c2"
    ),
    # FGT_200E-v7.2.10.M-build1706-FORTINET
    "91ffb074ae11fb0e690e789d7aa8da233025b6dbca85a3bbb76ca763c6f3916a": AesKey(
        "2c9db2648d657d4bd620bc3afb0804295f1f2fcc89cae0b48be36d17411f52a9", "7973a62786dabf31bcfeef620c12b126"
    ),
    # FGT_200F-v7.2.10.M-build1706-FORTINET
    "a6608b34311dc81950b7205dca3bf046303c24c22582088d7f63b9ec69bfaf58": AesKey(
        "feaf6ee6a11f4cbe6941f8912bf3194f435060cc31316f8d6fa951c8261704de", "4c180d394ffae3345f24c113621fa52d"
    ),
    # FGT_201E-v7.2.10.M-build1706-FORTINET
    "7354db57b741b0406dd61de777447a0c1429672c260e40b16dd28fbb27ebbdc8": AesKey(
        "1df7f86eeccf5b968ffa16bce9f569deae65fad2a7b335b3222c716888e44f74", "1350fea7ec825974a0e09d5edd0829a8"
    ),
    # FGT_201F-v7.2.10.M-build1706-FORTINET
    "535ed49122b4f7316fb77dee278deaacc5213537271adfed20a281f5dd4fa75f": AesKey(
        "83b637eaec1b586fdf3ebe35fb3264ee16c62a392cd7cbdee4fc81de5d9c3c68", "4ed55b06edfdd83d43558d2bf951a6b5"
    ),
    # FGT_2200E-v7.2.10.M-build1706-FORTINET
    "d431d059dc21476fc0a17279980a2eb6122420d21a4d58f15cdec3787efe8c81": AesKey(
        "5b41f9db5d4ff3abcfc2c78feba50e56b5b23d1b237e46f53367127ff6033c46", "290412ddda6db099b5b4c298bdb6d6f1"
    ),
    # FGT_2201E-v7.2.10.M-build1706-FORTINET
    "c465af1009da8e1c70b3d40ef7b315f27cb5d86936e7e2499d3eb3a15b252626": AesKey(
        "0b0079184de25964eba8b7260dd569284430bdbb99123d3a30f07481c5f6f48b", "583b01b3c46226825f69fb62529b2250"
    ),
    # FGT_2500E-v7.2.10.M-build1706-FORTINET
    "fa953eeea5b94f0f1e5afd41c4151c21e461158ca35ef3d6ec97cfea7e93b41a": AesKey(
        "3b0ea357b00914229ddb3ecafcafe459b9d7c75415ad4f629145922a513d1c7a", "1799ac75b1d5333977af634cd96ee90c"
    ),
    # FGT_2600F-v7.2.10.M-build1706-FORTINET
    "0e8c6d458d1f2377188287289db10b1957cbf71b36f635dba8a8cf0c09632561": AesKey(
        "882446b564449694a3d8caf21d6c0f3b591aa68c76cbc21090ba2f3637d9c5b8", "45ab453957e5b5da7ce46f80f06da8e4"
    ),
    # FGT_2601F-v7.2.10.M-build1706-FORTINET
    "af20f7905a8cf87b252ef4d2ed47ef835b522c39a0ddd3c61b7fe8effc4becbe": AesKey(
        "0e6724f399a4b1d6c83e3ce3fc9c2515c3cf7355c4a201fa86eafbcbe93b1e6a", "fa5eb1fe1cb50dbc391786e14abdc9af"
    ),
    # FGT_3000D-v7.2.10.M-build1706-FORTINET
    "35624aeee7d75e11636f1b1813e39e2695a47d572aca91a5801b0fea19a178c4": AesKey(
        "67c2151a30adf0bdfc89112fcbc9b42d9c998063b07116cdc5464987c8e5d9d1", "64e4e77529b59b2ecf72afaa9adecf5e"
    ),
    # FGT_3000F-v7.2.10.M-build1706-FORTINET
    "3a0f30727bdc4cc41ba963db4562af817be5210f986588002780594dc78233c2": AesKey(
        "430dbe75bee516d43ed506e47db44c96791b69e0de5a34041842ffaf7ce4d6e4", "df10a9d3c39fce5a47447abc26c6046b"
    ),
    # FGT_3001F-v7.2.10.M-build1706-FORTINET
    "599cd89e2b8707e724e0b0aed43c9ce8256fc7ff11a85c2712ef8bd36f1976a7": AesKey(
        "3525ca40e36747f694958b5dfe57991ff76c95f7401c205d8b809f7f9612a2d0", "f29b1aa267ef98ffa1380fba42239553"
    ),
    # FGT_300E-v7.2.10.M-build1706-FORTINET
    "ed1dae8a76c3971437410c63e02f583e335b2739ea21650b03661e8cf5dca722": AesKey(
        "9b8bcda1ea96513873de1eb51d50572017d9cfdbfa3c1a344b0f9dc93131a0ee", "e15ee9b0a46fc1cb891bdeec9a785fe9"
    ),
    # FGT_301E-v7.2.10.M-build1706-FORTINET
    "5208d9bf2e468e57fe6a60a35810a686859654cc7279d1177a9ea11a289d51d5": AesKey(
        "034bb690e7e73c74cdf71241973947045f61c54edd1bb4e4b5df7229558f26da", "68e5b90287dec9d2c9ab89dcf8bf2fb0"
    ),
    # FGT_3100D-v7.2.10.M-build1706-FORTINET
    "3acf6be0484b44164f441caf09652a1fb09a7e80408d0c1fb946420fec5c5f31": AesKey(
        "55d7fc22665b6121db899daccd1817144710e015a4d31ef346a0f3bd76070871", "db28828e6662265aa0873088e5989088"
    ),
    # FGT_3200D-v7.2.10.M-build1706-FORTINET
    "22e7177dee53f46af94617882e80f4bad0b65a6dcc09d4918aee544d1df45eb4": AesKey(
        "9d5042bcafb5d75a9512cf50ef1ceb726834fe6dff958baadb52b1f33c0e38ff", "6b7f65ec557a175b77ac730afba5ee1a"
    ),
    # FGT_3200F-v7.2.10.M-build1706-FORTINET
    "af77980087003ccaebacd386078659930dba07ff5218b94939e52cbebbb0005c": AesKey(
        "6cdf0aaf3e0c021ac990ad91ee91f7262ac39523bd62c8190c1e0b14c43683ed", "a939e0dc421f1b6f8a86c3c75b600993"
    ),
    # FGT_3201F-v7.2.10.M-build1706-FORTINET
    "16b1363873b231cfc7dd1a121c5e9f65adacfe970a4624245c7c2a47535a6cdb": AesKey(
        "0c0db6a79c9e673013f1805162d765851623dae71f511068cc3e27aaeb3ee057", "6c0a3e1301ece59044cc2d596db7fe59"
    ),
    # FGT_3300E-v7.2.10.M-build1706-FORTINET
    "afc74a1b59aad708c6615f7a6bb59a212d0d21fb522e726fdd1f13f4f95c4fda": AesKey(
        "82df15cf9b384fca62fb97a4bee302b473b0b4bdde49cb314467bf400a7a9640", "8086c9bcf709b04c98384644d3a14f21"
    ),
    # FGT_3301E-v7.2.10.M-build1706-FORTINET
    "2565bb3359c2a492c346b7c2c87db3d51ae99428aa5bf93893411a1d39ff355d": AesKey(
        "5801a00b26d57b66077a8cd1ae0fe850833d3cc46ca4e24b07d05355e8285df5", "e2c03b00fc47f62659d35b67d9ae40a8"
    ),
    # FGT_3400E-v7.2.10.M-build1706-FORTINET
    "e7345de4f8f267a8b9ade60645aa0368e6e7e0742212a87289a6f22d8901ef50": AesKey(
        "179ce7c7a90a54eaff93ffaf515f8b31aec7674b1dd84846cb75ab82595db887", "7f734bb044ea867aebf0bc4388af8d86"
    ),
    # FGT_3401E-v7.2.10.M-build1706-FORTINET
    "5ef0a0bfdfba4ace6c39195e345f095207ace2b2d5125cf3295b4268d1cfc438": AesKey(
        "053eeb6b3f12b00fab5496d462453d115dce034ad2f4d389fa6beba6cf8402cd", "c084253dffbda83845975aa67b301bc8"
    ),
    # FGT_3500F-v7.2.10.M-build1706-FORTINET
    "6e8d223ade66f2c94f2ed7e3003ee6ed7079a0b044c7f8e6ff12f55d06d0649c": AesKey(
        "55ef4733be9c76697f04c62fb8a4350e2e41c2377e00f99b8a2403ba4ec45ebc", "68a41f585ea36f3be6478c06fc2b6283"
    ),
    # FGT_3501F-v7.2.10.M-build1706-FORTINET
    "14e82b5da058cc0c801cede37c8ca96cb40046d7a7680565d09fdcb58f0082a4": AesKey(
        "75bfee1d833250e518aba94f8291e27738e359dc38272462a52fec3098aa9587", "d373efb3096a9d31139851799de10e63"
    ),
    # FGT_3600E-v7.2.10.M-build1706-FORTINET
    "3123329b0e63afd34d0b4db81bb53d9bd1d8d2e8f8bbf2d1e8a0dde7b30ab0d5": AesKey(
        "2c7e226c04127a7ca7a72cf7a9bac7cd19f96fa02b0de9d170f170440f0e9b16", "e346fea48cffb9a7375c76936d281ac7"
    ),
    # FGT_3601E-v7.2.10.M-build1706-FORTINET
    "5734d1fba9666b524193cb95ed6f875a3c07dff0d7d3c3ab701ad1536e42320f": AesKey(
        "34aef864ea3eb729e2041a2c86ec2107f2af9975dd8a5dd4c5929ed70c4ef18c", "0cb5dd2e65382b94ea76c91fb7c9c5d4"
    ),
    # FGT_3700D-v7.2.10.M-build1706-FORTINET
    "0d8c506ce7e5bededb8620e437ce9bf14060536fb6fc59b1589be2f0aca78465": AesKey(
        "7edc2dd5a6825dda1bf3f1e2a240fceb0c51c6ef88bba7c01ddb6875ccb673a4", "b2316ca3027e6e6c524178203ef4aec9"
    ),
    # FGT_3700F-v7.2.10.M-build1706-FORTINET
    "a7ed95100fe2cb9e3ce33c6f8bcd77543cb8305fdb1b728a67edad316238c1e4": AesKey(
        "41ff22d779cc98304e750ee7f995ccc49d71c7f6dceddbd15afa658843f0ba26", "5bf6e2c7349fc37b654bfd158362778a"
    ),
    # FGT_3701F-v7.2.10.M-build1706-FORTINET
    "4617aa5faee745784eedee7470dfda0e1e2eb31fb153e013dd49fa9d06638bc4": AesKey(
        "832bdcf0d45a09d236afe0c58a9d4368afe8f2445b76f5ceba59d14a27ba473f", "3e44ba30fd89ce5521e7c478ec845807"
    ),
    # FGT_3960E-v7.2.10.M-build1706-FORTINET
    "8828ccf4c30cc00e0786a913f056d2cfa79109fecf27d3fdd3e54d377f8b66ba": AesKey(
        "beffaba616766cdf336453a6ef87aadd4b49a5b7fe9ac4849cd2cda0f3aea223", "b33d4da57a640a0fff914bd4c6c8944c"
    ),
    # FGT_3980E-v7.2.10.M-build1706-FORTINET
    "6c5720b737c255812f2a9335cd43d514be579abe256a60ed0756788296811d0c": AesKey(
        "7941933446673449e575f0b79755121abd824212e8de0d964f3cffe307adc1ec", "2921cef21f2891cd49576bc71081e752"
    ),
    # FGT_400E-v7.2.10.M-build1706-FORTINET
    "d4f26fd564a176983d647d08add7da459630cd8081770fd5c7d128ec80c470b6": AesKey(
        "1b210b4563fb93694bc0bd3c85d7e9dc97ee4cea0f9e2c453fba52c9fbcb9410", "43a6ec8cdeeed38e2e9ea4bc9845a2f0"
    ),
    # FGT_400E_BP-v7.2.10.M-build1706-FORTINET
    "f5b4a8f12a944e2d52b6da88a19d338d103d916fc64e98a82a1b3ea6da24eb45": AesKey(
        "800ac9fa504eeb6e685e0db5be335f245f6606a388277ff90f61dd689b7e95a1", "47436c05ddecc0a3162396262d942359"
    ),
    # FGT_400F-v7.2.10.M-build1706-FORTINET
    "926745c66a6e0d0a7196a422b4d992d250a2ca272ca69e09dca72780c8b66721": AesKey(
        "ec23cc3de2f8879372c367475ddd21f862dea71ae8dc43e24ac426b72e835cc8", "bbc047dd40fb594db4ffc8b34cadbe3d"
    ),
    # FGT_401E-v7.2.10.M-build1706-FORTINET
    "6bec3af20a0cc96f160d1c9650e1d545dd9843d5f93323ca9e88edc9bbf5d79f": AesKey(
        "6ef5780a4d12aea7a4efffa003150c1883b08788fc8a86fb7bb93766db144efe", "e61770bd848f0c61285b5a40c9dcfa94"
    ),
    # FGT_401F-v7.2.10.M-build1706-FORTINET
    "c83346ffc705dd24da2bfd696e87887d5d4feda2222d277a9f367b065c037116": AesKey(
        "962ac2e0dd86ba784d0fb48330aa8488133533ad49fc55692a235282a3202d09", "fc257a21e1ced4d9adcb0b7cf6e6d35d"
    ),
    # FGT_40F-v7.2.10.M-build1706-FORTINET
    "def10c8ffa13d8520a3a1b5351f75febdf59181f49e5f1678c5474b90ba43c5e": AesKey(
        "f7ffe38ecdf669abc2dcf3fa8534b3c58238423fdb19d1211f3483670913c2b9", "9f975db96ee5be4f535786a6dfdba2e5"
    ),
    # FGT_40F_3G4G-v7.2.10.M-build1706-FORTINET
    "f1ca4cb6da606687136f0979b8e1f3b5579343a0e6cf0dcd60324b367aa8c647": AesKey(
        "0446804d4f68aeb63779b5c6780b01de3654db402fcda284afd8bc79e6ee7614", "6ff60d1214ea8652f569cd7c708aba16"
    ),
    # FGT_4200F-v7.2.10.M-build1706-FORTINET
    "ce77519c039ef58a6bba594cdab16c6f4e73597a8ddf6c31c54b64bb96ed569d": AesKey(
        "3332be567e6758374f8a45364dd7484c4dcbb5ba2d34cf94e3a3b0e24816f81c", "f70f75bb1ef8ca6b8a46e488fbe0a79b"
    ),
    # FGT_4201F-v7.2.10.M-build1706-FORTINET
    "208eb68e6fd1fca82893eceb4821f1a3b92b8d111b49d912e896ae6b74112f49": AesKey(
        "b787c5c1e4deb9fbc329e920db2d990273397c356d723646871b2108f4584f5f", "e96d0e6769f34e143b4f8bffc064554e"
    ),
    # FGT_4400F-v7.2.10.M-build1706-FORTINET
    "7f81cf47a08e5cea37e9e04cb0c8625a810c6e628fe94208165fb189e096841c": AesKey(
        "e7725f473cdb4eb07527dd94af7c68d32e4a9fe4f5906172b11bf4848ba44261", "ffa5e314bdd0d866ec6ae135793285e0"
    ),
    # FGT_4401F-v7.2.10.M-build1706-FORTINET
    "8c29a1763ad861eb891414680ccf2444aeb015f89d17bee6010bad3b8ad75dfa": AesKey(
        "7ff406137839dae51cc646049545faf321fee9ed55cc1f1583abd225ed814214", "cdc84c2b4372f9f947779501c1587ef7"
    ),
    # FGT_4800F-v7.2.10.M-build1706-FORTINET
    "e0a6e6b8e440ed7442eb3e0e8d0537042afb978100b41ae646db7ac262bf4081": AesKey(
        "a4b4b557f09261e5021b63082fd3be267b8c19bb7d45f359e7a73bdb89bc9a7f", "5ec1526d0be66e77fb8081ac4872fe9d"
    ),
    # FGT_4801F-v7.2.10.M-build1706-FORTINET
    "00bdacf14f2f37b50f08b6a5ee603d791d55ce0d1918ff806e0328eff9a69521": AesKey(
        "e83164b6040c70c9ae7bfb389c4de15f52ae5050a0cca202ab9a6fd8a71cd6a3", "4ecdacc30c3cf9c46ac21c11d88d5554"
    ),
    # FGT_5001E-v7.2.10.M-build1706-FORTINET
    "7088b80ef67ab24fe3b8c998233c3e0b5f9b5ec6cea9d8e313f3f06b1cfa703a": AesKey(
        "d1cfdc3c5e7ed52dd4558fd324d746862cd8e5225603a49eabc467d01e54a5d2", "ef35d85d26a5f7bc9217bec0ba3d445c"
    ),
    # FGT_5001E1-v7.2.10.M-build1706-FORTINET
    "be1eab105a3a96923a888a453e24aaf280da250ccc293163f8aac30a5534aca4": AesKey(
        "49d59c6e7df9ce64d5774fe4f10243e4425cabab346e986d445b00f6d5e909e5", "247673284988eed883d37561d4beaf4e"
    ),
    # FGT_500E-v7.2.10.M-build1706-FORTINET
    "3d9663457d1541a4df43b6060a42d6c8c4dd9fc0531b9dd3640b786251aba130": AesKey(
        "5cf15488fb4b2ced8b49303c321f9e91d378898ab792e8bc39347604fabf2220", "3cc4dbb7ef577fab50321fdec63592e6"
    ),
    # FGT_501E-v7.2.10.M-build1706-FORTINET
    "c0afa916b3497d9dcabee04d8a1530392f06f5bd74d505a2b99a11cd6e74c6c2": AesKey(
        "ef44f7f29c6f1092aa82f83a8081f9a69385a6930a4785ef78b3be984d68b74e", "0ee869d2ce559e6d8e203d7bae3d2a9c"
    ),
    # FGT_6000F-v7.2.10.M-build1706-FORTINET
    "c0ba967dfca49070f47b5f41df7bf690e10d32448c819433abf4a1a63553696e": AesKey(
        "532065c97425088d67a0618abd1d61d7897bff7c391eb591b7e5fb0ff8616c71", "794a5451436dffe5bafccdb9e8ee071d"
    ),
    # FGT_600E-v7.2.10.M-build1706-FORTINET
    "85bd770de683c001c87971511725bdf48963662a8c29b9a6bce2306fdddeef77": AesKey(
        "82769d807578b6a4dec2d3bb06d0671513426897d57b5bbab8adf19484313278", "786b1ef11195066d09f200d5c8a60423"
    ),
    # FGT_600F-v7.2.10.M-build1706-FORTINET
    "ebc186ec2b472b7269f3862a82a10b62e78b6a95d453e40b543f960eed9a83ec": AesKey(
        "97b9cdd7afdf109c4505fe17da18dd229edf503a652be7464e009b7cc5639137", "d526f112eeb76551dd14344c10592c8e"
    ),
    # FGT_601E-v7.2.10.M-build1706-FORTINET
    "a829b082379c3d0a01ebef3819ce9d57e245b33df22863c1f4b29106867f90d4": AesKey(
        "47796a61435eb25c926f89a3c4134d17a8e2d830203fd04e819844626b727dc9", "81066daecd865885616df21e7d9eb3e5"
    ),
    # FGT_601F-v7.2.10.M-build1706-FORTINET
    "443eef4642d3fe938db7504888d341f722ce945750b14e62c0d6ffaa2999c057": AesKey(
        "ee79d149e0caa7461a6953edc316148e5a4930b6cd69e62fbb42592ea028b2ee", "8962b25751e08c2d2ae6992190dcc01a"
    ),
    # FGT_60E-v7.2.10.M-build1706-FORTINET
    "f6f22fcb886e11b148aff43958982bc3f95a99a752028f426ebbb0132310dcf2": AesKey(
        "b1294e946af0cd237ed26e5550a8987276c759c507de800f60f30af444ddcdd2", "635db1a71ff045a408c8e7a0ddfd6c8f"
    ),
    # FGT_60E_DSL-v7.2.10.M-build1706-FORTINET
    "3334751da6a42f52c6e9fca6661da031dd404ccb71197797db0aae6b8a499677": AesKey(
        "58fbef0d342b8690de36709f8f55e680b971f77962a03995e17d1b599dc0b134", "c0e659fabf99b97acddfc4a2ed0021ca"
    ),
    # FGT_60E_DSLJ-v7.2.10.M-build1706-FORTINET
    "17fc0fe81cdf53cc54295e6ce7516b7d1e97bd76d0d7943e45f8706c632fc619": AesKey(
        "f93b8e24a5963a4bbee5c4cbf95e2b1bc594123f1fbef0f178826d5572ca02b1", "d0d5c841fca5f397fe392c3b2620aa8a"
    ),
    # FGT_60E_POE-v7.2.10.M-build1706-FORTINET
    "6d27a2be88127644b9abce13b444a48c7063ff460396a5ae3d197c085923e241": AesKey(
        "092d0e4dcb134412fe74aa9520ed3d33454d37e35d1418e1176bfd45d0dc175b", "3a5f7bc6951bf81668a79640ca5ded9b"
    ),
    # FGT_60F-v7.2.10.M-build1706-FORTINET
    "d8885a551798b3b9702e54125883169ca13704026c34efd8bf6fb625fdd69797": AesKey(
        "2481ed863190c42c5f4bb56e9878c231d48008952fcc5ceb437a37cf0b0ad098", "359a098583aea15b569740b163f37c06"
    ),
    # FGT_61E-v7.2.10.M-build1706-FORTINET
    "89ee69889a12a7e6e0ed85dc8a0fcfd70b7fa49b31acf67f7437c6d8ce407243": AesKey(
        "f63cfbdb0abaf0c37da07d8270f154c3dee1b34eed9e0040b5cce94c5b560f15", "27b8b83ad60992b86326adf0253069fc"
    ),
    # FGT_61F-v7.2.10.M-build1706-FORTINET
    "77519cb194d7941a252335c49529df78c995d2763174a09e86ee9ec627815d8e": AesKey(
        "982721eb4807d65d0858e29de4e24c284a0009634b549e0aca56f5ebe9a1b466", "50add858ff6c6c5d45f29a77dfbe728b"
    ),
    # FGT_7000E-v7.2.10.M-build1706-FORTINET
    "6793b74b7ff4ce70efc1b8c08d53fdb7cb5b4e173bfb54892d260df8e7482e93": AesKey(
        "77ce646806f380375618e5d9b7102c45cccf69f46c86f3ef7b1af1891cd92356", "31512fd53478128a26fb01991d203bb1"
    ),
    # FGT_7000F-v7.2.10.M-build1706-FORTINET
    "68f8168c81b72392d8218f1fee24262be55b74a02c65d073d368619f6b75e348": AesKey(
        "af492b8f5f5f37491be54b85b8f5b2b3d0f7d3302ecd0db82bae4b8057821922", "a47a50801fbfb0d0ad2e1a1ec34e0c3c"
    ),
    # FGT_70F-v7.2.10.M-build1706-FORTINET
    "cbb9b6e46b6f309a87e8c84841d1075623ad015cd02636be1ad1c84b723d8955": AesKey(
        "667977b6031743c96b8a371ec7e5ffd0266b2ce6dfbd0a3af8669b04d2ff3920", "7b26a22857ab7fc7c50428a356979918"
    ),
    # FGT_71F-v7.2.10.M-build1706-FORTINET
    "6ad8d067342feb83bac88f6cc0a62d11335b321ffa4d3c809cbba6511f332ae5": AesKey(
        "ba47e90a99402111b1c64c05c7a7299778c843c331cd0e246ad7036780ad46d2", "23eca828ec0dfd8d88b3e9bb88cfaeb8"
    ),
    # FGT_800D-v7.2.10.M-build1706-FORTINET
    "1839f975b7fef77186ba9cfea41f39a25ea02ad075f877a24c5b42c8c08bb9d3": AesKey(
        "49d68d15fdc67d00bcb9a61cf08bc96b34cf96a81db7d2611a30902061241f9a", "00cf4f46a119c5b8fe76e597ce20e2b0"
    ),
    # FGT_80E-v7.2.10.M-build1706-FORTINET
    "f7bef188ad937ef42f0c9d2c96c2d409b8f4be1ebaaf9781daeac46de6819feb": AesKey(
        "2950163d6eb26e76f71a232ed9c94caa9ef10b1181018bb85756291cc4fa8ba7", "8c25927f5ccd7a7595dff3b416012489"
    ),
    # FGT_80E_POE-v7.2.10.M-build1706-FORTINET
    "e9873544dc371d0459ae71f1ee9df275d22a9ff4ec164c14f72ab6e033d98d16": AesKey(
        "67bdbdac0de7732cfbaf0f10a0e7dad99cabde7b21da9fbbb154844606591311", "153e60f2ad505ff08455d071954e6a87"
    ),
    # FGT_80F-v7.2.10.M-build1706-FORTINET
    "f96e5290ec3f10490a30f491bcbe6950dc779a79f165cce72672f09daec35e53": AesKey(
        "d10dcd102cbaca0f0f29c7d66baa79d8c0930be61d8ef6855e4ac59f51a5757b", "54309be6e8d78558bb554b6f1f41e371"
    ),
    # FGT_80F_BP-v7.2.10.M-build1706-FORTINET
    "3b2fe22eb7dcaefe5af4ac2f188318134fadafa95e8a434975c449f14d5550b2": AesKey(
        "496b7cba9c54b30e497ce71ea80292fea6fabaafa5899c7a9ccd0b0263b69add", "71d19fef79686a31e09eebcfad0ee859"
    ),
    # FGT_80F_DSL-v7.2.10.M-build1706-FORTINET
    "ccd3988f9020b8b94efeea6f896d5e2e1d520fff8b2b150b391868b286117a76": AesKey(
        "808ff560cde2599668afec6fe6c6920d756e5dbd91a9c63af50109e0a3d0d71c", "07205400d55b8c9a61b5ce291dfe6f91"
    ),
    # FGT_80F_POE-v7.2.10.M-build1706-FORTINET
    "03b538d70412e1981c32da87ea4e0bdd954fd173f5cbee40021ba90c49422e86": AesKey(
        "04dacc1413ce3b31a2c648bf667a7d56c5fb882e4f70ea93eb9beef96fa8261c", "9272d4f5315a99e6f1a48c010113ea02"
    ),
    # FGT_81E-v7.2.10.M-build1706-FORTINET
    "f5580574c50810ec5d047043aeacdefde4c60662acbeba0c0f60be6137d29c0d": AesKey(
        "33a6757d89983282654b071a6e22c34a465ac5f9a29ef9f9ea843a1b73a258e2", "9343127f88a729b6854d6f624bf06c60"
    ),
    # FGT_81E_POE-v7.2.10.M-build1706-FORTINET
    "9848ddc49a9639dc40ee5d95a705f09da1211ad7a1e7958ecb39b7159f45173b": AesKey(
        "4b45c612cb9bd909b87b5de81f468a2d860f9439f63ecb8c66bcd941260d0527", "1ee39ce4d37e3c8bf9d67a36b6713751"
    ),
    # FGT_81F-v7.2.10.M-build1706-FORTINET
    "062a3c5a74674a854141a9cd44014b077061a9c875aea5a2639229c9df5c2112": AesKey(
        "d68c7b6aacfe3a6301d17ff82fa201a069c3b17cdaf98f8488ad5257e922df37", "d6130584b51bf041d475c40123380df2"
    ),
    # FGT_81F_POE-v7.2.10.M-build1706-FORTINET
    "4d9c48b78910f6d1d48ae7203326a3287f3c25b061e5d0e1a35c67519e1a3eb9": AesKey(
        "a210a8c5e62c70aef7b827a1871c6f1d9a243749a1559a3fc110f2d02a25b01c", "eb9381241789ce35c6a0e9098e0376d7"
    ),
    # FGT_900D-v7.2.10.M-build1706-FORTINET
    "4dfa2084afa976b223fbe516deca854070151d593c439d5f072b0791b15144a1": AesKey(
        "b44590ad6eeedc23b905a0ef4c0a1394b738ba6af10973de3eb395f64bcf41bb", "ce988c4584b425a3129df501899f804e"
    ),
    # FGT_900G-v7.2.10.M-build1706-FORTINET
    "4d0314cbd72475928f7a7445dc88018b70f267429a0f1d56212a11fb923a7380": AesKey(
        "ef77e8f033c0b151df029c4922a49c4bdb779c418fac5654997e8e7d381a99ab", "5bac102dc6a56b971749050f923bfeda"
    ),
    # FGT_901G-v7.2.10.M-build1706-FORTINET
    "a25c14e6a081ed8dc99b986d81148e73749d798688868301132cca599ebc98b4": AesKey(
        "42b39d35e27c680a4038d19276a5169a1fefd0962c9a7288ab4bb7f31896eba9", "aa0e7048a802b54e9c6292ab981ae162"
    ),
    # FGT_90E-v7.2.10.M-build1706-FORTINET
    "dc4ebeaa1f864b21166a43795ec6319fcee8077e15f686645b3db8464102f73e": AesKey(
        "b5f5fdacd78d5dd552e65726248b2fcaa32dcd03ad7c7faf3ce929ada992cac2", "70ebf8e018fbb22b7055079c0a967d1a"
    ),
    # FGT_90G-v7.2.10.M-build1706-FORTINET
    "d5a357033ff2faa7cf0e03e2a47794424c9925f3b5635a34c8625c310f6d5b39": AesKey(
        "c846ec747225592795b4763fcb17daa45f296e359bc2e25f11f15107a583e189", "1a719bbb3a834d8e31783573204e741b"
    ),
    # FGT_91E-v7.2.10.M-build1706-FORTINET
    "06d732a22272dd60b7bf828a173ac2f8c4d32f2ccdcae059b4b0c187f25281b3": AesKey(
        "40e7dba54b6fb6fa051231aa5c0b092cd64e07beafe45828e2f153076abdb7cb", "45cf1b3e4d02763a028480746f439e7f"
    ),
    # FGT_91G-v7.2.10.M-build1706-FORTINET
    "b98ff4a89daf8caf4b628954390c5766c73ecd3aff7308ffc7e539a9cdae3019": AesKey(
        "1879d5a3dd8a95fca9d318f94bb7bdb004e6b776112c5150a48622c38acc6c84", "081a23e65c0601a8aa8e809ef2a25821"
    ),
    # FGT_ARM64_AWS-v7.2.10.M-build1706-FORTINET
    "1fb81085e4e3e27df2ab3a654e4be0e77c99922ba2d9651b7c0afda836dcd38b": AesKey(
        "9228277ef345030368c83d2f0b788fc07a0ea4409092f0e8d5801486edac9683", "6058a156b37c0389e13d778192548621"
    ),
    # FGT_ARM64_AZURE-v7.2.10.M-build1706-FORTINET
    "e706ff9c6eab55c504fb033dc36a10b937db009c18f112fb2a3a7a9badfe9076": AesKey(
        "cd622d6e89bd74a570717c5ba1b2650dd388af7c3e62106534c63e9186327234", "3cfa5e02d0e9b71470b6367f22764bff"
    ),
    # FGT_ARM64_GCP-v7.2.10.M-build1706-FORTINET
    "ac34476c171ab2a7b8d02923406d4feddfc8ff6c598d1e5ec7276b1bf6b68990": AesKey(
        "5d7e5abaa0f796a9da6c9308d2ebef72769acd6ecaf8867f7fced165ea2259ad", "a7eab22b959f56e5ad439dd70854c13a"
    ),
    # FGT_ARM64_KVM-v7.2.10.M-build1706-FORTINET
    "9d39bd458395120bc2616bfc6f39b13ba78d738172a1302566214c733b079063": AesKey(
        "b3ce3c98e4817c2e07cceac9839ddb66956799b2dcd5c1cb57d144edf0ec32eb", "f24227b546d7ea4f91e673204fc0435f"
    ),
    # FGT_ARM64_OCI-v7.2.10.M-build1706-FORTINET
    "4847071b428dd5141397a62c8d672baf40df2b366d1255ec44988b8eb5e9051b": AesKey(
        "0de7c27cc0108cb219bb1ea4342f90410f436051c209acdd65798eeab0424fbe", "adbe33f9408a9760ae4b292ea0c0182d"
    ),
    # FGT_VM64-v7.2.10.M-build1706-FORTINET
    "10940daf6bb53202faac0d561929ad56c895b8627ffdf4c62290a0e1035a4703": AesKey(
        "a76140eb5d877f32ab9dc51a33b9637cdd3505caa4e30dfad8db30a504431497", "f4e98a5ae0d66542ab54258530b8a447"
    ),
    # FGT_VM64_ALI-v7.2.10.M-build1706-FORTINET
    "84913672df3dba72df501b94725ecb054b9b981d983c2dd3ff50323ba59004bd": AesKey(
        "4a5e3c4e5b17a6f7ebc564bb02d446b8fc66f5ca855ad6d29f1dce426f161f4f", "0458fb520aef7564dfaa47f96df3278b"
    ),
    # FGT_VM64_AWS-v7.2.10.M-build1706-FORTINET
    "ca81ba3beabf3b9cc6b7f7dbe37c1e3c160147a4a770cdb2c59ae8cedfd3a3bc": AesKey(
        "41b52f46305d037191430ca38eed83f21ca021a3e14d8d72546979fcec556dd9", "8775f311df120def6fcf086e1b75beb8"
    ),
    # FGT_VM64_AZURE-v7.2.10.M-build1706-FORTINET
    "bc0aed71441ff688ecf4e46e2125e1a97e0a000ef92ec77b71c5255ab33a41d4": AesKey(
        "e10334a0d5724ae4aa425be49b47f7cf0e0ff4816048640b2b3b715f40f767da", "a6be42041c37fdb05c180d159e79d1d3"
    ),
    # FGT_VM64_GCP-v7.2.10.M-build1706-FORTINET
    "3c03590726e2bb8da852eb88863ed72fc48034619765303ac573b6830279f996": AesKey(
        "52b391ecf8e99fd65230ceda58f1827f9ac876e2631f7eabb32b2eda057f02b2", "6d4826b11d940da05d3cb7916386404a"
    ),
    # FGT_VM64_HV-v7.2.10.M-build1706-FORTINET
    "3397e4717331b788662ff6b54a73f68854506bcf747dfc5cdff6a3aecf4f1126": AesKey(
        "cc67919304d4010846c37ce8d3cbc70b1aa86b7b3198bc259e829c839bd773e3", "23b1616ebd351b34325fc7483012d370"
    ),
    # FGT_VM64_IBM-v7.2.10.M-build1706-FORTINET
    "3a3e76b2f2f611419bdbc4493077c63d89a0ef47d8f707f95d3844a57d2a595c": AesKey(
        "194e3faa7d29e2227991cb7b532d62b0d48dc138e86219fb2368c4b6a87a6885", "8d2b78880bb7c8c63d690f6c663e1ebe"
    ),
    # FGT_VM64_KVM-v7.2.10.M-build1706-FORTINET
    "be5047334e70e38f934062199e9088d9c21a2f0b1ac208f2599328d09e8ed276": AesKey(
        "6402bb128363f097e65b3dfe84d7a4a5f9c90772b29619edd186fb281da16ba6", "e48a9bb030cf810b0ae060110ccaf17e"
    ),
    # FGT_VM64_OPC-v7.2.10.M-build1706-FORTINET
    "ae62dfa3517b212bab9783f8e3e5faa36024bff3e457eb5f88b6c1ebdfbc6b44": AesKey(
        "a0a534a893d73db437ffe61658b1e66bda2624d705971558198f1cb21368b907", "63fe57daf6a4f38976de0f091bf14eb4"
    ),
    # FGT_VM64_RAXONDEMAND-v7.2.10.M-build1706-FORTINET
    "cbfe6b9cd6d09945bd1484ce720f0482e44fcf4c1fad01a789f0d7954f755bdd": AesKey(
        "44c966c9364595ccb6bc97717f6e679dd2b1398ebb16df2179388a1309c69e52", "2055ed1acf8f5aa8b4633b15bd7921b2"
    ),
    # FGT_VM64_SVM-v7.2.10.M-build1706-FORTINET
    "432756e4f7abecaf3bb0f836149baaf6d71baf910fb91c7f374caedc6e086f87": AesKey(
        "e47b93004a5e1a35bf19e4a5daf387394aa9a559c0ab47f52dad8bc0b3b68b65", "37f30f78caacc1644549b80068aa1a3c"
    ),
    # FGT_VM64_VMX-v7.2.10.M-build1706-FORTINET
    "1067d68ad2bc2887612fc6e1ceca4b75ea985bf04da0a75a07df2904b6f9b91c": AesKey(
        "3036df3598a971d8466f57ef2b686d17663a40f86e667b4151a2c6ea426c11a8", "c1693f3a31205a80a131ea5d2eaec9b8"
    ),
    # FGT_VM64_XEN-v7.2.10.M-build1706-FORTINET
    "ed547face497b6cab98f18e8e596a29ab63480a0a428df69b9a13bcf727bcae2": AesKey(
        "42a53ed9271ae0ee43b895c69352bb832d6e043aff6b69f6beb39c9e9e75f2a6", "b71fb01197f466140f02f70a799077e1"
    ),
    # FOS_VM64-v7.2.10.M-build1706-FORTINET
    "80caa3631b2da985f89cdd14433a04c9fd8871c5022392791a019dc7ed940269": AesKey(
        "9f5baea7dce8f39f4a2bde9a2c38a0d884a0481358e7aa74d258f6a4beb4b5ec", "36d9bd3f82080562e89af5b40663c616"
    ),
    # FOS_VM64_HV-v7.2.10.M-build1706-FORTINET
    "d8113c00cb505c0a17748758ad5fda8a0da3c2a0430b638797ab6d9055047e71": AesKey(
        "e263202b7b7f9bf403f3bb4ea33bcd7c22f604606c4c1c491463206e686f6d09", "6fe90c4a8e4f77c72f97c490e6c43679"
    ),
    # FOS_VM64_KVM-v7.2.10.M-build1706-FORTINET
    "19b819416b31fcc7223f003a302cd03c63a89c88e035cbed02e0d99c8d04e27f": AesKey(
        "bd32df9613dad90f9fe6c5da2f1d3df5fb8d72a6958516b56453cca02a6d86c1", "b7e8caa2f79c0bb93877f8b76c6f3955"
    ),
    # FOS_VM64_XEN-v7.2.10.M-build1706-FORTINET
    "3e378a5cf1129795bb97445417c516df0603d4e3382c9155db4056760f65bb41": AesKey(
        "f4b4fce9214d93997d85e99e5ba78025464c0bb6194c299e3791ac23094fdc64", "05193176f9d727aa8ebc75d3a249e643"
    ),
    # FWF_40F-v7.2.10.M-build1706-FORTINET
    "cddd97c1c8c1cdb77e94abb4c46772bbc59473019b5e4affe88f63ebb51d7d80": AesKey(
        "50df63afc7c95b726b3b3ecb34aaf91e56e27a770b52b3832bf1a5fb06136f77", "d4edd83ef1f3f5f2e328670165e7a0e3"
    ),
    # FWF_40F_3G4G-v7.2.10.M-build1706-FORTINET
    "533f67bd3ef3fae38568e4aeb0e433f434031f92e56c67d96e04eb915a153392": AesKey(
        "55727fa84f7f02c431bd8aa29f38d9f806d8cc82c174d45df1400460cba5f137", "98dc9ec242aa9bbbd2c52b7e5c2cc821"
    ),
    # FWF_60E-v7.2.10.M-build1706-FORTINET
    "edd23b7096ff25ef3a67a438421a97613ab17c9c92f8d79bc5def04bbaf64e3a": AesKey(
        "ac5dbdfb58ec4cb3ce22146c4477e777d4a3ad6516ade5e1215a7f590205cc19", "2a0714cc81d48f515e88e9b478f0c203"
    ),
    # FWF_60E_DSL-v7.2.10.M-build1706-FORTINET
    "1250eb2a459067c2632e98e6da22e6f4df376f0ea6931ca31f42d955b42a3dd0": AesKey(
        "14e4e465d19cbd07ac1e78c4cedbef77bfeeef53f78d42eda0cf7e78e47d22ff", "f0542d0f1d48518105600c7a443e134d"
    ),
    # FWF_60E_DSLJ-v7.2.10.M-build1706-FORTINET
    "a479cfad7fd6ad698aeed12ac53301170bf062cc629aeeaabec6e6069b551c2f": AesKey(
        "78f392a9f5c39e722c81938fd414002e601e1bf134235a26d32c49da7c1ee835", "2e2d601336fbd4458cf9caad68a0b9f3"
    ),
    # FWF_60F-v7.2.10.M-build1706-FORTINET
    "5fc028c1f8cfb0a56561e8791231eeedafcf832c580222941e9e7a2810f73458": AesKey(
        "2d5933268be9e9030c3e830334ff74cc2e8cc584658c2e35298196d89d8e6e6e", "2fde0b8f53054115a7ea18c8e3b26fd2"
    ),
    # FWF_61E-v7.2.10.M-build1706-FORTINET
    "9e26b51681af94266e57123360eff472766f920c1a05f755203c66d9b5f72dc3": AesKey(
        "e9671c43c9f5321410d8df782eb1afa77ec7dc5998b5cfdca80967499557405b", "592dfaee37f058092344512e3dfd0dac"
    ),
    # FWF_61F-v7.2.10.M-build1706-FORTINET
    "83b4beb207f54b48c65a273a197aef3fe0ac745014066dcc7e4b05b69edd4807": AesKey(
        "b1d11e7859f8e7e2647c25469f9b435c6167c819bf99165214ab6008e1d34837", "1b55f2bf714b465704b859206d22442f"
    ),
    # FWF_80F_2R-v7.2.10.M-build1706-FORTINET
    "fc15447a3a257dbae1921b9f3f5c63a92a7e459c00c8f830df665a8d33829e41": AesKey(
        "55324004d313633dee62a15076764d18a112bb3b5be958423d506408d32b54fe", "4316b7fcd5811ecc5258095ec8611bc8"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.2.10.M-build1706-FORTINET
    "07f8733c0336d5e323fd68503ff53110115940bd80d1865a9805c3938ce0b75e": AesKey(
        "bd6a3f11383a19a4e50b2bde870a59482b4f3625cca694da257f4c5d96f2dfc8", "8aa9c572bb53fcfee1dd9af55f9b81e4"
    ),
    # FWF_81F_2R-v7.2.10.M-build1706-FORTINET
    "706a849e1e0585c71d5b3b10b214057f31e90865d57c99f7b56a6ae20ac2effa": AesKey(
        "4bb93f9a102623dddb47d1c47b4a6e05cfd4dc4e629f7a4582ea4239853c761f", "0429290c7aecfccaee8d314bfdab19c5"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.2.10.M-build1706-FORTINET
    "03c37af93348d78cff10ee2f002c9781c635b2e8be69542a6218a38b8b11059c": AesKey(
        "a693b234262690bd99ac347e4874856bb2fec07184edaaa4e01925813b6595a1", "98e20759a4a45a6accee28aa4ee32f06"
    ),
    # FWF_81F_2R_3G4G_POE-v7.2.10.M-build1706-FORTINET
    "b1c5654a8fa2f4adec1ab9b1ace25f4676ad24b74d96324d7fd87455e08cd257": AesKey(
        "a979f7b16befd9b598fa5bac5c3536584c46034c3988b47cb09044106d7eadb6", "5d137e136a48da10e8c2f505108bdff5"
    ),
    # FWF_81F_2R_POE-v7.2.10.M-build1706-FORTINET
    "9c8e5baae899b7d9148260e78a29a69acc0e19f90fa4703700b10d208113334c": AesKey(
        "85af3325c724451787f6540b193e9df0f608e527eee05de5162ee01fffbebc7f", "e305330625f649000b7b52e999ac8f97"
    ),
    # FFW_3980E-v7.4.1.F-build2463-FORTINET
    "9e914ac65820861e742791edbb8835bf2f4c14296b9fbe9cb511f8b514f186d8": ChaCha20Seed(
        "58e03d75d69f5d6e5147f4a13fe7a83f1d617ddefdb3cc77be6fd906c8a94aff"
    ),
    # FFW_VM64-v7.4.1.F-build2463-FORTINET
    "74aaf377c35e3592ba130f1a3ad393014d74ced6494b6f8857195ec0a8b45d3d": ChaCha20Seed(
        "65f520a90e4903cab66ec7e4cd0221e102befc69cac2e0bc60410654fe66ac56"
    ),
    # FFW_VM64_KVM-v7.4.1.F-build2463-FORTINET
    "7060dfc699c8a22ebaebbf8e00dcb9f49cebd49ebea26c596057bede82625274": ChaCha20Seed(
        "1a24e0ff0acb765d783005e7f2df5ab49457e8be37e62c7f4697e6830fd0ad13"
    ),
    # FGR_60F-v7.4.1.F-build2463-FORTINET
    "3fe6c6f47c12ace17d760fedd093a52d8433fd717c22b9d7273abd451f5235a5": ChaCha20Seed(
        "d320ec6ce7b871a9b305faa970b3b2cf5f1ba13af5bf18e390f4f588fa841071"
    ),
    # FGR_60F_3G4G-v7.4.1.F-build2463-FORTINET
    "9ba66cd40fabdb93e9e8882c8e79c4334d8220fc96e87aaba9c70ede64751202": ChaCha20Seed(
        "84b021083224541d6918d52d1ee6a90e11a658f2d914fbb2de87c760204e683c"
    ),
    # FGR_70F-v7.4.1.F-build2463-FORTINET
    "d5fb5113b2b101c2474f3f7d960a4f2e2f1e3b8ae85bba356c5919a3fb0e6077": ChaCha20Seed(
        "144e746d4a9801cd44ff9fd574e9715b40fdbe41e38a4e812d507426538792ba"
    ),
    # FGR_70F_3G4G-v7.4.1.F-build2463-FORTINET
    "9bde52b04bdc8d3096128a1ffb2b9488ed5d9e6ce8ae17840f75e5eb00d646ba": ChaCha20Seed(
        "e3c519e7dbdaff8486be227279699b5c56d9a9b01b843f4760eecbecb07b9694"
    ),
    # FGT_1000D-v7.4.1.F-build2463-FORTINET
    "4021c0d941530758a25ec4ca850dc356c93ec6401609d486b57bf22e3c62d7a7": ChaCha20Seed(
        "eec5093c5ccab0db90cc584564280f67288a1669b51938e9d19e38f3ca1ac869"
    ),
    # FGT_100F-v7.4.1.F-build2463-FORTINET
    "14ecf0d92e95da1a21c120c1c8baf430fb60e15c516345df952a7528e27eba20": ChaCha20Seed(
        "e4c32af8720f9be39d856cf444ca7a63794736c0389322a766079b2f5b8d0afb"
    ),
    # FGT_101F-v7.4.1.F-build2463-FORTINET
    "9b33ed129f7a948852d9eb108c8d6cd1686485e20cc46b9ec94e27dbefb6c7f0": ChaCha20Seed(
        "b9c987d835c1e36b51d3c8ef88cf3c6904f9812c4925fb44db1ab2e691193ba2"
    ),
    # FGT_1100E-v7.4.1.F-build2463-FORTINET
    "1e2ad1f744610d80acdf9f5158f341aaf03940ea1a24d3865fa8c418d1f9ba2a": ChaCha20Seed(
        "cd0a1a52b9ff0543c81bf548791e3aaaaa4575e6ff8b5e024550adf558b78269"
    ),
    # FGT_1101E-v7.4.1.F-build2463-FORTINET
    "af9cca9d00369f2e9c3fd43318e08224b31387fbb6521184cb7ba81979611b10": ChaCha20Seed(
        "4e6cfa4e761341aef416148a7f108f9f82f79e2bb802d5818820ba3971d6a468"
    ),
    # FGT_140E-v7.4.1.F-build2463-FORTINET
    "c5a3d2863f8bb8c3ac92504f85c50753240a9d274a44c6b940d244a55aa99e28": ChaCha20Seed(
        "ee35b84503c7d6af634337f72f7e62a6718713de8892599c7eff255ee3d25091"
    ),
    # FGT_140E_POE-v7.4.1.F-build2463-FORTINET
    "fe4d009649cc9f13bb9d1c09207afcf4330b08ee4e6b13128834d8ce41e512d7": ChaCha20Seed(
        "fc3b303b0323cdc142da9364c52c3bedfb61378ea87057e8fb7e1377971bca60"
    ),
    # FGT_1800F-v7.4.1.F-build2463-FORTINET
    "b4e60cfffd1c8d93fee0089aaa3306245052363075f5d293d9123682a2ec97b5": ChaCha20Seed(
        "46e2baca603dff00948c93902d1e7e00af48fe7cea196180742b0c259a23f2ba"
    ),
    # FGT_1801F-v7.4.1.F-build2463-FORTINET
    "abfdf3d215accd530df81794af4a1f337645a678dea53a7e8b22a3cb316894ae": ChaCha20Seed(
        "d7403be5ad2fbfac7c509f0df82c30de2809eef30d2e43a908c0435349d0f0af"
    ),
    # FGT_2000E-v7.4.1.F-build2463-FORTINET
    "0de726e2cca0d1e51807ad33ffc83eaf29fc35033dceaa5960b3a48868fa7121": ChaCha20Seed(
        "dfd8cb576bf67ee8904fc873d1056e44999987b332ca3eb5f4178f2c692afbac"
    ),
    # FGT_200E-v7.4.1.F-build2463-FORTINET
    "ddfd9fe3cf4451b0686cfb87688c9b5216265518c781c7a3d3fef0c65ea49629": ChaCha20Seed(
        "20d6371aad176da5a3ec54317057af54aedeca91a57c27e8623c79b05a3bb27a"
    ),
    # FGT_200F-v7.4.1.F-build2463-FORTINET
    "9895c2409ef038c891abd398c54b0a9100b1986e1e32212f603e558c88be95b0": ChaCha20Seed(
        "ddb5cfc02729c9b3c84d34b1beefd1eaaec59c8e888391e6842ccdb84964e037"
    ),
    # FGT_201E-v7.4.1.F-build2463-FORTINET
    "3a0389757fe757c7a993891932a8c2d9cdfbdcc914ef2e4263b55f27cd58570e": ChaCha20Seed(
        "f61bc5abc2f710631cf8a9ec8fbf4025d73b12ca298a8bdc8b0c9ef3b729de65"
    ),
    # FGT_201F-v7.4.1.F-build2463-FORTINET
    "7482c176dfcbc15571c642875557d3e2d0b1de6cefc8c7abbc2ea30c7b1a0bfd": ChaCha20Seed(
        "aaf8574343e704cdb66e57c1e053d0dcf99e9760d4c37bbd168940bb9f7686ee"
    ),
    # FGT_2200E-v7.4.1.F-build2463-FORTINET
    "8bf8d88a0183db5805dcd47471feb2c4d6e2ed06c1254d87b1a11fb8d895e99f": ChaCha20Seed(
        "ce270a51243a69311a31f1976fca09ca45a43406bb5d6bcf6f5571ed95b298fc"
    ),
    # FGT_2201E-v7.4.1.F-build2463-FORTINET
    "ca65b3e70b4bc1c312214a0d4eb6dfdbc9e3c4f63937ca71d67ebb530eecf3cd": ChaCha20Seed(
        "33cb01dcf5deb69a0b066bcddbec4325383fd1ca36674d31196577a65b5a0fe6"
    ),
    # FGT_2500E-v7.4.1.F-build2463-FORTINET
    "94708dc9a18056e90b52bd362e09a701ae64fb3d685ac0ce9c2d1ff2dafbce15": ChaCha20Seed(
        "1ae197abfc3e0bc695c6f66cec16feec8a047ba81e6331f5b2305f5827e7bd72"
    ),
    # FGT_2600F-v7.4.1.F-build2463-FORTINET
    "4f4788d23877fc6e372209890b1f5f16665454e44078ebeb2f33330657ecdad3": ChaCha20Seed(
        "02f6c77e03d4e41c4dbb352ff484685abea36066dc4116eb20964bb3d5554a55"
    ),
    # FGT_2601F-v7.4.1.F-build2463-FORTINET
    "74d9d1bde6cdb3671600cb168cb88ec0006a1ec3f39232d6549a15dcad9f6c15": ChaCha20Seed(
        "ad83d65a776bd3dc72177f3f94d350f302c11da5722811f60a1c631a7b154fac"
    ),
    # FGT_3000D-v7.4.1.F-build2463-FORTINET
    "38b9eb63255c742f1177ca040f0283938e71a7443fce0c706cd1a862bd1439c6": ChaCha20Seed(
        "7fa957c49d1ddd9a5388af320a0ff164a15c98879b65ca3baeb71d1d9cddaebc"
    ),
    # FGT_3000F-v7.4.1.F-build2463-FORTINET
    "ac6e4d86c5a6e75dd0daad64d137e20f6d5032ac379800852c776f732d1bf7a5": ChaCha20Seed(
        "2b8796ad2aecc297670b87d186377b4b68a582742ee8d76a061ad073782b21af"
    ),
    # FGT_3001F-v7.4.1.F-build2463-FORTINET
    "99dbf50cef9a349d84b82be86f005383d49a27a4482902aa9cb7daf3e0a7f52e": ChaCha20Seed(
        "38c2ca82a2a072be48534e6e42c30aee9c7dcb89da3be2526ec2a7787c8d3477"
    ),
    # FGT_300E-v7.4.1.F-build2463-FORTINET
    "b9da7b2dce7fcfd398a4183f37fd045ba90862e7848edbf0022d8b8f22c2a8ee": ChaCha20Seed(
        "6c41010043bd2184b4f7e10546982bdb5199acf3af70ab9ffa6215593fe349ae"
    ),
    # FGT_301E-v7.4.1.F-build2463-FORTINET
    "15ceee06b491ff908f2a89bab2a531fbfe611d5d008144afb96bc3fc400999fb": ChaCha20Seed(
        "cdb8b811d4b3187e53c16513d2f3c22ef33bb43935100caa18db935487fb7ef0"
    ),
    # FGT_3100D-v7.4.1.F-build2463-FORTINET
    "0895859b227d1206eeb7e55153d80f1480cc5fa92fbe511cd92472b309cb73ef": ChaCha20Seed(
        "8d955d19bbb4f3562a84154f674b82ef5cac7f0265fa956c57da99bbf3f08283"
    ),
    # FGT_3200D-v7.4.1.F-build2463-FORTINET
    "615532f8300b58effaaf4a9fd82f0fe426ce63d5bcb9b0c7eda8acc100730c2e": ChaCha20Seed(
        "cf8c2702806e2c8e3689d5d586d78eaa19f639fafa5935bef5fecd9bf602d94f"
    ),
    # FGT_3300E-v7.4.1.F-build2463-FORTINET
    "6f22018cc11841591ce26722290b8aee7415df52d5277e72dcccdc94e6d56b9f": ChaCha20Seed(
        "fdf306d45431426019ad231bf14477a1378a518822baa95bc08bf8c0722f2fd4"
    ),
    # FGT_3301E-v7.4.1.F-build2463-FORTINET
    "e2d32d1cc59d30c254405472c70dc994969bf445f4e9dd86a7d9a8202e99ea7c": ChaCha20Seed(
        "dac62efb21b5ba0c5f71b0d0ab4021b949d38ca681d07cfe52dacddac0f2e590"
    ),
    # FGT_3400E-v7.4.1.F-build2463-FORTINET
    "e9ccaa34ad2d763b9195c1098792bc6401bf6401730ebb74df4ef3278f286667": ChaCha20Seed(
        "e9b86f290db947142c453368fe0420e23897a9879fb2e62ca6db5350e51d24a0"
    ),
    # FGT_3401E-v7.4.1.F-build2463-FORTINET
    "7c0e748f231d36385d785084faaae69dbb500579eb3f2163536e3a2571563c88": ChaCha20Seed(
        "7db80a8cfcf4f52aa67eab091db39d79b13c015cc38f259914b4af58e539f36b"
    ),
    # FGT_3500F-v7.4.1.F-build2463-FORTINET
    "ae1456508a6f6efd84b9aa23ded90011fa16f816c2875eca4240e884361495d3": ChaCha20Seed(
        "c7a910fe0aa385b7260f9b8b784ade9e9fac3548d0f0813110f3645ba696bb2f"
    ),
    # FGT_3501F-v7.4.1.F-build2463-FORTINET
    "004e745f971eaed55f86e64a093f903776d995c2ebcbadc76cfd0085ddaa4ea5": ChaCha20Seed(
        "f60318d3a3ae8444466f3248332daad990276edf8380bab9da6eec8090fba0b3"
    ),
    # FGT_3600E-v7.4.1.F-build2463-FORTINET
    "83c5076b8c22543d25ba352e2b7acc0d124352713c160a8df1a91a475206bb23": ChaCha20Seed(
        "edc09f4f9bd3f6799939241570f3810ff260f6dd78bda474da7c3b199e761df4"
    ),
    # FGT_3601E-v7.4.1.F-build2463-FORTINET
    "f7c7df101cc46c8c7f16529cfa1a02e775a8a199ae03ec9237d76eeca9caa4a1": ChaCha20Seed(
        "5d973972922801ed0a0fb36c494db8475a458d7c034de7434cc5eaa70149028d"
    ),
    # FGT_3700D-v7.4.1.F-build2463-FORTINET
    "bed972eb871857ec65aeed6f8552f9a3455af2d90304b61e007c94c99af150e3": ChaCha20Seed(
        "a8f102a4db47f52860d864a290eadb71831fd145a76007a5f342b6ac34ec4104"
    ),
    # FGT_3960E-v7.4.1.F-build2463-FORTINET
    "1670efa7d1bb8bfb16d563412947444e484822897dac4534159bb12c8b1b8ecb": ChaCha20Seed(
        "66f1ddf3f18f204d7f229d5afacfa93012b19877afbec6aa16fe260b73015ea5"
    ),
    # FGT_3980E-v7.4.1.F-build2463-FORTINET
    "4a52366afc2c38f5ef1fe7cae6a71fd836da8b97f1766b59e973d8f03cd31c80": ChaCha20Seed(
        "68bd4971713049c47708c331906d24a5b72de65db48c311b5414693be5c368ba"
    ),
    # FGT_400E-v7.4.1.F-build2463-FORTINET
    "1b263f3ac490baa5ceddb1978c8190b9be97e63149efdb757fcbaf3c4fa4df86": ChaCha20Seed(
        "d5ec0b794ec2cf79334f5cd7532a9f5c3a702933dea99f3926044751c1b8d50b"
    ),
    # FGT_400E_BP-v7.4.1.F-build2463-FORTINET
    "5d2d33f0046a90f763fdc279e5208dc4e44536934afdf8d0d2bfdfcc32b610b8": ChaCha20Seed(
        "c03f87c61fba6e88af8d88422c5336e2fa41375df67c506ac5d4249a93c2437f"
    ),
    # FGT_400F-v7.4.1.F-build2463-FORTINET
    "4b7c7db4e41ab956b573061eb7ea512083891506ffd7c103fb322cc93f27748d": ChaCha20Seed(
        "db0f0fed91caad0a2bd10ca9005f8fa0d582f10c954932e822decab636bbb4b8"
    ),
    # FGT_401E-v7.4.1.F-build2463-FORTINET
    "776564183e0b30200046cf1ca459daca02d8609868609ba665bb2eadc33427ad": ChaCha20Seed(
        "431d614eb0578aa325666d7f02f521774dab1a208b6f5a13955b7d5237202725"
    ),
    # FGT_401F-v7.4.1.F-build2463-FORTINET
    "032733f15f5de007a0527273f2da2446e7d7d19d8a6cc4915b355288fb0184b3": ChaCha20Seed(
        "96dc2f48e003a6b701070da199f718e838dc3c96d27c7b58375eee63d7584491"
    ),
    # FGT_40F-v7.4.1.F-build2463-FORTINET
    "959afcba72c55339f3b1c064f0caadfe5aaa3ff5ee481839a9365ef296ee6990": ChaCha20Seed(
        "024ee6402987a06f9900feedf296bcaea7e02ee0c95cb1efcd8626b52e304698"
    ),
    # FGT_40F_3G4G-v7.4.1.F-build2463-FORTINET
    "1e759e7402e2f2e4008cf686709a6b6a81130fe73cd7337b6ad071fad45e828d": ChaCha20Seed(
        "efeba725d9f31d36eb8f7072cbbe25c826211382a90115e54fa24111058017f4"
    ),
    # FGT_4200F-v7.4.1.F-build2463-FORTINET
    "b2b8eeddb699680d4075055603e023de001b713811adc1066834bc6167858080": ChaCha20Seed(
        "2f1da9cc0a6ed94ce19b3dc71f33bf4455ea0f700fa885525c0dc1f671818c2d"
    ),
    # FGT_4201F-v7.4.1.F-build2463-FORTINET
    "3aa6f18e76e723de7cd26c12c472bcb2e7deb684b7c15858abad0a37b372d889": ChaCha20Seed(
        "1faf997b2436b9db97866d1ee7a8881e01bcea10dc9f3f92647f94f54e7203e2"
    ),
    # FGT_4400F-v7.4.1.F-build2463-FORTINET
    "0ac8be2c308d549b33f2c1bc0b616928d7aedc21323078511fc4fb70c0251d34": ChaCha20Seed(
        "d14c3688457e89db70b180f69b705638a86a8e6f975dddb77a8eb6e8595812a9"
    ),
    # FGT_4401F-v7.4.1.F-build2463-FORTINET
    "22addbdbee2ee5248dc98a6bbc0d2db9702d1669cf9d15b1aa575c924c7b0e98": ChaCha20Seed(
        "41179c279c465b2fe3c2ce08ccd1dd5e7d030f453b47fc239a18d6338dabd901"
    ),
    # FGT_5001E-v7.4.1.F-build2463-FORTINET
    "965c90e8f95d0522da91cabed7989eac603ca16379fc285574e5358e11f6a96c": ChaCha20Seed(
        "30ddb1745d0a1cc3383db94446c3118d5871aa2006edd5918ca523e79455b555"
    ),
    # FGT_5001E1-v7.4.1.F-build2463-FORTINET
    "ee490b6fe3f7fe5f1cf9ac63a038fb2a22d10e1018b560f41e79a102151c248c": ChaCha20Seed(
        "1cd37d86fec7e473a05ce880308a039b6803c55e90395ded8e308c4e0f9d614b"
    ),
    # FGT_500E-v7.4.1.F-build2463-FORTINET
    "0d769cdf540211be0fd952f5b99d09efccc9530715f19ae5a01f689a8e0aaf5e": ChaCha20Seed(
        "0d5de34c5f85fa4fd71d0924e437afd7ab60085783e9e6d8b9cb575e1579aa2e"
    ),
    # FGT_501E-v7.4.1.F-build2463-FORTINET
    "7caedb9823eb944baf95bcedd533ae0a793e5696e5314368888e209a8c7c559e": ChaCha20Seed(
        "f3a7a69b0ce2df0430415044a06abd667657e62d5b44428b87c45a736d1f71d0"
    ),
    # FGT_6000F-v7.4.1.F-build2463-FORTINET
    "e3452be4fbf44aecf13154a5b32887487ab7f38254540f1b20d5214ab5cc4728": ChaCha20Seed(
        "3583a1dc9a922f4311c87cda9c70015cbd985b0380774907b1f060ff95f64d85"
    ),
    # FGT_600E-v7.4.1.F-build2463-FORTINET
    "7f1a910e9ba7cb24c6ab47a31fa79104971a113cf5d239f4c1cc2c40edffcebb": ChaCha20Seed(
        "883e4fa95d7987886c93557c8447175b24a660a5a1290683a48aacfa9fd2c16f"
    ),
    # FGT_600F-v7.4.1.F-build2463-FORTINET
    "79fc37828ecad0c1cde6590f144aef72321307c8d96c7a82bf8cfe7e4024bee6": ChaCha20Seed(
        "1fd8e44190bfbe258c0d1b5b6b25347b06b20f45aaa4ded33ef2a3afc9c57cd0"
    ),
    # FGT_601E-v7.4.1.F-build2463-FORTINET
    "f53b9ae511c49dcbd2449a76322e1792e93f6c17b4f9b33cbea9e96b869bf166": ChaCha20Seed(
        "53a175bf8e900bb4da6e6751bf8131bfa68e1dc86c51a741d36d515e778b417b"
    ),
    # FGT_601F-v7.4.1.F-build2463-FORTINET
    "5d4717a07b519708a78921e2d5fed1ef4106e05c70aea87cd5ae62db95df8e66": ChaCha20Seed(
        "a1ce7149709b76acff56692f8794543b055fb98201233995a8e915f876a84263"
    ),
    # FGT_60E-v7.4.1.F-build2463-FORTINET
    "66c343c4962efa27ccef4d9a981861964c8795a7ca3a5fe72498a07fc51495d1": ChaCha20Seed(
        "f63876043426403d3f8fb0ee8050da42d5dede98204d4833aa1c0bb3ffcb0dab"
    ),
    # FGT_60E_DSL-v7.4.1.F-build2463-FORTINET
    "faa2eb670304663b0d054fc77ed20bf76c7c53cf275b1c5d4579debd97d7063a": ChaCha20Seed(
        "789091ec6deb08f0253e363d84603521af6138273ab75797a34ab0aaac95ae34"
    ),
    # FGT_60E_DSLJ-v7.4.1.F-build2463-FORTINET
    "25774d522e673c29582bea76abd3a4f7a34dce78667295841ab47ad3e05e094c": ChaCha20Seed(
        "c6eeea8b4a73d0e9e8465046a1c8fa358818c042adb309cc147b27f0ec157d67"
    ),
    # FGT_60E_POE-v7.4.1.F-build2463-FORTINET
    "ad59eb8db2ed817bb4b428349897467413f227a736137512e3e6f3626dff6078": ChaCha20Seed(
        "1fcfaa3de3d4fdd95e0f09f9678f118b30bab82591e98b010d5e5757cdb1528d"
    ),
    # FGT_60F-v7.4.1.F-build2463-FORTINET
    "d5dfeae31429f1dce2a03825dd980368bb27418260e9bc35148b09f0519eb602": ChaCha20Seed(
        "b8af6023379d2c87eda698bdd5cfc5ef1433a0172413a0587b2a7d665877aa20"
    ),
    # FGT_61E-v7.4.1.F-build2463-FORTINET
    "dca7669f8e16bf8f97b344a21d8c0d9569743969db9c5ef3cb567edd5e7a4544": ChaCha20Seed(
        "7ec486fc274137367e7af93d2a79247075a0c919499c4a919310cd0d833c0135"
    ),
    # FGT_61F-v7.4.1.F-build2463-FORTINET
    "7c2451fe465cf4fb17a67d685e36bd06dcda8a756217fd0d35e5eabd00475a8c": ChaCha20Seed(
        "cf65cc3fb029999b864b337e55008071751f4ef1c9d9cc93edd0c7b48d48ff9b"
    ),
    # FGT_7000E-v7.4.1.F-build2463-FORTINET
    "bfb01d306b3ee2994039a1993b045cbfdb3e1d862e57cfe23fa856c1f16aacbc": ChaCha20Seed(
        "1b7145f3b010febeca5bd4aedeabded85e359f2b2c0d6c6484419fe203e3d759"
    ),
    # FGT_7000F-v7.4.1.F-build2463-FORTINET
    "08ceeb4474ce6f2d4b2187a1abd5c7d3927f51bb5eeb36ba642b69095de2ad90": ChaCha20Seed(
        "add1ff85bb4143f74b6d13e96ec0ddff60c575d89c67d1b083316605d565e59a"
    ),
    # FGT_70F-v7.4.1.F-build2463-FORTINET
    "9b5907daa62eaa2da874cab078263a08ef1fd8a4960fa7131e8ca89e60effdde": ChaCha20Seed(
        "cfd25d831820e1f9c7314192dd13a88bd85f29f5f3c18d400476f54a58d13e79"
    ),
    # FGT_71F-v7.4.1.F-build2463-FORTINET
    "030596f32f995552487c9daa851b03c139706202ad4319a6473848c289515775": ChaCha20Seed(
        "f50ccf050f7819aafeb71788a22ad5e66fdb4832cfa6b4bc3cd275f1e3b44929"
    ),
    # FGT_800D-v7.4.1.F-build2463-FORTINET
    "04d802c0eda386402fbfe290a419b9ebd3bcb65403af0cca4cc55b0b1c491305": ChaCha20Seed(
        "9bb068f496ea2e719a82091332cd10ab4a1ba5fa574d1a2a42bfed587f4ed465"
    ),
    # FGT_80E-v7.4.1.F-build2463-FORTINET
    "241ac2c4104e75a283d414be93e5156c864d6c749521e14dd7fef9d67357f1ad": ChaCha20Seed(
        "8179d55acb80bd3b06920a818b7c008581d80146992d9c2efe2b1dcb4eae9ae1"
    ),
    # FGT_80E_POE-v7.4.1.F-build2463-FORTINET
    "1f43c91fc76347bbbd4ed95a734c84d70b85e73bbdb1e3bf1f449c3e258eb363": ChaCha20Seed(
        "ee5e9a12801cd60321bc4e1abad9025954cd0e8ee438dfdcb344f31fb6f2dd14"
    ),
    # FGT_80F-v7.4.1.F-build2463-FORTINET
    "19d9d8349de1e768813b44490f2381bab534e44474200e11b853197f52bd1485": ChaCha20Seed(
        "5a34da5b906f7a6243e1c8437ad83ca28fa2a738017025e3a34670afcd94b484"
    ),
    # FGT_80F_BP-v7.4.1.F-build2463-FORTINET
    "f006979ba01abfa8fb789e49d21a050d5d73201606672e6697bcf8d3a323a1ac": ChaCha20Seed(
        "41000e2fb91bd85029405a6aa587bfb637d56087e39bb9fc791fd950714599ab"
    ),
    # FGT_80F_POE-v7.4.1.F-build2463-FORTINET
    "5b1a30d580dd2259c97dbeadd2bc666c6b35a08f3a15256fd67a5b0634df8a38": ChaCha20Seed(
        "62c42edc6643d6c7cc0709ff6592878f7a6a6e21c4bd2b4488752aff130479df"
    ),
    # FGT_81E-v7.4.1.F-build2463-FORTINET
    "1b5a34a14c59fa4a1eabc0e85f535c6eee207f4c70153a408e642333b65ee3fc": ChaCha20Seed(
        "77fd00ea6aab837d170be4af3198b6de67c32b397549c6879e492e768c0cec4f"
    ),
    # FGT_81E_POE-v7.4.1.F-build2463-FORTINET
    "ef86a35f923e05d3502d3b0f15c1fd5b13b202ca81da7149cba5a4462396d8e3": ChaCha20Seed(
        "5c370df524abcfc36e59f25e86822fed2dd98b7bed0611bdfb0f66f7492c0209"
    ),
    # FGT_81F-v7.4.1.F-build2463-FORTINET
    "29f15a16a84f27c1d4151f6e7f672dda285c8f96380a0024de1235352c1145fe": ChaCha20Seed(
        "0091edabb21fca284ec67191c8f9552854369cb99371466a6e77804145a910da"
    ),
    # FGT_81F_POE-v7.4.1.F-build2463-FORTINET
    "c1aaf0dd54884f0159229a957ef8a504413e0b9105a3071412d3a5f73d3ccb5f": ChaCha20Seed(
        "b51db6649bcaec814ded0ffcb3e75f7ad11ac6b5bd56fd9b95c1a4b3b10f1ca7"
    ),
    # FGT_900D-v7.4.1.F-build2463-FORTINET
    "7cd5641f068187853d01cdf8ab7798b95018d522b08c7bf02e23159da5f3c91b": ChaCha20Seed(
        "2e4c11313eb24f36e1c190b005438104d1aad8fd737505de30638fc865d5b96e"
    ),
    # FGT_90E-v7.4.1.F-build2463-FORTINET
    "7950066cb41c57b817125f8f189a214b6670e2fcf4e878b885c73e82d7857739": ChaCha20Seed(
        "19d7be8c07851b4765c953a107bcf223d4ee21d48f5a6557549af8fa796fe518"
    ),
    # FGT_91E-v7.4.1.F-build2463-FORTINET
    "bd36ed29e02e10f72c01c7b978d7a08740d1046278a2f6b1fd322b02e6b15560": ChaCha20Seed(
        "c235155077ea44fcbb522955458912bd8f65c593c1d8f68bd2a4b866d3b50e1c"
    ),
    # FGT_ARM64_AWS-v7.4.1.F-build2463-FORTINET
    "eb573e7e497fcf3dbe5f896af53b612d233467f20d2cea144796daf55212bdce": ChaCha20Seed(
        "2aff45af42a414bf98322967c1af3e3cc1328c5e60ce4e892f8d17b341b7a9cd"
    ),
    # FGT_ARM64_AZURE-v7.4.1.F-build2463-FORTINET
    "dd427fe007668398e8ece8ae383a902076c3217232a2041059b4b57ed9317bd1": ChaCha20Seed(
        "c610b7119fd1564c8563f2bc364f3cb624d383e54150a30e9973ce1f973d0046"
    ),
    # FGT_ARM64_GCP-v7.4.1.F-build2463-FORTINET
    "6387ea04b0ef62276d71e5d6691e934ee0231246da7c9473c5a22abe477ca93b": ChaCha20Seed(
        "f2852af43ba8e2bd2f7007a4bfba65359400470e359f19e77151ab791cd1d677"
    ),
    # FGT_ARM64_KVM-v7.4.1.F-build2463-FORTINET
    "2de8d8e3067343cc9a5b33efda19bc602bde5ec8cbf64b44373fc8e5a54a7e11": ChaCha20Seed(
        "96932b0825425183d2f27e848eb525d8a5cc6ea9e169c8ceafd220c176657679"
    ),
    # FGT_ARM64_OCI-v7.4.1.F-build2463-FORTINET
    "889b3a98ac161859805752324b408334f81e484c5a0ce699b19424baafd84a87": ChaCha20Seed(
        "d97f2ba3e015caefdf1c433f6eee93296fa685efd60e0ecd5d5ecab19b89bd59"
    ),
    # FGT_VM64-v7.4.1.F-build2463-FORTINET
    "a008b47327293e48502a121ee8709f243ad5da4e63d6f663c253db27bd01ea28": ChaCha20Seed(
        "366486c0f2c6322ec23e4f33a98caa1b19d41c74bb4f25f6e8e2087b0655b30f"
    ),
    # FGT_VM64_ALI-v7.4.1.F-build2463-FORTINET
    "2043806c25d8de1edbe8c74e9c7414b415d10efd02368e8730b8bf8ce504ec9d": ChaCha20Seed(
        "6b3e00788380e9411750189079da5a4b53649a6b8d3e2b5dd67993e2498d119d"
    ),
    # FGT_VM64_AWS-v7.4.1.F-build2463-FORTINET
    "bf1e14e0daac5db5bcac2b26df49b1d5f6abdeda5155f1b759192fd7ea1db699": ChaCha20Seed(
        "dd83e3d39d5a7d0ff14efe63970fb814e3e3c9e2ecaa17565189150e9a63b9fe"
    ),
    # FGT_VM64_AZURE-v7.4.1.F-build2463-FORTINET
    "fe803c4d7e21662c9fd86b6daf6d4d27bf290084971fb0d6d43415ce16c6d413": ChaCha20Seed(
        "ca86df4fba6710d0c82d2afcdb526f4eb59541a58835dc0f33a17d4d39d01792"
    ),
    # FGT_VM64_GCP-v7.4.1.F-build2463-FORTINET
    "c70e0acab2acc0d287330993e83f81f81acd67a8810f85934ffd97b4eb6c6817": ChaCha20Seed(
        "c37334e6c13bc368b661b4698a5b6992acfbc502ab43736b1121407d064b2cc8"
    ),
    # FGT_VM64_HV-v7.4.1.F-build2463-FORTINET
    "10c165bee8d5c2acaae3cf6299ae70313c9b371b2cefa06c30a347092fbca064": ChaCha20Seed(
        "673f9672d4e4518e4c1b0676d2cf7bb73745130d223f55ae5706db7aeb7ee0cb"
    ),
    # FGT_VM64_IBM-v7.4.1.F-build2463-FORTINET
    "f4ff1e81eb7293e3b5abf228c15f96233ee51a9241edb124f0fde664641cf7ce": ChaCha20Seed(
        "e5d2ffaf5f9274d96ef7131b3f70b674a2356757a2b450b30ca3ad3db689b534"
    ),
    # FGT_VM64_KVM-v7.4.1.F-build2463-FORTINET
    "26af699026a09a72be0c243f55b933f4151aa39437a7f697e018e725e2a1110b": ChaCha20Seed(
        "811af728288a99ae490366de131ac4997abcc75d9b0ab5440ca832fdd5c5f6a2"
    ),
    # FGT_VM64_OPC-v7.4.1.F-build2463-FORTINET
    "030ab163586fa24f86712a7d2f2df7c2f9395049fabdb8abbfa078ebab20469c": ChaCha20Seed(
        "3a62d003e0f61ff73490dac0400cfa5f7543c23800cef99e00a207bceefa0182"
    ),
    # FGT_VM64_RAXONDEMAND-v7.4.1.F-build2463-FORTINET
    "b412991fdf45604b60e1640a9eaa11453027eca4928e4b6edf996dad9350ce96": ChaCha20Seed(
        "caa7b0a76a8b6271c7fa2b2cbbd869c87bb66536adaecf20c6ce033c2f25ae28"
    ),
    # FGT_VM64_XEN-v7.4.1.F-build2463-FORTINET
    "567d01ca73f65bfc02141845692464bf08885963f7ec923bbca2c9aae662193a": ChaCha20Seed(
        "2639a131fd458299af4be42a691bcb123bade8b314422d9e6084bb410b39315a"
    ),
    # FWF_40F-v7.4.1.F-build2463-FORTINET
    "1a1bafb78192ec9c56242fae30d58d402b9b4615ee9fd694ad55138ae25312d0": ChaCha20Seed(
        "33feab7920bb0e94445d814b5cd1c3c7ad623c0218673c7527d0578c391c077b"
    ),
    # FWF_40F_3G4G-v7.4.1.F-build2463-FORTINET
    "d6780265afab2535ba48eb1e5adf8fa643282f8ffad06eea500b09e78b21b1fb": ChaCha20Seed(
        "49f75204a9c11b9b87ce8a32aa1023ba882d771ed39cefe7c5cd218bd7d9801a"
    ),
    # FWF_60E-v7.4.1.F-build2463-FORTINET
    "21fde5b8c698eefcb9e9f43e7a4eca403c039b3a5598060b75aef2353dd6454c": ChaCha20Seed(
        "bf3e11babe4751b888784c47363a2a199dca08b7e459d014786638f0f6b11db4"
    ),
    # FWF_60E_DSL-v7.4.1.F-build2463-FORTINET
    "df59de5847053209abce36c940d533556200715ae4a88701f9041c844fd51b36": ChaCha20Seed(
        "c6f4a0ae98c6b35e135704ed017c6f09f81503fb2c92dcbf2a266d8791bf1a1b"
    ),
    # FWF_60E_DSLJ-v7.4.1.F-build2463-FORTINET
    "5878a52a3ab83ed5c9c04dc7335fe6dd3576321734448359007f1fcd68d39789": ChaCha20Seed(
        "ee971994fb34755ff45a932a18de1d6352808ea5ab126e18830f8b410c800791"
    ),
    # FWF_60F-v7.4.1.F-build2463-FORTINET
    "96efdd36eaaf0778601faf73b4afc27a09380512c5955d307c399aa711ac4f48": ChaCha20Seed(
        "f0088862942aa248ef8fb17fdf24c4ab47fc40e9de245da613781f7ba28ea97b"
    ),
    # FWF_61E-v7.4.1.F-build2463-FORTINET
    "5127b8bb64a06a189a686739c9ac2ef82304299192007683dc0aea4a19970617": ChaCha20Seed(
        "3c2ede28d15ffe3bc92026c0d34532d070327a090ad60c73a81328f21d242faa"
    ),
    # FWF_61F-v7.4.1.F-build2463-FORTINET
    "bc0d484b65d89c91d47a77fa8e2554eda4affb940788d3f248b36e539e77afe3": ChaCha20Seed(
        "8f199ff76c08b85db86bfeaa86e6cacb904d2a4d83a4aa16a67d5db15165b1d2"
    ),
    # FWF_80F_2R-v7.4.1.F-build2463-FORTINET
    "102c5d81d7f8f3dec96f48501b630275c492cff73733631299494cbc8d9b5d74": ChaCha20Seed(
        "3f0eabc3a1941bdfe83dd57903b5710b9fb4e2366d0bef6b4911dbda843cbc6f"
    ),
    # FWF_81F_2R-v7.4.1.F-build2463-FORTINET
    "38af187eb44f4dd0168d8f8b9c81abe006a8860bbde65749ff7800c1b7d74565": ChaCha20Seed(
        "7f39826fadbb7e952f3201f2aee1379d7cd6200140a088e09aeef2709d154192"
    ),
    # FWF_81F_2R_3G4G_POE-v7.4.1.F-build2463-FORTINET
    "42ab26aedde4fec68b62e4b593924d5bea927bc6466ae6c7c3a5356725185323": ChaCha20Seed(
        "c6b491a11cd564cb0c48a3f97333ed9c653f6ec939c2023ba7c4f64d88c5cfbd"
    ),
    # FWF_81F_2R_POE-v7.4.1.F-build2463-FORTINET
    "29279f4a168a7cb29f726962bf97af3bec26cb932e36d4dfa988ec9230a4b30d": ChaCha20Seed(
        "39252fe3cec846b2dbb3b5f0070b099089d3f0cf7f5cc616904027262bd6e267"
    ),
    # FFW_1801F-v7.4.2.F-build2571-FORTINET
    "d719f7fd533d05efb872907cf3711d0d895750d288b856ce70fefecbd7ace482": ChaCha20Seed(
        "71326d86adfd809fa4a7b754810be9a6552d9091cdd0ebe654944fd3e159eacf"
    ),
    # FFW_2600F-v7.4.2.F-build2571-FORTINET
    "0119cc4269540cbcd71ca054334e8856d4d22164d54459733408d46eb4bdf038": ChaCha20Seed(
        "c26422591c61d94810112c48472559efd22a795e4b14f81821559019674b9eeb"
    ),
    # FFW_3980E-v7.4.2.F-build2571-FORTINET
    "f62a56569f39355edff88412e3a572a064308a82c328bd276bccd3d41976563f": ChaCha20Seed(
        "e384310901d031ac678421088c81d69ced4cfd5c693bd0c6262528ea74385db6"
    ),
    # FFW_4200F-v7.4.2.F-build2571-FORTINET
    "94d96415062579eb9b311cdb9b94bf533374453793b4d9e62cc219d04aa8a00b": ChaCha20Seed(
        "911ee4ac1e798b18139a0f24ead468190f6e70ad19276723bdbea3b7e3f448dd"
    ),
    # FFW_4400F-v7.4.2.F-build2571-FORTINET
    "e31d269f28b15d1ab061d19a2dc2692f963c1997ff2e1c21bcbf1dadc9afdfc9": ChaCha20Seed(
        "20206bb7b2f67684c73e61b33554de860c6dcda3453f365b459717aba848619a"
    ),
    # FFW_4401F-v7.4.2.F-build2571-FORTINET
    "8db6ee76a3e488c259b067f701e2249207a751f3592b6c378b16a7b16570350d": ChaCha20Seed(
        "a429f4c6774c5f7aa9e34730775047cc24cf0f379898498168d4d110bcd810a5"
    ),
    # FFW_VM64-v7.4.2.F-build2571-FORTINET
    "5fb3d57c31e761aa3da3b8a61a3688f77a7940efd47955d5a134a297d523cf46": ChaCha20Seed(
        "d89857dce1809dffee19ee773249b5e2317a00e4f70de46407b3193c08e96ddc"
    ),
    # FFW_VM64_KVM-v7.4.2.F-build2571-FORTINET
    "ff962ae3aefcc47b8f5006c9f949258068101426da6952cf5948f074391f987b": ChaCha20Seed(
        "135cfb746f7131b3c4975aae05900803a3d71894b64a6e4cbe03e453e784f31f"
    ),
    # FGR_60F-v7.4.2.F-build2571-FORTINET
    "dbe2d8134a661bab6ba277f07675f650f49f06fcdb192252a75039e6d8f7d593": ChaCha20Seed(
        "c43220c28b6e17e30e33fa3570a5a0022bdc109c09a1ae9384a0832a0699a23b"
    ),
    # FGR_60F_3G4G-v7.4.2.F-build2571-FORTINET
    "8181edaa59a447d0f9ca8743d99c11a4f833732bcdd42147364915bd91cb78ae": ChaCha20Seed(
        "6db1ececa37141697c57628dc0f1152a1070187c5e930ce4e358fad559ddb317"
    ),
    # FGR_70F-v7.4.2.F-build2571-FORTINET
    "c3360f4c03c696e04c8a1b711847f6556110bff6c78b17264d7fb2f8f533e0d4": ChaCha20Seed(
        "50abb6d5e96ae397b626f3bf0ecbe9a67511f8b19e5037583dd8dadbdc685614"
    ),
    # FGR_70F_3G4G-v7.4.2.F-build2571-FORTINET
    "cd12f1b25a21cce69652f1e63848f342585d4b3b90a296c845757d30594f6d73": ChaCha20Seed(
        "e6767ff6445d043a00c72fa198c7c6c058dfddb5f95bda06292469f25cea1072"
    ),
    # FGT_1000D-v7.4.2.F-build2571-FORTINET
    "076bfbdcc2710e622182f07a8a2f39ce6f87c357eec4aebaf54c5a76ad522245": ChaCha20Seed(
        "8ea186dd8339181fed2234621bd9a27c88c691cd1ef37e829ebabba66e9f2447"
    ),
    # FGT_1000F-v7.4.2.F-build2571-FORTINET
    "f8c55cf34f7264ba2e97816dc073162975ad30aaa9722694764573138b8307b9": ChaCha20Seed(
        "7062a2e903e5b08ab6eefc325124588c005ec202afd786de152f7515257334b8"
    ),
    # FGT_1001F-v7.4.2.F-build2571-FORTINET
    "74561888b50a5f1e5606d8be1158082e0820101448306caa32df0e283f188525": ChaCha20Seed(
        "41b43c6edc3508c0acc2ad67e9c0f97010c2d509f1ae8cdf8a7c6777cabf26ae"
    ),
    # FGT_100F-v7.4.2.F-build2571-FORTINET
    "63d133ebd7aa7fdc6be5c957ea6e160f497140d5b594535c6ad5a64be571b2ec": ChaCha20Seed(
        "e68bc852fbb8d17f6b280925b270b375385e847e55985739236d9be0d6b49f4c"
    ),
    # FGT_101F-v7.4.2.F-build2571-FORTINET
    "d9e37266c18d0239dda5dfe212ab85aae27c9291fa7dc7529edce5eb40ac7043": ChaCha20Seed(
        "7a4d041aace35f15c5d4466807035e7447d6511729ddbcb53052ce1476234b65"
    ),
    # FGT_1100E-v7.4.2.F-build2571-FORTINET
    "ab0e974e922816840d585e92bf221fcef7d39119fc18c18f3a274e19e4f253e2": ChaCha20Seed(
        "a6f4d5d5555ebf9d0fcae4fe98ef95f94212cb46a88a19b7b2607f57867b473e"
    ),
    # FGT_1101E-v7.4.2.F-build2571-FORTINET
    "dcd9ec7b4117c0a423e98a6b2dd610ab17b50d9fef0b2dd3b949410538ac1ada": ChaCha20Seed(
        "5bc475f36bb6854ee0c2b78e8d0a0ed176397a7e71e6a72fee0e57b8e05152f0"
    ),
    # FGT_140E-v7.4.2.F-build2571-FORTINET
    "c0686e4230b318e706d7bb8c1b1550884c9d371b7730094d1af80787b08a4533": ChaCha20Seed(
        "b522f72f71741abd4a4b666ceec1ef15679c38ee8be8dc16b23e764643522975"
    ),
    # FGT_140E_POE-v7.4.2.F-build2571-FORTINET
    "73581118d9dd55225fe570d7adaf1dbe94c7b0b75a2a62b8dbe93267f9887f04": ChaCha20Seed(
        "32ad8e039cd90502b76d0c9b43150af59735d90eb59928ada6d5153407da41fe"
    ),
    # FGT_1800F-v7.4.2.F-build2571-FORTINET
    "e38da7c8f99553b68f40dafd08e78eb02d585bf6bc35c495fcbc0f8f193d1ec9": ChaCha20Seed(
        "06768bdd841113e7b2df7d91ac38638f81f7c3bef551bd0934d19d92310e7740"
    ),
    # FGT_1801F-v7.4.2.F-build2571-FORTINET
    "07dcfa8a295137f401aa258ca4c3d68dc16cb7591f6c413626c8ac68e9bd5882": ChaCha20Seed(
        "8ac42353ad24ed86c2f49eaeb8dc912c30732d9b6f6c4b89b33725dce2b1e9bb"
    ),
    # FGT_2000E-v7.4.2.F-build2571-FORTINET
    "ebc93e63aadaf46895dca67923a50b301da3a61d386d2f4a0e670859e1af6f50": ChaCha20Seed(
        "294e5ee185eb663824c9be6e2733383387686adaae71e8c1acb11a2d4dcde8c3"
    ),
    # FGT_200E-v7.4.2.F-build2571-FORTINET
    "347c8b6b20f002ec8e34bea2427b8d0f25ec44c2c000c9602676763c0e5eeb71": ChaCha20Seed(
        "94711929a3ecee9976798511fcbdb799eea823772947e6db9ac5429ed8080bbd"
    ),
    # FGT_200F-v7.4.2.F-build2571-FORTINET
    "f818e0d235b39fcad80b8a21829e6c52c9be7852c236a1fa8042835d0f28712e": ChaCha20Seed(
        "b7ceea44378f5b98beb87109f7124b84d6e7557aee011b68028a8fba4f56bcbc"
    ),
    # FGT_201E-v7.4.2.F-build2571-FORTINET
    "940b522d8d0018785aef3c59dda075790959d769d2c670a50c027ab13abde951": ChaCha20Seed(
        "e9d33f10bcef951960a1d7dbfd13c81f9ba9674d24fe30eb5030a49c3300874e"
    ),
    # FGT_201F-v7.4.2.F-build2571-FORTINET
    "9393d69dc693c76e0ddf19fd0a00a08e30d0c7eaff7a9977a3a7c6becd675b05": ChaCha20Seed(
        "41c4d71273d6f10dc39aaacd0b1f7595c3d9c858f3aee273b4f673b843d969b6"
    ),
    # FGT_2200E-v7.4.2.F-build2571-FORTINET
    "ae09a4e905eb1b63bf38bfb08dde09efca5f2ae52a83a9b5c06d51d187d82e6b": ChaCha20Seed(
        "46cefc1f75fc7f4862326cd92a1cebb8ca1535b269dd405be5e4f54083a9f98e"
    ),
    # FGT_2201E-v7.4.2.F-build2571-FORTINET
    "0dcb0964661d59fcdc159e2e825e060d736cbf85c41ff403e7b4e940806f4760": ChaCha20Seed(
        "f7b572a57b59669d36a21b22c46da6df82938c7ce12d27fb3f388faace0ce202"
    ),
    # FGT_2500E-v7.4.2.F-build2571-FORTINET
    "b7417971de2ab8f006846b88b69ad8aaecc7ab9dc21241833fa8a0f85aa21147": ChaCha20Seed(
        "13c93243d25bd34a12d1476a7e8f7ef4c27253aff41d0aaa9d744f3c5a9a4f50"
    ),
    # FGT_2600F-v7.4.2.F-build2571-FORTINET
    "95b5430be956fa33fab5c49fdd8f53f1905f58d1cf3cbff890c0f78f0372ffb4": ChaCha20Seed(
        "fec35189cb0b857af94cbd7b03e2bbeab9003f86ca0b9f2b975401c556804416"
    ),
    # FGT_2601F-v7.4.2.F-build2571-FORTINET
    "39bb0a900d45857e081647854c2190533100526dc7ade21cbc18cdd0d886dbc4": ChaCha20Seed(
        "7e811b3cb20c5eb25f76e0887b4e533f3cc335a35cf8ca159395252b86eb0e61"
    ),
    # FGT_3000D-v7.4.2.F-build2571-FORTINET
    "80d084e0679001f513f834b23a35defb9c03d19da241e73539ba150b20b91f47": ChaCha20Seed(
        "93acab9436e2e3eaa1c7b7af37e6d2304da87278316553b3f9992a9abf560d61"
    ),
    # FGT_3000F-v7.4.2.F-build2571-FORTINET
    "6f25e41c905ab057e0c3df6be416c2303c6bc3958ce98b1cca9431c3c3c8bf76": ChaCha20Seed(
        "eae0cedae4347b4c2c531bd53bf1723fd03b3cb67628122af2182919349d7b67"
    ),
    # FGT_3001F-v7.4.2.F-build2571-FORTINET
    "862590710ad5b4ad6447681982979cc2015730bd9a3494056d6162c4ba870225": ChaCha20Seed(
        "4c8eba66451123945bbf3a6bb30f8879fe111f0becd9dca07ab0a5979ab3c978"
    ),
    # FGT_300E-v7.4.2.F-build2571-FORTINET
    "cdcc05202cf3f8f838f85a75bebb98795f2ddaaa8ebc27d7a7d36e9e6d77e4db": ChaCha20Seed(
        "41767b3918a2e27ba92fb5652847efe2bbc5c754e70526548bedf35c8b4a55eb"
    ),
    # FGT_301E-v7.4.2.F-build2571-FORTINET
    "29973f634ae45118d9edd044fb5493bb5a01fc36a5bb83e3c83f8880b58f02c6": ChaCha20Seed(
        "c86e570f298468c15cd3bc2ec91f2f64fdbe23b46eed85484431b1da5c2adeb2"
    ),
    # FGT_3100D-v7.4.2.F-build2571-FORTINET
    "2c98618e5a861454fb834dca8a2562b39ab77c1a38c94e86113312588442e579": ChaCha20Seed(
        "d391ec9515c6b729c86a0f3cf75d3450b143c8773d29a4545ea4b2b1883ea239"
    ),
    # FGT_3200D-v7.4.2.F-build2571-FORTINET
    "0e6e452b54c14124a5cb218cc81e6cd76526b182e10eba122e8d262fe01b1076": ChaCha20Seed(
        "338c4c0dffa7192ad02bcfd4f1ea87eaf4903188d678252b03b94e28199a007a"
    ),
    # FGT_3200F-v7.4.2.F-build2571-FORTINET
    "1750eae5f320c88e3b6a939a193117650cd066641c40f252c04432525690b32a": ChaCha20Seed(
        "88e4b8e96ad822b9cf6861e9d9aad31c82d70978e4ebda255229b1d97f6b65d8"
    ),
    # FGT_3201F-v7.4.2.F-build2571-FORTINET
    "858d704d29e321eeef39e99cb2f0ac35b35530fe9b665bee8b099769d3fc9f26": ChaCha20Seed(
        "a7351af32ca979d6c859b9efe1b2d19b516f000e7c7c1fe97d9776b732ddceaf"
    ),
    # FGT_3300E-v7.4.2.F-build2571-FORTINET
    "4dbe62b952ab27a3f78e7adc17ed82607d79ecb0ef23cd62c4f9b0125f669a30": ChaCha20Seed(
        "515515ed4a3b1ea27a7aaf79a732e3de693c5e9b4276bfc9eaa1816d86d16394"
    ),
    # FGT_3301E-v7.4.2.F-build2571-FORTINET
    "1a527c837707a63175ae7a483c7bff00b030a1edc52be61f97173192d7e7bd0b": ChaCha20Seed(
        "d6ae4d8259cfde3413669df36846a94f1b0035c8550a778a74184cb7c2a0da70"
    ),
    # FGT_3400E-v7.4.2.F-build2571-FORTINET
    "9fae035eb9b17960bb644ff8ffd2f1233145bc88b551368fd82cf27caf1ca0f6": ChaCha20Seed(
        "3fb6a043d5b957ac9f4e46c851d059475d6113a9a98c45da4c73ec58cfd5e432"
    ),
    # FGT_3401E-v7.4.2.F-build2571-FORTINET
    "ff68d1fe60f5e92d5e66beb4ab98d15207d5e97f4c28c51b7347b1f9ba2854c8": ChaCha20Seed(
        "6ed774fa730f422aaf20b5c8d1bb135cf8aa629ccb3b245d71826b86fe8da142"
    ),
    # FGT_3500F-v7.4.2.F-build2571-FORTINET
    "0e353cd2c73450c8a36766f73dd200d35fb8211b6540585a0e3b312e7d87927f": ChaCha20Seed(
        "00146e4517e399ee7f9fc4dbc492d461762900fcdffb9af7e7f693d9d9d6eff9"
    ),
    # FGT_3501F-v7.4.2.F-build2571-FORTINET
    "06ec4c29f0270fdbd97ba99a4155469f3f6b5da4faad9a55f0923c856ca530b2": ChaCha20Seed(
        "0542af7a6f6c179efe21dff3f2b371b1346ae4617acdc81b2f803faf8276c72c"
    ),
    # FGT_3600E-v7.4.2.F-build2571-FORTINET
    "027dd95ba9f99ce2968fed67aa38b866e5ef424891d95e7e2620a6443d229406": ChaCha20Seed(
        "b425e05acb080efa6a28a9f93590cef55d08b11c753f5e1a447b4ce8f8dbdd92"
    ),
    # FGT_3601E-v7.4.2.F-build2571-FORTINET
    "c00934941568f16926bac683982abf76f99506cd3117ce205a7fcc2025021da6": ChaCha20Seed(
        "e32f2f3fe9f47a5b5b19919c7c0f4c62159809da9609a0645da62e49f260f318"
    ),
    # FGT_3700D-v7.4.2.F-build2571-FORTINET
    "e163b9a6d7f60bd07e192abc28ad5b43fb6eb992a2a76ba25010510fc31d7d8a": ChaCha20Seed(
        "8eb2ceba4ca5cf2e1785c92626b060cea335269e59b232022b7b401c0cdc73d8"
    ),
    # FGT_3700F-v7.4.2.F-build2571-FORTINET
    "e00fca729a8d1dccc67f491c35fc87f5d8217b889608a1d241df3d7d420eb5ea": ChaCha20Seed(
        "8d08d3d3853d35a5a270577bff1f0e64de2ad16f1970d1d9823a73333bb2236c"
    ),
    # FGT_3701F-v7.4.2.F-build2571-FORTINET
    "b467b86bf37179b47d2249c87bc92e952bc9972cdc1ddf47559145371f816c03": ChaCha20Seed(
        "31ccc0c474d5dfe10343c7209c1c75b5dc3d3e09433da092100d87ccaaf332af"
    ),
    # FGT_3960E-v7.4.2.F-build2571-FORTINET
    "dada3015aa28e6a207daa287ef4e37ddc6834cbad5b694c7c0337dcbd351d75b": ChaCha20Seed(
        "bcc983630b9beccef3e178301b81b0f5e41088f618bc56b883f13353a0d64c5f"
    ),
    # FGT_3980E-v7.4.2.F-build2571-FORTINET
    "cba0d61c3675703e3702fe1af2a0ac2820d3332ffb8f78db1f226011cb94291e": ChaCha20Seed(
        "43b710ecedb024838167562bdd60a1d6ed2eafca719036e032ebb061c198b1ca"
    ),
    # FGT_400E-v7.4.2.F-build2571-FORTINET
    "f3c1d335e592c4397b69b63a5b7d336befffbdc62bd4974f00209c10133c2bd3": ChaCha20Seed(
        "4cca4b550ee4ef1f124b93fbb62cf59ede7d358c93ba8ee6df4e603722089af8"
    ),
    # FGT_400E_BP-v7.4.2.F-build2571-FORTINET
    "a9318414b2c8fa8a8daba150c43cdadbc9643dbb8ad763bd2698290dcfe78303": ChaCha20Seed(
        "28038fc62a4eca42e3e55c212379f0b8287fb96ae5938c86f74bec61ecf2bdf8"
    ),
    # FGT_400F-v7.4.2.F-build2571-FORTINET
    "c66a34cde218fb36a042db72075ecf01b5d9aaf0bcbed9fc8b6385553f2373f1": ChaCha20Seed(
        "3580d17ef54522464f3f3376ecce47ccdb63f10bb44980a0bfb60097495b8c24"
    ),
    # FGT_401E-v7.4.2.F-build2571-FORTINET
    "7a2a7c7d90ed0eee91917a69e274755d7abbd11bf3f9662698af86e1b6f56dac": ChaCha20Seed(
        "055ccceb976ecbab4c8ac9f3d6362988c2ef9eebcc21aa61d608a9c8eb65efdd"
    ),
    # FGT_401F-v7.4.2.F-build2571-FORTINET
    "7193d7221ac960fb150564637e8756e0bca1df1acd85803403949fca793885d8": ChaCha20Seed(
        "8c36ff29e02dfda419e1c2f34a9acbe6d679e84332bc9b674962aabc15d37073"
    ),
    # FGT_40F-v7.4.2.F-build2571-FORTINET
    "2ce5a50515810756bcb5500171ebcc1d0e1224278589f53bb6c9a3991c395704": ChaCha20Seed(
        "ad94627918aaaf45a2f1446bf98fcf7e6b71fe4f59ffbe323b67a9d1214ceecb"
    ),
    # FGT_40F_3G4G-v7.4.2.F-build2571-FORTINET
    "8bd5c3be1fa780fec0381c52813435ba8640d71ce7d81be71e945ed3c5a69a2a": ChaCha20Seed(
        "e5505db82c0f77581185c8487c835a931ab5a6add9f535997727b4707df71ad8"
    ),
    # FGT_4200F-v7.4.2.F-build2571-FORTINET
    "6321083d414d20f79b4a985605fe73fdd68cdbd1f9849104ea01845a8dd9d43e": ChaCha20Seed(
        "53d4a72649d6a1d475a0572b4a6ac8db88b232b5cb5a657e87d7307267274aba"
    ),
    # FGT_4201F-v7.4.2.F-build2571-FORTINET
    "a4e41ed85ae2a2ce3982b4fd204850950e5b4e29164f360be42594f96bda615e": ChaCha20Seed(
        "458c3f3265d20cb6a2f406a3069feb8f11fb0c93ee2025d1f1915c7d76c05bab"
    ),
    # FGT_4400F-v7.4.2.F-build2571-FORTINET
    "59f7ccc29f73366abb1cd26699d1148d38c9c626fb79fb397f4aafe53f8040ec": ChaCha20Seed(
        "48bacc11cc3c9b573599bf27f8a4b4765e72ce54eba968b0946c8b85d1f87884"
    ),
    # FGT_4401F-v7.4.2.F-build2571-FORTINET
    "b975b174740188576ce081722d387df30352fed80300c8c4be179b5e5ad5af57": ChaCha20Seed(
        "0d4b09096931cb0411d1ebf060ed3f40a08ef4a7c8a37ff4c8c124495354ded8"
    ),
    # FGT_4800F-v7.4.2.F-build2571-FORTINET
    "6379172ac54f7d7e758b8927d3c3bf8f1cf11d0fd06a26efb98773157fa0cc1a": ChaCha20Seed(
        "bc8814fe4651adf4ffa9b9df6628ae22ab66e620b91b9abfae0e75ce4be28f59"
    ),
    # FGT_4801F-v7.4.2.F-build2571-FORTINET
    "9e8a3821b733a5463b9f456891143d6a10dc2726755feaaee7d3c4c0b6c2ad26": ChaCha20Seed(
        "b9440b366c4337eabe92553cb19baf07fb9ea0d9ef253a07cde2c07170b646f7"
    ),
    # FGT_5001E-v7.4.2.F-build2571-FORTINET
    "7f3e31168155da0b380587f1909e70fff278c68d77424cf5b1d9cd55c6d20141": ChaCha20Seed(
        "cb0da3d1d85c8f557ff816aeaedf960a3e1d03a0294c32e56dafc1af6b32663e"
    ),
    # FGT_5001E1-v7.4.2.F-build2571-FORTINET
    "9b0b72dcf24d487d056eb196dea9cbb22a23e4b27980010c7b4f231db6102c56": ChaCha20Seed(
        "2915b07cadc7d9144c6a1dc06b389c55a2c0b803f886225893cccaa40e7fe964"
    ),
    # FGT_500E-v7.4.2.F-build2571-FORTINET
    "38ff055ed268e4981e2205151983e27c36ec648a5945e1ef537a3f3f2015c9cc": ChaCha20Seed(
        "75e9b7b43394bfeac250d25634fec5bcd067339b251394c18209004bec6da32c"
    ),
    # FGT_501E-v7.4.2.F-build2571-FORTINET
    "070c606dfdf28595bf4c32329e2dc5fb443731b868b1d22b7c863f4e8b377572": ChaCha20Seed(
        "1e88b4e65142ff9a70431b49e4ca21d5c8a232d7e3411baeb31cdec6e4d74116"
    ),
    # FGT_6000F-v7.4.2.F-build2571-FORTINET
    "5138848c3def24d1738a476bd38d2b09aa2a02e4d172314cc477d65c484819d8": ChaCha20Seed(
        "8695181145e07e31f3cf64eb897859cdf44a0816303c419585a9c49144d8f50c"
    ),
    # FGT_600E-v7.4.2.F-build2571-FORTINET
    "0561ddf5561860e05fe74621f43a06b7bc1b9fa9925c1b0c80f404f886d33a82": ChaCha20Seed(
        "7e61d7d8c0c57919deabea69f28dc33002134b10d533137e3306ccff85c0b479"
    ),
    # FGT_600F-v7.4.2.F-build2571-FORTINET
    "9f7baeaf9d68e7b622e1b50826a44a366aa9a51370e43b81a8c3c390ab9b885e": ChaCha20Seed(
        "f02123d5692c1480d0031ea7c70cedbbbe78c26a678a3b35923cfc5059beca35"
    ),
    # FGT_601E-v7.4.2.F-build2571-FORTINET
    "81d82ab8b194f4c75f9f3680e77546a1ee9bf98e6ffcb3152ca061f517784c57": ChaCha20Seed(
        "c476a7e981fc230d125c58a3a9f8a5c91f8dab14d6c36656ef6057f7484544d3"
    ),
    # FGT_601F-v7.4.2.F-build2571-FORTINET
    "39bd20df5c6f735206f3ecdd159fd7dbdb600a8ed123808d57d8851054a6b93c": ChaCha20Seed(
        "ba5b010431863503df8740ad2eb13d3a5b4c49e85d39eed8ac07985ebae384c0"
    ),
    # FGT_60E-v7.4.2.F-build2571-FORTINET
    "269089b501d1a552af741e0247722fa05072159bc6ee0629c369271b62208ea5": ChaCha20Seed(
        "32f2177d720e57ec4d1559ab442ff2a0d0bcb2780f612871f1ce802c5e3f8d55"
    ),
    # FGT_60E_DSL-v7.4.2.F-build2571-FORTINET
    "6b52e5213bc2f2ce22d9e832ac7c7fe5b19b1916301bfce586af294b746d2992": ChaCha20Seed(
        "55f2ea608fedf0a76117326e9c39db47bfce18a57241741c30045050145cfec3"
    ),
    # FGT_60E_DSLJ-v7.4.2.F-build2571-FORTINET
    "c01bf5276dd7bd4e465cad5859914ad68a67b5013ec1d9fbb62417eb60041468": ChaCha20Seed(
        "bbf15e7e070f49ef6779981293cca99f67f2ded91a4c8d7b238b59d7669f0f29"
    ),
    # FGT_60E_POE-v7.4.2.F-build2571-FORTINET
    "59dbecc025b81675e2e9f3aa692114e52c0920e949be78f730dd1b3c9ba73833": ChaCha20Seed(
        "1b9805058602f020fbe5c0fd444a5036a07c25d9a0d74bf1e9c2627b5a377d7a"
    ),
    # FGT_60F-v7.4.2.F-build2571-FORTINET
    "ba8e439e40e45e15fb7b6f1f55d09029f68cfe7f458c13da5dbaeac172a1a7db": ChaCha20Seed(
        "7f8813062e87c1ea6b2a93f94919d2e35ca18ed34c18fdfa0ce16e71863e7ee9"
    ),
    # FGT_61E-v7.4.2.F-build2571-FORTINET
    "eab8038cbb6d34c6143fb59015f89e2594af9056348d37ca36c565d6dfe11bde": ChaCha20Seed(
        "192af77f1fc1d5ce749a2751613370af365dd4f00841f8bed4f1a45e5bb90e32"
    ),
    # FGT_61F-v7.4.2.F-build2571-FORTINET
    "72c6d12c88bbe857e45de1abf3251719e0083567daa73d3d6ebf59fda07a7a99": ChaCha20Seed(
        "eebf55b6bc8eac5216033f5589f8a01a0ebc07b456f72a60fcc450873529e764"
    ),
    # FGT_7000E-v7.4.2.F-build2571-FORTINET
    "0183e4f79bc18d8781c405988269932a958877b1808090fa879e45fd3508253f": ChaCha20Seed(
        "b4608f87950cb679a2056955b2e48b68eae0f4d9c98c282b6d5b606d1988e808"
    ),
    # FGT_7000F-v7.4.2.F-build2571-FORTINET
    "01dbf256572e7ba106c218a8d62c9bcaf1f358e000e9fd720b31c64df1a64df2": ChaCha20Seed(
        "647a77e93e98f9fab61cb227d220c09738f2a802fc1a3148b27782fab25b922b"
    ),
    # FGT_70F-v7.4.2.F-build2571-FORTINET
    "47591ff48b14d54a09bede9e5559867e36d0f8ad313906e8b00aa4b55780ea62": ChaCha20Seed(
        "0fccd6a228e24612109987557eb83799df49eb8f0005ef80a051a7f8e9de52b1"
    ),
    # FGT_71F-v7.4.2.F-build2571-FORTINET
    "e0013fad2da99ad2a7ebc5e4ad59a207ad46b8d4fbdf44f178be6e81d58c0378": ChaCha20Seed(
        "21ef7a4bfc00ee66ae38e546d3a3d68afe62c98f67624189cbb60bdb71922749"
    ),
    # FGT_800D-v7.4.2.F-build2571-FORTINET
    "5c9a1137a5f908e913f8b2c307fbe4b5ab2c25f790e15f6dd61d7c236a55f096": ChaCha20Seed(
        "a283d6c203ba1624656450b65191ca1daaad7ba1a65f7bbc1ca3f696de90d8a6"
    ),
    # FGT_80E-v7.4.2.F-build2571-FORTINET
    "98a22f1649a31c1672b40b8b5139b82b756a96c273bf75ba62dc9922c5abbe21": ChaCha20Seed(
        "155b8f5bf8aa826788335c6607abbf2ab79d0a4aae80929e0015dc6d77032429"
    ),
    # FGT_80E_POE-v7.4.2.F-build2571-FORTINET
    "09794f4982f07761559b6da59b506c3b993c01c3fe8353803d3816b8f2882a09": ChaCha20Seed(
        "114809b8e51c1a1f5ba471de6aebe5dc591d6db204bc0c6236c8b8d0896eb7e9"
    ),
    # FGT_80F-v7.4.2.F-build2571-FORTINET
    "5a10cae79dcb7e520518937ff19be653799e7dd8101a69bbd9a67e12a928449a": ChaCha20Seed(
        "64ba12e35eebbfb665575a681c43b3428829e2901720ec4c68af2c7ff33506b7"
    ),
    # FGT_80F_BP-v7.4.2.F-build2571-FORTINET
    "544c5ffafc9fb412aeb69c4aa31f7bfdeeb34a3566b90a290926492ca52a3631": ChaCha20Seed(
        "6b39b1926c50bfbce090d8c3b74a6f4d9386a7adeca7bbabc5a2d5a4438b9c41"
    ),
    # FGT_80F_POE-v7.4.2.F-build2571-FORTINET
    "c28c9c132c55ed9d8a7e0dd4810a1d6fe68e07a4299134acad3ba72fc9be672a": ChaCha20Seed(
        "6b588eded8fa7932df384f346b6160514327445968c1908e5b9e300a3b22492a"
    ),
    # FGT_81E-v7.4.2.F-build2571-FORTINET
    "97b660b0f2c3d2c958ed00853f5016390b0e042dcbec835fa39706929e35364c": ChaCha20Seed(
        "c2e6760a3307891b1d6d20ac6de6ddf512dc7389627f18766ceeb5451c8b0ef1"
    ),
    # FGT_81E_POE-v7.4.2.F-build2571-FORTINET
    "7655df3959dbb9174432fd779174281dc80aef02d609fed7c0ceaac757e38dd9": ChaCha20Seed(
        "fc77f686da3e6839c11c4abc4da8b9e94ef51b669737a0ce17b91b7536d214b4"
    ),
    # FGT_81F-v7.4.2.F-build2571-FORTINET
    "d8271cb81295e307b63cd012294d7fcaf575b920d68b4e831c8631aaf6d669bc": ChaCha20Seed(
        "8ba0efb794ed9ec1b3a5198fb5823dc5b34f96215c831e567c680e3b14ef8c83"
    ),
    # FGT_81F_POE-v7.4.2.F-build2571-FORTINET
    "951754e54ddf006c9da63fe041c20210f98bc61892c4706b251173d9c05b3a3c": ChaCha20Seed(
        "2a8d710cd8206407dbcb72bcc3ce57fa44d24447574fd129c7a30a9dba3babdd"
    ),
    # FGT_900D-v7.4.2.F-build2571-FORTINET
    "489b486b6fc290dafe5243bb69219ad50fb7aca4baaabb16d0f5443fbf0bbaa4": ChaCha20Seed(
        "8dc60d30171d413ec74f2bd8da1c64c24926928ba772711e3aed875b35c7f4f3"
    ),
    # FGT_900G-v7.4.2.F-build2571-FORTINET
    "499295de29a835b8e610dd0e829aa658d479d12e23c0990b9b0a4aa89ceca4c0": ChaCha20Seed(
        "b426ba6c2c463b4590d656a090e1f6f9ee7c2882db573173bf5050bb3b71c2a1"
    ),
    # FGT_901G-v7.4.2.F-build2571-FORTINET
    "b5b136ab23be53c60c832ced9b04e5c08e40058d27e5e070379cb8fa8a2eb964": ChaCha20Seed(
        "5db140dc60ca8d98cb58993ce47abf6846d2437e210531facaa060cdbc951275"
    ),
    # FGT_90E-v7.4.2.F-build2571-FORTINET
    "6b7778652a1ee7177470d4b2fd49875d63f6c0e93f2806f0b4417a0a91561f94": ChaCha20Seed(
        "079ab478b0dea7c4c6d352e80ec0615bb70f6c89b6e7c108a023d0e68a838f2a"
    ),
    # FGT_91E-v7.4.2.F-build2571-FORTINET
    "0ddf1e1de74261491fd1aea133748e7860c790f0dd726375236ffbdc5ae6694b": ChaCha20Seed(
        "3eaaeedcd7f0eb6fe2440ec6a5d625da46c20f23a3b92f7132e3209fdb79cd1f"
    ),
    # FGT_ARM64_AWS-v7.4.2.F-build2571-FORTINET
    "cbdbf15bc00cc612fe420009ec47a9d831005eb4691452ceb16e94db656ce949": ChaCha20Seed(
        "6c9fa2758813038f1709757f78deb8f19adb385eccb0dd4d5cd144334a510263"
    ),
    # FGT_ARM64_AZURE-v7.4.2.F-build2571-FORTINET
    "c12ae2394fe66edc2db6391b2bf08c13dadb7f9dfb3bb2fec0174736d6bf6deb": ChaCha20Seed(
        "5b524bbf44455b198ed554196e0bceea8747e85294c35957fa828cf7afd3df95"
    ),
    # FGT_ARM64_GCP-v7.4.2.F-build2571-FORTINET
    "8bc2990e4ce948f4ddc4b04e29db5dbcf1e9a95b7d45953fd9341bd2ddee9b5a": ChaCha20Seed(
        "75f2d38ed37c4397431924d11c2c38a7adff513252fe3ff27a75141282776191"
    ),
    # FGT_ARM64_KVM-v7.4.2.F-build2571-FORTINET
    "b41f6c3bbd9038f9dd5d254a4ffe640e9fd970011f6021ec0fab17c818d3464d": ChaCha20Seed(
        "e406774c8837e1244ebdb7fe5abc691f77ef332e7d0f34a3eb810d08e89dc573"
    ),
    # FGT_ARM64_OCI-v7.4.2.F-build2571-FORTINET
    "fe8f12cb4c838bf09fabd3167c7fc774ce36389e17710b207a6240a5601cfd96": ChaCha20Seed(
        "23bc5895e31f3295158256a492004a56f5de77c87d07f2a7320ac1b406811a20"
    ),
    # FGT_VM64-v7.4.2.F-build2571-FORTINET
    "c392cf83ab484e0b2419b2711b02cdc88a73db35634c10340037243394a586eb": ChaCha20Seed(
        "480767be539de28ee773497fa731dd6368adc9946df61da8e1253fa402ba0302"
    ),
    # FGT_VM64_ALI-v7.4.2.F-build2571-FORTINET
    "9ad30477bf7434f4d1b297e78d66e435c9209de1a0fefb4508fbd6b7e6691004": ChaCha20Seed(
        "cfddb90a8fe941254d8ebc8f97766c0d6c4f506c4d3ab7b4b01e6e58e5276cd5"
    ),
    # FGT_VM64_AWS-v7.4.2.F-build2571-FORTINET
    "52738f8a1e1cbed207160d6bfb2251c0ab173f941a401525e0a5274f6ddd8786": ChaCha20Seed(
        "5fa20dd9bfd254f59c2e0c899c8f7e348cbe89231c39d2eb6726f4dd3fd658a7"
    ),
    # FGT_VM64_AZURE-v7.4.2.F-build2571-FORTINET
    "d625d0ef804e27094b744e0d1a70da1d97901cde3d51bfe05ecc0f12c99feb30": ChaCha20Seed(
        "6199df9c95c510c1cfacc95acc1d56c825f4e8f05e519680fe72feb2a25a1e53"
    ),
    # FGT_VM64_GCP-v7.4.2.F-build2571-FORTINET
    "8cb9da64a28516715118c281c824eac0a09b4e8aeeff86fada7a4fbf86d39e97": ChaCha20Seed(
        "279d6d0b668d0dc2a6aa9dbff3cffaa8b41bd994236ccdf802b9326e65cb8b2e"
    ),
    # FGT_VM64_HV-v7.4.2.F-build2571-FORTINET
    "d9310c5274d9c606c4b64b6be9942589cf8f33c98f379ccd56b486a71d659036": ChaCha20Seed(
        "62cc0a65287de21fd0ae24d03baecbc3e08eee69f97bd5e41f4da5ed72859a10"
    ),
    # FGT_VM64_IBM-v7.4.2.F-build2571-FORTINET
    "c65be5c5380c3cf5393a556b5a0a38790e480aaeb496832f1b4920f38eb65b8b": ChaCha20Seed(
        "b6a99b440f17fd76958111e08efad9f19f27258f1d3c93d209666d3545efc4d3"
    ),
    # FGT_VM64_KVM-v7.4.2.F-build2571-FORTINET
    "d0b1f4e81b755156bda15852f06555f1ae48d809472246f50dd7cecf5ec766d0": ChaCha20Seed(
        "5dbb868fda6e5f3c9cf6cba5236cfdb09fb96a8495729f07f1f1480f05f37b60"
    ),
    # FGT_VM64_OPC-v7.4.2.F-build2571-FORTINET
    "72614a6d2ddcfe5b0729ab8fb00eb082aeadfa333a19cf21ee9641c6c53bf56b": ChaCha20Seed(
        "331f309ddee50673f624bc4f10acb1e0b31b067d0600ee2a30fc77f38964a332"
    ),
    # FGT_VM64_RAXONDEMAND-v7.4.2.F-build2571-FORTINET
    "1a7053c99fa69d7363d79d93b26b92a65c76f806d26b759c396777e355850b36": ChaCha20Seed(
        "15ce132217ec3570d9f51088abf9acf927903498d0cb8c75534c6b8cf869c79b"
    ),
    # FGT_VM64_XEN-v7.4.2.F-build2571-FORTINET
    "d0bbaa36723c36f7c763d869c7a6e946854ac5bc4c6c44ea03d1b5843cd42a89": ChaCha20Seed(
        "14c6607c1b675b7ec6d4c1a437e51dee2e742158b782f56c72d8fc5aae72f296"
    ),
    # FWF_40F-v7.4.2.F-build2571-FORTINET
    "b74e6808c9e6540c2892e7c62e24b4f85755bb3497432f8b89e2e4e8386846bb": ChaCha20Seed(
        "87e9221eb376e13dbc3ee4b09407b28a3215f101b917602fa35fdfa0a7367cb5"
    ),
    # FWF_40F_3G4G-v7.4.2.F-build2571-FORTINET
    "f258c027bac0b9c7bf746d04f97e41c387ddd43e3f8b5e8f4899388a3eb0ab4d": ChaCha20Seed(
        "031c37f376b30acc63a0cad67b324e40eed3c8a16df08a29a8075f6559630398"
    ),
    # FWF_60E-v7.4.2.F-build2571-FORTINET
    "09c3c9c070fdbcd00095962c5284524de6257893c8598515eb0504cd3c62a533": ChaCha20Seed(
        "8640e29714d03cd3019c1b3e23b50124f5aa35cf610628777f6d894cf4f11d89"
    ),
    # FWF_60E_DSL-v7.4.2.F-build2571-FORTINET
    "ceba9d64a9caf714b16059ed4962067c25d50bb14dbdb98a3148af6e8a1c75d6": ChaCha20Seed(
        "d16b12146ef53ef41833b784eeca07b31ce2e80a1d8a1b3695d8cbc6f1158b79"
    ),
    # FWF_60E_DSLJ-v7.4.2.F-build2571-FORTINET
    "50aeff4e538845cd5dc01e8ef818c2ea92fd0de2690bdc66fcd91f248f7d6b48": ChaCha20Seed(
        "b364657221ac797bccd4dc177901842eff198d89e8424a3184de90daf9c32f56"
    ),
    # FWF_60F-v7.4.2.F-build2571-FORTINET
    "c9f6940b3baeb2825fb4d9241c952bb9a3805c0af35a9b4d36b69f1e6a4d4a3d": ChaCha20Seed(
        "df70dd5112677e7e780ecd87299cd2cf73ae0b44038f3f7f3ca130fce78e2b85"
    ),
    # FWF_61E-v7.4.2.F-build2571-FORTINET
    "b02f629ba836274ce7dad7e1f8172cf4015f0c1222d60f98362d809ea047a790": ChaCha20Seed(
        "007fa3a4205ddcbb91d021bb647375573dc0e7b56802deba2fb71de3f6329bd9"
    ),
    # FWF_61F-v7.4.2.F-build2571-FORTINET
    "40960250e57687de2ffd2acc077fed0298a7eec0ff702a7487e4e4c6197128e4": ChaCha20Seed(
        "0d8a802e0bb63db38a54040e083cf81108f5cbb3c0d34258caa97ebcb5a72661"
    ),
    # FWF_80F_2R-v7.4.2.F-build2571-FORTINET
    "71f8f725734807a334b422f2fd3ccef472e15b50f84ecd916a3e0c34255dc42a": ChaCha20Seed(
        "248658ffb7219aaadc9dc166d5b36299a210faff955a01a411a95f5ce5abedb7"
    ),
    # FWF_81F_2R-v7.4.2.F-build2571-FORTINET
    "e06c2cf6431d668065a7473dd63ae5d8f5c9cdd229dea4d4745416a8e3727a14": ChaCha20Seed(
        "1123d1a11488630418e5dc571e7fce35ae1e42771ef43b82b38cb73fe609e5c4"
    ),
    # FWF_81F_2R_3G4G_POE-v7.4.2.F-build2571-FORTINET
    "42aec8606a6210e0b34675ebff5a5c084f637722c70157e4d72af8fd5a84b1d0": ChaCha20Seed(
        "d8ea12f10b1cb804e20193f065f44aadde8e588d77b4e0f594481916aa5ee966"
    ),
    # FWF_81F_2R_POE-v7.4.2.F-build2571-FORTINET
    "9898a78aa07a09ac0192ced7bcc422bb765b90ea16789329348c2b35000d94f7": ChaCha20Seed(
        "990074286787440b2c46307f3321f0ae500db6f9b1841b9866c438b6cc90fdb7"
    ),
    # FFW_1801F-v7.4.3.F-build2573-FORTINET
    "33b9f6b7367b3f47f447a68750059b0228f3fdbae4e5012d0dcca95223177dc3": ChaCha20Seed(
        "78960af1902b1518579baddb0f233e39443b1e699a6fdc7b6ba9995ced37900b"
    ),
    # FFW_2600F-v7.4.3.F-build2573-FORTINET
    "9e959835fb82f22ddda6913d78c1f17fa0dcb17aea296b38698efff19ecfa07d": ChaCha20Seed(
        "087803987aa975a5d7b01f542b966cf8067e8f4486da2a4cd0c375e17c3e4e2f"
    ),
    # FFW_3980E-v7.4.3.F-build2573-FORTINET
    "d4f13c28befbd8d68ab2b5a3a3c752696c0a199497a94b12943bf0f102b1c4bc": ChaCha20Seed(
        "ca0845b2f66fcef4e971e4e9327b2cb59eccdf0ce4cd53f54745508888088764"
    ),
    # FFW_4200F-v7.4.3.F-build2573-FORTINET
    "4e947cc0b1259f77cac04811bfdee87d3b4e854c5a7942d5b9ee3e360180797f": ChaCha20Seed(
        "a8c08f94c09809da70f14ee82f72b8f48b67f91decf327b223cf822258e06ee0"
    ),
    # FFW_4400F-v7.4.3.F-build2573-FORTINET
    "cf475e87fda031a5fdc03c716c8b8ad7d9733c17df95ba3a7addb0b668258b82": ChaCha20Seed(
        "fa90c8bc16ddc81ea4b595e395f87e57473b41944ffb2d77a7a6d358a73385f5"
    ),
    # FFW_4401F-v7.4.3.F-build2573-FORTINET
    "7f0ce4a821ee4c51bbdf7013aa02d87e00b5c48f1cdb5dd9a3a312b3b1f7b59c": ChaCha20Seed(
        "cc9a3faa14b66882fc029ee0535faa3390ae86eb9279464f419827d32e9f60f4"
    ),
    # FFW_VM64-v7.4.3.F-build2573-FORTINET
    "0f630720307dd95991a4d09621a61374c254d622b5a097d46abad6a5e80f90e0": ChaCha20Seed(
        "d3cf58939399d08891d338b81de71764bd67f5d1f74ddeeef923cdacdda6b410"
    ),
    # FFW_VM64_KVM-v7.4.3.F-build2573-FORTINET
    "a2f37aff58fffdf707bf6e56ce0d80c1334a40f478ca98a83a5b1a0827f83309": ChaCha20Seed(
        "b895eb46711bf617f25ffb703b3c1cfcdc51814920485fee7ab26b377a0bcf07"
    ),
    # FGR_60F-v7.4.3.F-build2573-FORTINET
    "4ea37206430d85573ddca94507dbeb1a520a8b3b966445d9afb6429ac4a6b25b": ChaCha20Seed(
        "ee5784e1ce094f71d721334a0d9440ce010939a3431babe2f6dfed2b32fb970b"
    ),
    # FGR_60F_3G4G-v7.4.3.F-build2573-FORTINET
    "3ce94092ee0589d93ecc7e2658dfd6bc727c73f382824afa74631116ac0f9ba5": ChaCha20Seed(
        "3903fd19b37c2cc405e3f4b6da5c0c2e73c5f1beb6492d93ec0d1c37a6edf01d"
    ),
    # FGR_70F-v7.4.3.F-build2573-FORTINET
    "99d581fb071750ce33d22c01e3e9ac16b7e74529721885012f5cf40472760548": ChaCha20Seed(
        "66d0f8614ead03a66de268fa997bd8d87fd04ca64fb74ab56b44f13d1db84744"
    ),
    # FGR_70F_3G4G-v7.4.3.F-build2573-FORTINET
    "14e6d04df495e46d1a6277e9c48824214f202705b873265b376f1013db498a42": ChaCha20Seed(
        "ebd2ef5eb94d9af0ce907d445a4ef83f55f00f0e132aefb5e3c0b966616c3ecb"
    ),
    # FGT_1000D-v7.4.3.F-build2573-FORTINET
    "e1dfac9f413cc265e4271057f3d96580410b0557bc563348a049c56fc1084595": ChaCha20Seed(
        "0bb0cf4c313a3d663ee55f4e8a26680586cf96457f8da54dbfff1d4849cbf9bf"
    ),
    # FGT_1000F-v7.4.3.F-build2573-FORTINET
    "4e13251763649becafbf132e1f2fc3d2885dca80d9eeffc1de110db5ae22e44f": ChaCha20Seed(
        "7d24c6056736146fac923fa962c9e65828746371ceacf57f73225c9438b6a6a3"
    ),
    # FGT_1001F-v7.4.3.F-build2573-FORTINET
    "18dca02eb20d946da6584aa8ed497966178d2d250863608671aa3aa636af72ef": ChaCha20Seed(
        "fe941b4bef1c09fd6f653ca74a60a76a3dffdd8d947d93ca5cb96846f54f0d4a"
    ),
    # FGT_100F-v7.4.3.F-build2573-FORTINET
    "535e3a94c0f7746d25bc580c47d18a23dd9991e16e84124392f9576a362453d7": ChaCha20Seed(
        "0d2e1976c226cd3f122b2084838be1f0e374567105979b95aa9047a1db85d2e7"
    ),
    # FGT_101F-v7.4.3.F-build2573-FORTINET
    "77d567cdabb07d02822dd269954b4c30be7d70bc6d90c11fff425656bc4a0ce5": ChaCha20Seed(
        "fc5d01e0c7376671542d65aa245b7f9e5a3bf48411b7ef6b362c205b4a496ad7"
    ),
    # FGT_1100E-v7.4.3.F-build2573-FORTINET
    "328c9958cdffba640796c0c7e72e261dc743532b47e16dad7be473b1dd6cbd24": ChaCha20Seed(
        "96eee6df04ca662bc430830f9b10f4ba0358b33e284e938fb09d0b5dcf5a2005"
    ),
    # FGT_1101E-v7.4.3.F-build2573-FORTINET
    "a0804adfa07fe12f13ab2324fd5e589bba41fc756c951799e3a3d7dab528f8fb": ChaCha20Seed(
        "f4ffd198e20af858310bfa39deded2638cae4c3aadcb2ec6d00efe303d62c24b"
    ),
    # FGT_140E-v7.4.3.F-build2573-FORTINET
    "11f08964d7426ea7277847e01cfaa5b71ef8fe6ba19791bbe67b54f3135a90e8": ChaCha20Seed(
        "6bb953bd1fae42887c8553f7e681d96a355198197fbfdf302220b641bb1bd4ea"
    ),
    # FGT_140E_POE-v7.4.3.F-build2573-FORTINET
    "13bb79243131fe159dc60ad2f82b856ae4e471114da7b6500f39f33ab18822cb": ChaCha20Seed(
        "fb3487e57d467c675562bcb21df93c9cb8adf38fddd7f2e987c14825df09a1af"
    ),
    # FGT_1800F-v7.4.3.F-build2573-FORTINET
    "e146bbeb4d5c9f3662f9205f9fbeb5689799a8b54fb25e07a2da27c3fff99210": ChaCha20Seed(
        "c0f205878897bd57fa6ae79aca31c762cd902f7b866ff308eba08208b1d101bc"
    ),
    # FGT_1801F-v7.4.3.F-build2573-FORTINET
    "93228d913a835128747df4d280c76607e6e1f4b0a4233246979fdcaf0789bdd1": ChaCha20Seed(
        "e88686832e1ac77d27fe9f2bf4a98d0d1f49cfef6777f6f90adba5e0fd80120a"
    ),
    # FGT_2000E-v7.4.3.F-build2573-FORTINET
    "69c19ad2fe0abda2b5a7a29529025188f42a1d99964412319dfb9b3ceeff4f86": ChaCha20Seed(
        "1262a44b37c62cb27b67e8b8bdbd4c43a7ae6d3aaaf7b04302bfae319491df68"
    ),
    # FGT_200E-v7.4.3.F-build2573-FORTINET
    "200282e2325fe0badc0d3883ba95c720f0fb0dff2c1e7d55a88d55243363a648": ChaCha20Seed(
        "243a30473a334f6ca8c06a3c806ff49e34e4286d7b3c80c80cf32fa6fabdb5aa"
    ),
    # FGT_200F-v7.4.3.F-build2573-FORTINET
    "6a2da9675ec5302f5da0a55087e67abaa835b92a7c9731e3d7caf810cb977df9": ChaCha20Seed(
        "c121c6749e0a77c90ded9d80cb2c1ceb32a11b66b9ecfdda1e36d04268587565"
    ),
    # FGT_201E-v7.4.3.F-build2573-FORTINET
    "283c59bfae478d1b4446d516210b053a1398b9aebca54e50504bbb72c7efbd0b": ChaCha20Seed(
        "9e4c3f618af16112ff6812d42d1b190321dfc66598838e51dfc71301520c3992"
    ),
    # FGT_201F-v7.4.3.F-build2573-FORTINET
    "a9006f06b361a8c83ff3341791be43489e5643cac63fe3749e60105f0f956574": ChaCha20Seed(
        "9f195496ef9ddb5f4fcd404e5af107a2fdcf542180400c178568f89aacbee31d"
    ),
    # FGT_2200E-v7.4.3.F-build2573-FORTINET
    "110cfbca94683f5a62eccb6875102e9b4074e5d209e79816f5081e70e4519a4f": ChaCha20Seed(
        "5e7b3c99e5830eca9007a5ad84e4222e9da49e43c1673645e741a472ea2b7017"
    ),
    # FGT_2201E-v7.4.3.F-build2573-FORTINET
    "f903755cee5328be65b78782a44f7ba86e0eb2830caa7b7058ff0cf329755d1f": ChaCha20Seed(
        "717eb166da6cbe84caab418168fdedfa1f3b5296230ac7b65c039b2e2bddc7cd"
    ),
    # FGT_2500E-v7.4.3.F-build2573-FORTINET
    "fbee3b15bb3d72cb1dbac58775f1b7b0741eb387c7410e6702fc1bb72fc6db0f": ChaCha20Seed(
        "3c2777d8ccf9b50947a85a0bf3ee644f70e13f5f9afc4e0643f577571d252fc1"
    ),
    # FGT_2600F-v7.4.3.F-build2573-FORTINET
    "17cac4572d9364eee47dd875819c1c2f6c522dcde372d4d0a57ec810e959e241": ChaCha20Seed(
        "9dfe11afb155d6fe0df6dfe95e67fc4c58959a2becd3a6ab9ec13af4f09b0ddb"
    ),
    # FGT_2601F-v7.4.3.F-build2573-FORTINET
    "948013155426951d6dca89bedc25060b7ee29073244964516e03c2a102b13cfe": ChaCha20Seed(
        "e14c0b96d0cc7550fb96320d2181a175b9fc35a53d2bd93489b71e2b89b0c119"
    ),
    # FGT_3000D-v7.4.3.F-build2573-FORTINET
    "196e58ead4a7cce385b38c5311a5ea7f456297f04ed0c2775f725794453b4525": ChaCha20Seed(
        "ffe048d441652a2b5b62f5bfcbfe51e71af51222b512480dd25aaa0a3ba7e919"
    ),
    # FGT_3000F-v7.4.3.F-build2573-FORTINET
    "24f01b46fae116dd452f26fa1adeb68330b98f67b60c723e8a9fec988fd615cb": ChaCha20Seed(
        "e0acbfad7c2d1efd371d97e448d7324df24d66d3c04aa5c24d51b14284ab2af6"
    ),
    # FGT_3001F-v7.4.3.F-build2573-FORTINET
    "882f70757cc1126c215848638cb1bd3556a350da8f217d22fc1fd7601c9d63a8": ChaCha20Seed(
        "25f9424b6431a2160dade1b2324879e1435f23b4db671f79d730d2ad73872512"
    ),
    # FGT_300E-v7.4.3.F-build2573-FORTINET
    "c6f809467dc4dea4d260cf881fbb65523d65d4b5059e2fdf512fa54ab4211514": ChaCha20Seed(
        "87a29b9c0083019e07a4a61c93962c011825755507910fb7a39b5288992266fe"
    ),
    # FGT_301E-v7.4.3.F-build2573-FORTINET
    "a743ce34be3c804d045528a6564233d3b3d8c6166a1f77465dd98ea6c6fba17d": ChaCha20Seed(
        "d3bbff256d47128eaea57efe95d8bf85cf272dc9a2fcd572fc14c84af871fb83"
    ),
    # FGT_3100D-v7.4.3.F-build2573-FORTINET
    "6ca700626d72744939b84b66049d920eb2d851c3f690917fbbcf1446a080fad6": ChaCha20Seed(
        "2a8651e58c42346df645944afb57d4e269c7de2b74b0816037ee47d49bf2f155"
    ),
    # FGT_3200D-v7.4.3.F-build2573-FORTINET
    "2ab22436ac247e79ac730dc9f1edafbe3acb424ab56a004c7a91ddb70ee80b92": ChaCha20Seed(
        "a2354017718db0367627efd0645adc50f7a475404e5daf5ac06f76ac018bfcad"
    ),
    # FGT_3200F-v7.4.3.F-build2573-FORTINET
    "ef43edacf450736f338e1ac54dd3671f653d021c6368862178098e8d9d0c971e": ChaCha20Seed(
        "cfdd494f913cdf2933b8a0594b841d0f3c2edf4db744f390b51879c18cd92721"
    ),
    # FGT_3201F-v7.4.3.F-build2573-FORTINET
    "09f1d5c6c3c9e92dc660a8c01ae9e2ee7cc620e421a9abee6ec83b6ec30187d2": ChaCha20Seed(
        "67c27552e6a20db1bd6ef872cb17d4f52ddacd147b5bd7afb63a7f20344afb75"
    ),
    # FGT_3300E-v7.4.3.F-build2573-FORTINET
    "7fad940f5b6c29cc9c2124fa01b7a7e7238786b2326ed425039b0c48ee9d4f1f": ChaCha20Seed(
        "a3872f77b2e8bda47d53cb98de99b9124f02c799628acab4410d5c0bba0bda7e"
    ),
    # FGT_3301E-v7.4.3.F-build2573-FORTINET
    "b926e879bac646b71b6cf89524e80eaca36974ec416abd936d24c6c16916b65c": ChaCha20Seed(
        "4c183284d80364fe6ddf34d9c93a7aa062d8ad49f056c02a2e15ae8119bd8383"
    ),
    # FGT_3400E-v7.4.3.F-build2573-FORTINET
    "5b3d2d66e575777699d3a9edb856f3ded5f30d9bf228754501d2888b9381890c": ChaCha20Seed(
        "39026bf05b8497509b6fc4fcdade412b6dcc2faac63a52f87cec292d59d8dbea"
    ),
    # FGT_3401E-v7.4.3.F-build2573-FORTINET
    "113db377e9331ebef33b931bcd26fa8fbfc6fd4ee5da3745d3e42405e790510b": ChaCha20Seed(
        "2e16c3bdc585c5a4c7bfc4ddc73c1a678526daf38c5b3d4ecb85eba57dc87e9e"
    ),
    # FGT_3500F-v7.4.3.F-build2573-FORTINET
    "f05a02d4611980e491285edbf74d766c0bd9e8e8c974af33fc62bd80856d3a4a": ChaCha20Seed(
        "3a9cae58dd86d9cb36e62763290728d6fc67da71bd8fd72ff0b5be049c2141ab"
    ),
    # FGT_3501F-v7.4.3.F-build2573-FORTINET
    "7623a601bce40ace3bf01d7625678304aa01e32aa6b63dae08564a112bb29d3c": ChaCha20Seed(
        "9209c08c5e92fad448be4d039619574eb350d5ba8b6db9062e14465ffa68296f"
    ),
    # FGT_3600E-v7.4.3.F-build2573-FORTINET
    "e000bb95f0aee65e78c9b8fb3da3ac8cf9fe1a357443692f065922fa444eeec6": ChaCha20Seed(
        "c74eb02f8cef63f9f52da5ec5903d9475dc1d3f8b12ea51221f3ddb2a8abd529"
    ),
    # FGT_3601E-v7.4.3.F-build2573-FORTINET
    "b6231e85434c8918f36868774591efdb98561b310f19d4dba4587f80a958765b": ChaCha20Seed(
        "5c80a3adab2930dc1c3a0e855ece937084dfb055512cecaab27b1f0c0034e4ea"
    ),
    # FGT_3700D-v7.4.3.F-build2573-FORTINET
    "ce407d0c64667c549ed3a9df7c67b96be4adeed8c350d589885e739cb8c24045": ChaCha20Seed(
        "c85f3042f6a220562bc121df7395b9dca0ec5218098858fd9647bd5f9fe409f2"
    ),
    # FGT_3700F-v7.4.3.F-build2573-FORTINET
    "10046afb23dddd807cd1a68fb6460851c833211676891df3a9771adc6b92dce6": ChaCha20Seed(
        "d8fa8833b1ae2d75685089dcbc3887250ba85b940c89db441ab849f78b82921d"
    ),
    # FGT_3701F-v7.4.3.F-build2573-FORTINET
    "68dc714ebbd4011d388c0dcaf934700a7a64ed29878aadfbcf134d052381dfab": ChaCha20Seed(
        "7229f19f080e53d85dcefeed5fbcebb269dd32ca15ae5e7d03a05518658e26f7"
    ),
    # FGT_3960E-v7.4.3.F-build2573-FORTINET
    "9370008736ded12586d04778ad7af09f8c5e9b3cc97a96671c5a955f44e3cee0": ChaCha20Seed(
        "bcfd5b869744be773abdf5ee76830df2639dadbd60d631225c24b627b256aa91"
    ),
    # FGT_3980E-v7.4.3.F-build2573-FORTINET
    "9a0317269267c0bc9199a2bbd1ed89f7d455b5b59c4bd3e44d3662fa3efa91b1": ChaCha20Seed(
        "a1c35521ce5c82d831f5feacc178a36e3f258e17eec7f0357374671c5dd2694c"
    ),
    # FGT_400E-v7.4.3.F-build2573-FORTINET
    "2373f78fee66055e306d67973d0593453798c434086021cc46a6237406d0775f": ChaCha20Seed(
        "a157210bc1bd469381c8038495003f5134d13a9430631e50fad64b49cd6924ca"
    ),
    # FGT_400E_BP-v7.4.3.F-build2573-FORTINET
    "406c5d5d3a63e807a0aff9f9b10a74a93b1739095c93863a9263a69eeae8d24b": ChaCha20Seed(
        "1d3b038c2e747ad3aa69855f08275719112134b96f31049dcd232cd7989461e7"
    ),
    # FGT_400F-v7.4.3.F-build2573-FORTINET
    "46d3b406bd8692e23a83155fdb95b704d75042fcd1bc7bd2ef29c860341d662e": ChaCha20Seed(
        "204c64991add7ff3b5f27fc1420e95ebca6ba7bb8236a08a6dd701d646e06c28"
    ),
    # FGT_401E-v7.4.3.F-build2573-FORTINET
    "70e97710a6e5fbbed4fc9633bd500b92681f9a78ecf3d7ef5c75023d7653cdd5": ChaCha20Seed(
        "9fb2b166a47a07d95634b8306a063e616ccfe16e6bffa43afb547dd57a3a48a7"
    ),
    # FGT_401F-v7.4.3.F-build2573-FORTINET
    "1b9cfd7cd6c0a62accb108b00d6ff80a889bf30caf6820b7200c635f63e5a356": ChaCha20Seed(
        "06d96dad65e0a3307ef225d7015a3733d4cce8113817cd7db2c99fc9503eb7c8"
    ),
    # FGT_40F-v7.4.3.F-build2573-FORTINET
    "7f2769d6c204d7b1e7d41b91d8de2295a179353aae4bd69192a7c0d48e40e24a": ChaCha20Seed(
        "9969a2064d9dafc48152aa5b10f4b5b8d1d7f5928f0f58ec34c67809415f2b96"
    ),
    # FGT_40F_3G4G-v7.4.3.F-build2573-FORTINET
    "4e6138a4193caa5a59e695cf32a8c045d32d310aa48a606e99402ed76205439a": ChaCha20Seed(
        "91b532eaf99eb9f976a32e5a920e20a4adb1c8cc56f4c2cec45c8c671e3ac2a5"
    ),
    # FGT_4200F-v7.4.3.F-build2573-FORTINET
    "3fa7fddbb1d6bd5367197c7e628dc28568f7d757627fdb6ef399d2847f2f33f1": ChaCha20Seed(
        "a535248c7ed14498192ba179e1e3605c2d00557da415cd0f710a0a4cc4c41a61"
    ),
    # FGT_4201F-v7.4.3.F-build2573-FORTINET
    "601e9dacc91a664307b8f3e66bdaa7a1a273990f525cf269be849df0b10ad083": ChaCha20Seed(
        "e31f74e424374251556d103122a155e54940b0543224e9f20cd86583f84d7fd9"
    ),
    # FGT_4400F-v7.4.3.F-build2573-FORTINET
    "9de173932ac9c0f201c42a89c2605c6a2d0dc41b6774ae047bd34c63fc491bc7": ChaCha20Seed(
        "283497a1e719ab5f93c9e9d869e42f5337d97e0e197cb28e7a98d3198f164d22"
    ),
    # FGT_4401F-v7.4.3.F-build2573-FORTINET
    "0be6464a2e09943fff0a22ecbb3d4c1133a377c9ba7e4423da5f44f2d81441ec": ChaCha20Seed(
        "3997b74e7693ad0f144bbfeb036afaa16d700de54d0986ac773d1710fbaf92a2"
    ),
    # FGT_4800F-v7.4.3.F-build2573-FORTINET
    "f359993a59d465f13720c865505cfe3a68581cadae8ee2d4eeeb8f6dd6916bc3": ChaCha20Seed(
        "74aa9af95926e0a5d42e1e1e96cfd3a8bb4b67531635a4b886ebbcc1334427e6"
    ),
    # FGT_4801F-v7.4.3.F-build2573-FORTINET
    "9868d5256f6900996b5ec243d282ea8766aa50ad4c6fae9eec172059c02aaf25": ChaCha20Seed(
        "6455923f3eb307db4cda8fce5dc8006496c1ce1cff2c395ed26ebd18312048d4"
    ),
    # FGT_5001E-v7.4.3.F-build2573-FORTINET
    "f6ba3bc6018e47b0ef84dc570379cb8509e78de5f5a9d5fe3d5c1e6db9c561bf": ChaCha20Seed(
        "5e0a76a02c8cb4e7f2d27b485bc8f3be2e9e803e72af7e111314637fd9926d5f"
    ),
    # FGT_5001E1-v7.4.3.F-build2573-FORTINET
    "4687ddb211278c294f3d23f524abdf0206f3c5149bfaa9a20b8360a5a2393fe9": ChaCha20Seed(
        "e290add642e8e5595398e4e8aae077a58ed8e8b79b651787d55a2dadc6ff3d78"
    ),
    # FGT_500E-v7.4.3.F-build2573-FORTINET
    "9c2a98dc04a08e40bddf29a3efed109b8ea00e4946ee34383175c62a7e20a0ba": ChaCha20Seed(
        "9663bef79e06e60c8b61938b182555d6db3f078488cbcb81a1048cc7298aafa3"
    ),
    # FGT_501E-v7.4.3.F-build2573-FORTINET
    "05a44b3fedd95bdd2c1b90a1c8f7795cbd104161779ef5cec16d85f63f5741cc": ChaCha20Seed(
        "4aa1740f56fee5b100e81014c35ec2c25633053146d34dc641626ed833aa09f5"
    ),
    # FGT_6000F-v7.4.3.F-build2573-FORTINET
    "0a2894459e4735ecda36c257db9b72f896e452cbe28db298f9679632c67ce9bb": ChaCha20Seed(
        "25122b668a4b876c712b5fa2fed4f12c5d02737e3f9a6591f14418d58a7abf98"
    ),
    # FGT_600E-v7.4.3.F-build2573-FORTINET
    "c038fe83385d57dc637b4519bbd546a42846f39c00499d77743effc4860ad559": ChaCha20Seed(
        "d57aac34a183d26f4fbad7848508c00fa040284b3add972fbf4067fdbdcd1ef6"
    ),
    # FGT_600F-v7.4.3.F-build2573-FORTINET
    "4964bb3f4f25af170ce29ebdcd890fbc4357da4971783913a33848ce2339137a": ChaCha20Seed(
        "de4409e998f0b463671cee8501dde4526cffd9aefcd15df8c33e531b2c6498d7"
    ),
    # FGT_601E-v7.4.3.F-build2573-FORTINET
    "20cfa85e4ce761094e81821101c8c0a86356d86223087db2474c7b46b8726654": ChaCha20Seed(
        "ced2acb0e8bec65051d08eeb5b4c22684f9400644e01c914538aa8891a632445"
    ),
    # FGT_601F-v7.4.3.F-build2573-FORTINET
    "cd083b0e6a213e9cc9ef87da9298f97f7e2f34edf2e2b6441a5440d6342f6ed0": ChaCha20Seed(
        "0d12012224e11b7e971e60b4e5ac5e56f701ceb8d1aec0ff9ba6767fdc56a11f"
    ),
    # FGT_60E-v7.4.3.F-build2573-FORTINET
    "f29b96e197e213f1ae6b27f34c8960d7d4f0560ad1e446b4ff3314026e12b655": ChaCha20Seed(
        "fc56ee0abeb7ce692d1265c60407bba9722363a84dcccf11ed08bd67d68fd752"
    ),
    # FGT_60E_DSL-v7.4.3.F-build2573-FORTINET
    "ba47c7442c7e2bae989bfa054dea8ddc9dbb9f8671cbd8357cacc03ca3e52d6f": ChaCha20Seed(
        "eddb2fa2645f994ce2ca1965b8283815fbc6fe3c1925002e1b2d0333ed91651d"
    ),
    # FGT_60E_DSLJ-v7.4.3.F-build2573-FORTINET
    "498c02e85dde980d66deee71d15755cafbabcf2827af4a29a751231d3b81afc6": ChaCha20Seed(
        "8e0e59d33746aa5ced2a00744a3c043ecdbcc6b56d89fec82b7657ee267c18a4"
    ),
    # FGT_60E_POE-v7.4.3.F-build2573-FORTINET
    "846a51fe987c6b57040b51f37303731cab0b7bc1995ca14ffe6f33690626fcf9": ChaCha20Seed(
        "d35ba950649fc5e9fcad3269bd5688cc31bcfd17b34664280d80acf2e0fa9925"
    ),
    # FGT_60F-v7.4.3.F-build2573-FORTINET
    "569d7dd505c7bcc55f5843eb6e64f6ee5c25413269fd65338fd9ee01457c8cba": ChaCha20Seed(
        "a505e70fdf117d5e48a3f62bb0083098590f78f5d4dcb8e9d32d9ebc4e5e3a10"
    ),
    # FGT_61E-v7.4.3.F-build2573-FORTINET
    "fcbfa464d95bc9256ea0e3183aaadf12689aa25b249e3840f31593ce1c08556a": ChaCha20Seed(
        "9354b1a205b849678dd6c3d7ec8b28c525086d51abc67692850ee43756620169"
    ),
    # FGT_61F-v7.4.3.F-build2573-FORTINET
    "d33ba7abb98c6c714352d1b4e3f56777708ee326e68c29b7671acc0ed74743c2": ChaCha20Seed(
        "1cc29ad32283330a47a1bb181b9a9281bc2378b7719358e64d202452f6529e13"
    ),
    # FGT_7000E-v7.4.3.F-build2573-FORTINET
    "c0b62417f69072192ecf17fcd253b5a64ea295833a9b25f2ae24c3306b252ac9": ChaCha20Seed(
        "6b5f79f90e04c2756d736b54e4e179b768685a921eecdea641d83e73b60c8af2"
    ),
    # FGT_7000F-v7.4.3.F-build2573-FORTINET
    "e06974675e4f9840180cd28267e27abc2d12650180946fa21c22c8f4c3cecfe4": ChaCha20Seed(
        "f7d2907173c9920277bf1382f83dace8fb2cf76e6b5337c3ca7d209817adc7be"
    ),
    # FGT_70F-v7.4.3.F-build2573-FORTINET
    "21ca1f7cf0752836c0910553e473d4eb20ba05326c8a9f5a17891585ceea3cb2": ChaCha20Seed(
        "0254cdc0e5aaf9904225fd69f2119c57bdf67b953e1146fde1979b2e18973dfc"
    ),
    # FGT_71F-v7.4.3.F-build2573-FORTINET
    "49fc706da4970b5a16fad06034ea160031647edead6f4df6c58607d6a2ed41b5": ChaCha20Seed(
        "9de5ceb6ac0efae1b032c7917cbadd0b3248ffde3187b70d992015e7c33fa55e"
    ),
    # FGT_800D-v7.4.3.F-build2573-FORTINET
    "7aff25c748b6a7c92e27bbdcfcf77aed0bc3cd25871ca90f3e5d0b69ee3c59a4": ChaCha20Seed(
        "620117abc66a1b1eeb3380433d132259fd1544c9da3a857929bc5311339e31d4"
    ),
    # FGT_80E-v7.4.3.F-build2573-FORTINET
    "4bc03ef0ab5520978318419a265f2aab903601a775de685be932151e203a180a": ChaCha20Seed(
        "f4d5cb7f1d80ce961194e1e23117829a1f2224da16497942da3ea3bcff0efd5e"
    ),
    # FGT_80E_POE-v7.4.3.F-build2573-FORTINET
    "e1c5492bb79263bc892f664f1102a75edfca2da1dd8f270beaa58355db7c3092": ChaCha20Seed(
        "9db628a7eb04bad2440f02bb9ac00ec3bc71d12a6c2959308d4cbc50c681c66b"
    ),
    # FGT_80F-v7.4.3.F-build2573-FORTINET
    "82459b46772901ebbafc2abab62f05827c52fc1e8d9c0eeb2c543f9ad0cb4e43": ChaCha20Seed(
        "e54a35f0aaf9764c9bbf129a64ade3b7ce347af28852f8478c490485f850a3ed"
    ),
    # FGT_80F_BP-v7.4.3.F-build2573-FORTINET
    "03b6d691ba27ca213d9a8f43615fee7eb2c38deadc62003117f3acb35f6b6c8e": ChaCha20Seed(
        "fbb8c865af02b6a74e4464f5a2871ac14fcdc19759d42b772c58674c0966bf75"
    ),
    # FGT_80F_POE-v7.4.3.F-build2573-FORTINET
    "e288d99496509c00a467b608cb5047e38ba8b98e703751033c569d166b7138d9": ChaCha20Seed(
        "0e1ecb9a0078dcb666d939f7c5024b2a059d99d5adc8f19e031d67e7e4614d18"
    ),
    # FGT_81E-v7.4.3.F-build2573-FORTINET
    "a2b2e6eac521a50d48b28e96df9eb694c08a6b0726e43bc1a89c78b54c149906": ChaCha20Seed(
        "2f50650b7a1b1fc5e5774dde8b632fa7d98df121b0918dfe088ee5f67c6cbbe5"
    ),
    # FGT_81E_POE-v7.4.3.F-build2573-FORTINET
    "e44583ca5f9aea111ba119af947aa5a4def09746e653bd6ac5c5292a334f4de1": ChaCha20Seed(
        "984d4df74f0c3392506132070ae80a8d516313a5da53230398862ab4d8c7930b"
    ),
    # FGT_81F-v7.4.3.F-build2573-FORTINET
    "ee19d86770ba920d6d11fbbb0c388cb5efc3bfcc7d94f2ffed854c8c1522df46": ChaCha20Seed(
        "1a9ec01e6387fc602baeacc269d60507ccce7be80289e905d1a9a05309b56cb9"
    ),
    # FGT_81F_POE-v7.4.3.F-build2573-FORTINET
    "500b94ab5dd1f13a55eb83ce862249f94aa728949d2e7601c0fd920a098a4f4f": ChaCha20Seed(
        "b51c59e45b8ea00ab9f7e788f5a4f34a1fb8924c0fe196076edc56d6ce1ae3c0"
    ),
    # FGT_900D-v7.4.3.F-build2573-FORTINET
    "370782d542af3f2ce52921cb4af5c14b78c1c04cbaef47fbe38fcc9552167623": ChaCha20Seed(
        "81c21e4190b7a3a8a461fd61d62c361c772ca5afba1ffdac02766c84e75fb099"
    ),
    # FGT_900G-v7.4.3.F-build2573-FORTINET
    "66207c46321d8acec06dd9d06793d47ca621941bbc551937d1377c3b515f110b": ChaCha20Seed(
        "264fc522ee956e7d296de4c86228cd3dedadb68d49c9b18a2a02eaed0b7d83f3"
    ),
    # FGT_901G-v7.4.3.F-build2573-FORTINET
    "d536533736a54526ba567781330c84e9ae1601250caefc889e28c805fb047f6e": ChaCha20Seed(
        "91cd13f7a412c422f0afa994160585dcd5daf38a4324aba81a5dca41f757be7b"
    ),
    # FGT_90E-v7.4.3.F-build2573-FORTINET
    "e92e2c46ce12018f7039b3f6a9c8f0def3e0ae5857bf40313a7606ebb11ba54f": ChaCha20Seed(
        "126bb34e092fa827a275e54695c300d4211d3eeda05426552317987477e0b98d"
    ),
    # FGT_91E-v7.4.3.F-build2573-FORTINET
    "7acb2f38b50e2780bfd9b8e2604f70fc7c8a2a4e8cb1a3f60212749512e91bf6": ChaCha20Seed(
        "81adb1979d5459fbd9a5d06c3c4ba922eb44eb3677d946e0107b96a2492bbb6d"
    ),
    # FGT_ARM64_AWS-v7.4.3.F-build2573-FORTINET
    "d0fd8a81274b57b08d4e14f5f3ddb420860e4794b2d1a551936a6cc54a4466c8": ChaCha20Seed(
        "2302cb858a750bd3fec4673a25e34809a38329595ff195a778cbe213b0d35284"
    ),
    # FGT_ARM64_AZURE-v7.4.3.F-build2573-FORTINET
    "956f5a05d336cd3714e7ff4283862cf052b89ca621e1cdc41ba7512e553684d5": ChaCha20Seed(
        "584c64a64f70081c03505d14a50260d825e0cf78ef006d9eeef9a855605ba263"
    ),
    # FGT_ARM64_GCP-v7.4.3.F-build2573-FORTINET
    "ea13f87536965c50c628d865dde5d05b1a7229fdb856c8f95c9baa78f4356b66": ChaCha20Seed(
        "30a1429124666084406f7d17aef13e67bfc127e15827957d03a8d4de76dce8d0"
    ),
    # FGT_ARM64_KVM-v7.4.3.F-build2573-FORTINET
    "0b6c678058e48add5786444233438c97da87cc0134384e38237e8decbb287e1f": ChaCha20Seed(
        "e6690a64563c3f8b6a2d628aaa5d10e42d1ca73b973902f04d4fdc1169f0e47f"
    ),
    # FGT_ARM64_OCI-v7.4.3.F-build2573-FORTINET
    "ba4226140c8f06a2900975d589235ba509b3e98899df91bff09e2888ad3f8f3e": ChaCha20Seed(
        "466d8f0888a4989eb094bd2364fc24e9dce76ffef6108076ba3e18fb7121ba5e"
    ),
    # FGT_VM64-v7.4.3.F-build2573-FORTINET
    "ba0450947e51844588b29bd302d2a1a3802f7718cf6840011c1b34f1c1f0bb89": ChaCha20Seed(
        "4cf7a950b99cf29b0343e7ba6c609e49d9766f16c6d2f075f72ad400542f0765"
    ),
    # FGT_VM64_ALI-v7.4.3.F-build2573-FORTINET
    "7ff5564f07c7d01caf62c5b8bb117dcde87c86e01d1f8f4ab1ce2e6c697b7457": ChaCha20Seed(
        "ac80c561989829b4c6089fa504b10053a640fd0b40e2c6d40ed6b1146e1ae70d"
    ),
    # FGT_VM64_AWS-v7.4.3.F-build2573-FORTINET
    "02b66d72be45af0286bfff14a12c1f61b6056f5c0cc90d9a7f6ee3939d3731b0": ChaCha20Seed(
        "99b34081d2cdf769318475f4f51cf9379501160f39937a9ba1f552e8dfd39c87"
    ),
    # FGT_VM64_AZURE-v7.4.3.F-build2573-FORTINET
    "e13d9002fc056c67233b5455f2e244263bfc95dac3261f1e029140a8b7cac37f": ChaCha20Seed(
        "01b453db8d358d2cf8928afe879e9129685e28c630efdee891a405677ab5fa12"
    ),
    # FGT_VM64_GCP-v7.4.3.F-build2573-FORTINET
    "5fc5e652bf02b3f42a7e1d2fd1e60a59e5af683a9bab5329ab0ac64d196f34a2": ChaCha20Seed(
        "7bfacb34cbbeda0f81b197cfed42220841210cf75c0806f0d5471f61a797e58d"
    ),
    # FGT_VM64_HV-v7.4.3.F-build2573-FORTINET
    "8ddbab6589233844d7943cbcf48de13d2816c8db2b85edc129e24fd6e68bb7d1": ChaCha20Seed(
        "042d15d38867f65b5b6719f09b4d5679d6c9278f6aed5cfe45a87804311a669c"
    ),
    # FGT_VM64_IBM-v7.4.3.F-build2573-FORTINET
    "a3f3de98f208ff20047029068bec8712e70da257b70212bb1f1b13a24e61777f": ChaCha20Seed(
        "38c2d0c81af5f96c5a92fc8b1853f38c0d677f761a48f14c757a2b1739598133"
    ),
    # FGT_VM64_KVM-v7.4.3.F-build2573-FORTINET
    "03fe6b59525a8d4b0e1dd789bbd44c95c288255a87d78255f1c774a04e2609aa": ChaCha20Seed(
        "64693c17f02c68ac537b30796080f9819c01ae89a4d5258c4f9485404847a025"
    ),
    # FGT_VM64_OPC-v7.4.3.F-build2573-FORTINET
    "c1d656b60c1e200b846656c84779c740d5f73bd13c631b70f143b2af0ff9d5f7": ChaCha20Seed(
        "7cde9af207208d0483a13eebe85a6af3c857616e71ff269d8bac0b7915c8d507"
    ),
    # FGT_VM64_RAXONDEMAND-v7.4.3.F-build2573-FORTINET
    "c57fbf31e70d8584823f96af1dabdf5178aa5e10703bbe446590b4505054de77": ChaCha20Seed(
        "c29c99ae52259613e8f647a6ee3ce082d321c8a6a7f37c8cdd0acecb8f44e017"
    ),
    # FGT_VM64_XEN-v7.4.3.F-build2573-FORTINET
    "8951cb7b09bc3db2b8046d0b07fcda9ca83896e2fede2823ed72f4650398de48": ChaCha20Seed(
        "9cf653a43614cd0ec349b6f1589c9911badd5dd9181b29eb2c511e6362279506"
    ),
    # FWF_40F-v7.4.3.F-build2573-FORTINET
    "f4f22eb0cf8c45c4c7781676c0f43524653fced4cdc24bcb104270bc987da425": ChaCha20Seed(
        "30690d3b35387ff1f817a35a00fa6187840fef6a0d84ef3f0144aca47edc319e"
    ),
    # FWF_40F_3G4G-v7.4.3.F-build2573-FORTINET
    "c5e78caf91976def879382c858fb19ef55b7d042b369f6cc8fe79226994c2841": ChaCha20Seed(
        "49c2cf5158dc8cdd04ffab1abf6f9280da9fa9ab68eead124af958a03f032ecd"
    ),
    # FWF_60E-v7.4.3.F-build2573-FORTINET
    "31f3773c79f50f8bf6b87cdc4b5fab93b47fc39a98317566f7d3fdbad03b078c": ChaCha20Seed(
        "5ef0a3a9bf70902e53d3b763ac073a7a5008e5a4962f6538ac7829e1089342c1"
    ),
    # FWF_60E_DSL-v7.4.3.F-build2573-FORTINET
    "b01ccd2a345f5e4e26a617326b6ed102a08d2528c552548c4e68a26f5d392345": ChaCha20Seed(
        "1059b71f798d20dae998130690b1c5407336e39ae3ef6d5ee5be222f7879e7f2"
    ),
    # FWF_60E_DSLJ-v7.4.3.F-build2573-FORTINET
    "e26e828e3129f10650a7b5d040f44b29d664718f58fffa56eb74abe837f3b869": ChaCha20Seed(
        "48cf7fcc44177e8649c3b9425f3afb4e628a5ba5f7179d0c8839e0865fff7572"
    ),
    # FWF_60F-v7.4.3.F-build2573-FORTINET
    "7ad838c2a1d02da37c76ea5f0edc0ec208471ecb01de3f9b32eba08a23ad3599": ChaCha20Seed(
        "f1a7527041b162c2f792085b5aa69982284af851e36676fbe7528c06bb2ce313"
    ),
    # FWF_61E-v7.4.3.F-build2573-FORTINET
    "be4d2903c4bb2ee9d5b2c5c0d124557f792b58f222952e8ad02d55092a2ee48f": ChaCha20Seed(
        "1e907f45969521939c85aec9d2af39c18d9ac2069b32b53b9f80d49f1cb10fe1"
    ),
    # FWF_61F-v7.4.3.F-build2573-FORTINET
    "837d8c07f5f51dae250e0ea63dd2b390ca2880c1bbaaab056044869b59bfd1a4": ChaCha20Seed(
        "342957358102b52a760987b70c842feb6fde1339833a44dabd11a777369ab4d8"
    ),
    # FWF_80F_2R-v7.4.3.F-build2573-FORTINET
    "f04841c636a6f7c1cd678f6b1a66a3c3375904d72d8a8f50a06f41bce33418a7": ChaCha20Seed(
        "da0e926c0867302003b6918b324c58e424ec3a171d3c5c7040a0a95e295e0c69"
    ),
    # FWF_81F_2R-v7.4.3.F-build2573-FORTINET
    "83d4e6bea7c3667d05072a7d4b7f301322d69164d5b7be0b242a8a7dc04511bc": ChaCha20Seed(
        "1a5b92a1a2fe3bc688e78c5d6296a997264b3395f78ffae42f94343ea219a46b"
    ),
    # FWF_81F_2R_3G4G_POE-v7.4.3.F-build2573-FORTINET
    "170786f39d54196144bb34b2bed561a4035018521637756d7ae39d680d8dfcb5": ChaCha20Seed(
        "1593383972801dd60ed435668b1b8cb40e503224ddd43848732dd427c59e3071"
    ),
    # FWF_81F_2R_POE-v7.4.3.F-build2573-FORTINET
    "29882e3f5945fe05459f8e890f092947575331919ec3efc8687fe1c8dd36de11": ChaCha20Seed(
        "580e131453a2f4c38cd1b6a36707d73155bebc4aa7d95e866cf5faa63a87616b"
    ),
    # FFW_1801F-v7.4.4.F-build2662-FORTINET
    "6527ec7925e07756d2e8a806bebe1d069347a7cc5a707b7f7eac48e071bbfe34": AesKey(
        "74212ca3e90566668291f88dd657d898c249322be0c761c680354edb233cedf1", "a3cae22f501ace02418e6ac0229418d6"
    ),
    # FFW_2600F-v7.4.4.F-build2662-FORTINET
    "40a6e2827531bbf34d977ff7176d524ba4880d31fc2639895d9dd4b358abcba4": AesKey(
        "f5ec2c8fb3ee0f12a7e4d851b3086278c0a6e5d8a2f8feaf0c8bd083c73d0d9d", "d525d6f28153be6624247c2b591188f0"
    ),
    # FFW_3001F-v7.4.4.F-build2662-FORTINET
    "e9822bc87914010df7a4f8dbc499c44149d45b35ab7831af941c62aec9d5e0ef": AesKey(
        "09f3bdac792ddba498f55d90a32c570aa80bcffca1b82c0055be355e36ccb80b", "145150fa6bc02bb8872cdbe3828361ad"
    ),
    # FFW_3980E-v7.4.4.F-build2662-FORTINET
    "ca23fd9ac91d90ee07c3fb71287c07eed4f5d4721cf34a833077307c8f7e475a": AesKey(
        "7493b1756c7af34e2bd7fd4c5e063cf9b9d5e395a043343b1698cc5b3450017c", "5db9ab52d7d63222c56359ab092ba126"
    ),
    # FFW_4200F-v7.4.4.F-build2662-FORTINET
    "20308c68cf7f5e356f1fffdb248f20013f7e738e372fccddd3c2bcc724ae3955": AesKey(
        "490bba371fb40e74c1f29b00a0e0d0ede3db1765fbd9a8725db02da868666652", "1b7fa6dc4c10c3a7366a4ee555169b62"
    ),
    # FFW_4400F-v7.4.4.F-build2662-FORTINET
    "8ec0f81e5dd0ce2ba3195e1dd68eb00519a2dfa71c2959a59af8707db86a3abf": AesKey(
        "617358683b154cd2892f5ff6d681e2de78d242ef1bd8ba361c09acf280e07fba", "f23d45d27c7efb43d831037b733b8da7"
    ),
    # FFW_4401F-v7.4.4.F-build2662-FORTINET
    "6b9fc1aaf81eda7d2320614bd4484d7ca5d3dde5304d0779fc6f2c3706bdd0e1": AesKey(
        "0d052ab5117b0ba35e1bff0824a903efa720543d223ba6fafcdc7bf6bdde83b1", "6cee1edc6f95c0fbcaea069be6a4b79a"
    ),
    # FFW_4801F-v7.4.4.F-build2662-FORTINET
    "8cac6cb03d62552a8e175359a3c56274f06addd6760c017e3edead1b4327e03c": AesKey(
        "a22eed1191b082432618829d6f6e949574ba4826c59d2ee143fda7aac6793bb3", "b58141616d7b5c4763feebebe9e0cba4"
    ),
    # FFW_VM64-v7.4.4.F-build2662-FORTINET
    "a3578c40de0437a8eb35040fd12fb615eaf4a073cba3eb7263d8c967ffe25a59": AesKey(
        "39c8ca5572173940627c391290cd59abc7605aa0c8c74831bfe0fd318692385e", "d686e9025dad357a4453af32af49b913"
    ),
    # FFW_VM64_KVM-v7.4.4.F-build2662-FORTINET
    "63bab86f0d4667ff4d34fcc79ff326a2c9e576d59ed6553c1919a9a3366c1676": AesKey(
        "a8bd56e20447e31f0c8592c9dd77ed234240f29519ac7eee776cb0e1d7dfedd0", "6481e7e3059604f61d34249c8a51277e"
    ),
    # FGR_60F-v7.4.4.F-build6003-FORTINET
    "1be7062f7989324acf05fac953a5fb880f4c7d1872a8430ddd31eacdf13028b0": AesKey(
        "833ca689d87531861b1eb2ff3312c1735396d486cd932f2d9cd3f53c8e21425b", "ed27b04ddf6ed2b32beb071914fdc037"
    ),
    # FGR_60F_3G4G-v7.4.4.F-build6003-FORTINET
    "50ee8d1434ba7a08357b6e7526a65d863e8d153ed6b6026e1f5dea7d1c96972c": AesKey(
        "f5d8ed444999aabcf1402ca5d8b4e3b56921387eb9e509c2382009d8575b9f2a", "3aaf191276abc11a46bc3da9a5bd749f"
    ),
    # FGR_70F-v7.4.4.F-build6003-FORTINET
    "0ce5fb0c5764e35192caf822358777c6877ba0b4c619f00b03ef051c8ce49bc7": AesKey(
        "f21080bc5bade5d13ba12053056302989902c2493ddb0d980b18efca1b7da742", "a6a745e166c879652dadd825e2e22e72"
    ),
    # FGR_70F_3G4G-v7.4.4.F-build6003-FORTINET
    "9ab99cfb2095be4f597f572511adc2dd2e9e55f6317afec29ce6643138a8bb42": AesKey(
        "67e1e2b56edc4007f5ba5c7618db8e981877108cad8b0dcf18cff556cf701ff7", "9fe42c0a55e5d8602946c64ca3a3224a"
    ),
    # FGT_1000D-v7.4.4.F-build2662-FORTINET
    "2a6570d0c3fedc06170f3d82185a645ad42eedeacb1593b56f32ed897381c790": AesKey(
        "ed9d1f94de6d32a8bdb0f20e1dd14d09d8484d64d9eb8d5931cc921d4a1bc05b", "94e472ecb701ea7ae9dc38db03c5cde3"
    ),
    # FGT_1000F-v7.4.4.F-build2662-FORTINET
    "d042914ac62d748423ab3951c0a161db4012da14e478d7d0fc887b522e7668b5": AesKey(
        "6cd535efff8223aaeb94a9d7dab09ad70ba8cc1f2754a84b0efc68b7dc79a7e1", "fcb218b025685f6328b9a30bf6a4a464"
    ),
    # FGT_1001F-v7.4.4.F-build2662-FORTINET
    "6bcab489615c0c537e389b01f191b2353d1ade20c4d2f96b433984794a8ee3e3": AesKey(
        "a3038f32844aaee99ce592bbf29090aca220bcf2bfe8c62da88f0d0466ad9d76", "a657968321001661861272c6a8acd54f"
    ),
    # FGT_100F-v7.4.4.F-build2662-FORTINET
    "480cceb24fb72ef98f935af269f19c23a4ab3fc59f45cd8ff6dec7bffd3e273c": AesKey(
        "dffba94bcbad1c74da2db342c776478f16e900477045f26f56a54dfea4c2b413", "01b6fa809a22de250191739b295ad87d"
    ),
    # FGT_101F-v7.4.4.F-build2662-FORTINET
    "70505a316c3c339851fba9c4591c6960b25cf95c1c2973a08759d8403e3e5767": AesKey(
        "2518d85c087fd37cc39ba5a3c30cf814ef1e9e1a0df6d12c3b2ff319db0ce336", "cb8adf55c488c73fd03c34bd6cfd131e"
    ),
    # FGT_1100E-v7.4.4.F-build2662-FORTINET
    "73234991f8a7a6b7ff361bda9888b2c3425e45531708b0305d952b274f3faf6f": AesKey(
        "44fe2963d84488a70b45017541618265c88cc2e1bede0eb1b5ac09043057762d", "c4da3a5664c5e7d99754cd88f3034a30"
    ),
    # FGT_1101E-v7.4.4.F-build2662-FORTINET
    "9da45f976d8814c2f44ec85aa55984cd93f0538e395644e554decb4f20dcf043": AesKey(
        "7f3b4d7c96daa9b5103472cc459c2019598ba8271ba8c158fad95b9e34cb7e33", "76389b5363d89ec97729fc8016c3b5b7"
    ),
    # FGT_140E-v7.4.4.F-build2662-FORTINET
    "994f1bd5b1d7625c21f9a4e003ca2f9084258e65b2a66586ea333479d909b2ef": AesKey(
        "eec4234ec07fc9519a912b05f024cb0ffcdb912c4a372bc63622556bb56b847d", "48fd369b0d3850bead5e5d56e3925872"
    ),
    # FGT_140E_POE-v7.4.4.F-build2662-FORTINET
    "2d15f940d091d890a131ae12eab5e5a44407babf19a1f364d63465498a71f5bf": AesKey(
        "953f37ffb0e146b1c293faf7b91c1c90fd2e230f790e8b17c33981bbac68e34e", "2022e8ef03c96f33aba4560225f2e079"
    ),
    # FGT_1800F-v7.4.4.F-build2662-FORTINET
    "b9aed505a266fe3a19d2f18dfa00a22a22f30bbbef55a26f42bf47760a2c0859": AesKey(
        "7d0a6d64ae80ee2d68a49411eeb32aae60db93c4bed18824114b1bf424d16737", "2e0021f7868038958940e97a6475d479"
    ),
    # FGT_1801F-v7.4.4.F-build2662-FORTINET
    "6b20027eb11b5c5ab5d29248f270181f3629a3c6f433885e910d01791c476151": AesKey(
        "fcc5fedc58d9c4f15dfde25c8d444ca638a0c9035506725245bc436de44ac454", "992b0bac1ab7ecff012c3093761532f4"
    ),
    # FGT_2000E-v7.4.4.F-build2662-FORTINET
    "9c8562bb7037e56e1fdd93be479c01b0fd0d75d25eabc4e7b5d26fe29c8900d0": AesKey(
        "a594ef2be3fdd64290f575819bcb503e3d77cd4fe6b1eedca84d33c516adba01", "7f124a3f2de89a54ae546cb8102acb81"
    ),
    # FGT_200E-v7.4.4.F-build2662-FORTINET
    "36ebec63fc20dafc5ed85636c312a9b028c5792429d9e00a691e34e181177043": AesKey(
        "962507cc78774437669d543fb7ac294d57580bf2b790674805893dd4e74311b1", "2e4e0ba7b779246a28494354ee84b2bb"
    ),
    # FGT_200F-v7.4.4.F-build2662-FORTINET
    "47f5139af74b9dcc596ce58f696666e88e22eea3a8d08be40a1e463de478ab4d": AesKey(
        "aad5142254ae474372773e5ebdafef2273b62654668c6473ee3400d413da4c4a", "567a2a502f1787f7a8e6ccd6cbef5592"
    ),
    # FGT_201E-v7.4.4.F-build2662-FORTINET
    "ff91ff30a8a0ca57be6672e3b9f5e5cfdfff94d4b97f119df27a98ab40a96a5b": AesKey(
        "a8a2e5469e25d45b8c1d42fcf2eba7297a4c9d03453b27be3ed909fa081d9196", "987ab869e5395fd8622e3361de87be19"
    ),
    # FGT_201F-v7.4.4.F-build2662-FORTINET
    "6fb3095c0e4e1c05f8d26d017a7bb0b31d737f279b19c35230975fd82339c19d": AesKey(
        "dc7d80ea4c98bc1c3ae77e815561224093ec6896397036b5b430bb30907ac380", "c2559b0aee0c6ab1984975381ea8291d"
    ),
    # FGT_2200E-v7.4.4.F-build2662-FORTINET
    "28874d876ade99584c4bda4764f1f058a8f453eb70eb59f9709d6867be89bbf4": AesKey(
        "136fd7469489871723c9673196a8701a1cd89053801fa1edfb66c8a3a31476ee", "620ba9a5721411c61d414d1b654f8104"
    ),
    # FGT_2201E-v7.4.4.F-build2662-FORTINET
    "30a3fffac5ff99e3571bcee2280ae2990b67f257be97ca280a18ad3285d2c9e0": AesKey(
        "117672267b9e7c356563f902f94bd085e02243bed14c2ae5f48f57e958269a02", "4cb72be82cc3a38277cb921046f87564"
    ),
    # FGT_2500E-v7.4.4.F-build2662-FORTINET
    "99b9c50b4b755b9624b4431828db1a49ccc9c0115d47a8461f0dac8090ca853a": AesKey(
        "e0407e9e21ef466a2ac1012a472c43200a3acc214be255d9be0bf4c0b1a95f90", "1571cd4d14c2407df85640f931f8ba61"
    ),
    # FGT_2600F-v7.4.4.F-build2662-FORTINET
    "8f64db87e469794a4e2aaef03b14fdcaa66a4ef287811752af0f49b640c2c4fc": AesKey(
        "17d2f78653e297c34b2e4a909e987ceb18c0b44b8cec11bed517a31f609f8e99", "3c408d4ef170bec993d36e39ba09323e"
    ),
    # FGT_2601F-v7.4.4.F-build2662-FORTINET
    "5fe716f00e59845dd5225a78c9eae6580032db410027c011f7ebdb0d5db2905d": AesKey(
        "a3c7d06415ff0e860c369a4cbb1352cb5e5f20907c82ede6febda7331819ea0a", "2d67a5691885e3d526f9db8195578958"
    ),
    # FGT_3000D-v7.4.4.F-build2662-FORTINET
    "4e18a1e94dfabe14c0f71cec9802cbe518442decf98fcd4e3a7aa0a2128db8b7": AesKey(
        "f42232e501a5c503878bcf27c96bfffce8b356f012010be4f7e63bcb6091e10c", "1368978b55201d166a9d5ff43104729f"
    ),
    # FGT_3000F-v7.4.4.F-build2662-FORTINET
    "4bff65e12a711de4da986d84beb978e124febb4f0a7b4401606569ab7f2924ac": AesKey(
        "9d9fc9257f585c8b9eeb8618ad92515c11556f1f9fa4caaeb7e0b527e6a3935a", "965ef2590f6bf37827f31e1cc1a11f9b"
    ),
    # FGT_3001F-v7.4.4.F-build2662-FORTINET
    "eed1e3a55ad997ec5bdf72e7c883114d13df2fe424d96f0f78e0247c7ec46ccc": AesKey(
        "fb52e1ddb57a824c2da93aab9e80189c5ab83bd2307f7c33dbe065df20b5a311", "b7a85ded102867f111bb7bc2993b4d14"
    ),
    # FGT_300E-v7.4.4.F-build2662-FORTINET
    "39d9ee16299a7dbd079072347629103366c06591aa7459ef0e7038bb325cec09": AesKey(
        "9121ec8f4616f9929f3060ace174f1d301bc95a1db75e16e26e2d3ef99320959", "498c94e9ef9f40bacca4e4ab6d584fa5"
    ),
    # FGT_301E-v7.4.4.F-build2662-FORTINET
    "05e1e24edf70a3c4c333520fe81ce0f54abd9a605573cd4e9df9efe5e5dcd70f": AesKey(
        "b3122cf4c0e6cc62f4afaeca321c8063a9ea533ff8230e6f2c9541b65b116447", "94557080788af42425ff357180a640bf"
    ),
    # FGT_3100D-v7.4.4.F-build2662-FORTINET
    "bc6cff2e0074ba39d27cba7f17756cd5ff3015a97e13a15449dae3ec9a1aa957": AesKey(
        "dcc8bef10cc2572797c2e84fa0069adfc1eb5f64299cd5930766aa8aa31e33bd", "812c0d2af6ace75c4826b74ca3d8af8b"
    ),
    # FGT_3200D-v7.4.4.F-build2662-FORTINET
    "589afeb826af30301faa75738c92f35ee12dca8ad01897790b67f8195f9272a4": AesKey(
        "ec94e1dd97a438bb5963c65c311f970176cabb40a467835364eabe6cb55365db", "6968751624c3934a131a55dc28d938f8"
    ),
    # FGT_3200F-v7.4.4.F-build2662-FORTINET
    "22d2df8c277952cc39875e51e6201ec51091601778526742515c53b5c425600a": AesKey(
        "c24c1b731b5f5d734231293081d977022c60ee1293705980144021bf11406f35", "20ff96164634f439c097d853a67072f6"
    ),
    # FGT_3201F-v7.4.4.F-build2662-FORTINET
    "79705d4c1ed5731ac896ce7c38ca0291be428ada504c42f91201fc296de0ccb4": AesKey(
        "cddcc3c6d5e57b101ba66f73c02c3695caad48fb0411c99b68cc70167861679a", "a657f4f3b3480c18bbf9bdbd35fce320"
    ),
    # FGT_3300E-v7.4.4.F-build2662-FORTINET
    "24677c33c39ca74c0133073c215edf6e3abbfb3a7313cfd2145d18c149fbb985": AesKey(
        "8ddf1f4ca1ca4281d055f97615599ec9b4a42a7370461d9d3b782c46d88bcdc7", "7d8212fdcf991cd84f604ff67f6143c5"
    ),
    # FGT_3301E-v7.4.4.F-build2662-FORTINET
    "9365c4275b77d2c2271b5aaf751d21c0d9f56158e9cff8767b233b2b14eea034": AesKey(
        "c5a0a71bfa2ab04b711a4b551aaa106e697c739bb9d39574a1c7590b24397af0", "176dd190341057aa2d8f79ff5d4a8619"
    ),
    # FGT_3400E-v7.4.4.F-build2662-FORTINET
    "872197d13af669b7d301e287fffd8c166cd1e5405a7e6df2371c994b36d87a72": AesKey(
        "af57cb82fbe51fe8fa98642d595c92fc1aff5497901b069f7f1b3e1c1f6e6e89", "2c89ca5bdc69301ae59904d05a5da1ce"
    ),
    # FGT_3401E-v7.4.4.F-build2662-FORTINET
    "95e6def4e741eab1101a89d1b9a8af6004204e611c157876c274661964e9e876": AesKey(
        "6e69e421f428fabe68bc552fe2fa319025d0ad76e7711622984aff55efbde331", "7325bf23fad5b0e7ff5bb169ad1a7003"
    ),
    # FGT_3500F-v7.4.4.F-build2662-FORTINET
    "98ad75f92c9e5cc2777530ed4f5d1da4ea3061ef2c1292d1aeb9588de3f63794": AesKey(
        "9279811bdb82cbf3ba9e82c91353bfa6bd8d3695e173fe167c7e595a393cabbd", "0bd5bf5fd33802d5b988967656c8e493"
    ),
    # FGT_3501F-v7.4.4.F-build2662-FORTINET
    "ed51708e3303df5417176cce3ebda5a6e5bd3b4e1dbe07892c7394ee79370fda": AesKey(
        "634ccdda05d98dc1872b32b53377595044931d2e14f333024c2c3a3ea5b9607f", "5e54d0d2f9808c00ae56e2995753c978"
    ),
    # FGT_3600E-v7.4.4.F-build2662-FORTINET
    "dcf7a1cdefe4739389ebcd77efdc847f5b8e3a83b123c56284bf4334d426b76f": AesKey(
        "d3c33261ce448da75a500e341377234e2a590bdcc58af33255d425dd05c938c3", "4f9d6262635a999a5329be10a418fe7f"
    ),
    # FGT_3601E-v7.4.4.F-build2662-FORTINET
    "a575407a5454114287a9e893341cd46eb2cb3b86a97a0a3f873e11ab18129b3d": AesKey(
        "78123ad50f19b21992b63f4b6efd9570cc0450de357bf1c7e4a8e59a49cf32f4", "2b2fee3ec8dde848131d2a08f480fcc3"
    ),
    # FGT_3700D-v7.4.4.F-build2662-FORTINET
    "bccf6d40bfaadb9fe2e28106d4c8db741d68ffdcdfc14f502a5ae6cd30026bd5": AesKey(
        "f1c9a377f0dd1032b2b1a92be36c21c323e7f89a9752189fca7f4ee2250c0ccf", "e47a72917c109a909af1b5c2319d385a"
    ),
    # FGT_3700F-v7.4.4.F-build2662-FORTINET
    "fd8f61042a58e5d02186d1f8edaad9e7d772f85b76a7fa2a76d1889328118c16": AesKey(
        "848c4ca4bdc9ecd387371d42c606c956f1493e72424c755355c56af2c2bebd41", "191d0139e8fc827d6089dc52fec5164f"
    ),
    # FGT_3701F-v7.4.4.F-build2662-FORTINET
    "0a11b7ff5caf99727075352906b202d1325c4ae72c36ecdb284f0b4d4897326e": AesKey(
        "a2351fce5611a27953b5e448b18145cbaf47903afa20a41398cd3bf68a64e9b8", "64fa8ec087ef8f972d6bce46b154d105"
    ),
    # FGT_3960E-v7.4.4.F-build2662-FORTINET
    "4f1051c5cb0a5f141002eb39f8dea4ff4399ed9bc36b727fda866d70f82c61ec": AesKey(
        "b9def74cb76d8bce35d240dd723d7d9677b522adfa7d97b5b033da1bd85e9421", "ec4a0c6a4ba2c9de6bd5ad477b2a514c"
    ),
    # FGT_3980E-v7.4.4.F-build2662-FORTINET
    "3271915381fbb43ca3932f8d74e6f1768f4343c54a7fa710fe84bb36e8030fa4": AesKey(
        "2c3710d88e363e6bdb875744cbd8d51f148ee9eb0bf56608750ccdae937540a7", "28c25502d7ce4be3dc5afa48ee8b99ac"
    ),
    # FGT_400E-v7.4.4.F-build2662-FORTINET
    "35a97595d112d38f2f352759f0899b55bc36eec6cef5efa33a086e8668baa167": AesKey(
        "c4a5c61285183a22c2132ab0a463042adf9aab7e927b3bbee4dc8b2deea35271", "67a97ec3123caceeef6a5eca854074ae"
    ),
    # FGT_400E_BP-v7.4.4.F-build2662-FORTINET
    "e9f166be61b9c42bef67a2949b27f0f5ed9b365b45aff254591b0ffd2f09e772": AesKey(
        "b32305962a1fc7fbdfddf4aa64ad5e007b2674ed43264f88cd5d8cb7e8ed992a", "cd5c7a500732f26427c8efa394b07428"
    ),
    # FGT_400F-v7.4.4.F-build2662-FORTINET
    "5b291c16f1aa9c4f8fdf55d3aa5d8f53a083116e4fe173ac22bcbf4235b49424": AesKey(
        "3e33c328cd27d981745de7ea8fc06359902bf8b91babc14b4634a78d9ba7ed9f", "7ae663933aa290063fbafed9eae89afe"
    ),
    # FGT_401E-v7.4.4.F-build2662-FORTINET
    "4badc831a416e39724716a8e92b592827ad99de2902c2a362988b65335b11bac": AesKey(
        "b34d512a1c3bedd993e50ca6fce0d2e09a86d47d33ed96ec00020ad9fe2c9aa5", "1bfb9129acddbfadbd8997d3c25cdbc6"
    ),
    # FGT_401F-v7.4.4.F-build2662-FORTINET
    "9b676859ddd0e62907fdc97a5e4a860edc4ae522bcf6f7e0890658dce9949abe": AesKey(
        "65923540859fbd76bf38e9eb58991388ab1bac4a76fb4d2901e3d7f84cfadbe2", "1742726263a265563246f32d83bf7815"
    ),
    # FGT_40F-v7.4.4.F-build2662-FORTINET
    "c0c0a4f8f2683ea739bcb4312535cec074e9f70d0290f138a04cbcd6b2f39055": AesKey(
        "0e00eb83ec8ae074650937eb84aafc0ca0ffe833dc21424e54cf94b8ae3a1f58", "1c4afe4d5b14c281a177f4e812c2703e"
    ),
    # FGT_40F_3G4G-v7.4.4.F-build2662-FORTINET
    "aa98c68969dd70f72a1da980b2b2af425782b2b5530b9456672a69ab2303dea7": AesKey(
        "46d399d411481cd2d5e7623c5a6f3228b7a657e3565ebda645a20fa85d4ae712", "5f29949c09a6fcdc47441e369e0b637d"
    ),
    # FGT_4200F-v7.4.4.F-build2662-FORTINET
    "baab1b5f526e8939ef76fbf6cfd578401de4b3a6990e6ded86ab21e5019d11bb": AesKey(
        "e7a50c5a390a3229e42047732b3304f9abb3aaf384ebba738309e4529c913620", "4a292b0d965dc335c6952e242e9509bd"
    ),
    # FGT_4201F-v7.4.4.F-build2662-FORTINET
    "a77e4db3ad2b112b56a70341ed7f1607427fad5a489fe58b1f8471954e45583b": AesKey(
        "abad8b78bdc73f071724392ae311c15d44fcd445e6f59a1a325239f8c8182d2b", "11d390fd52044c320c1418d2d44ee37d"
    ),
    # FGT_4400F-v7.4.4.F-build2662-FORTINET
    "baa92dfaedde887c00495d0b65367821d135dc745835b55ce66b38bfd8afc05e": AesKey(
        "aa3a40d332a54d98426debe211b58e853c5b1dad30a2ca5d85326a7cddae954b", "ab9bf221d68371d1d47e8ab28accb6a0"
    ),
    # FGT_4401F-v7.4.4.F-build2662-FORTINET
    "e8f1a7fef9cfa21c2116e9bb8cc92e26b8b24e78439838ebc23323c53a4b3628": AesKey(
        "2177e79be0420de0df6101c2031db9040debcd1a9aaa734d1956012ad8479f93", "b33231c8f45f6b7f2aa29ca5b3adbacd"
    ),
    # FGT_4800F-v7.4.4.F-build2662-FORTINET
    "07d1c8de287f8d45caec3853224e3215617653a2eb7a98ca946d8b371befb74e": AesKey(
        "7bd7f5d616f7717ae255124aa6aaf26712147220f93d36d85363b253747ef3f3", "2d9967d2188e9498e6095c7697e42f50"
    ),
    # FGT_4801F-v7.4.4.F-build2662-FORTINET
    "a7bf682f53fe332f5152fe0f87ad9d696d68e8045d167893d8cc7af704fcb095": AesKey(
        "0a1c78840aa4a29f7b95646d0f8491a6536120690bf6bb443dd0194b1e1be859", "ee35c5eb33d3c6a3e1e8cb01098adf06"
    ),
    # FGT_5001E-v7.4.4.F-build2662-FORTINET
    "bfb0014d20c36cef75e56e914a72f5ee630323a3165712323add0d8581c5c161": AesKey(
        "39c68f857ce58205d976785b8aeb43597cadd4d200a83cc489a1859c32e13f5c", "9900a315b970f5372ed59761f6e4aa02"
    ),
    # FGT_5001E1-v7.4.4.F-build2662-FORTINET
    "5e3348231d2d3978ffa96f363ec5f4bf58a0b596b70b76cb7d657d200b194871": AesKey(
        "3aadaa0df2ba9a91fb962d6533146c6a9cdb25036e0770d151e49f6f66f7110b", "d9a489f007640031a3cc0102f63c558d"
    ),
    # FGT_500E-v7.4.4.F-build2662-FORTINET
    "7efc7a231f29cd42a2603ea49ddab1f14d188a46b49efc35d8f7694d9d21ed01": AesKey(
        "0cf04992202a9a6582d472dc6abd8b0e90c46202024a3ecefaf34eb789a3024a", "25c1faed11eede56aa7c42ca78fef4db"
    ),
    # FGT_501E-v7.4.4.F-build2662-FORTINET
    "c8bfcbfc434ae73e5a3ed4241d230e0e28eacb1b695115baf9a585ec2a2e714f": AesKey(
        "8cf067f59843f3ecd61a465a77b3e4650466626c0b71f7359d4f7e6d772c256d", "c31a51776ca9d7f7da09d5eed4204ec8"
    ),
    # FGT_6000F-v7.4.4.F-build2662-FORTINET
    "b71b09d9d5309743e46ae653f33b11be4527e36d5e43284e503c4ee02e78790a": AesKey(
        "4ffc608a403145830e9c220999697aacbce70d75621309ac8636ef9c26b66cdd", "f4eeacc2e804189758e6008ae3d2e393"
    ),
    # FGT_600E-v7.4.4.F-build2662-FORTINET
    "b1959dfd93c66f5a12d61624d1f63db5a83db778f1f235ca436245364f57f82c": AesKey(
        "5fdebeeb78fd7403464dcceb8c17128325c0e858a5e016c3e0ce21ad27c0ad1c", "deb2c63cf941f519f9754c176c91987d"
    ),
    # FGT_600F-v7.4.4.F-build2662-FORTINET
    "832af7f3575c9ff878370945069fb13521c7093dce539f7b7e5d79681ee071cc": AesKey(
        "17ebca9d816560a13434ac7d776aba14d3e251b346b1c3129d01ae5a5034b999", "9e78c21fc50d9a5f55d3896afd570adf"
    ),
    # FGT_601E-v7.4.4.F-build2662-FORTINET
    "f81bbd925559820ed8bd23c86f1fa7d51372b498d8355ab71730f478abc5af89": AesKey(
        "bb9936f6163a6bd22ef7935a14fc8c91d151fbbfe38cf11afeaac2cb70c41811", "93c4530f898331ef846d4eb7b1f66eef"
    ),
    # FGT_601F-v7.4.4.F-build2662-FORTINET
    "972dca7d7a8c089fcf76e0824099a78e3dc591b8b40931667855d544d5ffab04": AesKey(
        "6263bfcec1f066f0e09783aad2e484529c83973e0e71b60d4607e9358589a9d2", "70f035a8f8f19318c53c1b6762679c12"
    ),
    # FGT_60E-v7.4.4.F-build2662-FORTINET
    "9147733e3d9c6c180c2fdba4431e5f4df6cf8f0d4e4d024849af5066e7f92fc2": AesKey(
        "f8a1163ae4b65cb6445064b367853be070ce46c8e99e5f61a685ed0faa7c6c5e", "4632214e8613180230eecdae8983e57b"
    ),
    # FGT_60E_DSL-v7.4.4.F-build2662-FORTINET
    "3a7538bd9696a1998d47b348e7659db5e44d3ba3f6f184a0a13cc6ad9b0014d4": AesKey(
        "5b1bb00c0b5d3ae9e936d60e5a9eb4140117f4989c00ffe6e4a377df7b1ebea6", "ba45b01df5b277eeadfb946efee0dff7"
    ),
    # FGT_60E_DSLJ-v7.4.4.F-build2662-FORTINET
    "bde7c95065b76053e56dd0a6cd8c91ef63d0489e3160abdfe092e851306ea6fb": AesKey(
        "f6144fd862ae1c7e785d1b5592ef29f92e27fa322a5174b47d7e0e124bb73533", "af71f4c316dcddeafaa2c1ca035e1d53"
    ),
    # FGT_60E_POE-v7.4.4.F-build2662-FORTINET
    "d622be9beff90af24542a2cac082166be20c314596597c4c03a07b5d5b8b474e": AesKey(
        "6b23639e8c42870acc38341211e851c8f864ade680b39775c36597607306f21f", "93cba337078712d2652088b1aada490e"
    ),
    # FGT_60F-v7.4.4.F-build2662-FORTINET
    "ae7ac9a0227f1c479feb2cd6ce394a73df28f2ce4940a86a501afa94a99ddc78": AesKey(
        "d989beb4767f36e094aec14e7ed0589ac3c7ec9bdb39ae4e34513595c683b839", "dd99c37ecc4288ad03ac5584d2cbbe28"
    ),
    # FGT_61E-v7.4.4.F-build2662-FORTINET
    "dca4aa8b62386b28abc83edfd6586974301a817cee4b8d434717214a9d008f25": AesKey(
        "bfbc1890c4a03ccf7a8ddb050a9a9a345c203ea067c8086bdbc09baa40b58787", "50e02e8e26e472b16953b22fed39796c"
    ),
    # FGT_61F-v7.4.4.F-build2662-FORTINET
    "d25fa4d4a36e61cd0015beb36fa595ba5524cc13e58b5c2118924bfec8866984": AesKey(
        "c142960629d273899f94c093aad84275fac7f235245b44d59f191db8ea46032a", "070f5f0ceaa9c8ab3d856c215b5bb7ab"
    ),
    # FGT_7000E-v7.4.4.F-build2662-FORTINET
    "e57c9c80da210f5b03e3beb3c7e796cc5823c15e9a951c070ba5af409b58bf9f": AesKey(
        "fd3b00f243217c0fb1d16f3afb220ed71760087668536df3d2349f68b63ea61e", "713bf6270740023dc99016afded4756a"
    ),
    # FGT_7000F-v7.4.4.F-build2662-FORTINET
    "1190ea2a40402ac29995d61da62f7a527566eba8af82c5f0e7c50a379b32149b": AesKey(
        "2e2f7d2d7189d13cab22641b326515f10e6c7a2643b8c4ac932682dffc9a12d2", "b1213d2403925809b89277485f10b400"
    ),
    # FGT_70F-v7.4.4.F-build2662-FORTINET
    "ec98ed884a5edfb38e416e5bf566d0085aa6b0249d01e8304b15d0ac1cacfae2": AesKey(
        "a180d186aeba58056b8644f4f7709efa1a3abbb33fb4d40fd598244b56a01e10", "8fbfa73431fef41dbe0b00bf66b71cd0"
    ),
    # FGT_71F-v7.4.4.F-build2662-FORTINET
    "f5032b1ed01a5642c08d1dbdf7724b513fd74069260734df27e490b1c66f0995": AesKey(
        "e362068fee076e3e39bef07932e4be47d69ec23f635687020d987715d9ae9a65", "78001e085d6ce225877b78d83758d6b4"
    ),
    # FGT_800D-v7.4.4.F-build2662-FORTINET
    "679fbc6f8d43c8d75809357f71d2b6d6b1ad9179dd699f5dee772f8e2818a0cd": AesKey(
        "c85045fd5757426f6acc9f46f097f7908ecafa9b18f9df4ec858e75287f3f92c", "e6e3b928b69af16f6a7fae0077e1f524"
    ),
    # FGT_80E-v7.4.4.F-build2662-FORTINET
    "8ef27130ae523575037f48cb34cb86397e0ed3c822a33781cfcc124cd1653f5f": AesKey(
        "b27a6c3f98784f27fc15f340a1dcc36b7cf1618d5736c65be892c3acc02e91c8", "e8e01491db25073c9d88efec7d5d367c"
    ),
    # FGT_80E_POE-v7.4.4.F-build2662-FORTINET
    "44f4e26c3155c5bf3a741c19cb300a06773b947190f46440df6b6dbe62acb798": AesKey(
        "dbde79bb132da4cdae6b9ca26bac4303c3c5dc5b086d93686303eb34c8427eeb", "05425317c0a0bb0fb33f26593f6db4b4"
    ),
    # FGT_80F-v7.4.4.F-build2662-FORTINET
    "8b9eebf352038d791753771b79ee99b5ce7d8e8613d6679c18f449629d84199c": AesKey(
        "1ff697ed338c848fefe97bcb6f2c2b043ad4fad99f4bafac5fc9205fe0083b9b", "c6bf8394878decb778ca4fe4dcff2ab8"
    ),
    # FGT_80F_BP-v7.4.4.F-build2662-FORTINET
    "aecf5660995bf3429bd05fb209d3cce8fb398c81dd934c0a7797be28242e605c": AesKey(
        "a45553b25c2655df02f24a07a84cbdeb5f96e19a5f8e235df150f72819dea361", "0da11f6b89187d205fc79980e202390a"
    ),
    # FGT_80F_DSL-v7.4.4.F-build2662-FORTINET
    "405cadf053af318f20a64bd76276585496956b519b488395aa8dfd62b94812fc": AesKey(
        "222c92234e202baa533ce4c82ad05ca241d65240d3305bb021f9e8213fa0dbed", "662737fde37680cc9d56e2eb67ccc4dc"
    ),
    # FGT_80F_POE-v7.4.4.F-build2662-FORTINET
    "c60c73b93fc3a68a2cf1ec117c6653abbd4841488d4dd16f3657ac76a16ad281": AesKey(
        "95fcc7d3e2ec330ffdc709716e925f4dd0fd2583b650947a7c37c0eeeaceaabb", "bf88d7363b82bb6f2be37a4867d111b8"
    ),
    # FGT_81E-v7.4.4.F-build2662-FORTINET
    "0edd88e06ffeaa45c44d78dee9e6a36c51830558631ed86adce38a4887b725d1": AesKey(
        "80f631d708b2fd595262bf92b55732404cc7bc818293c76a134678470c367303", "201e001ff6991dd1a120c03b46687acd"
    ),
    # FGT_81E_POE-v7.4.4.F-build2662-FORTINET
    "a827b2cbc6c0c582e9db13c989e589bd662387425304a74537081bf0d86f1762": AesKey(
        "34122213b834543f5c2da706731a1e6881527259246150d36391d2c14207c8a5", "d3a617de9f7f16192fcc56f5b4214fbc"
    ),
    # FGT_81F-v7.4.4.F-build2662-FORTINET
    "521e9357a53f797bee01a6cf5b6dedc171d7b0c1609af3830e1525c7ebad21e4": AesKey(
        "7500f04e0d8435446e86973dc7a37ebce0f666c097a7f7b640902a1e5cc80e6e", "03d842dd3750803cbb3118851564b536"
    ),
    # FGT_81F_POE-v7.4.4.F-build2662-FORTINET
    "bf060e65543c60a3e8a61be231a52f5c02dafb49129fe45c74d2b2447c2ae218": AesKey(
        "088bfcc3e7a383ff23779baf654ea74dfa14a027a65784b57042f489c1732117", "dc17c2b2642f361e6cb945ff9b370a37"
    ),
    # FGT_900D-v7.4.4.F-build2662-FORTINET
    "f176c5859379c4df5e017ce524e58008f4e3587e69044f6ba9cd95858add847f": AesKey(
        "8b9e9260a2df30a5a30a2b7c770918a153f2325c6b549e0dcccde4f7dde6225a", "c6235c61dec6b461ae04adcfc66aa85a"
    ),
    # FGT_900G-v7.4.4.F-build2662-FORTINET
    "4cc3e421ef3545395b00d8f7d1bee88299a8efec8ce73b0798cbf4330928bb6d": AesKey(
        "63b3527a297c5c0963bb32193a16374c2b77a91da79763c8048656328d2eb0bc", "267b32ae6321a87de381a99c998a5c18"
    ),
    # FGT_901G-v7.4.4.F-build2662-FORTINET
    "32651e901116911afda316337a6ef9a3d6857f98bc8b4f21134174a697fdf4c4": AesKey(
        "b58ba7457b48e2b0c0cbe4128f3d442fcef43101cd37516f9f599e619c79908d", "823247917c311a3dfc3d0610866aa1f5"
    ),
    # FGT_90E-v7.4.4.F-build2662-FORTINET
    "ca5788a509a5913931d44ff78b9f1ac3c4d592a2644f30c586cc908caa79a34f": AesKey(
        "a636ae0850f3d1d61cd187980792b5dd83afc5b2775b34b3112e215c33f12318", "d2d0267bdb18c472c70aac9d5ad1db29"
    ),
    # FGT_91E-v7.4.4.F-build2662-FORTINET
    "86d3a159e393760529bc9a86f5748590fb2e0b0db8751ec045491fda11d8152e": AesKey(
        "d443c62ff24359ae95aaa0aba03e300ed97c34683dbd66a9856f68eb8c832381", "5d0b10c24c90043963be497d94168fa3"
    ),
    # FGT_ARM64_AWS-v7.4.4.F-build2662-FORTINET
    "b45649c59df50937a972f5c59bf843f859ffcc3c6a7eac5e2853a566f162999f": AesKey(
        "0cfa8856115beb83cc39fb7eacf1b8eab113c7696e8ab949a21e70f86d14eb43", "08607ad8229d87b0215469f2531526c7"
    ),
    # FGT_ARM64_AZURE-v7.4.4.F-build2662-FORTINET
    "5c1df3d974677f8fcc2e15fc63794c154cf49312fc79c0bc96f32de0e4fe8451": AesKey(
        "d924491067258f1ec04f24fe9039defadc5493dcdbfcba5bf23e3e97c6bd2d8d", "a4ce1c4e0754907bae4900393f9c72fa"
    ),
    # FGT_ARM64_GCP-v7.4.4.F-build2662-FORTINET
    "1d10f113941ffaad88551fb081802fa434fdcbca86918e06cee8e6c07278ec10": AesKey(
        "9e49aa8dbe7c90ca4291d11097d25fb8db5b9a24812b01d4caff73412571a1f2", "6893ca83f2c10532ea7035123af3b0bd"
    ),
    # FGT_ARM64_KVM-v7.4.4.F-build2662-FORTINET
    "e824fddfadde8ef92323b2b51fc5a7a8c448143582ba5920aa79c4059390c021": AesKey(
        "c2810571a62825efd2ea702bedd4ba2a7f1992e8c5f89c5cea854d8069ba396f", "42d7fa89ba6e88a5d1e84cdf9b6d4d80"
    ),
    # FGT_ARM64_OCI-v7.4.4.F-build2662-FORTINET
    "4a27a6346fde0cf20567dd7b1c59c65e5110b8f2a931d5f38fe790a7908da5f7": AesKey(
        "1caf087ded3c6c7ce54426bb885b9f4fce0ab963f4d66734e41b4f7e23368cac", "ed822f903d43324a02c2c6eb1e9adb8f"
    ),
    # FGT_VM64-v7.4.4.F-build2662-FORTINET
    "556b4a63bc8229dede603469c92954a393ca9c5ee439379ffd82dc74e636baae": AesKey(
        "b41f184642db8aa759fe35c5786824650714e2db5dee79fb2ca6bcbebb60ab54", "d8919daa9a056c438bc75eb9f2ca064a"
    ),
    # FGT_VM64_ALI-v7.4.4.F-build2662-FORTINET
    "fb3b07989eb1a2c26bda84b00aa88264109db18929864614a8e538c7962e088c": AesKey(
        "4ba32bcf11f4ba8e20515d0241eae5e5e0354735fce07f3f9913ff886f7cef8b", "52fa260ec27d9b7a71b61252bfead8f1"
    ),
    # FGT_VM64_AWS-v7.4.4.F-build2662-FORTINET
    "8b072eaa29db546eb352eba2eb54dbb7555589d17f7b4b8dd8aca2ce7b6c2348": AesKey(
        "bd826dc449b2fd0f7a23ba0a335e89dc23cb0b69bca7696ec877be9738b7c158", "016a21529f783b1e07d3d9d9ceffd16c"
    ),
    # FGT_VM64_AZURE-v7.4.4.F-build2662-FORTINET
    "6fe79bde422aca11212f6f282e07b3c74685cdf22cdb142a45799d535ddd26c8": AesKey(
        "b4452fe983e9388f20ac923fe5ea58b99c04ff38ca368374fcd928b4a57a28ab", "7eef007215094399c07047471c00d5b6"
    ),
    # FGT_VM64_GCP-v7.4.4.F-build2662-FORTINET
    "51e38999a50f37ad2ada3b5c06c4a29e3453b8ff7d0168921e7eddf488290ef5": AesKey(
        "40842e5e26bda0fda775471a4013dbdea6159b35367b3b41343b45295c533277", "f547cbad1233637d21b847bcee250605"
    ),
    # FGT_VM64_HV-v7.4.4.F-build2662-FORTINET
    "687cb022b2be46141c39e3b628ef075fb205f2570452bcac1cc9917d6d7c2290": AesKey(
        "ef3cb7ddaaacfc0e11b52e26960e06a6644c424aca3a8afd0116370dbadae719", "978c05db42381c09f3e9a3067d539c94"
    ),
    # FGT_VM64_IBM-v7.4.4.F-build2662-FORTINET
    "1578bf93534fbd688762002ec865ca186014ffc9291c2c2c24a977688747217e": AesKey(
        "7514241c3af2463e4f694e569c82bdca798d73f0d89aa2536db46410368c07f7", "9def381ba9dbc628f83cac8943981343"
    ),
    # FGT_VM64_KVM-v7.4.4.F-build2662-FORTINET
    "63ebbfdb54e0603aa2729ed9768329e1ff0577da9bd65373a5885e95f6547891": AesKey(
        "0cc2ad893498dad5540c153000fdd766324611796ea8fa702792d7a48bb97c32", "5d38d79faeeb0050288193e107d3e91e"
    ),
    # FGT_VM64_OPC-v7.4.4.F-build2662-FORTINET
    "597e33df3e0ae2363ccc58f5a0dc06157660d6fb73e0944ab9aa55e6b136f021": AesKey(
        "5069af12e4a57be93e7cd3e72ce373eb59e3f245d6280857f5239285bd3ebe7d", "8e1a2ed275019666d11f7510521f393c"
    ),
    # FGT_VM64_RAXONDEMAND-v7.4.4.F-build2662-FORTINET
    "a825fa1c180eaab773508da17879a2b70ef2ddbf754fc950f8da6627f948a171": AesKey(
        "edcccf5294478cdd2cc49c21214f2636ad4923a7f0438b7301e8d3dbc5a03ee2", "cbdc82a73a62e1d46a508eace6a38ca6"
    ),
    # FGT_VM64_XEN-v7.4.4.F-build2662-FORTINET
    "19f5101ed515cd4ae7a1ece391ba6692e47e9333bb422a934e2a3fc470cba145": AesKey(
        "a75f8abc7a4dd84b6c9d49adcfaafd70e94e388370aeba27d2b0a2aed99e321b", "4fc2fd3d71c3af8a22c7c86455d4855e"
    ),
    # FWF_40F-v7.4.4.F-build2662-FORTINET
    "cd1265eac312dcedb03f773868d2df3eef7898270c95cd9cda09f584fc8957e2": AesKey(
        "d7f6f9330d300e7a2014aa4c65b3950380bdbfd2d784522456ab999f584d300e", "e6745534e2d44b8706189315903ef309"
    ),
    # FWF_40F_3G4G-v7.4.4.F-build2662-FORTINET
    "6b0cabdb2920c722d255cd3882bd4c90388323535881e759eac60701edae767c": AesKey(
        "46bd09c79adfa909b9b64c954999649aa913b7f5f9ff31f4a531efc689def04b", "a2df5c0e7fdc4a9d3b266b08871005c6"
    ),
    # FWF_60E-v7.4.4.F-build2662-FORTINET
    "12fc2b3fd13c754eed5e1b1f6b09f6cbe658acbf8b23e153ac07491c8e14f55e": AesKey(
        "b9ad051421cdd349404dda055db30202ed8b9a752948a44a31cf25bd7865b5b0", "792b832acef32ec1635daf52efcc5cd4"
    ),
    # FWF_60E_DSL-v7.4.4.F-build2662-FORTINET
    "6ede1c8cd9c8d28bab6b45daf26575cc7d0f4b8005428b41e661820093f134ed": AesKey(
        "8fe1881913e802b3d7e33d43ee5d105f40bca1188ea6426a435db5add2250d59", "4e6c49461a1a5f2dcef2ac4b0819f5bd"
    ),
    # FWF_60E_DSLJ-v7.4.4.F-build2662-FORTINET
    "ab1b54e29c0805f57077bff5a12f01773a867640e6e4eb38643b4fafba165d20": AesKey(
        "7e920316b8b90e7481e4da089eeb634091e12ef44e05c771b28061dd881ebf74", "3524586a80ea48eb72304a60a0c35c8c"
    ),
    # FWF_60F-v7.4.4.F-build2662-FORTINET
    "ef4edf0f416359e2291b3577d59e1a4e5fe85b60701530f6bd29ccabe10f0991": AesKey(
        "88b1c2917609fce0c9279f0505b9e216bf5bd98779994a972002152dd6fa1639", "ced912753af4af67bbe281cc395670af"
    ),
    # FWF_61E-v7.4.4.F-build2662-FORTINET
    "ece0b8fea2439422b0bed2f133b7cfa4ec65bd3ce57639d6a4554767d9bbb91e": AesKey(
        "787826496efc38195c72bc71d5426d81dd65c73575d4114a860dfad03f9a823f", "d6825310b4b2b232594d9a8e59957b16"
    ),
    # FWF_61F-v7.4.4.F-build2662-FORTINET
    "f6df8be4a50bf969ddbe998318fe9e56ce4cea45764e361787f25a3f591ee9cb": AesKey(
        "3bf57aec3e08e0704d45ee11d3cedad3c3700146abfdb6a714f434e9cfc9b5a8", "b8725779ed737f4a30905494fa579f63"
    ),
    # FWF_80F_2R-v7.4.4.F-build2662-FORTINET
    "36183ec303ebc614f8ba7f2c3fee186b1f88687a7337108efb898607636b0211": AesKey(
        "03a14b8dee97acb723423f3c04aacafb81920c193220d9fc5b0abf275f6f9bce", "b736d66a0bc4a4a8f810e6456a41f055"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.4.4.F-build2662-FORTINET
    "bdd5da74801142fabf2b0552b05865b60a184fc073b65b3ee1ca157535663942": AesKey(
        "9acc52e91b544f82ff1c8e518166fc4d13150e39ee713ea0f566b8048d3872b8", "6145ff6860a4cd54850f9ac7be177d9a"
    ),
    # FWF_81F_2R-v7.4.4.F-build2662-FORTINET
    "3ff28fa8b6ed54095b41cd127e3c20d1b83cac51f9eb5efd718a2747fcfbba37": AesKey(
        "9202189105b808bf32cf8b216b367564f8500534e2fce7b3743372cc10f87f82", "38a92b5586a2bb01aeb8138fed26c65d"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.4.4.F-build2662-FORTINET
    "e920d6fbadef9725c76de3c073d9cfef80da563c42a904b1f23bb36fa2ce4a03": AesKey(
        "c0841dca94306c108e97bd02a8b8c3f297726cc76498dc842558bfb0d3436040", "0dabe37c5624bc46ff9f7df59ea262bf"
    ),
    # FWF_81F_2R_3G4G_POE-v7.4.4.F-build2662-FORTINET
    "a01989b0ace6dba9cf76d9eec4d2991a1d183d7d24e1128ba149551afe2a26c5": AesKey(
        "5e4450a3309707b023d6e4673507f6799fe27e0e526e31b3daac9b27e99f76e8", "5d7855764d77232c28f0ac54ef4dc3a8"
    ),
    # FWF_81F_2R_POE-v7.4.4.F-build2662-FORTINET
    "4f68e82573ad675866a7d57c89f03b764206ce881ff57c57cfe6b036d538409b": AesKey(
        "853e831d4c96ed6f3b9ebd6e8e800ef189ce27a97e5eaa076368027d1643c88a", "018f4fbe1e9f127388921c92edc91eb9"
    ),
    # FFW_1801F-v7.4.5.M-build2702-FORTINET
    "c0165ac697454cb82123b9786adc05c3de71b569950fb2e174bf1b48c38302c4": AesKey(
        "7df35ca0ebdc440b3857944812113d0231fa79e60d5a7f4d9a457e7b987c984e", "80451293796629e05254220c460e8f79"
    ),
    # FFW_2600F-v7.4.5.M-build2702-FORTINET
    "60c12813eef7cf432fad03a17e24765764bd7b11416b3bb125d8bb334095bff8": AesKey(
        "0e28156eec5308bca71590c7166314b017b141929f9d0f8f5442c3a884baf764", "6e27bdb179460868a8f8d067cc71dd7a"
    ),
    # FFW_3001F-v7.4.5.M-build2702-FORTINET
    "4a58f75929ac4c50b20021e72d59f3cad65d6bd13173ce907d932fe8256fb2c5": AesKey(
        "fa07e18a039cd3790cf30273d49f32683af0e3fb629323f917917de5ebe43ce1", "874777b7024de0242defe2b03c68fb95"
    ),
    # FFW_3501F-v7.4.5.M-build2702-FORTINET
    "712fde39a48830eb5da0dc763ad43e1a0225c3c50bf4d65c8963793af8cbe8a4": AesKey(
        "b80830df6bf9dc9fc2dcc3e998d574f376b0a77a23c4a7a5eaddffee64847a68", "b8ce0699b7a228167ec11fd135578c2e"
    ),
    # FFW_3980E-v7.4.5.M-build2702-FORTINET
    "03c85cb2236f86ded7eb8e4cfb0046fbe6f7f5238b12a75d2fb9625f788f80ed": AesKey(
        "a64c2879e3201e06b0b4729765b7b2b5602affd555b2c75b5f0a78d8ee7a25b1", "7fa6138ce6ca419785bc948e13dbf5a7"
    ),
    # FFW_4200F-v7.4.5.M-build2702-FORTINET
    "936dbb18374f5f35bf419085d5bd754358a44fc3d3ce66ae351171f79be1d016": AesKey(
        "b3f0b821db1455cbfb86738b9ffa386370f2b6ee0e6953557a5dd8a98511d381", "22025bf9dd0226b11dbb403d05ba4f54"
    ),
    # FFW_4400F-v7.4.5.M-build2702-FORTINET
    "fd0b8391e8176dee364a897059adb467294076911f4fa17568532440dfd1ff6d": AesKey(
        "01b7422f278dd55257e95b585e090e56ece99489a74a2392af2685e2eb7f3f68", "c2a42080f762fc4f2b7a5571c13680b4"
    ),
    # FFW_4401F-v7.4.5.M-build2702-FORTINET
    "ec98f6c6db60178ccae0b35049596fe2abd1cc3141e12790bb029bb6b1f24f35": AesKey(
        "38841fa72644ea0e4870ce659c679980cd0dd4572643910a40a3dde577340d1b", "3db37bac54843799604eb0978ad40cb1"
    ),
    # FFW_4801F-v7.4.5.M-build2702-FORTINET
    "141d56da2fb94be7341477359e991f283126e529e14dd2a42a8bfb33ec8c6ce3": AesKey(
        "6e8ec35074b12e3716334a092a5791bf9ff5dd17a4a7049e4fb9aa08c350af78", "6a1db16947b190f12045d35a4e1d14cc"
    ),
    # FFW_VM64-v7.4.5.M-build2702-FORTINET
    "2bfb1dff3c57c2f349008e44eb97b8bdaea61a10817c1b1df1e216e8a17a28a8": AesKey(
        "df4499e36e059b454289f059012577d12caa58fa80db80552e0e1f55265a9373", "3b025f00d83f033d8930694720aa095f"
    ),
    # FFW_VM64_KVM-v7.4.5.M-build2702-FORTINET
    "e1e131f4f6dfdf9ce2d9abd378d67a41ab9c00a2e8afe4f062e7ccf388652578": AesKey(
        "5f5a764986dce6325c2471c9a65358979c741609f16240df0d9a448a21c3c4af", "a802942a9f96f21c1f3d9169a2c93ab7"
    ),
    # FGR_60F-v7.4.5.M-build2702-FORTINET
    "ec5acd7622b13c64358a68d1356aec193cd79cc7b117e8f171a5f15305580caa": AesKey(
        "37e4b1ebd48c4893d1bf27c02c2b83da141f728277a5cd50a00073fda77fd565", "2a0ad49be7eb86fe62e154b28e3ee989"
    ),
    # FGR_60F_3G4G-v7.4.5.M-build2702-FORTINET
    "1d3904b86f988a9a34195712bb061d17e3aef2ac75602b6d95af4e8460414071": AesKey(
        "214f4a705aa723eab7a05afd62903db20a17edae89148c9f40a25853d390b7bc", "337b88b41b11fff0f6b96d7b9df553bc"
    ),
    # FGR_70F-v7.4.5.M-build2702-FORTINET
    "f3b08dae773e62d5f0cf15a4e8ff3f2cc1cac4a337c009e0e7b0471121c2a027": AesKey(
        "35a1b96c95f37522685b1abf9165258154e6f86418368f35e07037986611b1e9", "869796a3810bfe4b97db6f8408990c73"
    ),
    # FGR_70F_3G4G-v7.4.5.M-build2702-FORTINET
    "71f20fceda8d9c6113c5952e5f8e4c3dd32d216ed877ec1beabc4fdc12aea47f": AesKey(
        "4b812cacad7d01fd57ef60cd96e8d70d6c1d56d6d1e037f319d4ed9ad501a685", "f425f8ba1df5a8342ba31f51ef639407"
    ),
    # FGT_1000D-v7.4.5.M-build2702-FORTINET
    "201587ef9ade727a61daf5cff3f009be9ad4db199f28a0bc1b36a38784c5c35f": AesKey(
        "967718c8fd0a506696d310c008756221b031a07c44ca396529792dc706cc182d", "caec463b72dd3c7c02b73bb644aeadcf"
    ),
    # FGT_1000F-v7.4.5.M-build2702-FORTINET
    "8db623440b633fae6447f92f6c252963c86b80757617378f9bb3e3882e2f240f": AesKey(
        "930c4f0c2d193dfdef9f5e8e67d83bea7e671d75a4486dec88588ab873bad6d6", "ce9332d0e59fa43346a95e24f4a2f959"
    ),
    # FGT_1001F-v7.4.5.M-build2702-FORTINET
    "29fecfada0326fde8ea4bdc25963e0605e964d800b9f1e87d6c36bac3e14d513": AesKey(
        "f3ce1e2e39107b15444fd28ab5b3321f7c60781dc8ffbe294550478d72bb4531", "ee89244ce09765925baa4dfa0f4ee308"
    ),
    # FGT_100F-v7.4.5.M-build2702-FORTINET
    "89d44a9ac0fd3d965cf54510e618c4a1b7add18087b5d269fdf704f29af821d3": AesKey(
        "86f9d02fd3f1b0f22887d79ead7d0a5000c005321114ac7163931f2b9626b327", "39ad82b508a83c9696afb07e0f81c0f8"
    ),
    # FGT_101F-v7.4.5.M-build2702-FORTINET
    "0407d393022a658240fad6d40fd8ae6327b8e72529e7d38b4b79b40e3230010f": AesKey(
        "3d2d8a9bbf3d403045bd0f8ae063da1abe74843bb3dd2f515b3e7304916979e9", "9481e864a2cc3f7c8083d91c77d9b908"
    ),
    # FGT_1100E-v7.4.5.M-build2702-FORTINET
    "8b8260fa5c6affc0c218a164b98e518f622cc5010bb348eaaeebdb14f2ef382d": AesKey(
        "efe77f00fa0dd5bf8d6c6bab16a21e39e357f826f1288ca62fb11c67dce59d69", "bdba8c6d220011adff0828792a166fd8"
    ),
    # FGT_1101E-v7.4.5.M-build2702-FORTINET
    "5509bbf02c0dd67fcf9c60fac8b7baa67617a8a343c32627cdee45af1f487909": AesKey(
        "c984eff6129cab9fdd80f86cb99bd392bd116a5440c42c6382d30b6f304f6460", "ea24bce156ca5d97ad157e5c1797a37a"
    ),
    # FGT_120G-v7.4.5.M-build2702-FORTINET
    "70091d536fd8f1315c6faedabfddadbdeaf953885c582cfe88835d08109dd2a4": AesKey(
        "557af07f9b0c59bab102f44a6b89f0eef3837593108093ef7c0de4d5e3ef0d7d", "44dff89dba4c79025ed1134c39996b79"
    ),
    # FGT_121G-v7.4.5.M-build2702-FORTINET
    "9cea484b85b5f873f95f1e0b26770fdf1afc64e080655942145835de0dc5d8df": AesKey(
        "c8a1137d88d9a521d9e6be479d36d48647752503600a2f1fbc4eb039eccc84d1", "ec4bca380e065975194d5d77d6185e8d"
    ),
    # FGT_140E-v7.4.5.M-build2702-FORTINET
    "55c254bf3c5380b5708ac9e55d7141ba51cbc259e693ce38cf444383c422bf90": AesKey(
        "b0d6c6ea0211961ac08e0b8f0618eb775547bfaddc355b80c8fecc0c37ee46f5", "86dc9c62821121266f47ffb58f81da64"
    ),
    # FGT_140E_POE-v7.4.5.M-build2702-FORTINET
    "88cf20afbb6157a66619b8d44e6f2d065214a0fea94701bafaa8d636447048ed": AesKey(
        "dcb41f251ab62a2942409745e37d4804f029f22a7f8f7a86991fbabcd488f37f", "65075c18d3eed2522dec8ee954d41ce9"
    ),
    # FGT_1800F-v7.4.5.M-build2702-FORTINET
    "da2506bb4e8defe3d69c6aae7a1e57e76f8f2a3795c09977c173bd957a3c23a5": AesKey(
        "58edac9a745dc008ac14e627995bb43b1be74f5d8a3baf2fa36a27d615c771f1", "2d9928bdb80d9facb7441dfe9e604166"
    ),
    # FGT_1801F-v7.4.5.M-build2702-FORTINET
    "5fb066dba2bff13b03bb0fbb67ed835b943e8b974490155b84fe6d79e33298aa": AesKey(
        "4a5b2509fec0d36dc300178469259bb6feee85b3a2efb0853af7d7a4aa7beb6e", "62dff9a31b817c00e1541bf13cee065f"
    ),
    # FGT_2000E-v7.4.5.M-build2702-FORTINET
    "e38a113d37310f85b110a5d646e8850f383412b83aaea1e150e11c61c6391441": AesKey(
        "e87b107ba1dcf01a25ec4df12f61dfd30bedd9d068005511734b56af06cbefd4", "2c80dfc4d0bef0d01ad7af802d76fa85"
    ),
    # FGT_200E-v7.4.5.M-build2702-FORTINET
    "6433615a3c3a83af9586b8eec4386ed74e41a38ec58177e5cb67fbfac8a4e3fa": AesKey(
        "c62910fe30ea0b3774aead99b77d34c890e627fed77ec61a0c051989a2eb54fd", "40ae13c6d1ef32f8bf6e703c219b69ab"
    ),
    # FGT_200F-v7.4.5.M-build2702-FORTINET
    "965ae9dfbcdcb9360bb1d7e6521ce7a41588df091ee9cf27adedf85992b4e8aa": AesKey(
        "d6fbbceffd61ee6020636f2be1a57728f90f8e1da67daff11cf2230cec5030e5", "830d5ff8f03ce4777319ae07a4f5fd6c"
    ),
    # FGT_201E-v7.4.5.M-build2702-FORTINET
    "5383870aa8518cdb29c06cac125de8b064aed8e1b0fe99e773a5bbcb0012965c": AesKey(
        "95e101e175623da762623991ea90d1823c7cef391a78ad31e00283b5cc59b16c", "9e07f401c1659e894bf8ca57ede73f7b"
    ),
    # FGT_201F-v7.4.5.M-build2702-FORTINET
    "d2e21f1632df2beb83f23fe50448642cb335c12c59f72f6ab966b94c53a25e0e": AesKey(
        "c9d9bd8ac22bc4dcc6affaf173fa320eaef4e1d5f1cd1f2d8234b07104ed8273", "c5f1f2962cc0dbf28047e17bbf937bf2"
    ),
    # FGT_2200E-v7.4.5.M-build2702-FORTINET
    "715d3578156e75870f7197d97ddd5c8a3f2890475327bab4497c4d6080240d9c": AesKey(
        "0aab6ebdb5f80b3c0a624970142e4d9e65ca0d352e870bf7d33a9fc97fe1d924", "21b5964b2741420da105d2d40e06092d"
    ),
    # FGT_2201E-v7.4.5.M-build2702-FORTINET
    "90045fc4b1bda02585db7bdce24e8bd5006692762ed7c6f61e4c1d846b1b06d1": AesKey(
        "c986d2d211b730fe7bac64a8041da2a2e7e14ebe52e8c32f9bc2b8546cac7953", "02ed202d6cbcfc19c85ac2026a281af3"
    ),
    # FGT_2500E-v7.4.5.M-build2702-FORTINET
    "d955662feb430ba611eb9b22b4a0d5beabb7badad503322d58a95fad6fa96ccd": AesKey(
        "ae9f7b7adc99e34817525ebf6282df5545d1f4907f01a5ed3c74a4e8f7af05d7", "da4402e644a5afcfabec6b584b24167e"
    ),
    # FGT_2600F-v7.4.5.M-build2702-FORTINET
    "ff33962599cc4d3242882d8b53becb6b9ff47424921d17debaed6185aecb7484": AesKey(
        "d45eb6e10325b62a165369d33ddc7831747b22601ed957b328495b9449235e01", "75e203ba4041352628dc92dacb3763f6"
    ),
    # FGT_2601F-v7.4.5.M-build2702-FORTINET
    "298933f0e5954a29e98707c9827ec71f5ea036a30e5f758eb7506b8a1b889741": AesKey(
        "3326876428bf5632046bc8b07c87bda74161e629c226481eee458c7f8f762592", "231310544f3477fc06e6238c90d07b44"
    ),
    # FGT_3000D-v7.4.5.M-build2702-FORTINET
    "9863ba786b28ce290c309d431b7f0d85eded33c744ff5dd7626053806f82d374": AesKey(
        "59ef40a6459566c3b2ed793431b4165de52f331b7c617f06f7114b2cb0646816", "4003f9aa501d3454eead32a0a93c9f47"
    ),
    # FGT_3000F-v7.4.5.M-build2702-FORTINET
    "d8c7a638e053727878a32673ca2e39faa39c12b377a491b5ce8144825d500835": AesKey(
        "ce004f9695a757369f6287ef1f45a00eb1447046cad33ed4b1d93fc24285cbea", "4dc8cdfb02b1d5940dbccb6a25213cba"
    ),
    # FGT_3001F-v7.4.5.M-build2702-FORTINET
    "cc04fb8b2e88f13f3d0ebed2703b0c1506a2cfedc7f2dfc0cfa75cad4e93b0ce": AesKey(
        "7ecee58941a1f40ffadbb8dc082b2df27babfb995eefa20350a77ed126c882fe", "1bc65131cf6b51057202b0d7452711ae"
    ),
    # FGT_300E-v7.4.5.M-build2702-FORTINET
    "e9da4cde53798508a635b5aa77e77bcdccb89da2ca14fbd8b29511f1ace8cb2b": AesKey(
        "06eb684a0f9d43b19e1a4cc1f71b6915bf02b950cc1f3c5a6ad8266286144ae0", "5b86671438d028303d1a490891d1a46b"
    ),
    # FGT_301E-v7.4.5.M-build2702-FORTINET
    "f65980a4acd68c50a77bfb5feee98400df828b96e3a1d334d822be093b5804ef": AesKey(
        "9fd284f933d39307764ef90ab05c785366ba79e93819067b1dde0e77d55474c4", "bd036b32573870a31b7b63a1fe794eb1"
    ),
    # FGT_3100D-v7.4.5.M-build2702-FORTINET
    "212743cafb93ce53f7578df9eeb48018972583a127fe545b330d9bc6cf045651": AesKey(
        "74f5149c1e4faf70dea266314f8a3bfbf62cd58fed93cb25ad8db77ed47098a6", "82de6b3f305f7c0c38640db1b5d320b5"
    ),
    # FGT_3200D-v7.4.5.M-build2702-FORTINET
    "71ca9a2eaa147daff4946eaf187002e6ecd7b134f9b30a4fcb3b02b945048495": AesKey(
        "9c54ea41be2b3c501324f09480ae905e6da26de24bdd4cf11043773bd006b942", "45df1f1321ed26189691fada380d7b46"
    ),
    # FGT_3200F-v7.4.5.M-build2702-FORTINET
    "17b064afd128c39334ba3a4a9b6719985ff7afc7480412d02b4f7a48fe24b39e": AesKey(
        "a36e026b6741027905439309620a6f32983e47de605285323b7444ac1618e720", "ec46702b5aabe22c39ba5ab6dabe6330"
    ),
    # FGT_3201F-v7.4.5.M-build2702-FORTINET
    "040bc8aa805da487eda1c5766a5f396e7ccd5316475b8d89f8ff44f0ddb87c43": AesKey(
        "4cbdc2360d3f4d6314fc7b70882beec1ee6ab443552b9afa64716890f9c7d715", "c0e4538546aab57aa85c7ef4cc154fc5"
    ),
    # FGT_3300E-v7.4.5.M-build2702-FORTINET
    "031453c8ca9c1feb09a29ede488ec36a314befee68a6ce0acedaf241d302abc0": AesKey(
        "a7d98f4b6f59989d4e5a3fcd81803728b53ac39002728dcc5d727a8799146606", "eaace927ecdbf412790f254fa6163619"
    ),
    # FGT_3301E-v7.4.5.M-build2702-FORTINET
    "37563c80716cc18ff202bbb34e497991c3d7ed352286de5b3133b039ef16b839": AesKey(
        "afcb6220b0721abc53d6a99188ac571496f915a6cbe255891d0edda4f0e2ad8d", "06369f3c9f4fce4d305a7680e09086a4"
    ),
    # FGT_3400E-v7.4.5.M-build2702-FORTINET
    "08525d93c23bd83f400fee2b326480b0d2b6b541534fd51548a01660bd66a558": AesKey(
        "604f1b3eb0658b4cb94f142215e9b4835f113e3abef6188194bdb9fc2909865e", "bd553220cf386a64f7f2980a027da97a"
    ),
    # FGT_3401E-v7.4.5.M-build2702-FORTINET
    "7fde328f46ee5e316fbe3e78df242475e15764ddf3e9e0ea46af91c7a68199b2": AesKey(
        "eeca182b542b74d3cd555cb80541a6e92bd3cb4be2efa80eafefeec78eee76fd", "3082db5a6441fc9fef26f535b1ee7ea6"
    ),
    # FGT_3500F-v7.4.5.M-build2702-FORTINET
    "1a34f0dcf3aea8885f15262d699fba723b1cccb9569c855c8da33cb4368f247c": AesKey(
        "7d1f6084c86a6139afd5d2a5aef93e0a4e4752978ae68a7482033d4071e07f50", "2fdf3870c1a06235bc9a8d23bc9c723b"
    ),
    # FGT_3501F-v7.4.5.M-build2702-FORTINET
    "ebc0f3b25db3192682eade212bcff48ecbc649621e9475171e71919b186b3e71": AesKey(
        "5d39e3ef073bf044e30210844de0590f899f875016156012146bd8d1a8a42e4e", "f1c34f439334a85ace1dfd3161becfec"
    ),
    # FGT_3600E-v7.4.5.M-build2702-FORTINET
    "49e0247f9d846fb0a5c5e8236a0eead7ffc26a42f9b3859b8d3e18a939ba6a06": AesKey(
        "16b43c95ef81c238601b8cb5342a811c322e8f0405d5c620b119653bd4d94189", "ddf017cb4fec925aed68b9f348236439"
    ),
    # FGT_3601E-v7.4.5.M-build2702-FORTINET
    "c4d80302d0718bd0f7d4d01cd070013fe19383c6d66eea1557788cefbe364bf9": AesKey(
        "f8d90844c2fb04fa80293c02a1c9b73046faa722f5d9ae1ea91d8c44ad8bc0a0", "717d7a141c707f3e3d7c18d767b4388d"
    ),
    # FGT_3700D-v7.4.5.M-build2702-FORTINET
    "4367779980417184db8a99aba6da3ef7a281feac5d5063ee156439b357c4ee7a": AesKey(
        "f4ad6c026540801beade95a0ecbcb562d319f5dfde838c2b5976e8837effc042", "454b28ab57ee8ae49d7483c8e7d554ce"
    ),
    # FGT_3700F-v7.4.5.M-build2702-FORTINET
    "c1273e93c0fc587412ade0577e3152a59f55ca551479ab61eb13784ce90a3a28": AesKey(
        "21ec57c48fb044584c3d886ce216f016103f43f6d9e57b7804581bc3ad513fc0", "57472d3234da0df3362210e5ae360b29"
    ),
    # FGT_3701F-v7.4.5.M-build2702-FORTINET
    "ceb1cfe1793e6cd93dd4cc40ee53d5525a782fc5892768427f3054006294bbca": AesKey(
        "4536c3866971189cf36fec2688cb7779a417ed6ee0fa0d4ecdbbba052187805c", "d21abf8fac688cf1743091d6db9cf815"
    ),
    # FGT_3960E-v7.4.5.M-build2702-FORTINET
    "b242b5c6e7b958388f2bc141221225dc15f197e69353a34345e5c7cedd05669e": AesKey(
        "2526d6b82023188e2fb54bb96033d621787e7ce7c4eb2662e106212082ea5c4c", "995f38f6f63284bd478db7a446a80639"
    ),
    # FGT_3980E-v7.4.5.M-build2702-FORTINET
    "a9387eebfc7b754b2a2237c33db1983d6f70843499e302a6daf1292691f713fc": AesKey(
        "a0a0cb13764fa09e8df4ff9735efc2c2cd92f0ee2259fc9bcdde9b174c47adfd", "40aed6f1d958cf21818801866c471d82"
    ),
    # FGT_400E-v7.4.5.M-build2702-FORTINET
    "4650418fa824ea22d666b158db7af413677e042166c2e3cec7bd896ef7e90871": AesKey(
        "c162aaf6aa5a03a91425ae8ca35f5a155491f2ecd6ad811d876aa1dff6c30a55", "44552b347f60af2c24b42e1fecba1f63"
    ),
    # FGT_400E_BP-v7.4.5.M-build2702-FORTINET
    "ba2b72d37f8c200d6ba9d75cade6d64ecf0bcac700d46264ca1870a1d24fab10": AesKey(
        "77a988be19c23a61444d89e55b707e3b2f144efc00f45a0a79e549fd18b33a0c", "2967314d92eea18a679eb1ca3a858d65"
    ),
    # FGT_400F-v7.4.5.M-build2702-FORTINET
    "5b32809ffddf4ca59cb5c1a22170980eabc892976e66f00e04f6765265470e8f": AesKey(
        "f46b931850570f3108916bbf943c63bd656acfd8976d12cadbb4699dd200e2f9", "e4264225e7f7cd7bd5a6bd2ac8d2ecaa"
    ),
    # FGT_401E-v7.4.5.M-build2702-FORTINET
    "51c8232c3ff088b1ffc88824d4e98f2636e7ce81a5bd470a8f9234ae4a84d45b": AesKey(
        "c07c28503fbe7d5ccbd371bb8fe789e2023b456bcab2b221b876ac8eca9aa12c", "46eb63232da7cb6e5ffe745e51dff801"
    ),
    # FGT_401F-v7.4.5.M-build2702-FORTINET
    "e5461fe4853067a2f01b165c208c348fad0c207f20862651088420b30a25a9e2": AesKey(
        "821020dfbf37a4e2fd21fb00e0e9f179b53c92d5fd7f5bb80d4e26c2c93d1528", "b48902d8ebaa0228393fd9a949d24d1d"
    ),
    # FGT_40F-v7.4.5.M-build2702-FORTINET
    "7dfd7c045bb19269a103bc95c1b7365554d8d4889a4983d99bec9676e90b5c80": AesKey(
        "68771d06cb9a88d647b4530e7da319049b29d858c428db58fd23f454ff79fa9e", "ffdbcc63161c2caa51828d435ed37392"
    ),
    # FGT_40F_3G4G-v7.4.5.M-build2702-FORTINET
    "0558e2fd4ecac56859c09532592c9cc08aa899fc3a1f30f3ce9052448c9ea5f1": AesKey(
        "c72a88f01c2fc138d5103eb04ac4e554b9636b67705fae597bb27394b6f91463", "dcc06200063fc51a591470b4be24a214"
    ),
    # FGT_4200F-v7.4.5.M-build2702-FORTINET
    "ef3ec0f956bde7b615a7a82a044f23351d87d30b8186dc0801650358f3313cc9": AesKey(
        "9302e49337e957c9e95a79a19a715899e35b98ad19e60a2c08d1ff4a0dd054dc", "56ad601818de2a4d7615ce9c0dc2e776"
    ),
    # FGT_4201F-v7.4.5.M-build2702-FORTINET
    "49fa82cc22262ab83daafdb3962530d02123ddf428a04e7a3c61230b115cfee1": AesKey(
        "2ca3133263445994564f12300f941cf461b006ca9feb27e158db50fa3529aec1", "89dc68e16cf29ec8aadbab1499062696"
    ),
    # FGT_4400F-v7.4.5.M-build2702-FORTINET
    "fc128c6cba72d1cecb27368564bf788183c807f5a327cdedbd28352d130df5e0": AesKey(
        "1b1c72b25b406b840c2d47e8e229f69abb0d5dad3636727f2d0ac40ae6abca05", "7090b66fc172034933514fc87ff9f5e0"
    ),
    # FGT_4401F-v7.4.5.M-build2702-FORTINET
    "411b08b16128e6ba2228721fe1d90de09aaebd721539c0190e5a7a521229aa40": AesKey(
        "2edb1834b2f91ec06a093edc45ee3c513c2e0ec16411d45821d107aa5e87a543", "3d7e40a6a8a641ed901e0d5737d3b2a7"
    ),
    # FGT_4800F-v7.4.5.M-build2702-FORTINET
    "9b01083999b3a876c8952ba69fb4494c42b99f7f3ad48653cff81df1787f7a57": AesKey(
        "b316c8ed6f1e68d418502323a8650a19a11c1d2bea4ee3076344a614fcae90e8", "2950ea68682e6792ee6a6975b352032a"
    ),
    # FGT_4801F-v7.4.5.M-build2702-FORTINET
    "ca26571b22be4693225eb3b31b66fdba11c6a73d896118d08012dc5b9a8e1232": AesKey(
        "e8ab0f07eee1a212879723ff5bb99c7b04572e22770a56a6257cba9ecada5c5f", "d10849bc0819c4cc5db46e7a041edbff"
    ),
    # FGT_5001E-v7.4.5.M-build2702-FORTINET
    "6b6b1abfa23772a624a69b2d4a8141a00f47313c21cea6f14bfddfba6e7251ac": AesKey(
        "18dcc4f5e8a74bc1d0cd96e2420041e3b20c7d80c20457400584034e6ef9f444", "f9a4bc6c1c31939255547c68eafd6ec0"
    ),
    # FGT_5001E1-v7.4.5.M-build2702-FORTINET
    "397602a9d55f33ecc1f8373100a26991de964ea537296c54f4786a8312bf9bcd": AesKey(
        "0dc4f6587fb13cfe3a1c3ab792457a5db0d807ed8cf8dda5cc49d9154e2fa074", "5945f7c6f335fbd29ad2eefa0e91a7ac"
    ),
    # FGT_500E-v7.4.5.M-build2702-FORTINET
    "6d8ead42700fef9f6d5a001f6fbe445b0ed330be6b619b034147bc2cc0332290": AesKey(
        "190de3d4dcb5cca944922ad79c605fe2388cd30ac205c3562301d123893cfb90", "c76c297d4c15fc13d4b7002caaa7bb73"
    ),
    # FGT_501E-v7.4.5.M-build2702-FORTINET
    "44a44d963337aa32a86deae56fbac91f019bef92c357078fa6c1660ee1729890": AesKey(
        "4a6f818ff3bd68269405f8bff91c4d503e9d4a6f4c3822b0664848a2b7fbee5c", "4231a92b127edba55990d6eca021a758"
    ),
    # FGT_6000F-v7.4.5.M-build2702-FORTINET
    "9ec76db1d638ef99184a2f4bd29ab405f48e3b7a00be6676e20ebf2b657d74a6": AesKey(
        "a07cf60918032b3f89cadf51a684eabfe69cb3d20f5ec4881c0fcca09fc953dc", "c6db441a30667f7f8d781ad5df9df9af"
    ),
    # FGT_600E-v7.4.5.M-build2702-FORTINET
    "3808fbe37c38c0037610d1a8ff5a13f03edd7dbf96e113fae032fa6083923fd8": AesKey(
        "e61fc353a4b06c28488d5babd07e21f77c4ab063002cc8533866177dc86502e8", "9d63f26922d518ee66b2781ab873be1a"
    ),
    # FGT_600F-v7.4.5.M-build2702-FORTINET
    "8ea03db5fb06fc1eb27992951dfd0c0c2f7985d4943f44e652189d88701c3fd5": AesKey(
        "ee139dc58ad1f3fe6d1ee0cf235720cd5dfe0e617ce6551a13b8bdeea25302af", "6cf64f9d3a201869293e3aff0f3d64ed"
    ),
    # FGT_601E-v7.4.5.M-build2702-FORTINET
    "003ba0ce899893178b2963c3c5eb98ea93ce8e4e37ebbd67636ee8a9d5aa45ec": AesKey(
        "d018f049aadcd80657c9d41998b9914d9be8d9b6ee8ab1175d39fb47db2ad828", "7e69887171b7cd4e27e12af78d139be0"
    ),
    # FGT_601F-v7.4.5.M-build2702-FORTINET
    "edd4b381c16b97c68ba755bc20c16071ba3363692f5f25e214367eaaef00b8b2": AesKey(
        "a2a3caa49ccedd968829ad0b8649b5c364bf499610dbf8cfefc1d118517c1119", "82d3dba0266bcb1546001228e7486627"
    ),
    # FGT_60E-v7.4.5.M-build2702-FORTINET
    "4b46be75f42f3cc1b83c784699fee27d03a9d3e0200b87467ef128a7e2e8a6e3": AesKey(
        "02435c1ceb79d5769a672160750a92596fb98eaa5335c13893edd259dad6dea8", "3098231ed8cad51c0466efc2d3070afc"
    ),
    # FGT_60E_DSL-v7.4.5.M-build2702-FORTINET
    "e1f25b31bf5f6e92a01ec81f1643d9609ab3b291f93521d5aaf7b4b2d10e313a": AesKey(
        "b0ff4d09d6f20d07da0d768759ed77c9f38810a0efec2d300eb8fe1f1c99233d", "34153d01cda406b8f34796d01052f320"
    ),
    # FGT_60E_DSLJ-v7.4.5.M-build2702-FORTINET
    "239e37afd8be6cd1ea1d9836c8215a6ba37cd78c1ddbadeb9da1274d2f2240ce": AesKey(
        "c1def618c4695c17a3c6cd5024debc9b116f8b4c91a5d60584654b701b1a81ac", "20811063b02f867dfbef14ace69f12a0"
    ),
    # FGT_60E_POE-v7.4.5.M-build2702-FORTINET
    "2690ef919880548747e5f8880272410697a927887355bed4beb3f73b4b39675a": AesKey(
        "2d9da2f9a5f5f01b76517ef007d59fbbfb0ff66e5ef21e379f0027aac7320516", "9506d412008a9cf66d8a410d5c8b3928"
    ),
    # FGT_60F-v7.4.5.M-build2702-FORTINET
    "ab2de38c5e75c9f301fde51d9cd5e16115ba72c60e5d6a9222c0cd12834e1e9d": AesKey(
        "d5ef1ffdff5bcca38437264bd6bcda4792ce31f1d1c3bf65f6643ca0ae3eb3ff", "ff5c048b1e722956689b3d551b70df5e"
    ),
    # FGT_61E-v7.4.5.M-build2702-FORTINET
    "1fc2deb51dd7ed0d0d05e78aafa7cedf2bff1760c200751705b9c5f55abfd8d1": AesKey(
        "7820dba32847497128d4d4a0c8f394b825f4bf11bcc6c54302edee547be4b91a", "9ac970672c01bfe216b8f3a1f4fe18e4"
    ),
    # FGT_61F-v7.4.5.M-build2702-FORTINET
    "37060049e34326b0cfd2fbeb2fc893f049192d61f8a7d979efadde3aba053e10": AesKey(
        "55ca80e7d961101516f852a1d7dfd64bd23bcc82807064b5d83b8320a7fef9c0", "bfe675db689bb1afd0424f2d90c489d2"
    ),
    # FGT_7000E-v7.4.5.M-build2702-FORTINET
    "001763f3091cd5ea4b3670d6c5b06a903a1b408b50be9517ffec76f092e61b36": AesKey(
        "fbb5dfac4354ce24414878cdfca31bc290f2b8b40f9e61ebf433351df6fad0c9", "31738c6337c1d19329d572b119235704"
    ),
    # FGT_7000F-v7.4.5.M-build2702-FORTINET
    "d08c7af225c69ede7bdd41a58af8c029a3a869969cfc59737332fa708e789dd7": AesKey(
        "5ce46cb7eb7195e4ea8eb6812855215845c0b8eaa0c25888583ac9277bd06fc2", "b2e8e6b35f04f77166f07fc3bbe51eb3"
    ),
    # FGT_70F-v7.4.5.M-build2702-FORTINET
    "03e9da9b59a047c5b94a231a0ba0ed85967d07365948b6658fcaef35637d7d57": AesKey(
        "22a455e78f44c53824e2021a101caefaa7fe21dd83ad12210290eef18d593ee6", "3d7b6afac2c87d5984362b1914cc9985"
    ),
    # FGT_71F-v7.4.5.M-build2702-FORTINET
    "49ee82ef62b65b6fa1a3f5ffa586343a933a0298a6d0dd1543360a1ece38ee79": AesKey(
        "45bdc9d8056f1f5cb6dde4567e1ef148a95a079410dbf3e6acae1bbbbe0dc256", "8e67f3bd90177c77ace57b4d81b313ff"
    ),
    # FGT_800D-v7.4.5.M-build2702-FORTINET
    "bae4d1227d2d593552c10e8ad2b7cfb742c290e000858e98260b25b9cacabae0": AesKey(
        "3a8ddf63321b953b23cd224f8839ecffefa25416122a0364e7179d14ab91b93f", "3347129b09af6d93ca0be751197ec13c"
    ),
    # FGT_80E-v7.4.5.M-build2702-FORTINET
    "391f6e987b059a855422fb737fb7e2a05198fc4208fd6d260c5df1f2e51f66d5": AesKey(
        "0935efc8569dd5c0995345251f35f6953274a3ffdfb7ee54dba67d08d8a2edef", "17338c0152f66a0786c1a32642784429"
    ),
    # FGT_80E_POE-v7.4.5.M-build2702-FORTINET
    "dccb7ed253eddf849c6000fb2032cd59716362390c49cf3ae1645523e181fdf8": AesKey(
        "1908ee9f354e31e138ba413a6a6acd9f4cb09a79fcf89aa5d8349b903e53c554", "f6b68d404f8dc8065e687a629d004dd3"
    ),
    # FGT_80F-v7.4.5.M-build2702-FORTINET
    "478d8c68776db80d0e11207458920eed9e2e4146f750814725d87bea8651ba6a": AesKey(
        "3f7ee38aae62f2cccdb0d32535766a3c53fd8f9691947e00b0f983838e53aa3e", "104fe8059c83b4eb3b42242d99d2e78a"
    ),
    # FGT_80F_BP-v7.4.5.M-build2702-FORTINET
    "08c514ad847e893b0ead1705db404f5618ef11e114ddc78a9c9e83c2de98bfd7": AesKey(
        "ecde1f7034aacb0aa9fd47a26bd8dfb4e0560c7f560983c9d3b232a5e4b7a813", "e3838254ef5ebebbc50a2a3dac4df8e1"
    ),
    # FGT_80F_DSL-v7.4.5.M-build2702-FORTINET
    "b7d72078186b96f1c23b1fa335b980c7da6b7b2797711f331f880f2ff85b522a": AesKey(
        "6b0cfa9d92b22272b323812d2b0b43ca04b3803e4c47018f62e31610f25f1239", "74f23aa650aa936d438632a736281fe7"
    ),
    # FGT_80F_POE-v7.4.5.M-build2702-FORTINET
    "871b243282d151b767145c7dd268b78ff1906f58cf8e99c2e35e64609fe2b8a4": AesKey(
        "7aaf64786dd93faef65bbe413923b33bc2e9fe95f409ad0a43a310d70e372847", "d97d7ea4bc0b717f01333e13daacc6fc"
    ),
    # FGT_81E-v7.4.5.M-build2702-FORTINET
    "49c59b4acdd98be033991fbcc2dd0ef2e1a25c1d5a7d74e6efc689193d94faeb": AesKey(
        "eef32fb505453ee080d472e1cb5d1a6c784c093b624a73a72046873b8d6c8663", "039188077282ab73a64a5d7ebabd724b"
    ),
    # FGT_81E_POE-v7.4.5.M-build2702-FORTINET
    "09840d673aea8159c01e92af4339a60e9e20102d413e7063de599c543e1879da": AesKey(
        "9ca7d8e2dcf42181cc9cccb0f6018af79c8eb4fb5b3f31780382a4bc01d61eb5", "8a94736464956d2924eb640530d48fa2"
    ),
    # FGT_81F-v7.4.5.M-build2702-FORTINET
    "22ea1b1b5a01135e4520118a71fa6c0fc937f42a6c97eca56c156a7c243efb0f": AesKey(
        "f05fa53a4fda65c507754259fb3cd7aed024c4d017488d1a80041679f8c989e4", "4451d9049653917bd1f7ec8c22fb3d33"
    ),
    # FGT_81F_POE-v7.4.5.M-build2702-FORTINET
    "a4c9a7ad380908eabaa7898dff2c7c08a056e1106b19a7298afc8a140d54c9cf": AesKey(
        "afba02104e5c0e1351ce322f285c7935d25b6e7f074ef97d4d3541cac075ec4b", "89973cb4cc95ede08647f0e393bb2c6a"
    ),
    # FGT_900D-v7.4.5.M-build2702-FORTINET
    "52d320ee98aaab633e403cf91e8e57694380724da778e960b1914f96c9c19010": AesKey(
        "36a9aee2ed9eb14ccf3a5f3dcc58a7e9b15c11b0509038c4396643e21a7a8960", "75e6018700337cdcd1a03c5f60b52c00"
    ),
    # FGT_900G-v7.4.5.M-build2702-FORTINET
    "c9db2e47cb966e2f897182b34358d69cd1b07ac401e5e468b23ec3727e69a396": AesKey(
        "9d15e116ad43bbdb9fddb8fa930a2f801cf21155b8fcd850765d0f36ee4f242c", "48b8df687d6adefa280b6312028dde81"
    ),
    # FGT_901G-v7.4.5.M-build2702-FORTINET
    "3e71b34dd14043455c3dfd308d217cd17380f0804bf3fa830d3bacfecae97849": AesKey(
        "dced0fbeccfd666a0a8a4aac862d152417ee62d7b9707ea8ade135e0836dc716", "17f2b4256cef2615728c83575ab33961"
    ),
    # FGT_90E-v7.4.5.M-build2702-FORTINET
    "97ee715e974d178ac3ecff456e4e9200a0beb4d15a189f5f80f79377d10ff98d": AesKey(
        "3e52a36c2b467a01dffa2c8c9c35e71e2e6c2b80a63f4a0fcfdea57c2f4092ef", "050ed428885330fa26328a3a6e6cb4c7"
    ),
    # FGT_90G-v7.4.5.M-build2702-FORTINET
    "a8c8dd0add73773733422fffed5c3dc75a3fb885774ad589f9e9a90ec580dc1e": AesKey(
        "2ef6d2ab7649456f131c3b73dee50f750f56194d8ba9c999ad4016e35fb8e53d", "4036d9dd504d3c1c98d616218c954435"
    ),
    # FGT_91E-v7.4.5.M-build2702-FORTINET
    "8180727ad89884a3a5be87d83035b9d1bf7fbfeaa4a729b72a91733eb85ec175": AesKey(
        "c49a1436e5723490743627178b5187ab2f9991ded7cd58bcbac7a980435e474e", "ba9770e680267eb72846f11082b46584"
    ),
    # FGT_91G-v7.4.5.M-build2702-FORTINET
    "5b1a67a37cce357d142c5ff3ea1f8adad275988b4727c240a40876c853466d32": AesKey(
        "f80b35cdc82a655b8af63127186967e032478285e04d2621581401462bb103e0", "26ef0363f5ed908f88ed5a1893a3103b"
    ),
    # FGT_ARM64_AWS-v7.4.5.M-build2702-FORTINET
    "e94d771c72d879dd7718c6880af90676088ac9d441e30fdca2d152f3d3e454ff": AesKey(
        "08654bbdb29442210493764ef2d533feb7331d9c065efb727b5eb7d09ff20aee", "ff53f414528d60ebfd39314ba94101c5"
    ),
    # FGT_ARM64_AZURE-v7.4.5.M-build2702-FORTINET
    "d495b1a56da55b0906f84586565206479dfe42c34065f6af25bcad845218ecbc": AesKey(
        "013dcc8f62e0084ebfe0b1967ea38782bfd1441b773f3162d2dfdb46e435eea5", "0a81ae52b35c00380a260a0a20b5088a"
    ),
    # FGT_ARM64_GCP-v7.4.5.M-build2702-FORTINET
    "7768b81be8c48895b22a7f27b83b528192107fffcb13e0207d272174fcc31135": AesKey(
        "99413fecde477dd40b70060bfae330d1f6a21f09a00fd14681e75352afc6de9a", "46082d0a8b63f9e60cf2904c2a1a89f9"
    ),
    # FGT_ARM64_KVM-v7.4.5.M-build2702-FORTINET
    "635f7751402f34bd48b4168c5fa408418e6dbef68bc6fcb4a22c292446c8d0be": AesKey(
        "4e6f4c38176f7a0790f05e197678dc30be73a3a790611ccf4f84469bc30a768e", "f751d46a3a4167bfc2088493b0fa91dc"
    ),
    # FGT_ARM64_OCI-v7.4.5.M-build2702-FORTINET
    "af5de8d3fdfed19a48cf68025be61866a62fbccce29b79880cf1203fb8ed42ba": AesKey(
        "202efe1a5497c7e4e1d69a30f7e0021299a2d33722cdf1338cffed40c9a3607c", "3f17fe6efb93048201170f6b33c97e99"
    ),
    # FGT_VM64-v7.4.5.M-build2702-FORTINET
    "c12236556065c84edb9dd9b78886bcdbd7b8f1c704c989633f1a3384919ff56a": AesKey(
        "a9db08ea23f5c42aa188a811235ffcb9088baeb7540a8fe792188d898369353b", "42bac0458b1a2f3f5fd73c976f7b0f78"
    ),
    # FGT_VM64_ALI-v7.4.5.M-build2702-FORTINET
    "96fb241c059a4da9d5eb19b3fd51c8e3a4a646c0878f84cb57a517ddef2e3235": AesKey(
        "6ecff0b502e9f49eb1add7a209a98c47e6b8cf972ab4cd86ebb76490259b7bc7", "d8ea4a6b62ae8a87d70ccb7dab097bd9"
    ),
    # FGT_VM64_AWS-v7.4.5.M-build2702-FORTINET
    "90b88c8d02488b52393dc0889e98fdc0a14f920d474af7aa5baff9cc9996e33a": AesKey(
        "3d5d603240d14872d29748f02843774a8cd6d5ebc6a16f2cdd6b719f0ba0a650", "d874a4993339e2c3f5fe6703bf5e5eba"
    ),
    # FGT_VM64_AZURE-v7.4.5.M-build2702-FORTINET
    "879a96dbbf09ebf30422dd9f7c83527be883fe71f80b7d3825012fed60c54b20": AesKey(
        "fbcb49c2a3265c12236323a00b9b1735822b6d9bfda59515f1ff0ca0c0a0d270", "1b735b372442d73e7dff63427ae6f65e"
    ),
    # FGT_VM64_GCP-v7.4.5.M-build2702-FORTINET
    "d049b2595c1faf7026e6a8e696bc5661c3faa861604664d6941bce1774cbf4ff": AesKey(
        "a5e2be9e357563140af10af5402c83eacd37d028fdf9b06986a46b24f267630d", "22315c5b7cd17e08223db6477563b7ca"
    ),
    # FGT_VM64_HV-v7.4.5.M-build2702-FORTINET
    "99cf8186ee8ba30aee62ff27af62d903aa0482b05353f20f57c30f661dfbda6d": AesKey(
        "064dace85c78444b109b001dcb423e7dd3ab8653d618841b4671adc60331dddd", "2484c58bbaeaaf4740a5298dd7622e32"
    ),
    # FGT_VM64_IBM-v7.4.5.M-build2702-FORTINET
    "5138b31aaa2ddcaabe615841ab3d7f394dc47663fb6be8d6e239f95a07d26f60": AesKey(
        "c17accc3e1a3b4e8d00b1bce65c058b69ef824820b7ef7032bc6769171016923", "2e2f95c024e3f0b8ec391b24f657a32f"
    ),
    # FGT_VM64_KVM-v7.4.5.M-build2702-FORTINET
    "f4da491872c48160080afa9d5682aeff7fb90915c75313c099c42eeb2fad3858": AesKey(
        "c2bc0e96511cafd2e1867090b49d599f3f6346c63a93319b8f3af96f68cab632", "e4e9ed1bcd01c52f9cc5aee816e12a28"
    ),
    # FGT_VM64_OPC-v7.4.5.M-build2702-FORTINET
    "262783bf5459a9cdbd0897e2d0698bc008571599b5ea5e44cd0348d8523e31c7": AesKey(
        "4104fffb10a9547458d3c66e8b82566ca344e16d19a75c34b76395a5587a99b1", "594e3615508241eadcfbff45c6114d25"
    ),
    # FGT_VM64_RAXONDEMAND-v7.4.5.M-build2702-FORTINET
    "107e66dd0da0758d24d55c808ce8f5654cafe780ae4450399aa972ec180ba386": AesKey(
        "0a7c1d4a811f17531c6b565f301de2670d2338bb7385a4feeadf8454c11a2c28", "d40b82adbd4d2f0e44b72a0a4b918e8c"
    ),
    # FGT_VM64_XEN-v7.4.5.M-build2702-FORTINET
    "0fa12e7e3b925e74e9b152ea727c2ec4e1e9bd1b73bcf3b0997e475cfeb5ed59": AesKey(
        "1e18cd66a8cbb26751e37a0bc26cbc8f3b5d5fe5579a35dfd9750afea7b19239", "efd2bad03fe8c4114bd4b56efd587385"
    ),
    # FWF_40F-v7.4.5.M-build2702-FORTINET
    "9317ac3645860ae1e63bb81fafed136fdf0ed5785d4099340711075ee65b6b60": AesKey(
        "963ee8aed7e16193d3d5a37aabbd1919436cef06cf2da40386c9072f6b06ac19", "45f75254abb2ce39ca56d41dd264a49d"
    ),
    # FWF_40F_3G4G-v7.4.5.M-build2702-FORTINET
    "3fa6d7c13b2502260181eba34e04f3a2b9926398997ffa5b604ed3a82ca3ede3": AesKey(
        "937806d4e72060b0a23d02b1f68dbd8bb15ede12b6dbb62635ed305e2b070c63", "cff6ad5ef1b223d55a93ef0534b29292"
    ),
    # FWF_60E-v7.4.5.M-build2702-FORTINET
    "a5d23a88e3f70717f57fb372d83f19de2af301dda9cf4159029d3a6dd04d0f2a": AesKey(
        "af23d6de24679a5e4716acf750428d2e88fe9695afc4731424bbb427190e81fa", "3aa6e0d2695c500f826c2d2dd5361ece"
    ),
    # FWF_60E_DSL-v7.4.5.M-build2702-FORTINET
    "19ecf0a6e66f205741dcae993f356f621624e7878b26355f5264356f114a03f5": AesKey(
        "44b852f79c5251c39b697dbeb620f817ff759a0ddf4e0c120b577d4b1e56bc84", "ef27b83da2d88a2da6701dcc15f7444a"
    ),
    # FWF_60E_DSLJ-v7.4.5.M-build2702-FORTINET
    "f1c6c1df421f3b84a797fe88fd2f7c66f48eeb93e9254f988c7ab995da27297a": AesKey(
        "bd8be3d83305f1b85a90acc8b0b1161394689ba80b37481033911fcefa5d6361", "7e124089536e63f80850beddbec0b36d"
    ),
    # FWF_60F-v7.4.5.M-build2702-FORTINET
    "ab24ae8d41f055d616e48bebe8b5041fc85bddb3d06759d33b1f84cfb2e2fce2": AesKey(
        "a0e41a8bd6f866ab871ae60eaa1e6bbd942062cc22291f21073c028d86fd6a93", "0275613f2b1ac4230d0917c121f3b21d"
    ),
    # FWF_61E-v7.4.5.M-build2702-FORTINET
    "17d433082c8fc3d76fb166465eb28970511ae0b71cb2af395e2ecd2ef8f13735": AesKey(
        "7d80af2e70ef521eff2a36fd1fa65a31748afc8efe0275462b5a2a653bc7e8c9", "2a2d88b836787b268fa3602166f51216"
    ),
    # FWF_61F-v7.4.5.M-build2702-FORTINET
    "d93fbb829ddb6f06cc3e0bb64d37fe62e49c5099c4b7662558bf7317052411c8": AesKey(
        "8fa18db13ac91fd718aeed0c9222c1e62c49d2049dfdd932dcbc892a4758069e", "0120e399099a96e6bd97e480778e22ca"
    ),
    # FWF_80F_2R-v7.4.5.M-build2702-FORTINET
    "4bbb2b5ba90457c766e4941305c6434633483d0a8a07ddd687125acfcfae3993": AesKey(
        "fdce2d4b72cd745717f421e854608e44b5b954c8847f5ea0393e184be65d27ab", "a7650be8a8027ec5cb757b6ec1fc07bd"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.4.5.M-build2702-FORTINET
    "b5a22ad363663afeeb2087bac1d4a5536eacddb148b7c79c6e801a1f3ff0bbea": AesKey(
        "e1ee774b1777b2dd2f441c295de67022e193cd8476eb34f22710fc28700844c9", "b0e5b64ce08c183f04cbab2012290575"
    ),
    # FWF_81F_2R-v7.4.5.M-build2702-FORTINET
    "1bd56559c4c977e786d7cf17a093347e95d7f6e01b68958e272f54014a0c8554": AesKey(
        "81b81b30544d601d6939c6033cb1b26b60c498b664198a6e6abe68fcc30b4f10", "152fd634c9b05c466a2cbd4731446e47"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.4.5.M-build2702-FORTINET
    "e6a74ca12b928666e981c75c973edc86dcf57c85cf03e4da4b6e1a1afe97b560": AesKey(
        "13557a54300283fdf8722c24393ba380ba00f878f5ddb109de24a9d62fe5831a", "7c383766605cfc57e28ce67935940f48"
    ),
    # FWF_81F_2R_3G4G_POE-v7.4.5.M-build2702-FORTINET
    "a122e542c15cc2d6a2e7a97a390894cd8a138775ec6fc9488d1e144e4892088d": AesKey(
        "76d683760f9bad4da47f9d2438814af44d4b6a8cadc1b91689755496f6101cfd", "7bc2d214fd608b7c2fd27b413af6ceb6"
    ),
    # FWF_81F_2R_POE-v7.4.5.M-build2702-FORTINET
    "cdbbb34a75a4d611bc55d8a86f0099843b368844f6d09918c362fabceed765d5": AesKey(
        "daa4ef4b7da3ffc627c16d7f3ba57f1482b5b11e8f007c185dee0a3955e665fb", "569e8b9684ae179a1db24af1c3b2a6bc"
    ),
    # FFW_1801F-v7.4.6.M-build2726-FORTINET
    "0ebb83a45571a74ba4d9181dffd132cac01f5739ef68a4edc5efd86fdf0c1a99": AesKey(
        "70e029167d07edba728ba31797c74ec090d7967e38db0082762fb923b74a799b", "099ce79d983fd7a4999c414beba1e552"
    ),
    # FFW_2600F-v7.4.6.M-build2726-FORTINET
    "915353221ed28d24bdf63a5d67136b5a5919be106f850da99a121bfbe6717a86": AesKey(
        "8da4e5a3b07e0ec7c8b19a1d2fd6e071300ed4f60eb394c1e1ba31f8d54e0731", "608236ecc906d4ff8bb4e3fdd4cec1c5"
    ),
    # FFW_3001F-v7.4.6.M-build2726-FORTINET
    "5243bb23a23f56d82d970b8267c63f3119876d31cbaf37320cc47af3d1c75b69": AesKey(
        "ca216d11c88fe51649b99641fa683fc36ce00936ee803d5623d6958bdd572c23", "23d6bcab29293e8f1a100717c5dbb400"
    ),
    # FFW_3501F-v7.4.6.M-build2726-FORTINET
    "8f6086d02fa33c1002119371bcbe77d09af88de3775b4361d56245dd33a567bb": AesKey(
        "e12c9ae1b326b2bfdee786a6aac3e67867a5f9a83a45e0942d694c3abbe16e44", "671416fc83e13de5095142531e9f052d"
    ),
    # FFW_3980E-v7.4.6.M-build2726-FORTINET
    "9dc462f852367e30cb0ab00a4ea6cb1f6ed18d9feb371e9d683fdf6979ad6dcf": AesKey(
        "4b840c9784534578e34fe22e61695a988467fa02610164c609009cad578c4ffc", "62623d8f1a9123a17391668dbd201ed9"
    ),
    # FFW_4200F-v7.4.6.M-build2726-FORTINET
    "7984240febb61c1f235a8ad21f2d649125163c0dd3e721f169b3902dbcbb2fba": AesKey(
        "9ead603081f611ea18ce04b22025fafe922f4f7977e2fa5869ba608f47a1f210", "a91e869da9900da019f65c95db407013"
    ),
    # FFW_4400F-v7.4.6.M-build2726-FORTINET
    "3ec5780e64b650206e9d0d39843647f66e3e9a8274bf8dc09f84207912d02d51": AesKey(
        "c6c4dc4b293f9704e17892db93aca42813cafb7a6b053d2d572b0d8cc7bd54f5", "b25ab86995dc8cc0d1f320de1bb1f5a5"
    ),
    # FFW_4401F-v7.4.6.M-build2726-FORTINET
    "d271702947cb753365af3be112fb31dd09df4a0f05a3c097e781a29ac7592ab5": AesKey(
        "9bc1108c9778e44728a058d39501317df32fe935ae040c62fa5b94aad06508d4", "ef7d8acfdee733cb80c340d3bf483358"
    ),
    # FFW_4801F-v7.4.6.M-build2726-FORTINET
    "a6376e8c8b34d734e98fdcff48bb0c612adf04217ba8842d9f2da0e1cc379004": AesKey(
        "e2c17cb5ab52a63b4a9004cf343436e79dcc32b6671d4f4e52958a629553c435", "379c5ba23ab318d434f2ba78e20ae75b"
    ),
    # FFW_VM64-v7.4.6.M-build2726-FORTINET
    "46e2740d13f10b8dc414206f9987da863110d56a63affd5a02381bbf78e60448": AesKey(
        "07f9c19f861ffc7411c1ac0917533286370a0a05a805a567824057a5ed8fead5", "ae592f93277a75eb8c232ee56f382453"
    ),
    # FFW_VM64_KVM-v7.4.6.M-build2726-FORTINET
    "852114aecd7934164b2a51425580094986f178e0085e0841f93144914cbc2499": AesKey(
        "112a36a63aa735b550af719f4fd2ad17e01a718de2d1f750dc90daf5df61450d", "9a82c87339527d67e969fd812672bae6"
    ),
    # FGR_60F-v7.4.6.M-build2726-FORTINET
    "7236e4ef2de0d45f96d03f4ca4b9a981a7f23becc0c5cb35b715d0224c0f09c7": AesKey(
        "01c52b64d570685ce864d7f739725846ef6f1efbcb35fc1c556adfe4bf6ad7d2", "842002483baef533a21bbb46a6b2b1b0"
    ),
    # FGR_60F_3G4G-v7.4.6.M-build2726-FORTINET
    "130fafa4cd63382b8dfbd114c9e67174668d9028ddca2ac302d26f223116315a": AesKey(
        "be9221f83e1095fc7b59ee9b21d1856f8cbb8c53cd1370e5a341292b6e7db585", "573fb3caf9ca3e35d14d34713053ff4c"
    ),
    # FGR_70F-v7.4.6.M-build2726-FORTINET
    "4a6cc0903a84ed9b53344e39025a48311fc90136b98a84de2b418a3d2373a88f": AesKey(
        "a12c5642fcaefee249d7f2b55723d165e2c44e0629e276d575ab96bfdd2c4e35", "4d70f48d0b9a8751119286b72733127f"
    ),
    # FGR_70F_3G4G-v7.4.6.M-build2726-FORTINET
    "b070dfe7ca216a5090d041ad29cc495d34a87b88c9969a46b4faed6d2f94868f": AesKey(
        "9f802b414366a66778b6a5a56c992eee61bf08aa3852105ab0cff6664bb701fb", "97a39f65df9cbf7222ceb3321a061f0e"
    ),
    # FGT_1000D-v7.4.6.M-build2726-FORTINET
    "529f8a0f2d7b91962c541b9da1b1ce513d9d4f494f007fb77eec9d02d45f8b2d": AesKey(
        "6c338d2aa115d55e535e1bf7057ce5e508024f03de45ce035ab112e5d7289dd6", "d3765504b598996c858efd08a3c71e4f"
    ),
    # FGT_1000F-v7.4.6.M-build2726-FORTINET
    "907ca193a8bbf7694146aae871dc159c99d87b0536e440529bbb964de4d5177b": AesKey(
        "3af09f772930cf13238b79c7803a3d047effcf972fedd29222ee4322dfb4a9b3", "1e302b1b8b7ef349b7fadfc9849bbf4c"
    ),
    # FGT_1001F-v7.4.6.M-build2726-FORTINET
    "73d08c4ae27c7268b5a45fc01d7b9ab2b73ce2da1eae687ab44d4e7484116ad2": AesKey(
        "82aba9cd29899fb906feb7e03136664b230bf347268ded08d382829b983c1353", "1d7fd8f4891f58972246c0c64da23f5a"
    ),
    # FGT_100F-v7.4.6.M-build2726-FORTINET
    "3bbee11329d2623fda94cd77c7f24f3c96958f6f845a382f5f83f71cb8f9f06c": AesKey(
        "8ae6dfa88362e8bf56a730e4f03fba1db7bc5f3854cfd73c261b6e89d506ace5", "d3ca8c32952c73bdd0436e667d739148"
    ),
    # FGT_101F-v7.4.6.M-build2726-FORTINET
    "bc0accdbd88708a8459b36a0edb7df7f6f094158276ceb9af62d0b8226085fa1": AesKey(
        "794264ce1201febdb212c71b44ab912d1acd167d343f172a3954ab88cbc4cf8c", "5ec0ff04708401b42b6bc9421ab96428"
    ),
    # FGT_1100E-v7.4.6.M-build2726-FORTINET
    "5c53e6cdc86251cd983a85efc2d38edb67940752c4fb7589cbae8ad5a5563c5a": AesKey(
        "20a7d27bc3eeb27e3351ad63a98931727cf5c3586a528c0680082f3518daa9fd", "690ffb889a18e25992e21b5cd1b14436"
    ),
    # FGT_1101E-v7.4.6.M-build2726-FORTINET
    "5b6547f16752e323078231e1cab08f48cdc8d82e252b949a0a696ca7c1cd859d": AesKey(
        "d12c1252234778f5cf1fc4c8504a6bf9aea81cfc24747884ba0a1a8d40231a0c", "cccf44c2bd6696c50c3f5979e157d851"
    ),
    # FGT_120G-v7.4.6.M-build2726-FORTINET
    "93d9bbd9bdd1a97ac04d614dacdfbc10605eccf1ee7b02ac1d29cfa2154df625": AesKey(
        "28cde5273a61f6694c07516ae78eb8a4df751db388382281e9f3ee0c4ed3429e", "0ab720ae3f99623254e2329bb7fa243f"
    ),
    # FGT_121G-v7.4.6.M-build2726-FORTINET
    "dda87c92e0170ca07b82c54e54c7dd739f22998f8b4834782af167aa447d534a": AesKey(
        "d90db433f8307f0b66dbdd77f340e36fbac8f02ff434074ce6d5b95e84826652", "9ea925fd03c9131040294228f6f75416"
    ),
    # FGT_140E-v7.4.6.M-build2726-FORTINET
    "3735011461cb7eb3930225c7ee4b9f265fc1e721efb662d358aa72ea59532ad1": AesKey(
        "c4c54c55b3ee41c2b9ca94354da7792327029ef414c1a265c983025d5b65ecdf", "bbb46b205ab372544537944037572dca"
    ),
    # FGT_140E_POE-v7.4.6.M-build2726-FORTINET
    "2ea09314ff66931bcf2630567eb9e7feb0d404ceff50166592c2d9e7789153da": AesKey(
        "8b33e424f751365d3e3acc27b1b6989ed53b2a15c5aa8dc2887e5056d39ed8c2", "98275ab1894b9e3e0b3e08462da955e4"
    ),
    # FGT_1800F-v7.4.6.M-build2726-FORTINET
    "8e5a272b68f804d8057c5a9bf242b756bca08e2babbd303dd375a4a014fc6df7": AesKey(
        "221384ace4b568ffc296eafa03e784668fa2c3e34bcf031452704a4a2818d195", "3d6bd4a85bcd5ef29e34304bac708359"
    ),
    # FGT_1801F-v7.4.6.M-build2726-FORTINET
    "cd24aba99f31da55e0c68d37508f33a9d9427015f6006222d32866a6ef53bfd6": AesKey(
        "a69e0880087a40f576cb7680b104aa0894ca17c71aa84cd4a2928df50667635e", "f486053b694e497d943b10519a72a3a5"
    ),
    # FGT_2000E-v7.4.6.M-build2726-FORTINET
    "ffb990d0eba165952bc1320c3370a399a8d955005dd850a5238ed1a6f9145df5": AesKey(
        "4b5ae4ab2ee40c0058a63c5aa033d3cc1f06a82d39b23cb4fce5d11bd58d8822", "b3165feed2d8150837c8c63eb2965f06"
    ),
    # FGT_200E-v7.4.6.M-build2726-FORTINET
    "0ec4bfcb5cb54bd4be8d1c84e594a09116cb5a46b87e297d7f7fdfaf0875ed41": AesKey(
        "3d5445959a95ef6fcd0afe8c309ddf736eaa4aefe415d2f810ea4208d6847f6f", "4760bb49666ac6b28778f95d2ecbe3c5"
    ),
    # FGT_200F-v7.4.6.M-build2726-FORTINET
    "823c21f1284571a8f54de866c026a4cc9d113f077cc908e37f8a4f9523594e4a": AesKey(
        "c36dbe631efbaf7282a87853413daa1008a6ad56904fa8878ea782788c50f8a6", "0f1b84d90dc457b867bac2c0036ea98f"
    ),
    # FGT_201E-v7.4.6.M-build2726-FORTINET
    "93545c5b5af31c61993640722e43f2b7b392e6287ad715c64f75982f4cf24cf1": AesKey(
        "35c6a2f0d01ec64c82cbf75d6bc6ffc813b2355cd4af4fa51db81320b2a67d75", "121bf9da49677f1ab3ff84cdb7bd888f"
    ),
    # FGT_201F-v7.4.6.M-build2726-FORTINET
    "031faeeabca4e7a5b369259ac3366ec3cc023ebde9b3d70a2e7c5407d0c5574e": AesKey(
        "37547930858d226c692df0ed89b3c14c30adc0bf460228e2dec8dfd6bfa4ab68", "ad7714f0605aa09d0bb9ce8708c737e1"
    ),
    # FGT_2200E-v7.4.6.M-build2726-FORTINET
    "2e9920e6574ad27624afde3f265e647bb673447b9a418bfbbdca56423c082fe5": AesKey(
        "4c3dbe02535e6e0ebf7701a801d20a716dc0b2cc45c42796d0556b78da04f166", "83bb15877d645946db4f4f3ef714010d"
    ),
    # FGT_2201E-v7.4.6.M-build2726-FORTINET
    "d0344e5351f38219ae8c56c61f50a6d70392fe96f26c85831bd87973c4012d72": AesKey(
        "18b69367a66a63608a2bc7ac8498985eebcdc9be6e609ec94d7620cd2506f5e6", "3fa4a63860525992e7aa3ec0a18e10d9"
    ),
    # FGT_2500E-v7.4.6.M-build2726-FORTINET
    "49833df2ce3a17cb851b1c22c603e8ce730c2b37e7b958ca1edad3c1cefdbc44": AesKey(
        "ef30a90c2c0681f9f9fb21290dda56de3fd523274edcb757b3bf89bbeb2180f8", "9a89c7264cdd13ae384de0ca56d7f5bd"
    ),
    # FGT_2600F-v7.4.6.M-build2726-FORTINET
    "2523bc680382b1bd97fdf1f235253efc3245928ee93c5e0f021d10415256b941": AesKey(
        "415553b8cc18bdccc4136adb42eef6f57bd670eaa30a16d6de2ca714e0fbf19d", "f3a88da89e8862908e5bf8ba6fe00c6c"
    ),
    # FGT_2601F-v7.4.6.M-build2726-FORTINET
    "17749bb285a24b6b0fb3b8eeb2858bc828e281a56e40ab267f7de3a2c42a4c0b": AesKey(
        "0e45c383cd161d7b052833967d799bd87b4e9eb30aeb1cbf5949d4b3bc93be97", "b4e6595c9830d76586237c13d1b3585c"
    ),
    # FGT_3000D-v7.4.6.M-build2726-FORTINET
    "f210891da4b69f36626e843c54fd9d213c86b1dc089eb566f8e5c9e384dee820": AesKey(
        "33446bd6c4c1ed7cf1550838286c90b2aea6f7cd90d13c3d87456932aa0022e7", "750b79a7a11cd080e213dcf56e3934b1"
    ),
    # FGT_3000F-v7.4.6.M-build2726-FORTINET
    "507d264794194ebcec746a051c43d4097f3bb0bdc5f80ff083289eb51b72b707": AesKey(
        "d48e9573c9bb1929933996d3ba4cce2185afa17d66b76e8b9a659c47bdcbb4e3", "3c7b5b66bb1a7eca5565fb92469cc330"
    ),
    # FGT_3001F-v7.4.6.M-build2726-FORTINET
    "31d320ca0f8afbca73c42d820ef1f22f5cc132723da6be10f6d45e1f5fa20f88": AesKey(
        "68efc7663bfa2bf9c9cee2b88d7f0e7713797f3ffd74a69bdc4e7fa066d583a6", "f5d64b2ea8a184da8bbe9ce0223d512f"
    ),
    # FGT_300E-v7.4.6.M-build2726-FORTINET
    "ddcc14245a10f5b62096857ef0c65ef8c7f10134066b4e223060b1a7b7c19dc2": AesKey(
        "2178f5cb4197ce8015ebdbca17f64a0f0cb8f0823066ae3609366c1c5d18bc1c", "a02adc839bbbb4234547f6a8ab2fc1bf"
    ),
    # FGT_301E-v7.4.6.M-build2726-FORTINET
    "f72ab4ca83792b515206c79d2880c60dedd992293e2f2004f4efee90ad30923b": AesKey(
        "ee3054e86dd9a3a01819c9b018f8af697b9d229bba6e20ee797727c5938b2627", "8cee218c139301e5dd255c7833f59d5d"
    ),
    # FGT_3100D-v7.4.6.M-build2726-FORTINET
    "c7933434a6aa1efc864d20ef2538d65f90b7d5c273719099ea905147a03107be": AesKey(
        "d711970c216ee0faaef134ed978e51b541d7b39bd971c2f739c37c7c8a34e53f", "248911bd0e55dd34e1170ffb271eafc7"
    ),
    # FGT_3200D-v7.4.6.M-build2726-FORTINET
    "0d7f2c206ddd7889c085a43fb27ff544d31e2e15285daa4b63daa73e51581dd6": AesKey(
        "75b569b17b259618a1224b60c5eb85f20a9a2c64b9749523f6c99cf26b6089db", "27628c165c1fa2e77ae7f7038d2c6ffc"
    ),
    # FGT_3200F-v7.4.6.M-build2726-FORTINET
    "c5b096d8d4b605999644fc84a6e25b03a4a6354d1fe83c46515984136763d4f1": AesKey(
        "5aa56b56a3a6b05db8a01e9e4da678ab7fb33f574b4febd1a7119d9358fae4ab", "def8a017b286fd72ffe446c251daea7f"
    ),
    # FGT_3201F-v7.4.6.M-build2726-FORTINET
    "9aeeeeadb4ac95c2a73802d982044e347cb0a8199aed304858c8371418a2123c": AesKey(
        "74b9bee0113640ea7bc7ff81a1bde283f3a35dfd1c477602be8694a17ee49443", "6a3eeea1e7b196f9269f3774d8e130f7"
    ),
    # FGT_3300E-v7.4.6.M-build2726-FORTINET
    "443d0e4e8a1a114d68e38e1eb69ee3007ccb594507ce9ac960df1ab32311b47c": AesKey(
        "3efad03105e468b3f7f5f44c7567821ede5bb3aa38f615e4e5f1468a002d53fb", "60918e9b72c893f82dc2dc94849aea05"
    ),
    # FGT_3301E-v7.4.6.M-build2726-FORTINET
    "db3c859bda1c0c9d61896a0a3f55bca3aefaaa0e7ed9ca7fb8d8f5516b1f5382": AesKey(
        "779e17f12ea3cab22f84b753be61f4ae56c7c7f16088b99d962a1dba89ced73e", "f4a0c33a3c67bf36178512f78a841488"
    ),
    # FGT_3400E-v7.4.6.M-build2726-FORTINET
    "0a67d560afbe12de0ba7c9dc2eafae1f438afe5e9794557bdb1a7013085ce574": AesKey(
        "c5ad66b7a012aaecec7444a00d9064a00400bde361a05cbce08195bf8900db82", "96964b2cbc8feda57bc659d6c356bdf4"
    ),
    # FGT_3401E-v7.4.6.M-build2726-FORTINET
    "a8fb6b2be3e1c46362a270480d1cd2ed3468de27fc822a83357781de79301624": AesKey(
        "b8a1df6192a8cae043e16ba2352d6caac9789f0d9a3fe7cd280d19b0a67cc303", "ee63e4c535b46740083bacf8a2acdd90"
    ),
    # FGT_3500F-v7.4.6.M-build2726-FORTINET
    "5145ac6991ca079366f425cd79d39b2470b80dc6b472a871e8bf28dc62af2545": AesKey(
        "4124e9d6a6758cb6a997b8f9b70656b31b466e9e5d6ad967fa20603888e28bb4", "0c32afce621da474a9bd801e9b3f67c7"
    ),
    # FGT_3501F-v7.4.6.M-build2726-FORTINET
    "de3f8e42aa6981b8dcf3912fd55522511e01f0726261316b922be9b3d23d5f50": AesKey(
        "9076dd7c271a46169bad025b7a9b2c02cc2d3039653dfe984dee699bd2259826", "2f903364b2508de2ffa8a5384450f020"
    ),
    # FGT_3600E-v7.4.6.M-build2726-FORTINET
    "fb804e73870ecf3939e074576b472ac0c773702917bf9fd51ec3f07eec9ea702": AesKey(
        "a3838e97db53c9880eb4dcf428b4e67a982896faa1f0b940761a4ec02b46e4d4", "513381621620c3729c581ba962a256c5"
    ),
    # FGT_3601E-v7.4.6.M-build2726-FORTINET
    "2eee0b5edf4c30bd6b4929bff60c3c6678c0981ac7bc7bf6cec9a3ca3a624aad": AesKey(
        "999ed76bc4b42360e1365c707745361a5abed4bb5016094159790a0bb3d3d643", "2eab2caea10ff2e3df72cf69bd800a7a"
    ),
    # FGT_3700D-v7.4.6.M-build2726-FORTINET
    "a934c1a84bfb0c15b63030909b1f2ee454aef3aad2dff0355ec6de4c362f6a66": AesKey(
        "be4f3c44c80963e95d69bc7078a95942dae0fecbb121ef57f5736cb494a14d93", "65c34bdda7f9a032678a9af64dd039c3"
    ),
    # FGT_3700F-v7.4.6.M-build2726-FORTINET
    "fa02480594cf7058f048dbbbe463b5a9410e3ec852ae901d37a34609b63d79aa": AesKey(
        "6b32033cbfa9e45c198ff6a4e1f8c6c8f2a3220fef1801885c121491524b9d3d", "3508a925fb8cf7cf9f5d64c7d3186b5f"
    ),
    # FGT_3701F-v7.4.6.M-build2726-FORTINET
    "2e4934e6f416c2d4a6d8f6b9fdadcd2a6a8092e5b00b3b73780a502dc53e0af3": AesKey(
        "fa61c35a3d4f3d6d83d1b7c980e6e3f502775b96309c2376e9aef20b744aafc8", "ac65179ee329490cc532958451c2e555"
    ),
    # FGT_3960E-v7.4.6.M-build2726-FORTINET
    "21fd59b5355a68bf0dcab1da66d63a07de09f57789f62619a7d3e06f61a64f20": AesKey(
        "415269cbf4599670d713acf9aae923c4227ca0a590fb8c0a429ea3ae37365c44", "1e1192a39a96057c46fed3a343d26154"
    ),
    # FGT_3980E-v7.4.6.M-build2726-FORTINET
    "1aab54964f255b75cc7783112d60284f204a00703e1402f2ddd9e1e458c78e99": AesKey(
        "bac6c2e5baf1580f701bbd27d30ec31ce8e9c5ce29fa3d8d35e4b82a87da4e62", "f6708822b9fcfa30b262f5fcee41096f"
    ),
    # FGT_400E-v7.4.6.M-build2726-FORTINET
    "efe6e45d7a14c95cca646a4b7695c113ec9d196efcd0229273eec4476ece4d4f": AesKey(
        "6912a1383c857b9f98187aa974e12ef9e66cf5034c0ad0e3c9451bb96630857c", "b116d8a3e4c81263ddba91adc74f9f86"
    ),
    # FGT_400E_BP-v7.4.6.M-build2726-FORTINET
    "1f065d6aec1fcbda36126ba04f63a7d7097df117d6bb88d313c1dfea333ea9f4": AesKey(
        "41881adf8f583dde541dcdc59f0b8463f5c7f32d818542e56af35898d715baf9", "36a3f6c7ecc271fe7c9239521fa43db6"
    ),
    # FGT_400F-v7.4.6.M-build2726-FORTINET
    "edc7e996e9626be5b14ecc970b9a2588bc9d507a0a8cd0bd6ffc85ceef216eb7": AesKey(
        "a73376eb6a39f17da7b1d084985cab7408c402de4c568ca0490cc1c810fd61c3", "001a81f48da8eb4b80e814aa133faacf"
    ),
    # FGT_401E-v7.4.6.M-build2726-FORTINET
    "d69ad5e899d4d38ef43225e0cc56ae6facf4187171183ef71fe0bd2694c738ce": AesKey(
        "1c9a3ccc56cea6ba88d77e37e5c6fa8f8f4739835e0becf146fc903bcdb0d952", "f74a68565262a13ab5ac598003b7a6d2"
    ),
    # FGT_401F-v7.4.6.M-build2726-FORTINET
    "936f6ef9cd9c7614206af614613551d55f91ea91c73a1d4adb06995c21188982": AesKey(
        "422497e784665e8a70109877671f802a4f87924f8a6c0c362810578659f9a691", "27673763f507f968dcf25dca80f52134"
    ),
    # FGT_40F-v7.4.6.M-build2726-FORTINET
    "9269505a63482861670139337de5c3ee1a129e3120eda01d1e31778ff8dda51f": AesKey(
        "f7238cc72c8dcb81e06a6b5c51140cace5b1c7bcc60399ec87c7a9759fd61746", "696810f786fb8e99916a7425c698b3cb"
    ),
    # FGT_40F_3G4G-v7.4.6.M-build2726-FORTINET
    "e87fde2da9e0bae68c1b0093912314659b8d3609ae88e8dc2d9924e33e6e57ee": AesKey(
        "c0a367beb11e95131f81a7de5ccbc4c99463521e2d940afa79bdaa1a4b59edfe", "4da77cb2d785a47f22f2d5f5a16e8f35"
    ),
    # FGT_4200F-v7.4.6.M-build2726-FORTINET
    "52c510afe6995720851fd929f6ead1ecc7a5acda0329eb2689e0c71d5898e92b": AesKey(
        "d908e169b33d0cfb29149f2e2e65184fa78a91104203e824bed725462ab516ee", "2472045c971f66b3b2c8741236eed691"
    ),
    # FGT_4201F-v7.4.6.M-build2726-FORTINET
    "625630ea33b8e70e5eb286025c86fd90c835b479a32eaa7cb0511731a576dc1f": AesKey(
        "715108cde73aea3e36d834f391668cf8913ec476888cf1de97f9f24561414f67", "626037a061402f350f8d4966c49da74c"
    ),
    # FGT_4400F-v7.4.6.M-build2726-FORTINET
    "c63709d087fe6de5ba6849b292030f721e0971e7530dbc28a8ac0496eee921e7": AesKey(
        "ceed0f121f689d7be359a7fe0d390026b9df0a51b77415c9c0634a360c9c89a1", "184a3d0d10eeb5704e81e94cd93b742a"
    ),
    # FGT_4401F-v7.4.6.M-build2726-FORTINET
    "9e9972a0b952302d9f10cc818910c3c7a7f1e1c279f60826d11035fd482366a7": AesKey(
        "fffca1890dd7db530640933752511593e7f096bd9f2cde7b549b0b25395a4c03", "c0c9e48779615b0dab03a970994eb5bb"
    ),
    # FGT_4800F-v7.4.6.M-build2726-FORTINET
    "c367022167ce18ee076a50207997a17fbfdead1e0b0022b99e79a141042cf322": AesKey(
        "082c208ce0ab85657b3392160b1d10a10406bf3b2b849ca2d602fad54af37386", "6c8d4080ef2820d65d087b22dc445dba"
    ),
    # FGT_4801F-v7.4.6.M-build2726-FORTINET
    "6b6f362e293c7e1b031365bc0b5324ad42c8c0d773227fb03791a56545925617": AesKey(
        "bf012230869d132650dfdb4d546ee601f406b371dad09696564ed7d2cacf1311", "507d703c6bab05681ab7317ff0a08dc9"
    ),
    # FGT_5001E-v7.4.6.M-build2726-FORTINET
    "46aa917474ced5dc2dafea521e86d27d14828050d169a8489a44f15941083625": AesKey(
        "9734fffc6a04bebeabdc0868b0785c60da833b50168e8f6eb55af5112ec4dd49", "c1f7822f6932b2d368e9fadec989039d"
    ),
    # FGT_5001E1-v7.4.6.M-build2726-FORTINET
    "63776fe730ed4374b01dda5cc3c739b81a9f77ccab0e6159d43779ac4931b4b1": AesKey(
        "760122708d8648aad2efc2c6504c0b6337f477ec2b82bc152296308d8fffc32f", "e82511259c77d3fb97dc80dac28634e8"
    ),
    # FGT_500E-v7.4.6.M-build2726-FORTINET
    "766f4908f2ec8eed28b3735345865608a26504201efd01e8f0097c1cb88de4f4": AesKey(
        "1387f44729d43baf40bf126d38c22698eb97553187a388afe8128a826c85598d", "b76e5b537b56fa2ee76cee27226359d0"
    ),
    # FGT_501E-v7.4.6.M-build2726-FORTINET
    "8b125004e970b835430dc406c7910d136673ba24e05aab4310cc710d51e1db48": AesKey(
        "1d4732298e4e5e17c39df7640f5face12d921a2f1bc3178814041c5b1ebd99c3", "fefe113fdc616e6348972efb37bda523"
    ),
    # FGT_6000F-v7.4.6.M-build8336-FORTINET
    "f0ad3db898c22a05b3a53087a663b6cf85d339aefcf5f14419b5612ae77123d2": AesKey(
        "b308461a772fb0281e07599c4f7f22a971fb24b69e7015cece0b52d8f8b45f96", "65fef4d08315c41ac9a90dc80f7785fa"
    ),
    # FGT_600E-v7.4.6.M-build2726-FORTINET
    "50842b5becf5db1c776fd53e587bcb8507243486aac90f5723682650a84cda32": AesKey(
        "817b48fe5f218f2321c78d45024ce54d4e934ca52cd26815c85232e3244b3928", "3280474d6c8ed6f2640e00798be7d1e6"
    ),
    # FGT_600F-v7.4.6.M-build2726-FORTINET
    "4811d3efd9c2a493d820c5a515dcfeb8da2d2cc0d4673d417b1959c86266c0ce": AesKey(
        "a27e3b27706b1f0089109153db5c1054760150bcde559660d5c2d43d21a8011c", "f7e88ebc667e48b7af0880407531b417"
    ),
    # FGT_601E-v7.4.6.M-build2726-FORTINET
    "5c608761afdfcd582141622f34699083353582bdb82872c3793d8f7cba3875a2": AesKey(
        "e4b67ab543b47359d6da6bb749ff048c75ef6405ebd9de86d6c882886c9b5698", "697961ee03bf2a08d6b4369d1b02c4a9"
    ),
    # FGT_601F-v7.4.6.M-build2726-FORTINET
    "cd7ca8a5539536cba08a03bacd9f95f3c6b9a809d28d3908a648afef94dc1aae": AesKey(
        "82f86417d2ac4a103c6fbdb3253283dfa6805711bb7a9882b669fef7dcc3495c", "04adb6ef285df86a6668603618f05c76"
    ),
    # FGT_60E-v7.4.6.M-build2726-FORTINET
    "2f66479eba90eb7f0fa8ad3ec37ca3144256dbd49050bf022cf6a5ba31a59d01": AesKey(
        "f88b4d5aa6f01cc9733861bfba42036543c42562e40e24c15b18cc639954b00f", "40b8ff8395b04956d8dbbb71f392429a"
    ),
    # FGT_60E_DSL-v7.4.6.M-build2726-FORTINET
    "375de2a8977335aae45ae796325fedeaeb1010a970553ab4dbbc950f3e385b1a": AesKey(
        "86032db4934b3e51f2a482e8bb7f64c5d68adb31f4b4aa6f549c9105712b4536", "58cd65a76ab87103147b89292461ba72"
    ),
    # FGT_60E_DSLJ-v7.4.6.M-build2726-FORTINET
    "ec38794949fd13280ec1e66e1e0d895bbd9be625161c64703bf26bc354970323": AesKey(
        "9dea920bf561c0096e87dcf86589e6c60a7866c28b84b01d6a2a28d66710a90a", "f1dee5e56c14d5815c1e5cbe0a82c2d4"
    ),
    # FGT_60E_POE-v7.4.6.M-build2726-FORTINET
    "3e4460550c2d3db47d032cca8dec0dfd55dbcc3f810478de18caab653b736ca8": AesKey(
        "863e466f3d714497996079489e6915c439fa0dba43575ff271d811b8e5cdeab2", "b0623c1dc0d4bdb4c0183fc7c6e08490"
    ),
    # FGT_60F-v7.4.6.M-build2726-FORTINET
    "cdb5b7d4f636666a36a470d653b81ee35362020d59b4db18eadd07c01ec058a3": AesKey(
        "f957d83faad31ac1c4ac6ff0e5acc6fa5075a0a03b1a723c0dc6823595259aeb", "7e0e6bb3083c25de77f60280840a103c"
    ),
    # FGT_61E-v7.4.6.M-build2726-FORTINET
    "a89fc2145610799fbea69734601afbe75c3c94096e97fd4eeb8c8e01d5ababb8": AesKey(
        "7074e9264a9cbf3ddf94356548c62f8ab904202c0f5b30cbe47e1d1fdf9f10ee", "d2694c3ca9fed553d91aa0a38f19fa11"
    ),
    # FGT_61F-v7.4.6.M-build2726-FORTINET
    "9ff0deae77a15e548b3a159fb10974d030726169b110fa060d0bc6ef82075daf": AesKey(
        "5be6b4c1825fce87e8dd1428114a84b9978340408adfa318f4cbb2197e79dc3c", "6bcfaec81789cd407bf008dfd9a72b70"
    ),
    # FGT_7000E-v7.4.6.M-build8336-FORTINET
    "bf8fdb46fd6b588fe2703bf631a2e0be2c921d386a800662ef29c8a7b42acdcd": AesKey(
        "5c134cd94f61b205a66f8b7340d71a945cb93d98c1e99d624f52beb9e3ea4ccb", "48607bf939543387c40073b70288e0f8"
    ),
    # FGT_7000F-v7.4.6.M-build8336-FORTINET
    "b8be6930f2b0e257d4f5256cceb089e02f35d2561273dc23c2b397d30dd9ea78": AesKey(
        "94978665c4d33fc3e655d927d8db53b7bac521c8bf380102a35f6b5dce35335c", "f5fe7555de2b9b9bcf6929222d91ba24"
    ),
    # FGT_70F-v7.4.6.M-build2726-FORTINET
    "9a2c72f82d7df0fb03dd90f82f4a167a7be08c4691a0f4336be6bfcea07143c1": AesKey(
        "9b88cdb7ee97fdf55fbacdd4952923d11b6697733ed2ca0f31b0b24145a20add", "47d404a4a666c45bb3405bd00da836dc"
    ),
    # FGT_71F-v7.4.6.M-build2726-FORTINET
    "5564ed3cadf8010a01233c0200334fafded6b0fdf91791614606d743681ad407": AesKey(
        "7286a8c78252b0550609ea0fa480d24cb2e79863eb44ae2b8293911d45da41bd", "0f882c4b28d789786831cb6a668b11b9"
    ),
    # FGT_800D-v7.4.6.M-build2726-FORTINET
    "7e9aec02da65c84c896be9ea569b00b5efcbbe905595d34049a44ee759fc1279": AesKey(
        "b17cab5dc06e7e15a0de7e94196c56068291c8200a8bad64ff6f30d262475cba", "b6fe7db9c24b91700ad38f5063da49ce"
    ),
    # FGT_80E-v7.4.6.M-build2726-FORTINET
    "5dc6a04bfced7e7eb6336e448d611b139bcf024813524da0a1b8d2533672afd1": AesKey(
        "afe654c7327c1f8680aa38a50e1c10b3e94cf6a72231da70b79c957bfdec89ea", "9882a3bd81f0016f120e7619c702e8e5"
    ),
    # FGT_80E_POE-v7.4.6.M-build2726-FORTINET
    "4cc1dfd68002ab3ec2a35d4666632c220746d3bf1d66bffaa6927292354b1bef": AesKey(
        "e11e895cf048afb497464cd42196c3f937afe50e870f1cae45558552b858f89c", "c05a5d9974bf23a0f5af73b043337f59"
    ),
    # FGT_80F-v7.4.6.M-build2726-FORTINET
    "0ab54736290587a2f765a94474964e3d22410d58fd84c7cc5bdbed7fc13826e0": AesKey(
        "00f45a417a7ed99292ec555bc7b79d8df262815d136bd88288760b2399a16fd5", "b9bb96a5a18776f91e218616e00a7e71"
    ),
    # FGT_80F_BP-v7.4.6.M-build2726-FORTINET
    "80eaddffa3fe68348b9c7ecd4826c69cec6cd47035f396b899e01a74a41f15b6": AesKey(
        "81916193b1e19b970f9f9bf5ca9c7b04c6763820abbf832851cc7c84b70ab77b", "eecd84972c0fa8db24519a8c0147246c"
    ),
    # FGT_80F_DSL-v7.4.6.M-build2726-FORTINET
    "b804bbb175b5b4e2d760b8a5e1b75974917460421739bfa3e9607dae95c6cfe1": AesKey(
        "e3a87141bd6b9d232ad0d553f4de2e6259c430742403a8d5c5e87bbfb52f1758", "17a05723fbd65e6d81f1d5875fccc088"
    ),
    # FGT_80F_POE-v7.4.6.M-build2726-FORTINET
    "0713731dea20158364543392762adde104e5923d694ac6874b99de2202cb86a5": AesKey(
        "5a67121db9581836db2b1a5d0157c0cab4d1ee9d50e1e9ca454abfdc06e1bb42", "e63d0264cdaae785704e2d5c2d5f747b"
    ),
    # FGT_81E-v7.4.6.M-build2726-FORTINET
    "db21e516bfc1a15e094c676853bd1fc8c3201abb0f5eda94107f3d2703a64bb5": AesKey(
        "3df98e54af757cea182345574b7420a32bfc67152346c410f4aa2c44749c002c", "55a180e3ecb271c6009dcfb36fc8b10d"
    ),
    # FGT_81E_POE-v7.4.6.M-build2726-FORTINET
    "974682d387e2374ef4acc4c39d3805fae23c0f96c0efce01452776146882add4": AesKey(
        "7378b60b09bdc9565f1afe93fa50e6407fd72b437f518abf929e61bbfc76ad87", "1c0bb8444724a631e3d74e050f1bc3e9"
    ),
    # FGT_81F-v7.4.6.M-build2726-FORTINET
    "bf1d5203219bd3effc3ba00e33911ea8c925c194daff16c03c241b4073abc823": AesKey(
        "53aa57b6be11d25b2ca490f58b1bc9152318084028621df28cb7988ad7374496", "186bb501c1975efb87024a5a06807267"
    ),
    # FGT_81F_POE-v7.4.6.M-build2726-FORTINET
    "fb58274af2990713b67b8923113a6276303682c91809744446c6fd91b8ec42f2": AesKey(
        "742ad579faed2dbf5d8da2419bc47205df263e3e6cd717aefa768b85ca19d2f6", "87301d1135c96dda3a3ba31a575510f2"
    ),
    # FGT_900D-v7.4.6.M-build2726-FORTINET
    "02ad13a07b86572515314a234b85eaeacee9049b167f88e4b916797ae9dff98f": AesKey(
        "3c72ff2e6aea3077d211bbde1ee248d43f597e32dd74fa9e7bfdaa61848a9a11", "116cdb297440f311b13fac93cc1b3c3e"
    ),
    # FGT_900G-v7.4.6.M-build2726-FORTINET
    "aae1bdca09d12a3a1aa11218c7e95d882fbc409bf8283d7e4854d235d554f025": AesKey(
        "d448f6d58321413924145c061dad5aff68a973314d7ebcf2bb2bc0135bf8284d", "34f22c70d4807a8ef9ae8e017be41c28"
    ),
    # FGT_901G-v7.4.6.M-build2726-FORTINET
    "b184919f9f8231e756a3421fbcc18bec0c34057d418679c2f6d827823eaa58e2": AesKey(
        "a496c5ffbc13d32329c35be02390ba084323ee1a2fbb39b092495c94be025d52", "b41efb097625dbe5abfed92e532f867b"
    ),
    # FGT_90E-v7.4.6.M-build2726-FORTINET
    "20326689f5904e71f0741a4ef5eb8c4f12232dfdce386d1f930471d3714a7ae5": AesKey(
        "9de78ff391b36910011fe607de7281d5b0bec495bc66f6078a7c2446e8fb3755", "d9535263d5115619aeed587281c1e2d1"
    ),
    # FGT_90G-v7.4.6.M-build2726-FORTINET
    "159e1f927d8bfe2aeeeb94aaca62de657e0a5f9b686045bea1a173046c57a654": AesKey(
        "02ce438db6868022c45a1ff427edd69c71e92600930506ca7ecd60824630f997", "365c22871dad29f98a8ef4c7286f131b"
    ),
    # FGT_91E-v7.4.6.M-build2726-FORTINET
    "2a173b7570030bf1595135c32cac01988e5df9b2a85e30652b07748814e6b04d": AesKey(
        "5003dcafbbd5aaf9255d57a9e5005eaa4130f1a169aa6ceef1f98ec9a9ea5cfd", "1064b29415ee4887856165edb9b52066"
    ),
    # FGT_91G-v7.4.6.M-build2726-FORTINET
    "30c828f6304db8bddb2322f33839c4ef85d651b591a45bc46da9fd417bef82e8": AesKey(
        "02eb081001efa76285ae669bbeb14ea3b164dd83424b517e5a4b8cdfd114237c", "55101bcdef58efc1e851840b7e327c44"
    ),
    # FGT_ARM64_AWS-v7.4.6.M-build2726-FORTINET
    "802733bff54c4ee510e721124cdc368303a16900677bbf26913e37bee08de674": AesKey(
        "6460dd0a4a7f62c42d4c3a4e98dd5d316c613c236594d1db82fd5599dac3a731", "cc09881bc312dc89d64c1d76fa0fe8d0"
    ),
    # FGT_ARM64_AZURE-v7.4.6.M-build2726-FORTINET
    "5a489dc498e9927c77048ca5b70b6ac15656afd45ea272fae3ffeca5da9843a1": AesKey(
        "937dfbc0b3b94f1c92e02b1ceff67452306275ea614e3e48d4b8decc998bb1c6", "85de1a060ac6cf1db21c97c6b3d0dc2f"
    ),
    # FGT_ARM64_GCP-v7.4.6.M-build2726-FORTINET
    "711f5b708c8fbab620ce8d01bbd5a8c1c447670ed2f1bb02ec36103f36aa321a": AesKey(
        "b6c7bb2852480b1629f0071af02065429523770550c65ee784b64ef6f51a8517", "399430b4bdb3f5de7127b4fb5e50da75"
    ),
    # FGT_ARM64_KVM-v7.4.6.M-build2726-FORTINET
    "5be7f96b36ddd5302c92938de88f7224f3051997a8e7cb1979d6bc9a39de1d67": AesKey(
        "114a841aefe1b8a080f516e75401a5ac333f3fc7bc2138ac65fe8db5bdb23af2", "a613389d0c7d86b5413a2fefc747dc49"
    ),
    # FGT_ARM64_OCI-v7.4.6.M-build2726-FORTINET
    "4e99a3aad5adafd46191b1ae080b750de6573d61003c167be706b200cab74469": AesKey(
        "69db251e6e621b594bba2f7761afbb7f19f38f2d1b0c1e85dcd47698a667a620", "4427c0b9e658600bdfbc745b7ae8f89f"
    ),
    # FGT_VM64-v7.4.6.M-build2726-FORTINET
    "d86f523ba74a519fd4dd3f0cebb2f076f0be7565ced6fdc2a2bb6d621c7e2468": AesKey(
        "3b3796380127578b5447501babf2f47a39ea086a78164a97202c5d6163b58033", "6c0ab4aeb16c746dddefe41ac9856f04"
    ),
    # FGT_VM64_ALI-v7.4.6.M-build2726-FORTINET
    "a812181ea05eb830dfaa65a8d898d9fc36109a8d1b3c0de726ab359776b85e6b": AesKey(
        "5dc100e316f639588b9fdbf2baa21c1a1964cc6aafbeacded6d5cf95053683ef", "d40c99d7aee975afdc8fb174157af6d9"
    ),
    # FGT_VM64_AWS-v7.4.6.M-build2726-FORTINET
    "5aad06248360c042085405ddf674a0ff549c6e0b95c5aa4ff50ee990e96f8521": AesKey(
        "1aba3c44f46a840c3093db616c4d4eeb72ec21bc88933f9ca6f742f80bf55486", "05e3bf054f7a1af9b20cb60b9e62c7a0"
    ),
    # FGT_VM64_AZURE-v7.4.6.M-build2726-FORTINET
    "182aa4c60180126f95b3970976813c2521fe9d61e4d4e00ba214e563c21f9dba": AesKey(
        "dcd3c775010cd2a680dccaf2b547542d0e838b8c66b349e21749175ddd2b87bc", "f6fbb1eb470c105aa845bed96b87693f"
    ),
    # FGT_VM64_GCP-v7.4.6.M-build2726-FORTINET
    "32dbe66a5e0e9622dd05d293c33e8dc394e6ed716b90609be90caf2f6998d426": AesKey(
        "13068655316846099a008cffe4d242e3a55d1ca7aeb9918895cb4085669727e2", "0b2576b0039d3885ad5e18ce919e6b29"
    ),
    # FGT_VM64_HV-v7.4.6.M-build2726-FORTINET
    "f3e75717866b1b2da811a05c68f8b5e01e835356a0d71010ec9bda34fbf27011": AesKey(
        "070705cf5bc216484fbe663fa73ebe84c42bb92d4c864d11639f0b986cf0ca55", "3aa78e4661fad61437640e51b37feff6"
    ),
    # FGT_VM64_IBM-v7.4.6.M-build2726-FORTINET
    "1cf14a39e46ba7838c50387f8faddabcf741a4b8c151366f4fe49ab50bcd3fdb": AesKey(
        "cdcaaa14448985a94d40984f8fe448c41c35fc308f7fb6e8629631a31579885a", "fda516e1a06404354230a28d73e0db7e"
    ),
    # FGT_VM64_KVM-v7.4.6.M-build2726-FORTINET
    "a0883836542c7a3d4768379f4a9b3526543f97a3a0dd3239818b56ceb4ec8cf1": AesKey(
        "40d3c5d9c6f92e467a51067c96b026bbcb56e8dfe4c7032e59c9a64431053004", "bb2ebca7941f99fef6302dff649973e6"
    ),
    # FGT_VM64_OPC-v7.4.6.M-build2726-FORTINET
    "bd94fec1a7b9c399bedef12c52fdf0b2bcecd03dfc1cadad2fe0e148871b18c4": AesKey(
        "ecc69a25c3ed27bd8131b1be6caab0f9f11745bfde386646adafa638e9ca3107", "3ff1707f4086a5e8b8b8604a757dc484"
    ),
    # FGT_VM64_RAXONDEMAND-v7.4.6.M-build2726-FORTINET
    "f8a454e5ec73a27c613b5c51ab8ab8c8c8e8cc8763174cef493b906e12da5be8": AesKey(
        "499772080ea64a47d9dab27bcce9f6c13310e366096a6f494ce6796b394eba46", "083db454b8890b8ae9f9e842f4599b3a"
    ),
    # FGT_VM64_XEN-v7.4.6.M-build2726-FORTINET
    "881dc84e6a48c7d8d8431eb8cb148004c24c8c268ec1797f64b1449aa1e974e4": AesKey(
        "e75dc3600a68dc41d0bb668123d65428617d4d667cf26a9b7a13aab2f8f6b102", "80edf86ce7f29a2380fdf7adf4d95a02"
    ),
    # FWF_40F-v7.4.6.M-build2726-FORTINET
    "464a20572958cb1bb9b2e401b1b4295c08fe69cd8d38066f6932d793abc9fee5": AesKey(
        "beaeefa6bfb001060f2f6778a2a15b51724300e755ad2223ab40e4d0c563216c", "0145ecd0f8a3a56e30343fef580579a1"
    ),
    # FWF_40F_3G4G-v7.4.6.M-build2726-FORTINET
    "dfc943e4c80b43072eb762294292bb3afb4a6eeee22c44ea4f60f5272d35bd46": AesKey(
        "9e28265e29df81e6ab0b867591027340039a318e6be2d3db9497837bf862a462", "157f4cc9199383f2676d032e43d6cc42"
    ),
    # FWF_60E-v7.4.6.M-build2726-FORTINET
    "a9fe09c499f642ec76c65b868ac266d2dea832c256e5ef947e1bab5228d12804": AesKey(
        "a643cc4b4c1a29575a04ec51eb4b370704314be04de6be78bf1d134c4bdc89d1", "999681df5fd427f6d85c311c6c94652a"
    ),
    # FWF_60E_DSL-v7.4.6.M-build2726-FORTINET
    "410024331d360a289c99136f9db50596d32b0b39eeb664f54fc02d19509d3907": AesKey(
        "f8b172ac624312c4caea9d0f1a1889be249a714804655245576bec6bc5533ba4", "52230ec799f5cab1cb892ac0ed29d157"
    ),
    # FWF_60E_DSLJ-v7.4.6.M-build2726-FORTINET
    "7fd049c7f83034c0a2b67b16a231b171c8137d33b9a8890069cc6e43e4eb3a32": AesKey(
        "1e765ed51cf4c75f83228133596e53074fb807836022366cc9bf7f4307aa7cf2", "d0d31c19751872f825dad4f85d1b765b"
    ),
    # FWF_60F-v7.4.6.M-build2726-FORTINET
    "2a34fa0f2d6129b742994102dcacf601a97dfe153cbcace6b30b7fabc313f6bf": AesKey(
        "5ac0e68f0d84765fe1a054b5057bae2ebf5612051de7c5768dbdcfa08ec2a00c", "04e020d4af9b9c866d0024c58a1c0ae9"
    ),
    # FWF_61E-v7.4.6.M-build2726-FORTINET
    "2e82f055955152678ec5beb1f64694ee58d4246aadd438e1b68b722b31868439": AesKey(
        "8e56d421155666f30b94b83102bea23a81ae83582f1dab6b53682252d52f08f6", "4a4f9c38929ee2f9b2adb9dd2a212a70"
    ),
    # FWF_61F-v7.4.6.M-build2726-FORTINET
    "893a3568b561f0f779607d1c83faf4d2a18556d19555002f183f35e842e2bfb9": AesKey(
        "4e588df119368939c191961199abf23e6fea5e686d0c5e4c7f336287ec55312a", "5f8e6e8e41510a112eeee87a1da4281b"
    ),
    # FWF_80F_2R-v7.4.6.M-build2726-FORTINET
    "bf98c55e1a4b726fef911c0751a2c8c8c5f7f673d2592a46664f303578563ec2": AesKey(
        "049069da4039f9d8bf0be8a9bf468cdd7ac0d5cf04efd937a9c3e9248051ecba", "447fc92ad0808879cd038e97500118c7"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.4.6.M-build2726-FORTINET
    "cba593b3dd35242ce8c3ba85713c2d62779dd49862c7e7330272ac2ecfc1aa47": AesKey(
        "f3feee80e2bcedb1886f3c78e7f42efe1168d679329d081ebf9e46459b46e521", "cb9a03c962b3d3ceea3492fc6d0faa41"
    ),
    # FWF_81F_2R-v7.4.6.M-build2726-FORTINET
    "4834a31bdf559e9eb8dfd1fdddf74262235d37e3cdde8fd08b9cd4d5ea513ab9": AesKey(
        "b10ea3784679730360cd4e4bb165055b75eb3fa11c231324cea8bee627aff905", "785b00662d22d1f0938f5a57622c4921"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.4.6.M-build2726-FORTINET
    "667f781b90868773215b4c4b5ce89e5abd72e360b4871c01a1443bb1981a1594": AesKey(
        "19aacec26fe154977b8f34be4632b0d473fb327df4a630c282221785c22efd33", "be6869070e4981d8cfdaa3e230319b36"
    ),
    # FWF_81F_2R_3G4G_POE-v7.4.6.M-build2726-FORTINET
    "9482f54465493717c389d292c32fc3a034063f1f1429545fa1aff4c706b07906": AesKey(
        "6da05a8f06112965f57ccf7989f323685e1f8e861f44708a4a58c7c135d2f707", "7d70ebae87436868862646ca23917108"
    ),
    # FWF_81F_2R_POE-v7.4.6.M-build2726-FORTINET
    "a4b8d656918f8e244106bcf6962e1b87e9ebb93a7603a4dc22a41f3be9cb1df7": AesKey(
        "2f5ae50658ba0260d820ab4982bd9bbadabe2e7374f3dbaf5f297300f04a53ac", "137e60a5c40013cd29f0584890eed90c"
    ),
    # FFW_1801F-v7.6.0.F-build3401-FORTINET
    "cd372a8baf1141a9cfc727fd66add68759360cf8b69d3162a370b56da077f8e6": AesKey(
        "e71adb577a0ec68993c4077bbeee92e00b00bdbc2a62b963bb35f4d2abccc700", "b0f3a6bf1772c2f99e8d989175e32634"
    ),
    # FFW_2600F-v7.6.0.F-build3401-FORTINET
    "1fd756d33745fb38d507f6883c9c245f47504a03158fe6eaba295e65b208eb70": AesKey(
        "905c2f8289abdbe7419703eba620677e7f559d1d2123e95feb04033c732fc01a", "2f09e2d2ccf845fa2b44435023d0b7cc"
    ),
    # FFW_3001F-v7.6.0.F-build3401-FORTINET
    "a9da1e5a0c860f0240d2eb32c37e9b8bebfb9c591939daab55f56be9570fe08c": AesKey(
        "1d36b0007f70298ea2830c5fcbab0c0ac2419fc9220361f651281796443e6de6", "97f5b70ca7056aa5e177959c2b445242"
    ),
    # FFW_3501F-v7.6.0.F-build3401-FORTINET
    "8131a11211a8cfaa37e8bcafb1baf77d52a5d5e1c8f377664c0300d9aa01ddce": AesKey(
        "215e2d191e8af96143ae113d3457bd8f64fab2471aad1248f8e8ba0a783e5031", "b6d5c6bf24e1b2c283cf5ad9a8329d91"
    ),
    # FFW_3980E-v7.6.0.F-build3401-FORTINET
    "deb01cbe2752fc65623924136c74f3b9ef9deb3aef22b0dc1b8a3901901fa441": AesKey(
        "42a1b7e926c14054cf060400f0cd8ee47002301be7e961c0ef87178974bd011a", "f9c5659b076893fb81c09529c0ebc68f"
    ),
    # FFW_4200F-v7.6.0.F-build3401-FORTINET
    "5d819bf24458b95b946e5795e6fdf8d25f2fcbfed18d6d87a9a73fcdd4fd0713": AesKey(
        "6bac51762cf8d608885e911419c56a51159439b350807c72e987f01e7e528b7e", "4259c00cddee4cc9f33f61e7f9f9cffe"
    ),
    # FFW_4400F-v7.6.0.F-build3401-FORTINET
    "5a08846e3ead3b4fc7bb67dff362276d6ab4284e7cbfc8bd5d4cfb475bbc10e6": AesKey(
        "03153d8c89f6494e61d9eb053f2bb1cd04b7eb5e8ffd23eaee1c1123284ab236", "11b4cc51425d0625719b68a3d30b4d53"
    ),
    # FFW_4401F-v7.6.0.F-build3401-FORTINET
    "4a069ef875cb6dc8b1a3c6293f1ac698dab0951db663cc99dd623498f5d1c487": AesKey(
        "71c5a2ebe0dfdcd22c575ff3139afbf46d2427d9b9d87f52e19070b609560140", "c4410b258d8e71ee78153c5bea1b4862"
    ),
    # FFW_4801F-v7.6.0.F-build3401-FORTINET
    "4c39a3896c86d6b5e655cb02fa38897648c59966a9847d8494ce16fd8c7a9cb4": AesKey(
        "6b583e9cd74877ee325ebbb5614206496c240e7f1d2bb52e2890ac789c7e17fa", "9d4a29b46a629c4d78d50a8a2b4259c4"
    ),
    # FFW_VM64-v7.6.0.F-build3401-FORTINET
    "eb2631855bcf4f478c9f6dc887922efbd19b36f32760d505abf5a3333bcec834": AesKey(
        "3b4903d1e99c34d2624069ba245b873b6d69f1999b59a79115c78d6db5669fd0", "eb2e1e19e5b12ef211f1d5b950d2de9b"
    ),
    # FFW_VM64_KVM-v7.6.0.F-build3401-FORTINET
    "96c6c7a6ebdc740a28ad9e78c28f9b8e8a042e9daf849e7b2dd1ecd09bd6ba63": AesKey(
        "2e8aa9ba583ed308d21f7c420a2a31c556134b0891805fced3a77ca0f6806d9e", "a3f4f7f0ff99234381d017e4289e2384"
    ),
    # FGR_60F-v7.6.0.F-build3401-FORTINET
    "61d02b9cc069e861563a3324e75fbf0fcb3f6c256708327f48b79f5078bdcd91": AesKey(
        "2429ad173752efd0f1f7d0bda05b1d0a9c867a8e4d1a1dcb80bcc01f9801ab61", "79a0c315c2b629499ac717cce93b43ab"
    ),
    # FGR_60F_3G4G-v7.6.0.F-build3401-FORTINET
    "5146f762575f9faca7305db76304650845a95cf8b3ae5d2adff88d86d52f5c88": AesKey(
        "e44fb7b8e310571a48be34d66665f8016ebb3e07e3444ff776efc14d6c9645cb", "fbfc2d0d19452592e8a888fc87e32d02"
    ),
    # FGR_70F-v7.6.0.F-build3401-FORTINET
    "04c998b5910d668afba8d18c63ce96fb830a446beb4c54a4ce2e51e871cd3189": AesKey(
        "7fa07266bff08b3bf05dacc6519fd91c784c5e6c01abe54d8144c23a3d870fed", "45255cb428f1fc5f29fc04d4237736e9"
    ),
    # FGR_70F_3G4G-v7.6.0.F-build3401-FORTINET
    "3005d90a100083d6edf9491b52ca930561d7fec933ae46a46965f04725948bc4": AesKey(
        "bfd1904eaba459e192699fe7206e39a7d0dd5e59c4b4f57f5da9a3c041b1f0d5", "404891e0549ba88c476e599dd43dd180"
    ),
    # FGT_1000D-v7.6.0.F-build3401-FORTINET
    "ccbbbe6bc02944f30b241b36fe9956189a31ce08f6368683b0b28a464732ea4a": AesKey(
        "8842c10cc887e045ad8e38c50b9a5e1c7680febbd828e2714b9b428ecaea28fc", "42aa3cb195f23a57e2ef73d2d9355bf7"
    ),
    # FGT_1000F-v7.6.0.F-build3401-FORTINET
    "2ba2579652df1c5da429a3976b276d484f0cbf6615c11de4a5946e96dc07c7d9": AesKey(
        "af5f4747ea990d187d9a1d73c4db6be2915a35914f1c6f46cb9f426872f330fd", "217cfc5aac3a9fb135445715766eb5cb"
    ),
    # FGT_1001F-v7.6.0.F-build3401-FORTINET
    "567a724f81ce02b76458c18327122ce50412634ec3d7f0b03e9822ce1e6c2227": AesKey(
        "cf341d7ca1772829ffce63317869335a08426e4215e02073f8eaa6320597d54f", "933e32f3d831489deb9b676a033e5af8"
    ),
    # FGT_100F-v7.6.0.F-build3401-FORTINET
    "09a83938967671ea8bf05829ac9c5cc11c4a87e010ed47217921e7e16c4963fb": AesKey(
        "e8c34ddd95ece07436f9ef0e8f97ec1d9cbc6ca54b831f4561858875ba243fb7", "80600999407452ea95b421675a323f81"
    ),
    # FGT_101F-v7.6.0.F-build3401-FORTINET
    "fa7a61b18f578d2caf5b727605d84823669251ffdff3284e4bae9b319072056b": AesKey(
        "3f0e4790a65634c08674d7f1faef40a61c773d8e8a0d5ff4586aa3d2ec256787", "42eb804cc89f10ac0a386fded10b4aad"
    ),
    # FGT_1100E-v7.6.0.F-build3401-FORTINET
    "1022eb837c41f69027c8665a92119c3a7591f5e5d0e3137204bcc37ba6914230": AesKey(
        "82babf3e3205bb7511b8f4d7625ed8cf8c9cb686966d5691a9621b5e84946276", "49ab8c5e82396052e1182e98cc1df561"
    ),
    # FGT_1101E-v7.6.0.F-build3401-FORTINET
    "659e0e57251b89cae2d5920c7abd1716e340c8ea7112fbf1069d1a87524dfe4b": AesKey(
        "e104e92bfb0cea6ee0b10caabebd3386680f343c4669ad73e064ee0566958f02", "31396d3e46e7ee3380e11e2077e7a42b"
    ),
    # FGT_1800F-v7.6.0.F-build3401-FORTINET
    "a40bbda9466af220a5aa47e0c39771ef372e0762e14c22a271efe87f68a3a4bf": AesKey(
        "040252d89e52ad2786bedbf01cc5d83de326f6b65eda86824a2b55d9d73a59a8", "4503c9c302fdc97a2b784011f5f45d08"
    ),
    # FGT_1801F-v7.6.0.F-build3401-FORTINET
    "b65a80151d37bd3a2cb7653b954bbf6f7f309c0b7c56463304662ad48dbdbaa2": AesKey(
        "c9e3ee41fc24cc637034538fffb8ca469a8d1014ff920c94ea0da8ea2cf632d6", "c5abc57efc6b97f47c45bc5d6fe7d32e"
    ),
    # FGT_2000E-v7.6.0.F-build3401-FORTINET
    "595cde83c7fd218cb238bc5c94b16377b1df25433ce27520d1c11a262a348f5d": AesKey(
        "a945f3c93705340011cbc1331ecaf8cd9e53377b93dfe92a95cfe36e17999f98", "12c207947157a2c7b65f43a48837369d"
    ),
    # FGT_200E-v7.6.0.F-build3401-FORTINET
    "0d2149f0136a051f22e0142cbf87994e6587d7ea56ade11d29a53ebfa886628f": AesKey(
        "a10c7a06297c299b261b4883b5178fc4e7c0077294efdeb9e557689ed5ed868c", "c7bcb2971f6d03a2424798fc788be4e5"
    ),
    # FGT_200F-v7.6.0.F-build3401-FORTINET
    "da530de8fc77cdaefe92e163aa0e3fc2e720f51b29e035f3fc82321effb03761": AesKey(
        "faecc8b912d39cd2ddc23b59f3b7daa5ee97d77e58ef0b67f300ddc24d69717d", "04787b9f391a5dc66398b86213d3c8cc"
    ),
    # FGT_201E-v7.6.0.F-build3401-FORTINET
    "84ee586edde4b07e80804c89147a3c0da22d46cabf5198584c2eb6dd5fef149d": AesKey(
        "7bcb6cabbd11351dbae145e69e1faae8005e0986cfab2776ccff434496269eb1", "a387bd1109ec15955730771377bcdc85"
    ),
    # FGT_201F-v7.6.0.F-build3401-FORTINET
    "6f653fe8d40b0361b9580e3f55889ecc070ad9c1a98e3b95e33ffc55ffb053cb": AesKey(
        "e573fc0243253f35ffc764d9794a4523860dcb7ccb0c388f2652ca4c3590243b", "402eb35f780bd1e585d487d4e61af804"
    ),
    # FGT_2200E-v7.6.0.F-build3401-FORTINET
    "e736bc034354b72b4ee4c7042a403abce2c3338df2eb513812ea004dbc1e9305": AesKey(
        "24d4ef66f3ba227c3bbf5372b098f1d9ea65c24a63d1bc43e69613ce27ec3435", "9b068d6bf7e565d9406c54607401905f"
    ),
    # FGT_2201E-v7.6.0.F-build3401-FORTINET
    "179e69f53a4be7dc8c00cf99f54ad3ba80f9fa74363539fa3faa69fd0bb9c829": AesKey(
        "4386981be68d9aa1ce9c75a449327344c16159774dca97e3a83562bd3f3a47d6", "65b599802c8cc3c958bac34ab42bc329"
    ),
    # FGT_2500E-v7.6.0.F-build3401-FORTINET
    "19b33e16b8164d0b72814e6ee6ff2f2b5f3618b194aa6c0f50eeb6cb90b0a213": AesKey(
        "e4f0654775711fae19bed7c8984d4fac0c446ce985caa9a4c8e3d8c58844ee9c", "6c63a38ec5162bdd217bc8e9f6e8ba94"
    ),
    # FGT_2600F-v7.6.0.F-build3401-FORTINET
    "70e07c0bde30327b1faf56ea6e40ad65c46495c8fe2382017c1e6c7a9fcbf4c1": AesKey(
        "bbfae12511df0de93948a3381e15370aedb27f3e771235f2d1b1f69800702e8b", "e4632657fb0ef44489623e71c413f133"
    ),
    # FGT_2601F-v7.6.0.F-build3401-FORTINET
    "cae071bef893a8d5aeee25915e948ab0ceb5a5a290801f2a169a7a2b35cb82c3": AesKey(
        "7fd4097f2f1b078738b4a9ddcf023bd3752202a485f5eea8fc3091cb6333da22", "b87a478dc428d55e2702eb28c5913eca"
    ),
    # FGT_3000D-v7.6.0.F-build3401-FORTINET
    "9d974fc5753e8d22388be42cdadc4a0dabcd57774706f3755a95db14213f8e74": AesKey(
        "7e205209bbd7cc7bab7709f6690e5c9e40fe418b6f0cbbb6e66aa671a109c3a6", "9347ba8e17b11f374970f595fa281fdd"
    ),
    # FGT_3000F-v7.6.0.F-build3401-FORTINET
    "2364670ed29f8e16b7c2f7156e0306de3a6fb432f2551d0c40c2645970da9516": AesKey(
        "b73ebb68a35a4e8f22f8aa205fd1e0e0f73060556064e84bae43ebcc95b60cc6", "f9387ba72d76ec28bb31cd44dec694f2"
    ),
    # FGT_3001F-v7.6.0.F-build3401-FORTINET
    "8b3c49cf66d48763ee64bf8335f7841431000fbd843a85ba62fb978e914bdb6f": AesKey(
        "f7c2f07cb6b306c9473aeffe8b73d376784ecbc9c73de3e02b8e3d0817fa6138", "3f05d2c73c5cde6e7091e7d3ad5c79cf"
    ),
    # FGT_300E-v7.6.0.F-build3401-FORTINET
    "4218333b0a44cd3389f363cb52b559cd1e6fe8f6b23ffe66418c96578114172e": AesKey(
        "b3f0cf8246a6d5b56dbd6eefb6eb69bb08ffb373fc90e802ebbdbc82909d6037", "090e576b11c18f75a905312eb135312f"
    ),
    # FGT_301E-v7.6.0.F-build3401-FORTINET
    "7ce053013cb615e201995011d6c599ea301f7384df7e479402cd5c1538023685": AesKey(
        "061ab1355ae900a39a577a431195bf33dedf3b02bf0ec3837756c7253d13fd0e", "71e77976c9a17aac1ca6510254e52227"
    ),
    # FGT_3100D-v7.6.0.F-build3401-FORTINET
    "51b1169815f09c4b4ab83655272baa08d106fc59a96002ae248548697b0134e9": AesKey(
        "a0cb854fde26e6e3f785ed22dc03430a91e2f4b10daf31c61c9ea3b51f010fb4", "b9176186b15b55acbe1769a6e961452a"
    ),
    # FGT_3200D-v7.6.0.F-build3401-FORTINET
    "97364e363d343a1f77541f3330948544b45aec5dcb28b956c6715abe96f8c703": AesKey(
        "43ab72a9f7b6450a70a855be7184ec2904f5b14f97db0376eba12444658ad274", "1b468885441b8a71c8d9757238785efe"
    ),
    # FGT_3200F-v7.6.0.F-build3401-FORTINET
    "c310ecc5a42cbe7ec84a32d0e12d944a0b0be18724bfd60462f855733ae6c0c7": AesKey(
        "ec036d85eb43509fa10c0085e4ad0b4c247e5f170be096d4406a12efae283609", "ac0176adf4d7fc903366e2e96679837d"
    ),
    # FGT_3201F-v7.6.0.F-build3401-FORTINET
    "85c4a13912d62e7d07e59fa22259dd0df872fa564adb907206ca43c0a4c5360d": AesKey(
        "c54e8f73b44cb3c4f9f5463fbec47ecb92cfafd57a3143f28242e7f2c7172182", "1b50d35b0eb8956959ea14bf5d1efb68"
    ),
    # FGT_3300E-v7.6.0.F-build3401-FORTINET
    "fb1b1b64d18eb92ddffc02e431bbfe6ac16ca2746d60eeb57a95bc3461c3b590": AesKey(
        "9a026cbf9a2fa4b5598d0e4d9f386c806656390a0e8b427bf8d30f0cab541106", "c12bfa3479f048fbb85d3e775bc048c1"
    ),
    # FGT_3301E-v7.6.0.F-build3401-FORTINET
    "8853203f9648c18abf015a841b719545aa2ca8a124ab494eeec6bb5e6bbd0661": AesKey(
        "48aca7ec7bfc7016479d8bdce4123707927386ad4604d080c7b81afdf6cfc38d", "67fac94f895d78249c1f925d173fec62"
    ),
    # FGT_3400E-v7.6.0.F-build3401-FORTINET
    "a843885d974d71632abcc21e4d496bd1f5351ee86ad4bc3366d9446c52b4bcf5": AesKey(
        "36aaba91071c4cd601e8c219b310e5a27e6f5a201332c6e105e65d3b0da85700", "894fb5e2325b6192912e458da5132300"
    ),
    # FGT_3401E-v7.6.0.F-build3401-FORTINET
    "8719b63b225d3cef3e615ce3de41c87182292a115c7104880f877aa81dc4f1cc": AesKey(
        "6ebb0a00f39ab7407348d4fdaf64a12550bf84691bdd121f361b213cd31778d8", "10baa99b403185fb7e65ed702209443e"
    ),
    # FGT_3500F-v7.6.0.F-build3401-FORTINET
    "95b2e8ce1f32665cc44a1e77f25c26b815c481394e3eeb5b3bc593739aac5db9": AesKey(
        "41fee16a472ae230d67910e2bd3cd321e26a901f86ae2cba7b6d4774660938cf", "49067b7bc76e02269364f971f13a4d0b"
    ),
    # FGT_3501F-v7.6.0.F-build3401-FORTINET
    "18542845192a93e8af84ac6a7dc6fd1f3ac6704f77f0588f27031031df6ef5ea": AesKey(
        "e2ab8383bce64bc2542c29340dee610f22f8888a776837fcf42f7a2fc8f5a820", "e13fc22275c90c0fbb420aa09b4f5a1b"
    ),
    # FGT_3600E-v7.6.0.F-build3401-FORTINET
    "a1a9cfad71738f05be621d10f155ab63f12052bf145f15a585dbca3bafbd5493": AesKey(
        "7cc0062943dd6f0de201a9832ca6b5a5e14f212918de8135837bef94f370363f", "3beba0f9ee028e6d6b95b1e4eaa57819"
    ),
    # FGT_3601E-v7.6.0.F-build3401-FORTINET
    "9e8a269338f03cdd33d7d866f1880d8234ae591594016e45ee1c7b9014cc9bb7": AesKey(
        "59a977e094a62256195ae7a231ee8659ed8f51d60cebb5a58709ca7e5f76de6c", "7ede7f54c4915a97aaabfec56423a2c6"
    ),
    # FGT_3700D-v7.6.0.F-build3401-FORTINET
    "64e16cb799b91a7dd7d6ade14776713fb075d22d10aebd874dd9a2d8183d87f3": AesKey(
        "8c302d9c7da27e5a9e59144380cdc06ad66e9ad81dbf397521523e8ac5c45ae3", "994d1ca4abbd364aa187f6731458b954"
    ),
    # FGT_3700F-v7.6.0.F-build3401-FORTINET
    "cb40af01be3b2360b084d452fc2b96998d33c7c0b1aa188383aafc8494fef1a9": AesKey(
        "977ea41d469d84fa3aeea3730a4f60a0f82431d609764dd4e8dc4513432c6fcb", "823f3cf7a0153e2de77077272cadbf3f"
    ),
    # FGT_3701F-v7.6.0.F-build3401-FORTINET
    "3f2ce9e18622d8054ff8ed3ba32ddc7282b847690cf935e102eaeefbbcd53fe0": AesKey(
        "61d9b4b69f4974abf6846415506cf4a76ce7242de68e69e78e28961a04d1a40a", "555aa629b26fcf841b0f855da7205f7c"
    ),
    # FGT_3960E-v7.6.0.F-build3401-FORTINET
    "4350c03e9f3ae8647052ed143eb65694de685d14e0e2e9dbc9150181e9707371": AesKey(
        "cd0be2a2b1e94fea1b8c100ee8c37be7e5e083b98b06dc73ed2571213ccb1b44", "c48a17e33598ad2c3b009abfa27f04ec"
    ),
    # FGT_3980E-v7.6.0.F-build3401-FORTINET
    "efe594458970e24361e62ea0b1c34aa727045da11c395ff979611695031693f8": AesKey(
        "d1c4a1ccfc04cf7904264719fbe7b521fb3d4f067d2957cf4dce0b8b2b899664", "5996677fd6be55312a71c8f77b7423f1"
    ),
    # FGT_400E-v7.6.0.F-build3401-FORTINET
    "adb5c63a0d00354558d9d4b7be8f7654d73fd9362f4cd3539bfd0ab58942c5f3": AesKey(
        "3f4dc497ac1d2d8a7f46a9d91b6adf5ab97b62d8cb4bb4ba734f0117a2574191", "5e3423e88243ec4310f48f2813284746"
    ),
    # FGT_400E_BP-v7.6.0.F-build3401-FORTINET
    "f4dff869c5589df8f0c087f5f19566f0b38ad4e2b13876ec57f6abee28f00a38": AesKey(
        "181055baa3fa97d876d2c4af83a1fd3506b9dcb4b16b207f6271e11d4bee1b20", "fdfe082445b11ee23b1f88556fc764f2"
    ),
    # FGT_400F-v7.6.0.F-build3401-FORTINET
    "158820ad385629449bccc17e53876b06defdb11f50ddb3144516eb84267c5b39": AesKey(
        "07781d342b7f0394daf7752219b7c4d58277b5bfef4cef72c5482069e9f48063", "06f4c02b09bf5056b74713ee3f444fd9"
    ),
    # FGT_401E-v7.6.0.F-build3401-FORTINET
    "88fc20f24ae9b8179206b300a8b8d730a5c8609f57f8ee3a999364c3cd433d84": AesKey(
        "f8eb2569447db30a2d9e6b274cc3cb46c41e189e71aeb101affa70a8e935bede", "b29f7569455fe3a0f8996a8c8c724237"
    ),
    # FGT_401F-v7.6.0.F-build3401-FORTINET
    "fb70abb61ace778b1f04cee46863173a15cd1ecacd7494e5afce996541e35113": AesKey(
        "3a2aa85d89da523024e82e5e679a610c50664d0a40d07cc1bd0b9efefe468714", "a6c3906023e9c346f497ce4b65b624a6"
    ),
    # FGT_40F-v7.6.0.F-build3401-FORTINET
    "a329559c601196e9e4b49cc6abef131070eef13b8a1685bf5cbcecec7656455d": AesKey(
        "6e30d609dff4971fdfcbedc8a1916c12c3b259cd57de105e1b30910e72d2a10d", "9d733b76388ae9f88dedd9d62746415f"
    ),
    # FGT_40F_3G4G-v7.6.0.F-build3401-FORTINET
    "004965f158bd85f9f168075b7cd7b850345e6bd454590e920b6ec8043c56c93f": AesKey(
        "c137e2ec9e5d67b521b38b88488bb46a377f967faef2b858ad0b82ead87637f1", "239abe7f2ef92e7652b65a10ff86ecde"
    ),
    # FGT_4200F-v7.6.0.F-build3401-FORTINET
    "bf88f5b9705fe856f3e5ec31f0f03259f18094ff102f9d305e230086eeb68e62": AesKey(
        "61092c1d6db1c26c478276b2897e9a8abd12d8c455ab72ee01c4cd826427ff2e", "47f13371ba91ceffb99264dcab668fdd"
    ),
    # FGT_4201F-v7.6.0.F-build3401-FORTINET
    "a6e1a24fc7c026a024cbcce7e9f5236e6e7d0018d34acb07f18bdfa15775f55a": AesKey(
        "a72d3bde628a07c3176463e953a0a9a0e70e6be69999a91a3c909fc3c425705d", "2c06675cbbaadeccd746fa5762a0de81"
    ),
    # FGT_4400F-v7.6.0.F-build3401-FORTINET
    "a6fff328a1a626973f2a4961b6525fee6e964f8a6fb16fe2302b1223ffe2a71a": AesKey(
        "fc75d599b2b30919e5a6063bd62e72464d2d9b71db0792da78fb40c515234fa7", "076f57127441a53722af22df0a26d62b"
    ),
    # FGT_4401F-v7.6.0.F-build3401-FORTINET
    "e394518c2b8601152dc63d8f040e0b21d538669143d34514a220602a25f46896": AesKey(
        "08d5694d810443c47367e40a21e7eeafc53d35d1877cf389fc58f6802e58ebac", "1aacecd9cec48b96594f03d841e2f584"
    ),
    # FGT_4800F-v7.6.0.F-build3401-FORTINET
    "5a119ec009c316cdc823b9ff22f98e25dbbad6a757014fac7dead04aed47b07c": AesKey(
        "5d3083078ba07773b26f0d20cbdf03c7a92aa9f97fa0d8b90c99285bec487764", "d425b54693f98cf9362f4bdb3fc548af"
    ),
    # FGT_4801F-v7.6.0.F-build3401-FORTINET
    "90c8ea1f38da7f41886be5b2b21c7e1abef418f4eabc10510e247f6bac8bfc8c": AesKey(
        "dead576ba0999d6d2fa58e98dbf219e6e5cc06b4b94e28c4de8538ff9045b369", "8af9208649b13d18fa454e64338eeacc"
    ),
    # FGT_5001E-v7.6.0.F-build3401-FORTINET
    "22f27d2dcc4ce99377527331b5ea00a7ea06dc8bf0ef05fbea2aec450653a8a0": AesKey(
        "f4c54708e03edda15fe374d05eea77fedda02a99b116a58dea1fc744e411bcbf", "f0a5e2660d5f970ebd1918c0b2ca7ab1"
    ),
    # FGT_5001E1-v7.6.0.F-build3401-FORTINET
    "47f9a1f0a90eb29ef399bce5d924cffa47a1d3e85674056832e11588f5d0ad7a": AesKey(
        "3e615b63adfd6a71d1d25399b040aaddcce8605d73a98ad5f95bda3cb397fd72", "7ec2e0a95e53be38404476265eb4527c"
    ),
    # FGT_500E-v7.6.0.F-build3401-FORTINET
    "fff34f6c2019508d0b66ebbe66c12c662ce67be186dd03ac94fa47c22add69f7": AesKey(
        "f14b943aa4b9de912f4801d8ed585e7e48e7350d59b94bae36d2e64c054840c7", "e0bfc5fb28128ce665fe8bcd0a6f8dac"
    ),
    # FGT_501E-v7.6.0.F-build3401-FORTINET
    "e9e740f61ae9707db004cf60d04e5d379e8e4a58f25ae21576befc3d455c3a1f": AesKey(
        "eda9e8ba94ccdd739c9af3160eb720787f1a341eaf7d27d24ee560fbf7a28b6f", "3c420a5907f71d6daffaa36abc41b8ba"
    ),
    # FGT_6000F-v7.6.0.F-build3401-FORTINET
    "7c5ca1bf9f2135efcdbb5f0999157c94b91e54a33e116c7d3981d5c462eb53fe": AesKey(
        "51057347d2a8475237bd0f30c90a3b6784dc6a62d96fee719283ee7aaf4375bb", "fc41eb453f13d91a1dcbe7785724e857"
    ),
    # FGT_600E-v7.6.0.F-build3401-FORTINET
    "af8e674080218b51b82dc30430bf2d20dcdf6d0c7fa101b11b20db2b1bb408d5": AesKey(
        "46e422dd1b440878df1cecf84bf61ecd1e1b6e818714b846ab7393e7a3f4849a", "89a5bc6a78148517c78065080b09f52a"
    ),
    # FGT_600F-v7.6.0.F-build3401-FORTINET
    "6866a46d6cb9cec644c1ccb3418ebbbdc08a5c3abb71035c1cd6e6e07cb22fef": AesKey(
        "77719f6f9e1373041899a9c01ab55f9327ad9328b07e647ccd0bbbbbcc5ec1a3", "f09d1385861d8fd9f3adf718ea77035a"
    ),
    # FGT_601E-v7.6.0.F-build3401-FORTINET
    "0222b3e904efc77bfe965383aedbc362e98544d2279d0af19c1e602b1c7a7cce": AesKey(
        "8c72135b29c49ce433eebc0aa37e2f5a6c292cd29b3f1c579fab21fb2de83391", "acff02d554cf50bce47612433f4edfd8"
    ),
    # FGT_601F-v7.6.0.F-build3401-FORTINET
    "ccc8d6cf929bce146852ff446ec422f7d978101be2191327ee0a3ddbd8ebe97c": AesKey(
        "b91f8bb65c62481adc87c74c8700ee7b3fc32f2d53dd5e1e9e80b769d192a698", "ded89d390180bd298c7b2f02fb32f90a"
    ),
    # FGT_60F-v7.6.0.F-build3401-FORTINET
    "f0e7c67cb8e377cd41dd08b6aa46163adc2290a9ac044dad058f4d719b655a4f": AesKey(
        "db91c5ac281a3518886e8c597d462f66d428489b623493d37bd6e3e0b5892cc2", "d5adf4f4e10e25aa2653c3264330f3ef"
    ),
    # FGT_61F-v7.6.0.F-build3401-FORTINET
    "fd9d622ddb95aab5f808f7849d7067784c42430f32c92c22e6a01551efbabbd9": AesKey(
        "dac5cd1ee85dd619d2a9a3600bdd81d672f8514dbfc86c9193d020a57d6be31b", "92d319cf82121e6026cffae3f1538321"
    ),
    # FGT_7000E-v7.6.0.F-build3401-FORTINET
    "98142159dfd5caf94066506f2844f8fe2f643a54a601aaf869566d5b3ab10604": AesKey(
        "e43fdef0ab9bf2dd63cf7ecd42254a3df0f5135bc21e82ec1d43179704ef905d", "c051205e688afdf2ba225213c08020c5"
    ),
    # FGT_7000F-v7.6.0.F-build3401-FORTINET
    "8157180a5518ec28c299b4636bbe0a865d4413a89d272120cf42be43ec3d293d": AesKey(
        "eb18df93371f93cc7ea468a7c569c73060b348255de944db2e81f948fd58457d", "557bcc4fd7545d6286b0e1a58efdc6f0"
    ),
    # FGT_70F-v7.6.0.F-build3401-FORTINET
    "f7be4bca51210900c5ee89c52588b97f53b2edc23ad626fe62c3e60ddf049f50": AesKey(
        "44a068f4c78b96b0c072fc1b88d1e4784e7d3f5d89c26611e4f66efd70d33b82", "9af5c7ea90efd941fd4154d8ed8adb81"
    ),
    # FGT_71F-v7.6.0.F-build3401-FORTINET
    "f13e8e5134981993f68478adf13c48a2a4c76e26e8b78b10f8322bed9c762f8f": AesKey(
        "137790e5f06f3834350610d761e433786182981c673853246f88a73f00e897f2", "c1e85ae9067e53c27176e68bdfbaa89c"
    ),
    # FGT_800D-v7.6.0.F-build3401-FORTINET
    "e453df8084f9680f41211d973d92c99f01751ed6fe23b6016a2e1ba8c2074697": AesKey(
        "fad2f3228c66e437177bd792f0c9827d29ef7745351041d51e28765f7d2598da", "a792e8dc3a3ace914bc0840f4f47b668"
    ),
    # FGT_80F-v7.6.0.F-build3401-FORTINET
    "d894f2959920627e79e660f7d351206491bf4a28c0e8a6b4c1d3b7de42325148": AesKey(
        "12a69c6555a835dfd1555bd324bb62faf6723d414f4687357b3f6905cccf544b", "9ba8da2e493c7516a650b46e39591f2e"
    ),
    # FGT_80F_BP-v7.6.0.F-build3401-FORTINET
    "3b16a38c2faf7f768e3dca407d481c4f260eb8452a8ddfb5b0ddbbc986956862": AesKey(
        "a6868917e498f03f40e423c4c677a35aa1bea5a2ea067e153a30cf3ea393f07c", "8dffb2fbe8d2e82df7c9311ed1dcc2db"
    ),
    # FGT_80F_DSL-v7.6.0.F-build3401-FORTINET
    "e6907b3e238a2b60f0002f043e120b14693e1ad9ce6a915c5e59297cb64d00f7": AesKey(
        "63a54d695325e6a063a0bfdb25d9ca757a0a83e9f404a660d9a4969cd5b2e86a", "10119b1c1dc6a07b4a6e90832eb1cb81"
    ),
    # FGT_80F_POE-v7.6.0.F-build3401-FORTINET
    "64be7042034959501997d41a52389557729d6d9bce4c6615aa087b0fe2b71df8": AesKey(
        "70533cf3166b2da81b382e38a7ace80ec1cf180df693fdc40a688ab91e1fdc0d", "bf5676aacf6888c29967f7b1622a2ae7"
    ),
    # FGT_81F-v7.6.0.F-build3401-FORTINET
    "ec47111a4e2cd0aba76f4fac7473282165e41ff1d6e78e30b92102cf6c6ca3e3": AesKey(
        "55b6fcd359337ba11bbc6d6103cada97ed7b732f59b274206e5320ae1baaafb1", "a9c182823e7f9c51cce1c2cfeba93f9f"
    ),
    # FGT_81F_POE-v7.6.0.F-build3401-FORTINET
    "c4fff4bd3db395f0596821beae23ee481a1916ecec006748fda331cbb68eac2f": AesKey(
        "46247f15e0d62e751bbf3bba6e59b1f2996fc7e794cde9788c12b4a16922ca7a", "7ee1df4da674c43075600e0e6499e55c"
    ),
    # FGT_900D-v7.6.0.F-build3401-FORTINET
    "5bcd6a6b0d83d0195c775c49aa0628ee7f4db375abfddf0d930146c5bc7204c8": AesKey(
        "8cc50439236cb2998c0543c6553e75692b800ec501ea6bfa1ea176b285a289f7", "e1e2da2655d32544e9d325007c835eff"
    ),
    # FGT_900G-v7.6.0.F-build3401-FORTINET
    "2ca9802f6719c4a8d30d9a8f2a351d3a2889e87c714cf4b307ce0486b67d72b6": AesKey(
        "f78f5ba5f001b8f7773d4d706e284b0b71f411f4d86664767e430adfa9227dec", "47421123b843712eb942c05068b40a3c"
    ),
    # FGT_901G-v7.6.0.F-build3401-FORTINET
    "c4c4d86b0d68624641dfba9a311c0098b6dff0b8fce25529bad9d23fbe944af8": AesKey(
        "13b4370495bc78048dcae58120cb7100ac77974c8c7acacfa4b26d5d42d2d465", "043e7c992934e87a33751d73a8df3bf6"
    ),
    # FGT_ARM64_AWS-v7.6.0.F-build3401-FORTINET
    "2ae439f8525aaaa4b674749b23823b226a918f280b6f915ab55c46bef1b67c96": AesKey(
        "ca4960089c6952a23c406adb1b7bb3598b20ba9c030a3a200415d7b17f1bca25", "8e58bdc0d78fabc64af4ecdf222b92e8"
    ),
    # FGT_ARM64_AZURE-v7.6.0.F-build3401-FORTINET
    "7d28aa9608e987b4356cda93629c7386981ad7efcbd54bc05e5cf86030894c0c": AesKey(
        "2263dd5355d77a67d258e9ce92d678a1092d4f3e1d8440d9d3865ab683fc7e4d", "fc7ec1b42111d73d6096510bbedf840b"
    ),
    # FGT_ARM64_GCP-v7.6.0.F-build3401-FORTINET
    "30107ce4b895b1de56aa9485f38bb81b8d8ad406de9f1ee286206466b8f62029": AesKey(
        "b5ce42a8f567ca9512d59a1666235735c121824dabf58b108ccdecb9883963cf", "57d5623d7d70b57bb4ec00bd9bbcf10a"
    ),
    # FGT_ARM64_KVM-v7.6.0.F-build3401-FORTINET
    "73154683195e09f4d77f6ff0a2f8a847d71624ea43981fc288432967af005665": AesKey(
        "a7d979f816c4e950f185f288086edb1586d7d41bd919c91ef764da15a2063275", "83aedf397dd9f96ddab55bdd515ca98b"
    ),
    # FGT_ARM64_OCI-v7.6.0.F-build3401-FORTINET
    "eaab212c0dbf64aff9ef8b962e338479142f9d65d9bc0a538a777081a46e0b86": AesKey(
        "ad100888ab89f9f6f134bbedbb62081690e55ddf6c983b097479f60dbebcc11a", "fd9cb938b50c1941f27836754ce50de2"
    ),
    # FGT_VM64-v7.6.0.F-build3401-FORTINET
    "1d3807cfd41d3ecfa6a894daf30c54f0d291eb5cb5ed170e4a733066bbbc55cb": AesKey(
        "60c5417f82d9baf62c08f24703220b173d446e3a01980a01e43b1d4857d3db28", "a20f7cb69318eb6ad4e57f1382a10669"
    ),
    # FGT_VM64_ALI-v7.6.0.F-build3401-FORTINET
    "ca0e9c10fcbb2bdd35432d54f110a4672eaa9e2ed4d707e87ed9b767c13d4ffb": AesKey(
        "932e8c5275657cb6c063284563694a7f51c1d5cf5b5dc371da91aba4ac03ea2a", "dba02f27cdef3f980885dd7e8d9eb2c3"
    ),
    # FGT_VM64_AWS-v7.6.0.F-build3401-FORTINET
    "e3e3b58ee03d326b3013884bc3b0e9b796e841f77b8f77015816bd54c22d9517": AesKey(
        "11d63db7c50e810ac15f6411d3c186896235cd434bac192c5c00d20048305aea", "859bf46905760490f5ccbf283cbe40a3"
    ),
    # FGT_VM64_AZURE-v7.6.0.F-build3401-FORTINET
    "5a4c18b9118124049955caa29824188d92fc51741f9576f90ea7a9df082b7657": AesKey(
        "8ff23905bd11c2ddbbc8a52b725d964bb8fe65613ac3a1e0c2907fc9d94f2613", "36d443825e7a653969e6b817c5f81d3a"
    ),
    # FGT_VM64_GCP-v7.6.0.F-build3401-FORTINET
    "dab1b3328113f2a373d288c8d311ccb8ed19a564d6b753f206c5a171eec1d079": AesKey(
        "d5c0e38cda966e4caf14ad65d8ad769e8b8b90ef7cb07771064c213c1c9de406", "d8b06fa72121d761bb6cf3ed8d21e473"
    ),
    # FGT_VM64_HV-v7.6.0.F-build3401-FORTINET
    "b31df2f69216f75aded13ec1d4238a08064c6c5fb9dbaed246197efdae8db043": AesKey(
        "05f9ce40e47b3575854e74417d275af19825567b607fa5ae52440f6dbd0c75ba", "76fa7d7b991f3c9021b021d3cb13754d"
    ),
    # FGT_VM64_IBM-v7.6.0.F-build3401-FORTINET
    "18d397b276eb995b3da1dafa9df348c339890c4b36c6f9b68b8d0e37ab6eaa0d": AesKey(
        "baf50ef32102c4c573f0f2084b692c5bbbda65928d091fbb2725954ec798e450", "d271a422e129566a7010612d639fd69c"
    ),
    # FGT_VM64_KVM-v7.6.0.F-build3401-FORTINET
    "fade571b90681aa514aea0f7b4e0e64805d3ecc712d6df851d008224aa029a99": AesKey(
        "b36294c5330a159d5e90a12f590549f249be690f3294ba54a0bd9ec896bfc4ca", "310e91cb89ce4e14d7bb9bf0ee547f6f"
    ),
    # FGT_VM64_OPC-v7.6.0.F-build3401-FORTINET
    "3f6b1dc3a58a2605e8644cdf3d5ca94ef66c9643812f5a05259d3a36b6768aee": AesKey(
        "4850761e7928ce556536be55467cbc45729a9da593fa840f97199ae5b3c8bb3f", "e9072d232aa0c0b91077aa3a40207fe7"
    ),
    # FGT_VM64_RAXONDEMAND-v7.6.0.F-build3401-FORTINET
    "00f0749f5c43c534e7b1933c3815221d74c487eb45531a34214bf806736e0c1c": AesKey(
        "6e9af0f4b30694b9b06e9afbbc64bb5cc316ec2fd9c08fb149237ec0ed765874", "29507c5ca77213c04569f959cdbbe1c2"
    ),
    # FGT_VM64_XEN-v7.6.0.F-build3401-FORTINET
    "2e782443b9e8657ef3340f1fc9ed7e9ada84ecfaef343870cd47456b72feb754": AesKey(
        "93f66fcc67887280c412f0b1dcaa367bd2656cc57d35c1e5899e9f5ae29ce405", "fc2499d4c5961b61972ea681ea152418"
    ),
    # FWF_40F-v7.6.0.F-build3401-FORTINET
    "0d236b2f41059e598462a204718d8ab4768b5b1bb2580f40d6d5f511854f301f": AesKey(
        "d92a6b170821449833030d00d805ee025fb078345616d805c7cf75d6f1c61807", "244b226b0dd74b44eddb1e159231f057"
    ),
    # FWF_40F_3G4G-v7.6.0.F-build3401-FORTINET
    "46d69c9f6c11b69eea088ecfdf66fc1b770f3c9c817bde1ee23295adf73c7bb3": AesKey(
        "9c79e0f4d0ca869c8ba59e41e7d088817cd84187a8f40b132a693bd9f2945e6a", "d86216e1105a9fa7f60e901bef588d1b"
    ),
    # FWF_60F-v7.6.0.F-build3401-FORTINET
    "6ba8092ca2ae4587b8bdae31a0b0ee819b010ed96e2243b6d043368104d02056": AesKey(
        "27c134253faccbc06f85aba93984cf3ff1411b9d39e7942d5fa4e43a3349b7c0", "2b9fab2bc631e2c82b3073ea249bb516"
    ),
    # FWF_61F-v7.6.0.F-build3401-FORTINET
    "2adeacd97f06ae56d530714a2a709c540f979f47b787d1ac97a3fb7471941410": AesKey(
        "450ee9daf5972baf67db2ab9f2ef072d4a39328baf1717b72b1801281c2dcbc9", "b181688964cc5bdf42916346f53d3b46"
    ),
    # FWF_80F_2R-v7.6.0.F-build3401-FORTINET
    "3cee0af0654603593b7f30d7f055347ba1160d30a327114de696d9799e01c121": AesKey(
        "0a4900e35171bacfc50cdf8d50c0314da8971d28a8e48df6dd7af80d02b6339f", "0cf35a6a429dda67a79481d196fc59a0"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.6.0.F-build3401-FORTINET
    "f172e948eadf8908d9dbb621915281b86735aa21a1f71be34d77f1f86931c8a8": AesKey(
        "4915e10fb58241cc086dbe726db8dfaf09d31e259432e7f2852fd4700b7b7a3c", "42cf5c4cb42f7916e5917d53b9a6f2ec"
    ),
    # FWF_81F_2R-v7.6.0.F-build3401-FORTINET
    "ca87e921567ef9549ae649113bba68c4a0e2bb102c2d9c6d7bfb383b2c35dc51": AesKey(
        "d353907781f57751fdf311b7da52e1ffe2e58f8fc793c709f2bfbcbfa3758386", "4c80a26ccabfce6695953d453b40f5aa"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.6.0.F-build3401-FORTINET
    "32e793fea232b4bda79ca6dd142ee0b40ba9a74db68a4874f1102ef9a9e5aa05": AesKey(
        "06b8647c4152953400718dca84f590950030c237ff9978dc18a87ff592d87581", "abb94b06a02b5407ca5066b02440c203"
    ),
    # FWF_81F_2R_3G4G_POE-v7.6.0.F-build3401-FORTINET
    "8706c43a8c33644341b2fb8bcd626ca9a25169d590d011af2ed8f90d98b1a4d6": AesKey(
        "b17311cb94ace035f5b2fe2455cc2cb23347ae22aad42c4f147c46774eb0065e", "3fcca198739d89868036f4dbdbab1832"
    ),
    # FWF_81F_2R_POE-v7.6.0.F-build3401-FORTINET
    "b5b11c8a517f340b1e726d0cac916c48bb2d97bd95bb4b5777c9e24de36493ad": AesKey(
        "3d73e1632931949fbf0fa99fe0a9740eb3b29fd64757df2fe1aea9780b48e837", "6dd1a8937e6da72b5b18d0c5bd597e91"
    ),
    # FFW_1801F-v7.6.1.F-build3457-FORTINET
    "a35c6efef6d695d86cd334d634dae4204957621d441012a36aa398308220dfec": AesKey(
        "4407726c71851c6ca17d3ba2001d9ed004da05aef1cda9706d3b4480cd7c3327", "b71e548540866749cf3146e7b0b24c43"
    ),
    # FFW_2600F-v7.6.1.F-build3457-FORTINET
    "46026408d7956259d6e27eca202ff418d17564415cd6a92a668dc7924cc89a4f": AesKey(
        "4220b792d3094218bc1d9d37fb9bdb503ac3905693c8cdb1044b2fbf3b71089c", "a2eec06cffd6bfafa6e160ea6ccb06ef"
    ),
    # FFW_3001F-v7.6.1.F-build3457-FORTINET
    "d46a64eaf6906d7c4d973f9b6eaed51a1b78f0fa3c589f5544b4426b0f262c7c": AesKey(
        "ec6784b417fcf394cea0ffd9334552ec3843ff41522e20dd15de1338cc0566ca", "3d0c4d9e017329b22a9a691ea907ab45"
    ),
    # FFW_3501F-v7.6.1.F-build3457-FORTINET
    "b4b24169e1f5349da6be84b9874bf1506764fa8ae5779b72bcb2b9f22c8b448e": AesKey(
        "c201e0c3bbd08204ae15f68b6fda582ab1abc175493576f7b0742b1ca61a27e6", "5b2bec4b5b14531a6b72e97e398243db"
    ),
    # FFW_3980E-v7.6.1.F-build3457-FORTINET
    "42318f60a144e2772acf5cf60a28beb0181f6978827f6809d182215b6fdbafa0": AesKey(
        "6edfdf448da173c1ce6ce76ae59141d3ceea5ff5a6e5875a521d39037c257d9a", "2e73d97bfaa3bebea43483f3df2b31bc"
    ),
    # FFW_4200F-v7.6.1.F-build3457-FORTINET
    "c6386b46c3f7d4558ddea0b80ca2e45951b50327f34e8e796f9956b20a62d40c": AesKey(
        "b5bc92a76475e92a5c2a2cde5f209ed1b15d2f968b6f39e02cd6d9e7db42a4bf", "429d10686b64fd8ff55479bc7c6be51c"
    ),
    # FFW_4400F-v7.6.1.F-build3457-FORTINET
    "4fe6599454f9a5ae51de47e489b6433cb0ec93335375bc9a3f58cfab1490ced6": AesKey(
        "3383360710f8eb2351352a48f00911166dd096b8df781090e8b3fac39ba82b11", "a7ef99835bb5623f10c853ea8ce9ecbd"
    ),
    # FFW_4401F-v7.6.1.F-build3457-FORTINET
    "771f4f0ad70ec1bd12544d3d4eca2fbcdd1bdaf62b3aa163c93903b679f68abe": AesKey(
        "b668409919afbf09f195c8cb1b56b708888b4da31d1fae8c122e3fe12903ab6c", "92faa4ecd252a24fb52507cdbd6a0754"
    ),
    # FFW_4801F-v7.6.1.F-build3457-FORTINET
    "38842f2c72b6b95e0353766520528f7c7e577d72e3ad69837623506f3a227835": AesKey(
        "039b083f39d1f716b1b1e06d9ebf77404092239a57156426376e463b194ebb7b", "de0da649bcbc5ede12256c6d511948b5"
    ),
    # FFW_VM64-v7.6.1.F-build3457-FORTINET
    "8360434939be643745f80d8fb6daf791487b1372b78a9d74a487da6f7c807aaf": AesKey(
        "4c889cb0895d30a4fb29593296034878f78b45d35dd89fb3f4b9ec3896d965bf", "0ed04fd2bd59a8ed173e0df7ba60b763"
    ),
    # FFW_VM64_KVM-v7.6.1.F-build3457-FORTINET
    "d91e1a180a082972a47d94a710203e934d8018fc1aeda5d373e944b9fdaff4f7": AesKey(
        "f855cc72b38fd660784734927f12e7f94f0c7daa750c258ba272abcc1f6d9de0", "739ca2248cee17b2e08af581f2cf139b"
    ),
    # FGR_60F-v7.6.1.F-build3457-FORTINET
    "e61de80cf10b84e802ec5e050f288a4f41103476a5825a357a4a7db243f98497": AesKey(
        "f4e6d7777c82ea449aa98d1d3ebdcaaf289360787b314cfef56c17c59740f40b", "16503313c2b99f59174d382d4583324d"
    ),
    # FGR_60F_3G4G-v7.6.1.F-build3457-FORTINET
    "6424162bb217c8137c6df5ea0858ede7ba6b32744b88b787378580b5fc27c20f": AesKey(
        "2b3cdac46c14a1ff6664a43e5f4c7b2f7f2d7049a2120abda72949022d65ff52", "32fe0afb079912361dd45fc2ddd731f1"
    ),
    # FGR_70F-v7.6.1.F-build3457-FORTINET
    "3e7279148cd4b4d784dbd5723490a3ba964d7e535901ed964a50cb39ac1d4839": AesKey(
        "dc12d357896c6b820481319c94a1c573f1fdd5b6622321af4d32596f56419971", "0a496f9313966db1c1592ef2fa28e170"
    ),
    # FGR_70F_3G4G-v7.6.1.F-build3457-FORTINET
    "b0f256d1a297dbdc88bc382f9b56b7ed93b9ec66a5c00335409bb480835ff8c2": AesKey(
        "7f5c244b2c46bbb406bb96042b10a4bfd8b9e1de2f6c65cb4b1e577f9e2cd79c", "8cbaf48b729a452095bc8105f3be504a"
    ),
    # FGT_1000D-v7.6.1.F-build3457-FORTINET
    "a83128e966b8d6dc37688cdd98274f3059cb10cef24268ca78fb5d6b1b68b3bc": AesKey(
        "a227dbe65ff40b07e8a37795e417df4f30f2efe3110d58cd70c8d4b47c8fbebf", "d12d7623e0d233e533253caff8583a77"
    ),
    # FGT_1000F-v7.6.1.F-build3457-FORTINET
    "11ee14fa472615dd53ed5f18862915ae4d473b102f52aedc37f512644b05d1cb": AesKey(
        "703bea596568259af6df5e973a72d9a987699f88d443e560b429c252dbb4d832", "0894ced4d4c51513849e0fb61928e34b"
    ),
    # FGT_1001F-v7.6.1.F-build3457-FORTINET
    "c8d0047e30ad46e671cbbfcd6e3c185fdcb535aa30977f466f40d88ed0973296": AesKey(
        "95e7c010e567fe53f66013f26d030ac09fb7649c4614a390f688fe5d5546d01d", "0f8c33a85e2a0e1141b30531f0209614"
    ),
    # FGT_100F-v7.6.1.F-build3457-FORTINET
    "9ab16a657b599f15ca9421b0b74c1a4c933d839f68ff17503e4a9ca7bb528c20": AesKey(
        "e41ec493f12d62ad933d51ad4a4714a7257b0787c0588e9ec739c77ea3fd9e87", "ee3d170ec157b542f39faa333b10ee13"
    ),
    # FGT_101F-v7.6.1.F-build3457-FORTINET
    "541c72cf6013d67aa4304572fb840fd8eb6ff43e40802b8667bd95fea92ebabc": AesKey(
        "ee8e716b31592297a9b3ec5e9176df1099d12e3b99d4b0fbab30316e4c409bd7", "0e83987814e1a92d898250ea4caba6df"
    ),
    # FGT_1100E-v7.6.1.F-build3457-FORTINET
    "8210b28929272fcb0c998891719d4424f838ce6c4df47578c698ccff72a0b54c": AesKey(
        "8bbd265cbbca4883062232999d5e6b0110c0393319e3752d15c02f34237fff3f", "f6c3691a3f6125dc8d2321c1525abd43"
    ),
    # FGT_1101E-v7.6.1.F-build3457-FORTINET
    "72c93166d226fa7d453f365080dba2c50ff7351c2944d36978a30580e3a19ee8": AesKey(
        "3120f74f9f9b081c4a7fff8d06e73042d3899edd24ea92bbdc77060e54c00fc3", "51d48bd46a6d3a5f85ff118d9e739fc7"
    ),
    # FGT_120G-v7.6.1.F-build3457-FORTINET
    "d84c3b17116cc5b35060b93f9684f0f1ced40a7ee68f327a590d13a096a7395d": AesKey(
        "766ae119a90616ba8db9ce5c29c4929c2de77386365050d5a7a498f42e94d001", "13229ce66265587f60ed796c060c599c"
    ),
    # FGT_121G-v7.6.1.F-build3457-FORTINET
    "fb92b41245318febbd001b165f225fd581436156915f9307f00ec973ce5aaae7": AesKey(
        "7f0b5485ac3a6a0148995fce14c1352abf01c9b5423b8ce5f12025795a8b35ad", "e67dd2e701d6a62d5bbd7c8790910ca1"
    ),
    # FGT_1800F-v7.6.1.F-build3457-FORTINET
    "a55cd80877313534eda2c02212f558248263f09ddac4cbba4b65a55e251b6fc8": AesKey(
        "3fa4a7981b986626352331e33545b810e4064ce76361b8771f4cff88cf975947", "eedca476ab51f197eac00fb77f51caec"
    ),
    # FGT_1801F-v7.6.1.F-build3457-FORTINET
    "ebf7cc61914e500c860475311f507359b314605cad0f631d6d0187146e83bf38": AesKey(
        "2775c4f926908ee1900a71d5cb8d6e037de67fec93ad2deab5c59b19a07d29b7", "e47c424052fe71b37638f28259f656b5"
    ),
    # FGT_2000E-v7.6.1.F-build3457-FORTINET
    "171770a5045f7a01ed1d45f2dc1cf5811034b63332ce804db4c20dfae99cce6e": AesKey(
        "f5c1881d88a25a373d09799ab2f01171f7a76a84d1d7fa366edaad73a84506ee", "df4032935f03f824a0c257d71f08dd5f"
    ),
    # FGT_200E-v7.6.1.F-build3457-FORTINET
    "30fa4fb6e671e7223d97f5e0aef240d3470984074239cbf0d28d5509e23ef63f": AesKey(
        "46a556c6b58d4e6b2dba8a90f976516044ca001f60645dbf2ac27884c9c3cd1e", "b5825053ad97667e6a7b2c997269736c"
    ),
    # FGT_200F-v7.6.1.F-build3457-FORTINET
    "6ed9feac88c69ddc3624942e0cc9a2a63e744eec00283009db0c651ecf4bebd2": AesKey(
        "4f45a063dd2c2766afba880aa9c07d08c664b4e909871f1c594d8efb9f8bc3c0", "1ce49347e1e1646c9960ba6aa8d8b1fd"
    ),
    # FGT_201E-v7.6.1.F-build3457-FORTINET
    "302c2d3664c10324197b5aa80a2caa34382d708c2eb05dc58fd28953d31e8e6f": AesKey(
        "6aa5cb64d75d79581d79065d3d12864a5d008a41ff36375c7905e81235d4269d", "da30a1e7e3f8a80e3521983dec941e0c"
    ),
    # FGT_201F-v7.6.1.F-build3457-FORTINET
    "3b6f9a5f40e9b7170f87e8fa4b018ebb40189805385da186f347cbcc25a914e8": AesKey(
        "44e32802417ef49b2a64ff41b98cbb2e57997f470d5dbbbee8dfda00e92c84b4", "eda0870738c6ea40d2deee61c9583d66"
    ),
    # FGT_2200E-v7.6.1.F-build3457-FORTINET
    "9910dc0dea9ca59410ad762b7e924ff3b25e94e0e93849af76fe3a10ce900b4b": AesKey(
        "a272ff584bdb8200fe5b6d863a3330d9a8596e891713400c46b6d5211a062ad6", "dade75ff35ea643acfb1441da0fba37d"
    ),
    # FGT_2201E-v7.6.1.F-build3457-FORTINET
    "42aa24d4de5142399a8240ea8af00931154a129324a42e22977c2e4aa37892ee": AesKey(
        "a1ee60096e4aa7ca303550c110c2a213a834c63ce3e117ff12c8e1f7b557c7d4", "714b29a346155485ae1d2c5fafcf4946"
    ),
    # FGT_2500E-v7.6.1.F-build3457-FORTINET
    "c8676683ad7aeabb70fcc7f7bff9bc2b0f130ccfcdb25efeb0cb6d62eaf508ee": AesKey(
        "b4cc4b9fdbf23725a0ccfdef3d62c37e1eb687da7cbb77b37caa0181835b4d19", "a8df03516180ffc1597c003ad484c9b5"
    ),
    # FGT_2600F-v7.6.1.F-build3457-FORTINET
    "53d99e07f0ca71c67cdca0efc906b78684dc0d53fe11cf912cd664210966aebb": AesKey(
        "3928792856c6588c036337bbf99eb6d1e93198765354ebb3a17b112a7f5870c1", "3abbce316891b75005d4353cf6a60bc5"
    ),
    # FGT_2601F-v7.6.1.F-build3457-FORTINET
    "fb90b90bab31557783e4908124b12878e62f08e5833093c68279d0942341b7f5": AesKey(
        "24f731d6a474f27a4164308528315c5ad19b8eeb8158453c306757990f73eed5", "27992d55aa36a7e683f20b2785580581"
    ),
    # FGT_3000D-v7.6.1.F-build3457-FORTINET
    "6a9d25e9ceb0109bcfa56c3719cf5ec28ba6d79901e744e0ea493096bd3f57e9": AesKey(
        "88244f60c1a50e3ec10b31624e5a7fa642aba781b2dca2afa6c35054a82e602a", "4d93c94b3014b7eaf0ef311b903ca708"
    ),
    # FGT_3000F-v7.6.1.F-build3457-FORTINET
    "81d60bf88403c58e38ccf5490dd2ac33122ed9750d01c9988e30e45345d93ceb": AesKey(
        "f4097f36bd410c2b45ad187447d792e8c2741f4a1c19ebc5728e58d043aa2543", "0e24332ba0c2de3e01e47ad6233fd30b"
    ),
    # FGT_3001F-v7.6.1.F-build3457-FORTINET
    "3e42b5daa1935d54ad68196e63d2c27176ceeca7a2d30290df2e285a3ad0434f": AesKey(
        "84822df8198e433670f0fa35d2e7fc0eb8b934becb1150ecfbfd3a850ee0663c", "65a9fdfcb4b1ac9d0348895efa390d6d"
    ),
    # FGT_300E-v7.6.1.F-build3457-FORTINET
    "c88d25b4856341a3530ac9ed96a2cdd76d5bfb76707d3361db5f45298cbbf03a": AesKey(
        "565181ff6f7a02655cab298b2703742d321953ec37ecce2faca58538132696d7", "79c78e8468d40b9aec7c6246c01db615"
    ),
    # FGT_301E-v7.6.1.F-build3457-FORTINET
    "0959d2d1853752c206cd1b6cba0ee7d17a20617ea16b1d5442f1cadbb331be30": AesKey(
        "4dccfa2906a74ea2597d4d2e89cb214b066a6bbe238fb3fcf154ba9d63eb0891", "30c1e4581d938fbe25e6e32a8ac4eddf"
    ),
    # FGT_3100D-v7.6.1.F-build3457-FORTINET
    "b7c017ec520da4ff9a83d1a9edfa5465b255b33314af39f6d8366cbdfc86ce2a": AesKey(
        "900762158496d88c243e75c18a318993cf3d83d35a64bef23423bb9a4a2e21da", "71381b2320474ca0c923acd50ba25eb4"
    ),
    # FGT_3200D-v7.6.1.F-build3457-FORTINET
    "1f1e999f21a9e4baec3a3357b51c262f6fab107e65c7df0a51aba73ff4c5a47f": AesKey(
        "34ff0833ed55ff4866b0f29d917abe8883753327d257499ee0ccb3ed621a136e", "bb67d51f4e8df827bf64508494495c6f"
    ),
    # FGT_3200F-v7.6.1.F-build3457-FORTINET
    "36e0c994a7ddf1395fa83455ebe6e3aecec72b0b821d0682ca504c6502d942e8": AesKey(
        "e5c106bb8bc2ea9747b2b5373ba774b040288be062b1aa34f5161ec5ed924500", "3085ca61ac4995594f20ec6f0d3e5f6e"
    ),
    # FGT_3201F-v7.6.1.F-build3457-FORTINET
    "bbefbd377e522613953eb245a009d8aa22c1e261e2d55cae1e56428b6e7a7a18": AesKey(
        "f59be276e6dd44b1e0dc32a6ed1e5caaba3fbef09d026380f970a7812b75fd2d", "e3e757ce1afffa174bace528ae95f943"
    ),
    # FGT_3300E-v7.6.1.F-build3457-FORTINET
    "c7cae5652cb3af3187a117adc94a42c1dfc761e2678d3f69bc4bc215c36ac941": AesKey(
        "3b8a62536bc8285265b54138477c18a420d2bd208f36c787b3ab714ac836849e", "96d8e30e4f6fd000c40d0d80537cff68"
    ),
    # FGT_3301E-v7.6.1.F-build3457-FORTINET
    "e1532b4cabc81feb2ec89787213e52bff69106856a3e15f8d39808ae7fedbc69": AesKey(
        "19ee10cb09ea6422e34695b5096d03fb9b7bf9a52aa09ba3134c01926fa5f24f", "b726870cca1cf8540994f5bf292eff05"
    ),
    # FGT_3400E-v7.6.1.F-build3457-FORTINET
    "1024464854f802915f74d6c54dfc3bb91a07181544d882c190e744514b6ed4a4": AesKey(
        "b4bc7fe2ac4fc9b29b8aaa7c4dedbe76406f74d9afc7595bc907cc2d468f8041", "294d2b3014921c880d80cead8a07880b"
    ),
    # FGT_3401E-v7.6.1.F-build3457-FORTINET
    "19b4eca7b50c655eeafd1d30f4202e1fee014e90cb615c5c89d63258babaefe4": AesKey(
        "8f538d74e374617d1d75c78a0fdeb559fdf5504614eb5e694468a65d0d8e50fa", "f86ac48f766f6a1055d773f1912eb5b5"
    ),
    # FGT_3500F-v7.6.1.F-build3457-FORTINET
    "5e119b8d2459da5948116e8becf8b47c0c641548fadf137f74ac66eb7ffeec70": AesKey(
        "03ce8496db9a1f3603eb65d560a4299779715bc18198d75fcf0988fb5de3fa77", "a9946ff61a528288c5bd12a696c45b86"
    ),
    # FGT_3501F-v7.6.1.F-build3457-FORTINET
    "80cee997b46504a3bbd27e2d85a93774028e9385c970b17c8ae46e39df6fdfea": AesKey(
        "306427b7a5b0ad59e1c23ce08505e98a7170bf3dea68b1364acb6330a3a15b80", "72ae1f545dda828b559d281e595d3241"
    ),
    # FGT_3600E-v7.6.1.F-build3457-FORTINET
    "f0f136907302cc1c23f25f9b63d343de59176d1bfea0caaac107bd3e60c1469f": AesKey(
        "f7d200e19871f4e3cf704549cce801474e2e4ce4359243eedac0178f3b0a4766", "ac8d4104707d3444f3a18934520b4d95"
    ),
    # FGT_3601E-v7.6.1.F-build3457-FORTINET
    "3a0e5229e73cc1ead77f3d87647421e81b5cee9448b5d23a994de12a7e1e3031": AesKey(
        "262ea9af59a354996f8f96fb5e1c7de894e0813c035f124aa3f4b5e776920081", "2d61ea4fedac80dfda7325025862f4d5"
    ),
    # FGT_3700D-v7.6.1.F-build3457-FORTINET
    "d4757b8835bf6c50641005da37cc192e89ab6f964923f6eb8b34753f9377a7b9": AesKey(
        "85b509cb539598ae5f7b3eabd6786220b3d1820d6798c4e9cc918cd126099134", "4fbb01d3788dea93e76c02aeb941104f"
    ),
    # FGT_3700F-v7.6.1.F-build3457-FORTINET
    "ebdca6aa48de43815a8258def762c8bac3f9d16068fee12c2fdfdd8a32f0347f": AesKey(
        "5eb48dcef7266580384e96c74748b857cbdae09ee1e1cc774479a2a0607453ca", "1665864286cc3e43efae21e86f2db9e0"
    ),
    # FGT_3701F-v7.6.1.F-build3457-FORTINET
    "71f055b0cc29dcb7db65d39ff6b09e51d5d42e83a04446aace348b001f67465e": AesKey(
        "0f7329673eb9a354c6c4cfe9c83fb3e0cc7e6e06520bfc2dd4ad6ce1d290dba6", "845f54b73ccea5fc0a7bd261e63fce64"
    ),
    # FGT_3960E-v7.6.1.F-build3457-FORTINET
    "b71f25e3ec392d320a5d6ec50970019374cddb6d6154ca2f0a728dc53cfef336": AesKey(
        "80cc77052e667c7a9f21aab9a6c53946ea2cb88965b8028b7dce9365bb3bfdd6", "fa703cd0a9839019b464cbf4a39932c5"
    ),
    # FGT_3980E-v7.6.1.F-build3457-FORTINET
    "f011c0532b016b78454a23ba61291865da0f674c2a225e2298c7da116947a87a": AesKey(
        "75121c024b8c9cd7e2ae44b71fa306b66544f877cfeaf64090d66e02937c500d", "964d7f85ba8d9bb7f55aac2e4d10d7bc"
    ),
    # FGT_400E-v7.6.1.F-build3457-FORTINET
    "02b6095fa7da584e56dd3f9d6d5eb565eb06ff0ff7c7c05a4c6460a904980d38": AesKey(
        "c7505c3c611df3992be77eeaedb65c01007fb06c9bab1b37e2f29e42c14434e8", "758c844a09124cc7bbd3ac606e1ff83c"
    ),
    # FGT_400E_BP-v7.6.1.F-build3457-FORTINET
    "a847cdbfaf98896bdd5fdca63aa4835105aa2799032ad17f0555ca410131a050": AesKey(
        "80d61a048c52c53845b43c422c0943bb039845c90cce724302892e52434f1f35", "2408ec387a8fc0f967c1da73c376a713"
    ),
    # FGT_400F-v7.6.1.F-build3457-FORTINET
    "14c782fdd80204bec8af2f69582f56598f70b6ecb3098f909543a144b2e52339": AesKey(
        "d8b1c02c6f93746ca4691ab4472d579744cca1ede053adb6a25c1e2b275e1276", "9964dca67a2959e75622f7cb67ca3598"
    ),
    # FGT_401E-v7.6.1.F-build3457-FORTINET
    "45d89540a2223345a751ecb3d80f8af9e23f628685796f5e74857b3384a635a1": AesKey(
        "31fc14158b7fc08d7919fa02aefb2760772b2e8710c8516de9d32e143fc9f922", "68d9192aff7618bea8d439480d552874"
    ),
    # FGT_401F-v7.6.1.F-build3457-FORTINET
    "d8a744efc823e2582134437107a139107fe8c4a15986a1ccc89fd1d9d67c5da9": AesKey(
        "0d267565b1d4b98967fc615a93ea3b5ef12eb72caa1a1eff24ac65c50731c2b9", "e98497e80889772c2ae393b8a4114b76"
    ),
    # FGT_40F-v7.6.1.F-build3457-FORTINET
    "74f265f9364ba49994d9c7598b4e6b9d1d070580ea69ee69ec594a69f66acc12": AesKey(
        "081b5b0d8bb329dc99c3c20570e100ed3ca9c9152bccadf8e94c2f138d82e862", "8765d3502370b43315017cdaaac6dd15"
    ),
    # FGT_40F_3G4G-v7.6.1.F-build3457-FORTINET
    "4d865ea589eea8fbc3c3d7c0a40a6948957eb487739055be4290d1d424fe3ab7": AesKey(
        "724f5863d3d4571e3cd1cfef6626ad3f65cc5f63bdb1e403029f65638e537dd0", "124a3493868536b7b0580438db0b742c"
    ),
    # FGT_4200F-v7.6.1.F-build3457-FORTINET
    "b8d4fe7f89a90dcfa76362f97a85ce305d78d6d426de5a71149b95234923618a": AesKey(
        "57e83ea9135a9e1abd6d9bd66166428c4581de01b1d6c6dc7e0a77c13c28aebd", "10f8163f4c871c36fca2407092221053"
    ),
    # FGT_4201F-v7.6.1.F-build3457-FORTINET
    "213b0e4c125c40ed4b187dd549e9b27a4b70c74182c115ea45a758022950cc8d": AesKey(
        "d2574124c8699c14e090daeaf79e8619d4fa073e3a0e9315fe95d08921fdefd1", "d5f2836048b289711e5e99c7850032fd"
    ),
    # FGT_4400F-v7.6.1.F-build3457-FORTINET
    "064da284376b4ae055c3b5b6379c1b72692d79f86e9ec728a1bdf0586ea7bfc1": AesKey(
        "a96cb5fa4341dbaff32f6cd2a95f58dd5204130fc8f55c735285da6d83f3afaa", "58b7f3e00e6997524025e7c83e86c999"
    ),
    # FGT_4401F-v7.6.1.F-build3457-FORTINET
    "f2d21e03bba897cab92702f8c0e4e751b7979c7c4b89d697f328dfdc72d0b944": AesKey(
        "f5bfa32af2c58b3cc73c2b2aaa8139ed098b89a9fe55b7d873d4d0ed78af74fe", "57e378b836fbac16051b2850bdf993c1"
    ),
    # FGT_4800F-v7.6.1.F-build3457-FORTINET
    "586113308a44ec906be3132036af10d497a278377708ac673a514519dab3b2f7": AesKey(
        "412f00b029c90a32558a54e3b0c4b9efed9c8631d1c33b94c34be7d3c5a984b4", "724584626d8c74b3ef99681692108dc7"
    ),
    # FGT_4801F-v7.6.1.F-build3457-FORTINET
    "3fcd90a455f1c0c99ba9b90b21d96c32d40ba3066220cf4a05434c7499c5b261": AesKey(
        "d457ca68b2ffc1297447e80a8fc39431d1108621cf10c4a75f7563145b489fb9", "d67be48faa22df140594e434e2d458b8"
    ),
    # FGT_5001E-v7.6.1.F-build3457-FORTINET
    "55fbe1eabe2f6d41830737c05b6fae99871d37cfdcee82263a077ce5fce7f4e2": AesKey(
        "826410d8a69f5e684d0fffc933fea717a134113549dbdae7ebf235d767b07cbe", "cdd77fcc230879cdf84f319ad53f7b64"
    ),
    # FGT_5001E1-v7.6.1.F-build3457-FORTINET
    "1d7e8cdab24fd8572f32e8b450e2fbb519a9e54a8462fdb1d913e21a385d8b24": AesKey(
        "da0347dc84b4384cf836a25d3fd6f0a38c572e3e4696e067136fd98c4674f6f1", "43b70b817d4af61214f9802c5966dd7d"
    ),
    # FGT_500E-v7.6.1.F-build3457-FORTINET
    "4d18e3b9b18de450e6e743394b4b9f4abc64bc7f6651f6bd22b04af546cf181c": AesKey(
        "7e84dc45783c2e92dc69f47bd6c4201f64076c73c8aef6e59ab4cd604f5026c8", "155128ef0f0a6bab7194593bf02b9934"
    ),
    # FGT_501E-v7.6.1.F-build3457-FORTINET
    "a7fa9a912c3f66692610589ef8c5375c64f2a2ef3411396e28e3d8c0b84c3aca": AesKey(
        "009f407db7ddca296189ba0e231ae925abc0ce58a197bd3ff3745841af336f85", "442506e633e6f9af5a55aec82ef6b800"
    ),
    # FGT_6000F-v7.6.1.F-build3457-FORTINET
    "fc766026eba51b364eac75fc4277984dcc246f63d0862cdae912e7fe554e4bd7": AesKey(
        "d55b16420258137cc169b7f5a8425d2ed108a8b8b5d10655c4c06a3e8b37eccd", "c2b6dfa49926a47031eac5bc943022e1"
    ),
    # FGT_600E-v7.6.1.F-build3457-FORTINET
    "f8c5590b63c1223dcee8008f977f2aaf7d13c3b8876f8440c2b00f7cf1cde75a": AesKey(
        "00ab9b6b8a1ca6433ba74ce184b74bafd694fa2171a13087715b75eb323d2f05", "5b2309a4c5e228951c95f0ef5a5b9a72"
    ),
    # FGT_600F-v7.6.1.F-build3457-FORTINET
    "5047053a820be0e8e643f0e9802f11eb31fc4eb2910f2845ed6ea38125f2df59": AesKey(
        "db763648500da56b99d1f81a667b099c1ec9128d6db57eaef5d8752ed3654025", "7a5a1b04c735789949ec62e1a5a880a4"
    ),
    # FGT_601E-v7.6.1.F-build3457-FORTINET
    "7806e26149ebd80ffa032e16ab5ab2bf9bf4d19020c204f5f04c6b516be3d6fc": AesKey(
        "25e18958006d50d640b02fae73de9b671679c24e51c6510270c1ede5294c9f2f", "b1d8862c90305a8f5c501e9dfd56ee2d"
    ),
    # FGT_601F-v7.6.1.F-build3457-FORTINET
    "8031cb9f8088567dbe7103da08c57b3f232321ababeab72acb365e661aa56e68": AesKey(
        "ffef0c01481a581c706f17b8fce25331894ededb14686435e30a0b92321c9a5c", "52788b05906caeae027687f7762058a1"
    ),
    # FGT_60F-v7.6.1.F-build3457-FORTINET
    "cf68f65898faad59298f44049697acb59304394f8a54f9bdf0480d12dcb76c26": AesKey(
        "d11e9281bb9aa422220eddf8eabbd669afb85fe4e0778404e20330b86b6e4179", "8650e2303063833a8a1e2d5439404ef3"
    ),
    # FGT_61F-v7.6.1.F-build3457-FORTINET
    "a916b3be02074b5ac327c0b7c640a056a5171341ed264c41902832101ec24f9a": AesKey(
        "e7d84928ead689f8bb83c96ccdf7e36ea563299241e3d1bc9e4cfd99a85a75a8", "3fbea7d50aea59229afafc491d401926"
    ),
    # FGT_7000E-v7.6.1.F-build3457-FORTINET
    "463e98c20c43e0dd132725b6c973ec6140bd27c71e30ec7eaeb5676572d40d42": AesKey(
        "cf8f92b8879124041673deb620b66fd2adf5fa01116dbb0a4281feb7a829e5cc", "4ac701bea77f5052b3b9efa99b6b4f31"
    ),
    # FGT_7000F-v7.6.1.F-build3457-FORTINET
    "8a9f05eb6472a6fac6b433c908666096ce1ab7364b918c3ce08575d4ba729e79": AesKey(
        "80ffcf11c3329182be2c377cd23b9553a37a7962f621f30234439dc8795e4cab", "df72a2e978160a9e88c229c5b16cbf7c"
    ),
    # FGT_70F-v7.6.1.F-build3457-FORTINET
    "c984454b60d0bc46d494c5ae58f757482961b4926d9ea1fe9fa5eec25c76f03b": AesKey(
        "938609ab1788559335028919ef565dced1e003666d794be2f16e83e8bfc51594", "04c39becbceb9c0066439afeca731bc1"
    ),
    # FGT_71F-v7.6.1.F-build3457-FORTINET
    "879fcd8e5438bd4d6635dda22aaf0bc472be1943e90e7385615e56d0bc363d47": AesKey(
        "f417d2072a1cadb00d6b6d26ba3d2de28d3769014019607b590d8114e45aaaf9", "acf0877879cb1f67b5594ee855add713"
    ),
    # FGT_800D-v7.6.1.F-build3457-FORTINET
    "48e5568bffe330aeff8224c9df8f047d3091702860ed659d44479cff4501c383": AesKey(
        "f87694a967cf7d146dbc6cd09b2a76c160978fd6489399532a13052ce2172ea1", "7b5bbdb9dbb8078e9e9f67224b20877c"
    ),
    # FGT_80F-v7.6.1.F-build3457-FORTINET
    "3622e06f217cb02e910ae4c3b507848e69412033ce9a037e3ca1dd8570e925f6": AesKey(
        "03bc5652da2d0b88029b048d8de570ac36e6dd3f25ee716038582bf790a102a6", "15e30c17912cda65497e15acb1ea8d55"
    ),
    # FGT_80F_BP-v7.6.1.F-build3457-FORTINET
    "3c7c04c50295010230349c123b7b668e2b3aa18126962f3eab5cb916b20c8d66": AesKey(
        "259a240c7784c226c0d327bf1c544147ddfb3c9d306c66e7ea3da6538a6d1212", "bde9fc6270fbf4bcaa82d41b57ec13ac"
    ),
    # FGT_80F_DSL-v7.6.1.F-build3457-FORTINET
    "0e14b4a347f6d22c2e253149547cf6e1cc26ec40359a4009f09ada96617585e4": AesKey(
        "763714c559e45dbd4039afdab086af34f2756cdfb96f68975defd7d37da45abc", "24f229ceff509d5898b6fe7dca33886c"
    ),
    # FGT_80F_POE-v7.6.1.F-build3457-FORTINET
    "405acbe40dfecc82ba7ec44ecda5c2c7390f6a2a603e39f24be7eeb1012ae64f": AesKey(
        "5ba348587ca88cdf79f2365287247c05268ed1a4f43165eaccf698b1151f2f0b", "2871ae07292f876180daad8aeed7c21f"
    ),
    # FGT_81F-v7.6.1.F-build3457-FORTINET
    "89b0ff7eeae2152a7ef651533121184c81aefa628b96918af3d5c6feb1b43fce": AesKey(
        "2bebf90c3f33808f4d225e71143ebcf9f985eb4035c098e41e5b953f202c1396", "8f02ca95db1f181c57c6625038cc952c"
    ),
    # FGT_81F_POE-v7.6.1.F-build3457-FORTINET
    "5f698b7cc7c8c6212dfe6df56758a73cde3cccb4988ed3cac0464cd949c14f35": AesKey(
        "6c6867c391234b6d33a1e3ec520500f22df22f3b700fca672f1d8097e9dafaa5", "f0baab6d118b5ef0a022d3359c5aa511"
    ),
    # FGT_900D-v7.6.1.F-build3457-FORTINET
    "90e254b906f92ea02af3f97c1d84c6e792bcf6aceeb4e422f3eea4488e383285": AesKey(
        "a663e89662edec5d1daeb9ed2f6df00e7aa9be5e3acf4cbad49bff435584778f", "3d43880d3f6ed6e4652c4994d559c2b3"
    ),
    # FGT_900G-v7.6.1.F-build3457-FORTINET
    "7a31e0d849c7dd155497538d5b41d12b4833b2d0861c2c388f2ca60926c44b16": AesKey(
        "81c96ea747f8a7547dfefd3f654340854f499c2775435768fee76f5802c67924", "f33818706b98ebc67e7f9bee746e9dfe"
    ),
    # FGT_901G-v7.6.1.F-build3457-FORTINET
    "86c929372b89c5b766a38636bdfca1938768afcf1981001ed38676d1f35315e7": AesKey(
        "88937a641180237ed7f47742ba96c850bba61477b11e6681e8979b001ff47831", "f259e146c28cbd31ac258e599a628c40"
    ),
    # FGT_90G-v7.6.1.F-build3457-FORTINET
    "7b012b55e67c189dfd96994afa50c25feb7e92cedb2503085cab7fe56c4d3c1f": AesKey(
        "9880d3f39ea843a7cd6ec1940a8663e1d36cdb80ea05e6e77d0dbeb8184bb76b", "ed01b3d00d4f132713a8c55ff4741f80"
    ),
    # FGT_91G-v7.6.1.F-build3457-FORTINET
    "0fb28c0d0a8dfba9b927257d20c993188c2aae23f140c89b77cf217ca7e4e148": AesKey(
        "e574a81895b935e0b30a5a8b0e6d664e9a139897456a48505c6d0a9edef2c2b5", "e28882fe6e44a744a7d7acfef32a43cf"
    ),
    # FGT_ARM64_AWS-v7.6.1.F-build3457-FORTINET
    "bfa18af5597528114757c6bd24e9851cae3f18ec2ddb1657135af5c08991fd52": AesKey(
        "38e21bb9afd2a18c129f0786ce8e883ce7c1ccd63e1f51cddcb84608d17cd6ae", "2f16099980c2811520f681ad89452d17"
    ),
    # FGT_ARM64_AZURE-v7.6.1.F-build3457-FORTINET
    "e24777f5655bf40b51ac74c4b33910b0965c9a7d33a7f330e6cfe9965c2906de": AesKey(
        "afba906a7474078a47275ecc5eae6ac392fe1bf29b4251e483ef19e38b65b88c", "3ac4b067e0a3b3aee928b633aea11401"
    ),
    # FGT_ARM64_GCP-v7.6.1.F-build3457-FORTINET
    "29a53d275fd5db025a82d7709051435fbd8993c612d4e9cd1a5b42ed29652f1e": AesKey(
        "6546f362e40abd42f1f6b2f7947978f34a2d8ef03621dece019590c4f147bc0e", "e0700db3e5e7030d266d9da5b6ea761d"
    ),
    # FGT_ARM64_KVM-v7.6.1.F-build3457-FORTINET
    "edbfe60b0e4c56b89358e60261dec5344eb43149372ea14c912fc200babaa068": AesKey(
        "44b28f29fe961ad3c046233497c5f334fc1c3875a69697a4f1b928047556565c", "cb946457fad13bb2f299fd98e7974e0a"
    ),
    # FGT_ARM64_OCI-v7.6.1.F-build3457-FORTINET
    "e5d3a35f4cbba7a3dc904c8925493d6c3264955b47dc9f2399aacdaa668e8589": AesKey(
        "1894c7e1fc14a379f51e2bf82120d52bcbf74c377af6aa466b1e70393f2bfa0f", "3e557b20cce1da89098b4ad21efcf8df"
    ),
    # FGT_VM64-v7.6.1.F-build3457-FORTINET
    "b3394fc2a9a9219c4fd2df8a9a6540a9aec3e5955374a04d87ecc4915415a238": AesKey(
        "fe48b5a5b5e24cd8cb24deafd2848174e4d7817343bbb913512f2fea20bcd345", "fdff013a7e949fff65e07c36ae3f3476"
    ),
    # FGT_VM64_ALI-v7.6.1.F-build3457-FORTINET
    "b3e82007d20792a77bc9ce343bc6b9a4ebabe08fb041890404b52c1d41f1be8b": AesKey(
        "68c4c3cae6bed924304c758a7a25fcd36490fe089883ae9459ed69d77e91d9b4", "d7c4edb237e61a95b0cca9eb0c6efa8c"
    ),
    # FGT_VM64_AWS-v7.6.1.F-build3457-FORTINET
    "f38efeeb93d80e9084d1c5202652c7604a00a31cb67363384c7d529e520de52f": AesKey(
        "372d3719085108f692368dd1a21d4b227403cc2bed8018619b9a5586b007f4b6", "930d573b2225d8ea09b652ebf5efbd21"
    ),
    # FGT_VM64_AZURE-v7.6.1.F-build3457-FORTINET
    "de6753f57893dcc68374bcc79998b99d36339d67146f9c2bd7365fdb894ea0f1": AesKey(
        "5d545803ff3fb593a8ade8cf072a02e84da8cd31b947c28de8bcb62f41f6fbca", "203651d6ec69155422dc1fff701352ba"
    ),
    # FGT_VM64_GCP-v7.6.1.F-build3457-FORTINET
    "57b1c7185950e401f9c1ed25456dd2abeae0f9e392e0bcc1e50bc43219693dcb": AesKey(
        "c57e960f2fbfc59a2b6eb5a7ff2c5778b8c03cc767c17e93f6efca7b6da46f16", "4d1b5e05f1cf5211106f55c8158bb4bd"
    ),
    # FGT_VM64_HV-v7.6.1.F-build3457-FORTINET
    "e9690ae7620040bfabbaf63abb15c77724ddb97a8f36ce5bb81745a5cb9d587e": AesKey(
        "558212a0107eecb86f24a2451c2bf8b7db128f03cda733378483f3c3cf84fbb4", "07bd22eb6164d96b8e51a32eb46d43a5"
    ),
    # FGT_VM64_IBM-v7.6.1.F-build3457-FORTINET
    "141c74e5545116c85168970a54fa55f09f1704fe80522a57bd3bca82df0e5b4e": AesKey(
        "21dcd0bb55ba71619810766c3f37b2bb16642cacffe79151ee1c7c1332efd6b0", "bbc2528caed26529b6bc563d807e58b0"
    ),
    # FGT_VM64_KVM-v7.6.1.F-build3457-FORTINET
    "637bb45cd2933bc9e454ad59b690da32be94abacebcff60617f269cb55fdbcfe": AesKey(
        "65d12e74f16e7bf942224712ce9dd6de7c5a09960f510d61d9ccf7c2b9aec348", "2bc0bc7d48a4e241a8010df3bb572812"
    ),
    # FGT_VM64_OPC-v7.6.1.F-build3457-FORTINET
    "6533ebaa80b82682d9cf2129669cf74c0c3f7790ce31f2d76fb981d507081eac": AesKey(
        "b3fcc2c6ddf86a78e3c73b0dcb35f06137514212c2c6f5e3cb296a2d1850359d", "bfab6c3adf99b6256299e81893704b59"
    ),
    # FGT_VM64_RAXONDEMAND-v7.6.1.F-build3457-FORTINET
    "b0e26881a914535598c29e2499440a7bd87a45df62a98792f21c18ba7c32571f": AesKey(
        "7ab6646d42039241717e47221db4de055c64d28de8e0f39c59e977f5630a9c0b", "a2ce4b22ef2ce8ffadd483d4fccbaa5e"
    ),
    # FGT_VM64_XEN-v7.6.1.F-build3457-FORTINET
    "49e1757c7f26a9ba4dbdbab18351f59f71e4550608b437a2dc1d8ce89edac578": AesKey(
        "426b7b8c764bf29233be2e3b0a1f685c28f5766038ed8869ea679b2e5c28ba15", "001cc943226ef846923678fad18fb941"
    ),
    # FWF_40F-v7.6.1.F-build3457-FORTINET
    "d3b5d0ac4d10d48a66ccb4761f98bdb3369fbb7205c29e73ffb2ea07eca454fb": AesKey(
        "bca3358f116322c8a827d572ea0f69a2aef4f592b8a7749c78291448e62e2e9c", "fa9390586be56ee0bf15eaff93d226ae"
    ),
    # FWF_40F_3G4G-v7.6.1.F-build3457-FORTINET
    "20dbffa5cf7a5cbdda5a0ccf13bbaaf8b582d2088fbed2d06db53bfac87191d7": AesKey(
        "511f37d853367475eeddd20974c9d8636f086a23183bb53bcb2e0852ce4d001c", "12d84ef00a2b01f4fc34467b07087cf0"
    ),
    # FWF_60F-v7.6.1.F-build3457-FORTINET
    "c55eba1ff08ebb6a71f220730646727b5a80e3cc302b2698a92441907dcc0474": AesKey(
        "d14f85b20f088ded09a27030db89290f895976abae589deee034a1ca29a6e37c", "c8ccde19e707bfceb3464357486f9cf1"
    ),
    # FWF_61F-v7.6.1.F-build3457-FORTINET
    "9a32a967ef5899606c12609d4c04402dc59538ccf096e9b51f5748ca4bf4951f": AesKey(
        "83c13822be558f652ab45eb4922436837cdbb98d1171646eac3a10031741a80c", "34d02386f6b2e5cbac9ba1c673eb97f0"
    ),
    # FWF_80F_2R-v7.6.1.F-build3457-FORTINET
    "95dd10e20ccba7c53c6513965f0a7258a48a2cdbca17a17992a498f29a85d3ad": AesKey(
        "b97536583d042732bf074b73daadff5a88f276049b7329ea168ab013232067ad", "ca74d39816e34b0afda04977371ce33c"
    ),
    # FWF_80F_2R_3G4G_DSL-v7.6.1.F-build3457-FORTINET
    "a710abf076cbe5bbf24a0d5e0fca9944e24c14f2a5f7c3bd7dd55c3d4b8933d0": AesKey(
        "220783e4388556fca3615cb4601a8ddf0289b9a123dab7107bdc9fba47dbf71d", "57f15932f8e08cc82126b08786294ebc"
    ),
    # FWF_81F_2R-v7.6.1.F-build3457-FORTINET
    "6fa6d33d5160f844807eedf6a5cda5639f625efc5f98d286edd8c50ba31ccad8": AesKey(
        "cfa33e1849fd284f4965ed9673c815e50b4fd5f4e1ea04873eb7f2e2516dd854", "c30a64392a145aa60625cb2bd7315b1f"
    ),
    # FWF_81F_2R_3G4G_DSL-v7.6.1.F-build3457-FORTINET
    "4494332f928155f52ab1c4dd06ec35bd26ec6258e3ac08a17e717d198fccc04a": AesKey(
        "7bc0310a7c1a14c149701076292e4a8eb1ce95272a49f9923c97aacaa494dfed", "82c5e22cf3e605734b5e53ab08c48605"
    ),
    # FWF_81F_2R_3G4G_POE-v7.6.1.F-build3457-FORTINET
    "faa8327fde20878277680a4d5f1d38293066dba2b4064906282829930c698981": AesKey(
        "06b6e2d33e03624ae3681fa92cf307695d9dace392f2a4d8635cfd9039aa83bc", "b2bcf19dcd5050bdecd25253a197d107"
    ),
    # FWF_81F_2R_POE-v7.6.1.F-build3457-FORTINET
    "a446cc158d2c119bfaa19ce566f59edf533134edb19a143a2bf61161e205129b": AesKey(
        "2ddac6d3884286a095dff3d1d5879aa7bac8a77c6026803f66d4997a0aad7b55", "1a4aab8fb43d5af04270d214907a5688"
    ),
}
