"""
The :mod:`scadnano` Python module is a library for describing synthetic DNA nanostructures
(e.g., DNA origami).
Installation instructions are at the
`GitHub repository <https://github.com/UC-Davis-molecular-computing/scadnano-python-package>`_.

This module is used to write Python scripts outputting ``*.dna`` files readable
by `scadnano <https://web.cs.ucdavis.edu/~doty/scadnano/>`_, a web application useful for displaying
and manually editing these structures.
The purpose of this module is to help automate some of the task of creating DNA designs,
as well as making large-scale changes to them that are easier to describe programmatically than
to do by hand in scadnano.

This library uses typing hints from the Python typing library.
(https://docs.python.org/3/library/typing.html)
Each function and method indicate intended types of the parameters.
However, due to Python's design, these types are not enforced at runtime.
It is suggested to use a static analysis tool such as that provided by an IDE such as PyCharm
(https://www.jetbrains.com/pycharm/)
to see warnings when the typing rules are violated. 
Such warnings probably indicate an erroneous usage.

Most of the classes in this module are Python dataclasses
(https://docs.python.org/3/library/dataclasses.html)
whose fields show up in the documentation.
Their types are listed in parentheses after the name of the class;
for example :any:`Color` has ``int`` fields :py:data:`Color.r`, :py:data:`Color.g`, :py:data:`Color.b`.
In general it is safe to read these fields directly, but not to write to them directly.
Setter methods (named ``set_<fieldname>``) are provided for fields where it makes sense to set it to another
value than it had originally.
However, due to Python naming conventions for dataclass fields and property setters,
it is not straightforward to enforce that the fields cannot be written, 
so the user must take care not to set them.
"""

# needed to use forward annotations: https://docs.python.org/3/whatsnew/3.7.html#whatsnew37-pep563
from __future__ import annotations

import dataclasses
from abc import abstractmethod, ABC
import json
import enum
import itertools
import re
from dataclasses import dataclass, field, InitVar, replace
from typing import Tuple, List, Set, Dict, Union, Optional, FrozenSet
from collections import defaultdict, OrderedDict, Counter
import sys
import os.path
import xlwt


# TODO: make explicit rules about when strands can be added and sequences assigned.
#  For instance, if we add a strand to overlap one that already has a DNA sequence sequence assigned,
#  should the complement be automatically assigned?

# TODO: see if :param the_parameter: and :return: can be used with Sphinx


def _pairwise(iterable):
    """s -> (s0,s1), (s1,s2), (s2, s3), ..."""
    a, b = itertools.tee(iterable)
    next(b, None)
    return zip(a, b)


##############################################################################
# JSON serialization
# There are external libraries to handle JSON
# in Python, but I want this to be a simple, single-file library, so we just
# implement what we need below.


class _JSONSerializable(ABC):

    @abstractmethod
    def to_json_serializable(self, suppress_indent: bool = True):
        raise NotImplementedError()


def _json_encode(obj: _JSONSerializable, suppress_indent: bool = True) -> str:
    encoder = _SuppressableIndentEncoder if suppress_indent else json.JSONEncoder
    serializable = obj.to_json_serializable(suppress_indent=suppress_indent)
    return json.dumps(serializable, cls=encoder, indent=2)


class _NoIndent:
    """ Value wrapper. """

    def __init__(self, value):
        self.value = value


class _SuppressableIndentEncoder(json.JSONEncoder):
    def __init__(self, *args, **kwargs):
        self.unique_id = 0
        super(_SuppressableIndentEncoder, self).__init__(*args, **kwargs)
        self.kwargs = dict(kwargs)
        del self.kwargs['indent']
        self._replacement_map = {}

    def default(self, obj):
        if isinstance(obj, _NoIndent):
            # key = uuid.uuid1().hex # this caused problems with Brython.
            key = self.unique_id
            self.unique_id += 1
            self._replacement_map[key] = json.dumps(obj.value, **self.kwargs)
            return "@@%s@@" % (key,)
        else:
            return super().default(obj)

    def encode(self, obj):
        result = super().encode(obj)
        for k, v in self._replacement_map.items():
            result = result.replace('"@@%s@@"' % (k,), v)
        return result


#
# END JSON serialization
##############################################################################


##############################################################################
# Colors
# As with JSON serialization, there are external libraries to handle colors
# in Python, but I want this to be a simple, single-file library, so we just
# implement what we need below.

@dataclass
class Color(_JSONSerializable):
    r: int = None
    """Red component: 0-255.
    
    Optional if :py:data:`Color.hex` is given."""

    g: int = None
    """Green component: 0-255.
    
    Optional if :py:data:`Color.hex` is given."""

    b: int = None
    """Blue component: 0-255.
    
    Optional if :py:data:`Color.hex` is given."""

    hex: InitVar[str] = None
    """Hex color preceded by # sign, e.g., "#ff0000" is red.
    
    Optional if :py:data:`Color.r`, :py:data:`Color.g`, :py:data:`Color.b` are all given."""

    def __post_init__(self, hex):
        if hex is None:
            assert (self.r is not None and self.g is not None and self.b is not None)
        else:
            assert (self.r is None and self.g is None and self.b is None)
            hex = hex.lstrip('#')
            self.r = int(hex[0:2], 16)
            self.g = int(hex[2:4], 16)
            self.b = int(hex[4:6], 16)

    def to_json_serializable(self, suppress_indent: bool = True):
        # Return object representing this Color that is JSON serializable.
        # return NoIndent(self.__dict__) if suppress_indent else self.__dict__
        return f'#{self.r:02x}{self.g:02x}{self.b:02x}'

    def to_cadnano_v2_int_hex(self):
        return int(f'{self.r:02x}{self.g:02x}{self.b:02x}', 16)

    @classmethod
    def from_cadnano_v2_int_hex(cls, hex_int):
        hex_str = "0x{:06x}".format(hex_int)
        return Color(hex=hex_str[2:])


class ColorCycler:
    """
    Calling ``next(color_cycler)`` on a ColorCycler named ``color_cycler``
    returns a the next :any:`Color` from a fixed size list,
    cycling after reaching the end of the list.

    To choose new colors, set ``color_cycler.colors`` to a new list of :any:`Color`'s.
    """

    # These are copied from cadnano:
    # https://github.com/sdouglas/cadnano2/blob/master/views/styles.py#L97
    _colors = [Color(50, 184, 108),
               Color(204, 0, 0),
               Color(247, 67, 8),
               Color(247, 147, 30),
               Color(170, 170, 0),
               Color(87, 187, 0),
               Color(0, 114, 0),
               Color(3, 182, 162),
               # Color(23, 0, 222), # don't like this because it looks too much like scaffold
               Color(50, 0, 150),  # this one is better contrast with scaffold
               Color(184, 5, 108),
               Color(51, 51, 51),
               Color(115, 0, 222),
               Color(136, 136, 136)]
    """List of colors to cycle through."""

    def __init__(self):
        self._current_color_idx = 0
        # random order
        order = [3, 11, 0, 12, 8, 1, 10, 6, 5, 9, 4, 7, 2]
        colors_shuffled = [None] * len(self._colors)
        for i, color in zip(order, self._colors):
            colors_shuffled[i] = color
        self._colors = colors_shuffled

    def __next__(self):
        color = self._colors[self._current_color_idx]
        self._current_color_idx = (self._current_color_idx + 1) % len(self._colors)
        return color

    @property
    def colors(self):
        """The colors that are cycled through when calling ``next()`` on some :any:`ColorCycler`."""
        return list(self._colors)

    @colors.setter
    def colors(self, newcolors):
        self._colors = newcolors
        self._current_color_idx = 0


default_scaffold_color = Color(0, 102, 204)
"""Default color for scaffold strand(s)."""

default_strand_color = Color(0, 0, 0)
"""Default color for non-scaffold strand(s)."""

color_cycler = ColorCycler()


#
# END Colors
##############################################################################


@enum.unique
class Grid(str, enum.Enum):
    """
    Represents default patterns for laying out helices in the side view.
    Each :any:`Grid` except :py:data:`Grid.none` has an interpretation of a "grid position",
    which is a 2D integer coordinate (`h`, `v`).
    (scadnano also allows a 3rd coordinate (`h`, `v`, `b`) specifying a "base offset" at which to position
    the start of the :any:`Helix`, which is not relevant for the side view but will eventually be
    supported to adjust the main view.)
    """

    square = "square"
    """
    Square lattice. 
    Increasing `h` moves right and increasing `v` moves down. 
    (i.e., "computer screen coordinates" rather than Cartesian coordinates where positive `y` is up.)
    """

    hex = "hex"
    """
    Hexagonal lattice. Uses the *"odd-r horizontal layout"* coordinate system described here: 
    https://www.redblobgames.com/grids/hexagons/. 
    Incrementing `v` moves down and to the right if `h` is even, 
    and moves down and to the left if `h` is odd.
    """

    honeycomb = "honeycomb"
    """
    Honeycomb lattice. This consists of all the hex lattice positions except where 
    honeycomb lattice disallows grid positions (`h`, `v`) with 
    `v` even and `h` a multiple of 3 or
    `v` odd and `h` = 1 + a multiple of 3.  
    
    However, we use the same convention as cadnano for encoding hex coordinates see `misc/cadnano-format-specs/v2.txt`.
    That convention is different from simply excluding coordinates from the hex lattice.

    """

    none = "none"
    """No fixed grid."""


# convenience names for users
reverse = False
forward = True
square = Grid.square
hexagonal = Grid.hex  # should not use identifier "hex" because that's a Python built-in function
honeycomb = Grid.honeycomb

##########################################################################
# constants

current_version: str = "0.6.8"
initial_version: str = "0.1.0"

default_idt_scale = "25nm"
default_idt_purification = "STD"


def default_major_tick_distance(grid: Grid) -> int:
    return 7 if grid in (Grid.hex, Grid.honeycomb) else 8


default_grid: Grid = Grid.square

default_helix_rotation: float = -90.0
default_helix_rotation_anchor: int = 0

base_width_svg: float = 10.0
"""Width of a single base in the SVG main view of scadnano."""

base_height_svg: float = 10.0
"""Height of a single base in the SVG main view of scadnano."""

distance_between_helices_svg: float = (base_width_svg * 2.5 / 0.34)
"""Distance between tops of two consecutive helices (using default positioning rules).

This is set to (:const:`base_width_svg` * 2.5/0.34) based on the following calculation,
to attempt to make the DNA structure appear to scale in 2D drawings:
The width of one base pair of double-stranded DNA bp is 0.34 nm. In a DNA origami, 
AFM images let us estimate that the average distance between adjacent double helices is 2.5 nm.
(A DNA double-helix is only 2 nm wide, but the helices electrostatically repel each other so the spacing
in a DNA origami or an other DNA nanostructure with many parallel DNA helices---e.g., single-stranded tile
lattices---is larger than 2 nm.)
Thus the distance between the helices is 2.5/0.34 ~ 7.5 times the width of a single DNA base.
"""

DNA_base_wildcard: str = '?'
"""Symbol to insert when a DNA sequence has been assigned to a strand through complementarity, but
some regions of the strand are not bound to the strand that was just assigned. Also used in case the
DNA sequence assigned to a strand is too short; the sequence is padded with :any:`DNA_base_wildcard` to 
make its length the same as the length of the strand."""


def _rotate_string(string: str, rotation: int):
    rotation = rotation % len(string)
    return string[rotation:] + string[:rotation]


class M13Variant(enum.Enum):
    """Variants of M13mp18 viral genome. "Standard" variant is p7249. Other variants are longer."""

    p7249 = "p7249"
    """"Standard" variant of M13mp18; 7249 bases long."""

    p7560 = "p7560"
    """Variant of M13mp18 that is 7560 bases long."""

    p8064 = "p8064"
    """Variant of M13mp18 that is 8064 bases long."""


def m13(rotation: int = 5587, variant: M13Variant = M13Variant.p7249):
    """
    The M13mp18 DNA sequence (commonly called simply M13).
    
    By default, starts from cyclic rotation 5587 
    (with 0-based indexing;  commonly this is called rotation 5588, which assumes that indexing begins at 1), 
    as defined in
    `GenBank <https://www.ncbi.nlm.nih.gov/nuccore/X02513.1>`_.
    
    By default, returns the "standard" variant of consisting of 7249 bases, sold by companies such as  
    `New England Biolabs <https://www.neb.com/~/media/nebus/page%20images/tools%20and%20resources/interactive%20tools/dna%20sequences%20and%20maps/m13mp18_map.pdf>`_
    and
    `Tilibit <https://cdn.shopify.com/s/files/1/1299/5863/files/Product_Sheet_single-stranded_scaffold_DNA_type_7249_M1-10.pdf?14656642867652657391>`_.
    
    The actual M13 DNA strand itself is circular, 
    so assigning this sequence to the scaffold :any:`Strand` in a :any:`DNADesign`
    means that the "5' end" of the scaffold :any:`Strand` 
    (which is a fiction since the actual circular DNA strand has no endpoint) 
    will have the sequence starting at position 5587 starting at the displayed 5' in scadnano,
    assigned until the displayed 3' end. 
    Assuming the displayed scaffold :any:`Strand` has length :math:`n < 7249`, then a loopout of length 
    :math:`7249 - n` consisting of the undisplayed bases will be present in the actual DNA structure.
    For a more detailed discussion of why this particular rotation of M13 is chosen,
    see 
    `Supplementary Note S8 <http://www.dna.caltech.edu/Papers/DNAorigami-supp1.linux.pdf>`_ 
    in
    [`Folding DNA to create nanoscale shapes and patterns. Paul W. K. Rothemund, Nature 440:297-302 (2006) <http://www.nature.com/nature/journal/v440/n7082/abs/nature04586.html>`_].
    
    :param rotation: rotation of circular strand. Valid values are 0 through length-1.
    :param variant: variant of M13 strand to use
    :return: M13 strand sequence
    """  # noqa (suppress PEP warning)
    seq = _m13_variants[variant]
    return _rotate_string(seq, rotation)


_7249 = "" \
        "AATGCTACTACTATTAGTAGAATTGATGCCACCTTTTCAGCTCGCGCCCCAAATGAAAATATAGCTAAACAGGTTATTGACCATTTGCGAAATGTATCTA" \
        "ATGGTCAAACTAAATCTACTCGTTCGCAGAATTGGGAATCAACTGTTATATGGAATGAAACTTCCAGACACCGTACTTTAGTTGCATATTTAAAACATGT" \
        "TGAGCTACAGCATTATATTCAGCAATTAAGCTCTAAGCCATCCGCAAAAATGACCTCTTATCAAAAGGAGCAATTAAAGGTACTCTCTAATCCTGACCTG" \
        "TTGGAGTTTGCTTCCGGTCTGGTTCGCTTTGAAGCTCGAATTAAAACGCGATATTTGAAGTCTTTCGGGCTTCCTCTTAATCTTTTTGATGCAATCCGCT" \
        "TTGCTTCTGACTATAATAGTCAGGGTAAAGACCTGATTTTTGATTTATGGTCATTCTCGTTTTCTGAACTGTTTAAAGCATTTGAGGGGGATTCAATGAA" \
        "TATTTATGACGATTCCGCAGTATTGGACGCTATCCAGTCTAAACATTTTACTATTACCCCCTCTGGCAAAACTTCTTTTGCAAAAGCCTCTCGCTATTTT" \
        "GGTTTTTATCGTCGTCTGGTAAACGAGGGTTATGATAGTGTTGCTCTTACTATGCCTCGTAATTCCTTTTGGCGTTATGTATCTGCATTAGTTGAATGTG" \
        "GTATTCCTAAATCTCAACTGATGAATCTTTCTACCTGTAATAATGTTGTTCCGTTAGTTCGTTTTATTAACGTAGATTTTTCTTCCCAACGTCCTGACTG" \
        "GTATAATGAGCCAGTTCTTAAAATCGCATAAGGTAATTCACAATGATTAAAGTTGAAATTAAACCATCTCAAGCCCAATTTACTACTCGTTCTGGTGTTT" \
        "CTCGTCAGGGCAAGCCTTATTCACTGAATGAGCAGCTTTGTTACGTTGATTTGGGTAATGAATATCCGGTTCTTGTCAAGATTACTCTTGATGAAGGTCA" \
        "GCCAGCCTATGCGCCTGGTCTGTACACCGTTCATCTGTCCTCTTTCAAAGTTGGTCAGTTCGGTTCCCTTATGATTGACCGTCTGCGCCTCGTTCCGGCT" \
        "AAGTAACATGGAGCAGGTCGCGGATTTCGACACAATTTATCAGGCGATGATACAAATCTCCGTTGTACTTTGTTTCGCGCTTGGTATAATCGCTGGGGGT" \
        "CAAAGATGAGTGTTTTAGTGTATTCTTTTGCCTCTTTCGTTTTAGGTTGGTGCCTTCGTAGTGGCATTACGTATTTTACCCGTTTAATGGAAACTTCCTC" \
        "ATGAAAAAGTCTTTAGTCCTCAAAGCCTCTGTAGCCGTTGCTACCCTCGTTCCGATGCTGTCTTTCGCTGCTGAGGGTGACGATCCCGCAAAAGCGGCCT" \
        "TTAACTCCCTGCAAGCCTCAGCGACCGAATATATCGGTTATGCGTGGGCGATGGTTGTTGTCATTGTCGGCGCAACTATCGGTATCAAGCTGTTTAAGAA" \
        "ATTCACCTCGAAAGCAAGCTGATAAACCGATACAATTAAAGGCTCCTTTTGGAGCCTTTTTTTTGGAGATTTTCAACGTGAAAAAATTATTATTCGCAAT" \
        "TCCTTTAGTTGTTCCTTTCTATTCTCACTCCGCTGAAACTGTTGAAAGTTGTTTAGCAAAATCCCATACAGAAAATTCATTTACTAACGTCTGGAAAGAC" \
        "GACAAAACTTTAGATCGTTACGCTAACTATGAGGGCTGTCTGTGGAATGCTACAGGCGTTGTAGTTTGTACTGGTGACGAAACTCAGTGTTACGGTACAT" \
        "GGGTTCCTATTGGGCTTGCTATCCCTGAAAATGAGGGTGGTGGCTCTGAGGGTGGCGGTTCTGAGGGTGGCGGTTCTGAGGGTGGCGGTACTAAACCTCC" \
        "TGAGTACGGTGATACACCTATTCCGGGCTATACTTATATCAACCCTCTCGACGGCACTTATCCGCCTGGTACTGAGCAAAACCCCGCTAATCCTAATCCT" \
        "TCTCTTGAGGAGTCTCAGCCTCTTAATACTTTCATGTTTCAGAATAATAGGTTCCGAAATAGGCAGGGGGCATTAACTGTTTATACGGGCACTGTTACTC" \
        "AAGGCACTGACCCCGTTAAAACTTATTACCAGTACACTCCTGTATCATCAAAAGCCATGTATGACGCTTACTGGAACGGTAAATTCAGAGACTGCGCTTT" \
        "CCATTCTGGCTTTAATGAGGATTTATTTGTTTGTGAATATCAAGGCCAATCGTCTGACCTGCCTCAACCTCCTGTCAATGCTGGCGGCGGCTCTGGTGGT" \
        "GGTTCTGGTGGCGGCTCTGAGGGTGGTGGCTCTGAGGGTGGCGGTTCTGAGGGTGGCGGCTCTGAGGGAGGCGGTTCCGGTGGTGGCTCTGGTTCCGGTG" \
        "ATTTTGATTATGAAAAGATGGCAAACGCTAATAAGGGGGCTATGACCGAAAATGCCGATGAAAACGCGCTACAGTCTGACGCTAAAGGCAAACTTGATTC" \
        "TGTCGCTACTGATTACGGTGCTGCTATCGATGGTTTCATTGGTGACGTTTCCGGCCTTGCTAATGGTAATGGTGCTACTGGTGATTTTGCTGGCTCTAAT" \
        "TCCCAAATGGCTCAAGTCGGTGACGGTGATAATTCACCTTTAATGAATAATTTCCGTCAATATTTACCTTCCCTCCCTCAATCGGTTGAATGTCGCCCTT" \
        "TTGTCTTTGGCGCTGGTAAACCATATGAATTTTCTATTGATTGTGACAAAATAAACTTATTCCGTGGTGTCTTTGCGTTTCTTTTATATGTTGCCACCTT" \
        "TATGTATGTATTTTCTACGTTTGCTAACATACTGCGTAATAAGGAGTCTTAATCATGCCAGTTCTTTTGGGTATTCCGTTATTATTGCGTTTCCTCGGTT" \
        "TCCTTCTGGTAACTTTGTTCGGCTATCTGCTTACTTTTCTTAAAAAGGGCTTCGGTAAGATAGCTATTGCTATTTCATTGTTTCTTGCTCTTATTATTGG" \
        "GCTTAACTCAATTCTTGTGGGTTATCTCTCTGATATTAGCGCTCAATTACCCTCTGACTTTGTTCAGGGTGTTCAGTTAATTCTCCCGTCTAATGCGCTT" \
        "CCCTGTTTTTATGTTATTCTCTCTGTAAAGGCTGCTATTTTCATTTTTGACGTTAAACAAAAAATCGTTTCTTATTTGGATTGGGATAAATAATATGGCT" \
        "GTTTATTTTGTAACTGGCAAATTAGGCTCTGGAAAGACGCTCGTTAGCGTTGGTAAGATTCAGGATAAAATTGTAGCTGGGTGCAAAATAGCAACTAATC" \
        "TTGATTTAAGGCTTCAAAACCTCCCGCAAGTCGGGAGGTTCGCTAAAACGCCTCGCGTTCTTAGAATACCGGATAAGCCTTCTATATCTGATTTGCTTGC" \
        "TATTGGGCGCGGTAATGATTCCTACGATGAAAATAAAAACGGCTTGCTTGTTCTCGATGAGTGCGGTACTTGGTTTAATACCCGTTCTTGGAATGATAAG" \
        "GAAAGACAGCCGATTATTGATTGGTTTCTACATGCTCGTAAATTAGGATGGGATATTATTTTTCTTGTTCAGGACTTATCTATTGTTGATAAACAGGCGC" \
        "GTTCTGCATTAGCTGAACATGTTGTTTATTGTCGTCGTCTGGACAGAATTACTTTACCTTTTGTCGGTACTTTATATTCTCTTATTACTGGCTCGAAAAT" \
        "GCCTCTGCCTAAATTACATGTTGGCGTTGTTAAATATGGCGATTCTCAATTAAGCCCTACTGTTGAGCGTTGGCTTTATACTGGTAAGAATTTGTATAAC" \
        "GCATATGATACTAAACAGGCTTTTTCTAGTAATTATGATTCCGGTGTTTATTCTTATTTAACGCCTTATTTATCACACGGTCGGTATTTCAAACCATTAA" \
        "ATTTAGGTCAGAAGATGAAATTAACTAAAATATATTTGAAAAAGTTTTCTCGCGTTCTTTGTCTTGCGATTGGATTTGCATCAGCATTTACATATAGTTA" \
        "TATAACCCAACCTAAGCCGGAGGTTAAAAAGGTAGTCTCTCAGACCTATGATTTTGATAAATTCACTATTGACTCTTCTCAGCGTCTTAATCTAAGCTAT" \
        "CGCTATGTTTTCAAGGATTCTAAGGGAAAATTAATTAATAGCGACGATTTACAGAAGCAAGGTTATTCACTCACATATATTGATTTATGTACTGTTTCCA" \
        "TTAAAAAAGGTAATTCAAATGAAATTGTTAAATGTAATTAATTTTGTTTTCTTGATGTTTGTTTCATCATCTTCTTTTGCTCAGGTAATTGAAATGAATA" \
        "ATTCGCCTCTGCGCGATTTTGTAACTTGGTATTCAAAGCAATCAGGCGAATCCGTTATTGTTTCTCCCGATGTAAAAGGTACTGTTACTGTATATTCATC" \
        "TGACGTTAAACCTGAAAATCTACGCAATTTCTTTATTTCTGTTTTACGTGCAAATAATTTTGATATGGTAGGTTCTAACCCTTCCATTATTCAGAAGTAT" \
        "AATCCAAACAATCAGGATTATATTGATGAATTGCCATCATCTGATAATCAGGAATATGATGATAATTCCGCTCCTTCTGGTGGTTTCTTTGTTCCGCAAA" \
        "ATGATAATGTTACTCAAACTTTTAAAATTAATAACGTTCGGGCAAAGGATTTAATACGAGTTGTCGAATTGTTTGTAAAGTCTAATACTTCTAAATCCTC" \
        "AAATGTATTATCTATTGACGGCTCTAATCTATTAGTTGTTAGTGCTCCTAAAGATATTTTAGATAACCTTCCTCAATTCCTTTCAACTGTTGATTTGCCA" \
        "ACTGACCAGATATTGATTGAGGGTTTGATATTTGAGGTTCAGCAAGGTGATGCTTTAGATTTTTCATTTGCTGCTGGCTCTCAGCGTGGCACTGTTGCAG" \
        "GCGGTGTTAATACTGACCGCCTCACCTCTGTTTTATCTTCTGCTGGTGGTTCGTTCGGTATTTTTAATGGCGATGTTTTAGGGCTATCAGTTCGCGCATT" \
        "AAAGACTAATAGCCATTCAAAAATATTGTCTGTGCCACGTATTCTTACGCTTTCAGGTCAGAAGGGTTCTATCTCTGTTGGCCAGAATGTCCCTTTTATT" \
        "ACTGGTCGTGTGACTGGTGAATCTGCCAATGTAAATAATCCATTTCAGACGATTGAGCGTCAAAATGTAGGTATTTCCATGAGCGTTTTTCCTGTTGCAA" \
        "TGGCTGGCGGTAATATTGTTCTGGATATTACCAGCAAGGCCGATAGTTTGAGTTCTTCTACTCAGGCAAGTGATGTTATTACTAATCAAAGAAGTATTGC" \
        "TACAACGGTTAATTTGCGTGATGGACAGACTCTTTTACTCGGTGGCCTCACTGATTATAAAAACACTTCTCAGGATTCTGGCGTACCGTTCCTGTCTAAA" \
        "ATCCCTTTAATCGGCCTCCTGTTTAGCTCCCGCTCTGATTCTAACGAGGAAAGCACGTTATACGTGCTCGTCAAAGCAACCATAGTACGCGCCCTGTAGC" \
        "GGCGCATTAAGCGCGGCGGGTGTGGTGGTTACGCGCAGCGTGACCGCTACACTTGCCAGCGCCCTAGCGCCCGCTCCTTTCGCTTTCTTCCCTTCCTTTC" \
        "TCGCCACGTTCGCCGGCTTTCCCCGTCAAGCTCTAAATCGGGGGCTCCCTTTAGGGTTCCGATTTAGTGCTTTACGGCACCTCGACCCCAAAAAACTTGA" \
        "TTTGGGTGATGGTTCACGTAGTGGGCCATCGCCCTGATAGACGGTTTTTCGCCCTTTGACGTTGGAGTCCACGTTCTTTAATAGTGGACTCTTGTTCCAA" \
        "ACTGGAACAACACTCAACCCTATCTCGGGCTATTCTTTTGATTTATAAGGGATTTTGCCGATTTCGGAACCACCATCAAACAGGATTTTCGCCTGCTGGG" \
        "GCAAACCAGCGTGGACCGCTTGCTGCAACTCTCTCAGGGCCAGGCGGTGAAGGGCAATCAGCTGTTGCCCGTCTCACTGGTGAAAAGAAAAACCACCCTG" \
        "GCGCCCAATACGCAAACCGCCTCTCCCCGCGCGTTGGCCGATTCATTAATGCAGCTGGCACGACAGGTTTCCCGACTGGAAAGCGGGCAGTGAGCGCAAC" \
        "GCAATTAATGTGAGTTAGCTCACTCATTAGGCACCCCAGGCTTTACACTTTATGCTTCCGGCTCGTATGTTGTGTGGAATTGTGAGCGGATAACAATTTC" \
        "ACACAGGAAACAGCTATGACCATGATTACGAATTCGAGCTCGGTACCCGGGGATCCTCTAGAGTCGACCTGCAGGCATGCAAGCTTGGCACTGGCCGTCG" \
        "TTTTACAACGTCGTGACTGGGAAAACCCTGGCGTTACCCAACTTAATCGCCTTGCAGCACATCCCCCTTTCGCCAGCTGGCGTAATAGCGAAGAGGCCCG" \
        "CACCGATCGCCCTTCCCAACAGTTGCGCAGCCTGAATGGCGAATGGCGCTTTGCCTGGTTTCCGGCACCAGAAGCGGTGCCGGAAAGCTGGCTGGAGTGC" \
        "GATCTTCCTGAGGCCGATACTGTCGTCGTCCCCTCAAACTGGCAGATGCACGGTTACGATGCGCCCATCTACACCAACGTGACCTATCCCATTACGGTCA" \
        "ATCCGCCGTTTGTTCCCACGGAGAATCCGACGGGTTGTTACTCGCTCACATTTAATGTTGATGAAAGCTGGCTACAGGAAGGCCAGACGCGAATTATTTT" \
        "TGATGGCGTTCCTATTGGTTAAAAAATGAGCTGATTTAACAAAAATTTAATGCGAATTTTAACAAAATATTAACGTTTACAATTTAAATATTTGCTTATA" \
        "CAATCTTCCTGTTTTTGGGGCTTTTCTGATTATCAACCGGGGTACATATGATTGACATGCTAGTTTTACGATTACCGTTCATCGATTCTCTTGTTTGCTC" \
        "CAGACTCTCAGGCAATGACCTGATAGCCTTTGTAGATCTCTCAAAAATAGCTACCCTCTCCGGCATTAATTTATCAGCTAGAACGGTTGAATATCATATT" \
        "GATGGTGATTTGACTGTCTCCGGCCTTTCTCACCCTTTTGAATCTTTACCTACACATTACTCAGGCATTGCATTTAAAATATATGAGGGTTCTAAAAATT" \
        "TTTATCCTTGCGTTGAAATAAAGGCTTCTCCCGCAAAAGTATTACAGGGTCATAATGTTTTTGGTACAACCGATTTAGCTTTATGCTCTGAGGCTTTATT" \
        "GCTTAATTTTGCTAATTCTTTGCCTTGCCTGTATGATTTATTGGATGTT"

_7560 = "" \
        "AGCTTGGCACTGGCCGTCGTTTTACAACGTCGTGACTGGGAAAACCCTGGCGTTACCCAACTTAATCGCCTTGCAGCACATCCCCCTTTCGCCAGCTGGC" \
        "GTAATAGCGAAGAGGCCCGCACCGATCGCCCTTCCCAACAGTTGCGCAGCCTGAATGGCGAATGGCGCTTTGCCTGGTTTCCGGCACCAGAAGCGGTGCC" \
        "GGAAAGCTGGCTGGAGTGCGATCTTCCTGAGGCCGATACTGTCGTCGTCCCCTCAAACTGGCAGATGCACGGTTACGATGCGCCCATCTACACCAACGTG" \
        "ACCTATCCCATTACGGTCAATCCGCCGTTTGTTCCCACGGAGAATCCGACGGGTTGTTACTCGCTCACATTTAATGTTGATGAAAGCTGGCTACAGGAAG" \
        "GCCAGACGCGAATTATTTTTGATGGCGTTCCTATTGGTTAAAAAATGAGCTGATTTAACAAAAATTTAATGCGAATTTTAACAAAATATTAACGTTTACA" \
        "ATTTAAATATTTGCTTATACAATCTTCCTGTTTTTGGGGCTTTTCTGATTATCAACCGGGGTACATATGATTGACATGCTAGTTTTACGATTACCGTTCA" \
        "TCGATTCTCTTGTTTGCTCCAGACTCTCAGGCAATGACCTGATAGCCTTTGTAGATCTCTCAAAAATAGCTACCCTCTCCGGCATTAATTTATCAGCTAG" \
        "AACGGTTGAATATCATATTGATGGTGATTTGACTGTCTCCGGCCTTTCTCACCCTTTTGAATCTTTACCTACACATTACTCAGGCATTGCATTTAAAATA" \
        "TATGAGGGTTCTAAAAATTTTTATCCTTGCGTTGAAATAAAGGCTTCTCCCGCAAAAGTATTACAGGGTCATAATGTTTTTGGTACAACCGATTTAGCTT" \
        "TATGCTCTGAGGCTTTATTGCTTAATTTTGCTAATTCTTTGCCTTGCCTGTATGATTTATTGGATGTTAATGCTACTACTATTAGTAGAATTGATGCCAC" \
        "CTTTTCAGCTCGCGCCCCAAATGAAAATATAGCTAAACAGGTTATTGACCATTTGCGAAATGTATCTAATGGTCAAACTAAATCTACTCGTTCGCAGAAT" \
        "TGGGAATCAACTGTTATATGGAATGAAACTTCCAGACACCGTACTTTAGTTGCATATTTAAAACATGTTGAGCTACAGCATTATATTCAGCAATTAAGCT" \
        "CTAAGCCATCCGCAAAAATGACCTCTTATCAAAAGGAGCAATTAAAGGTACTCTCTAATCCTGACCTGTTGGAGTTTGCTTCCGGTCTGGTTCGCTTTGA" \
        "AGCTCGAATTAAAACGCGATATTTGAAGTCTTTCGGGCTTCCTCTTAATCTTTTTGATGCAATCCGCTTTGCTTCTGACTATAATAGTCAGGGTAAAGAC" \
        "CTGATTTTTGATTTATGGTCATTCTCGTTTTCTGAACTGTTTAAAGCATTTGAGGGGGATTCAATGAATATTTATGACGATTCCGCAGTATTGGACGCTA" \
        "TCCAGTCTAAACATTTTACTATTACCCCCTCTGGCAAAACTTCTTTTGCAAAAGCCTCTCGCTATTTTGGTTTTTATCGTCGTCTGGTAAACGAGGGTTA" \
        "TGATAGTGTTGCTCTTACTATGCCTCGTAATTCCTTTTGGCGTTATGTATCTGCATTAGTTGAATGTGGTATTCCTAAATCTCAACTGATGAATCTTTCT" \
        "ACCTGTAATAATGTTGTTCCGTTAGTTCGTTTTATTAACGTAGATTTTTCTTCCCAACGTCCTGACTGGTATAATGAGCCAGTTCTTAAAATCGCATAAG" \
        "GTAATTCACAATGATTAAAGTTGAAATTAAACCATCTCAAGCCCAATTTACTACTCGTTCTGGTGTTTCTCGTCAGGGCAAGCCTTATTCACTGAATGAG" \
        "CAGCTTTGTTACGTTGATTTGGGTAATGAATATCCGGTTCTTGTCAAGATTACTCTTGATGAAGGTCAGCCAGCCTATGCGCCTGGTCTGTACACCGTTC" \
        "ATCTGTCCTCTTTCAAAGTTGGTCAGTTCGGTTCCCTTATGATTGACCGTCTGCGCCTCGTTCCGGCTAAGTAACATGGAGCAGGTCGCGGATTTCGACA" \
        "CAATTTATCAGGCGATGATACAAATCTCCGTTGTACTTTGTTTCGCGCTTGGTATAATCGCTGGGGGTCAAAGATGAGTGTTTTAGTGTATTCTTTTGCC" \
        "TCTTTCGTTTTAGGTTGGTGCCTTCGTAGTGGCATTACGTATTTTACCCGTTTAATGGAAACTTCCTCATGAAAAAGTCTTTAGTCCTCAAAGCCTCTGT" \
        "AGCCGTTGCTACCCTCGTTCCGATGCTGTCTTTCGCTGCTGAGGGTGACGATCCCGCAAAAGCGGCCTTTAACTCCCTGCAAGCCTCAGCGACCGAATAT" \
        "ATCGGTTATGCGTGGGCGATGGTTGTTGTCATTGTCGGCGCAACTATCGGTATCAAGCTGTTTAAGAAATTCACCTCGAAAGCAAGCTGATAAACCGATA" \
        "CAATTAAAGGCTCCTTTTGGAGCCTTTTTTTTGGAGATTTTCAACGTGAAAAAATTATTATTCGCAATTCCTTTAGTTGTTCCTTTCTATTCTCACTCCG" \
        "CTGAAACTGTTGAAAGTTGTTTAGCAAAATCCCATACAGAAAATTCATTTACTAACGTCTGGAAAGACGACAAAACTTTAGATCGTTACGCTAACTATGA" \
        "GGGCTGTCTGTGGAATGCTACAGGCGTTGTAGTTTGTACTGGTGACGAAACTCAGTGTTACGGTACATGGGTTCCTATTGGGCTTGCTATCCCTGAAAAT" \
        "GAGGGTGGTGGCTCTGAGGGTGGCGGTTCTGAGGGTGGCGGTTCTGAGGGTGGCGGTACTAAACCTCCTGAGTACGGTGATACACCTATTCCGGGCTATA" \
        "CTTATATCAACCCTCTCGACGGCACTTATCCGCCTGGTACTGAGCAAAACCCCGCTAATCCTAATCCTTCTCTTGAGGAGTCTCAGCCTCTTAATACTTT" \
        "CATGTTTCAGAATAATAGGTTCCGAAATAGGCAGGGGGCATTAACTGTTTATACGGGCACTGTTACTCAAGGCACTGACCCCGTTAAAACTTATTACCAG" \
        "TACACTCCTGTATCATCAAAAGCCATGTATGACGCTTACTGGAACGGTAAATTCAGAGACTGCGCTTTCCATTCTGGCTTTAATGAGGATTTATTTGTTT" \
        "GTGAATATCAAGGCCAATCGTCTGACCTGCCTCAACCTCCTGTCAATGCTGGCGGCGGCTCTGGTGGTGGTTCTGGTGGCGGCTCTGAGGGTGGTGGCTC" \
        "TGAGGGTGGCGGTTCTGAGGGTGGCGGCTCTGAGGGAGGCGGTTCCGGTGGTGGCTCTGGTTCCGGTGATTTTGATTATGAAAAGATGGCAAACGCTAAT" \
        "AAGGGGGCTATGACCGAAAATGCCGATGAAAACGCGCTACAGTCTGACGCTAAAGGCAAACTTGATTCTGTCGCTACTGATTACGGTGCTGCTATCGATG" \
        "GTTTCATTGGTGACGTTTCCGGCCTTGCTAATGGTAATGGTGCTACTGGTGATTTTGCTGGCTCTAATTCCCAAATGGCTCAAGTCGGTGACGGTGATAA" \
        "TTCACCTTTAATGAATAATTTCCGTCAATATTTACCTTCCCTCCCTCAATCGGTTGAATGTCGCCCTTTTGTCTTTGGCGCTGGTAAACCATATGAATTT" \
        "TCTATTGATTGTGACAAAATAAACTTATTCCGTGGTGTCTTTGCGTTTCTTTTATATGTTGCCACCTTTATGTATGTATTTTCTACGTTTGCTAACATAC" \
        "TGCGTAATAAGGAGTCTTAATCATGCCAGTTCTTTTGGGTATTCCGTTATTATTGCGTTTCCTCGGTTTCCTTCTGGTAACTTTGTTCGGCTATCTGCTT" \
        "ACTTTTCTTAAAAAGGGCTTCGGTAAGATAGCTATTGCTATTTCATTGTTTCTTGCTCTTATTATTGGGCTTAACTCAATTCTTGTGGGTTATCTCTCTG" \
        "ATATTAGCGCTCAATTACCCTCTGACTTTGTTCAGGGTGTTCAGTTAATTCTCCCGTCTAATGCGCTTCCCTGTTTTTATGTTATTCTCTCTGTAAAGGC" \
        "TGCTATTTTCATTTTTGACGTTAAACAAAAAATCGTTTCTTATTTGGATTGGGATAAATAATATGGCTGTTTATTTTGTAACTGGCAAATTAGGCTCTGG" \
        "AAAGACGCTCGTTAGCGTTGGTAAGATTCAGGATAAAATTGTAGCTGGGTGCAAAATAGCAACTAATCTTGATTTAAGGCTTCAAAACCTCCCGCAAGTC" \
        "GGGAGGTTCGCTAAAACGCCTCGCGTTCTTAGAATACCGGATAAGCCTTCTATATCTGATTTGCTTGCTATTGGGCGCGGTAATGATTCCTACGATGAAA" \
        "ATAAAAACGGCTTGCTTGTTCTCGATGAGTGCGGTACTTGGTTTAATACCCGTTCTTGGAATGATAAGGAAAGACAGCCGATTATTGATTGGTTTCTACA" \
        "TGCTCGTAAATTAGGATGGGATATTATTTTTCTTGTTCAGGACTTATCTATTGTTGATAAACAGGCGCGTTCTGCATTAGCTGAACATGTTGTTTATTGT" \
        "CGTCGTCTGGACAGAATTACTTTACCTTTTGTCGGTACTTTATATTCTCTTATTACTGGCTCGAAAATGCCTCTGCCTAAATTACATGTTGGCGTTGTTA" \
        "AATATGGCGATTCTCAATTAAGCCCTACTGTTGAGCGTTGGCTTTATACTGGTAAGAATTTGTATAACGCATATGATACTAAACAGGCTTTTTCTAGTAA" \
        "TTATGATTCCGGTGTTTATTCTTATTTAACGCCTTATTTATCACACGGTCGGTATTTCAAACCATTAAATTTAGGTCAGAAGATGAAATTAACTAAAATA" \
        "TATTTGAAAAAGTTTTCTCGCGTTCTTTGTCTTGCGATTGGATTTGCATCAGCATTTACATATAGTTATATAACCCAACCTAAGCCGGAGGTTAAAAAGG" \
        "TAGTCTCTCAGACCTATGATTTTGATAAATTCACTATTGACTCTTCTCAGCGTCTTAATCTAAGCTATCGCTATGTTTTCAAGGATTCTAAGGGAAAATT" \
        "AATTAATAGCGACGATTTACAGAAGCAAGGTTATTCACTCACATATATTGATTTATGTACTGTTTCCATTAAAAAAGGTAATTCAAATGAAATTGTTAAA" \
        "TGTAATTAATTTTGTTTTCTTGATGTTTGTTTCATCATCTTCTTTTGCTCAGGTAATTGAAATGAATAATTCGCCTCTGCGCGATTTTGTAACTTGGTAT" \
        "TCAAAGCAATCAGGCGAATCCGTTATTGTTTCTCCCGATGTAAAAGGTACTGTTACTGTATATTCATCTGACGTTAAACCTGAAAATCTACGCAATTTCT" \
        "TTATTTCTGTTTTACGTGCAAATAATTTTGATATGGTAGGTTCTAACCCTTCCATTATTCAGAAGTATAATCCAAACAATCAGGATTATATTGATGAATT" \
        "GCCATCATCTGATAATCAGGAATATGATGATAATTCCGCTCCTTCTGGTGGTTTCTTTGTTCCGCAAAATGATAATGTTACTCAAACTTTTAAAATTAAT" \
        "AACGTTCGGGCAAAGGATTTAATACGAGTTGTCGAATTGTTTGTAAAGTCTAATACTTCTAAATCCTCAAATGTATTATCTATTGACGGCTCTAATCTAT" \
        "TAGTTGTTAGTGCTCCTAAAGATATTTTAGATAACCTTCCTCAATTCCTTTCAACTGTTGATTTGCCAACTGACCAGATATTGATTGAGGGTTTGATATT" \
        "TGAGGTTCAGCAAGGTGATGCTTTAGATTTTTCATTTGCTGCTGGCTCTCAGCGTGGCACTGTTGCAGGCGGTGTTAATACTGACCGCCTCACCTCTGTT" \
        "TTATCTTCTGCTGGTGGTTCGTTCGGTATTTTTAATGGCGATGTTTTAGGGCTATCAGTTCGCGCATTAAAGACTAATAGCCATTCAAAAATATTGTCTG" \
        "TGCCACGTATTCTTACGCTTTCAGGTCAGAAGGGTTCTATCTCTGTTGGCCAGAATGTCCCTTTTATTACTGGTCGTGTGACTGGTGAATCTGCCAATGT" \
        "AAATAATCCATTTCAGACGATTGAGCGTCAAAATGTAGGTATTTCCATGAGCGTTTTTCCTGTTGCAATGGCTGGCGGTAATATTGTTCTGGATATTACC" \
        "AGCAAGGCCGATAGTTTGAGTTCTTCTACTCAGGCAAGTGATGTTATTACTAATCAAAGAAGTATTGCTACAACGGTTAATTTGCGTGATGGACAGACTC" \
        "TTTTACTCGGTGGCCTCACTGATTATAAAAACACTTCTCAGGATTCTGGCGTACCGTTCCTGTCTAAAATCCCTTTAATCGGCCTCCTGTTTAGCTCCCG" \
        "CTCTGATTCTAACGAGGAAAGCACGTTATACGTGCTCGTCAAAGCAACCATAGTACGCGCCCTGTAGCGGCGCATTAAGCGCGGCGGGTGTGGTGGTTAC" \
        "GCGCAGCGTGACCGCTACACTTGCCAGCGCCCTAGCGCCCGCTCCTTTCGCTTTCTTCCCTTCCTTTCTCGCCACGTTCGCCGGCTTTCCCCGTCAAGCT" \
        "CTAAATCGGGGGCTCCCTTTAGGGTTCCGATTTAGTGCTTTACGGCACCTCGACCCCAAAAAACTTGATTTGGGTGATGGTTCACGTAGTGGGCCATCGC" \
        "CCTGATAGACGGTTTTTCGCCCTTTGACGTTGGAGTCCACGTTCTTTAATAGTGGACTCTTGTTCCAAACTGGAACAACACTCAACCCTATCTCGGGCTA" \
        "TTCTTTTGATTTATAAGGGATTTTGCCGATTTCGGAACCACCATCAAACAGGATTTTCGCCTGCTGGGGCAAACCAGCGTGGACCGCTTGCTGCAACTCT" \
        "CTCAGGGCCAGGCGGTGAAGGGCAATCAGCTGTTGCCCGTCTCACTGGTGAAAAGAAAAACCACCCTGGCGCCCAATACGCAAACCGCCTCTCCCCGCGC" \
        "GTTGGCCGATTCATTAATGCAGCTGGCACGACAGGTTTCCCGACTGGAAAGCGGGCAGTGAGCGCAACGCAATTAATGTGAGTTAGCTCACTCATTAGGC" \
        "ACCCCAGGCTTTACACTTTATGCTTCCGGCTCGTATGTTGTGTGGAATTGTGAGCGGATAACAATTTCACACAGGAAACAGCTATGACCATGATTACGAA" \
        "TTCGAGCTCGGTACCCGGGGATCCTCCGTCTTTATCGAGGTAACAAGCACCACGTAGCTTAAGCCCTGTTTACTCATTACACCAACCAGGAGGTCAGAGT" \
        "TCGGAGAAATGATTTATGTGAAATGCGTCAGCCGATTCAAGGCCCCTATATTCGTGCCCACCGACGAGTTGCTTACAGATGGCAGGGCCGCACTGTCGGT" \
        "ATCATAGAGTCACTCCAGGGCGAGCGTAAATAGATTAGAAGCGGGGTTATTTTGGCGGGACATTGTCATAAGGTTGACAATTCAGCACTAAGGACACTTA" \
        "AGTCGTGCGCATGAATTCACAACCACTTAGAAGAACATCCACCCTGGCTTCTCCTGAGAA"

_8064 = "" \
        "GGCAATGACCTGATAGCCTTTGTAGATCTCTCAAAAATAGCTACCCTCTCCGGCATTAATTTATCAGCTAGAACGGTTGAATATCATATTGATGGTGATT" \
        "TGACTGTCTCCGGCCTTTCTCACCCTTTTGAATCTTTACCTACACATTACTCAGGCATTGCATTTAAAATATATGAGGGTTCTAAAAATTTTTATCCTTG" \
        "CGTTGAAATAAAGGCTTCTCCCGCAAAAGTATTACAGGGTCATAATGTTTTTGGTACAACCGATTTAGCTTTATGCTCTGAGGCTTTATTGCTTAATTTT" \
        "GCTAATTCTTTGCCTTGCCTGTATGATTTATTGGATGTTAATGCTACTACTATTAGTAGAATTGATGCCACCTTTTCAGCTCGCGCCCCAAATGAAAATA" \
        "TAGCTAAACAGGTTATTGACCATTTGCGAAATGTATCTAATGGTCAAACTAAATCTACTCGTTCGCAGAATTGGGAATCAACTGTTATATGGAATGAAAC" \
        "TTCCAGACACCGTACTTTAGTTGCATATTTAAAACATGTTGAGCTACAGCATTATATTCAGCAATTAAGCTCTAAGCCATCCGCAAAAATGACCTCTTAT" \
        "CAAAAGGAGCAATTAAAGGTACTCTCTAATCCTGACCTGTTGGAGTTTGCTTCCGGTCTGGTTCGCTTTGAAGCTCGAATTAAAACGCGATATTTGAAGT" \
        "CTTTCGGGCTTCCTCTTAATCTTTTTGATGCAATCCGCTTTGCTTCTGACTATAATAGTCAGGGTAAAGACCTGATTTTTGATTTATGGTCATTCTCGTT" \
        "TTCTGAACTGTTTAAAGCATTTGAGGGGGATTCAATGAATATTTATGACGATTCCGCAGTATTGGACGCTATCCAGTCTAAACATTTTACTATTACCCCC" \
        "TCTGGCAAAACTTCTTTTGCAAAAGCCTCTCGCTATTTTGGTTTTTATCGTCGTCTGGTAAACGAGGGTTATGATAGTGTTGCTCTTACTATGCCTCGTA" \
        "ATTCCTTTTGGCGTTATGTATCTGCATTAGTTGAATGTGGTATTCCTAAATCTCAACTGATGAATCTTTCTACCTGTAATAATGTTGTTCCGTTAGTTCG" \
        "TTTTATTAACGTAGATTTTTCTTCCCAACGTCCTGACTGGTATAATGAGCCAGTTCTTAAAATCGCATAAGGTAATTCACAATGATTAAAGTTGAAATTA" \
        "AACCATCTCAAGCCCAATTTACTACTCGTTCTGGTGTTTCTCGTCAGGGCAAGCCTTATTCACTGAATGAGCAGCTTTGTTACGTTGATTTGGGTAATGA" \
        "ATATCCGGTTCTTGTCAAGATTACTCTTGATGAAGGTCAGCCAGCCTATGCGCCTGGTCTGTACACCGTTCATCTGTCCTCTTTCAAAGTTGGTCAGTTC" \
        "GGTTCCCTTATGATTGACCGTCTGCGCCTCGTTCCGGCTAAGTAACATGGAGCAGGTCGCGGATTTCGACACAATTTATCAGGCGATGATACAAATCTCC" \
        "GTTGTACTTTGTTTCGCGCTTGGTATAATCGCTGGGGGTCAAAGATGAGTGTTTTAGTGTATTCTTTTGCCTCTTTCGTTTTAGGTTGGTGCCTTCGTAG" \
        "TGGCATTACGTATTTTACCCGTTTAATGGAAACTTCCTCATGAAAAAGTCTTTAGTCCTCAAAGCCTCTGTAGCCGTTGCTACCCTCGTTCCGATGCTGT" \
        "CTTTCGCTGCTGAGGGTGACGATCCCGCAAAAGCGGCCTTTAACTCCCTGCAAGCCTCAGCGACCGAATATATCGGTTATGCGTGGGCGATGGTTGTTGT" \
        "CATTGTCGGCGCAACTATCGGTATCAAGCTGTTTAAGAAATTCACCTCGAAAGCAAGCTGATAAACCGATACAATTAAAGGCTCCTTTTGGAGCCTTTTT" \
        "TTTGGAGATTTTCAACGTGAAAAAATTATTATTCGCAATTCCTTTAGTTGTTCCTTTCTATTCTCACTCCGCTGAAACTGTTGAAAGTTGTTTAGCAAAA" \
        "TCCCATACAGAAAATTCATTTACTAACGTCTGGAAAGACGACAAAACTTTAGATCGTTACGCTAACTATGAGGGCTGTCTGTGGAATGCTACAGGCGTTG" \
        "TAGTTTGTACTGGTGACGAAACTCAGTGTTACGGTACATGGGTTCCTATTGGGCTTGCTATCCCTGAAAATGAGGGTGGTGGCTCTGAGGGTGGCGGTTC" \
        "TGAGGGTGGCGGTTCTGAGGGTGGCGGTACTAAACCTCCTGAGTACGGTGATACACCTATTCCGGGCTATACTTATATCAACCCTCTCGACGGCACTTAT" \
        "CCGCCTGGTACTGAGCAAAACCCCGCTAATCCTAATCCTTCTCTTGAGGAGTCTCAGCCTCTTAATACTTTCATGTTTCAGAATAATAGGTTCCGAAATA" \
        "GGCAGGGGGCATTAACTGTTTATACGGGCACTGTTACTCAAGGCACTGACCCCGTTAAAACTTATTACCAGTACACTCCTGTATCATCAAAAGCCATGTA" \
        "TGACGCTTACTGGAACGGTAAATTCAGAGACTGCGCTTTCCATTCTGGCTTTAATGAGGATTTATTTGTTTGTGAATATCAAGGCCAATCGTCTGACCTG" \
        "CCTCAACCTCCTGTCAATGCTGGCGGCGGCTCTGGTGGTGGTTCTGGTGGCGGCTCTGAGGGTGGTGGCTCTGAGGGTGGCGGTTCTGAGGGTGGCGGCT" \
        "CTGAGGGAGGCGGTTCCGGTGGTGGCTCTGGTTCCGGTGATTTTGATTATGAAAAGATGGCAAACGCTAATAAGGGGGCTATGACCGAAAATGCCGATGA" \
        "AAACGCGCTACAGTCTGACGCTAAAGGCAAACTTGATTCTGTCGCTACTGATTACGGTGCTGCTATCGATGGTTTCATTGGTGACGTTTCCGGCCTTGCT" \
        "AATGGTAATGGTGCTACTGGTGATTTTGCTGGCTCTAATTCCCAAATGGCTCAAGTCGGTGACGGTGATAATTCACCTTTAATGAATAATTTCCGTCAAT" \
        "ATTTACCTTCCCTCCCTCAATCGGTTGAATGTCGCCCTTTTGTCTTTGGCGCTGGTAAACCATATGAATTTTCTATTGATTGTGACAAAATAAACTTATT" \
        "CCGTGGTGTCTTTGCGTTTCTTTTATATGTTGCCACCTTTATGTATGTATTTTCTACGTTTGCTAACATACTGCGTAATAAGGAGTCTTAATCATGCCAG" \
        "TTCTTTTGGGTATTCCGTTATTATTGCGTTTCCTCGGTTTCCTTCTGGTAACTTTGTTCGGCTATCTGCTTACTTTTCTTAAAAAGGGCTTCGGTAAGAT" \
        "AGCTATTGCTATTTCATTGTTTCTTGCTCTTATTATTGGGCTTAACTCAATTCTTGTGGGTTATCTCTCTGATATTAGCGCTCAATTACCCTCTGACTTT" \
        "GTTCAGGGTGTTCAGTTAATTCTCCCGTCTAATGCGCTTCCCTGTTTTTATGTTATTCTCTCTGTAAAGGCTGCTATTTTCATTTTTGACGTTAAACAAA" \
        "AAATCGTTTCTTATTTGGATTGGGATAAATAATATGGCTGTTTATTTTGTAACTGGCAAATTAGGCTCTGGAAAGACGCTCGTTAGCGTTGGTAAGATTC" \
        "AGGATAAAATTGTAGCTGGGTGCAAAATAGCAACTAATCTTGATTTAAGGCTTCAAAACCTCCCGCAAGTCGGGAGGTTCGCTAAAACGCCTCGCGTTCT" \
        "TAGAATACCGGATAAGCCTTCTATATCTGATTTGCTTGCTATTGGGCGCGGTAATGATTCCTACGATGAAAATAAAAACGGCTTGCTTGTTCTCGATGAG" \
        "TGCGGTACTTGGTTTAATACCCGTTCTTGGAATGATAAGGAAAGACAGCCGATTATTGATTGGTTTCTACATGCTCGTAAATTAGGATGGGATATTATTT" \
        "TTCTTGTTCAGGACTTATCTATTGTTGATAAACAGGCGCGTTCTGCATTAGCTGAACATGTTGTTTATTGTCGTCGTCTGGACAGAATTACTTTACCTTT" \
        "TGTCGGTACTTTATATTCTCTTATTACTGGCTCGAAAATGCCTCTGCCTAAATTACATGTTGGCGTTGTTAAATATGGCGATTCTCAATTAAGCCCTACT" \
        "GTTGAGCGTTGGCTTTATACTGGTAAGAATTTGTATAACGCATATGATACTAAACAGGCTTTTTCTAGTAATTATGATTCCGGTGTTTATTCTTATTTAA" \
        "CGCCTTATTTATCACACGGTCGGTATTTCAAACCATTAAATTTAGGTCAGAAGATGAAATTAACTAAAATATATTTGAAAAAGTTTTCTCGCGTTCTTTG" \
        "TCTTGCGATTGGATTTGCATCAGCATTTACATATAGTTATATAACCCAACCTAAGCCGGAGGTTAAAAAGGTAGTCTCTCAGACCTATGATTTTGATAAA" \
        "TTCACTATTGACTCTTCTCAGCGTCTTAATCTAAGCTATCGCTATGTTTTCAAGGATTCTAAGGGAAAATTAATTAATAGCGACGATTTACAGAAGCAAG" \
        "GTTATTCACTCACATATATTGATTTATGTACTGTTTCCATTAAAAAAGGTAATTCAAATGAAATTGTTAAATGTAATTAATTTTGTTTTCTTGATGTTTG" \
        "TTTCATCATCTTCTTTTGCTCAGGTAATTGAAATGAATAATTCGCCTCTGCGCGATTTTGTAACTTGGTATTCAAAGCAATCAGGCGAATCCGTTATTGT" \
        "TTCTCCCGATGTAAAAGGTACTGTTACTGTATATTCATCTGACGTTAAACCTGAAAATCTACGCAATTTCTTTATTTCTGTTTTACGTGCAAATAATTTT" \
        "GATATGGTAGGTTCTAACCCTTCCATTATTCAGAAGTATAATCCAAACAATCAGGATTATATTGATGAATTGCCATCATCTGATAATCAGGAATATGATG" \
        "ATAATTCCGCTCCTTCTGGTGGTTTCTTTGTTCCGCAAAATGATAATGTTACTCAAACTTTTAAAATTAATAACGTTCGGGCAAAGGATTTAATACGAGT" \
        "TGTCGAATTGTTTGTAAAGTCTAATACTTCTAAATCCTCAAATGTATTATCTATTGACGGCTCTAATCTATTAGTTGTTAGTGCTCCTAAAGATATTTTA" \
        "GATAACCTTCCTCAATTCCTTTCAACTGTTGATTTGCCAACTGACCAGATATTGATTGAGGGTTTGATATTTGAGGTTCAGCAAGGTGATGCTTTAGATT" \
        "TTTCATTTGCTGCTGGCTCTCAGCGTGGCACTGTTGCAGGCGGTGTTAATACTGACCGCCTCACCTCTGTTTTATCTTCTGCTGGTGGTTCGTTCGGTAT" \
        "TTTTAATGGCGATGTTTTAGGGCTATCAGTTCGCGCATTAAAGACTAATAGCCATTCAAAAATATTGTCTGTGCCACGTATTCTTACGCTTTCAGGTCAG" \
        "AAGGGTTCTATCTCTGTTGGCCAGAATGTCCCTTTTATTACTGGTCGTGTGACTGGTGAATCTGCCAATGTAAATAATCCATTTCAGACGATTGAGCGTC" \
        "AAAATGTAGGTATTTCCATGAGCGTTTTTCCTGTTGCAATGGCTGGCGGTAATATTGTTCTGGATATTACCAGCAAGGCCGATAGTTTGAGTTCTTCTAC" \
        "TCAGGCAAGTGATGTTATTACTAATCAAAGAAGTATTGCTACAACGGTTAATTTGCGTGATGGACAGACTCTTTTACTCGGTGGCCTCACTGATTATAAA" \
        "AACACTTCTCAGGATTCTGGCGTACCGTTCCTGTCTAAAATCCCTTTAATCGGCCTCCTGTTTAGCTCCCGCTCTGATTCTAACGAGGAAAGCACGTTAT" \
        "ACGTGCTCGTCAAAGCAACCATAGTACGCGCCCTGTAGCGGCGCATTAAGCGCGGCGGGTGTGGTGGTTACGCGCAGCGTGACCGCTACACTTGCCAGCG" \
        "CCCTAGCGCCCGCTCCTTTCGCTTTCTTCCCTTCCTTTCTCGCCACGTTCGCCGGCTTTCCCCGTCAAGCTCTAAATCGGGGGCTCCCTTTAGGGTTCCG" \
        "ATTTAGTGCTTTACGGCACCTCGACCCCAAAAAACTTGATTTGGGTGATGGTTCACGTAGTGGGCCATCGCCCTGATAGACGGTTTTTCGCCCTTTGACG" \
        "TTGGAGTCCACGTTCTTTAATAGTGGACTCTTGTTCCAAACTGGAACAACACTCAACCCTATCTCGGGCTATTCTTTTGATTTATAAGGGATTTTGCCGA" \
        "TTTCGGAACCACCATCAAACAGGATTTTCGCCTGCTGGGGCAAACCAGCGTGGACCGCTTGCTGCAACTCTCTCAGGGCCAGGCGGTGAAGGGCAATCAG" \
        "CTGTTGCCCGTCTCACTGGTGAAAAGAAAAACCACCCTGGCGCCCAATACGCAAACCGCCTCTCCCCGCGCGTTGGCCGATTCATTAATGCAGCTGGCAC" \
        "GACAGGTTTCCCGACTGGAAAGCGGGCAGTGAGCGCAACGCAATTAATGTGAGTTAGCTCACTCATTAGGCACCCCAGGCTTTACACTTTATGCTTCCGG" \
        "CTCGTATGTTGTGTGGAATTGTGAGCGGATAACAATTTCACACAGGAAACAGCTATGACCATGATTACGAATTCGAGCTCGGTACCCGGGGATCCTCAAC" \
        "TGTGAGGAGGCTCACGGACGCGAAGAACAGGCACGCGTGCTGGCAGAAACCCCCGGTATGACCGTGAAAACGGCCCGCCGCATTCTGGCCGCAGCACCAC" \
        "AGAGTGCACAGGCGCGCAGTGACACTGCGCTGGATCGTCTGATGCAGGGGGCACCGGCACCGCTGGCTGCAGGTAACCCGGCATCTGATGCCGTTAACGA" \
        "TTTGCTGAACACACCAGTGTAAGGGATGTTTATGACGAGCAAAGAAACCTTTACCCATTACCAGCCGCAGGGCAACAGTGACCCGGCTCATACCGCAACC" \
        "GCGCCCGGCGGATTGAGTGCGAAAGCGCCTGCAATGACCCCGCTGATGCTGGACACCTCCAGCCGTAAGCTGGTTGCGTGGGATGGCACCACCGACGGTG" \
        "CTGCCGTTGGCATTCTTGCGGTTGCTGCTGACCAGACCAGCACCACGCTGACGTTCTACAAGTCCGGCACGTTCCGTTATGAGGATGTGCTCTGGCCGGA" \
        "GGCTGCCAGCGACGAGACGAAAAAACGGACCGCGTTTGCCGGAACGGCAATCAGCATCGTTTAACTTTACCCTTCATCACTAAAGGCCGCCTGTGCGGCT" \
        "TTTTTTACGGGATTTTTTTATGTCGATGTACACAACCGCCCAACTGCTGGCGGCAAATGAGCAGAAATTTAAGTTTGATCCGCTGTTTCTGCGTCTCTTT" \
        "TTCCGTGAGAGCTATCCCTTCACCACGGAGAAAGTCTATCTCTCACAAATTCCGGGACTGGTAAACATGGCGCTGTACGTTTCGCCGATTGTTTCCGGTG" \
        "AGGTTATCCGTTCCCGTGGCGGCTCCACCTCTGAAAGCTTGGCACTGGCCGTCGTTTTACAACGTCGTGACTGGGAAAACCCTGGCGTTACCCAACTTAA" \
        "TCGCCTTGCAGCACATCCCCCTTTCGCCAGCTGGCGTAATAGCGAAGAGGCCCGCACCGATCGCCCTTCCCAACAGTTGCGCAGCCTGAATGGCGAATGG" \
        "CGCTTTGCCTGGTTTCCGGCACCAGAAGCGGTGCCGGAAAGCTGGCTGGAGTGCGATCTTCCTGAGGCCGATACTGTCGTCGTCCCCTCAAACTGGCAGA" \
        "TGCACGGTTACGATGCGCCCATCTACACCAACGTGACCTATCCCATTACGGTCAATCCGCCGTTTGTTCCCACGGAGAATCCGACGGGTTGTTACTCGCT" \
        "CACATTTAATGTTGATGAAAGCTGGCTACAGGAAGGCCAGACGCGAATTATTTTTGATGGCGTTCCTATTGGTTAAAAAATGAGCTGATTTAACAAAAAT" \
        "TTAATGCGAATTTTAACAAAATATTAACGTTTACAATTTAAATATTTGCTTATACAATCTTCCTGTTTTTGGGGCTTTTCTGATTATCAACCGGGGTACA" \
        "TATGATTGACATGCTAGTTTTACGATTACCGTTCATCGATTCTCTTGTTTGCTCCAGACTCTCA"

_m13_variants = {
    M13Variant.p7249: _7249,

    M13Variant.p7560: _7560,

    M13Variant.p8064: _8064,
}

##################
# keys

# DNADesign keys
version_key = 'version'
grid_key = 'grid'
major_tick_distance_key = 'major_tick_distance'
major_ticks_key = 'major_ticks'
rotation_key = 'rotation'
rotation_anchor_key = 'rotation_anchor'
helices_key = 'helices'
strands_key = 'strands'
scaffold_key = 'scaffold'
helices_view_order_key = 'helices_view_order'
is_origami_key = 'is_origami'
design_modifications_key = 'modifications_in_design'

# Helix keys
idx_on_helix_key = 'idx'
max_offset_key = 'max_offset'
min_offset_key = 'min_offset'
grid_position_key = 'grid_position'
svg_position_key = 'svg_position'
position3d_key = 'position'

# Strand keys
color_key = 'color'
dna_sequence_key = 'sequence'
legacy_dna_sequence_keys = 'dna_sequence'  # support legacy names for these ideas
domains_key = 'domains'
legacy_domains_keys = ['substrands']  # support legacy names for these ideas
idt_key = 'idt'
is_scaffold_key = 'is_scaffold'
modification_5p_key = '5prime_modification'
modification_3p_key = '3prime_modification'
modifications_int_key = 'internal_modifications'

# Domain keys
helix_idx_key = 'helix'
forward_key = 'forward'
legacy_forward_keys = 'right'  # support legacy names for these ideas
start_key = 'start'
end_key = 'end'
deletions_key = 'deletions'
insertions_key = 'insertions'

# Loopout keys
loopout_key = 'loopout'

# Modification keys
mod_location_key = 'location'
mod_display_text_key = 'display_text'
mod_id_key = 'id'
mod_idt_text_key = 'idt_text'
mod_font_size_key = 'font_size'
mod_display_connector_key = 'display_connector'
mod_allowed_bases_key = 'allowed_bases'


# end keys
##################

# end constants
##########################################################################


##########################################################################
# modification abstract base classes


@dataclass(frozen=True, eq=True)
class Modification(_JSONSerializable):
    """Base class of modifications (to DNA sequences, e.g., biotin or Cy3).
    Use :any:`Modification3Prime`, :any:`Modification5Prime`, or :any:`ModificationInternal`
    to instantiate."""

    display_text: str
    """Short text to display in the web interface as an "icon"
    visually representing the modification, e.g., ``'B'`` for biotin or ``'Cy3'`` for Cy3."""

    id: str = None
    """Short representation as a string; used to write in :any:`Strand` json representation,
    while the full description of the modification is written under a global key in the :any:`DNADesign`."""

    font_size: Optional[int] = None
    """Font size to use when displaying :py:data:`Modification.display_text`, in units of px.
    optional; if not specified, default value used in web interface is 8."""

    idt_text: Optional[str] = None
    """IDT text string specifying this modification (e.g., '/5Biosg/' for 5' biotin). optional"""

    display_connector: bool = True
    """Indicates whether in the web interface to display a fake "connector" between the DNA strand and
    the visual depiction of the modification. If ``True``, then it will be displayed slight above 
    (for forward domains) or below (for reverse domains) the DNA strand itself, to keep from blocking
    the view of the rest of the design. If ``False``, the modification will
    be displayed at the same veritical height as the DNA strand. The latter option is useful to 
    visually test where the modifications will appear on the surface of a 2D design, for instance, whereas
    the former distorts the locations of the modifications and does not as accurately represent their
    positions relative to each other and the rest of the DNA design."""

    def to_json_serializable(self, suppress_indent: bool = True):
        ret = {
            mod_display_text_key: self.display_text,
            # mod_id_key: self.id,
        }
        if self.idt_text is not None:
            ret[mod_idt_text_key] = self.idt_text
        if self.font_size is not None:
            ret[mod_font_size_key] = self.font_size
        if self.display_connector == False:
            ret[mod_display_connector_key] = False
        return ret

    @staticmethod
    def from_json(json_map: dict) -> Modification:
        location = json_map[mod_location_key]
        if location == "5'":
            return Modification5Prime.from_json(json_map)
        elif location == "3'":
            return Modification3Prime.from_json(json_map)
        elif location == "internal":
            return ModificationInternal.from_json(json_map)
        else:
            raise IllegalDNADesignError(f'unknown Modification location "{location}"')


@dataclass(frozen=True, eq=True)
class Modification5Prime(Modification):
    """5' modification of DNA sequence, e.g., biotin or Cy3."""

    def to_json_serializable(self, suppress_indent: bool = True):
        ret = super().to_json_serializable(suppress_indent)
        ret[mod_location_key] = "5'"
        return ret

    @staticmethod
    def from_json(json_map: dict) -> Modification5Prime:
        display_text = json_map[mod_display_text_key]
        font_size = json_map.get(mod_font_size_key)
        display_connector = json_map.get(mod_display_connector_key, True)
        # id = json_map[mod_id_key]
        location = json_map[mod_location_key]
        assert location == "5'"
        idt_text = json_map.get(mod_idt_text_key)
        return Modification5Prime(display_text=display_text, idt_text=idt_text, font_size=font_size,
                                  display_connector=display_connector)


@dataclass(frozen=True, eq=True)
class Modification3Prime(Modification):
    """3' modification of DNA sequence, e.g., biotin or Cy3."""

    def to_json_serializable(self, suppress_indent: bool = True):
        ret = super().to_json_serializable(suppress_indent)
        ret[mod_location_key] = "3'"
        return ret

    @staticmethod
    def from_json(json_map: dict) -> Modification3Prime:
        display_text = json_map[mod_display_text_key]
        font_size = json_map.get(mod_font_size_key)
        display_connector = json_map.get(mod_display_connector_key, True)
        # id = json_map[mod_id_key]
        location = json_map[mod_location_key]
        assert location == "3'"
        idt_text = json_map.get(mod_idt_text_key)
        return Modification3Prime(display_text=display_text, idt_text=idt_text, font_size=font_size,
                                  display_connector=display_connector)


@dataclass(frozen=True, eq=True)
class ModificationInternal(Modification):
    """Internal modification of DNA sequence, e.g., biotin or Cy3."""

    allowed_bases: Optional[FrozenSet[str]] = None
    """If None, then this is an internal modification that goes between bases. 
    If instead it is a list of bases, then this is an internal modification that attaches to a base,
    and this lists the allowed bases for this internal modification to be placed at. 
    For example, internal biotins for IDT must be at a T. If any base is allowed, it should be
    ``['A','C','G','T']``."""

    def to_json_serializable(self, suppress_indent: bool = True):
        ret = super().to_json_serializable(suppress_indent)
        ret[mod_location_key] = "internal"
        if self.allowed_bases is not None:
            ret[mod_allowed_bases_key] = _NoIndent(
                list(self.allowed_bases)) if suppress_indent else list(self.allowed_bases)
        return ret

    @staticmethod
    def from_json(json_map: dict) -> ModificationInternal:
        display_text = json_map[mod_display_text_key]
        font_size = json_map.get(mod_font_size_key)
        display_connector = json_map.get(mod_display_connector_key, True)
        # id = json_map[mod_id_key]
        location = json_map[mod_location_key]
        assert location == "internal"
        idt_text = json_map.get(mod_idt_text_key)
        allowed_bases_list = json_map.get(mod_allowed_bases_key)
        allowed_bases = frozenset(allowed_bases_list) if allowed_bases_list is not None else None
        return ModificationInternal(display_text=display_text, idt_text=idt_text, font_size=font_size,
                                    display_connector=display_connector, allowed_bases=allowed_bases)


# end modification abstract base classes
##########################################################################

@dataclass
class Position3D(_JSONSerializable):
    """
    Position (x,y,z) and orientation (pitch,roll,yaw) in 3D space.
    """

    x: float = 0
    y: float = 0
    z: float = 0
    pitch: float = 0
    roll: float = 0
    yaw: float = 0

    def to_json_serializable(self, suppress_indent: bool = True):
        dct = self.__dict__
        # return NoIndent(dct) if suppress_indent else dct
        return dct


def in_browser() -> bool:
    """Test if this code is running in the browser.

    Checks for existence of package "pyodide" used in pyodide. If present it is assumed the code is
    running in the browser."""
    try:
        import pyodide
        return True
    except ModuleNotFoundError:
        return False


@dataclass
class Helix(_JSONSerializable):
    """
    Represents a "helix" where :any:`Domain`'s could go. Technically a :any:`Helix` can contain no
    :any:`Domain`'s. More commonly, some partial regions of it may have only 1 or 0 :any:`Domain`'s.
    So it is best thought of as a "potential" double-helix.

    It has a 1-dimensional integer coordinate system given by "offsets", integers between
    :py:data:`Helix.min_offset` (inclusive) and :py:data:`Helix.max_offset` (exclusive).
    At any valid offset for this :any:`Helix`, at most two :any:`Domain`'s may share that offset
    on this :any:`Helix`, and if there are exactly two, then one must have
    :py:data:`Domain.forward` = ``true`` and the other must have
    :py:data:`Domain.forward` = ``false``.

    Once part of a :any:`DNADesign`, a :any:`Helix` has an index (accessible  via :py:meth:`Helix.idx`
    once the :any:`DNADesign` is created)
    representing its order in the list of all :any:`Helix`'s. This index is how a :any:`Domain` is
    associated to the :any:`Helix` via the integer index :any:`Domain.helix`.
    """

    max_offset: int = None
    """Maximum offset (exclusive) of :any:`Domain` that can be drawn on this :any:`Helix`. 
    If unspecified, it is calculated when the :any:`DNADesign` is instantiated as 
    the largest :any:`Domain.end` offset of any :any:`Domain` in the design.
    """

    min_offset: int = 0
    """Minimum offset (inclusive) of :any:`Domain` that can be drawn on this :any:`Helix`. 
    If unspecified, it is set to 0.
    """

    major_tick_distance: int = -1
    """If positive, overrides :any:`DNADesign.major_tick_distance`."""

    major_ticks: List[int] = None
    """If not ``None``, overrides :any:`DNADesign.major_tick_distance` and :any:`Helix.major_tick_distance`
    to specify a list of offsets at which to put major ticks."""

    grid_position: Tuple[int, int, int] = None
    """`(h,v,b)` position of this helix in the side view grid,
    if :const:`Grid.square`, :const:`Grid.hex` , or :const:`Grid.honeycomb` is used
    in the :any:`DNADesign` containing this helix.
    `h` and `v` are in units of "helices": incrementing `h` moves right one helix in the grid
    and incrementing `v` moves down one helix in the grid. 
    In the case of the hexagonal lattice, 
    The convention is that incrementing `v` moves down and to the right if h is even, 
    and moves down and to the left if `h` is odd.
    This is the "odd-r horizontal layout" coordinate system here: 
    https://www.redblobgames.com/grids/hexagons/)
    `b` goes in and out of the screen in the side view, and it is in units of "bases".
    Incrementing `b` moves the whole helix one base into the screen.
    In the main view, a helix with `b` = 1 would have its base offset 0 line up with base offset 1
    of a helix with `b` = 0.
    However, the default y svg_position for helices does not otherwise depend on grid_position.
    The default is to list the y-coordinates in order by helix idx.
    
    Default is `h` = 0, `v` = index of :any:`Helix` in :py:data:`DNADesign.helices`, `b` = 0.
    
    In the case of the honeycomb lattice, we use the same convention as cadnano for encoding hex coordinates see `misc/cadnano-format-specs/v2.txt`.
    That convention is different from simply excluding coordinates from the hex lattice.
    """

    svg_position: Tuple[float, float] = None
    """`(x,y)` SVG coordinates of base offset 0 of this Helix in the main view. 
    
    If `grid_position` and `position` are both omitted, then the default is 
    `x` = 0, `y` = [index of helix] * :any:`scadnano.distance_between_helices_svg`.
    
    If `grid_position = (h,v,b)` is specified but `position` is omitted, then the default is
    `x` = b * BASE_WIDTH_SVG, `y` = [index of :any:`Helix`] * :any:`scadnano.distance_between_helices_svg`."""

    rotation: float = 0
    """Rotation angle (in degrees) of backbone of the :any:`Domain` on this :any:`Helix` with 
    :py:data:`Domain.forward` = ``True``. 
    
    The angle is relative to the offset :py:data:`Helix.rotation_anchor`, and 0 degrees is defined to
    be pointing *up* in both the side view and main view.
    
    A positive rotation angle rotates *clockwise* in the side view.
    This violates standard Cartesian coordinate conventions:
    https://en.wikipedia.org/wiki/Rotation_matrix, 
    but it is consistent with SVG rotation conventions:
    https://www.w3.org/TR/SVG11/coords.html#ExampleRotateScale.
    
    For example, a rotation of 90 degrees points right in the side view 
    and out of the screen in the main view.
    
    Default is 0 degrees."""

    rotation_anchor: int = 0
    """Offset on this :any:`Helix` that is the reference point for 0 degrees.
    The rotation at offset ``o`` is 360 degrees times the remainder of ``o - rotation_anchor`` 
    when divided by 10.5.
    
    For example, if :py:data:`Helix.rotation` = 0 and :py:data:`Helix.rotation_anchor` = 42, then
    at offsets of the form :math:`42 + 21k` for integer :math:`k` 
    (i.e., 42 itself, as well as 21, 0, -21, -42, ..., 63, 84, 105, ...),
    the rotation angle is also 0 at those offsets since
    they are integer multiples of 21 (hence also multiples of 10.5) from 42.
    
    Default is 0."""

    position3d: Position3D = None
    """Position (x,y,z) and orientation (pitch,roll,yaw) of this :any:`Helix` in 3D space.
    
    Optional if :py:data:`Helix.grid_position` is specified. 
    Default is pitch, roll, yaw are 0, and x,y,z are determined by grid position h, v, b."""

    idx: int = None
    """Index of this :any:`Helix`.
    
    Optional if no other :any:`Helix` specifies a value for *idx*.
    Default is the order of the :any:`Helix` is listed in constructor for :any:`DNADesign`."""

    # for optimization; list of domains on that Helix
    _domains: List['Domain'] = field(default_factory=list)

    def __post_init__(self):
        if self.major_ticks is not None and self.max_offset is not None and self.min_offset is not None:
            for major_tick in self.major_ticks:
                if major_tick > self.max_offset - self.min_offset:
                    raise IllegalDNADesignError(f'major tick {major_tick} in list {self.major_ticks} is '
                                                f'outside the range of available offsets since max_offset = '
                                                f'{self.max_offset}')

    def to_json_serializable(self, suppress_indent: bool = True):
        dct = dict()

        if self.min_offset != 0:
            dct[min_offset_key] = self.min_offset

        dct[max_offset_key] = self.max_offset

        if self.position3d is None:
            if self.grid_position[
                2] == 0:  # don't bother writing grid position base coordinate if it is 0
                dct[grid_position_key] = (self.grid_position[0], self.grid_position[1])
            else:
                dct[grid_position_key] = (
                    self.grid_position[0], self.grid_position[1], self.grid_position[2])
        else:
            dct[position3d_key] = self.position3d.to_json_serializable(suppress_indent)

        # print(f'self.svg_position()    = {self.svg_position}')
        # print(f'default_svg_position() = {self.default_svg_position()}')
        default_x, default_y = self.default_svg_position()
        if not (_is_close(self.svg_position[0], default_x) and _is_close(self.svg_position[1],
                                                                         default_y)):
            dct[svg_position_key] = (self.svg_position[0], self.svg_position[1])

        if self.major_tick_distance is not None and self.major_tick_distance > 0:
            dct[major_tick_distance_key] = self.major_tick_distance

        if self.major_ticks is not None:
            dct[major_ticks_key] = self.major_ticks

        if self.rotation != 0:
            dct[rotation_key] = self.rotation

        if self.rotation_anchor != 0:
            dct[rotation_anchor_key] = self.rotation_anchor

        dct[idx_on_helix_key] = self.idx

        return _NoIndent(dct) if suppress_indent else dct

    def default_svg_position(self):
        return 0, self.idx * distance_between_helices_svg

    def default_grid_position(self):
        return (0, self.idx, 0)

    def calculate_major_ticks(self, default_major_tick_distance: int):
        """
        Calculates full list of major tick marks, whether using `default_major_tick_distance` (from
        :any:`DNADesign`), :py:data:`Helix.major_tick_distance`, or :py:data:`Helix.major_ticks`.
        They are used in reverse order to determine precedence. (e.g., :py:data:`Helix.major_ticks`
        overrides :py:data:`Helix.major_tick_distance`, which overrides
        `default_major_tick_distance` from :any:`DNADesign`.
        """
        if self.major_ticks is not None:
            return self.major_ticks
        distance = default_major_tick_distance if self.major_tick_distance <= 0 else self.major_tick_distance
        return list(range(self.min_offset, self.max_offset + 1, distance))

    @staticmethod
    def from_json(json_map: dict) -> Helix:
        grid_position = None
        if grid_position_key in json_map:
            gp_list = json_map[grid_position_key]
            if len(gp_list) not in [2, 3]:
                raise IllegalDNADesignError("list of grid_position coordinates must be length 2 or 3, "
                                            f"but this is the list: {gp_list}")
            if len(gp_list) == 2:
                gp_list.append(0)
            grid_position = tuple(gp_list)

        svg_position = None
        if svg_position_key in json_map:
            sp_list = json_map[grid_position_key]
            if len(sp_list) != 2:
                raise IllegalDNADesignError("svg_position must have exactly two integers, "
                                            f"but instead it has {len(sp_list)}: {sp_list}")
            svg_position = tuple(sp_list)

        major_tick_distance = json_map.get(major_tick_distance_key)
        major_ticks = json_map.get(major_ticks_key)
        min_offset = json_map.get(min_offset_key)
        max_offset = json_map.get(max_offset_key)
        rotation = json_map.get(rotation_key, default_helix_rotation)
        rotation_anchor = json_map.get(rotation_anchor_key, default_helix_rotation_anchor)
        position3d = json_map.get(position3d_key)
        idx = json_map.get(idx_on_helix_key)

        return Helix(
            major_tick_distance=major_tick_distance,
            major_ticks=major_ticks,
            grid_position=grid_position,
            svg_position=svg_position,
            min_offset=min_offset,
            max_offset=max_offset,
            rotation=rotation,
            rotation_anchor=rotation_anchor,
            position3d=position3d,
            idx=idx,
        )


def _is_close(x1: float, x2: float):
    return abs(x1 - x2) < 0.00000001


@dataclass
class Domain(_JSONSerializable):
    """
    A maximal portion of a :any:`Strand` that is continguous on a single :any:`Helix`.
    A :any:`Strand` contains a list of :any:`Domain`'s (and also potentially :any:`Loopout`'s).
    """

    helix: int
    """index of the :any:`Helix` on which this :any:`Domain` resides."""

    forward: bool
    """Whether the strand "points" forward (i.e., its 3' end has a larger offset than its 5' end).
    If :any:`Domain.forward` is ``True``, then 
    :any:`Domain.start` is the 5' end of the :any:`Domain` and 
    :any:`Domain.end` is the 3' end of the :any:`Domain`.
    If :any:`Domain.forward` is ``False``, these roles are reversed."""

    start: int
    """
    The smallest offset position of any base on this Domain
    (3' end if :any:`Domain.forward` = ``False``,
    5' end if :any:`Domain.forward` = ``True``).
    """

    # TODO: give option to user in constructor to specify that end is inclusive (default exclusive)
    end: int
    """
    1 plus the largest offset position of any base on this Domain
    (5' end if :any:`Domain.forward` = ``False``,
    3' end if :any:`Domain.forward` = ``True``).
    Note that the set of base offsets occupied by this Domain is {start, start+1, ..., end-1},
    i.e., inclusive for :py:data:`Strand.start` but exclusive for :py:data:`Strand.end`,
    the same convention used in Python for slices of lists and strings.
    (e.g., :samp:`"abcdef"[1:3] == "bc"`)
    
    Some methods (such as :py:meth:`Domain.dna_sequence_in`) use the convention of being inclusive on 
    both ends and are marked with the word "INCLUSIVE".
    (Such a convention is easier to reason about when there are insertions and deletions.)
    """

    deletions: List[int] = field(default_factory=list)
    """List of positions of deletions on this Domain."""

    insertions: List[Tuple[int, int]] = field(default_factory=list)
    """List of (position,num_insertions) pairs on this Domain.
    
    This is the number of *extra* bases in addition to the base already at this position. 
    The total number of bases at this offset is num_insertions+1."""

    # not serialized; for efficiency
    _parent_strand: Strand = field(init=False, repr=False, compare=False, default=None)

    def __post_init__(self):
        self._check_start_end()

    def __repr__(self):
        rep = (f'Domain(helix={self.helix}'
               f', forward={self.forward}'
               f', start={self.start}'
               f', end={self.end}') + \
              (f', deletions={self.deletions}' if len(self.deletions) > 0 else '') + \
              (f', insertions={self.insertions}' if len(self.insertions) > 0 else '') + \
              ')'
        return rep

    def __str__(self):
        return repr(self)

    def strand(self) -> Strand:
        return self._parent_strand

    def _check_start_end(self):
        if self.start >= self.end:
            raise StrandError(self._parent_strand,
                              f'start = {self.start} must be less than end = {self.end}')

    def to_json_serializable(self, suppress_indent: bool = True):
        dct = OrderedDict()
        dct[helix_idx_key] = self.helix
        dct[forward_key] = self.forward
        dct[start_key] = self.start
        dct[end_key] = self.end
        if len(self.deletions) > 0:
            dct[deletions_key] = self.deletions
        if len(self.insertions) > 0:
            dct[insertions_key] = self.insertions
        return _NoIndent(dct) if suppress_indent else dct

    @staticmethod
    def is_loopout() -> bool:
        """Indicates if this is a :any:`Loopout` (always false)
        Useful when object could be either :any:`Loopout` or :any:`Domain`."""
        return False

    @staticmethod
    def is_domain() -> bool:
        """Indicates if this is a :any:`Domain` (always true)
        Useful when object could be either :any:`Loopout` or :any:`Domain`."""
        return True

    def set_start(self, new_start: int):
        self.start = new_start
        self._check_start_end()

    def set_end(self, new_end: int):
        self.end = new_end
        self._check_start_end()

    def offset_5p(self) -> int:
        """5' offset of this :any:`Domain`, INCLUSIVE."""
        if self.forward:
            return self.start
        else:
            return self.end - 1
        # return self.start if self.forward else self.end - 1

    def offset_3p(self) -> int:
        """3' offset of this :any:`Domain`, INCLUSIVE."""
        return self.end - 1 if self.forward else self.start

    def _num_insertions(self) -> int:
        # total number of insertions in this Domain
        return sum(insertion[1] for insertion in self.insertions)

    def contains_offset(self, offset: int) -> bool:
        """Indicates if `offset` is the offset of a base on this :any:`Domain`.

        Note that offsets refer to visual portions of the displayed grid for the Helix.
        If for example, this Domain starts at position 0 and ends at 10, and it has 5 deletions,
        then it contains the offset 7 even though there is no base 7 positions from the start."""
        return self.start <= offset < self.end

    def __len__(self):
        """Same as :meth:`Domain.dna_length`.

        See also :meth:`Domain.visual_length`."""
        return self.dna_length()

    def dna_length(self) -> int:
        """Number of bases in this Domain."""
        return self.end - self.start - len(self.deletions) + self._num_insertions()

    def dna_length_in(self, left, right) -> int:
        """Number of bases in this Domain between `left` and `right` (INCLUSIVE)."""
        if not left <= right + 1:
            raise ValueError(f'left = {left} and right = {right} but we should have left <= right + 1')
        if not self.start <= left:
            raise ValueError(f'left = {left} should be at least self.start = {self.start}')
        if not right < self.end:
            raise ValueError(f'right = {right} should be at most self.end - 1 = {self.end - 1}')
        num_deletions = sum(1 for offset in self.deletions if left <= offset <= right)
        num_insertions = sum(length for (offset, length) in self.insertions if left <= offset <= right)
        return (right - left + 1) - num_deletions + num_insertions

    def visual_length(self) -> int:
        """Distance between :any:`Domain.start` offset and :any:`Domain.end` offset.

        This can be more or less than the :meth:`Domain.dna_length` due to insertions and deletions."""
        return self.end - self.start

    def dna_sequence(self) -> Optional[str]:
        """Return DNA sequence of this Domain, or ``None`` if no DNA sequence has been assigned
        to this :any:`Domain`'s :any:`Strand`."""
        return self.dna_sequence_in(self.start, self.end - 1)

    def dna_sequence_in(self, offset_left: int, offset_right: int) -> Optional[str]:
        """Return DNA sequence of this Domain in the interval of offsets given by
        [`offset_left`, `offset_right`], INCLUSIVE, or ``None`` if no DNA sequence has been assigned
        to this :any:`Domain`'s :any:`Strand`.

        WARNING: This is inclusive on both ends,
        unlike other parts of this API where the right endpoint is exclusive.
        This is to make the notion well-defined when one of the endpoints is on an offset with a
        deletion or insertion."""
        strand_seq = self._parent_strand.dna_sequence
        if strand_seq is None:
            return None

        # if on a deletion, move inward until we are off of it
        while offset_left in self.deletions:
            offset_left += 1
        while offset_right in self.deletions:
            offset_right -= 1

        if offset_left > offset_right:
            return ''
        if offset_left >= self.end:
            return ''
        if offset_right < 0:
            return ''

        str_idx_left = self.domain_offset_to_strand_dna_idx(offset_left, self.forward)
        str_idx_right = self.domain_offset_to_strand_dna_idx(offset_right, not self.forward)
        if not self.forward:  # these will be out of order if strand is left
            str_idx_left, str_idx_right = str_idx_right, str_idx_left
        subseq = strand_seq[str_idx_left:str_idx_right + 1]
        return subseq

    def get_seq_start_idx(self) -> int:
        """Starting DNA subsequence index for first base of this :any:`Domain` on its
        Parent :any:`Strand`'s DNA sequence."""
        domains = self._parent_strand.domains
        # index of self in parent strand's list of domains
        self_domain_idx = domains.index(self)
        # index of self's position within the DNA sequence of parent strand
        self_seq_idx_start = sum(prev_domain.dna_length()
                                 for prev_domain in domains[:self_domain_idx])
        return self_seq_idx_start

    def domain_offset_to_strand_dna_idx(self, offset: int, offset_closer_to_5p: bool) -> int:
        """ Convert from offset on this :any:`Domain`'s :any:`Helix`
        to string index on the parent :any:`Strand`'s DNA sequence.

        If `offset_closer_to_5p` is ``True``, (this only matters if `offset` contains an insertion)
        then the only leftmost string index corresponding to this offset is included,
        otherwise up to the rightmost string index (including all insertions) is included."""
        if offset in self.deletions:
            raise ValueError(f'offset {offset} illegally contains a deletion from {self.deletions}')

        # length adjustment for insertions depends on whether this is a left or right offset
        len_adjust = self._net_ins_del_length_increase_from_5p_to(offset, offset_closer_to_5p)

        # get string index assuming this Domain is first on Strand
        if self.forward:
            offset += len_adjust  # account for insertions and deletions
            domain_str_idx = offset - self.start
        else:
            # account for insertions and deletions
            offset -= len_adjust  # account for insertions and deletions
            domain_str_idx = self.end - 1 - offset

        # correct for existence of previous Domains on this Strand
        return domain_str_idx + self.get_seq_start_idx()

    def _net_ins_del_length_increase_from_5p_to(self, offset_edge: int, offset_closer_to_5p: bool) -> int:
        """Net number of insertions from 5'/3' end to offset_edge,
        INCLUSIVE on 5'/3' end, EXCLUSIVE on offset_edge.

        Set `five_p` ``= False`` to test from 3' end to `offset_edge`."""
        length_increase = 0
        for deletion in self.deletions:
            if self._between_5p_and_offset(deletion, offset_edge):
                length_increase -= 1
        for (insertion_offset, insertion_length) in self.insertions:
            if self._between_5p_and_offset(insertion_offset, offset_edge):
                length_increase += insertion_length
        # special case for when offset_edge is an endpoint closer to the 3' end,
        # we add its extra insertions also in this case
        if not offset_closer_to_5p:
            insertion_map: Dict[int, int] = dict(self.insertions)
            if offset_edge in insertion_map:
                insertion_length = insertion_map[offset_edge]
                length_increase += insertion_length
        return length_increase

    def _between_5p_and_offset(self, offset_to_test: int, offset_edge: int) -> bool:
        return ((self.forward and self.start <= offset_to_test < offset_edge) or
                (not self.forward and offset_edge < offset_to_test < self.end))

    # def _between_3p_and_offset(self, offset_to_test: int, offset_edge: int) -> bool:
    #     return ((self.direction == Direction.left and self.start <= offset_to_test < offset_edge) or
    #             (self.direction == Direction.forward and offset_edge < offset_to_test < self.end))

    # The type hint 'Domain' must be in quotes since Domain is not yet defined.
    # This is a "forward reference": https://www.python.org/dev/peps/pep-0484/#forward-references
    def overlaps(self, other: Domain) -> bool:
        r"""Indicates if this :any:`Domain`'s set of offsets (the set
        :math:`\{x \in \mathbb{N} \mid`
        ``self.start``
        :math:`\leq x \leq`
        ``self.end``
        :math:`\}`)
        has nonempty intersection with those of `other`,
        and they appear on the same helix,
        and they point in opposite directions."""  # noqa (suppress PEP warning)
        return (self.helix == other.helix and
                self.forward == (not other.forward) and
                self.compute_overlap(other)[0] >= 0)

    def overlaps_illegally(self, other: Domain):
        r"""Indicates if this :any:`Domain`'s set of offsets (the set
        :math:`\{x \in \mathbb{N} \mid`
        ``self.start``
        :math:`\leq x \leq`
        ``self.end``
        :math:`\}`)
        has nonempty intersection with those of `other`,
        and they appear on the same helix,
        and they point in the same direction."""  # noqa (suppress PEP warning)
        return (self.helix == other.helix and
                self.forward == other.forward and
                self.compute_overlap(other)[0] >= 0)

    def compute_overlap(self, other: Domain) -> Tuple[int, int]:
        """Return [left,right) offset indicating overlap between this Domain and `other`.

        Return ``(-1,-1)`` if they do not overlap (different helices, or non-overlapping regions
        of the same helix)."""
        overlap_start = max(self.start, other.start)
        overlap_end = min(self.end, other.end)
        if overlap_start >= overlap_end:  # overlap is empty
            return -1, -1
        return overlap_start, overlap_end

    def insertion_offsets(self) -> List[int]:
        """Return offsets of insertions (but not their lengths)."""
        return [ins_off for (ins_off, _) in self.insertions]

    @staticmethod
    def from_json(json_map):
        helix = json_map[helix_idx_key]
        if forward_key in json_map:
            forward = json_map[forward_key]
        else:
            forward = None
            for legacy_forward_key in legacy_dna_sequence_keys:
                if legacy_forward_key in json_map:
                    forward = json_map[legacy_forward_key]
                    break
            if forward is None:
                raise IllegalDNADesignError(f'key {forward_key} missing from Domain description')
        start = json_map[start_key]
        end = json_map[end_key]
        deletions = json_map.get(deletions_key, [])
        insertions = list(map(tuple, json_map.get(insertions_key, [])))
        return Domain(
            helix=helix,
            forward=forward,
            start=start,
            end=end,
            deletions=deletions,
            insertions=insertions,
        )


'''
    var forward = util.get_value(json_map, constants.forward_key, name);
    var helix = util.get_value(json_map, constants.helix_idx_key, name);
    var start = util.get_value(json_map, constants.start_key, name);
    var end = util.get_value(json_map, constants.end_key, name);
//    List<int> deletions =
//        json_map.containsKey(constants.deletions_key) ? List<int>.from(json_map[constants.deletions_key]) : [];
//    List<Tuple2<int, int>> insertions =
//        json_map.containsKey(constants.insertions_key) ? parse_json_insertions(json_map[constants.insertions_key]) : [];
    var deletions = List<int>.from(util.get_value_with_default(json_map, constants.deletions_key, []));
    var insertions =
        parse_json_insertions(util.get_value_with_default(json_map, constants.insertions_key, []));
'''


@dataclass
class Loopout(_JSONSerializable):
    """Represents a single-stranded loopout on a :any:`Strand`.

    One could think of a :any:`Loopout` as a type of :any:`Domain`, but none of the fields of
    :any:`Domain` make sense for :any:`Loopout`, so they are not related to each other in the type
    hierarchy. It is interpreted that a :any:`Loopout` is a single-stranded region bridging two
    :any:`Domain`'s that are connected to :any:`Helix`'s, or if it occurs on the end of a :any:`Strand`,
    then it is a single-stranded extension. It is illegal for two consecutive :any:`Domain`'s to both
    be :any:`Loopout`'s, and for a :any:`Strand` to have only one element of :any:`Strand.domains`
    that is a :any:`Loopout`.

    Loopout has only a single field :py:data:`Loopout.length` that specifies the length of the loopout.

    For example, one use of a loopout is to describe a hairpin (a.k.a.,
    `stem-loop <https://en.wikipedia.org/wiki/Stem-loop>`_).
    The following creates a :any:`Strand` that represents a hairpin with a stem length of 10 and a loop
    length of 5.

    .. code-block:: Python

        import scadnano as sc

        domain_f = sc.Domain(helix=0, forward=True, start=0, end=10)
        loop = sc.Loopout(length=5)
        domain_r = sc.Domain(helix=0, forward=False, start=0, end=10)
        hairpin = sc.Strand([domain_f, loop, domain_r])
    """

    length: int
    """Length (in DNA bases) of this Loopout."""

    # not serialized; for efficiency
    _parent_strand: Strand = field(init=False, repr=False, compare=False, default=None)

    def to_json_serializable(self, suppress_indent: bool = True):
        dct = {loopout_key: self.length}
        return _NoIndent(dct)

    def __repr__(self):
        return f'Loopout({self.length})'

    def __str__(self):
        return repr(self)

    @staticmethod
    def is_loopout() -> bool:
        """Indicates if this is a :any:`Loopout` (always true).
        Useful when object could be either :any:`Loopout` or :any:`Domain`."""
        return True

    @staticmethod
    def is_domain() -> bool:
        """Indicates if this is a :any:`Domain` (always false)
        Useful when object could be either :any:`Loopout` or :any:`Domain`."""
        return False

    def __len__(self):
        """Same as :any:`Loopout.dna_length`"""
        return self.dna_length()

    def dna_length(self) -> int:
        """Length of this :any:`Loopout`; same as field :py:data:`Loopout.length`."""
        return self.length

    def dna_sequence(self) -> Optional[str]:
        """Return DNA sequence of this :any:`Loopout`, or ``None`` if no DNA sequence has been assigned
        to the :any:`Strand` of this :any:`Loopout`."""
        strand_seq = self._parent_strand.dna_sequence
        if strand_seq is None:
            return None

        str_idx_left = self.get_seq_start_idx()
        str_idx_right = str_idx_left + self.length  # EXCLUSIVE (unlike similar code for Domain)
        subseq = strand_seq[str_idx_left:str_idx_right]
        return subseq

    def get_seq_start_idx(self) -> int:
        """Starting DNA subsequence index for first base of this :any:`Loopout` on its
        :any:`Strand`'s DNA sequence."""
        domains = self._parent_strand.domains
        # index of self in parent strand's list of domains
        self_domain_idx = domains.index(self)
        # index of self's position within the DNA sequence of parent strand
        self_seq_idx_start = sum(prev_domain.dna_length()
                                 for prev_domain in domains[:self_domain_idx])
        return self_seq_idx_start

    @staticmethod
    def from_json(json_map):
        if loopout_key not in json_map:
            raise IllegalDNADesignError(f'no key "{loopout_key}" in JSON map')
        length = int(json_map[loopout_key])
        return Loopout(length=length)


_wctable = str.maketrans('ACGTacgt', 'TGCAtgca')


def wc(seq: str) -> str:
    """Return reverse Watson-Crick complement of `seq`."""
    return seq.translate(_wctable)[::-1]


@dataclass
class IDTFields(_JSONSerializable):
    """Data required when ordering DNA strands from the synthesis company
    `IDT DNA Technologies <https://www.idtdna.com/>`_.
    This data is used when automatically generating files used to order DNA from IDT."""

    name: str
    """Name of the strand (first field in IDT bulk input: https://www.idtdna.com/site/order/oligoentry).
    
    Non-optional field.
    """

    scale: str = default_idt_scale
    """Synthesis scale at which to synthesize the strand (third field in IDT bulk input:
    https://www.idtdna.com/site/order/oligoentry).
    Choices supplied by IDT at the time this was written: 
    ``"25nm"``, ``"100nm"``, ``"250nm"``, ``"1um"``, ``"5um"``, 
    ``"10um"``, ``"4nmU"``, ``"20nmU"``, ``"PU"``, ``"25nmS"``.
    
    Optional field.
    """

    purification: str = default_idt_purification
    """Purification options (fourth field in IDT bulk input:
    https://www.idtdna.com/site/order/oligoentry). 
    Choices supplied by IDT at the time this was written: 
    ``"STD"``, ``"PAGE"``, ``"HPLC"``, ``"IEHPLC"``, ``"RNASE"``, ``"DUALHPLC"``, ``"PAGEHPLC"``.
    
    Optional field.
    """

    plate: Optional[str] = None
    """Name of plate in case this strand will be ordered on a 96-well or 384-well plate.
    
    Optional field, but non-optional if :py:data:`IDTField.well` is not ``None``.
    """

    well: Optional[str] = None
    """Well position on plate in case this strand will be ordered on a 96-well or 384-well plate.
    
    Optional field, but non-optional if :py:data:`IDTField.plate` is not ``None``.
    """

    def __post_init__(self):
        _check_idt_string_not_none_or_empty(self.name, 'name')
        _check_idt_string_not_none_or_empty(self.scale, 'scale')
        _check_idt_string_not_none_or_empty(self.purification, 'purification')
        if self.plate is None and self.well is not None:
            raise IllegalDNADesignError(f'IDTFields.plate cannot be None if IDTFields.well is not None\n'
                                        f'IDTFields.well = {self.well}')
        if self.plate is not None and self.well is None:
            raise IllegalDNADesignError(f'IDTFields.well cannot be None if IDTFields.plate is not None\n'
                                        f'IDTFields.plate = {self.plate}')

    def to_json_serializable(self, suppress_indent: bool = True):
        dct = self.__dict__
        if self.plate is None:
            del dct['plate']
        if self.well is None:
            del dct['well']
        return _NoIndent(dct)


def _check_idt_string_not_none_or_empty(value: str, field_name: str):
    if value is None:
        raise IllegalDNADesignError(f'field {field_name} in IDTFields cannot be None')
    if len(value) == 0:
        raise IllegalDNADesignError(f'field {field_name} in IDTFields cannot be empty')


@dataclass
class Strand(_JSONSerializable):
    """
    Represents a single strand of DNA.

    Each maximal portion that is continguous on a single :any:`Helix` is a :any:`Domain`.
    Crossovers from one :any:`Helix` to another are implicitly from the 3' end of one of this
    Strand's :any:`Domain`'s to the 5' end of the next :any:`Domain`.

    A portion of the :any:`Strand` not associated to any :any:`Helix` is represented by a :any:`Loopout`.
    Two :any:`Loopout`'s cannot occur consecutively on a :any:`Strand`, nor can a :any:`Strand`
    contain only a :any:`Loopout` but no :any:`Domain`.


    One can set the strand to be a scaffold in the constructor:

    .. code-block:: Python

        import scadnano as sc

        scaffold_domains = [ ... ]
        scaffold_strand = sc.Strand(domains=scaffold_domains, is_scaffold=True)

    or by calling :py:meth:`Strand.set_scaffold` on the :any:`Strand` object:

    .. code-block:: Python

        import scadnano as sc

        scaffold_domains = [ ... ]
        scaffold_strand = sc.Strand(domains=scaffold_domains)
        scaffold_strand.set_scaffold()

    Both will give the strand the same color that
    `cadnano <https://cadnano.org/>`_
    uses for the scaffold.
    """

    domains: List[Union[Domain, Loopout]]
    """:any:`Domain`'s (or :any:`Loopout`'s) composing this Strand. 
    Each :any:`Domain` is contiguous on a single :any:`Helix` 
    and could be either single-stranded or double-stranded, 
    whereas each :any:`Loopout` is single-stranded and has no associated :any:`Helix`."""

    dna_sequence: Optional[str] = None
    """Do not assign directly to this field. Always use :any:`DNADesign.assign_dna` 
    (for complementarity checking) or :any:`Strand.set_dna_sequence` 
    (without complementarity checking, to allow mismatches)."""

    color: Optional[Color] = None
    """Color to show this strand in the main view. If not specified in the constructor,
    a color is assigned by cycling through a list of defaults given by 
    :meth:`ColorCycler.colors`"""

    automatically_assign_color: bool = field(repr=False, default=True)
    """If `automatically_assign_color` = ``False`` and `color` = ``None``, do not automatically
    assign a :any:`Color` to this :any:`Strand`. 
    In this case color will be set to its default of ``None`` and will not be
    written to the JSON with :py:meth:`DNADesign.write_scadnano_file` or :py:meth:`DNADesign.to_json`."""

    idt: Optional[IDTFields] = None
    """Fields used when ordering strands from the synthesis company IDT 
    (Integrated DNA Technologies, Coralville, IA). If present (i.e., not equal to :const:`None`)
    then the method :py:meth:`DNADesign.write_idt_bulk_input_file` can be called to automatically
    generate an text file for ordering strands in test tubes: 
    https://www.idtdna.com/site/order/oligoentry,
    as can the method :py:meth:`DNADesign.write_idt_plate_excel_file` for writing a Microsoft Excel 
    file that can be uploaded to IDT's website for describing DNA sequences to be ordered in 96-well
    or 384-well plates."""

    use_default_idt: bool = False
    """If ``True``, assigns an :any:`IDTFields` to this :any:`Strand` with same naming convention as
    cadnano, i.e., :py:data:`IDTFields.name` = "ST{h5}[{s}]{h3}[{e}]", where {h5} and {h3} are the 
    :any:`Helix`'s of the 5' and 3' ends, respectively, of the :any:`Strand`, 
    and {s} and {e} are the respective start and end offsets on those helices.
    """

    is_scaffold: bool = False
    """Indicates whether this :any:`Strand` is a scaffold for a DNA origami. If any :any:`Strand` in a
    :any:`DNADesign` is a scaffold, then the design is considered a DNA origami design."""

    modification_5p: Optional[Modification5Prime] = None
    """5' modification; None if there is no 5' modification."""

    modification_3p: Optional[Modification3Prime] = None
    """3' modification; None if there is no 5' modification."""

    modifications_int: Dict[int, ModificationInternal] = field(default_factory=dict)
    """:any:`Modification`'s to the DNA sequence (e.g., biotin, Cy3/Cy5 fluorphores). Maps offset to 
    modification. If the internal modification is attached to a base 
    (e.g., internal biotin, /iBiodT/ from IDT), 
    then the offset is that of the base.
    If it goes between two bases 
    (e.g., internal Cy3, /iCy3/ from IDT),
    then the offset is that of the previous base, 
    e.g., to put a Cy3 between bases at offsets 3 and 4, the offset should be 3. 
    So for an internal modified base on a sequence of length n, the allowed offsets are 0,...,n-1,
    and for an internal modification that goes between bases, the allowed offsets are 0,...,n-2."""

    # not serialized; efficient way to see a list of all domains on a given helix
    _helix_idx_domain_map: Dict[int, List[Domain]] = field(
        init=False, repr=False, compare=False, default=None)

    def to_json_serializable(self, suppress_indent: bool = True):
        dct = OrderedDict()
        if self.color is not None:
            dct[color_key] = self.color.to_json_serializable(suppress_indent)
        if self.dna_sequence is not None:
            dct[dna_sequence_key] = self.dna_sequence
        if self.idt is not None:
            dct[idt_key] = self.idt.to_json_serializable(suppress_indent)
        dct[domains_key] = [domain.to_json_serializable(suppress_indent) for domain in self.domains]
        if hasattr(self, is_scaffold_key) and self.is_scaffold == True:
            dct[is_scaffold_key] = self.is_scaffold

        if self.modification_5p is not None:
            dct[modification_5p_key] = self.modification_5p.id
        if self.modification_3p is not None:
            dct[modification_3p_key] = self.modification_3p.id
        if len(self.modifications_int) > 0:
            mods_dict = {}
            for offset, mod in self.modifications_int.items():
                mods_dict[f"{offset}"] = mod.id
            dct[modifications_int_key] = _NoIndent(mods_dict) if suppress_indent else mods_dict

        return dct

    def __post_init__(self):
        # if color not specified, pick one by cycling through list of staple colors,
        # unless caller specified not to
        global color_cycler
        if self.color is None and self.automatically_assign_color:
            if self.is_scaffold:
                self.color = default_scaffold_color
            else:
                self.color = next(color_cycler)

        self._helix_idx_domain_map = defaultdict(list)

        for domain in self.domains:
            if domain.is_domain():
                self._helix_idx_domain_map[domain.helix].append(domain)

        for domain in self.domains:
            domain._parent_strand = self

        if len(self.domains) == 1:
            if self.first_domain().is_loopout():
                raise StrandError(self, 'strand cannot have a single Loopout as its only domain')

        for domain1, domain2 in _pairwise(self.domains):
            if domain1.is_loopout() and domain2.is_loopout():
                raise StrandError(self, 'cannot have two consecutive Loopouts in a strand')

        if self.use_default_idt:
            self.set_default_idt(True)

        self._ensure_modifications_legal()

    def __eq__(self, other: Strand) -> bool:
        if not isinstance(other, Strand):
            return False
        return self.domains == other.domains

    def __hash__(self):
        return hash(self.domains)

    def set_scaffold(self, is_scaf: bool = True):
        """Sets this :any:`Strand` as a scaffold. Alters color to default scaffold color.

        If `is_scaf` == ``False``, sets this strand as not a scaffold, and leaves the color alone."""
        self.is_scaffold = is_scaf
        if is_scaf:
            self.color = default_scaffold_color

    def set_color(self, color: Color):
        """Sets color of this :any:`Strand`."""
        self.color = color

    def set_default_idt(self, use_default_idt: bool = True, skip_scaffold: bool = True):
        """Sets idt field to be the default given the Domain data of this :any:`Strand`.

        """
        if skip_scaffold and self.is_scaffold: return
        self.use_default_idt = use_default_idt
        if use_default_idt:
            start_helix = self.first_bound_domain().helix
            end_helix = self.last_bound_domain().helix
            start_offset = self.first_bound_domain().offset_5p()
            end_offset = self.last_bound_domain().offset_3p()
            self.idt = IDTFields(name=f'ST{start_helix}[{start_offset}]{end_helix}[{end_offset}]')
        else:
            self.idt = None

    def set_modification_5p(self, mod: Modification5Prime = None):
        """Sets 5' modification to be `mod`."""
        self.modification_5p = mod

    def set_modification_3p(self, mod: Modification3Prime = None):
        """Sets 3' modification to be `mod`."""
        self.modification_3p = mod

    def remove_modification_5p(self):
        """Removes 5' modification."""
        self.modification_5p = None

    def remove_modification_3p(self):
        """Removes 3' modification."""
        self.modification_3p = None

    def set_modification_internal(self, idx: int, mod: ModificationInternal, warn_on_no_dna: bool = True):
        """Adds internal modification `mod` at given DNA index `idx`."""
        if idx < 0:
            raise IllegalDNADesignError('idx of modification must be nonnegative')
        if idx >= self.dna_length():
            raise IllegalDNADesignError(f'idx of modification must be at most length of DNA: '
                                        f'{self.dna_length()}')
        if self.dna_sequence is not None:
            if mod.allowed_bases is not None and self.dna_sequence[idx] not in mod.allowed_bases:
                raise IllegalDNADesignError(f'only bases {",".join(mod.allowed_bases)} are allowed at '
                                            f'index {idx}, but sequence has base {self.dna_sequence[idx]} '
                                            f'\nDNA sequence: {self.dna_sequence}'
                                            f'\nmodification: {mod}')
        elif warn_on_no_dna:
            print('WARNING: no DNA sequence has been assigned, so certain error checks on the internal '
                  'modification were not done. To be safe, first assign DNA, then add the modifications.')
        self.modifications_int[idx] = mod

    def remove_modification_internal(self, idx: int):
        """Removes internal modification at given DNA index `idx`."""
        if idx in self.modifications_int:
            del self.modifications_int[idx]

    def first_domain(self) -> Union[Domain, Loopout]:
        """First domain (of type either :any:`Domain` or :any:`Loopout`) on this :any:`Strand`."""
        return self.domains[0]

    def last_domain(self) -> Union[Domain, Loopout]:
        """Last domain (of type either :any:`Domain` or :any:`Loopout`) on this :any:`Strand`."""
        return self.domains[-1]

    def set_dna_sequence(self, sequence: str):
        """Set this :any:`Strand`'s DNA sequence to `seq`
        WITHOUT checking for complementarity with overlapping
        :any:`Strand`'s or automatically assigning their sequences.
        To assign a sequence to a :any:`Strand` and have the overlapping
        :any:`Strand`'s automatically have the appropriate Watson-Crick complements assigned,
        use :any:`DNADesign.assign_dna`.

        All whitespace in `sequence` is removed,
        and lowercase bases 'a', 'c', 'g', 't' are converted to uppercase.

        `sequence`, after all whitespace is removed, must be exactly the same length as
        :py:meth:`Strand.dna_length`.
        Wildcard symbols (:py:const:`DNA_case_wildcard`) are allowed to leave part of the DNA unassigned.
        """
        trimmed_seq = _remove_whitespace_and_uppercase(sequence)
        if len(trimmed_seq) != self.dna_length():
            domain = self.first_domain()
            raise StrandError(self, f"strand starting at helix {domain.helix} offset {domain.offset_5p()} "
                                    f"has length {self.dna_length()}, but you attempted to assign a "
                                    f"DNA sequence of length {len(trimmed_seq)}: {sequence}")
        self.dna_sequence = trimmed_seq

    def dna_length(self) -> int:
        """Return sum of DNA length of :any:`Domain`'s and :any:`Loopout`'s of this :any:`Strand`."""
        acc = 0
        for domain in self.domains:
            acc += domain.dna_length()
        return acc

    def bound_domains(self) -> List[Domain]:
        """:any:`Domain`'s of this :any:`Strand` that are not :any:`Loopout`'s."""
        return [domain for domain in self.domains if domain.is_domain()]

    def offset_5p(self) -> int:
        """5' offset of this entire :any:`Strand`, INCLUSIVE."""
        return self.first_domain().offset_5p()

    def offset_3p(self) -> int:
        """3' offset of this entire :any:`Strand`, INCLUSIVE."""
        return self.last_domain().offset_3p()

    def overlaps(self, other: Strand) -> bool:
        """Indicates whether `self` overlaps `other_strand`, meaning that the set of offsets occupied
        by `self` has nonempty intersection with those occupied by `other_strand`."""
        for domain_self in self.bound_domains():
            for domain_other in other.bound_domains():
                if domain_self.overlaps(domain_other):
                    return True
        return False

    def assign_dna_complement_from(self, other: Strand):
        """Assuming a DNA sequence has been assigned to `other`, assign its Watson-Crick
        complement to the portions of this Strand that are bound to `other`.

        Generally this is not called directly; use :py:meth:`DNADesign.assign_dna` to assign
        a DNA sequence to a :any:`Strand`. The method :py:meth:`DNADesign.assign_dna` will calculate
        which other :any:`Strand`'s need
        to be assigned via :py:meth:`Strand.assign_dna_complement_from`.

        However, it is permitted to assign the field :py:data:`Strand.dna_sequence` directly
        via the method :py:meth:`Strand.set_dna_sequence`.
        This is used, for instance, to assign a DNA sequence to a :any:`Strand` bound to another
        :any:`Strand`
        with an assigned DNA sequence where they overlap. In this case no error checking
        about sequence complementarity is done. This can be used to intentionally assign *mismatching*
        DNA sequences to :any:`Strand`'s that are bound on a :any:`Helix`."""

        already_assigned = self.dna_sequence is not None

        # put DNA sequences to assign to domains in List, one position per domain
        strand_complement_builder = []
        if already_assigned:
            for domain in self.domains:
                strand_complement_builder.append(domain.dna_sequence())
        else:
            for domain in self.domains:
                wildcards = DNA_base_wildcard * domain.dna_length()
                strand_complement_builder.append(wildcards)

        for (domain_idx, domain_self) in enumerate(self.domains):
            if domain_self.is_loopout():
                domain_self_dna_sequence = DNA_base_wildcard * domain_self.dna_length()
            else:
                helix = domain_self.helix

                # for helix, domains_on_helix_self in self._helix_idx_domain_map.items():
                domains_on_helix_other = other._helix_idx_domain_map[helix]
                # for domain_self in domains_on_helix_self:
                overlaps = []
                for domain_other in domains_on_helix_other:
                    if domain_self != domain_other and domain_self.overlaps(domain_other):
                        overlap = domain_self.compute_overlap(domain_other)
                        overlaps.append((overlap, domain_other))

                overlaps.sort()

                domain_complement_builder = []
                start_idx = domain_self.start
                # repeatedly insert wildcards into gaps, then reverse WC complement
                for ((overlap_left, overlap_right), domain_other) in overlaps:
                    # wildcards = DNA_base_wildcard * (overlap_left - start_idx)
                    num_wildcard_bases = domain_self.dna_length_in(start_idx, overlap_left - 1)
                    wildcards = DNA_base_wildcard * num_wildcard_bases

                    other_seq = domain_other.dna_sequence_in(overlap_left, overlap_right - 1)
                    overlap_complement = wc(other_seq)
                    domain_complement_builder.append(wildcards)
                    domain_complement_builder.append(overlap_complement)
                    start_idx = overlap_right

                # last wildcard for gap between last overlap and end
                # last_wildcards = DNA_base_wildcard * (domain_self.end - start_idx)
                num_wildcard_bases = domain_self.dna_length_in(start_idx, domain_self.end - 1)
                last_wildcards = DNA_base_wildcard * num_wildcard_bases

                domain_complement_builder.append(last_wildcards)

                # If pointing left, each individual overlap sequence was reverse orientation in wc(),
                # but not the list of all of them put together until now.
                if not domain_self.forward:
                    domain_complement_builder.reverse()

                domain_self_dna_sequence = ''.join(domain_complement_builder)

            # merge with existing pre-assigned sequence
            existing_domain_self_dna_sequence = strand_complement_builder[domain_idx]
            merged_domain_self_dna_sequence = _string_merge_wildcard(domain_self_dna_sequence,
                                                                     existing_domain_self_dna_sequence,
                                                                     DNA_base_wildcard)
            strand_complement_builder[domain_idx] = merged_domain_self_dna_sequence

        strand_complement = ''.join(strand_complement_builder)
        new_dna_sequence = strand_complement
        if self.dna_sequence is not None:
            try:
                new_dna_sequence = _string_merge_wildcard(self.dna_sequence, new_dna_sequence,
                                                          DNA_base_wildcard)
            except ValueError:
                domain_self = self.first_domain()
                domain_other = other.first_domain()
                msg = f'strand starting at helix {domain_self.helix}, offset {domain_self.offset_5p()} ' \
                      f'has length ' \
                      f'{self.dna_length()} and already has a partial DNA sequence assignment of length ' \
                      f'{len(self.dna_sequence)}, which is \n' \
                      f'{self.dna_sequence}, ' \
                      f'but you tried to assign sequence of length {len(new_dna_sequence)} to it, which ' \
                      f'is\n{new_dna_sequence} (this assignment was indirect, since you assigned directly ' \
                      f'to a strand bound to this one). This occurred while directly assigning a DNA ' \
                      f'sequence to the strand whose 5\' end is at helix {domain_other.helix}, and is of ' \
                      f'length {other.dna_length()}.'
                raise IllegalDNADesignError(msg)

        self.set_dna_sequence(new_dna_sequence)
        # self.dna_sequence = _pad_dna(new_dna_sequence, self.dna_length())

    def _insert_domain(self, order, domain):
        """Only intended to be called by DNADesign.insert_domain"""
        self.domains.insert(order, domain)
        domain._parent_strand = self
        if domain.is_domain():
            self._helix_idx_domain_map[domain.helix].append(domain)
        if self.use_default_idt:
            self.set_default_idt()

    def _remove_domain(self, domain):
        """Only intended to be called by DNADesign.remove_domain"""
        self.domains.remove(domain)
        domain._parent_strand = None
        if domain.is_domain():
            self._helix_idx_domain_map[domain.helix].remove(domain)
        if self.use_default_idt:
            self.set_default_idt()

    def contains_loopouts(self):
        for domain in self.domains:
            if domain.is_loopout():
                return True
        return False

    def first_bound_domain(self):
        """First :any:`Domain` (i.e., not a :any:`Loopout`) on this :any:`Strand`.

        Currently the first and last strand must not be :any:`Loopout`'s, so this should return the same
        domain as :py:meth:`Strand.first_domain`, but in case an initial or final :any:`Loopout` is
        supported in the future, this method is provided."""
        for domain in self.domains:
            if domain.is_domain():
                return domain

    def last_bound_domain(self):
        """Last :any:`Domain` (i.e., not a :any:`Loopout`) on this :any:`Strand`.

        Currently the first and last strand must not be :any:`Loopout`'s, so this should return the same
        domain as :py:meth:`Strand.first_domain`, but in case an initial or final :any:`Loopout` is
        supported in the future, this method is provided."""
        domain_rev = list(self.domains)
        domain_rev.reverse()
        for domain in domain_rev:
            if domain.is_domain():
                return domain

    def reverse(self):
        """
        Reverses "polarity" of this :any:`Strand`.

        Does NOT check whether this keeps the :any:`DNADesign` legal, so be cautious in calling this method
        directly. To reverse every :any:`Strand`, called :py:meth:`DNADesign.reverse_all`.
        If the design was legal before, it will be legal after calling that method.
        """
        self.domains.reverse()
        for domain in self.bound_domains():
            domain.forward = not domain.forward

    @staticmethod
    def from_json(json_map: dict) -> Strand:
        if domains_key not in json_map:
            domain_jsons = None
            for legacy_domain_key in legacy_domains_keys:
                domain_jsons = json_map[legacy_domain_key]
            if domain_jsons is None:
                raise IllegalDNADesignError(
                    f'key "{domains_key}" (as well as legacy key {",".join(legacy_domains_keys)}) '
                    f'is missing from the description of a Strand:'
                    f'\n  {json_map}')
        else:
            domain_jsons = json_map[domains_key]
        if len(domain_jsons) == 0:
            raise IllegalDNADesignError(f'{domains_key} list cannot be empty')

        domains = []
        for domain_json in domain_jsons:
            if loopout_key in domain_json:
                domains.append(Loopout.from_json(domain_json))
            else:
                domains.append(Domain.from_json(domain_json))
        if isinstance(domains[0], Loopout):
            raise IllegalDNADesignError('Loopout at beginning of Strand not supported')
        if isinstance(domains[-1], Loopout):
            raise IllegalDNADesignError('Loopout at end of Strand not supported')

        is_scaffold = json_map.get(is_scaffold_key, False)

        dna_sequence = json_map.get(dna_sequence_key)
        if dna_sequence_key not in json_map:
            for legacy_dna_sequence_key in legacy_dna_sequence_keys:
                if legacy_dna_sequence_key in json_map:
                    dna_sequence = json_map.get(legacy_dna_sequence_key)
                    break

        idt = json_map.get(idt_key)
        color_str = json_map.get(color_key,
                                 default_scaffold_color if is_scaffold else default_strand_color)
        color = Color(hex=color_str)

        return Strand(
            domains=domains,
            dna_sequence=dna_sequence,
            color=color,
            idt=idt,
            is_scaffold=is_scaffold,
        )

    def _ensure_modifications_legal(self, check_offsets_legal=False):
        if check_offsets_legal:
            if self.dna_sequence is None:
                raise IllegalDNADesignError(f"must assign DNA sequence first")
            modI_offsets_list = list(self.modifications_int.keys())
            min_offset = min(modI_offsets_list) if len(modI_offsets_list) > 0 else None
            max_offset = max(modI_offsets_list) if len(modI_offsets_list) > 0 else None
            if min_offset is not None and min_offset < 0:
                raise IllegalDNADesignError(f"smallest offset is {min_offset} but must be nonnegative: "
                                            f"{self.modifications_int}")
            if max_offset is not None and max_offset > len(self.dna_sequence):
                raise IllegalDNADesignError(f"largeest offset is {max_offset} but must be at most "
                                            f"{len(self.dna_sequence)}: "
                                            f"{self.modifications_int}")

    def idt_dna_sequence(self):
        self._ensure_modifications_legal(check_offsets_legal=True)

        ret_list = []
        if self.modification_5p is not None:
            ret_list.append(self.modification_5p.idt_text)

        for offset, base in enumerate(self.dna_sequence):
            ret_list.append(base)
            if offset in self.modifications_int:  # if internal mod attached to base, replace base
                mod = self.modifications_int[offset]
                if mod.allowed_bases is not None:
                    if base not in mod.allowed_bases:
                        msg = f'internal modification {mod} can only replace one of these bases: ' \
                              f'{",".join(mod.allowed_bases)}, but the base at offset {offset} is {base}'
                        raise IllegalDNADesignError(msg)
                    ret_list[-1] = mod.idt_text  # replace base with modified base
                else:
                    ret_list.append(mod.idt_text)  # append modification between two bases

        if self.modification_3p is not None:
            ret_list.append(self.modification_3p.idt_text)

        return ''.join(ret_list)

    def unmodified_version(self):
        strand_nomods = replace(self, modification_3p=None, modification_5p=None, modifications_int={})
        return strand_nomods


def _string_merge_wildcard(s1: str, s2: str, wildcard: str) -> str:
    """Takes a "union" of two equal-length strings `s1` and `s2`.
    Whenever one has a symbol `wildcard` and the other does not, the result has the non-wildcard symbol.

    Raises :py:class:`ValueError` if `s1` and `s2` are not the same length or do not agree on non-wildcard
    symbols at any position."""
    if len(s1) != len(s2):
        raise ValueError(f'\ns1={s1} and\ns2={s2}\nare not the same length.')
    union_builder = []
    for i in range(len(s1)):
        c1, c2 = s1[i], s2[i]
        if c1 == wildcard:
            union_builder.append(c2)
        elif c2 == wildcard:
            union_builder.append(c1)
        elif c1 != c2:
            raise ValueError(f's1={s1} and s2={s2} have unequal symbols {c1} and {c2} at position {i}.')
        elif c1 == c2:
            union_builder.append(c1)
        else:
            raise AssertionError('should be unreachable')
    return ''.join(union_builder)


class IllegalDNADesignError(ValueError):
    """Indicates that some aspect of the :any:`DNADesign` object is illegal."""

    def __init__(self, the_cause: str):
        self.cause = the_cause

    # __str__ is to print() the value
    def __str__(self):
        return repr(self.cause)


class StrandError(IllegalDNADesignError):
    """Indicates that the :any:`DNADesign` is illegal due to some specific :any:`Strand`.
    Information about the :any:`Strand` is embedded in the error message when this exception is
    raised that helps to identify which :any:`Strand` caused the problem."""

    def __init__(self, strand: Strand, the_cause: str):
        first_domain = strand.first_bound_domain()
        last_domain = strand.last_bound_domain()

        msg = (f'{the_cause}\n'
               f'strand length        =  {strand.dna_length()}\n'
               f'DNA length           =  {len(strand.dna_sequence) if strand.dna_sequence else "N/A"}\n'
               f'DNA sequence         =  {strand.dna_sequence}'
               f"strand 5' helix      =  {first_domain.helix if first_domain else 'N/A'}\n"
               f"strand 5' end offset =  {first_domain.offset_5p() if first_domain else 'N/A'}\n"
               f"strand 3' helix      =  {last_domain.helix if last_domain else 'N/A'}\n"
               f"strand 3' end offset =  {last_domain.offset_3p() if last_domain else 'N/A'}\n")

        super().__init__(msg)
        # super(IllegalDNADesignError, self).__init__(msg)


def _plates(idt_strands):
    plates = set()
    for strand in idt_strands:
        if strand.idt is not None and strand.idt.plate is not None:
            plates.add(strand.idt.plate)
    return list(plates)


_96WELL_PLATE_ROWS: List[str] = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H']
_96WELL_PLATE_COLS: List[int] = list(range(1, 13))

_384WELL_PLATE_ROWS: List[str] = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
                                  'O',
                                  'P']
_384WELL_PLATE_COLS: List[int] = list(range(1, 25))


@enum.unique
class PlateType(int, enum.Enum):
    """Represents two different types of plates in which DNA sequences can be ordered."""

    wells96 = 96
    """96-well plate."""

    wells384 = 384
    """384-well plate."""

    def rows(self) -> List[str]:
        return _96WELL_PLATE_ROWS if self is PlateType.wells96 else _384WELL_PLATE_ROWS

    def cols(self) -> List[int]:
        return _96WELL_PLATE_COLS if self is PlateType.wells96 else _384WELL_PLATE_COLS


class _PlateCoordinate:

    def __init__(self, plate_type: PlateType):
        self._plate_type = plate_type
        self._plate: int = 1
        self._row_idx: int = 0
        self._col_idx: int = 0

    def increment(self):
        self._row_idx += 1
        if self._row_idx == len(self._plate_type.rows()):
            self._row_idx = 0
            self._col_idx += 1
            if self._col_idx == len(self._plate_type.cols()):
                self._col_idx = 0
                self._plate += 1

    def plate(self) -> int:
        return self._plate

    def row(self) -> str:
        return self._plate_type.rows()[self._row_idx]

    def col(self) -> int:
        return self._plate_type.cols()[self._col_idx]

    def well(self) -> str:
        return f'{self.row()}{self.col()}'


def remove_helix_idxs_if_default(helices: List[Dict]):
    # removes indices from each helix if they are the default (order of appearance in list)
    default = True
    for expected_idx, helix in enumerate(helices):
        idx = helix[idx_on_helix_key]
        if idx != expected_idx:
            default = False
            break

    if default:
        for helix in helices:
            del helix[idx_on_helix_key]


@dataclass
class DNADesign(_JSONSerializable):
    """Object representing the entire design of the DNA structure."""

    strands: List[Strand]
    """All of the :any:`Strand`'s in this :any:`DNADesign`.
    
    Required field."""

    helices: Dict[int, Helix] = None
    """All of the :any:`Helix`'s in this :any:`DNADesign`. 
    This is a dictionary mapping index to the :any:`Helix` with that index; if helices have indices 
    0, 1, ..., num_helices-1, then this can be used as a list of Helices. 
    
    Optional field. If not specified, then the number of helices will be just large enough to store the
    largest index :py:data:`Domain.helix` 
    stored in any :any:`Domain` 
    in :py:data:`DNADesign.strands`."""

    grid: Grid = Grid.square
    """Common choices for how to arrange helices relative to each other.
    
    Optional field."""

    major_tick_distance: int = -1
    """Distance between major ticks (bold) delimiting boundaries between bases.
    
    Optional field.
    If not specified, default value is 8 unless overridden by :py:data:`DNADesign.grid`.
    If 0 then no major ticks are drawn.
    If negative then the default value is assumed, but `major_tick_distance` is not stored in the JSON file
    when serialized.
    If :any:`DNADesign.grid` = :any:`Grid.square` then the default value is 8.
    If :any:`DNADesign.grid` = :any:`Grid.hex` or :any:`Grid.honeycomb` then the default value is 7."""

    helices_view_order: List[int] = None
    """A list of the order in which the helix should be displayed in the main view of scadnano.
       
    This list must be a permutation containing each integer 0, 1, 2, ..., len(helices)-1 exactly once.

    Optional field. If not specified, it will be set to the identity permutation [0, ..., len(helices)-1].
    """

    @property
    def scaffold(self) -> Optional[Strand]:
        """Returns the first scaffold in this :any:`DNADesign`, if there is one, or ``None`` otherwise."""
        for strand in self.strands:
            if strand.is_scaffold:
                return strand
        return None

    def __init__(self, *,
                 helices: Optional[Union[List[Helix], Dict[int, Helix]]] = None,
                 strands: List[Strand] = None,
                 grid: Grid = Grid.square,
                 major_tick_distance: int = -1,
                 helices_view_order: List[int] = None):
        self.helices = helices
        self.strands = strands
        self.grid = grid
        self.major_tick_distance = major_tick_distance
        self.helices_view_order = helices_view_order

        if self.strands is None:
            self.strands = []

        if self.major_tick_distance < 0 or self.major_tick_distance is None:
            self.major_tick_distance = default_major_tick_distance(self.grid)

        if self.helices is None:
            if len(self.strands) > 0:
                max_helix_idx = max(
                    domain.helix for strand in self.strands for domain in strand.bound_domains())
                self.helices = {idx: Helix(idx=idx) for idx in range(max_helix_idx + 1)}
            else:
                self.helices = {}

        self.helices = DNADesign._normalize_helices_as_dict(self.helices)

        self.__post_init__()

    @staticmethod
    def _normalize_helices_as_dict(helices: Union[List[Helix], Dict[int, Helix]]) -> Dict[int, Helix]:
        def idx_of(helix: Helix, order: int):
            return order if helix.idx is None else helix.idx

        if isinstance(helices, list):
            indices = [idx_of(helix, idx) for idx, helix in enumerate(helices)]
            if len(set(indices)) < len(indices):
                duplicates = [index for index, count in Counter(indices).items() if count > 1]
                raise IllegalDNADesignError(
                    'No two helices can share an index, but these indices appear on '
                    f'multiple helices: {", ".join(map(str, duplicates))}')
            helices = {idx_of(helix, idx): helix for idx, helix in enumerate(helices)}

        for idx, helix in helices.items():
            helix.idx = idx

        return helices

    def __post_init__(self):
        # XXX: exact order of these calls is important
        self._set_helices_idxs()
        self._set_helices_grid_and_svg_positions()
        self._build_domains_on_helix_lists()
        self._set_helices_min_max_offsets(update=False)
        self._check_legal_design()

        self._set_and_check_helices_view_order()

    @staticmethod
    def from_scadnano_file(filename: str) -> DNADesign:
        """
        Loads a :any:`DNADesign` from the file with the given name.

        :param filename: name of the file with the design. Should be a JSON file ending in .dna
        :return: DNADesign described in the file
        """
        with open(filename) as f:
            json_str = f.read()
        json_map = json.loads(json_str)
        return DNADesign._from_scadnano_json(json_map)

    @staticmethod
    def _from_scadnano_json(json_map: dict) -> DNADesign:
        # reads scadnano .dna file format into a DNADesign object
        version = json_map.get(version_key, initial_version)  # not sure what to do with this
        grid = json_map.get(grid_key, Grid.square)
        grid_is_none = grid == Grid.none

        if (major_tick_distance_key in json_map):
            major_tick_distance = json_map[major_tick_distance_key]
        elif not grid_is_none:
            if grid in [Grid.hex, Grid.honeycomb]:
                major_tick_distance = 7
            else:
                major_tick_distance = 8
        else:
            major_tick_distance = -1

        helices = []
        deserialized_helices_list = json_map[helices_key]
        num_helices = len(deserialized_helices_list)

        # create Helices
        idx_default = 0
        for helix_json in deserialized_helices_list:
            helix = Helix.from_json(helix_json)
            if grid_is_none and grid_position_key in helix_json:
                raise IllegalDNADesignError(
                    f'grid is none, but Helix {idx_default} has grid_position = {helix_json[grid_position_key]}')
            elif not grid_is_none and position3d_key in helix_json:
                raise IllegalDNADesignError(
                    'grid is not none, but Helix $idx has position = ${helix_json[constants.position3d_key]}')
            helices.append(helix)
            idx_default += 1

        # view order of helices
        helices_view_order = json_map.get(helices_view_order_key)
        if helices_view_order is not None:
            helix_idxs = [helix.idx for helix in helices]
            if len(helices_view_order) != num_helices:
                raise IllegalDNADesignError(f'length of helices ({num_helices}) does not match '
                                            f'length of helices_view_order ({len(helices_view_order)})')
            if sorted(helices_view_order) != sorted(helix_idxs):
                raise IllegalDNADesignError(f'helices_view_order = {helices_view_order} is not a '
                                            f'permutation of the set of helices {helix_idxs}')

        # strands
        strands = []
        strand_jsons = json_map[strands_key]
        for strand_json in strand_jsons:
            strand = Strand.from_json(strand_json)
            strands.append(strand)

        # modifications in whole design
        if design_modifications_key in json_map:
            all_mods_json = json_map[design_modifications_key]
            all_mods = {}
            for mod_key, mod_json in all_mods_json.items():
                mod = Modification.from_json(mod_json)
                mod = dataclasses.replace(mod, id=mod_key)
                all_mods[mod_key] = mod
            DNADesign.assign_modifications_to_strands(strands, strand_jsons, all_mods)

        return DNADesign(
            helices=helices,
            strands=strands,
            grid=grid,
            major_tick_distance=major_tick_distance,
            helices_view_order=helices_view_order,
        )

    @staticmethod
    def assign_modifications_to_strands(strands: List[Strand], strand_jsons: List[dict],
                                        all_mods: Dict[str, Modification]):
        for strand, strand_json in zip(strands, strand_jsons):
            if modification_5p_key in strand_json:
                mod_name = strand_json[modification_5p_key]
                strand.modification_5p = all_mods[mod_name]
            if modification_3p_key in strand_json:
                mod_name = strand_json[modification_3p_key]
                strand.modification_3p = all_mods[mod_name]
            if modifications_int_key in strand_json:
                mod_names_by_offset = strand_json[modifications_int_key]
                for offset_str, mod_name in mod_names_by_offset.items():
                    offset = int(offset_str)
                    strand.modifications_int[offset] = all_mods[mod_name]

    @staticmethod
    def _cadnano_v2_import_find_5_end(vstrands, strand_type: str, helix_num: int, base_id: int,
                                      id_from: int,
                                      base_from: int):
        """ Routine which finds the 5' end of a strand in a cadnano v2 import. It returns the
        helix and the base of the 5' end.
        """
        id_from_before = helix_num
        base_from_before = base_id
        while not (id_from == -1 and base_from == -1):
            id_from_before = id_from
            base_from_before = base_from
            id_from, base_from, _, _ = vstrands[id_from][strand_type][base_from]
        return id_from_before, base_from_before

    @staticmethod
    def _cadnano_v2_import_find_strand_color(vstrands, strand_type: str, strand_5_end_base: int,
                                             strand_5_end_helix: int):
        """ Routines which finds the color of a cadnano v2 strand. """
        color = default_scaffold_color
        if strand_type == 'stap':
            for base_id, stap_color in vstrands[strand_5_end_helix]['stap_colors']:
                if base_id == strand_5_end_base:
                    color = Color.from_cadnano_v2_int_hex(stap_color)
                    break
        return color

    @staticmethod
    def _cadnano_v2_import_extract_deletions(skip_table, start, end):
        """ Routines which converts cadnano skips to scadnano deletions """
        to_return = []
        for base_id in range(start, end):
            if skip_table[base_id] == -1:
                to_return.append(base_id)
        return to_return

    @staticmethod
    def _cadnano_v2_import_extract_insertions(loop_table, start, end):
        """ Routines which converts cadnano skips to scadnano insertions """
        to_return = []
        for base_id in range(start, end):
            if loop_table[base_id] != 0:
                to_return.append([base_id, loop_table[base_id]])
        return to_return

    @staticmethod
    def _cadnano_v2_import_explore_domains(vstrands, seen, strand_type: str, strand_5_end_base: int,
                                           strand_5_end_helix: int):
        """Finds all domains of a cadnano v2 strand. """
        curr_helix = strand_5_end_helix
        curr_base = strand_5_end_base
        domains = []

        direction_forward = (strand_type == 'scaf' and curr_helix % 2 == 0) or (
            (strand_type == 'stap' and curr_helix % 2 == 1))
        start, end = -1, -1
        if direction_forward:
            start = curr_base
        else:
            end = curr_base

        while not (curr_helix == -1 and curr_base == -1):
            old_helix = curr_helix
            old_base = curr_base
            seen[(curr_helix, curr_base)] = True
            curr_helix, curr_base = vstrands[curr_helix][strand_type][curr_base][2:]
            # Add crossover
            # We have a crossover when we jump helix or when order is broken on same helix
            if curr_helix != old_helix or (not direction_forward and curr_base > old_base) or (
                    direction_forward and curr_base < old_base):
                if direction_forward:
                    end = old_base
                else:
                    start = old_base

                domains.append(
                    Domain(old_helix, direction_forward, min(start, end), max(start, end) + 1,
                           deletions=DNADesign._cadnano_v2_import_extract_deletions(
                               vstrands[old_helix]['skip'], start, end),
                           insertions=DNADesign._cadnano_v2_import_extract_insertions(
                               vstrands[old_helix]['loop'], start, end)))

                direction_forward = (strand_type == 'scaf' and curr_helix % 2 == 0) or (
                    (strand_type == 'stap' and curr_helix % 2 == 1))
                start, end = -1, -1
                if direction_forward:
                    start = curr_base
                else:
                    end = curr_base

        return domains

    @staticmethod
    def _cadnano_v2_import_explore_strand(vstrands, num_bases: int, strand_type: str, seen,
                                          helix_num: int,
                                          base_id: int):
        """ Routine that will follow a cadnano v2 strand accross helices and create
            cadnano domains and strand accordingly.
        """

        seen[(helix_num, base_id)] = True
        id_from, base_from, id_to, base_to = vstrands[helix_num][strand_type][base_id]

        if (id_from, base_from, id_to, base_to) == (-1, -1, -1, -1):
            return None

        strand_5_end_helix, strand_5_end_base = DNADesign._cadnano_v2_import_find_5_end(vstrands,
                                                                                        strand_type,
                                                                                        helix_num,
                                                                                        base_id,
                                                                                        id_from,
                                                                                        base_from)
        strand_color = DNADesign._cadnano_v2_import_find_strand_color(vstrands, strand_type,
                                                                      strand_5_end_base,
                                                                      strand_5_end_helix)
        domains = DNADesign._cadnano_v2_import_explore_domains(vstrands, seen, strand_type,
                                                               strand_5_end_base,
                                                               strand_5_end_helix)
        strand = Strand(domains=domains, is_scaffold=(strand_type == 'scaf'), color=strand_color)

        return strand

    @staticmethod
    def from_cadnano_v2(directory: str = None, filename: str = None, json_dict: dict = None) -> DNADesign:
        """ Creates a DNADesign from a cadnano v2 file.
        """

        if json_dict is None:
            file_path = os.path.join(directory, filename)
            f = open(file_path, 'r')
            cadnano_v2_design = json.load(f)
            f.close()
        else:
            cadnano_v2_design = json_dict

        num_bases = len(cadnano_v2_design['vstrands'][0]['scaf'])
        grid_type = Grid.square
        if num_bases % 21 == 0:
            grid_type = Grid.honeycomb

        min_row, min_col = None, None
        for cadnano_helix in cadnano_v2_design['vstrands']:
            col, row = cadnano_helix['col'], cadnano_helix['row']
            min_row = row if min_row is None else min_row
            min_col = col if min_col is None else min_col
            min_row = row if row < min_row else min_row
            min_col = col if col < min_col else min_col

        helices = OrderedDict({})
        for cadnano_helix in cadnano_v2_design['vstrands']:
            col, row = cadnano_helix['col'], cadnano_helix['row']
            num = cadnano_helix['num']
            helix = Helix(idx=num, max_offset=num_bases,
                          grid_position=(col, row, 0))
            helices[num] = helix

        # We do a DFS on strands
        seen = {'scaf': {}, 'stap': {}}
        strands = []
        cadnano_helices = OrderedDict({})
        for cadnano_helix in cadnano_v2_design['vstrands']:
            helix_num = cadnano_helix['num']
            cadnano_helices[helix_num] = cadnano_helix

        for cadnano_helix in cadnano_v2_design['vstrands']:
            helix_num = cadnano_helix['num']
            for strand_type in ['scaf', 'stap']:
                for base_id in range(num_bases):
                    if (helix_num, base_id) in seen[strand_type]:
                        continue

                    strand = DNADesign._cadnano_v2_import_explore_strand(cadnano_helices, num_bases,
                                                                         strand_type,
                                                                         seen[strand_type], helix_num,
                                                                         base_id)
                    if not strand is None:
                        strands.append(strand)

        design = DNADesign(grid=grid_type, helices=helices, strands=strands)
        design.set_helices_view_order([num for num in helices])

        return design

    def _all_modifications(self) -> Set[Modification]:
        # List of all modifications.
        mods_5p = {strand.modification_5p for strand in self.strands if
                   strand.modification_5p is not None}
        mods_3p = {strand.modification_3p for strand in self.strands if
                   strand.modification_3p is not None}
        mods_int = {mod for strand in self.strands for mod in strand.modifications_int.values()}
        return mods_5p | mods_3p | mods_int

    def assign_m13_to_scaffold(self, rotation: int = 5588, variant: M13Variant = M13Variant.p7249):
        """Assigns the scaffold to be the sequence of M13: :py:func:`m13` with the given `rotation`.

        Raises :any:`IllegalDNADesignError` if the number of scaffolds is not exactly 1.
        """
        scaffold = None
        num_scafs = 0
        for strand in self.strands:
            if strand.is_scaffold:
                num_scafs += 1
                if scaffold is None:
                    scaffold = strand
        if num_scafs == 0:
            raise IllegalDNADesignError(
                'Tried to assign DNA to scaffold, but there is no scaffold strand. '
                'You must set strand.is_scaffold to True for exactly one strand.')
        elif num_scafs > 1:
            raise IllegalDNADesignError(
                'Tried to assign DNA to scaffold, but there are multiple scaffold '
                'strands. You must set strand.is_scaffold to True for exactly one '
                'strand.')
        self.assign_dna(scaffold, m13(rotation, variant))

    def to_json_serializable(self, suppress_indent: bool = True):
        dct = OrderedDict()
        dct[version_key] = current_version
        if self.grid != default_grid:
            dct[grid_key] = str(self.grid)[5:]  # remove prefix 'Grid.'
        if self.major_tick_distance >= 0:  # and (
            # self.major_tick_distance != default_major_tick_distance(self.grid)):
            dct[major_tick_distance_key] = self.major_tick_distance

        dct[helices_key] = [helix.to_json_serializable(suppress_indent) for helix in
                            self.helices.values()]

        # remove idx key from list of helices if they have the default index
        unwrapped_helices = dct[helices_key]
        if len(unwrapped_helices) > 0 and isinstance(unwrapped_helices[0], _NoIndent):
            unwrapped_helices = [wrapped.value for wrapped in unwrapped_helices]
        remove_helix_idxs_if_default(unwrapped_helices)

        default_helices_view_order = list(range(0, len(self.helices)))
        if self.helices_view_order != default_helices_view_order:
            dct[helices_view_order_key] = _NoIndent(self.helices_view_order)

        # modifications
        mods = self._all_modifications()
        if len(mods) > 0:
            mods_dict = {}
            for mod in mods:
                if mod.id not in mods_dict:
                    mods_dict[mod.id] = mod.to_json_serializable(suppress_indent)
            dct[design_modifications_key] = mods_dict

        dct[strands_key] = [strand.to_json_serializable(suppress_indent) for strand in self.strands]

        for helix_list_order, helix in enumerate(self.helices.values()):
            helix_json = dct[helices_key][helix_list_order]
            if suppress_indent:
                helix_json = helix_json.value  # get past NoIndent surrounding helix
            # XXX: no need to check here because key was already deleted by Helix.to_json_serializable
            # max_offset still needs to be checked here since it requires global knowledge of Strands
            # if 0 == helix_json[min_offset_key]:
            #     del helix_json[min_offset_key]
            max_offset = max((domain.end for domain in helix._domains), default=-1)
            if max_offset == helix_json[max_offset_key]:
                del helix_json[max_offset_key]

        return dct

    @staticmethod
    def _get_multiple_of_x_sup_closest_to_y(x: int, y: int) -> int:
        return y if y % x == 0 else y + (x - y % x)

    @staticmethod
    def _cadnano_v2_place_strand_segment(helix_dct, domain: Domain,
                                         strand_type: str = 'scaf') -> None:
        """Converts a strand region with no crossover to cadnano v2.
        """
        # Insertions and deletions
        for deletion in domain.deletions:
            helix_dct['skip'][deletion] = -1
        for loop_where, loop_nb in domain.insertions:
            helix_dct['loop'][loop_where] = loop_nb

        start, end, forward = domain.start, domain.end, domain.forward
        strand_helix = helix_dct['num']

        for i_base in range(start, end):
            if forward:
                from_helix, from_base = strand_helix, i_base - 1
                to_helix, to_base = strand_helix, i_base + 1
            else:
                from_helix, from_base = strand_helix, i_base + 1
                to_helix, to_base = strand_helix, i_base - 1

            if i_base == start:
                if forward:
                    helix_dct[strand_type][i_base][2:] = [to_helix, to_base]
                else:
                    helix_dct[strand_type][i_base][:2] = [from_helix, from_base]
            elif i_base < end - 1:
                helix_dct[strand_type][i_base] = [from_helix, from_base, to_helix, to_base]
            else:
                if forward:
                    helix_dct[strand_type][i_base][:2] = [from_helix, from_base]
                else:
                    helix_dct[strand_type][i_base][2:] = [to_helix, to_base]
        return

    @staticmethod
    def _cadnano_v2_place_crossover(helix_from_dct: dict, helix_to_dct: dict,
                                    domain_from: Domain, domain_to: Domain,
                                    strand_type: str = 'scaf') -> None:
        """Converts a crossover to cadnano v2 format.
        Returns a conversion table from ids in the structure self.helices to helices ids
        as given by helix.idx.
        """

        helix_from = helix_from_dct['num']
        start_from, end_from, forward_from = domain_from.start, domain_from.end, domain_from.forward

        helix_to = helix_to_dct['num']
        start_to, end_to = domain_to.start, domain_to.end

        if forward_from:
            helix_from_dct[strand_type][end_from - 1][2:] = [helix_to, end_to - 1]
            helix_to_dct[strand_type][end_to - 1][:2] = [helix_from, end_from - 1]
        else:
            helix_from_dct[strand_type][start_from][2:] = [helix_to, start_to]
            helix_to_dct[strand_type][start_to][:2] = [helix_from, start_from]

    @staticmethod
    def _cadnano_v2_color_of_stap(color: Color, domain: Domain) -> List[int]:
        base_id = domain.start if domain.forward else domain.end - 1
        cadnano_color = color.to_cadnano_v2_int_hex()
        return [base_id, cadnano_color]

    def _cadnano_v2_place_strand(self, strand: Strand, dct: dict,
                                 helices_ids_reverse: Dict[int, int]) -> None:
        """Place a scadnano strand in cadnano v2.
        """
        strand_type = 'stap'
        if hasattr(strand, is_scaffold_key) and strand.is_scaffold:
            strand_type = 'scaf'

        for i, domain in enumerate(strand.domains):
            which_helix_id = helices_ids_reverse[domain.helix]
            which_helix = dct['vstrands'][which_helix_id]

            if strand_type == 'stap':
                which_helix['stap_colors'].append(self._cadnano_v2_color_of_stap(strand.color, domain))

            self._cadnano_v2_place_strand_segment(which_helix, domain, strand_type)

            if i != len(strand.domains) - 1:
                next_domain = strand.domains[i + 1]
                next_helix_id = helices_ids_reverse[next_domain.helix]
                next_helix = dct['vstrands'][next_helix_id]
                self._cadnano_v2_place_crossover(which_helix, next_helix,
                                                 domain, next_domain, strand_type)

    def _cadnano_v2_fill_blank(self, dct: dict, num_bases: int) -> dict:
        """Creates blank cadnanov2 helices in and initialized all their fields.
        """
        helices_ids_reverse = {}
        for i, helix in self.helices.items():
            helix_dct = OrderedDict()
            helix_dct['num'] = helix.idx

            if self.grid == Grid.square:
                helix_dct['row'] = helix.grid_position[1]
                helix_dct['col'] = helix.grid_position[0]

            if self.grid == Grid.honeycomb:
                helix_dct['row'], helix_dct['col'] = helix.grid_position[1], helix.grid_position[0]

            helix_dct['scaf'] = []
            helix_dct['stap'] = []
            helix_dct['loop'] = []
            helix_dct['skip'] = []

            for _ in range(num_bases):
                helix_dct['scaf'].append([-1, -1, -1, -1])
                helix_dct['stap'].append([-1, -1, -1, -1])
                helix_dct['loop'].append(0)
                helix_dct['skip'].append(0)

            helix_dct['stap_colors'] = []
            helix_dct['scafLoop'] = []
            helix_dct['stapLoop'] = []

            helices_ids_reverse[helix_dct['num']] = i
            dct['vstrands'].append(helix_dct)
        return helices_ids_reverse

    def to_cadnano_v2(self):
        """Converts the design to the cadnano v2 format.
        Please see the spec `misc/cadnano-format-specs/v2.txt` for more info on that format.
        """
        dct = OrderedDict()
        dct['vstrands'] = []

        if self.__class__ != DNADesign:
            raise ValueError(
                'Please export DNAOrigamiDesign only as we need to know which strand is the scaffold.')

        '''Figuring out the type of grid.
        In cadnano v2, all helices have the same max offset 
        called `num_bases` and the type of grid is determined as follows:
            if num_bases % 32 == 0: then we are on grid square
            if num_bases % 21 == 0: then we are on grid honey
        '''
        num_bases = 0
        for helix in self.helices.values():
            num_bases = max(num_bases, helix.max_offset)

        if self.grid == Grid.square:
            num_bases = self._get_multiple_of_x_sup_closest_to_y(32, num_bases)
        elif self.grid == Grid.honeycomb:
            num_bases = self._get_multiple_of_x_sup_closest_to_y(21, num_bases)
        else:
            raise NotImplementedError('We can export to cadnano v2 `square` and `honeycomb` grids only.')

        '''Figuring out if helices numbers have good parity.
        In cadnano v2, only even helices have the scaffold go forward, only odd helices
        have the scaffold go backward.

        '''
        for strand in self.strands:
            if hasattr(strand, is_scaffold_key) and strand.is_scaffold:
                for domain in strand.domains:
                    if type(domain) == Loopout:
                        raise ValueError(
                            'We cannot handle designs with Loopouts as it is not a cadnano v2 concept')
                    if domain.helix % 2 != int(not domain.forward):
                        raise ValueError('We can only convert designs where even helices have the scaffold \
                                              going forward and odd helices have the scaffold going backward see the spec v2.txt Note 4. {}'.format(
                            domain))

        '''Filling the helices with blank.
        '''
        helices_ids_reverse = self._cadnano_v2_fill_blank(dct, num_bases)
        '''Putting the scaffold in place.
        '''

        for strand in self.strands:
            self._cadnano_v2_place_strand(strand, dct, helices_ids_reverse)

        return dct

    def _set_and_check_helices_view_order(self):
        identity = list(self.helices.keys())
        if self.helices_view_order is None:
            self.helices_view_order = identity
        self._check_helices_view_order_is_bijection()

    def set_helices_view_order(self, helices_view_order: List[int]):
        self.helices_view_order = helices_view_order
        self._check_helices_view_order_is_bijection()

    def _check_helices_view_order_is_bijection(self):
        if not (sorted(self.helices_view_order) == sorted(self.helices.keys())):
            raise IllegalDNADesignError(
                f"The specified helices view order: {self.helices_view_order}\n "
                f"is not a bijection on helices indices: {self.helices_view_order} {self.helices.keys()}.")

    def _set_helices_idxs(self):
        for idx, helix in self.helices.items():
            if helix.idx is None:
                helix.idx = idx

    def _set_helices_grid_and_svg_positions(self):
        for idx, helix in self.helices.items():
            if helix.grid_position is None:
                helix.grid_position = helix.default_grid_position()
            if helix.svg_position is None:
                helix.svg_position = helix.default_svg_position()

    def _set_helices_min_max_offsets(self, update: bool):
        """update = whether to overwrite existing Helix.max_offset and Helix.min_offset.
        Don't do this when DNADesign is first created, but do it later when updating."""
        for helix in self.helices.values():

            if update or helix.max_offset is None:
                max_offset = None if len(helix._domains) == 0 else helix._domains[0].end
                for domain in helix._domains:
                    max_offset = max(max_offset, domain.end)
                helix.max_offset = max_offset

            if update or helix.min_offset is None:
                min_offset = None if len(helix._domains) == 0 else helix._domains[0].start
                for domain in helix._domains:
                    min_offset = min(min_offset, domain.start)
                if min_offset is None or min_offset > 0:
                    min_offset = 0
                helix.min_offset = min_offset

    def set_default_idt(self, use_default_idt: bool = True):
        """If ``True``, sets :py:data:`Strand.use_default_idt` to ``True`` for every :any:`Strand` in this
        :any:`DNADesign` and calls :py:meth:`Strand.set_default_idt` on each of them to assign a
        default idt field.

        If ``False``, removes IDT field from each :any:`Strand`."""
        for strand in self.strands:
            strand.set_default_idt(use_default_idt)

    def strands_starting_on_helix(self, helix: int) -> List[Strand]:
        """Return list of :any:`Strand`'s that begin (have their 5' end)
        on the :any:`Helix` with index `helix`."""
        return [strand for strand in self.strands if strand.domains[0].helix == helix]

    def strands_ending_on_helix(self, helix: int) -> List[Strand]:
        """Return list of :any:`Strand`'s that finish (have their 3' end)
        on the :any:`Helix` with index `helix`."""
        return [strand for strand in self.strands if strand.domains[-1].helix == helix]

    def _check_legal_design(self):
        # self._check_helix_indices()
        self._check_helix_offsets()
        self._check_strands_reference_helices_legally()
        self._check_loopouts_not_consecutive_or_singletons_or_zero_length()
        self._check_strands_overlap_legally()
        self._check_grid_honeycomb_positions_legal()

    def _check_grid_honeycomb_positions_legal(self):
        # ensures grid positions are legal if honeycomb lattice is used
        if self.grid == Grid.honeycomb:
            for helix in self.helices.values():
                x = helix.grid_position[0]
                y = helix.grid_position[1]

                # following is for odd-q system: https://www.redblobgames.com/grids/hexagons/
                # if x % 2 == 1 and y % 2 == 0:
                #     raise IllegalDNADesignError('honeycomb lattice disallows grid positions of first two '
                #                                 'coordinates (x,y,_) if x is odd and y is even, '
                #                                 f'but helix {helix.idx} has grid position '
                #                                 f'{helix.grid_position}')

                # following is for even-q system: https://www.redblobgames.com/grids/hexagons/
                # if x % 2 == 1 and y % 2 == 1:
                #     raise IllegalDNADesignError('honeycomb lattice disallows grid positions of first two '
                #                                 'coordinates (x,y,_) if both x and y are odd, '
                #                                 f'but helix {helix.idx} has grid position '
                #                                 f'{helix.grid_position}')

                # following is for odd-r system: https://www.redblobgames.com/grids/hexagons/
                # if x % 3 == 0 and y % 2 == 0:
                #     raise IllegalDNADesignError('honeycomb lattice disallows grid positions of first two '
                #                                 'coordinates (x,y,_) with y even and x a multiple of 3, '
                #                                 f'but helix {helix.idx} has grid position '
                #                                 f'{helix.grid_position}')
                # if x % 3 == 1 and y % 2 == 1:
                #     raise IllegalDNADesignError('honeycomb lattice disallows grid positions of first two '
                #                                 'coordinates (x,y) with y odd and x = 1 + a multiple of 3, '
                #                                 f'but helix {helix.idx} has grid position '
                #                                 f'{helix.grid_position}')

    # TODO: come up with reasonable default behavior when no strands are on helix and max_offset not given
    def _check_helix_offsets(self):
        for helix in self.helices.values():
            if helix.min_offset is not None \
                    and helix.max_offset is not None \
                    and helix.min_offset >= helix.max_offset:
                err_msg = f'for helix {helix.idx}, ' \
                          f'helix.min_offset = {helix.min_offset} must be strictly less than ' \
                          f'helix.max_offset = {helix.max_offset}'
                raise IllegalDNADesignError(err_msg)

    # def _check_helix_indices(self):
    #     # ensure if there are H helices, the list of sorted indices is 0,1,...,H-1
    #     indices_helices = sorted([(helix.idx, helix) for helix in self.helices],
    #                              key=lambda x: x[0])
    #     for (correct_idx, (helix_idx, helix)) in enumerate(indices_helices):
    #         if correct_idx != helix_idx:
    #             if correct_idx < helix_idx:
    #                 err_msg = f"missing Helix with helix {correct_idx}"
    #             else:
    #                 err_msg = f"duplicate Helices with helix {helix_idx}"
    #             raise IllegalDNADesignError(err_msg)

    def _check_strands_overlap_legally(self, domain_to_check: Domain = None):
        """If `Domain_to_check` is None, check all.
        Otherwise only check pairs where one is domain_to_check."""

        def err_msg(domain1, domain2, h_idx):
            return f"two domains overlap on helix {h_idx}: " \
                   f"\n{domain1}\n  and\n{domain2}\n  but have the same direction"

        # ensure that if two strands overlap on the same helix,
        # they point in opposite directions
        for helix_idx, domains in enumerate(helix._domains for helix in self.helices.values()):
            if domain_to_check is not None and domain_to_check.helix != helix_idx:
                # TODO: if necessary, we can be more efficient by only checking this one domain
                continue

            if len(domains) == 0:
                continue

            # check all consecutive domains on the same helix, sorted by start/end indices
            offsets_data = []
            for domain in domains:
                offsets_data.append((domain.start, True, domain))
                offsets_data.append((domain.end, False, domain))
            offsets_data.sort(key=lambda offset_data: offset_data[0])

            current_domains: List[Domain] = []
            for offset, is_start, domain in offsets_data:
                if is_start:
                    if len(current_domains) >= 2:
                        if offset >= current_domains[1].end:
                            del current_domains[1]
                    if len(current_domains) >= 1:
                        if offset >= current_domains[0].end:
                            del current_domains[0]
                    current_domains.append(domain)
                    if len(current_domains) > 2:
                        domain0, domain1, domain2 = current_domains[0:3]
                        for d_first, d_second in [(domain0, domain1), (domain1, domain2), (domain0, domain2)]:
                            if d_first.forward == d_second.forward:
                                raise IllegalDNADesignError(err_msg(d_first, d_second, helix_idx))
                        raise AssertionError(
                            f"since current_domains = {current_domains} has at least three domains, "
                            f"I expected to find a pair of illegally overlapping domains")
                    elif len(current_domains) == 2:
                        d_first, d_second = current_domains
                        if d_first.forward == d_second.forward:
                            raise IllegalDNADesignError(err_msg(d_first, d_second, helix_idx))

    def _check_loopouts_not_consecutive_or_singletons_or_zero_length(self):
        for strand in self.strands:
            DNADesign._check_loopout_not_singleton(strand)
            DNADesign._check_two_consecutive_loopouts(strand)
            DNADesign._check_loopouts_length(strand)

    @staticmethod
    def _check_two_consecutive_loopouts(strand):
        for domain1, domain2 in _pairwise(strand.domains):
            if domain1.is_loopout() and domain2.is_loopout():
                raise StrandError(strand, 'cannot have two consecutive Loopouts in a strand')

    @staticmethod
    def _check_loopout_not_singleton(strand):
        if len(strand.domains) == 1:
            if strand.first_domain().is_loopout():
                raise StrandError(strand, 'strand cannot have a single Loopout as its only domain')

    @staticmethod
    def _check_loopouts_length(strand):
        for loopout in strand.domains:
            if loopout.is_loopout() and loopout.length <= 0:
                raise StrandError(strand, f'loopout length must be positive but is {loopout.length}')

    def _check_strands_reference_helices_legally(self):
        # ensure each strand refers to an existing helix
        for strand in self.strands:
            self._check_strand_references_legal_helices(strand)
            self._check_strand_has_legal_offsets_in_helices(strand)

    def _check_strand_has_legal_offsets_in_helices(self, strand: Strand):
        for domain in strand.domains:
            if domain.is_domain():
                helix = self.helices[domain.helix]
                if domain.start < helix.min_offset:
                    err_msg = f"domain {domain} has start offset {domain.start}, " \
                              f"beyond the end of " \
                              f"Helix {domain.helix} that has min_offset = {helix.min_offset}"
                    raise StrandError(domain._parent_strand, err_msg)
                if domain.end > helix.max_offset:
                    err_msg = f"domain {domain} has end offset {domain.end}, " \
                              f"beyond the end of " \
                              f"Helix {domain.helix} that has max_offset = {helix.max_offset}"
                    err = StrandError(domain._parent_strand, err_msg)
                    raise err

    def _check_strand_references_legal_helices(self, strand: Strand):
        for domain in strand.domains:
            if domain.is_domain() and domain.helix not in self.helices:
                err_msg = f"domain {domain} refers to nonexistent Helix index {domain.helix}; " \
                          f"here are the list of valid helices: {self._helices_to_string()}"
                raise StrandError(domain._parent_strand, err_msg)

        # ensure helix_idx's are never negative twice in a row
        for domain1, domain2 in _pairwise(strand.domains):
            if domain1.is_loopout() and domain2.is_loopout():
                err_msg = f"Loopouts {domain1} and {domain2} are consecutive on strand {strand}. " \
                          f"At least one of any consecutive pair must be a Domain, not a Loopout."
                raise StrandError(strand, err_msg)

    def set_helix_idx(self, old_idx: int, new_idx: int):
        if new_idx in self.helices:
            raise IllegalDNADesignError(f'cannot assign idx {new_idx} to helix {old_idx}; '
                                        'another helix already has that index')
        helix: Helix = self.helices[old_idx]
        del self.helices[old_idx]
        self.helices[new_idx] = helix
        helix.idx = new_idx
        for domain in helix._domains:
            domain.helix = new_idx

    def domain_at(self, helix: int, offset: int, forward: bool):
        """
        Return :any:`Domain` that overlaps `offset` on helix with idx `helix` and has
        :py:data:`Domain.forward` = ``True``, or ``None`` if there is no such :any:`Domain`.

        :param helix: TODO
        :param offset: TODO
        :param forward: TODO
        :return: TODO
        """
        for domain in self.domains_at(helix, offset):
            if domain.forward == forward:
                return domain
        return None

    def domains_at(self, helix: int, offset: int) -> List[Domain]:
        """Return list of :any:`Domain`'s that overlap `offset` on helix with idx `helix`.

        If constructed properly, this list should have 0, 1, or 2 elements."""
        domains_on_helix = self.helices[helix]._domains
        # TODO: replace this with a faster algorithm using binary search
        domains_on_helix = [domain for domain in domains_on_helix if
                            domain.contains_offset(offset)]
        if len(domains_on_helix) not in [0, 1, 2]:
            raise AssertionError(f'There should be at most 2 domains on helix {helix}, '
                                 f'but there are {len(domains_on_helix)}:\n{domains_on_helix}')
        return domains_on_helix

    # TODO: add_strand and insert_domain should check for existing deletions/insertion parallel strands
    def add_strand(self, strand: Strand):
        """Add `strand` to this design."""
        self._check_strand_references_legal_helices(strand)
        self.strands.append(strand)
        for domain in strand.domains:
            if domain.is_domain():
                self.helices[domain.helix]._domains.append(domain)

    def remove_strand(self, strand: Strand):
        """Remove `strand` from this design."""
        self.strands.remove(strand)
        for domain in strand.domains:
            if domain.is_domain():
                self.helices[domain.helix]._domains.remove(domain)

    def insert_domain(self, strand: Strand, order: int, domain: Union[Domain, Loopout]):
        """Insert `Domain` into `strand` at index given by `order`. Uses same indexing as Python lists,
        e.g., ``strand.insert_domain(domain, 0)`` inserts ``domain`` as the new first :any:`Domain`."""
        assert strand in self.strands
        strand._insert_domain(order, domain)
        self._check_strand_references_legal_helices(strand)
        self._check_loopouts_not_consecutive_or_singletons_or_zero_length()
        if domain.is_domain():
            self.helices[domain.helix]._domains.append(domain)

    def remove_domain(self, strand: Strand, domain: Union[Domain, Loopout]):
        """Remove `Domain` from `strand`."""
        assert strand in self.strands
        strand._remove_domain(domain)
        if domain.is_domain():
            self.helices[domain.helix]._domains.remove(domain)

    def _build_domains_on_helix_lists(self):
        for helix in self.helices.values():
            helix._domains = []
        for strand in self.strands:
            for domain in strand.domains:
                if domain.is_domain():
                    if domain.helix in self.helices:
                        self.helices[domain.helix]._domains.append(domain)
                    else:
                        msg = f"domain's helix is {domain.helix} but no helix has that index; here " \
                              f"are the list of helix indices: {self._helices_to_string()}"
                        raise StrandError(strand=strand, the_cause=msg)

    def _helices_to_string(self):
        return ', '.join(map(str, self.helices.keys()))

    def to_json(self, suppress_indent: bool = True) -> str:
        """Return string representing this DNADesign, suitable for reading by scadnano if written to
        a JSON file ending in extension .dna"""
        # if isinstance(self, DNAOrigamiDesign):
        #     scaf = None
        #     for strand in self.strands:
        #         if strand.is_scaffold == True:
        #             scaf = strand
        #             break
        #     if self.scaffold is None:
        #         msg = 'No scaffold specified for DNADesign. You can delay assigning the scaffold ' \
        #               'until after creating the DNADesign object, but you must assign a scaffold ' \
        #               'using the method Strand.set_scaffold() before calling to_json().'
        #         if scaf is not None:
        #             msg += f'There is a strand marked as a scaffold. Try calling set_scaffold with it as ' \
        #                    f'a parameter:\n{scaf}'
        #         raise IllegalDNADesignError(msg)
        return _json_encode(self, suppress_indent)

    # TODO: create version of add_deletion and add_insertion that simply changes the major tick distance
    #  on the helix at that position, as well as updating the end offset of the domain (and subsequent
    #  domains on the same helix)

    def add_deletion(self, helix: int, offset: int):
        """Adds a deletion to every :class:`scadnano.Strand` at the given helix and base offset."""
        domains = self.domains_at(helix, offset)
        if len(domains) == 0:
            raise IllegalDNADesignError(f"no domains are at helix {helix} offset {offset}")
        for domain in domains:
            if domain.contains_offset(offset):
                domain.deletions.append(offset)

    def add_insertion(self, helix: int, offset: int, length: int):
        """Adds an insertion with the given length to every :class:`scadnano.Strand`
        at the given helix and base offset, with the given length."""
        domains = self.domains_at(helix, offset)
        if len(domains) == 0:
            raise IllegalDNADesignError(f"no domains are at helix {helix} offset {offset}")
        for domain in domains:
            if domain.contains_offset(offset):
                domain.insertions.append((offset, length))

    def set_start(self, domain: Domain, start: int):
        """Sets ``Domain.start`` to `start`."""
        assert domain in (domain for strand in self.strands for domain in strand.domains)
        domain.set_start(start)
        self._check_strands_overlap_legally(domain)

    def set_end(self, domain: Domain, end: int):
        """Sets ``Domain.end`` to `end`."""
        assert domain in (domain for strand in self.strands for domain in strand.domains)
        domain.set_end(end)
        self._check_strands_overlap_legally(domain)

    def move_strand_offsets(self, delta: int):
        """Moves all strands backward (if `delta` < 0) or forward (if `delta` > 0) by `delta`."""
        for strand in self.strands:
            for domain in strand.domains:
                domain.start += delta
                domain.end += delta
        self._check_strands_overlap_legally()

    def move_strands_on_helices(self, delta: int):
        """Moves all strands up (if `delta` < 0) or down (if `delta` > 0) by the number of helices given by
        `delta`."""
        for strand in self.strands:
            for domain in strand.domains:
                domain.helix += delta
        self._check_strands_reference_helices_legally()

    def assign_dna(self, strand: Strand, sequence: str):
        """
        Assigns `sequence` as DNA sequence of `strand`.

        If any :class:`scadnano.Strand` is bound to `strand`,
        it is assigned the reverse Watson-Crick complement of the relevant portion,
        and any remaining portions of the other strand that have not already been assigned a DNA sequence
        are assigned to be the symbol :py:data:`DNA_base_wildcard`.

        Before assigning, `sequence` is first forced to be the same length as `strand`
        as follows:
        If `sequence` is longer, it is truncated.
        If `sequence` is shorter, it is padded with :py:data:`DNA_base_wildcard`'s.

        All whitespace in `sequence` is removed, and lowercase bases
        'a', 'c', 'g', 't' are converted to uppercase.
        """
        padded_sequence = _pad_and_remove_whitespace(sequence, strand)
        if strand is None:
            raise IllegalDNADesignError('strand cannot be None to assign DNA to it')
        if strand not in self.strands:
            raise StrandError(strand, 'strand is not in the given DNADesign')

        if strand.dna_sequence is None:
            merged_sequence = padded_sequence
        else:
            try:
                merged_sequence = _string_merge_wildcard(strand.dna_sequence, padded_sequence,
                                                         DNA_base_wildcard)
            except ValueError:
                first_domain = strand.first_domain()
                msg = f'strand starting at helix {first_domain.helix}, offset {first_domain.offset_5p()} has ' \
                      f'length ' \
                      f'{strand.dna_length()} and already has a DNA sequence assignment of length ' \
                      f'{len(strand.dna_sequence)}, which is \n' \
                      f'{strand.dna_sequence}, ' \
                      f'but you tried to assign a different sequence of length {len(padded_sequence)} to ' \
                      f'it, which is\n{padded_sequence}.'
                raise IllegalDNADesignError(msg)

        strand.set_dna_sequence(merged_sequence)

        for other_strand in self.strands:
            # note that possibly strand==other_strand; it might bind to itself at some point and we want to
            # allow a partial assignment to one domain to automatically assign the complement to the
            # bound domain.
            # However, if there are no wildcards in the assigned sequence we can safely skip strand.
            if strand == other_strand and DNA_base_wildcard not in strand.dna_sequence:
                continue
            if other_strand.overlaps(strand):
                # we do this even if other_strand has a complete DNA sequence,
                # because we get complementarity checks this way
                other_strand.assign_dna_complement_from(strand)

    def to_idt_bulk_input_format(self, delimiter: str = ',', warn_duplicate_name: bool = False,
                                 warn_on_non_idt_strands: bool = False,
                                 export_non_modified_strand_version: bool = False) -> str:
        """Return string that is written to the file in the method
        :py:meth:`DNADesign.write_idt_bulk_input_file`.

        `delimiter` is the symbol to delimit the four IDT fields name,sequence,scale,purification.

        `warn_duplicate_name` if ``True`` prints a warning when two different :any:`Strand`'s have the same
        :py:attr:`IDTField.name` and the same :any:`Strand.dna_sequence`. An :any:`IllegalDNADesignError` is
        raised (regardless of the value of this parameter)
        if two different :any:`Strand`'s have the same name but different sequences, IDT scales, or IDT
        purifications.

        `warn_on_non_idt_strands` specifies whether to print a warning for strands that lack the field
        :any:`Strand.idt`. Such strands will not be part of the output. No warning is ever issued for the
        scaffold (regardless of the value of the parameter `warn_on_non_idt_strands`).

        `export_non_modified_strand_version` specifies whether, for each strand that has modifications,
        to also output a version of the strand with no modifications, but otherwise having the same data.
        """
        added_strands = self._idt_strands(warn_duplicate_name, warn_on_non_idt_strands,
                                          export_non_modified_strand_version)

        idt_lines = [
            delimiter.join(
                [strand.idt.name, strand.idt_dna_sequence(), strand.idt.scale, strand.idt.purification])
            for strand in added_strands.values()]

        idt_string = '\n'.join(idt_lines)
        return idt_string

    def _idt_strands(self, warn_duplicate_name: bool, warn_on_non_idt_strands: bool,
                     export_non_modified_strand_version: bool = False) -> Dict[str, Strand]:
        added_strands: Dict[str, Strand] = {}  # dict: name -> strand
        for strand in self.strands:
            if strand.idt is not None:
                name = strand.idt.name
                if name in added_strands:
                    existing_strand = added_strands[name]
                    assert existing_strand.idt.name == name
                    domain = strand.first_domain()
                    existing_domain = existing_strand.first_domain()
                    if strand.dna_sequence != existing_strand.dna_sequence:
                        raise IllegalDNADesignError(
                            f'two strands with same IDT name {name} but different sequences:\n'
                            f'  strand 1: helix {domain.helix}, 5\' end at offset {domain.offset_5p()}, '
                            f'sequence: {strand.dna_sequence}\n'
                            f'  strand 2: helix {existing_domain.helix}, 5\' end at offset '
                            f'{existing_domain.offset_5p()}, '
                            f'sequence: {existing_strand.dna_sequence}\n')
                    elif strand.idt.scale != existing_strand.idt.scale:
                        raise IllegalDNADesignError(
                            f'two strands with same IDT name {name} but different IDT scales:\n'
                            f'  strand 1: helix {domain.helix}, 5\' end at offset {domain.offset_5p()}, '
                            f'scale: {strand.idt.scale}\n'
                            f'  strand 2: helix {existing_domain.helix}, 5\' end at offset '
                            f'{existing_domain.offset_5p()}, '
                            f'scale: {existing_strand.idt.scale}\n')
                    elif strand.idt.purification != existing_strand.idt.purification:
                        raise IllegalDNADesignError(
                            f'two strands with same IDT name {name} but different purifications:\n'
                            f'  strand 1: helix {domain.helix}, 5\' end at offset {domain.offset_5p()}, '
                            f'purification: {strand.idt.purification}\n'
                            f'  strand 2: helix {existing_domain.helix}, 5\' end at offset '
                            f'{existing_domain.offset_5p()}, '
                            f'purification: {existing_strand.idt.purification}\n')
                    elif warn_duplicate_name:
                        print(
                            f'WARNING: two strands with same IDT name {name}:\n'
                            f'  strand 1: helix {domain.helix}, 5\' end at offset {domain.offset_5p()}\n'
                            f'  strand 2: helix {existing_domain.helix}, 5\' end at offset '
                            f'{existing_domain.offset_5p()}\n')
                added_strands[name] = strand
                if export_non_modified_strand_version:
                    added_strands[name + '_nomods'] = strand.unmodified_version()
            elif warn_on_non_idt_strands and not strand.is_scaffold:
                print(f"WARNING: strand with 5' end at (helix, offset) "
                      f"({strand.first_domain().helix}, {strand.first_domain().offset_5p()}) "
                      f"does not have a field idt, so will not be part of IDT output.")
        return added_strands

    def write_idt_bulk_input_file(self, directory: str = '.', filename: str = None, delimiter: str = ',',
                                  warn_duplicate_name: bool = True, warn_on_non_idt_strands: bool = True,
                                  export_non_modified_strand_version: bool = False):
        """Write ``.idt`` text file encoding the strands of this :any:`DNADesign` with the field
        :any:`Strand.idt`, suitable for pasting into the "Bulk Input" field of IDT
        (Integrated DNA Technologies, Coralville, IA, https://www.idtdna.com/),
        with the output file having the same name as the running script but with ``.py`` changed to ``.idt``,
        unless `filename` is explicitly specified.
        For instance, if the script is named ``my_origami.py``,
        then the sequences will be written to ``my_origami.idt``.

        `directory` specifies a directory in which to place the file, either absolute or relative to
        the current working directory. Default is the current working directory.

        `delimiter` is the symbol to delimit the four IDT fields name,sequence,scale,purification.

        `warn_duplicate_name` if ``True`` prints a warning when two different :any:`Strand`'s have the same
        :py:attr:`IDTField.name` and the same :any:`Strand.dna_sequence`. An :any:`IllegalDNADesignError` is
        raised (regardless of the value of this parameter)
        if two different :any:`Strand`'s have the same name but different sequences, IDT scales, or IDT
        purifications.

        `warn_on_non_idt_strands` specifies whether to print a warning for strands that lack the field
        :any:`Strand.idt`. Such strands will not be output into the file.

        The string written is that returned by :meth:`DNADesign.to_idt_bulk_input_format`.
        """
        contents = self.to_idt_bulk_input_format(delimiter, warn_duplicate_name, warn_on_non_idt_strands,
                                                 export_non_modified_strand_version)
        _write_file_same_name_as_running_python_script(contents, 'idt', directory, filename)

    def write_idt_plate_excel_file(self, directory: str = '.', filename: str = None,
                                   warn_duplicate_name: bool = False, warn_on_non_idt_strands: bool = False,
                                   use_default_plates: bool = False, warn_using_default_plates: bool = True,
                                   plate_type: PlateType = PlateType.wells96,
                                   export_non_modified_strand_version: bool = False):
        """Write ``.xls`` (Microsoft Excel) file encoding the strands of this :any:`DNADesign` with the field
        :py:data:`Strand.idt`, suitable for uploading to IDT
        (Integrated DNA Technologies, Coralville, IA, https://www.idtdna.com/)
        to describe a 96-well or 384-well plate
        (https://www.idtdna.com/site/order/plate/index/dna/),
        with the output file having the same name as the running script but with ``.py`` changed to ``.xls``,
        unless `filename` is explicitly specified.
        For instance, if the script is named ``my_origami.py``,
        then the sequences will be written to ``my_origami.xls``.

        `directory` specifies a directory in which to place the file, either absolute or relative to
        the current working directory. Default is the current working directory.

        `warn_duplicate_name` if ``True`` prints a warning when two different :any:`Strand`'s have the same
        :py:attr:`IDTField.name` and the same :any:`Strand.dna_sequence`. An :any:`IllegalDNADesignError` is
        raised (regardless of the value of this parameter)
        if two different :any:`Strand`'s have the same name but different sequences, IDT scales, or IDT
        purifications.

        `warn_on_non_idt_strands` specifies whether to print a warning for strands that lack the field
        :py:data:`Strand.idt`. Such strands will not be output into the file. No warning is ever issued
        for the scaffold (regardless of the value of the parameter `warn_on_non_idt_strands`).

        `warn_using_default_plates` specifies whether to print a warning for strands whose
        :py:data:`Strand.idt` have the fields

        `plate_type` is a :any:`PlateType` specifying whether to use a 96-well plate or a 384-well plate
        if the `use_default_plates` parameter is ``True``.
        Ignored if `use_default_plates` is ``False``, because in that case the wells are explicitly set
        by the user, who is free to use coordinates for either plate type.
        """

        idt_strands = list(self._idt_strands(warn_duplicate_name, warn_on_non_idt_strands,
                                             export_non_modified_strand_version).values())

        if not use_default_plates:
            self._write_plates_assuming_explicit_in_each_strand(directory, filename, idt_strands)
        else:
            self._write_plates_default(directory=directory, filename=filename, idt_strands=idt_strands,
                                       plate_type=plate_type,
                                       warn_using_default_plates=warn_using_default_plates)

    def _write_plates_assuming_explicit_in_each_strand(self, directory: str, filename: str,
                                                       idt_strands: List[Strand]):
        plates = list({strand.idt.plate for strand in idt_strands if strand.idt is not None if
                       strand.idt.plate is not None})
        if len(plates) == 0:
            raise ValueError('Cannot write a a plate file since no plate data exists in any Strands '
                             'in the design.\n'
                             'Set the option use_default_plates=True in '
                             "DNADesign.write_idt_plate_excel_file\nif you don't want to enter plate "
                             'and well positions for each Strand you wish to write to the Excel file.')
        plates.sort()
        filename_plate, workbook = self._setup_excel_file(directory, filename)
        for plate in plates:
            worksheet = self._add_new_excel_plate_sheet(plate, workbook)

            strands_in_plate = [strand for strand in idt_strands if
                                strand.idt is not None and strand.idt.plate == plate]

            strands_in_plate.sort(key=lambda s: (int(s.idt.well[1:]), s.idt.well[0]))

            for row, strand in enumerate(strands_in_plate):
                worksheet.write(row + 1, 0, strand.idt.well)
                worksheet.write(row + 1, 1, strand.idt.name)
                worksheet.write(row + 1, 2, strand.idt_dna_sequence())

            workbook.save(filename_plate)

    @staticmethod
    def _add_new_excel_plate_sheet(plate_name: str, workbook: xlwt.Workbook) -> xlwt.Worksheet:
        worksheet = workbook.add_sheet(plate_name)
        worksheet.write(0, 0, 'Well Position')
        worksheet.write(0, 1, 'Name')
        worksheet.write(0, 2, 'Sequence')
        return worksheet

    @staticmethod
    def _setup_excel_file(directory, filename):
        plate_extension = f'xls'
        if filename is None:
            filename_plate = _get_filename_same_name_as_running_python_script(
                directory, plate_extension, filename)
        else:
            filename_plate = _create_directory_and_set_filename(directory, filename)
        workbook = xlwt.Workbook()
        return filename_plate, workbook

    def _write_plates_default(self, directory: str, filename: str, idt_strands: List[Strand],
                              plate_type: PlateType = PlateType.wells96,
                              warn_using_default_plates: bool = True):
        plate_coord = _PlateCoordinate(plate_type=plate_type)
        plate = 1
        excel_row = 1
        filename_plate, workbook = self._setup_excel_file(directory, filename)
        worksheet = self._add_new_excel_plate_sheet(f'plate{plate}', workbook)

        for strand in idt_strands:
            if warn_using_default_plates and strand.idt.plate is not None:
                print(
                    f"WARNING: strand {strand} has plate entry {strand.idt.plate}, which is being ignored "
                    f"since we are using default plate/well addressing")
            if warn_using_default_plates and strand.idt.well is not None:
                print(
                    f"WARNING: strand {strand} has well entry {strand.idt.well}, which is being ignored "
                    f"since we are using default plate/well addressing")

            well = plate_coord.well()
            worksheet.write(excel_row, 0, well)
            worksheet.write(excel_row, 1, strand.idt.name)
            worksheet.write(excel_row, 2, strand.idt_dna_sequence())
            plate_coord.increment()
            if plate != plate_coord.plate():
                workbook.save(filename_plate)
                plate = plate_coord.plate()
                worksheet = self._add_new_excel_plate_sheet(f'plate{plate}', workbook)
                excel_row = 1
            else:
                excel_row += 1

        workbook.save(filename_plate)

    def write_scadnano_file(self, directory: str = '.', filename=None):
        """Write ``.dna`` file representing this :any:`DNADesign`, suitable for reading by scadnano,
        with the output file having the same name as the running script but with ``.py`` changed to ``.dna``,
        unless `filename` is explicitly specified.
        For instance, if the script is named ``my_origami.py``,
        then the design will be written to ``my_origami.dna``.

        `directory` specifies a directory in which to place the file, either absolute or relative to
        the current working directory. Default is the current working directory.

        The string written is that returned by :meth:`DNADesign.to_json`.
        """
        contents = self.to_json()
        _write_file_same_name_as_running_python_script(contents, 'dna', directory, filename)

    def export_cadnano_v2(self, directory: str = '.', filename=None):
        """Write ``.json`` file representing this :any:`DNADesign`, suitable for reading by cadnano v2,
        with the output file having the same name as the running script but with ``.py`` changed to ``.json``,
        unless `filename` is explicitly specified.
        For instance, if the script is named ``my_origami.py``,
        then the design will be written to ``my_origami.json``.

        `directory` specifies a directory in which to place the file, either absolute or relative to
        the current working directory. Default is the current working directory.

        The string written is that returned by :meth:`DNADesign.to_cadnano_v2`.
        """
        content_serializable = OrderedDict({})
        content_serializable['name'] = _get_filename_same_name_as_running_python_script(directory, 'json',
                                                                                        filename)
        content_serializable_final = self.to_cadnano_v2()
        content_serializable.update(content_serializable_final)

        encoder = _SuppressableIndentEncoder
        contents = json.dumps(content_serializable, cls=encoder, indent=2)

        _write_file_same_name_as_running_python_script(contents, 'json', directory, filename)

    def add_nick(self, helix: int, offset: int, forward: bool):
        """Add nick to :any:`Domain` on :any:`Helix` with index `helix`,
        in direction given by `forward`, at offset `offset`. The two :any:`Domain`'s created by this nick
        will have 5'/3' ends at offsets `offset` and `offset-1`.

        For example, if there is a :any:`Domain` with
        :py:data:`Domain.helix` = ``0``,
        :py:data:`Domain.forward` = ``True``,
        :py:data:`Domain.start` = ``0``,
        :py:data:`Domain.end` = ``10``,
        then calling ``add_nick(helix=0, offset=5, forward=True)`` will split it into two :any:`Domain`'s,
        with one domains having the fields
        :py:data:`Domain.helix` = ``0``,
        :py:data:`Domain.forward` = ``True``,
        :py:data:`Domain.start` = ``0``,
        :py:data:`Domain.end` = ``5``,
        (recall that :py:data:`Domain.end` is exclusive, meaning that the largest offset on this
        :any:`Domain` is 4 = ``offset-1``)
        and the other domain having the fields
        :py:data:`Domain.helix` = ``0``,
        :py:data:`Domain.forward` = ``True``,
        :py:data:`Domain.start` = ``5``,
        :py:data:`Domain.end` = ``10``.
        """
        for domain_to_remove in self.domains_at(helix, offset):
            if domain_to_remove.forward == forward:
                break
        else:
            raise IllegalDNADesignError(f'no domain at helix {helix} in direction '
                                        f'{"forward" if forward else "reverse"} at offset {offset}')
        strand = domain_to_remove.strand()
        domains = strand.domains
        order = domains.index(domain_to_remove)
        domains_before = domains[:order]
        domains_after = domains[order + 1:]
        domain_left = Domain(helix, forward, domain_to_remove.start, offset)
        domain_right = Domain(helix, forward, offset, domain_to_remove.end)

        if domain_to_remove.forward:
            domain_to_add_before = domain_left
            domain_to_add_after = domain_right
        else:
            domain_to_add_before = domain_right
            domain_to_add_after = domain_left

        if strand.dna_sequence:
            dna_sequence_before = ''.join(domain.dna_sequence() for domain in domains_before)
            dna_sequence_after = ''.join(domain.dna_sequence() for domain in domains_after)
            dna_sequence_on_domain_left = domain_to_remove.dna_sequence_in(
                domain_to_remove.start,
                offset - 1)
            dna_sequence_on_domain_right = domain_to_remove.dna_sequence_in(offset,
                                                                            domain_to_remove.end - 1)
            if domain_to_remove.forward:
                dna_sequence_on_domain_before = dna_sequence_on_domain_left
                dna_sequence_on_domain_after = dna_sequence_on_domain_right
            else:
                dna_sequence_on_domain_before = dna_sequence_on_domain_right
                dna_sequence_on_domain_after = dna_sequence_on_domain_left
            dna_sequence_before_whole = dna_sequence_before + dna_sequence_on_domain_before
            dna_sequence_after_whole = dna_sequence_on_domain_after + dna_sequence_after
        else:
            dna_sequence_before_whole = None
            dna_sequence_after_whole = None

        self.strands.remove(strand)

        idt_present = strand.idt is not None
        strand_before = Strand(domains=domains_before + [domain_to_add_before],
                               dna_sequence=dna_sequence_before_whole,
                               color=strand.color, idt=strand.idt if idt_present else None)

        strand_after = Strand(domains=[domain_to_add_after] + domains_after,
                              dna_sequence=dna_sequence_after_whole,
                              automatically_assign_color=True, use_default_idt=idt_present)

        self.helices[helix]._domains.remove(domain_to_remove)
        self.helices[helix]._domains.extend([domain_to_add_before, domain_to_add_after])

        self.strands.extend([strand_before, strand_after])

    def add_half_crossover(self, helix1: int, helix2: int, offset1: int, forward1: bool,
                           offset2: int = None, forward2: bool = None):
        """
        Add a half crossover from helix `helix1` at offset `offset1` to `helix2`, on the strand
        with :py:data:`Strand.forward` = `forward`.

        Unlike :py:meth:`DNADesign.add_full_crossover`, which automatically adds a nick between the two
        half-crossovers, to call this method, there must *already* be nicks adjacent to the given
        offsets on the given helices. (either on the left or right side)

        :param helix1: index of one helix of half crossover
        :param helix2: index of other helix of half crossover
        :param offset1: offset on `helix1` at which to add half crossover
        :param forward1: direction of :any:`Strand` on `helix1` to which to add half crossover
        :param offset2: offset on `helix2` at which to add half crossover.
            If not specified, defaults to `offset1`
        :param forward2: direction of :any:`Strand` on `helix2` to which to add half crossover.
            If not specified, defaults to the negation of `forward1`

        """
        if offset2 is None:
            offset2 = offset1
        if forward2 is None:
            forward2 = not forward1
        domain1 = self.domain_at(helix1, offset1, forward1)
        domain2 = self.domain_at(helix2, offset2, forward2)
        if domain1 is None:
            raise IllegalDNADesignError(
                f"Cannot add half crossover at (helix={helix1}, offset={offset1}). "
                f"There is no Domain there.")
        if domain2 is None:
            raise IllegalDNADesignError(
                f"Cannot add half crossover at (helix={helix2}, offset={offset2}). "
                f"There is no Domain there.")
        strand1 = domain1.strand()
        strand2 = domain2.strand()

        if strand1 == strand2:
            raise IllegalDNADesignError(f"Cannot add crossover from "
                                        f"(helix={helix1}, offset={offset1}) to "
                                        f"(helix={helix2}, offset={offset2}) "
                                        f"because that would join two Domains "
                                        f"already on the same Strand! "
                                        f"Currently circular Strands are not supported. "
                                        f"Instead, try adding nicks first, or rearrange the order of "
                                        f"crossover addition, to ensure that all strands are "
                                        f"non-circular, even in intermediate stages.")

        if domain1.offset_3p() == offset1 and domain2.offset_5p() == offset2:
            strand_first = strand1
            strand_last = strand2
        elif domain1.offset_5p() == offset1 and domain2.offset_3p() == offset2:
            strand_first = strand2
            strand_last = strand1
        else:
            raise IllegalDNADesignError("Cannot add half crossover. Must have one domain have its "
                                        "5' end at the given offset and the other with its 3' end at the "
                                        "given offset, but this is not the case.")

        new_domains = strand_first.domains + strand_last.domains
        if strand_first.dna_sequence is None and strand_last.dna_sequence is None:
            new_dna = None
        elif strand_first.dna_sequence is not None and strand_last.dna_sequence is not None:
            new_dna = strand_first.dna_sequence + strand_last.dna_sequence
        else:
            raise IllegalDNADesignError(
                'cannot add crossover between two strands if one has a DNA sequence '
                'and the other does not')
        new_strand = Strand(domains=new_domains, color=strand_first.color, dna_sequence=new_dna,
                            idt=strand_first.idt)

        self.strands.remove(strand_first)
        self.strands.remove(strand_last)
        self.strands.append(new_strand)

    def add_full_crossover(self, helix1: int, helix2: int, offset1: int, forward1: bool,
                           offset2: int = None, forward2: bool = None):
        """
        Adds two half-crossovers, one at `offset1` and another at `offset1`-1.
        Other arguments have the same meaning as in :py:meth:`DNADesign.add_half_crossover`.
        A nick is automatically added on helix `helix1` between
        `offset1` and `offset1`-1 if one is not already present,
        and similarly for `offset2` on helix `helix2`.
        """
        if offset2 is None:
            offset2 = offset1
        if forward2 is None:
            forward2 = not forward1
        for helix, forward, offset in [(helix1, forward1, offset1), (helix2, forward2, offset2)]:
            self._prepare_nicks_for_full_crossover(helix, forward, offset)
        self.add_half_crossover(helix1=helix1, helix2=helix2, offset1=offset1 - 1, offset2=offset2 - 1,
                                forward1=forward1, forward2=forward2)
        self.add_half_crossover(helix1=helix1, helix2=helix2, offset1=offset1, offset2=offset2,
                                forward1=forward1, forward2=forward2)

    def add_crossovers(self, crossovers: List[Crossover]):
        """
        Adds a list of :any:`Crossover`'s in batch.

        This helps to avoid problems where adding them one at a time
        creates an intermediate design with circular strands.

        :param crossovers: list of :any:`Crossover`'s to add. Its fields have the same meaning as in
            :py:meth:`DNADesign.add_half_crossover`
            and
            :py:meth:`DNADesign.add_full_crossover`,
            with the extra field `Crossover.half` indicating whether it represents a half or full crossover.

        """
        for crossover in crossovers:
            if not crossover.half:
                for helix, forward, offset in [(crossover.helix1, crossover.forward1, crossover.offset1),
                                               (crossover.helix2, crossover.forward2, crossover.offset2)]:
                    self._prepare_nicks_for_full_crossover(helix, forward, offset)

        for crossover in crossovers:
            if crossover.half:
                self.add_half_crossover(helix1=crossover.helix1, helix2=crossover.helix2,
                                        forward1=crossover.forward1, forward2=crossover.forward2,
                                        offset1=crossover.offset1, offset2=crossover.offset2)
            else:
                self.add_full_crossover(helix1=crossover.helix1, helix2=crossover.helix2,
                                        forward1=crossover.forward1, forward2=crossover.forward2,
                                        offset1=crossover.offset1, offset2=crossover.offset2)

    def _prepare_nicks_for_full_crossover(self, helix, forward, offset):
        domain_right = self.domain_at(helix, offset, forward)
        if domain_right is None:
            raise IllegalDNADesignError(f'You tried to create a full crossover at '
                                        f'(helix={helix}, offset={offset}) '
                                        f'but there is no Strand there.')
        domain_left = self.domain_at(helix, offset - 1, forward)
        if domain_left is None:
            raise IllegalDNADesignError(f'You tried to create a full crossover at '
                                        f'(helix={helix}, offset={offset}) '
                                        f'but there is no Strand at offset {offset - 1}.')
        if domain_left == domain_right:
            self.add_nick(helix, offset, forward)
        else:
            assert domain_left.end == domain_right.start

    def inline_deletions_insertions(self):
        """
        Converts deletions and insertions by "inlining" them. Insertions and deletions are removed,
        and their domains have their lengths altered. Also, major tick marks on the helices will be
        shifted to preserve their adjacency to bases already present. For example, if there are major
        tick marks at 0, 8, 18, 24, and a deletion between 0 and 8, then
        the domain is shortened by 1,
        the tick marks become 0, 7, 15, 23,
        and the helix's maximum offset is shrunk by 1.

        We assume that a major tick mark appears just to the LEFT of the offset it encodes,
        so the minimum and maximum offsets for tick marks are respectively the helix's minimum offset
        and 1 plus its maximum offset, the latter being just to the right of the last offset on the helix.
        """
        for helix in self.helices.values():
            self._inline_deletions_insertions_on_helix(helix)

    def _inline_deletions_insertions_on_helix(self, helix: Helix):
        ###################################################
        # first gather information before changing anything

        # gather all mods on helix
        deletions = [deletion for domain in helix._domains for deletion in domain.deletions]
        insertions = [insertion for domain in helix._domains for insertion in domain.insertions]

        # change max offset
        delta_length = sum(length for (offset, length) in insertions) - len(deletions)

        # combined collection of deletions/insertions into one dict mapping offset --> None/len, where
        # value of -1 indicates deletion, and otherwise is length of insertion
        dels_ins = dict()
        for deletion in deletions:
            dels_ins[deletion] = -1
        for insertion in insertions:
            dels_ins[insertion[0]] = insertion[1]

        # put offsets in sorted order
        dels_ins_offsets_sorted = sorted(dels_ins.keys())

        # fix helix major ticks
        major_ticks = sorted(helix.calculate_major_ticks(self.major_tick_distance))

        ###################################################
        # now that info is gathered, start changing things

        helix.max_offset += delta_length
        if len(major_ticks) > 0:
            major_tick_idx = 0
            delta_acc = 0  # accumulated delta; insertions add to this and deletions subtract from it
            for offset in dels_ins_offsets_sorted:
                # go to first major tick great than offset, updating passed ones by delta_acc
                while major_tick_idx < len(major_ticks) and major_ticks[major_tick_idx] <= offset:
                    major_ticks[major_tick_idx] += delta_acc
                    major_tick_idx += 1
                delta_acc += dels_ins[offset]
            # if necessary, update major ticks beyond last ins/del
            while major_tick_idx < len(major_ticks):
                major_ticks[major_tick_idx] += delta_acc
                major_tick_idx += 1
            # TODO: check if regularly spaced and reaching both ends, and if so set helix.major_tick_distance
            helix.major_ticks = major_ticks

        # fix domain start/end offsets
        domains = sorted(helix._domains, key=lambda domain: domain.start)
        delta_acc = 0
        for domain in domains:
            domain.start += delta_acc
            delta_acc += domain.dna_length() - domain.visual_length()
            domain.end += delta_acc
            domain.deletions = []
            domain.insertions = []

    def reverse_all(self):
        """
        Reverses "polarity" of every :any:`Strand` in this :any:`DNADesign`.

        No attempt is made to make any assigned DNA sequences match by reversing or rearranging them.
        Every :any:`Strand` keeps the same DNA sequence it had before (unreversed), if one was assigned.
        It is recommended to assign/reassign DNA sequences *after* doing this operation.
        """
        for strand in self.strands:
            strand.reverse()

    def set_major_tick_distance(self, major_tick_distance: int):
        self.major_tick_distance = major_tick_distance


def _name_of_this_script() -> str:
    """Return name of the currently running script, WITHOUT the .py extension."""
    return os.path.basename(sys.argv[0])[:-3]


def _write_file_same_name_as_running_python_script(contents: str, extension: str, directory: str = '.',
                                                   filename=None):
    relative_filename = _get_filename_same_name_as_running_python_script(directory, extension, filename)
    with open(relative_filename, 'w') as out_file:
        out_file.write(contents)


def _get_filename_same_name_as_running_python_script(directory, extension, filename):
    if filename is None:
        filename = _name_of_this_script() + f'.{extension}'
    relative_filename = _create_directory_and_set_filename(directory, filename)
    return relative_filename


def _create_directory_and_set_filename(directory, filename):
    if not os.path.exists(directory):
        os.makedirs(directory)
    relative_filename = os.path.join(directory, filename)
    return relative_filename


def _remove_whitespace_and_uppercase(sequence):
    sequence = re.sub(r'\s*', '', sequence)
    sequence = sequence.upper()
    return sequence


def _pad_and_remove_whitespace(sequence, strand):
    sequence = _remove_whitespace_and_uppercase(sequence)
    padded_sequence = _pad_dna(sequence, strand.dna_length())
    return padded_sequence


def _pad_dna(sequence: str, length: int) -> str:
    """Return `sequence` modified to have length `length`.

    If len(sequence) < length, pad with  :py:data:`DNA_base_wildcard`.
    If len(sequence) > length, remove extra symbols."""
    if len(sequence) > length:
        sequence = sequence[:length]
    elif len(sequence) < length:
        sequence += DNA_base_wildcard * (length - len(sequence))
    return sequence


@dataclass
class Crossover:
    """
    A :any:`Crossover` object represents the parameters to the methods
    :py:meth:`DNADesign.add_half_crossover`
    and
    :py:meth:`DNADesign.add_full_crossover`,
    with one more field :py:data:`Crossover.half` to identify whether it is a half or full crossover.

    It is used in conjection with :py:meth:`DNADesign.add_crossovers` to add many crossovers in batch.
    This helps avoid the issue that adding crossovers one at a time can lead to an intermediate
    :any:`DNADesign` with circular strands, which are currently unsupported.
    """

    helix1: int
    """index of one helix of half crossover"""

    helix2: int
    """index of other helix of half crossover"""

    offset1: int
    """offset on `helix1` at which to add half crossover"""

    forward1: bool
    """direction of :any:`Strand` on `helix1` to which to add half crossover"""

    offset2: int = None
    """
    offset on `helix2` at which to add half crossover. If not specified, defaults to `offset1`
    """

    forward2: bool = None
    """
    direction of :any:`Strand` on `helix2` to which to add half crossover. 
    If not specified, defaults to the negation of `forward1`
    """

    half: bool = False
    """
    Indicates whether this is a half or full crossover.
    If not specified, defaults to ``False``.
    """

    def __post_init__(self):
        if self.offset2 is None:
            self.offset2 = self.offset1
        if self.forward2 is None:
            self.forward2 = not self.forward1
