from typing import List, Union

import numpy as np

# tempting to use typing.get_args
# and use generics for nicer syntax to capture event type, but requires Python 3.8 onwards
# https://stackoverflow.com/questions/48572831/how-to-access-the-type-arguments-of-typing-generic


class VulnerabilityDistrib:
    """Vulnerability distribution as a discrete matrix."""

    __slots__ = ["_event_type", "_intensity_bins", "_impact_bins", "_prob_matrix"]

    def __init__(
        self,
        event_type: type,
        intensity_bins: Union[List[float], np.ndarray],
        impact_bins: Union[List[float], np.ndarray],
        prob_matrix: Union[List[List[float]], np.ndarray],
    ):
        """Create a new vulnerability distribution.
        Args:
            event_type: type of event
            intensity_bins: non-decreasing intensity bin bounds
            impact_bins: non-decreasing impact bin bounds
            prob_matrix: matrix of probabilities with size [len(intensity_bins) - 1, len(impact_bins) - 1]
        """
        self._event_type = event_type
        self._intensity_bins = np.array(intensity_bins)
        self._impact_bins = np.array(impact_bins)
        self._prob_matrix = np.array(prob_matrix)

    # sdef from_mean_impact(intensity_bins: List[float], impact_bins: List[float]):

    @property
    def event_type(self) -> type:
        return self._event_type

    def intensity_bin_bounds(self):
        return zip(self._intensity_bins[0:-1], self._intensity_bins[1:])

    def impact_bin_bounds(self):
        return zip(self._impact_bins[0:-1], self._impact_bins[1:])

    @property
    def impact_bins(self) -> np.ndarray:
        return self._impact_bins

    @property
    def intensity_bins(self) -> np.ndarray:
        return self._intensity_bins

    @property
    def prob_matrix(self) -> np.ndarray:
        return self._prob_matrix
