"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepScalingPolicy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_common_1 = require("@aws-cdk/aws-autoscaling-common");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const step_scaling_action_1 = require("./step-scaling-action");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Define a scaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 */
class StepScalingPolicy extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_applicationautoscaling_StepScalingPolicyProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, StepScalingPolicy);
            }
            throw error;
        }
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        if (props.datapointsToAlarm !== undefined && props.datapointsToAlarm < 1) {
            throw new RangeError(`datapointsToAlarm cannot be less than 1, got: ${props.datapointsToAlarm}`);
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: props.metricAggregationType ?? aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: props.evaluationPeriods ?? 1,
                datapointsToAlarm: props.datapointsToAlarm,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: props.metricAggregationType ?? aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                scalingTarget: props.scalingTarget,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: props.evaluationPeriods ?? 1,
                datapointsToAlarm: props.datapointsToAlarm,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
_a = JSII_RTTI_SYMBOL_1;
StepScalingPolicy[_a] = { fqn: "@aws-cdk/aws-applicationautoscaling.StepScalingPolicy", version: "1.181.1" };
function aggregationTypeFromMetric(metric) {
    const statistic = metric.toMetricConfig().metricStat?.statistic;
    if (statistic == null) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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